
unit FVektor2;

interface

uses Mode13;

const MaxVertex = 2048;  { Nmero mximo de vrtices en un slido }
      MaxVnum = 10;      { Nmero mximo de vrtices en un polgono }
      MaxPoly = 2048;    { Nmero mximo de polgonos en un slido }

      {Shading types}
      NoShading = 0;        { Sin sombreado (sirve para simular sombras) }
      WireFrame = 1;        { Malla de alambre }
      PhongShading = 4;     { Sombreado Phong }
      TextureMap = 5;       { Mapeado de texturas }
      EnviromentMap = 6;    { Mapeado de entorno }
      TextureWrap = 7;      { Mapeado de textura (envolviendo) }

      long = $66;           { opcode para 32 bits }

{ *** Math funx *** }

type Fixed = longint;

function Sgn(n : longint) : integer;

function Single2Fixed(n : single) : Fixed;      {   Real --> Punto Fijo }
function Fixed2Int(const n : Fixed) : integer;  { Punto Fijo --> Entero }
function RFixed2Int(const n : Fixed) : integer; { Fijo --> Entero (redondeo) }
function Fixed2Single(n : Fixed) : single;      { Punto Fijo --> Real }
function FMul(const n1, n2 : Fixed) : Fixed;    { Multiplicacin }
function FDiv(const n1, n2 : Fixed) : Fixed;    { Divisin }
function FSquare(const n : Fixed) : Fixed;      { Cuadrado }
function FHSqrt(const n : Fixed) : Fixed;       { Raz Cuadrada +precisin }

var SinT, CosT : array[0..359] of Fixed;        { Tablas trigonomtricas }


{ *** Types and object classes *** }

type PTTexture = ^TTexture;
     TTexture = array[0..254] of array [0..255] of byte;

     TFrame = object  { Regin en donde se dibujan los slidos }
            private
                   Xmin, Ymin : Fixed;  { Esquina superior izquierda }
                   Xmax, Ymax : Fixed;  { Esquina inferior derecha }
                   Xcenter, Ycenter : Fixed;  { Coordenadas del centro }
            public
                  procedure Reset; { Fija el marco segn CurrentMode }
                  procedure SetFrame(x1, y1, x2, y2 : word);
                  procedure SetCenter(x1, y1 : word);
                  procedure Center; { Fija el centro con respecto al marco }
            end;


     PTVector = ^TVector; { Objeto vector y su apuntador }
     TVector = object
             private
                    x, y, z : Fixed; { componentes del vector }
             public
                   Zoff : Fixed;   { Se usa para Calc2d }
                   X2d, Y2d : Fixed; { Coordenadas de la proyeccin 2D }

                   constructor Init(nx, ny, nz : integer);
                   procedure SetP(nx, ny, nz : integer);
                   procedure SetPSingle(nx, ny, nz : single);
                   procedure GetP(var nx, ny, nz : integer);
                   procedure VRotate(ax, ay, az : integer);
                   procedure Calc2d; { Calcula la proyeccin a 2D }
                   function FMagnitude : Fixed;
             end;

     PTVertex = ^TVertex; { Objeto vrtice y su apuntador }
     TVertex = object(TVector)
             private
                    normX, normY, normZ : Fixed; { componentes de la normal }
                    u, v : Fixed; { Coordenadas en la textura }
                    procedure Normalize; { Normaliza la normal }
                    procedure CalcUV; { Calcula (u, v) para Env. Mapping }
                    procedure SetUV(n : byte); { (u, v) para Tex. Mapping }
             public
                   constructor Init(nx, ny, nz : integer);
                   procedure Rotate(ax, ay, az : integer);
             end;


     PTPoly = ^TPoly;
     TPoly = object
           private
                  Vnum : byte;
                  colStart : byte; { primer color del sombreado }
                  colWidth : byte; { nmero de colores para el sombreado }
                  MaxColor : byte;
                  Style : byte;
                  normX, normY, normZ : Fixed; { coord. de la normal }
                  function ZOrder : Fixed;
                  function Visible : boolean;
           public
                 Vertex : array[1..MaxVnum] of PTVertex;
                 TexSeg, TexOff : word;
                 TColWidth : byte;

                 constructor Init;
                 procedure AddVertex(v : PTVertex);
                 procedure Draw(where : word);
                 procedure CalcNormal;
                 procedure RotateNormal(ax, ay, az : integer);
                 procedure Rotate(ax, ay, az : integer);
                 procedure Scale(sx, sy, sz : single);
           end;


     PTPolyList = ^TPolyList;
     TPolyList = object(TPoly)
               public
                     next : PTPolyList;
                     constructor Init;
                     destructor Done;
               end;


     PTSolid = ^TSolid;
     TSolid = object
            private
                   Pnum : word; { nmero de polgonos }
                   ox, oy, oz : Fixed; { coord. del origen }
                   Style : byte;
                   ColStart : byte; { color de inicio }
                   ColWidth : byte; { nmero de colores }
                   TexSeg, TexOff : word;
                   TColWidth : byte;
            public
                  Vnum : word; { nmero de vrtices }
                  Vertex : array[1..MaxVertex] of PTVertex;
                  Pfirst : PTPolyList;
                  HideFaces : boolean;
                  constructor Init;
                  destructor Done;
                  procedure AddVertex(vx, vy, vz : integer);
                  procedure AddPoly(poly : PTPolyList);
                  procedure CalcNormals;
                  procedure ReCalcUV;
                  procedure SetUV(n : byte);
                  procedure SetOrigin(nx, ny, nz : integer);
                  procedure GetOrigin(var nx, ny, nz : integer);
                  procedure SetColor(cstart, cwidth : byte);
                  procedure SetStyle(ns : byte);
                  procedure SetTexture(t : PTTexture; tcwidth : byte);
                  function CountVertex : word;
                  function CountPoly : word;

                  procedure MakePhongTable;
                  procedure Draw(where : word);
                  procedure Rotate(ax, ay, az : integer);
                  procedure FastRot(ax, ay, az : integer);
                  procedure Scale(sx, sy, sz : single);

                  procedure Load(filename : string);
                  procedure Save(filename : string);
            end;


var Frame : TFrame;
    Light : record x, y, z : Fixed; end;


{ LightSource procedures }
procedure MoveLight(x, y, z : integer);
procedure RotateLight(ax, ay, az : integer);

{ Texture procedures }
procedure LoadTexture(fn : string; t : PTTexture; var p : TPalette;
                      ostart, oend : longint);


{ Palette procedures }
procedure MakePhongPal( ra, rd, rs, ga, gd, gs, ba, bd, bs : single;
                        n : word; var p : TPalette; start, range : word);

procedure MakeTexturePal( ra, rs, ga, gs, ba, bs, m : single; n : word;
                          var p : TPalette; start, range : byte;
                          tpal : TPalette; twidth : byte);



implementation

{ *** Funciones Matemticas *** }
var PhongCol : array[0..360] of byte;           { tabla de colores }
    ArcCos : array[-128..128] of Fixed;         { coseno inverso }

    left_v, right_v : array[0..2] of PTVertex;
    left_n, right_n : byte;
    left_height, right_height : integer;
    left_x, left_dx, right_x, right_dx : Fixed;


function Sgn(n : longint) : integer;
begin
     if n = 0 then Sgn := 0
        else if n > 0 then Sgn := 1
             else Sgn := -1;
end;


function Single2Fixed(n : single) : Fixed;
begin
     Single2Fixed := round(n * 65536);
end;

function Fixed2Int(const n : Fixed) : integer; assembler;
asm
   db long; mov ax, word ptr [n]     { mov eax, n }
   db long; sar ax, 16               { sar eax, 16 }
end;

function RFixed2Int(const n : Fixed) : integer; assembler;
asm
   db long; mov ax, word ptr [n]     { mov eax, n }
   db long; add ax, 8000h; dw 0      { add eax, 8000h }
   db long; sar ax, 16               { sar eax, 16 }
end;

function Fixed2Single(n : Fixed) : single;
begin
     Fixed2Single := n / 65536;
end;

function FMul(const n1, n2 : Fixed) : Fixed; assembler;
asm
   db long; mov ax, word ptr [n1]     { mov eax, n1 }
   db long; mov dx, word ptr [n2]     { mov edx, n2 }
   db long; imul dx                   { imul edx    }
   db long; add ax, 8000h; dw 0       { add eax, 8000h }
   db long; adc dx, 0                 { adc edx, 0  }
   db long, $0F, $AC, $D0, $10        { shrd eax, edx, 16 }
end;

function FDiv(const n1, n2 : Fixed) : FIxed; assembler;
asm
   db long; mov dx, word ptr [n1]     { mov edx, n1 }
   db long; mov bx, word ptr [n2]     { mov ebx, n2 }
   db long; xor ax, ax                { xor eax, eax }
   db long, $0F, $AC, $D0, $10        { shrd eax, edx, 16 }
   db long; sar dx, 16                { sar edx, 16 }
   db long; idiv bx                   { idiv ebx }
   db long, $0F, $A4, $C2, $10        { shld edx, eax, 16 }
end;

function FSquare(const n : Fixed) : Fixed; assembler;
asm
   db long; mov ax, word ptr [n]     { mov eax, n }
   db long; imul ax                  { imul eax }
   db long; add ax, 8000h; dw 0      { add eax, 8000h }
   db long; adc dx, 0                { adc edx, 0 }
   db long, $0F, $AC, $D0, $10       { shrd eax, edx, 16 }
end;

function FHSqrt(const n : Fixed) : Fixed; assembler;
asm
   db long; mov cx, word ptr [n]
   db long; xor ax, ax
   db long; mov bx, 0; dw 4000h

@loop1: db long; mov dx, cx
        db long; sub dx, bx
                 jb @loop2
        db long; sub dx, ax
                 jb @loop2
        db long; mov cx, dx
        db long; shr ax, 1
        db long; or ax, bx
        db long; shr bx, 2
                 jnz @loop1
                 jz @loop5

@loop2: db long; shr ax, 1
        db long; shr bx, 2
                 jnz @loop1

@loop5: db long; mov bx, 4000h; dw 0
        db long; shl ax, 16
        db long; shl cx, 16

@loop3: db long; mov dx, cx
        db long; sub dx, bx
                 jb @loop4
        db long; sub dx, ax
                 jb @loop4
        db long; mov cx, dx
        db long; shr ax, 1
        db long; or ax, bx
        db long; shr bx, 2
                 jnz @loop3
                 jmp @loop6

@loop4: db long; shr ax, 1
        db long; shr bx, 2
                 jnz @loop3

@loop6: db long, $0F, $A4, $C2, $10
end;


{ El siguiente procedimiento calcula las tablas de senos y cosenos }

procedure CalcTrigTables;
var angle : integer;
    value1, value2 : single;
begin
     for angle := 0 to 359 do
     begin
          sint[angle] := Single2Fixed(sin(angle * Pi / 180));
          cost[angle] := Single2Fixed(cos(angle * Pi / 180));
     end;

     for angle := -128 to 128 do
     begin
          if angle = 0 then value1 := 90
          else
          begin
               value1 := angle / 128;
               value2 := arctan(sqrt(1 - sqr(value1)) / value1);
               value1 := value2 * 180 / Pi;
               if value1 < 0 then value1 := value1 + 180;
          end;
          ArcCos[angle] := Single2Fixed(value1);
     end;
end;

{ *** Funciones auxiliares para interpolacin *** }
function DoRightSide : integer;
var height : integer;
    v1, v2 : PTVertex;
begin
     v1 := right_v[right_n];
     v2 := right_v[right_n - 1];
     height := (v2^.y2d - v1^.y2d) shr 16;
     if height = 0 then
     begin
          DoRightSide := 0;
          exit;
     end;
     right_dx := (v2^.x2d - v1^.x2d) div height;
     right_x := v1^.x2d;
     right_height := height;
     DoRightSide := height;
end;

function DoLeftSide : integer;
var height : integer;
    v1, v2 : PTVertex;
begin
     v1 := left_v[left_n];
     v2 := left_v[left_n - 1];
     height := (v2^.y2d - v1^.y2d) shr 16;
     if height = 0 then
     begin
          DoLeftSide := 0;
          exit;
     end;
     left_dx := (v2^.x2d - v1^.x2d) div height;
     left_x := v1^.x2d;
     left_height := height;
     DoLeftSide := height;
end;

{ *** Dibujo de tringulos *** }

{ *** Tringulo Plano *** }
procedure FlatTriangle(vertex1, vertex2, vertex3 : PTVertex; color : byte; where : word);
var v1, v2, v3, v0 : PTVertex;
    height : integer;
    temp, longest : Fixed;
    ScrOff : word;
    x1, width : integer;
begin
     v1 := vertex1;
     v2 := vertex2;
     v3 := vertex3;
     if v1^.y2d > v2^.y2d then begin v0 := v1; v1 := v2; v2 := v0; end;
     if v1^.y2d > v3^.y2d then begin v0 := v1; v1 := v3; v3 := v0; end;
     if v2^.y2d > v3^.y2d then begin v0 := v2; v2 := v3; v3 := v0; end;

     height := (v3^.y2d - v1^.y2d) shr 16;
     if height = 0 then exit;
     temp := (v2^.y2d - v1^.y2d) div height;
     longest := FMul(temp, v3^.x2d - v1^.x2d) + v1^.x2d - v2^.x2d;
     if longest = 0 then exit;

     if longest < 0 then
     begin
          right_v[0] := v3;
          right_v[1] := v2;
          right_v[2] := v1;
          right_n := 2;
          left_v[0] := v3;
          left_v[1] := v1;
          left_n := 1;

          if DoLeftSide <= 0 then exit;

          if DoRightSide <= 0 then
          begin
               dec(right_n);
               if DoRightSide <= 0 then exit;
          end;

          if longest > -65536 then longest := -65536;
     end
     else
     begin
          left_v[0] := v3;
          left_v[1] := v2;
          left_v[2] := v1;
          left_n := 2;
          right_v[0] := v3;
          right_v[1] := v1;
          right_n := 1;

          if DoRightSide <= 0 then exit;

          if DoLeftSide <= 0 then
          begin
               dec(left_n);
               if DoLeftSide <= 0 then exit;
          end;

          if longest < 65536 then longest := 65536;
     end;

     ScrOff := (v1^.y2d shr 16) * 320;

     while true do
     begin
          x1 := left_x shr 16;
          width := (right_x shr 16) - x1;
          if width > 0 then
          begin
               asm
                  mov di, x1
                  mov es, where
                  add di, ScrOff
                  mov al, color
                  mov cx, width
               @loop1:
                       mov es:[di], al
                       inc di
                       dec cx
                       jnz @loop1
               end;
          end;
          inc(ScrOff, 320);

          dec(left_height);
          if left_height <= 0 then
          begin
               dec(left_n);
               if left_n <= 0 then exit;
               if DoLeftSide <= 0 then exit;
          end
          else inc(left_x, left_dx);

          dec(right_height);
          if right_height <= 0 then
          begin
               dec(right_n);
               if right_n <= 0 then exit;
               if DoRightSide <= 0 then exit;
          end
          else inc(right_x, right_dx);
     end;
end;

{ *** Tringulo Phong *** }
procedure PhongTriangle(vertex1, vertex2, vertex3 : PTVertex; where : word);
var v1, v2, v3, v0 : PTVertex;
    height, dummy : integer;
    temp, longest : Fixed;
    lefta, leftda, dadx : Fixed;
    ScrOff : word;
    x1, width : integer;
    a, da, a1, a2, a3 : Fixed;
begin
     v1 := vertex1;
     v2 := vertex2;
     v3 := vertex3;
     if v1^.y2d > v2^.y2d then begin v0 := v1; v1 := v2; v2 := v0; end;
     if v1^.y2d > v3^.y2d then begin v0 := v1; v1 := v3; v3 := v0; end;
     if v2^.y2d > v3^.y2d then begin v0 := v2; v2 := v3; v3 := v0; end;

     height := (v3^.y2d - v1^.y2d) shr 16;
     if height = 0 then exit;
     temp := (v2^.y2d - v1^.y2d) div height;
     longest := FMul(temp, v3^.x2d - v1^.x2d) + v1^.x2d - v2^.x2d;
     if longest = 0 then exit;

     a1 := ArcCos[( FMul(v1^.normX, Light.X) + FMul(v1^.normY, Light.Y) +
                    FMul(v1^.normZ, Light.Z)) div 512];
     a2 := ArcCos[( FMul(v2^.normX, Light.X) + FMul(v2^.normY, Light.Y) +
                    FMul(v2^.normZ, Light.Z)) div 512];
     a3 := ArcCos[( FMul(v3^.normX, Light.X) + FMul(v3^.normY, Light.Y) +
                    FMul(v3^.normZ, Light.Z)) div 512];

{     a1 := FMul(v1^.normX, Light.X) + FMul(v1^.normY, Light.Y) +
           FMul(v1^.normZ, Light.Z) + 65536;
     a2 := FMul(v2^.normX, Light.X) + FMul(v2^.normY, Light.Y) +
           FMul(v2^.normZ, Light.Z) + 65536;
     a3 := FMul(v3^.normX, Light.X) + FMul(v3^.normY, Light.Y) +
           FMul(v3^.normZ, Light.Z) + 65536;
     asm
        db long; mov bx, word ptr [a1]
        db long; shr bx, 9
        db long; shl bx, 2
        db long; mov ax, word ptr [ArcCos + bx]
        db long; mov word ptr [a1], ax
        db long; mov bx, word ptr [a2]
        db long; shr bx, 9
        db long; shl bx, 2
        db long; mov ax, word ptr [ArcCos + bx]
        db long; mov word ptr [a2], ax
        db long; mov bx, word ptr [a3]
        db long; shr bx, 9
        db long; shl bx, 2
        db long; mov ax, word ptr [ArcCos + bx]
        db long; mov word ptr [a3], ax
     end;}

     if longest < 0 then
     begin
          right_v[0] := v3;
          right_v[1] := v2;
          right_v[2] := v1;
          right_n := 2;
          left_v[0] := v3;
          left_v[1] := v1;
          left_n := 1;

          dummy := DoLeftSide;
          if dummy <= 0 then exit;
          leftda := (a3 - a1) div dummy;
          lefta := a1;

          if DoRightSide <= 0 then
          begin
               dec(right_n);
               if DoRightSide <= 0 then exit;
          end;

          if longest > -65536 then longest := -65536;
     end
     else
     begin
          left_v[0] := v3;
          left_v[1] := v2;
          left_v[2] := v1;
          left_n := 2;
          right_v[0] := v3;
          right_v[1] := v1;
          right_n := 1;

          if DoRightSide <= 0 then exit;

          dummy := DoLeftSide;
          if dummy <= 0 then
          begin
               dec(left_n);
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftda := (a3 - a2) div dummy;
               lefta := a2;
          end
          else
          begin
               leftda := (a2 - a1) div dummy;
               lefta := a1;
          end;

          if longest < 65536 then longest := 65536;
     end;

     dadx := FDiv(FMul(temp, a3 - a1) + a1 - a2, longest);

     ScrOff := (v1^.y2d shr 16) * 320;
     while true do
     begin
          x1 := left_x shr 16;
          width := (right_x shr 16) - x1;
          if width > 0 then
          begin
               da := dadx;
               a := lefta - dadx;

               asm
                  mov di, scroff
                  mov es, where
                  add di, x1
                  db long; mov dx, word ptr [a]
                  db long; mov si, word ptr [da]
                  mov cx, width

               @loop1:
                       db long; mov bx, dx
                       db long; add dx, si
                       db long; shr bx, 15
                       mov al, byte ptr [PhongCol + bx]
                       mov es:[di], al
                       inc di
                       dec cx
                       jnz @loop1
               end;
          end;
          inc(ScrOff, 320);

          dec(left_height);
          if left_height <= 0 then
          begin
               dec(left_n);
               if left_n <= 0 then exit;
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftda := (a3 - a2) div dummy;
               lefta := a2;
          end
          else
          begin
               inc(left_x, left_dx);
               inc(lefta, leftda);
          end;

          dec(right_height);
          if right_height <= 0 then
          begin
               dec(right_n);
               if right_n <= 0 then exit;
               if DoRightSide <= 0 then exit;
          end
          else inc(right_x, right_dx);
     end;
end;


{ *** Tringulo con mapeo de textura *** }
procedure TextureTriangle(vertex1, vertex2, vertex3 : PTVertex; tSeg, where : word);
var v1, v2, v3, v0 : PTVertex;
    height, dummy : integer;
    temp, longest : Fixed;
    leftu, leftdu, leftv, leftdv, dudx, dvdx : Fixed;
    ScrOff : word;
    x1, width : integer;
    u, v, du, dv : Fixed;
begin
     v1 := vertex1;
     v2 := vertex2;
     v3 := vertex3;
     if v1^.y2d > v2^.y2d then begin v0 := v1; v1 := v2; v2 := v0; end;
     if v1^.y2d > v3^.y2d then begin v0 := v1; v1 := v3; v3 := v0; end;
     if v2^.y2d > v3^.y2d then begin v0 := v2; v2 := v3; v3 := v0; end;

     height := (v3^.y2d - v1^.y2d) shr 16;
     if height = 0 then exit;
     temp := (v2^.y2d - v1^.y2d) div height;
     longest := FMul(temp, v3^.x2d - v1^.x2d) + v1^.x2d - v2^.x2d;
     if longest = 0 then exit;

     if longest < 0 then
     begin
          right_v[0] := v3;
          right_v[1] := v2;
          right_v[2] := v1;
          right_n := 2;
          left_v[0] := v3;
          left_v[1] := v1;
          left_n := 1;

          dummy := DoLeftSide;
          if dummy <= 0 then exit;
          leftdu := (v3^.u - v1^.u) div dummy;
          leftu := v1^.u;
          leftdv := (v3^.v - v1^.v) div dummy;
          leftv := v1^.v;

          if DoRightSide <= 0 then
          begin
               dec(right_n);
               if DoRightSide <= 0 then exit;
          end;

          if longest > -65536 then longest := -65536;
     end
     else
     begin
          left_v[0] := v3;
          left_v[1] := v2;
          left_v[2] := v1;
          left_n := 2;
          right_v[0] := v3;
          right_v[1] := v1;
          right_n := 1;

          if DoRightSide <= 0 then exit;

          dummy := DoLeftSide;
          if dummy <= 0 then
          begin
               dec(left_n);
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftdu := (v3^.u - v2^.u) div dummy;
               leftu := v2^.u;
               leftdv := (v3^.v - v2^.v) div dummy;
               leftv := v2^.v;
          end
          else
          begin
               leftdu := (v2^.u - v1^.u) div dummy;
               leftu := v1^.u;
               leftdv := (v2^.v - v1^.v) div dummy;
               leftv := v1^.v;
          end;

          if longest < 65536 then longest := 65536;
     end;

     dudx := FDiv(FMul(temp, v3^.u - v1^.u) + v1^.u - v2^.u, longest);
     dvdx := FDiv(FMul(temp, v3^.v - v1^.v) + v1^.v - v2^.v, longest);

     ScrOff := (v1^.y2d shr 16) * 320;
     asm
        mov ax, tSeg
        db $8E, $E0 { mov fs, ax }
     end;
     while true do
     begin
          x1 := left_x shr 16;
          width := (right_x shr 16) - x1;
          if width > 0 then
          begin
               u := leftu shr 8;
               v := leftv shr 8;
               du := dudx shr 8;
               dv := dvdx shr 8;

               asm
                  mov es, where
                  mov di, scroff
                  db long; mov si, word ptr [v]
                  db long; mov dx, word ptr [u]
                  add di, x1
                  mov cx, width
               @loop1:
                       mov bx, si
                       db long; add si, word ptr [dv]
                       mov bl, dh
                       db long; add dx, word ptr [du]
                       db $64, $8A, $07 { mov al, fs:[bx] }
                       mov es:[di], al
                       inc di
                       dec cx
                       jne @loop1
               end;
          end;
          inc(ScrOff, 320);

          dec(left_height);
          if left_height <= 0 then
          begin
               dec(left_n);
               if left_n <= 0 then exit;
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftdu := (v3^.u - v2^.u) div dummy;
               leftu := v2^.u;
               leftdv := (v3^.v - v2^.v) div dummy;
               leftv := v2^.v;
          end
          else
          begin
               inc(left_x, left_dx);
               inc(leftu, leftdu);
               inc(leftv, leftdv);
          end;

          dec(right_height);
          if right_height <= 0 then
          begin
               dec(right_n);
               if right_n <= 0 then exit;
               if DoRightSide <= 0 then exit;
          end
          else inc(right_x, right_dx);
     end;
end;


{ *** Tringulo Phong con mapeo de textura *** }
procedure PhongTextureTriangle(vertex1, vertex2, vertex3 : PTVertex; tSeg, where : word);
var v1, v2, v3, v0 : PTVertex;
    height, dummy : integer;
    temp, longest : Fixed;
    leftu, leftdu, leftv, leftdv, dudx, dvdx : Fixed;
    lefta, leftda, dadx : Fixed;
    ScrOff, dest : word;
    plane : byte;
    x1, width : integer;
    u, v, du, dv, a, da, a1, a2, a3 : Fixed;
begin
     v1 := vertex1;
     v2 := vertex2;
     v3 := vertex3;
     if v1^.y2d > v2^.y2d then begin v0 := v1; v1 := v2; v2 := v0; end;
     if v1^.y2d > v3^.y2d then begin v0 := v1; v1 := v3; v3 := v0; end;
     if v2^.y2d > v3^.y2d then begin v0 := v2; v2 := v3; v3 := v0; end;

     height := (v3^.y2d - v1^.y2d) shr 16;
     if height = 0 then exit;
     temp := (v2^.y2d - v1^.y2d) div height;
     longest := FMul(temp, v3^.x2d - v1^.x2d) + v1^.x2d - v2^.x2d;
     if longest = 0 then exit;

     a1 := ArcCos[( FMul(v1^.normX, Light.X) + FMul(v1^.normY, Light.Y) +
                    FMul(v1^.normZ, Light.Z)) div 512];
     a2 := ArcCos[( FMul(v2^.normX, Light.X) + FMul(v2^.normY, Light.Y) +
                    FMul(v2^.normZ, Light.Z)) div 512];
     a3 := ArcCos[( FMul(v3^.normX, Light.X) + FMul(v3^.normY, Light.Y) +
                    FMul(v3^.normZ, Light.Z)) div 512];


     if longest < 0 then
     begin
          right_v[0] := v3;
          right_v[1] := v2;
          right_v[2] := v1;
          right_n := 2;
          left_v[0] := v3;
          left_v[1] := v1;
          left_n := 1;

          dummy := DoLeftSide;
          if dummy <= 0 then exit;
          leftdu := (v3^.u - v1^.u) div dummy;
          leftu := v1^.u;
          leftdv := (v3^.v - v1^.v) div dummy;
          leftv := v1^.v;
          leftda := (a3 - a1) div dummy;
          lefta := a1;

          if DoRightSide <= 0 then
          begin
               dec(right_n);
               if DoRightSide <= 0 then exit;
          end;

          if longest > -65536 then longest := -65536;
     end
     else
     begin
          left_v[0] := v3;
          left_v[1] := v2;
          left_v[2] := v1;
          left_n := 2;
          right_v[0] := v3;
          right_v[1] := v1;
          right_n := 1;

          if DoRightSide <= 0 then exit;

          dummy := DoLeftSide;
          if dummy <= 0 then
          begin
               dec(left_n);
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftdu := (v3^.u - v2^.u) div dummy;
               leftu := v2^.u;
               leftdv := (v3^.v - v2^.v) div dummy;
               leftv := v2^.v;
               leftda := (a3 - a2) div dummy;
               lefta := a2;
          end
          else
          begin
               leftdu := (v2^.u - v1^.u) div dummy;
               leftu := v1^.u;
               leftdv := (v2^.v - v1^.v) div dummy;
               leftv := v1^.v;
               leftda := (a2 - a1) div dummy;
               lefta := a1;
          end;

          if longest < 65536 then longest := 65536;
     end;

     dudx := FDiv(FMul(temp, v3^.u - v1^.u) + v1^.u - v2^.u, longest);
     dvdx := FDiv(FMul(temp, v3^.v - v1^.v) + v1^.v - v2^.v, longest);
     dadx := FDiv(FMul(temp, a3 - a1) + a1 - a2, longest);

     ScrOff := (v1^.y2d shr 16) * 320;
     asm
        mov ax, tSeg
        db $8E, $E0 { mov fs, ax }
     end;
     while true do
     begin
          x1 := left_x shr 16;
          width := (right_x shr 16) - x1;
          if width > 0 then
          begin
               u := leftu shr 8;
               v := leftv shr 8;
               du := dudx shr 8;
               dv := dvdx shr 8;
               da := dadx;
               a := lefta - dadx;

               asm
                  mov es, where
                  mov di, scroff
                  db long; mov si, word ptr [v]
                  db long; mov dx, word ptr [u]
                  add di, x1
                  mov cx, width
               @loop1:
                       mov bx, si
                       db long; add si, word ptr [dv]
                       mov bl, dh
                       db long; add dx, word ptr [du]
                       db $64, $8A, $07 { mov al, fs:[bx] }

                       db long; mov bx, word ptr [a]
                       db long; add bx, word ptr [da]
                       db long; mov word ptr [a], bx
                       db long; shr bx, 15
                       add al, byte ptr [PhongCol + bx]

                       mov es:[di], al
                       inc di
                       dec cx
                       jne @loop1
               end;
          end;
          inc(ScrOff, 320);

          dec(left_height);
          if left_height <= 0 then
          begin
               dec(left_n);
               if left_n <= 0 then exit;
               dummy := DoLeftSide;
               if dummy <= 0 then exit;
               leftdu := (v3^.u - v2^.u) div dummy;
               leftu := v2^.u;
               leftdv := (v3^.v - v2^.v) div dummy;
               leftv := v2^.v;
               leftda := (a3 - a2) div dummy;
               lefta := a2;
          end
          else
          begin
               inc(left_x, left_dx);
               inc(leftu, leftdu);
               inc(leftv, leftdv);
               inc(lefta, leftda);
          end;

          dec(right_height);
          if right_height <= 0 then
          begin
               dec(right_n);
               if right_n <= 0 then exit;
               if DoRightSide <= 0 then exit;
          end
          else inc(right_x, right_dx);
     end;
end;



{ Arbol de polgonos ordenados }
type PTTreeNode = ^TTreeNode;
     TTreeNode = object
               private
                      poly : PTPolyList;
                      Zorder : Fixed;
                      left, right : PTTreeNode;
                      constructor Init(p : PTPolyList; z : Fixed);
                      destructor Done;
                      procedure Draw(where : word);
               end;

     PTTree = ^TTree;
     TTree = object
           private
                  root : PTTreeNode;
                  constructor Init;
                  destructor Done;
                  procedure AddNode(p : PTPolyList);
                  procedure Draw(where : word);
           end;

     { Implementacin de TTreeNode }
     constructor TTreeNode.Init(p : PTPolyList; z : Fixed);
     begin
          poly := p;
          left := nil;
          right := nil;
          Zorder := z;
     end;

     destructor TTreeNode.Done;
     begin
          if left <> nil then dispose(left, Done);
          if right <> nil then dispose(right, Done);
     end;

     procedure TTreeNode.Draw(where : word);
     begin
          if left <> nil then left^.Draw(where);
          poly^.Draw(where);
          if right <> nil then right^.Draw(where);
     end;

     { Implementacin de TTree }
     constructor TTree.Init;
     begin
          root := nil;
     end;

     destructor TTree.Done;
     begin
          if root <> nil then dispose(root, Done);
     end;

     procedure TTree.AddNode(p : PTPolyList);
     var temp : PTTreeNode;
         flag : boolean;
         Zorder : Fixed;
     begin
          temp := root;
          flag := true;
          Zorder := p^.Zorder;

          if temp = nil then root := new(PTTreeNode, init(p, Zorder))
          else
          while flag do
                if Zorder <= temp^.Zorder then
                   if temp^.left = nil then
                   begin
                        temp^.left := new(PTTreeNode, init(p, Zorder));
                        flag := false;
                   end
                   else temp := temp^.left
                else
                    if temp^.right = nil then
                    begin
                         temp^.right := new(PTTreeNode, init(p, Zorder));
                         flag := false;
                    end
                    else temp := temp^.right;
     end;

     procedure TTree.Draw(where : word);
     begin
          if root <> nil then root^.Draw(where);
     end;


{ Implementacin de TFrame }
procedure TFrame.Reset;
begin
     SetFrame(0, 0, 319, 199);
     Center;
end;

procedure TFrame.SetFrame(x1, y1, x2, y2 : word);
begin
     Xmin := Fixed(x1) shl 16;
     Ymin := Fixed(y1) shl 16;
     Xmax := Fixed(x2) shl 16;
     Ymax := Fixed(y2) shl 16;
end;

procedure TFrame.SetCenter(x1, y1 : word);
begin
     Xcenter := Fixed(x1) shl 16;
     Ycenter := Fixed(y1) shl 16;
end;

procedure TFrame.Center;
begin
     Xcenter := (Xmin + Xmax + 65536) div 2;
     Ycenter := (Ymin + Ymax + 65536) div 2;
end;


{ Implementacin de TVector }
constructor TVector.Init(nx, ny, nz : integer);
begin
     SetP(nx, ny, nz);
     Zoff := 0;
end;

procedure TVector.SetP(nx, ny, nz : integer);
begin
     x := Fixed(nx) shl 16;
     y := Fixed(ny) shl 16;
     z := Fixed(nz) shl 16;
end;

procedure TVector.SetPSingle(nx, ny, nz : single);
begin
     x := Single2Fixed(nx);
     y := Single2Fixed(ny);
     z := Single2Fixed(nz);
end;

procedure TVector.GetP(var nx, ny, nz : integer);
begin
     nx := RFixed2Int(x);
     ny := RFixed2Int(y);
     nz := RFixed2Int(z);
end;

procedure TVector.VRotate(ax, ay, az : integer);
var nx, ny, nz : Fixed;     { nuevas coordenadas }

begin
     if ax < 0 then ax := ax + 360;
     if ay < 0 then ay := ay + 360;
     if az < 0 then az := az + 360;

     if (ax > 0) and (ax < 360) then
     begin              { rotacin sobre el eje X }
          ny := FMul(y, cost[ax]) - FMul(z, sint[ax]);
          nz := FMul(y, sint[ax]) + FMul(z, cost[ax]);
          y := ny;
          z := nz;
     end;

     if (ay > 0) and (ay < 360) then
     begin             { rotacin sobre el eje Y }
          nx := FMul(x, cost[ay]) - FMul(z, sint[ay]);
          nz := FMul(x, sint[ay]) + FMul(z, cost[ay]);
          x := nx;
          z := nz;
     end;

     if (az > 0) and (az < 360) then
     begin            { rotacin sobre el eje Z }
          nx := FMul(x, cost[az]) - FMul(y, sint[az]);
          ny := FMul(x, sint[az]) + FMul(y, cost[az]);
          x := nx;
          y := ny;
     end;
end;

function TVector.FMagnitude : longint;
begin
     FMagnitude := FHSqrt(FSquare(x) + FSquare(y) + FSquare(z));
end;

procedure TVector.Calc2d;
var p : Fixed;
begin
     p := 67108864 - Zoff - z;  { Int2Fixed(1024) - Zoff - Z }
     X2d := (FDiv(x, p) shl 9 + Frame.XCenter) and $FFFF0000;
     Y2d := (FDiv(y, -p) shl 9 + Frame.YCenter) and $FFFF0000;
end;


{ Implementacin de TVertex }
constructor TVertex.Init(nx, ny, nz : integer);
begin
     inherited Init(nx, ny, nz);
     normX := 0;
     normY := 0;
     normZ := 0;
end;

procedure TVertex.Rotate(ax, ay, az : integer);
var nx, ny, nz : Fixed;     { nuevas coordenadas del vrtice }

begin
     if ax < 0 then ax := ax + 360;
     if ay < 0 then ay := ay + 360;
     if az < 0 then az := az + 360;

     if (ax > 0) and (ax < 360) then
     begin              { rotacin sobre el eje X }
          ny := FMul(y, cost[ax]) - FMul(z, sint[ax]);
          nz := FMul(y, sint[ax]) + FMul(z, cost[ax]);
          y := ny;
          z := nz;
          ny := FMul(normY, cost[ax]) - FMul(normZ, sint[ax]);
          nz := FMul(normY, sint[ax]) + FMul(normZ, cost[ax]);
          normY := ny;
          normZ := nz;
     end;

     if (ay > 0) and (ay < 360) then
     begin             { rotacin sobre el eje Y }
          nx := FMul(x, cost[ay]) - FMul(z, sint[ay]);
          nz := FMul(x, sint[ay]) + FMul(z, cost[ay]);
          x := nx;
          z := nz;
          nx := FMul(normX, cost[ay]) - FMul(normZ, sint[ay]);
          nz := FMul(normX, sint[ay]) + FMul(normZ, cost[ay]);
          normX := nx;
          normZ := nz;
     end;

     if (az > 0) and (az < 360) then
     begin            { rotacin sobre el eje Z }
          nx := FMul(x, cost[az]) - FMul(y, sint[az]);
          ny := FMul(x, sint[az]) + FMul(y, cost[az]);
          x := nx;
          y := ny;
          nx := FMul(normX, cost[az]) - FMul(normY, sint[az]);
          ny := FMul(normX, sint[az]) + FMul(normY, cost[az]);
          normX := nx;
          normY := ny;
     end;
end;

procedure TVertex.Normalize;
var m : Fixed;
begin
     m := FHSqrt(FSquare(normX) + FSquare(normY) + FSquare(normZ));
     if m = 0 then exit;
     normX := FDiv(normX, m);
     normY := FDiv(normY, m);
     normZ := FDiv(normZ, m);
end;

procedure TVertex.CalcUV;
begin
     u := (normX + 65536) shl 7;
     v := (normY + 65536) shl 7 - normY - 65536;
end;

procedure TVertex.SetUV(n : byte);
begin
     case n of
          0 : begin
                   u := FHSqrt(FSquare(normX) + FSquare(normZ)) * 255;
                   v := FHSqrt(FSquare(normY) + FSquare(normZ)) * 254;
              end;
          1 : begin
                   u := (FMul(normX, normZ) + 65536) * 128;
                   v := (FMul(normY, normZ) + 65536) * 127;
              end;
          2 : begin
                   u := (FMul(normX, Light.Z) + 65536) * 128;
                   v := (FMul(normY, Light.Z) + 65536) * 127;
              end;
          3 : begin
                   u := (normX + 65536) shl 7;
                   v := (normY + 65536) shl 7 - normY - 65536;
              end;
     end;
end;

{ Implementacin de TPoly }
constructor TPoly.Init;
begin
     Vnum := 0;
     colStart := 0;
     colWidth := 0;
     Style := NoShading;
     normX := 0;
     normY := 0;
     normZ := 0;
end;

procedure TPoly.AddVertex(v : PTVertex);
begin
     if Vnum < MaxVnum then
     begin
          inc(Vnum);
          Vertex[Vnum] := v;
     end;
end;

procedure TPoly.Draw(where : word);
var r, v, w : byte; { quitar v, w }
    c, c1, c2, c3 : Fixed;
    a1, a2, a3 : Fixed;

begin
     if Vnum < 3 then exit;

     case Style of

          NoShading : FlatTriangle(Vertex[1], Vertex[2], Vertex[3], ColStart, where);

          WireFrame : begin
                           Line(Vertex[1]^.X2d shr 16,
                                Vertex[1]^.Y2d shr 16,
                                Vertex[2]^.X2d shr 16,
                                Vertex[2]^.Y2d shr 16, ColStart, where);
                           Line(Vertex[2]^.X2d shr 16,
                                Vertex[2]^.Y2d shr 16,
                                Vertex[3]^.X2d shr 16,
                                Vertex[3]^.Y2d shr 16, ColStart, where);
                           Line(Vertex[3]^.X2d shr 16,
                                Vertex[3]^.Y2d shr 16,
                                Vertex[1]^.X2d shr 16,
                                Vertex[1]^.Y2d shr 16, ColStart, where);
                      end;


          PhongShading : PhongTriangle(Vertex[1], Vertex[2], Vertex[3], where);

          TextureMap : begin
                            Vertex[1]^.U := 0;
                            Vertex[1]^.V := 0;
                            Vertex[2]^.U := 16711680;
                            Vertex[2]^.V := 0;
                            Vertex[3]^.U := 16711680;
                            Vertex[3]^.V := 16646144;
                            PhongTextureTriangle( Vertex[1], Vertex[2],
                                             Vertex[3], texSeg, where);
                       end;

          EnviromentMap,
          TextureWrap : TextureTriangle(Vertex[1], Vertex[2], Vertex[3],
                                        texSeg, where);

     end;
end;


procedure TPoly.CalcNormal;
var ax, ay, az, bx, by, bz, m : Fixed;
begin
     if Vnum < 3 then exit;
     ax := Vertex[2]^.x - Vertex[1]^.x;
     ay := Vertex[2]^.y - Vertex[1]^.y;
     az := Vertex[2]^.z - Vertex[1]^.z;
     bx := Vertex[3]^.x - Vertex[1]^.x;
     by := Vertex[3]^.y - Vertex[1]^.y;
     bz := Vertex[3]^.z - Vertex[1]^.z;
     normX := FMul(ay, bz) - FMul(by, az);
     normY := FMul(bx, az) - FMul(ax, bz);
     normZ := FMul(ax, by) - FMul(bx, ay);
     normX := normX div 256;
     normY := normY div 256;
     normZ := normZ div 256;
     m := FHSqrt(FSquare(normX) + FSquare(normY) + FSquare(normZ));
     if m = 0 then exit;
     normX := FDiv(normX, m);
     normY := FDiv(normY, m);
     normZ := FDiv(normZ, m);
end;

procedure TPoly.RotateNormal(ax, ay, az : integer);
var nx, ny, nz : Fixed;     { nuevas coordenadas }

begin
     if ax < 0 then ax := ax + 360;
     if ay < 0 then ay := ay + 360;
     if az < 0 then az := az + 360;

     if (ax > 0) and (ax < 360) then
     begin              { rotacin sobre el eje X }
          ny := FMul(normY, cost[ax]) - FMul(normZ, sint[ax]);
          nz := FMul(normY, sint[ax]) + FMul(normZ, cost[ax]);
          normY := ny;
          normZ := nz;
     end;

     if (ay > 0) and (ay < 360) then
     begin             { rotacin sobre el eje Y }
          nx := FMul(normX, cost[ay]) - FMul(normZ, sint[ay]);
          nz := FMul(normX, sint[ay]) + FMul(normZ, cost[ay]);
          normX := nx;
          normZ := nz;
     end;

     if (az > 0) and (az < 360) then
     begin            { rotacin sobre el eje Z }
          nx := FMul(normX, cost[az]) - FMul(normY, sint[az]);
          ny := FMul(normX, sint[az]) + FMul(normY, cost[az]);
          normX := nx;
          normY := ny;
     end;
end;

procedure TPoly.Rotate(ax, ay, az : integer);
var v : byte;
begin
     RotateNormal(ax, ay, az);
     for v := 1 to Vnum do Vertex[v]^.Rotate(ax, ay, az);
end;

procedure TPoly.Scale(sx, sy, sz : single);
var v : byte;
begin
     if (sx = 0) or (sy = 0) or (sz = 0) then exit;
     normX := round(normX / sx);
     normY := round(normY / sy);
     normZ := round(normZ / sz);
     for v := 1 to Vnum do
         with Vertex[v]^ do
         begin
              x := round(x * sx);
              y := round(y * sy);
              z := round(z * sz);
         end;
end;

function TPoly.Zorder : Fixed;
var v : byte;
    z : Fixed;
begin
     z := 0;
     for v := 1 to Vnum do inc(z, Vertex[v]^.z);
     Zorder := z;
end;

{function TPoly.Visible : boolean;
begin
     if normZ >= -2048 then Visible := true else Visible := false;
end;}

function TPoly.Visible : boolean;
begin
     if (FMul(Vertex[2]^.x - Vertex[1]^.x, Vertex[3]^.y - Vertex[1]^.y) -
         FMul(Vertex[3]^.x - Vertex[1]^.x, Vertex[2]^.y - Vertex[1]^.y)) > 0
         then Visible := true else Visible := false;
end;

{ Implementacin de TPolyList }
constructor TPolyList.Init;
begin
     inherited Init;
     next := nil;
end;

destructor TPolyList.Done;
begin
     if next <> nil then dispose(next, Done);
     next := nil;
end;


{ Implementacin de TSolid }
constructor TSolid.Init;
begin
     Vnum := 0;
     Pnum := 0;
     ox := 0;
     oy := 0;
     oz := 0;
     Style := NoShading;
     HideFaces := true;
     ColStart := 0;
     ColWidth := 0;
     PFirst := nil;
end;

destructor TSolid.Done;
var v : word;
begin
     if PFirst <> nil then dispose(PFirst, Done);
     PFirst := nil;
     for v := 1 to Vnum do dispose(Vertex[v]);
     Vnum := 0;
     Pnum := 0;
end;

procedure TSolid.AddVertex(vx, vy, vz : integer);
var v : PTVertex;
begin
     if Vnum < MaxVertex then
     begin
          v := new(PTVertex, Init(vx, vy, vz));
          inc(Vnum);
          Vertex[Vnum] := v;
     end;
end;

procedure TSolid.AddPoly(poly : PTPolyList);
var temp : PTPolyList;
begin
     if Pnum = MaxPoly then exit;
     temp := Pfirst;
     if temp <> nil then
     begin
          while temp^.next <> nil do temp := temp^.next;
          temp^.next := poly;
     end
     else Pfirst := poly;
     inc(Pnum);
end;

procedure TSolid.CalcNormals;
var p : PTPolyList;
    v, vv : word;
begin
     p := Pfirst;
     while p <> nil do
     begin
          p^.CalcNormal;
          p := p^.next;
     end;

     for v := 1 to Vnum do
     begin
          Vertex[v]^.normX := 0;
          Vertex[v]^.normY := 0;
          Vertex[v]^.normZ := 0;
          p := Pfirst;
          while p <> nil do
          begin
               for vv := 1 to p^.Vnum do
                   if Vertex[v] = p^.Vertex[vv] then
                   begin
                        Vertex[v]^.normX := Vertex[v]^.normX + p^.normX;
                        Vertex[v]^.normY := Vertex[v]^.normY + p^.normY;
                        Vertex[v]^.normZ := Vertex[v]^.normZ + p^.normZ;
                   end;
               p := p^.next;
          end;
          Vertex[v]^.Normalize;
     end;
end;

procedure TSolid.ReCalcUV;
var i : word;
begin
     for i := 1 to VNum do Vertex[i]^.CalcUV;
end;

procedure TSolid.SetUV(n : byte);
var i : word;
begin
     for i := 1 to Vnum do Vertex[i]^.SetUV(n);
end;

procedure TSolid.SetOrigin(nx, ny, nz : integer);
begin
     ox := Fixed(nx) shl 16;
     oy := Fixed(ny) shl 16;
     oz := Fixed(nz) shl 16;
end;

procedure TSolid.GetOrigin(var nx, ny, nz : integer);
begin
     nx := Fixed2Int(ox);
     ny := Fixed2Int(oy);
     nz := Fixed2Int(oz);
end;

procedure TSolid.SetColor(cstart, cwidth : byte);
var p : PTPolyList;
begin
     ColStart := cstart;
     ColWidth := cwidth;
     p := Pfirst;
     while p <> nil do
     begin
          p^.ColStart := cstart;
          p^.ColWidth := cwidth;
          p^.MaxColor := cstart + cwidth;
          p := p^.next;
     end;
end;

procedure TSolid.SetStyle(ns : byte);
var p : PTPolyList;
begin
     Style := ns;
     p := Pfirst;
     while p <> nil do
     begin
          p^.Style := ns;
          p := p^.next;
     end;
end;

procedure TSolid.SetTexture(t : PTTexture; tcwidth : byte);
var p : PTPolyList;
begin
     TexSeg := seg(t^);
     TexOff := ofs(t^);
     TColWidth := tcwidth;
     p := Pfirst;
     while p <> nil do
     begin
          p^.TexSeg := TexSeg;
          p^.TexOff := TexOff;
          p^.TColWidth := tcwidth;
          p := p^.next;
     end;
end;

function TSolid.CountVertex : word;
begin
     CountVertex := Vnum;
end;

function TSolid.CountPoly : word;
begin
     CountPoly := Pnum;
end;

procedure TSolid.MakePhongTable;
var value : single;
    i : integer;
begin
     for i := 0 to 360 do
     begin
          value := (cos(i * Pi / 360.0) + 1) * (ColWidth div 2);
          if Style = TextureMap then value := trunc(value) * TColWidth;
          PhongCol[i] := trunc(value) + ColStart;
     end;
end;


procedure TSolid.Draw;
var tree : PTTree;
    p : PTPolyList;
    v : word;
begin
     for v := 1 to Vnum do
         with Vertex[v]^ do
         begin
              Zoff := oz;
              Calc2d;
              X2d := X2d + ox;
              Y2d := Y2d - oy;
              if X2d < Frame.Xmin then X2d := Frame.Xmin
                 else if X2d > Frame.Xmax then X2d := Frame.Xmax;
              if Y2d < Frame.Ymin then Y2d := Frame.Ymin
                 else if Y2d > Frame.Ymax then Y2d := Frame.Ymax;
         end;

     if Style = EnviromentMap then for v := 1 to Vnum do Vertex[v]^.CalcUV;

     tree := new(PTTree, Init);
     p := Pfirst;
     while p <> nil do
     begin
{          if (not HideFaces) or p^.Visible then tree^.AddNode(p);}
           if p^.Visible then tree^.AddNode(p);
          p := p^.next;
     end;
     tree^.Draw(where);
     dispose(tree, Done);
end;

procedure TSolid.Rotate(ax, ay, az : integer);
var v : word;
{    p : PTPolyList;}
begin
     for v := 1 to Vnum do Vertex[v]^.Rotate(ax, ay, az);
{     p := Pfirst;
     while p <> nil do
     begin
          p^.RotateNormal(ax, ay, az);
          p := p^.next;
     end;}
end;

procedure TSolid.FastRot(ax, ay, az : integer);
var v : word;
begin
     for v := 1 to Vnum do Vertex[v]^.VRotate(ax, ay, az);
end;


procedure TSolid.Scale(sx, sy, sz : single);
var v : word;
begin
     for v := 1 to Vnum do
         with Vertex[v]^ do
         begin
              x := round(x * sx);
              y := round(y * sy);
              z := round(z * sz);
         end;
     CalcNormals;
end;

procedure TSolid.Load(filename : string);
var f : file of Fixed;
    i, j, nv, np : word;
    x, y, z, u, v : Fixed;
    p : PTPolyList;
begin
     assign(f, filename);
     reset(f);
     read(f, x); { Leer ID y versin }
     if (x and $1FAC0000) <> $1FAC0000 then { Comprobar ID }
     begin
          close(f); { Salir si ID es incorrecta }
          exit;
     end;
     if Pfirst <> nil then Done; { Destrur lo que haba antes }
     read(f, x);
     nv := x;     { Leer # de vrtices }
     read(f, x);
     np := x;     { Leer # de polgonos }
     for i := 1 to 5 do read(f, x); { Leer 5 DWORDS de basura }
     for i := 1 to nv do
     begin
          AddVertex(0, 0, 0);      { Aadir un vrtice }
          read(f, x, y, z, u, v);  { Leer su informacin y asignarla }
          Vertex[vnum]^.x := x;
          Vertex[vnum]^.y := y;
          Vertex[vnum]^.z := z;
          Vertex[vnum]^.u := u shl 8;
          Vertex[vnum]^.v := v * 255;
     end;
     for i := 1 to np do
     begin
          p := new(PTPolyList, Init); { Crear un polgono nuevo }
          read(f, u);                 { Leer su nmero de vrtices }
          for j := 1 to u do
          begin
               read(f, v);            { Leer el ndice del vrtice }
               inc(v);                { Que empiecen desde 1 }
               p^.AddVertex(Vertex[v]); { Aadir el vrtice al polgono }
          end;
          AddPoly(p);                   { Aadir el polgono al slido }
     end;
     close(f);
     CalcNormals;       { Calcular las normales }
end;

procedure TSolid.Save(filename : string);
{ Formato de archivo .VKX

          file of longint:

                 - Identificacin de archivo = 1FAC0100h
                        - High word = ID
                        - Low word  = Versin
                 - Nmero de vrtices en el slido
                 - Nmero de polgonos em el slido
                 - 5 dobles palabras reservadas (sin uso)
                 - Para cada vrtice:
                        - X (en punto fijo 16.16)
                        - Y (en punto fijo 16.16)
                        - Z (en punto fijo 16.16)
                        - U (en punto fijo y rango [0 - 1))
                        - V (en punto fijo y rango [0 - 1))
                 - Para cada polgono:
                        - Nmero de vrtices en el polgono
                        - Indices de los vrtices (empezando en 0)
                 - Fin de archivo;
}
        function VIndex(v : PTVertex) : word;
        var q : word;
        begin
             q := 1;
             while (Vertex[q] <> v) and (q <= MaxVertex) do inc(q);
             dec(q);
             VIndex := q;
        end;

var f : file of Fixed;
    i : word;
    a, u, v : Fixed;
    p : PTPolyList;
begin
     assign(f, filename);
     rewrite(f);
     a := $1FAC0100;
     write(f, a); { ID & Versin }
     a := Vnum;
     write(f, a); { Nmero de vrtices }
     a := Pnum;
     write(f, a); { Nmero de polgonos }
     a := 0;
     for i := 1 to 5 do write(f, a); { 5 DWORDS sin uso }

     for i := 1 to Vnum do
     begin
          write(f, Vertex[i]^.x, Vertex[i]^.y, Vertex[i]^.z);
          u := Vertex[i]^.u shr 8;
          v := Vertex[i]^.v div 255;
          write(f, u, v);
     end;

     p := Pfirst;
     while p <> nil do
     begin
          a := p^.Vnum;
          write(f, a);
          for i := 1 to p^.Vnum do
          begin
               a := Fixed(VIndex(p^.Vertex[i]));
               write(f, a);
          end;
          p := p^.next;
     end;
     close(f);
end;


{ Procedimientos para la fuente de iluminacin }
procedure MoveLight(x, y, z : integer);
var m, xx, yy, zz : Fixed;
begin
     xx := Fixed(x) shl 16;
     yy := Fixed(y) shl 16;
     zz := Fixed(z) shl 16;
     m := FHSqrt(FSquare(xx) + FSquare(yy) + FSquare(zz));
     if m = 0 then exit;
     Light.x := FDiv(xx, m);
     Light.y := FDiv(yy, m);
     Light.z := FDiv(zz, m);
end;

procedure RotateLight(ax, ay, az : integer);
var nx, ny, nz : Fixed;     { nuevas coordenadas }

begin
     if ax < 0 then ax := ax + 360;
     if ay < 0 then ay := ay + 360;
     if az < 0 then az := az + 360;

     if (ax > 0) and (ax < 360) then
     begin              { rotacin sobre el eje X }
          ny := FMul(Light.y, cost[ax]) - FMul(Light.z, sint[ax]);
          nz := FMul(Light.y, sint[ax]) + FMul(Light.z, cost[ax]);
          Light.y := ny;
          Light.z := nz;
     end;

     if (ay > 0) and (ay < 360) then
     begin             { rotacin sobre el eje Y }
          nx := FMul(Light.x, cost[ay]) - FMul(Light.z, sint[ay]);
          nz := FMul(Light.x, sint[ay]) + FMul(Light.z, cost[ay]);
          Light.x := nx;
          Light.z := nz;
     end;

     if (az > 0) and (az < 360) then
     begin            { rotacin sobre el eje Z }
          nx := FMul(Light.x, cost[az]) - FMul(Light.y, sint[az]);
          ny := FMul(Light.x, sint[az]) + FMul(Light.y, cost[az]);
          Light.x := nx;
          Light.y := ny;
     end;
end;


{ Texture procs }
procedure LoadTexture(fn : string; t : PTTexture; var p : TPalette;
                      ostart, oend : longint);
var f : file of byte;
    x : word;
    y, c, i, r, g, b : byte;
    flag : boolean;
function IncPos : boolean;
begin
     inc(x);
     if x = 256 then
     begin
          x := 0;
          inc(y);
     end;
     if y = 255 then IncPos := true else IncPos := false;
end;

begin
     assign(f, fn);
     reset(f);
     seek(f, ostart + 128);
     flag := false;
     x := 0;
     y := 0;
     while not flag do
     begin
          read(f, i);
          if (i and $C0) = $C0 then
          begin
               read(f, c);
               for i := 1 to (i and $3F) do
               begin
                    t^[y, x] := c;
                    flag := IncPos;
               end;
          end
          else
          begin
               t^[y, x] := i;
               flag := IncPos;
          end;
     end;
     seek(f, oend - 768);
     for i := 0 to 255 do
     begin
          read(f, r, g, b);
          p[i][0] := r div 4;
          p[i][1] := g div 4;
          p[i][2] := b div 4;
     end;
     close(f);
end;


{ Palette Prox }
procedure MakePhongPal( ra, rd, rs, ga, gd, gs, ba, bd, bs : single;
                        n : word; var p : TPalette; start, range : word);
var i, r, g, b : word;
    ang, angadd : single;
    dif, spec : single;

    function pow(base : single; expo : word) : single;
    var x : single;
        l : word;
    begin
         x := 1.0;
         if expo = 0 then expo := 1;
         for l := 1 to expo do x := x * base;
         pow := x;
    end;

begin
     if range > 256 then range := 256;
     ang := pi / 2;
     angadd := (pi / range) / 2;
     for i := 0 to (range - 1) do
     begin
          dif := cos(ang) * rd;
          spec := pow(cos(ang), n) * rs;
	  r := trunc((ra + dif + spec) / 4.0);
          if r > 63 then r := 63;

          dif := cos(ang) * gd;
          spec := pow(cos(ang), n) * gs;
	  g := trunc((ga + dif + spec) / 4.0);
          if g > 63 then g := 63;

	  dif := cos(ang) * bd;
          spec := pow(cos(ang), n) * bs;
	  b := trunc((ba + dif + spec) / 4.0);
          if b > 63 then b := 63;

	  p[start + i][0] := r;
          p[start + i][1] := g;
          p[start + i][2] := b;

	  ang := ang - angadd;
     end;
end;

procedure MakeTexturePal( ra, rs, ga, gs, ba, bs, m : single; n : word;
                          var p : TPalette; start, range : byte;
                          tpal : TPalette; twidth : byte);
var i, j : byte;
    r, g, b : word;
    ang, angadd : single;
    dif, spec : single;

    function pow(base : single; expo : word) : single;
    var x : single;
        l : word;
    begin
         x := 1.0;
         if expo = 0 then expo := 1;
         for l := 1 to expo do x := x * base;
         pow := x;
    end;

begin
     ang := pi / 2;
     angadd := (pi / range) / 2;
     if (range * twidth) >= 256 then dec(range);
     for i := 0 to (range - 1) do
     begin
          for j := 0 to (twidth - 1) do
          begin
               dif := cos(ang) * tpal[j][0] * m;
               spec := pow(cos(ang), n) * rs;
	       r := trunc((ra + dif + spec) / 4.0);
               if r > 63 then r := 63;

               dif := cos(ang) * tpal[j][1] * m;
               spec := pow(cos(ang), n) * gs;
	       g := trunc((ga + dif + spec) / 4.0);
               if g > 63 then g := 63;

	       dif := cos(ang) * tpal[j][2] * m;
               spec := pow(cos(ang), n) * bs;
	       b := trunc((ba + dif + spec) / 4.0);
               if b > 63 then b := 63;

	       p[start + (i * twidth) + j][0] := r;
               p[start + (i * twidth) + j][1] := g;
               p[start + (i * twidth) + j][2] := b;
          end;
	  ang := ang - angadd;
     end;
end;


{ Inicializacin }
begin
     CalcTrigTables;
     Frame.SetFrame(0, 0, 0, 0);
     Frame.Center;
     MoveLight(0, 0, 1);
end.