/*
        JiNX Sound System V1.0f - Windows demo programme

        (C) 1997-98, Mark Hodson (Bulk Sound System Code, IT Player, Drivers)
                     Saxon Druce (Windows Port)
                     Chris Bevan (Sound Output Code Prototype)
                     Jeffrey Lim (IT2.14+ Sample Compression Code)
*/

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <dsound.h>
#include <time.h>
#include <string.h>

#include "resource.h"

#include "jss_win.h"

// *******
// DEFINES
// *******

#define DM_UPDATE_OUTPUT (WM_USER + 0x100)

#define SAMPLE_SIZE 53053

// *****
// TYPES
// *****

typedef struct
{
	int valid;
	
	int order;
	int pattern;
	int row;
	int active1,active2;
	float occupancy;
} demo_output;

// *******
// GLOBALS
// *******

JTM_module *mod;
int bits_per_sample;
int channels;
int frequency;
int virtual_channels;
int reverb_delay;
int reverb_level;
int device;

demo_output DemOut[4];

unsigned char *test_sample;
int pcm_loaded=0;

int low_pass_frequency;
int high_pass_frequency;
int reverb_on;

// **********
// PROTOTYPES
// **********

BOOL CALLBACK DLG_Setup(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
// Callback function for the setup dialog box.

BOOL CALLBACK DLG_Control(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
// Callback function for the main runtime dialog box.

int speed_test(void);
// Returns an integer count based on amount of CPU available.

void __cdecl etrace(char *format, ... );
// Displays a message in an error dialog, then quits.

// *********
// FUNCTIONS
// *********

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;
	HWND hDlg,hDlgControl;
	OPENFILENAME of;
	
	char filename[256];
	int type;
	int i;
	int this_count, fastest_count, active;
	int total_active=0, active_samples=0;

	// show the 'please wait' dialog
	hDlg=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_SPEED_TEST),NULL,NULL);
	if (hDlg==NULL) etrace("Unable to create dialog box");
	ShowWindow(hDlg,SW_SHOW);
	while (PeekMessage(&msg, hDlg, 0, 0, PM_REMOVE))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	// setup vars for demo
	fastest_count=0;
	for (i=0; i<8; i++) fastest_count += speed_test();
	fastest_count = (fastest_count + 7)>>3;
	
	for (i=0; i<4; i++) DemOut[i].valid=0;
	
	// search for available sound hardware
	if (!JiNX_SS_EnumerateSoundDevices())
		etrace("JiNX_SS_EnumerateSoundDevices() failed -\n%s",JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);

	// kill 'please wait' dialog
	DestroyWindow(hDlg);

	// show the setup dialog box
	if (!DialogBox(hInstance,MAKEINTRESOURCE(IDD_SETUP),NULL,(DLGPROC)DLG_Setup))
		exit(1); // selected cancel

	if (bits_per_sample==8)
	{
		if (channels==1) type=0;
		else type=2;
	}
	else
	{
		if (channels==1) type=1;
		else type=3;
	}
	
	// create runtime control dialog box
	hDlgControl=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_CONTROL),NULL,(DLGPROC)DLG_Control);
	if (hDlgControl==NULL) etrace("Unable to create dialog box");
	ShowWindow(hDlgControl,SW_SHOW);
	
	// set chosen sound device
	JiNX_SS_SetDevice(device);

	// setup reverb options
	if (!JiNX_SS_PreInitSoundSystem(500,reverb_delay,reverb_level,0))
		etrace("JiNX_SS_InitSoundSystem() failed -\n%s",JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);

	// initialise JSS
	if (!JiNX_SS_InitSoundSystem(hDlgControl,type,frequency,virtual_channels))
		etrace("JiNX_SS_InitSoundSystem() failed -\n%s",JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);

	// install the timer
	JiNX_SS_InstallAccurateTimer();

	// get name of module
	of.lStructSize=sizeof(OPENFILENAME);
	of.hwndOwner=hDlgControl;
	of.hInstance=hInstance;
	of.lpstrFilter="Impulse Tracker files (*.it)\0*.IT\0All files (*.*)\0*.*\0\0\0";
	of.lpstrCustomFilter=NULL;
	of.nFilterIndex=1;
	strcpy(filename,"*.it");
	of.lpstrFile=filename;
	of.nMaxFile=256;
	of.lpstrFileTitle=NULL;
	of.lpstrInitialDir=NULL;
	of.lpstrTitle="Select Impulse Tracker File";
	of.Flags=OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOCHANGEDIR;
	of.lpstrDefExt="IT";
	
	if (!GetOpenFileName(&of))
		exit(1); // selected cancel
	
	// load it
	if ((mod=JiNX_SS_LoadModuleFile(filename,0))==NULL)
		etrace("JiNX_SS_LoadModuleFile() failed -\n%s",JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
	
	// begin playback
	JiNX_SS_PlayModule(mod);
	
	// main loop
	for (;;)
	{
		// perform some calculations to find output values to send to the control dialog box
		
		this_count = speed_test();
		if (this_count > fastest_count) this_count = fastest_count;
		
		// Count the number of non-zero-volume active channels
		for (active=0, i=0; i<mod->last_virtual_channel; i++)
		{
			if ((mod->virtual_channel[i].active_status != 0) && 
				(mod->virtual_channel[i].final_volume != 0)) active++;
		}
		
		active_samples++; total_active += active;

		// shuffle current display info up
		DemOut[0]=DemOut[1];
		DemOut[1]=DemOut[2];
		DemOut[2]=DemOut[3];

		// put new info in the bottom entry
		DemOut[3].valid=1;
		DemOut[3].order=mod->current_order;
		DemOut[3].pattern=mod->current_pattern;
		DemOut[3].row=mod->current_row;
		DemOut[3].active1=active;
		DemOut[3].active2=mod->last_virtual_channel;
		DemOut[3].occupancy=100.0f - ((100.0f * this_count) / fastest_count);
		
		// send a message to the dialog telling it to update it's output
		PostMessage(hDlgControl,DM_UPDATE_OUTPUT,0,0);
		
		// check for windows messages
		while (PeekMessage(&msg,hDlgControl,0,0,PM_NOREMOVE))
		{
			if (GetMessage(&msg,hDlgControl,0,0)==0)
			{
				// received WM_QUIT

				// uninstall timer
				JiNX_SS_UninstallAccurateTimer();

				// close JSS
				if (!JiNX_SS_CloseSoundSystem())
					etrace("JiNX_SS_CloseSoundSystem() failed -\n%s",JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
				
				// free module's memory
				JiNX_SS_UnloadModule(mod);

				// close dialog box
				DestroyWindow(hDlgControl);  

				return msg.wParam;
			}

			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	
	return 0;
}

BOOL CALLBACK DLG_Setup(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	char buf[200];
	int i;
	int num_devices;
	HWND focus;
	
	switch(msg)
	{
		case WM_INITDIALOG:

			CheckRadioButton(hDlg,IDC_8_BIT,IDC_16_BIT,IDC_16_BIT); // default 16 bit
			bits_per_sample=16;
			CheckRadioButton(hDlg,IDC_MONO,IDC_STEREO,IDC_STEREO); // default stereo
			channels=2;
			
			SetWindowText(GetDlgItem(hDlg,IDC_FREQUENCY),"44100"); // default 44100 Hz
			frequency=44100;
			
			SetWindowText(GetDlgItem(hDlg,IDC_VIRTUAL),"64"); // default 64 channels
			virtual_channels=64;

			SetWindowText(GetDlgItem(hDlg,IDC_REVERB_DELAY),"400"); // default 400 ms
			reverb_delay=400;
			
			SetWindowText(GetDlgItem(hDlg,IDC_REVERB_LEVEL),"2"); // default level 2
			reverb_level=2;

			// fill in available devices
			num_devices=JiNX_SS_ReturnNumberOfDevices();

			for (i=0; i<num_devices; i++)
				SendDlgItemMessage(hDlg,IDC_DEVICES,CB_ADDSTRING,0,(DWORD)JiNX_SS_DeviceDescription(i));
			
			if (num_devices>1)
				SendDlgItemMessage(hDlg,IDC_DEVICES,CB_SETCURSEL,1,0);
			else
				SendDlgItemMessage(hDlg,IDC_DEVICES,CB_SETCURSEL,0,0);

			SetFocus(GetDlgItem(hDlg,IDOK));

			return 1;
			break;
			
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDOK:
					// don't 'click OK' if pressed enter while editing an edit control
					focus=GetFocus();
					if (focus==GetDlgItem(hDlg,IDC_FREQUENCY) ||
						focus==GetDlgItem(hDlg,IDC_VIRTUAL) ||
						focus==GetDlgItem(hDlg,IDC_REVERB_DELAY) ||
						focus==GetDlgItem(hDlg,IDC_REVERB_LEVEL))
					{
						PostMessage(hDlg,WM_NEXTDLGCTL,0,0);
						return 1;
					}

					// find which device is selected
					device=SendDlgItemMessage(hDlg,IDC_DEVICES,CB_GETCURSEL,0,0);
					
					// clicked on OK button
					EndDialog(hDlg, 1);
					return 1;
					break;
					
				case IDCANCEL:
					// clicked on Cancel button
					EndDialog(hDlg, 0);
					return 1;
					break;
					
				case IDC_8_BIT:
					// clicked on 8bit radio button
					bits_per_sample=8;
					CheckRadioButton(hDlg,IDC_8_BIT,IDC_16_BIT,IDC_8_BIT);
					return 1;
				case IDC_16_BIT:
					// clicked on 16bit radio button
					bits_per_sample=16;
					CheckRadioButton(hDlg,IDC_8_BIT,IDC_16_BIT,IDC_16_BIT);
					return 1;
				case IDC_MONO:
					// clicked on mono radio button
					channels=1;
					CheckRadioButton(hDlg,IDC_MONO,IDC_STEREO,IDC_MONO);
					return 1;
				case IDC_STEREO:
					// clicked on stereo radio button
					channels=2;
					CheckRadioButton(hDlg,IDC_MONO,IDC_STEREO,IDC_STEREO);
					return 1;
					
				case IDC_FREQUENCY:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						if (GetFocus()==GetDlgItem(hDlg,IDCANCEL)) return 1; // pressed cancel, don't check
						
						// check that the frequency setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_FREQUENCY),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_FREQUENCY));
							return 1;
						}
						frequency=atoi(buf);
						if (frequency<1024 || frequency>100000)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n in the range 1024 to 100000","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_FREQUENCY));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
					break;
            
				case IDC_VIRTUAL:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						if (GetFocus()==GetDlgItem(hDlg,IDCANCEL)) return 1; // pressed cancel, don't check
						
						// check that the max virtual channels setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_VIRTUAL),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_VIRTUAL),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_VIRTUAL));
							return 1;
						}
						virtual_channels=atoi(buf);
						if (virtual_channels<1)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n greater than 0","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_VIRTUAL),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_VIRTUAL));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
					break;
            
				case IDC_REVERB_DELAY:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						if (GetFocus()==GetDlgItem(hDlg,IDCANCEL)) return 1; // pressed cancel, don't check
						
						// check that the frequency setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_REVERB_DELAY),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_REVERB_DELAY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_REVERB_DELAY));
							return 1;
						}
						reverb_delay=atoi(buf);
						if (reverb_delay<400 || reverb_delay>10000)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n in the range 400 to 10000","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_REVERB_DELAY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_REVERB_DELAY));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
					break;
            
				case IDC_REVERB_LEVEL:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						if (GetFocus()==GetDlgItem(hDlg,IDCANCEL)) return 1; // pressed cancel, don't check
						
						// check that the frequency setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_REVERB_LEVEL),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_REVERB_LEVEL),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_REVERB_LEVEL));
							return 1;
						}
						reverb_level=atoi(buf);
						if (reverb_level<1 || reverb_level>4)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n in the range 1 to 4","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_REVERB_LEVEL),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_REVERB_LEVEL));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
					break;
            
				default:
					break;
			}
			break;
		  
		default:
			return 0;
	}
  
	return 0;
}

BOOL CALLBACK DLG_Control(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	char buf[200];
	HWND htemp[4];
	int i;
	
	switch(msg)
	{
		case WM_INITDIALOG:
			SetWindowText(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY),"2000"); // default 2000 Hz
			low_pass_frequency=2000;
			SetWindowText(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY),"200"); // default 200 Hz
			high_pass_frequency=200;
			
			SetWindowText(GetDlgItem(hDlg,IDC_REVERB),"Reverb On");
			reverb_on=0;
			
			return 0; // didn't call SetFocus()
			break;
			
		case DM_UPDATE_OUTPUT:
			// update the table of output values
			
			// update orders
			htemp[0]=GetDlgItem(hDlg,IDC_ORDER1);
			htemp[1]=GetDlgItem(hDlg,IDC_ORDER2);
			htemp[2]=GetDlgItem(hDlg,IDC_ORDER3);
			htemp[3]=GetDlgItem(hDlg,IDC_ORDER4);
			for (i=0; i<4; i++)
			{
				if (DemOut[i].valid) sprintf(buf,"%0d",DemOut[i].order);
				else strcpy(buf,"-");
				SetWindowText(htemp[i],buf);
			}
			
			// update patterns
			htemp[0]=GetDlgItem(hDlg,IDC_PATTERN1);
			htemp[1]=GetDlgItem(hDlg,IDC_PATTERN2);
			htemp[2]=GetDlgItem(hDlg,IDC_PATTERN3);
			htemp[3]=GetDlgItem(hDlg,IDC_PATTERN4);
			for (i=0; i<4; i++)
			{
				if (DemOut[i].valid) sprintf(buf,"%0d",DemOut[i].pattern);
				else strcpy(buf,"-");
				SetWindowText(htemp[i],buf);
			}
			
			// update rows
			htemp[0]=GetDlgItem(hDlg,IDC_ROW1);
			htemp[1]=GetDlgItem(hDlg,IDC_ROW2);
			htemp[2]=GetDlgItem(hDlg,IDC_ROW3);
			htemp[3]=GetDlgItem(hDlg,IDC_ROW4);
			for (i=0; i<4; i++)
			{
				if (DemOut[i].valid) sprintf(buf,"%0d",DemOut[i].row);
				else strcpy(buf,"-");
				SetWindowText(htemp[i],buf);
			}
			
			// update active channels
			htemp[0]=GetDlgItem(hDlg,IDC_ACTIVE1);
			htemp[1]=GetDlgItem(hDlg,IDC_ACTIVE2);
			htemp[2]=GetDlgItem(hDlg,IDC_ACTIVE3);
			htemp[3]=GetDlgItem(hDlg,IDC_ACTIVE4);
			for (i=0; i<4; i++)
			{
				if (DemOut[i].valid) sprintf(buf,"%0d/%0d",DemOut[i].active1,DemOut[i].active2);
				else strcpy(buf,"-");
				SetWindowText(htemp[i],buf);
			}
			
			// update occupancy
			htemp[0]=GetDlgItem(hDlg,IDC_OCCUPANCY1);
			htemp[1]=GetDlgItem(hDlg,IDC_OCCUPANCY2);
			htemp[2]=GetDlgItem(hDlg,IDC_OCCUPANCY3);
			htemp[3]=GetDlgItem(hDlg,IDC_OCCUPANCY4);
			for (i=0; i<4; i++)
			{
				if (DemOut[i].valid) sprintf(buf,"%5.2f%%",DemOut[i].occupancy);
				else strcpy(buf,"-");
				SetWindowText(htemp[i],buf);
			}

			// update the current time
			sprintf(buf,"Current time: %6.2fs",JiNX_SS_WhatTimeIsIt());
			SetWindowText(GetDlgItem(hDlg,IDC_TIME),buf);
			
			return 1;
			break;
			
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDC_EXIT:
					PostQuitMessage(0);
					return 1;
					break;
					
				case IDC_PLAY:
					JiNX_SS_PlayModule(mod);
					// pressing play resets reverb, so:
					SetWindowText(GetDlgItem(hDlg,IDC_REVERB),"Reverb On");
					reverb_on=0;
					return 1;
					break;
					
				case IDC_STOP:
					JiNX_SS_StopModule();
					return 1;
					break;
					
				case IDC_SKIP_BACK:
					JiNX_SS_GotoOrderRow(mod->current_order-1, 0);
					return 1;
					break;
					
				case IDC_SKIP_FORWARD:
					JiNX_SS_GotoOrderRow(mod->current_order+1, 0);
					return 1;
					break;
					
				case IDC_INTERNAL:
					JiNX_SS_TriggerInternal(63, JiNX_SS_ConvertNoteString("C-5"), 1);
					return 1;
					break;
					
				case IDC_EXTERNAL:
					
					if (!pcm_loaded)
					{
						FILE *fptr;
						
						test_sample=(unsigned char *)malloc(SAMPLE_SIZE);
						if (test_sample == NULL) etrace("Insufficient memory");
						
						fptr = fopen("inject.pcm", "rb");
						if (fptr == NULL)
						{
							MessageBox(hDlg,"Someone stole Mulder!\n(INJECT.PCM missing...)","Error",MB_OK);
							return 1;
						}
						
						fread(test_sample,sizeof(char),SAMPLE_SIZE,fptr);

						fclose(fptr);
						
						pcm_loaded=1;
					}
					
					JiNX_SS_TriggerExternal(test_sample, 8, SAMPLE_SIZE, 22050, 256, 0);
					
					return 1;
					break;
					
				case IDC_LOW_PASS_FREQUENCY:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						// check that the frequency setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY));
							return 1;
						}
						low_pass_frequency=atoi(buf);
						if (low_pass_frequency<10 || low_pass_frequency>44100)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n in the range 10 to 44100","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_LOW_PASS_FREQUENCY));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
				case IDC_LOW_PASS_SET:
					JiNX_SS_LPfilterOn(low_pass_frequency);
					return 1;
					break;
					
				case IDC_LOW_PASS_OFF:
					JiNX_SS_LPfilterOff();
					return 1;
					break;
					
				case IDC_HIGH_PASS_FREQUENCY:
					if (HIWORD(wParam)==EN_KILLFOCUS)
					{
						// edit control has lost the focus
						
						// check that the frequency setting is valid
						GetWindowText(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY),buf,200);
						for (i=0; i<(int)strlen(buf); i++)
						{
							if (!isdigit(buf[i])) break;
						}
						if (i==0 || i!=(int)strlen(buf))
						{
							// not an integer
							MessageBox(hDlg,"Please enter an integer","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY));
							return 1;
						}
						high_pass_frequency=atoi(buf);
						if (high_pass_frequency<10 || high_pass_frequency>44100)
						{
							// not in range
							MessageBox(hDlg,"Please enter an integer\n in the range 10 to 44100","Invalid Data",MB_OK);
							PostMessage(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY),EM_SETSEL,0,-1);
							SetFocus(GetDlgItem(hDlg,IDC_HIGH_PASS_FREQUENCY));
							return 1;
						}                      
					}
					else
					{
						return 0;
					}
					
				case IDC_HIGH_PASS_SET:
					JiNX_SS_HPfilterOn(high_pass_frequency);
					return 1;
					break;
					
				case IDC_HIGH_PASS_OFF:
					JiNX_SS_HPfilterOff();
					return 1;
					break;
					
				case IDC_REVERB:
					if (reverb_on)
					{
						// turn reverb off
						JiNX_SS_ReverbOff();
						SetWindowText(GetDlgItem(hDlg,IDC_REVERB),"Reverb On");
						reverb_on=0;
					}
					else
					{
						// turn reverb on
						JiNX_SS_ReverbOn();
						SetWindowText(GetDlgItem(hDlg,IDC_REVERB),"Reverb Off");
						reverb_on=1;
					}
					return 1;
					break;
					
				default:
					break;
			}
			break;
		  
		default:
			return 0;
	}
  
	return 0;
}

int speed_test(void)
{
	int count;
	
	clock_t start;
	
	for (start=clock(); (clock() - start) == 0; );
	
	for (count=0, start=clock(); (clock() - start) < (CLOCKS_PER_SEC/4); count++);
	
	return count;
}

void __cdecl etrace(char *format, ... )
{
	va_list arglist;
	char buf[1000];
  
	va_start(arglist, format);
	vsprintf(buf, format, arglist);

	MessageBox(NULL,buf,"Fatal Error!!",MB_OK);

	exit(1);
}

