/*
        JiNX Sound System V1.0e - Demo Programme

        (C) 1997-99, Mark Hodson (Bulk Sound System Code, IT Player, Drivers)
                     Chris Bevan (Sound Output Code Prototype)
                     Jeffrey Lim (IT2.14+ Sample Compression Code)
*/

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <malloc.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "jss_dos.h"

#define SAMPLE_SIZE 53053
#define ZERO_VOL_TICKS 500
#define CONT_SAMPLES 1024

// Prototypes

int speed_test(void);

// Main demo programme

void main(int numargs, char *args[])
{
    int i,j, freq, vcs, reverb_delay, reverb_level, result;

    int this_count, fastest_count, active;
    int total_count=0, max_count=0, total_active=0, active_samples=0;

    char it_filename[100];
    signed char *test_sample;

    JTM_module *mod;

    FILE *fptr;

    // Load an external sample

    test_sample = malloc(SAMPLE_SIZE);
    if (test_sample == NULL)
    {
        printf("Out of Memory!\n");
        exit(1);
    }

    fptr = fopen("inject.pcm", "rb");
    if (fptr == NULL)
    {
        printf("Someone stole Mulder! (INJECT.PCM missing...)\n");
        exit(1);
    }

    for (i=0; i<SAMPLE_SIZE; i++) test_sample[i] = fgetc(fptr);

    fclose(fptr);

    // Check args and create strings

    if (numargs<4)
    {
        printf("\n\nSyntax: JSS_DEMO <Filename> <Frequency> <Channels> [<Reverb Delay> <Level>]\n\n");
        exit(1);
    }

    // Take filename and append .IT if necessary

    strcpy(it_filename,args[1]); strupr(it_filename);

    for (i=0, j=-1; i<strlen(it_filename); i++)
    {
        if (it_filename[i]=='.') j=i;
        if (it_filename[i]=='\\') j=-1;
    }

    if (j!=-1) it_filename[j]=0;

    strcat(it_filename,".IT");

    // Take the other args and look for optional reverb args

    freq = atoi(args[2]); vcs = atoi(args[3]);

    reverb_delay=100; if (numargs>4) reverb_delay = atoi(args[4]);
    reverb_level=2;   if (numargs>5) reverb_level = atoi(args[5]);

    // SetUp Sound System - Choose a device (compulsory)

    result = JiNX_SS_ChooseDevice(0);
    if (result == NULL)
    {
        printf("ChooseDevice Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        exit(1);
    }

    // SetUp Sound System - PreInit settings (optional)

    result = JiNX_SS_PreInitSoundSystem(ZERO_VOL_TICKS, reverb_delay, reverb_level, CONT_SAMPLES);
    if (result == NULL)
    {
        printf("PreInitSoundSystem Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        exit(1);
    }

    // SetUp Sound System - Initialise soundcard and sound system (compulsory)

    result = JiNX_SS_InitSoundSystemMax(freq, vcs);
    if (result == NULL)
    {
        printf("InitSoundSystem Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        exit(1);
    }

    // Print out some info regarding successful initialisation

    printf("\n%s in %s at %0dHz.\n\n",
           JiNX_SS_ReturnDevice(), JiNX_SS_ReturnType(), JiNX_SS_ReturnFrequency());

    // From now on the sound system is active and playing
    // If any errors are encountered, the JSS must be closed before exiting

    // Speed test b4 anything happens (obtain long average)

    printf("Testing operational non-playing speed (2 seconds).\n\n");

    for (fastest_count=0, i=0; i<8; i++) fastest_count += speed_test();

    fastest_count = (fastest_count + 7)>>3;

    // Load Module (give user option of enabling interpolation)

    printf("Enable Pre-Interpolation? [Y/N]\n\n"); i = getch();

    mod = JiNX_SS_LoadModuleFile(it_filename, i=='y' || i=='Y');
    if (mod == NULL)
    {
        printf("LoadModule Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        result = JiNX_SS_CloseSoundSystem();
        exit(1);
    }

    // SetUp Sound System - Setup timer function (optional)

    result = JiNX_SS_InstallAccurateTimer(0);
    if (result == NULL)
    {
        printf("InstallAccurateTimer Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        result = JiNX_SS_CloseSoundSystem();
        exit(1);
    }

    // Play Module

    JiNX_SS_PlayModule(mod);

    // Information Loop

PLAY_CONTINUE:

    do
    {
        // Update speed tests for occupancy calcs

        this_count = speed_test();
        if (this_count > fastest_count) this_count = fastest_count;
        total_count += this_count; max_count += fastest_count;

        // Count the number of non-zero-volume active channels

        for (active=0, i=0; i<mod->last_virtual_channel; i++)
        {
            if ((mod->virtual_channel[i].active_status != 0)
             && (mod->virtual_channel[i].final_volume != 0)) active++;
        }

        active_samples++; total_active += active;

        // Print out an info line

        printf("O:%03d P:%03d R:%03d [%03d/%03d active] | MaxChan: %4.0f | Occ: %5.1f%% | Time %6.1fs\n",
            mod->current_order, mod->current_pattern, mod->current_row,
            active, mod->last_virtual_channel,
            (1.0 * total_active * max_count)/(1.0 * active_samples * (max_count - total_count)),
            100.0 - ((100.0 * this_count) / fastest_count),
            JiNX_SS_WhatTimeIsIt());
    }
    while (!kbhit());

    // Look for keyboard hits

    switch(getch())
    {
        case 'p': case 'P': // Play module from beginning
            JiNX_SS_PlayModule(mod);
            goto PLAY_CONTINUE;

        case 's': case 'S': // Stop module
            JiNX_SS_StopModule();
            goto PLAY_CONTINUE;

        case 'r': case 'R': // Enable reverb
            JiNX_SS_ReverbOn();
            printf("Setting Reverb Echo On\n");
            goto PLAY_CONTINUE;

        case 'l': case 'L': // Enable lowpass filtering
            printf("Enter LowPass Filter Setting [1 -> 9]\n");
            i = 1 + getch() - '1'; if (i<1) i=1; if (i>9) i=9;
            printf("Setting LowPass Filter CutOff To %0dHz\n", 1000*i);
            JiNX_SS_LPfilterOn(1000*i); goto PLAY_CONTINUE;

        case 'h': case 'H': // Enable highpass filtering
            printf("Enter HighPass Filter Setting [1 -> 9]\n");
            i = 1 + getch() - '1'; if (i<1) i=1; if (i>9) i=9;
            printf("Setting HighPass Filter CutOff To %0dHz\n", 100*i);
            JiNX_SS_HPfilterOn(100*i); goto PLAY_CONTINUE;

        case 'n': case 'N': // Normal play (disable reverb and filters)
            JiNX_SS_ReverbOff();
            JiNX_SS_LPfilterOff(); JiNX_SS_HPfilterOff();
            printf("Setting All Filters and Reverbs Off\n");
            goto PLAY_CONTINUE;

        case ',': // Skip back one order
            JiNX_SS_GotoOrderRow(mod->current_order-1, 0);
            printf("Skipping Back 1 Order\n");
            goto PLAY_CONTINUE;

        case '.': // Skip forward one order
            JiNX_SS_GotoOrderRow(mod->current_order+1, 0);
            printf("Skipping Forward 1 Order\n");
            goto PLAY_CONTINUE;

        case 'i': case 'I': // Trigger internal - an IT instrument
            JiNX_SS_TriggerInternal(63, JiNX_SS_ConvertNoteString("C-5"), 1);
            printf("TriggerInternal C-5 in Channel 63 with Instrument 1\n");
            goto PLAY_CONTINUE;

        case 'e': case 'E': // Trigger external - the loaded sample
            i = JiNX_SS_TriggerExternal(test_sample, 8, SAMPLE_SIZE, 22050, 256, 0);
            printf("TriggerSample in Virtual Channel %0d\n", i);
            goto PLAY_CONTINUE;

        case 27: break; // Escape - exit programme

        default: goto PLAY_CONTINUE;
    }

    // Close Sound System

    result = JiNX_SS_UninstallAccurateTimer();
    if (result == NULL)
    {
        printf("UninstallAccurateTimer Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        result = JiNX_SS_CloseSoundSystem();
        exit(1);
    }

    result = JiNX_SS_CloseSoundSystem();
    if (result == NULL)
    {
        printf("CloseSoundSystem Returned: %s\n", JiNX_SS_ErrorString[JiNX_SS_ErrorCode]);
        exit(1);
    }

    // Free Module's Memory

    JiNX_SS_UnloadModule(mod);
}


int speed_test(void)
{
    int count;

    clock_t start;

    for (start=clock(); (clock() - start) == 0; );

    for (count=0, start=clock(); (clock() - start) < (CLOCKS_PER_SEC/4); count++);

    return count;
}



