; A submission to the 'Vintage Computing Christmas Challenge 2025'
; for the 16/48k ZX Spectrum, assembled using sjasmplus assembler.
; https://logiker.com/Vintage-Computing-Christmas-Challenge-2025
;
; Written by spaceWumpus, December 2025.
;
;
    DEVICE ZXSPECTRUM48
    CSPECTMAP "build/main.map"

CENTRECOORD     = 9           ; Character coord (x and y) for centre of snowflake

;   From https://worldofspectrum.net/pub/sinclair/games-info/z/ZXSpectrumAssembler.txt
;       "In the Spectrum, the address of the first character after the REM in a REM
;       statement in line 1 of a program is always 23760, unless microdrives are in use.""

    org 23760 ; First char after REM in line 1 REM in a BASIC program.

	
; Note that SAVE### directives have 'main' listed as entry point.

main:
; 1B So we can terminate with HALT, and not be messed with by RUPTs
  di
  
; DE used to hold constants
; 3B
  ld de, 256*(((2*CENTRECOORD)+1)) + 3

; C,B is X,Y
; 7,9 range limit nicely trims the horizontals and diagonals to the right length
;
; BUT, instead of setting BC to $0907, iterating down, then detecting and stopping after zero,
; we can exploit the fact that if we continue with negative values then the hozizontal and vertical
; lines are redrawn (which is harmless, and actually gives us more flexibility on start value of BC),
; but when we reach -10 this triggers an integer out of range error in the ROM, which calls RST $08, 
; which then returns to the BASIC interpreter, which immediately issues a 'halt'.
; Since we have disabled rupts, this cleanly stops our code.
;
; BC must be in the range $0707 to $09ff to give correct output
; ($0706 misses a char, $0a00 gives out of range char at start)
; Setting B to $08 or $09 is sufficient, C can be anything
; 
; Could save two more bytes by swapping use of BC and DE throughout, and then
; also swapping use of B and C, and then assembling to a start address 
; ending in $08, so that we would get C loaded correctly for free.
; But, it's Christmas eve, and I want to get finished, so I'm leaving that. Maybe after the contest
; 2B
  ld b, $08

loop:
  ; 15B
  ld a,c
  cp 8
  jr nc, skipChar  ; We skip anything with C >= 8
  or a ; Is A zero
  jr z, printChar:  ; Print if c == 0, i.e vertical axis (b=1..9)
  cp b
  jr z, printChar:  ; Print if b == c, i.e. on diagonal (b=1..7)
  ld a,5
  jr c, aboveDiag:

belowDiag:
  ; 5B
  ; c > b (so, below the diagonal to top-right)
  ;ld a,5
  cp b
  jr z, printChar  ; Print if b == 5, this gives the spike on the diagonals
  jr skipChar

aboveDiag:
  ; 10B
  sub e ; 3
  cp c
  jr c, skipChar
  ; If here, then c < 2
  ; When c == 0, we don't care whether we print, because we print stars for c==0 in all cases earlier
  ; So we only care about c==1 and c==2.
  ; We need to detect the following coordinates:
  ; C,B   B-C
  ; 1,4   3   0b011
  ; 2,5   3   
  ; 1,7   6   0b110
  ; 2,8   6
  ; a = b-c
  ld a,b
  sub c
  ; We want to detect a = 3, or 6  
  sub e ; 3
  jr z, printChar ; 'middle' horizontal spike
  sub e ; 3
; Fallthrough

printChar:
  ; Trashes HL:
  ; 'call z' instead of 'call' is safe here because every jump to printChar is 'jr z printChar', so the test will always pass
  ; 3B
  call z, printOctalMirroredAtBCCentreOrigin
; Fallthrough

skipChar:
  ; 3B
  dec bc
  jr loop
  
; ----------------

  ; Mirrored around 11,11
printOctalMirroredAtBCCentreOrigin:
  ; 4B
  ld h, CENTRECOORD
  ld l, h
  add hl, bc
  
  ; Deliberate fallthrough instead of call
  ;call printQuadMirroredAtHL

printOctalMirroredAtHL:
  ; 6B
  call printQuadMirroredAtHL
  ld a,h
  ld h,l
  ld l,a
  ; Deliberate fallthrough instead of call
  ;call printQuadMirroredAtHL

printQuadMirroredAtHL:
  ; 7B
  call printLeftRightMirroredAtHL
  ld a,l
  cpl
  add d ; (2*CENTRECOORD)+1
  ld l,a
  ; Deliberate fallthrough instead of call
  ;call printLeftRightMirroredAtHL

printLeftRightMirroredAtHL:
  ; 7B
  call printAtHL
  ld a,h
  cpl
  add d ; (2*CENTRECOORD)+1
  ld h,a

  ; Deliberate fallthrough instead of call
  ;call printAtHL

printAtHL:
  ; If called with out of range Row or Col, triggers a RST $8 error restart
  ; which ultimately calls ret, returning to BASIC interpreted which then
  ; immediately issues a 'halt'. Since we have disabled rupts, this stops the program.
  ; 11B
  ld a, $16 ; AT
  rst $10
  ld a, l ; Row
  rst $10
  ld a, h ; Col
  rst $10
  ld a, $2a ; Char
  rst $10
  ret

codeEnd

; To query code length:
; In debug console:
;   -eval codeEnd - main
; e.g. report
;   77, 4Dh, 1001101b
; means 77 bytes of code

; Top of stack
stack_top:

    ; Deployment, using 'main' as entrypoint
    SAVESNA "main.sna", main

    EMPTYTAP "main.tap"
    SAVETAP "main.tap", main

    ; Save machine code out as a code file:
    ; This file contains 2 bytes loading address, then 77 bytes of code:
    EMPTYTAP "code.tap"
    SAVETAP "code.tap", CODE, "code", main, $-main

    ; To create REM embedded BASIC version (compo submission):
    ;
    ; 1. In Spectrum emulator, create the following BASIC program (77 digits here in first line - need enough to hold the code):
    ;   1 REM 12345678901234567890123456789012345678901234567890123456789012345678901234567
    ;     REM is 'e'
    ;   2 RANDOMIZE USR 23760
    ;     RANDOMIZE is 'T'
    ;     USR is Ctrl-Shift 'L'
    ;     
    ;
    ; 2. 'Insert' the 'code.tap' file into the emulator
    ;
    ; 3. Load the assembled code into the REM:
    ;   LOAD "" CODE            J, Ctrl-P, Ctrl-P, Shift-Ctrl, I
    ;
    ; 3b. In Fuse: Media->Tape->Clear
    ;
    ; 4. Save the BASIC program with the embedded assembly:
    ;   SAVE "x"
    ;
    ; 4b. In Fuse: Media->Tape->Write, save as "submit.tap".
    ;
    ; 5. Run it
    ;   RUN
    ;
    ;
    ; Alternately, for 4, to create auto-runner, though this doesn't clear the screen before running:
    ;   SAVE "x" LINE 2
    ;     LINE is Ctrl-Shift Ctrl-3
    