;;; Author: Flewww
;;;-----------------------------------------------------------------------------
;;; Header

	PROCESSOR 6502
	INCLUDE "vcs.h"		; Provides RIOT & TIA memory map
	INCLUDE "macro.h"	; This file includes some helper macros

;;; A few constants
LINE_HEIGHT = 7                 ; including one spacing line
RNG_BUFFER_LEN = 6
BG_COLOR = $80
        IFCONST HARDWARE_BUILD
SHADE_COLOR = $06               ; Hardware value
BRICK_BRIGHTNESS = $08          ; Hardware value
        ELSE
SHADE_COLOR = $02               ; Stella value
BRICK_BRIGHTNESS = $04          ; Stella value
        ENDIF

;;;-----------------------------------------------------------------------------
;;; RAM segment
	SEG.U   ram
	ORG     $0080
RAM_START equ *
rng_buf         ds.b RNG_BUFFER_LEN     ; 6 bytes pseudo-random number generator
rng_seed        ds.b RNG_BUFFER_LEN     ; RNG seed
frame_cnt       ds.b 2
hdr_cnt         ds.b 1                  ; Picture header counter - for
                                        ; smooth scrolling
mask            ds.b 6
audv0_proxy     ds.b 1
tmp             ds.b 1
        echo "Used RAM:", (* - RAM_START)d, "bytes"

    MAC WAIT_TIMINT
.wait_timint
	lda TIMINT
	beq .wait_timint
    ENDM

;;; Padding hardware image
        IFCONST HARDWARE_BUILD
        SEG padding
        ORG $F000
        dc.b $ff
        ENDIF

;;;-----------------------------------------------------------------------------
;;; Code segment
	SEG code
	ORG $FF00

init:   SUBROUTINE
        ;; Borrowed from DASM's macro.h - a bit shaved
        ;; Initializes Registers & Memory
        cld
        ldx #0
        txa
        tay
.CLEAR_STACK:
        dex
        txs
        pha
        bne .CLEAR_STACK        ; SP=$FF, X = A = Y = 0

        lda #$01
        sta CTRLPF              ; Mirror playfield

        ;; Initialize rng_seed (Random Number Generator Seed)
        ldx #(RNG_BUFFER_LEN - 1)
.rng_init:      SUBROUTINE
        txa
        sta rng_seed,Y
        dex
        bpl .rng_init

        ;; Initial instrument for music
        lda #$0c                ; div 6 : pure tone
        sta AUDC0

        ;; initialize mask
        lda #$ff
        sta mask+3
        sta mask+4
        sta mask+5

        lda #BG_COLOR
        sta COLUBK
;;; End initialization

main_loop:	SUBROUTINE
	VERTICAL_SYNC		; 4 scanlines Vertical Sync signal
.vblank:
;;; 52 vblank scanlines
        lda #62
        sta TIM64T
        inc frame_cnt

        ;; Reinitialize RNG buffer with seed every frame
        ldx #(RNG_BUFFER_LEN - 1)
.copy_rng_seed:
        lda rng_seed,X
        sta rng_buf,X
        dex
        bpl .copy_rng_seed

        lda hdr_cnt
        bpl .same_line

        ;; Do that each time a line disappears
        ;; Reset header count
        lda #(LINE_HEIGHT - 1)
        sta hdr_cnt

        jsr rng_next_4         ; Update RNG buffer
        ;; Copy updated RNG buffer back to seed
        ldx #(RNG_BUFFER_LEN - 1)
.copy_rng_buf:
        lda rng_buf,X
        sta rng_seed,X
        dex
        bpl .copy_rng_buf

        ;; Play new notes
        ;; Note volume:
        lda rng_buf
        and #$0f
        cmp audv0_proxy
        bmi .same_line
        sta audv0_proxy
        sta AUDV0

        ;; Note frequency:
        lda mask+5
        and mask+3
        ora rng_buf+1
        and #$07
        tax
        lda scale_table,X
        sta AUDF0

        jmp .line_updated
.same_line:
        ;; note fade out
        ldx audv0_proxy
        beq .line_updated
        dex
        stx audv0_proxy
        stx AUDV0

.line_updated:
        ;; Update Playfield mask (rotation) - once a while
        lda frame_cnt
        and #$3f
        bne .dont_update_mask
.update_mask:
        asl mask+5
        rol mask+4
        rol mask+3
        ror mask+2
        rol mask+1
        ror mask
        lda mask+5              ; deal with last carry
        adc #0
        sta mask+5
.dont_update_mask:
        WAIT_TIMINT

.kernel:
;;; 256 kernel + overscan scanlines
        lda #19
        sta T1024T

        ;; Display a few blank lines for smooth scrolling
        ldx hdr_cnt
.header_loop:
        sta WSYNC
        dex
        bpl .header_loop

        lda #31                 ; Count of brick lines do display
        sta tmp
.kernel_loop:
        ;; Update display
        sta WSYNC
        lda rng_buf
        ora #BRICK_BRIGHTNESS
        sta COLUPF
        lda rng_buf+1
        and mask
        sta PF0
        lda rng_buf+2
        and mask+1
        sta PF1
        lda rng_buf+3
        and mask+2
        sta PF2

        ;; Generate new random values for next line
        jsr rng_next_4          ; Uses A, X and Y
        ;; Draw a thin shadow between bricks
        sta WSYNC
        lda #SHADE_COLOR
        sta COLUPF

        dec tmp
        bne .kernel_loop

        ;; Overscan
        lda #BG_COLOR
        sta COLUPF
        dec hdr_cnt

        WAIT_TIMINT
	jmp main_loop

;;; Generates 4 random numbers in rng_buf (from rng_buf to rng_buf+3).
;;; From The_First_Book_of_KIM.pdf (page 172) by Jim Butterfield.
;;; Uses A,X and Y registers
rng_next_4:       SUBROUTINE
        ldy #3
.generate_numbers:
        sec
        lda rng_buf+1
        adc rng_buf+4
        adc rng_buf+5
        sta rng_buf
        ldx #4
.rpl    lda rng_buf,X
        sta rng_buf+1,X
        dex
        bpl .rpl
        dey
        bpl .generate_numbers
        rts

;;; 4 Notes: D, E, G, A - 2 octaves
scale_table:
        ;;                                 computed -> heard
        dc.b  7                 ; 625.00 Hz -> Eb  -> E
        dc.b  8                 ; 555.56 Hz -> Db  -> D
        dc.b 11                 ; 416.67 Hz -> Ab  -> A
        dc.b 15                 ; 312.50 Hz -> Eb  -> E
        dc.b 17                 ; 277.78 Hz -> Db  -> D
        dc.b 23                 ; 208.33 Hz -> Ab  -> A
        dc.b 26                 ; 185.19 Hz -> Gb  -> G
        dc.b 31                 ; 156.25 Hz -> Eb  -> E

;;; End of Code
;;; ----------------------------------------------------------------------------
	echo "Used ROM:", (* - $ff00)d, "bytes"
	SEG reset
	ORG $FFFC
	DC.W init
        dc.b $46,$4c            ; sig
