;
; synth85.s
; Synth85
;
; Created by Stefan Koch on 28.02.16.
; Copyright (c) 2016 Moods Plateau.
; All rights reserved.
;

	INCLUDE "system/ctc.i"
	INCLUDE "lib/add16.i"

; ---------- constants -----------

_S85_NOTE:	EQU	0
_S85_EFFECT:	EQU	1
_S85_PARAM:	EQU	2

PERIODS_COUNT:	EQU	85	; PERIODS_END-PERIODS
PERIODS_VTC:	EQU	47	; PERIODS_VTE-PERIODS

VOLUME_MAX:	EQU	40H

; ---------- code start ----------

_SYNTH85_INIT:

	; init count with 1 (pre-decrement)
	LD	A,1
	LD	(_S85_TICK),A

	; reset notify, end
	XOR	A
	LD	(SYNTH85_NOTIFY),A
	LD	(SYNTH85_END),A

	; reset channels
	call	aud_ch0_off
	call	aud_ch1_off

	; init volume
	LD	A,31
	CALL	aud_vol_set

	IF SYNTH85_MASTER_VOLUME != 0

	; init master volume
	LD	A,0F0H
	LD	(_S85_MASTER_VOLUME),A

	ENDIF

	RET

; Load given module, return error in A
;
SYNTH85_LOAD:	; (ModulePtr/HL) Error/A

	CALL	_SYNTH85_INIT

	; set default speed on load
	LD	A,6
	LD	(_S85_SPEED),A

	CALL	mod_parse

	RET

; Sets playing position.
;
SYNTH85_SET:	; (Position/A)

	CALL	mod_set_position

	RET

	IF SYNTH85_CUSTOM_INT == 0

; Play currently loaded module
;
SYNTH85_PLAY:	; ()

	CALL	_SYNTH85_INIT

	LD	HL,_SYNTH85_INT_HANDLER
	LD	A,1	; Teiler = 1 -> 50 Hz
	CALL	CTC_INIT

	RET

; Stop currently playing module
;
SYNTH85_STOP:	; ()

	; ----- cleanup ctc -----

	; restore original isr
	CALL	CTC_CLEANUP

	; sound off
	CALL	SYNTH85_SOUND_OFF

	RET

	ENDIF

; Stop sound
;
SYNTH85_SOUND_OFF:	; ()

	LD	A,3
	OUT	(8CH),A
	OUT	(8DH),A

	RET

	IF SYNTH85_MASTER_VOLUME != 0

; Sets master volume.
;
; A = master volume (0..31)
;
SYNTH85_SET_MASTER_VOLUME:

	PUSH	AF
	PUSH	BC

	CALL	aud_vol_set

	POP	BC
	POP	AF

	RLCA
	RLCA
	RLCA
	AND	0F0H

	LD	(_S85_MASTER_VOLUME),A

	RET

	ENDIF

	IF SYNTH85_CHANNEL_MUTE != 0

; Mutes / unmutes channel 1
;
; A = muted (mute: 1, unmute: 0)
;
SYNTH85_MUTE_CH1:

	LD	(SYNTH85_CH1_MUTED),A

	AND	A
	RET	Z

	JP	aud_ch0_off

; Mutes / unmutes channel 2
;
; A = muted (mute: 1, unmute: 0)
;
SYNTH85_MUTE_CH2:

	LD	(SYNTH85_CH2_MUTED),A

	AND	A
	RET	Z

	JP	aud_ch1_off

	ENDIF

; ---------- interrupt ----------

	IF SYNTH85_CUSTOM_INT == 0

_SYNTH85_INT_HANDLER:

	CALL	IRET	; allow interrupts (preserve keyboard input)

	PUSH	HL
	PUSH	DE
	PUSH	BC
	PUSH	AF

	IF SYNTH85_RAM8 != 0

	; ensure RAM8_1 here
	LD	A,(IX+1)
	OR	1<<RAM8SEL
	OUT	(84H),A

	; disable irm for playing from RAM8
	IN	A,(PIOA)
	PUSH	AF
	AND	~(1<<IRM)
	OUT	(PIOA),A

	ENDIF

	CALL	SYNTH85_TICK

	IF SYNTH85_RAM8 != 0

	; restore irm
	POP	AF
	OUT	(PIOA),A

	; restore RAM8 block
	LD	A,(IX+1)
	OUT	(84H),A

	ENDIF

	POP	AF
	POP	BC
	POP	DE
	POP	HL

	RET

	ENDIF

; ---------- tick ----------

; Call from interrupt at 50 Hz
;
SYNTH85_TICK:	; ()

	; ----- ticks -----

	LD	HL,_S85_TICK
	DEC	(HL)
	JP	Z,.NEXT_NOTE

	IF SYNTH85_USE_SFX != 0

	; check sfx
	LD	A,(S85_SFX_PLAYING)
	AND	A
	RET	NZ

	ENDIF

	; ----- effect -----

	; CH1

	; volume control
	LD	A,(mod_song_attrs)
	AND	MOD_SA_LEFT
	LD	(_S85_VOLUME_FLAG),A

	; setup last values from ch1
	CALL	params_setup_ch1

	LD	A,(_S85_CH1_PERIOD)
	LD	(_S85_CHX_PERIOD),A

	LD	A,(_S85_CH1_VORTEILER)
	LD	(_S85_CHX_VORTEILER),A

	LD	A,(_S85_CH1_VOLUME)
	LD	(_S85_CHX_VOLUME),A

	LD	A,(_S85_CH1+_S85_NOTE)
	LD	(_S85_CHX+_S85_NOTE),A

	LD	A,(_S85_CH1+_S85_EFFECT)
	LD	(_S85_CHX+_S85_EFFECT),A

	LD	A,(_S85_CH1+_S85_PARAM)
	LD	(_S85_CHX+_S85_PARAM),A

	LD	A,(SLIDE_TO_PERIOD_CH1)
	LD	(SLIDE_TO_PERIOD),A

	LD	A,(VIBRATO_SPEED_CH1)
	LD	(VIBRATO_SPEED),A
	LD	A,(VIBRATO_DEPTH_CH1)
	LD	(VIBRATO_DEPTH),A
	LD	A,(VIBRATO_INDEX_CH1)
	LD	(VIBRATO_INDEX),A

	CALL	EFFECT

	LD	A,(_S85_CHX_PERIOD)
	LD	(_S85_CH1_PERIOD),A

	LD	A,(_S85_CHX_VORTEILER)
	LD	(_S85_CH1_VORTEILER),A

	LD	A,(_S85_CHX_VOLUME)
	LD	(_S85_CH1_VOLUME),A

	CALL	params_store_ch1

	LD	A,(SLIDE_TO_PERIOD)
	LD	(SLIDE_TO_PERIOD_CH1),A

	LD	A,(VIBRATO_SPEED)
	LD	(VIBRATO_SPEED_CH1),A
	LD	A,(VIBRATO_DEPTH)
	LD	(VIBRATO_DEPTH_CH1),A
	LD	A,(VIBRATO_INDEX)
	LD	(VIBRATO_INDEX_CH1),A

	; CH2

	; volume control
	LD	A,(mod_song_attrs)
	AND	MOD_SA_RIGHT
	LD	(_S85_VOLUME_FLAG),A

	; setup last values from ch2
	CALL	params_setup_ch2

	LD	A,(_S85_CH2_PERIOD)
	LD	(_S85_CHX_PERIOD),A

	LD	A,(_S85_CH2_VORTEILER)
	LD	(_S85_CHX_VORTEILER),A

	LD	A,(_S85_CH2_VOLUME)
	LD	(_S85_CHX_VOLUME),A

	LD	A,(_S85_CH2+_S85_NOTE)
	LD	(_S85_CHX+_S85_NOTE),A

	LD	A,(_S85_CH2+_S85_EFFECT)
	LD	(_S85_CHX+_S85_EFFECT),A

	LD	A,(_S85_CH2+_S85_PARAM)
	LD	(_S85_CHX+_S85_PARAM),A

	LD	A,(SLIDE_TO_PERIOD_CH2)
	LD	(SLIDE_TO_PERIOD),A

	LD	A,(VIBRATO_SPEED_CH2)
	LD	(VIBRATO_SPEED),A
	LD	A,(VIBRATO_DEPTH_CH2)
	LD	(VIBRATO_DEPTH),A
	LD	A,(VIBRATO_INDEX_CH2)
	LD	(VIBRATO_INDEX),A

	CALL	EFFECT

	LD	A,(_S85_CHX_PERIOD)
	LD	(_S85_CH2_PERIOD),A

	LD	A,(_S85_CHX_VORTEILER)
	LD	(_S85_CH2_VORTEILER),A

	LD	A,(_S85_CHX_VOLUME)
	LD	(_S85_CH2_VOLUME),A

	CALL	params_store_ch2

	LD	A,(SLIDE_TO_PERIOD)
	LD	(SLIDE_TO_PERIOD_CH2),A

	LD	A,(VIBRATO_SPEED)
	LD	(VIBRATO_SPEED_CH2),A
	LD	A,(VIBRATO_DEPTH)
	LD	(VIBRATO_DEPTH_CH2),A
	LD	A,(VIBRATO_INDEX)
	LD	(VIBRATO_INDEX_CH2),A

	JP	.PLAY_SOUND

	; ----- next note -----

.NEXT_NOTE:

	; ----- ticks -----

	LD	A,(_S85_SPEED)
	LD	(_S85_TICK),A

	; ----- check sfx -----

	IF SYNTH85_USE_SFX != 0

	LD	A,(S85_SFX_PLAYING)
	AND	A
	RET	NZ

	ENDIF

	; fetch notes (ch1 + ch2)
	LD	DE,_S85_CH1
	CALL	mod_fetch

	; notify new values
	LD	A,1
	LD	(SYNTH85_NOTIFY),A

	; CH1

	; volume control
	LD	A,(mod_song_attrs)
	AND	MOD_SA_LEFT
	LD	(_S85_VOLUME_FLAG),A

	; setup last values from ch1
	CALL	params_setup_ch1

	LD	A,(_S85_CH1_PERIOD)
	LD	(_S85_CHX_PERIOD),A

	LD	A,(_S85_CH1_VORTEILER)
	LD	(_S85_CHX_VORTEILER),A

	LD	A,(_S85_CH1_VOLUME)
	LD	(_S85_CHX_VOLUME),A

	LD	A,(_S85_CH1+_S85_NOTE)
	LD	(_S85_CHX+_S85_NOTE),A

	LD	A,(_S85_CH1+_S85_EFFECT)
	LD	(_S85_CHX+_S85_EFFECT),A

	LD	A,(_S85_CH1+_S85_PARAM)
	LD	(_S85_CHX+_S85_PARAM),A

	LD	A,(SLIDE_TO_PERIOD_CH1)
	LD	(SLIDE_TO_PERIOD),A

	LD	A,(VIBRATO_SPEED_CH1)
	LD	(VIBRATO_SPEED),A
	LD	A,(VIBRATO_DEPTH_CH1)
	LD	(VIBRATO_DEPTH),A
	LD	A,(VIBRATO_INDEX_CH1)
	LD	(VIBRATO_INDEX),A

	CALL	PLAY_NOTE

	LD	A,(_S85_CHX_PERIOD)
	LD	(_S85_CH1_PERIOD),A

	LD	A,(_S85_CHX_VORTEILER)
	LD	(_S85_CH1_VORTEILER),A

	LD	A,(_S85_CHX_VOLUME)
	LD	(_S85_CH1_VOLUME),A

	CALL	params_store_ch1

	LD	A,(SLIDE_TO_PERIOD)
	LD	(SLIDE_TO_PERIOD_CH1),A

	LD	A,(VIBRATO_SPEED)
	LD	(VIBRATO_SPEED_CH1),A
	LD	A,(VIBRATO_DEPTH)
	LD	(VIBRATO_DEPTH_CH1),A
	LD	A,(VIBRATO_INDEX)
	LD	(VIBRATO_INDEX_CH1),A

	; CH2

	; volume control
	LD	A,(mod_song_attrs)
	AND	MOD_SA_RIGHT
	LD	(_S85_VOLUME_FLAG),A

	; setup last values from ch2
	CALL	params_setup_ch2

	LD	A,(_S85_CH2_PERIOD)
	LD	(_S85_CHX_PERIOD),A

	LD	A,(_S85_CH2_VORTEILER)
	LD	(_S85_CHX_VORTEILER),A

	LD	A,(_S85_CH2_VOLUME)
	LD	(_S85_CHX_VOLUME),A

	LD	A,(_S85_CH2+_S85_NOTE)
	LD	(_S85_CHX+_S85_NOTE),A

	LD	A,(_S85_CH2+_S85_EFFECT)
	LD	(_S85_CHX+_S85_EFFECT),A

	LD	A,(_S85_CH2+_S85_PARAM)
	LD	(_S85_CHX+_S85_PARAM),A

	LD	A,(SLIDE_TO_PERIOD_CH2)
	LD	(SLIDE_TO_PERIOD),A

	LD	A,(VIBRATO_SPEED_CH2)
	LD	(VIBRATO_SPEED),A
	LD	A,(VIBRATO_DEPTH_CH2)
	LD	(VIBRATO_DEPTH),A
	LD	A,(VIBRATO_INDEX_CH2)
	LD	(VIBRATO_INDEX),A

	CALL	PLAY_NOTE

	LD	A,(_S85_CHX_PERIOD)
	LD	(_S85_CH2_PERIOD),A

	LD	A,(_S85_CHX_VORTEILER)
	LD	(_S85_CH2_VORTEILER),A

	LD	A,(_S85_CHX_VOLUME)
	LD	(_S85_CH2_VOLUME),A

	CALL	params_store_ch2

	LD	A,(SLIDE_TO_PERIOD)
	LD	(SLIDE_TO_PERIOD_CH2),A

	LD	A,(VIBRATO_SPEED)
	LD	(VIBRATO_SPEED_CH2),A
	LD	A,(VIBRATO_DEPTH)
	LD	(VIBRATO_DEPTH_CH2),A
	LD	A,(VIBRATO_INDEX)
	LD	(VIBRATO_INDEX_CH2),A

.PLAY_SOUND:

	CALL	params_load_ch1

	AND	A
	JR	Z,.NO_NEW_PARAMS_CH1

	IF SYNTH85_CHANNEL_MUTE != 0
	LD	A,(SYNTH85_CH1_MUTED)
	AND	A
	JR	NZ,.NO_NEW_PARAMS_CH1
	ENDIF

	LD	A,L
	AND	A
	JR	Z,.OFF1

	call	aud_ch0_set

	JR	.NO_NEW_PARAMS_CH1

.OFF1:
	call	aud_ch0_off

.NO_NEW_PARAMS_CH1:

	CALL	params_load_ch2

	AND	A
	JR	Z,.NO_NEW_PARAMS_CH2

	IF SYNTH85_CHANNEL_MUTE != 0
	LD	A,(SYNTH85_CH2_MUTED)
	AND	A
	JR	NZ,.NO_NEW_PARAMS_CH2
	ENDIF
	
	LD	A,L
	AND	A
	JR	Z,.OFF2

	call	aud_ch1_set

	JR	.NO_NEW_PARAMS_CH2

.OFF2:
	call	aud_ch1_off


.NO_NEW_PARAMS_CH2:

	; ----- volume -----

	CALL	params_load_volume

	AND	A
	JR	Z,.NO_NEW_VOLUME

	LD	A,C
	CP	40H
	JP	C,.VOL_OK

	LD	A,3FH

.VOL_OK:
	; preserve volume level 1
	CP	1
	JR	Z,.VOL_SKIP_SCALE

	SRL	A	; 0..31

.VOL_SKIP_SCALE:

	IF SYNTH85_MASTER_VOLUME != 0

	SRL	A	; 0..15
	LD	HL,_S85_MASTER_VOLUME
	OR	(HL)
	ADDC16	_S85_MASTER_VOLUME_TABLE
	LD	A,(HL)

	ENDIF

	CALL	aud_vol_set

.NO_NEW_VOLUME:

	RET

; ---------- note ----------

PLAY_NOTE:

	; set default volume if new note
	LD	A,(_S85_CHX+_S85_NOTE)
	AND	A
	JR	Z,.SKIP_DEFAULT_VOLUME

	; volume changes?
	LD	A,(_S85_CHX_VOLUME)
	CP	VOLUME_MAX
	JR	Z,.SKIP_DEFAULT_VOLUME

	; set default volume
	LD	A,VOLUME_MAX
	LD	(_S85_CHX_VOLUME),A ; for effects
	CALL	params_set_volume

.SKIP_DEFAULT_VOLUME:

	; immediate effects

	; ***** ARPEGGIO *****

	LD	A,(_S85_CHX+_S85_EFFECT)

	AND	A	; 0 = arpeggio
	JR	NZ,.NO_ARPEGGIO

	; any parameters given?
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	A
	JP	Z,.CONTINUE

	; note value given?
	; Fixes arpeggio glitch
	; TODO: use NOTE_PLAYING_CHX
	LD	A,(_S85_CHX+_S85_NOTE)
	AND	A
	JP	Z,.CONTINUE

	; lookup main period
	; TODO: use NOTE_PLAYING_CHX
	LD	A,(_S85_CHX+_S85_NOTE)
	LD	HL,PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)	; Period
	LD	(ARPEGGIO_PERIODS),A

	; lookup Vorteiler
	LD	H,0	; VT -> H
	LD	A,(_S85_CHX+_S85_NOTE)
	CP	PERIODS_VTC	; 47
	JR	NC,.ARPEGGIO_NO_VORTEILER
	INC	H
.ARPEGGIO_NO_VORTEILER:
	LD	A,H
	LD	(ARPEGGIO_VORTEILER),A

	; lookup semi period 1
	LD	A,(_S85_CHX+_S85_PARAM)
	RRCA
	RRCA
	RRCA
	RRCA
	AND	0FH
	LD	B,A
	LD	A,(_S85_CHX+_S85_NOTE)
	ADD	B
	LD	HL,PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)	; Period
	LD	(ARPEGGIO_PERIODS+1),A

	; lookup Vorteiler 1
	LD	H,0	; VT -> H
	LD	A,(_S85_CHX+_S85_NOTE)
	CP	PERIODS_VTC	; 47
	JR	NC,.ARPEGGIO_NO_VORTEILER1
	INC	H
.ARPEGGIO_NO_VORTEILER1:
	LD	A,H
	LD	(ARPEGGIO_VORTEILER+1),A

	; lookup semi period 2
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0FH
	LD	B,A
	LD	A,(_S85_CHX+_S85_NOTE)
	ADD	B
	LD	HL,PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)	; Period
	LD	(ARPEGGIO_PERIODS+2),A

	; lookup Vorteiler 2
	LD	H,0	; VT -> H
	LD	A,(_S85_CHX+_S85_NOTE)
	CP	PERIODS_VTC	; 47
	JR	NC,.ARPEGGIO_NO_VORTEILER2
	INC	H
.ARPEGGIO_NO_VORTEILER2:
	LD	A,H
	LD	(ARPEGGIO_VORTEILER+2),A

	; reset arpeggio state
	XOR	A			;  4/1
	LD	(ARPEGGIO_STATE),A	; 13/3

	JP	.CONTINUE

.NO_ARPEGGIO:

	; ***** SLIDE_TO_NOTE *****

	CP	3	; 3 = slide to note
	JR	NZ,.NO_SLIDE_TO_NOTE

	; note value given?
	LD	A,(_S85_CHX+_S85_NOTE)
	AND	A
	JR	NZ,.S2N_VALUE

	; set current values (do nothing)
	LD	A,(_S85_CHX_VORTEILER)
	LD	(SLIDE_TO_VORTEILER),A
	LD	A,(_S85_CHX_PERIOD)
	LD	(SLIDE_TO_PERIOD),A

	JP	.CONTINUE

.S2N_VALUE:

	; any parameters given?
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	A
	JP	Z,.CONTINUE

	; lookup period
	LD	A,(_S85_CHX+_S85_NOTE)
	LD	HL,PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)	; Period
	LD	(SLIDE_TO_PERIOD),A

	; lookup Vorteiler
	LD	H,0	; VT -> H
	LD	A,(_S85_CHX+_S85_NOTE)
	CP	PERIODS_VTC	; 47
	JR	NC,.S2N_NO_VORTEILER
	INC	H
.S2N_NO_VORTEILER:
	LD	A,H
	LD	(SLIDE_TO_VORTEILER),A

	JP	.NO_NEW_NOTE

.NO_SLIDE_TO_NOTE:

	; ***** Vibrato *****

	CP	4
	JR	NZ,.NO_VIBRATO

	; both parameters given?
	LD	A,(_S85_CHX+_S85_PARAM)
	LD	L,A
	AND	0F0H
	IF SYNTH85_PATTERN_REPEAT != 0
	JP	Z,.CONTINUE ; no speed
	ELSE
	JR	Z,.CONTINUE ; no speed
	ENDIF

	LD	A,L
	AND	0FH
	IF SYNTH85_PATTERN_REPEAT != 0
	JP	Z,.CONTINUE ; no depth
	ELSE
	JR	Z,.CONTINUE ; no depth
	ENDIF

	; vibrato depth
	LD	(VIBRATO_DEPTH),A

	; vibrato speed
	LD	A,L
	SRL	A
	SRL	A
	SRL	A
	SRL	A
	LD	(VIBRATO_SPEED),A

	JR	.CONTINUE

.NO_VIBRATO:

	; in place effects

	; ***** POSITION JUMP *****

	CP	0BH	; position jump
	JR	NZ,.NO_POSITION_JUMP

	IF SYNTH85_PATTERN_REPEAT != 0

	LD	A,(mod_repeat_pattern)
	AND	A
	JR	Z,.POSITION_JUMP_TO_SONGPOS

	LD	A,(mod_division)
	NEG	A
	ADD	MOD_PATTERN_LEN-1
	LD	L,A
	LD	H,0
	ADD	HL,HL	; *2
	LD	D,H
	LD	E,L
	ADD	HL,HL	; *4
	ADD	HL,DE	; *6
	EX	DE,HL
	LD	HL,(_mod_divptr)
	ADD	HL,DE
	LD	(_mod_divptr),HL

	JR	.POSITION_JUMP_SKIP

.POSITION_JUMP_TO_SONGPOS:

	ENDIF

	; find target position
	LD	A,(_S85_CHX+_S85_PARAM)
	DEC	A	; jump to position 0 would not work here!
	LD	(SYNTH85_SONGPOS),A

	IF SYNTH85_PATTERN_REPEAT != 0

.POSITION_JUMP_SKIP:

	ENDIF

	; skip pattern
	LD	A,MOD_PATTERN_LEN-1
	LD	(SYNTH85_PATTPOS),A

	JR	.CONTINUE

.NO_POSITION_JUMP:

	; in place effects

	; ***** VOLUME *****

	CP	0CH	; volume
	JR	NZ,.NO_VOLUME

	; set volume
	LD	A,(_S85_CHX+_S85_PARAM)
	LD	(_S85_CHX_VOLUME),A ; for effects

	CALL	params_set_volume

	JR	.CONTINUE

.NO_VOLUME:

	; in place effects

	; ***** PATTERN BREAK *****

	CP	0DH	; pattern break
	JR	NZ,.NO_PATTERN_BREAK

	IF SYNTH85_PATTERN_REPEAT != 0

	LD	A,(mod_repeat_pattern)
	AND	A
	JR	Z,.PATTERN_BREAK

	LD	A,(mod_division)
	NEG	A
	ADD	MOD_PATTERN_LEN-1
	LD	L,A
	LD	H,0
	ADD	HL,HL	; *2
	LD	D,H
	LD	E,L
	ADD	HL,HL	; *4
	ADD	HL,DE	; *6
	EX	DE,HL
	LD	HL,(_mod_divptr)
	ADD	HL,DE
	LD	(_mod_divptr),HL

.PATTERN_BREAK:

	ENDIF

	; TODO: find next pattern and set position
	LD	A,MOD_PATTERN_LEN-1
	LD	(SYNTH85_PATTPOS),A

	JR	.CONTINUE

.NO_PATTERN_BREAK:

	; ***** SPEED *****

	CP	0FH	; speed
	JR	NZ,.NO_SPEED

	; any parameters given?
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	A
	JR	Z,.CONTINUE

	; ticks per row (TPR)
	CP	1FH ; >= 20H -> skip
	JR	NC,.CONTINUE ; BPM not supported

	LD	(_S85_SPEED),A

.NO_SPEED:

.CONTINUE:

	; ----- new note value? -----

	LD	A,(_S85_CHX+_S85_NOTE)
	AND	A
	JR	Z,.NO_NEW_NOTE

	; lookup period
	LD	A,(_S85_CHX+_S85_NOTE)
	LD	HL,PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)
	LD	L,A	; Period -> L

	; lookup Vorteiler
	LD	H,0	; VT -> H
	LD	A,(_S85_CHX+_S85_NOTE)
	CP	PERIODS_VTC	; 47
	JR	NC,.NO_VORTEILER

	INC	H

.NO_VORTEILER:

	; period and prescale needed
	; for slide up/down effect
	LD	A,H
	LD	(_S85_CHX_VORTEILER),A
	LD	A,L
	LD	(_S85_CHX_PERIOD),A

	; store vorteiler and period here
	CALL	params_set_period

	JR	.NO_RESET

.NO_NEW_NOTE:

	IF 0	; no reset

	; arpeggio and vibrato reset
	; do nothing, if volume is 0
	LD	A,(_S85_CHX_VOLUME)
	AND	A
	JR	Z,.NO_RESET

	; arpeggio reset?
	LD	A,(ARPEGGIO_STATE)
	AND	A
	JR	Z,.VIBRATO_RESET

	; do arpeggio reset?
	LD	A,(_S85_CHX+_S85_EFFECT)
	CP	0 ; arpeggio
	JR	Z,.VIBRATO_RESET

	XOR	A
	LD	(ARPEGGIO_STATE),A

	JR	.RESET_NOTE

.VIBRATO_RESET:

	; vibrato reset?
	LD	A,(VIBRATO_SPEED)
	AND	A
	JR	Z,.NO_RESET

	; do vibrato reset?
	LD	A,(_S85_CHX+_S85_EFFECT)
	CP	4 ; vibrato
	JR	Z,.NO_RESET
	CP	6 ; vibrato + volume slide
	JR	Z,.NO_RESET

.RESET_NOTE:

	; do reset
	LD	A,(_S85_CHX_VORTEILER)
	LD	H,A
	LD	A,(_S85_CHX_PERIOD)
	LD	L,A

	CALL	params_set_period

	XOR	A
	LD	(VIBRATO_SPEED),A ; speed = 0

	; reset vibtab index
	XOR	A
	LD	(VIBRATO_INDEX),A

	ENDIF

.NO_RESET:

	RET

; ---------- effects ----------

EFFECT:

	; dispatch effects (ch1)

	LD	A,(_S85_CHX+_S85_EFFECT)
	ADD	A	; offset in bytes
	ADD	COMMAND_TABLE % 256
	LD	L,A
	ADC	A,COMMAND_TABLE / 256
	SUB	L
	LD	H,A
	LD	A,(HL)
	INC	HL
	LD	H,(HL)
	LD	L,A
	JP	(HL)

; ---------- effect commands ----------

CMD_DUMMY:
	RET

CMD_ARPEGGIO:

	; any parameters given?
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	A
	RET	Z

	; next state
	LD	A,(ARPEGGIO_STATE)
	INC	A
	AND	3
	LD	(ARPEGGIO_STATE),A

	; get period
	LD	HL,ARPEGGIO_PERIODS
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	E,(HL)

	; get Vorteiler
	LD	HL,ARPEGGIO_VORTEILER
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	H,(HL)	; VT
	LD	L,E	; ZK

	; set period
	CALL	params_set_period

	RET

ARPEGGIO_STATE:
	DEFB	0
ARPEGGIO_PERIODS:
	DEFS	3,0
ARPEGGIO_VORTEILER:
	DEFS	3,0

CMD_SLIDE_UP:

	; check vorteiler
	LD	A,(_S85_CHX_VORTEILER)
	AND	A
	JR	NZ,.SLIDE_UP_VT1

	; ----- VT = 0 -----

	; dec period and store new value
	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM
	SUB	(HL)

	LD	H,0 ; VT = 0

	CP	27
	JR	NC,.SLIDE_UP_STORE

	LD	A,27

	JR	.SLIDE_UP_STORE

	; ----- VT = 1 -----

.SLIDE_UP_VT1:

	; dec period and store new value
	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM
	SUB	(HL)

	LD	H,1	; VT = 1

	CP	15
	JR	NC,.SLIDE_UP_STORE

	LD	H,0
	LD	A,229

.SLIDE_UP_STORE:

	; store new values
	LD	L,A	; ZK
	LD	(_S85_CHX_PERIOD),A
	LD	A,H	; VT
	LD	(_S85_CHX_VORTEILER),A

	; set period
	CALL	params_set_period

	RET

CMD_SLIDE_DOWN:

	; check vorteiler
	LD	A,(_S85_CHX_VORTEILER)
	AND	A
	JR	NZ,.SLIDE_DOWN_VT1

	; ----- VT = 0 -----

	; inc period and store new value
	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM
	ADD	(HL)

	LD	H,0 ; VT = 0

	CP	230
	JR	C,.SLIDE_DOWN_STORE

	LD	H,1
	LD	A,15

	JR	.SLIDE_DOWN_STORE

	; ----- VT = 1 -----

.SLIDE_DOWN_VT1:

	; inc period and store new value
	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM
	ADD	(HL)

	LD	H,1 ; VT = 1

	CP	205
	JR	C,.SLIDE_DOWN_STORE

	LD	A,204

.SLIDE_DOWN_STORE:

	; store new values
	LD	L,A	; ZK
	LD	(_S85_CHX_PERIOD),A
	LD	A,H	; VT
	LD	(_S85_CHX_VORTEILER),A

	; set period
	CALL	params_set_period

	RET

CMD_STN_AND_VOL_SLIDE:

	; check for slide down
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0F0H
	JR	NZ,.STN_VOL_UP

	; check for volume already here
	LD	A,(_S85_CHX_VOLUME)
	AND	A
	RET	Z

	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0FH
	JR	Z,CMD_SLIDE_TO_NOTE

	; volume down

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)

	SUB	B

	CP	1
	JR	NC,.STN_VOL_DONE ; A > 0 -> done

	XOR	A ; set to 0

	; store new volume
	LD	(_S85_CHX_VOLUME),A

	; set new volume
	CALL	params_set_volume

	RET ; no slide

.STN_VOL_UP:

	; volume up

	SRL	A
	SRL	A
	SRL	A
	SRL	A

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)
	CP	VOLUME_MAX
	JR	Z,CMD_SLIDE_TO_NOTE ; A = MAX -> skip

	ADD	B

	CP	VOLUME_MAX
	JR	NC,.STN_VOL_DONE ; A <= MAX -> done

	LD	A,VOLUME_MAX ; set to MAX

.STN_VOL_DONE:

	; store new volume
	LD	(_S85_CHX_VOLUME),A

	; set new volume
	CALL	params_set_volume

CMD_SLIDE_TO_NOTE:

	LD	A,(_S85_CHX_VORTEILER)
	AND	A
	JR	NZ,.SLIDE_TO_VT1

	; ----- VT = 0 -----

	LD	A,(SLIDE_TO_VORTEILER)
	AND	A
	JR	NZ,.SLIDE_VT01_DOWN

	LD	A,(_S85_CHX_PERIOD)
	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	RET	Z

	LD	E,0 ; VT = 0
	LD	HL,_S85_CHX+_S85_PARAM

	JR	NC,.SLIDE_VT00_UP

	; ----- slide down VT00 -----

	ADD	(HL)

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	C,.SLIDE_END

	JR	.SLIDE_FINISH

.SLIDE_VT00_UP:

	; ----- slide up VT00 -----

	SUB	(HL)

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	NC,.SLIDE_END

	JR	.SLIDE_FINISH

.SLIDE_VT01_DOWN:

	; ----- slide down VT01 -----

	LD	E,0

	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM

	ADD	(HL)

	CP	230
	JR	C,.SLIDE_V01_DOWN_NO_VT

	LD	E,1	; VT
	LD	A,15	; ZK

.SLIDE_V01_DOWN_NO_VT:

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	C,.SLIDE_END

	JR	.SLIDE_FINISH

	; ----- VT = 1 -----

.SLIDE_TO_VT1:

	LD	A,(SLIDE_TO_VORTEILER)
	AND	A
	JR	Z,.SLIDE_VT10_UP

	LD	A,(_S85_CHX_PERIOD)
	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	RET	Z

	LD	E,1 ; VT = 1
	LD	HL,_S85_CHX+_S85_PARAM

	JR	C,.SLIDE_VT11_DOWN

	; ----- slide up VT11 -----

	SUB	(HL)

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	NC,.SLIDE_END

	JR	.SLIDE_FINISH

.SLIDE_VT11_DOWN:

	; ----- slide down VT11 -----

	ADD	(HL)

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	C,.SLIDE_END

	JR	.SLIDE_FINISH

.SLIDE_VT10_UP:

	; ----- slide up VT10 -----

	LD	E,1

	LD	A,(_S85_CHX_PERIOD)
	LD	HL,_S85_CHX+_S85_PARAM

	SUB	(HL)

	CP	15
	JR	NC,.SLIDE_V10_UP_VT

	LD	E,0	; VT
	LD	A,229	; ZK

.SLIDE_V10_UP_VT:

	LD	HL,SLIDE_TO_PERIOD
	CP	(HL)
	JR	C,.SLIDE_END

.SLIDE_FINISH:

	; set exact target values
	LD	A,(SLIDE_TO_VORTEILER)
	LD	E,A

	LD	A,(SLIDE_TO_PERIOD)

.SLIDE_END:

	; store new values
	LD	(_S85_CHX_PERIOD),A
	LD	A,E
	LD	(_S85_CHX_VORTEILER),A

	; load argument registers
	LD	H,A	; VT
	LD	A,(_S85_CHX_PERIOD)
	LD	L,A	; ZK

	; set new prescaler and period
	CALL	params_set_period

	RET

SLIDE_TO_PERIOD_CH1:
	DEFB	0
SLIDE_TO_PERIOD_CH2:
	DEFB	0
SLIDE_TO_PERIOD:
	DEFB	0
SLIDE_TO_VORTEILER_CH1:
	DEFB	0
SLIDE_TO_VORTEILER_CH2:
	DEFB	0
SLIDE_TO_VORTEILER:
	DEFB	0

CMD_VIB_AND_VOL_SLIDE:

	; check for slide down
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0F0H
	JR	NZ,.VIB_VOL_UP

	; check for volume already here
	LD	A,(_S85_CHX_VOLUME)
	AND	A
	RET	Z

	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0FH
	JR	Z,CMD_VIBRATO

	; volume down

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)

	SUB	B

	CP	1
	JR	NC,.VIP_VOL_DONE ; A > 0 -> done

	XOR	A ; set to 0

	; store new volume
	LD	(_S85_CHX_VOLUME),A

	; set new volume
	CALL	params_set_volume

	RET ; no vibrato

.VIB_VOL_UP:

	; volume up

	SRL	A
	SRL	A
	SRL	A
	SRL	A

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)
	CP	VOLUME_MAX
	JR	Z,CMD_VIBRATO ; A = MAX -> skip

	ADD	B

	CP	VOLUME_MAX
	JR	NC,.VIP_VOL_DONE ; A <= MAX -> done

	LD	A,VOLUME_MAX ; set to MAX

.VIP_VOL_DONE:

	; store new volume
	LD	(_S85_CHX_VOLUME),A

	; set new volume
	CALL	params_set_volume

CMD_VIBRATO:

	; select vib table
	LD	A,(VIBRATO_DEPTH)
	LD	B,A
	LD	D,0
	LD	E,32
	LD	HL,VIBTAB
	JR	.VIBRATO_LOOP_START

.VIBRATO_LOOP:
	ADD	HL,DE
.VIBRATO_LOOP_START:
	DJNZ	.VIBRATO_LOOP

	; lookup value
	LD	A,(VIBRATO_INDEX)
	LD	C,32
	AND	31	; 0..31
	LD	B,0
	LD	C,A
	ADD	HL,BC
	LD	A,(HL)

	; add value
	LD	B,A
	LD	A,(_S85_CHX_PERIOD)

	LD	HL,VIBRATO_INDEX
	BIT	5,(HL)
	JR	NZ,.VIBRATO_NEG

	ADD	B
	JR	.VIBRATO_CONT

.VIBRATO_NEG:
	SUB	B

.VIBRATO_CONT:
	LD	B,A ; store period

	; check bounds
	LD	A,(_S85_CHX_VORTEILER)
	LD	H,A	; VT -> H
	AND	A	; VT = 0?
	JR	NZ,.VIBRATO_VT_1

	; ----- VT = 0 -----
	LD	A,B	; period -> A
	CP	27
	JR	NC,.VIBRATO_VT_0_CHK2

	LD	A,27
	JR	.VIBRATO_VT_CONT

.VIBRATO_VT_0_CHK2:

	CP	229
	JR	C,.VIBRATO_VT_CONT

	LD	H,1	; VT = 1
	LD	A,15
	JR	.VIBRATO_VT_CONT

	; ----- VT = 1 -----
.VIBRATO_VT_1:
	LD	A,B	; period -> A
	CP	15
	JR	NC,.VIBRATO_VT_1_CHK2

	LD	H,0	; VT = 0
	LD	A,229
	JR	.VIBRATO_VT_CONT

.VIBRATO_VT_1_CHK2:
	CP	204
	JR	C,.VIBRATO_VT_CONT

	LD	A,204
	JR	.VIBRATO_VT_CONT

.VIBRATO_VT_CONT:
	LD	L,A	; ZK
	CALL	params_set_period

	; next index
	LD	A,(VIBRATO_SPEED)
	LD	B,A
	LD	A,(VIBRATO_INDEX)
	ADD	B
	AND	63	; 0..63

	; store new index
	LD	(VIBRATO_INDEX),A

	RET

VIBRATO_SPEED:
	DEFB	0
VIBRATO_DEPTH:
	DEFB	0
VIBRATO_INDEX:
	DEFB	0
VIBRATO_SPEED_CH1:
	DEFB	0
VIBRATO_DEPTH_CH1:
	DEFB	0
VIBRATO_INDEX_CH1:
	DEFB	0
VIBRATO_SPEED_CH2:
	DEFB	0
VIBRATO_DEPTH_CH2:
	DEFB	0
VIBRATO_INDEX_CH2:
	DEFB	0

CMD_VOLUME_SLIDE:

	; check for slide down
	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0F0H
	JR	NZ,.VOL_UP

	LD	A,(_S85_CHX+_S85_PARAM)
	AND	0FH
	RET	Z

	; volume down

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)
	AND	A
	RET	Z	; A = 0 -> skip

	SUB	B

	JR	NC,.VOL_DONE ; A >= 0 -> done

	XOR	A ; set to 0

	JR	.VOL_DONE

.VOL_UP:

	; volume up

	SRL	A
	SRL	A
	SRL	A
	SRL	A

	LD	B,A

	LD	A,(_S85_CHX_VOLUME)
	CP	VOLUME_MAX
	RET	Z ; A = MAX -> skip

	ADD	B

	CP	VOLUME_MAX+1
	JR	C,.VOL_DONE ; A <= MAX -> done

	LD	A,VOLUME_MAX ; set to MAX

.VOL_DONE:

	; store new volume
	LD	(_S85_CHX_VOLUME),A

	; set new volume
	CALL	params_set_volume

	RET

; ---------- source includes ----------

	INCLUDE "lib/audio.s"
	INCLUDE "lib/interrupt.s"

	INCLUDE "synth85/params.s"
	INCLUDE "synth85/module.s"

	IF SYNTH85_MASTER_VOLUME != 0

_S85_MASTER_VOLUME_TABLE:
	INCLUDE "synth85/volume_table.s"

	ENDIF

; ---------- static data ----------

_S85_SPEED:	DEFB	6	; steps per note

_S85_TICK:
	DEFB	0	; current tick

	IF SYNTH85_CHANNEL_MUTE != 0
SYNTH85_CH1_MUTED:
	DEFB	0

SYNTH85_CH2_MUTED:
	DEFB	0
	ENDIF

; current period (for effects)
_S85_CH1_PERIOD:
	DEFB	0
_S85_CH2_PERIOD:
	DEFB	0
_S85_CHX_PERIOD:
	DEFB	0

; current Vorteiler (for effects)
_S85_CH1_VORTEILER:
	DEFB	0
_S85_CH2_VORTEILER:
	DEFB	0
_S85_CHX_VORTEILER:
	DEFB	0

; current volume (for effects, 0 -> note off)
_S85_CH1_VOLUME:
	DEFB	0
_S85_CH2_VOLUME:
	DEFB	0
_S85_CHX_VOLUME:
	DEFB	0

; current note (value, effect, param)
_S85_CH1:
	DEFM	0,0,0
_S85_CH2:
	DEFM	0,0,0
_S85_CHX:
	DEFM	0,0,0

_S85_VOLUME_FLAG:
	DEFB	0

; master volume
	IF SYNTH85_MASTER_VOLUME != 0

; 0..15 (value in upper tetrade)
_S85_MASTER_VOLUME:
	DEFB	0

	ENDIF

; original isr
OLDINT:	DEFW	0
SYNTH85_END:
	DEFB	1	; Player Stop

SYNTH85_NOTIFY:
	DEFB	0

; jump table
COMMAND_TABLE:
	DEFW	CMD_ARPEGGIO		; 0 Arpeggio
	DEFW	CMD_SLIDE_UP		; 1 Slide up
	DEFW	CMD_SLIDE_DOWN		; 2 Slide down
	DEFW	CMD_SLIDE_TO_NOTE	; 3 Tone Portamento
	DEFW	CMD_VIBRATO		; 4 Vibrato
	DEFW	CMD_STN_AND_VOL_SLIDE	; 5 Tone Portamento + Volume Slide
	DEFW	CMD_VIB_AND_VOL_SLIDE	; 6 Vibrato + Volume Slide
	DEFW	CMD_DUMMY		; 7 Tremolo
	DEFW	CMD_DUMMY		; 8 ???
	DEFW	CMD_DUMMY		; 9 Set Sample Offset
	DEFW	CMD_VOLUME_SLIDE	; A Volume Slide
	DEFW	CMD_DUMMY		; B Position Jump (immediate)
	DEFW	CMD_DUMMY		; C Set Volume (immediate)
	DEFW	CMD_DUMMY		; D Pattern Break (immediate)
	DEFW	CMD_DUMMY		; E Exx Commands
	DEFW	CMD_DUMMY		; F Set Speed (immediate)

	; periods table
PERIODS:
	; Oktave 1
	DEFB	204,199,196,181,173,160
	DEFB	152,144,137,127,118,116

	; Oktave 2
	DEFB	102,97,95,90,86,81
	DEFB	75,72,68,64,60,58

	; Oktave 3
	DEFB	54,50,48,45,43,40
	DEFB	38,36,34,32,30,29

	; Oktave 4
	DEFB	27,25,24,23,21,20
	DEFB	19,18,17,16,15

PERIODS_VTE:

	DEFB	229

	; Oktave 5
	DEFB	216,204,192,182,171,162
	DEFB	153,144,136,128,120,114

	; Oktave 6
	DEFB	108,102,96,91,86,81
	DEFB	76,72,68,64,60,57

	; Oktave 7
	DEFB	54,51,48,45,43,40
	DEFB	38,36,34,32,30,29

	; Oktave 8
	DEFB	27

PERIODS_END:

	; vibrato table (half period)
VIBTAB:

; depth 1
	DEFM	0,0,0,0,0,0,1,1
	DEFM	1,1,1,1,1,1,1,1
	DEFM	1,1,1,1,1,1,1,1
	DEFM	1,1,1,0,0,0,0,0

; depth 2
	DEFM	0,0,0,1,1,1,2,2
	DEFM	2,3,3,3,3,3,3,3
	DEFM	3,3,3,3,3,3,3,3
	DEFM	2,2,2,1,1,1,0,0

; depth 3
	DEFM	0,0,1,1,2,2,3,3
	DEFM	4,4,4,5,5,5,5,5
	DEFM	5,5,5,5,5,5,4,4
	DEFM	4,3,3,2,2,1,1,0

; depth 4
	DEFM	0,0,1,2,3,3,4,5
	DEFM	5,6,6,7,7,7,7,7
	DEFM	7,7,7,7,7,7,6,6
	DEFM	5,5,4,3,3,2,1,0

; depth 5
	DEFM	0,0,1,2,3,4,5,6
	DEFM	7,7,8,8,9,9,9,9
	DEFM	9,9,9,9,9,8,8,7
	DEFM	7,6,5,4,3,2,1,0
; depth 6
	DEFM	0,1,2,3,4,5,6,7
	DEFM	8,9,9,10,11,11,11,11
	DEFM	11,11,11,11,11,10,9,9
	DEFM	8,7,6,5,4,3,2,1
; depth 7
	DEFM	0,1,2,4,5,6,7,8
	DEFM	9,10,11,12,12,13,13,13
	DEFM	13,13,13,13,12,12,11,10
	DEFM	9,8,7,6,5,4,2,1
; depth 8
	DEFM	0,1,3,4,6,7,8,10
	DEFM	11,12,13,14,14,15,15,15
	DEFM	15,15,15,15,14,14,13,12
	DEFM	11,10,8,7,6,4,3,1
; depth 9
	DEFM	0,1,3,5,6,8,9,11
	DEFM	12,13,14,15,16,17,17,17
	DEFM	17,17,17,17,16,15,14,13
	DEFM	12,11,9,8,6,5,3,1
; depth 10
	DEFM	0,1,3,5,7,9,11,12
	DEFM	14,15,16,17,18,19,19,19
	DEFM	19,19,19,19,18,17,16,15
	DEFM	14,12,11,9,7,5,3,1
; depth 11
	DEFM	0,2,4,6,8,10,12,13
	DEFM	15,16,18,19,20,20,21,21
	DEFM	21,21,21,20,20,19,18,16
	DEFM	15,13,12,10,8,6,4,2
; depth 12
	DEFM	0,2,4,6,9,11,13,15
	DEFM	16,18,19,21,22,22,23,23
	DEFM	23,23,23,22,22,21,19,18
	DEFM	16,15,13,11,9,6,4,2
; depth 13
	DEFM	0,2,4,7,9,12,14,16
	DEFM	18,20,21,22,23,24,25,25
	DEFM	25,25,25,24,23,22,21,20
	DEFM	18,16,14,12,9,7,4,2
; depth 14
	DEFM	0,2,5,8,10,13,15,17
	DEFM	19,21,23,24,25,26,27,27
	DEFM	27,27,27,26,25,24,23,21
	DEFM	19,17,15,13,10,8,5,2
; depth 15
	DEFM	0,2,5,8,11,14,16,18
	DEFM	21,23,24,26,27,28,29,29
	DEFM	29,29,29,28,27,26,24,23
	DEFM	21,18,16,14,11,8,5,2
