let canvas, ctx, canvasBuffer
let showDebugInfo = false
let doPerf = false
const NUM_BUFFERS = 2
let greyscaleBuffers = []
let bufferLocks = []
let outputBufferId

const bpm = sync.globals.BeatsPerMin
const ticksPerLine = sync.globals.TicksPerLine
const linesPerBeat = sync.globals.LinesPerBeat
const syncs = sync.patterns['beat']

const msPerBeat = (1000*60)/bpm
const msPerLine = msPerBeat / linesPerBeat

const instrumentIds = {
    // name: [instrumentId, noteLength, decaySpeed, generatedNoteData]
    'kick':     [2, 20, 0.1, []],
    'distort':  [3, 35, 0.01, []],
    'distort2': [4, 35, 0.01, []],
    'distort3': [5, 35, 0.01, []],
    'snare':    [6, 20, 0.1, []],
}
Object.values(instrumentIds).forEach(instrumentData => {
    const instrumentId = instrumentData[0]
    const noteLength = instrumentData[1]
    const noteDecay = instrumentData[2]

    let lastNoteOnSyncCursor = null
    let value = 0
    let noteOnLength
    for (let i = 0; i < 200*300; ++i) {
        const ms = (i * 10) * (60 / bpm)
        const syncCursor = Math.floor(ms / msPerLine)
        if (syncCursor !== lastNoteOnSyncCursor) {
            lastNoteOnSyncCursor = syncCursor
            const currentLine = syncs[syncCursor]
                //console.log(currentLine)
            if (currentLine && currentLine.notes && currentLine.notes[0].Instrument === instrumentId) {
                noteOnLength = 0
                value = 1
            }
        } else {
            if (noteOnLength++ > noteLength) {
                value = value - noteDecay
            }
        }
        instrumentData[3].push(Math.max(0, Math.min(1, value)))
    }
})
function readInstrumentSync (instrumentName) {
    const instrumentData = instrumentIds[instrumentName]
    const time = Math.floor(currentFrameTime * 100)
    const value = instrumentData[3][time]
    return value
}
{
    instrumentIds['rewind'] = [0, 0, 0, []]
    instrumentIds['!rewind'] = [0, 0, 0, []]
    for (let i = 0; i < 200*300; ++i) {
        const beat = i / 100
        const value = beat >= 340 && beat <  424 ? 1 : 0
        instrumentIds['rewind'][3].push(value)
        instrumentIds['!rewind'][3].push(1-value)
    }
}

const colourFields = []
{
    const colourSources = [
        [
            // Tweak these four hsl values..
            [0.0, 0.5, 0.2], [0.0, 0.6, 0.7],
            [0.0, 0.4, 0.4], [0.3, 0.3, 0.3],
        ],
        [
            // Tweak these four hsl values..
            [0.3, 0.6, 0.4], [0.4, 0.6, 0.3],
            [0.6, 0.6, 0.3], [0.6, 0.8, 0.3],
        ],
        [
            // Tweak these four hsl values..
            [0.7, 0.8, 0.6], [0.7, 0.9, 0.7],
            [0.6, 0.6, 0.3], [0.7, 0.8, 0.3],
        ]
    ]
    colourSources.forEach(colourSource => {
        let colourField = new Float32Array(640*480*3)
        let writeIndex = 0
        for (var y = 0; y < 480; ++y) {
            const ay = y / (480-1)
            for (var x = 0; x < 640; ++x) {
                const ax = x / (640-1)
                const h = lerp(lerp(colourSource[0][0], colourSource[1][0], ax), lerp(colourSource[2][0], colourSource[3][0], ax), ay)
                const s = lerp(lerp(colourSource[0][1], colourSource[1][1], ax), lerp(colourSource[2][1], colourSource[3][1], ax), ay)
                const l = lerp(lerp(colourSource[0][2], colourSource[1][2], ax), lerp(colourSource[2][2], colourSource[3][2], ax), ay)
                const [r, g, b] = hslToRgb(h, s, l)
                colourField[writeIndex++] = r * 2
                colourField[writeIndex++] = g * 2
                colourField[writeIndex++] = b * 2
            }
        }
        colourFields.push(colourField)
    })
}

function lerp(a, b, alpha) {
    return a + (b - a) * alpha
}

let audio;
const greets = []
const overlays = {}
const titleScreenImage = new Image()
titleScreenImage.src = `gfx/loading.png`

let transport = null
let requestAnimationFrameId = null

function requestUpdate () {
    window.cancelAnimationFrame(requestAnimationFrameId)
    requestAnimationFrameId = window.requestAnimationFrame(update)
}

function startPressed () {
    transport.setIsPlaying(true)
    requestUpdate()
}

function degToRad (degrees) {
    return degrees * (Math.PI/180)
}

const maxAssets = window.overlays.length  + window.greets.length + 1 // +1 for music
let totalLoaded = 0;

function load () {
    console.log('                ___  ___                  ________                  _____')
    console.log('      __ _____  \\ ///   \\  _/\\__ ____ ___/_  _    \\_ _______    __ /    /_')
    console.log('  ____\\// ___ \\--\\/-\\    \\//    X    \\\\    \\ \\\\____//   __  \\---\\//   ___ \\')
    console.log(' /  __ \\  X/   \\     \\    \\   _/_\\    \\\\_   \\ \\ ___/_   X/   \\ __ \\   X/   \\')
    console.log('/   \\X     \\    \\     \\    ____   \\    X/   /  X     \\  /     \\\\\\      \\_   \\')
    console.log('\\     \\____/\\____\\____/\\    X/____/\\_______/_________/________/  \\------/____>.')
    console.log(' \\_____\\!NE7            \\____\\                              \\_____\\')

    const params = new URLSearchParams(window.location.search)
    if (params.has('capture')) {
        const background = document.getElementById('background')
        background.style.backgroundColor = "#888"
    }
    
    
    const loaded = () => totalLoaded += 1

    audio = new Audio('./audio/kaneel_-_romucomu_demover.mp3')
    audio.addEventListener("canplaythrough", loaded)
    audio.load()

    window.overlays.forEach(x => {
        const image = new Image()
        image.onload = () => {
            loaded()
        }
        image.onerror = () => {
            throw new Error(`Failed to load overlay: ${x}`)
        }
        image.src = `gfx/${x}.png`
        overlays[x] = image
    })

    window.greets.map(( src, i ) => {
        const greet = []
        const greetImg = new Image()

        greetImg.onload = () => {
            const width = greetImg.width
            const height = greetImg.height
            const c  = document.createElement('canvas')
            c.width = width
            c.height = height
            
            const greetCtx = c.getContext('2d')
            greetCtx.drawImage(greetImg, 0, 0);
            const id = greetCtx.getImageData(0, 0, width, height).data
            let index = 0
            for (let y = 0; y < height; ++y) {
                let line = []
                for (x = 0; x < width; ++x) {
                    line.push(id[index] / 255)
                    index += 4
                }
                while (line.length && line[line.length - 1] === 0) {
                    line.pop()
                }
                greet.push(line)
            }
            greets[i] = greet
            loaded()
        }

        greetImg.onerror = () => {
            throw new Error(`Failed to load greet image ${src}`)
        }

        greetImg.src = src
    })

    let loading = () => {
        fitScreen()

        ctx.fillStyle = '#000'
        ctx.fillRect(73, 0, 494, 480)
        ctx.fillStyle = '#fff'
        ctx.fillRect(73, 0, (totalLoaded/maxAssets) * 494, 480)

        ctx.drawImage(titleScreenImage, 0, 0)

        if (totalLoaded < maxAssets) {
            window.requestAnimationFrame(loading)
        } else {
            const params = new URLSearchParams(window.location.search)
            if (params.has('dev')) {
                main()
            } else {
                return setTimeout(() => main(true), 1000)
            }
        }
    }
    console.log('Loading..')
    loading()
}


window.onresize = () => { fitScreen() }
canvas = document.getElementById('canvas')
ctx = canvas.getContext('2d', { alpha: false })
canvasBuffer = ctx.createImageData(640, 480)
for (var i = 0; i < canvasBuffer.data.length; ++i) {
    canvasBuffer.data[i] = 255
}

function main () {
    console.log('Loaded!')

    console.log("              ,._   _.,         ____")
    console.log(" __ ________ xXXXx xXXXx ____  /    \\  _________ __ _________  ____ __ ____")
    console.log(" \\/  ___    XXXXXXXXXXXXX.   \\/ ___  \\/         \\\\//   ___   \\/    \\\\/___  \\")
    console.log(" |    X/  __/XXXXXXXXXXXX!  \\    X/   \\    \\____/__     X/    \\   ___  X/   \\")
    console.log(" |     \\      `XXXXXXXXXX    \\    \\    \\    \\      \\           \\   X/   \\    \\")
    console.log(" |_____/\\       ^XXXXX!NE7___/\\___/\\___/\\__________/\\__________/___/\\___/\\___/.")
    console.log("         \\_____/ `6X^'")
    console.log("                  '")

    fitScreen()

    const params = new URLSearchParams(window.location.search)

    if (params.has('nomusic')) {
        transport = new Transport(60 * 4)
    } else {
        transport = new TransportMp3(audio, 76)
    }

    if (params.has('time')) {
        const time = parseInt(params.get('time')) * 60 / bpm
        transport.seekExact(time)
    }

    if (params.has('controls')) {
        document.addEventListener('keydown', (event) => {
            if (event.key === ' ') {
                transport.setIsPlaying(!transport.getIsPlaying())
                requestUpdate()
            } else if (event.key === 'ArrowUp' ) {
                transport.seekExact(0)
                requestUpdate()
            } else if (event.key === 'ArrowLeft' ) {
                transport.seekDelta(-(event.shiftKey ? 1 / 60 : 1))
                requestUpdate()
            } else if (event.key === 'ArrowRight' ) {
                transport.seekDelta(event.shiftKey ? 1 / 60 : 1)
                requestUpdate()
            } else if (event.key === 'f') {
                openFullscreen()
            }
        })
    } else {
        document.addEventListener('keydown', (event) => {
            if (event.key === ' ') {
                transport.setIsPlaying(!transport.getIsPlaying())
                requestUpdate()
            } else if (event.key === 'f') {
                openFullscreen()
            }
        })
        document.addEventListener('click', () => {
            transport.setIsPlaying(true)
            requestUpdate()
        })
        canvas.addEventListener('touchstart', () => {
            transport.setIsPlaying(true)
            openFullscreen()
            requestUpdate()
        })
    }

    if (params.has('dev')) {
        showDebugInfo = true        
    } else {
        const el = document.getElementById('debugStuff')
        el.style = "display:none"
    }

    if (params.has('perf')) {
        doPerf = true
    }

    for (let i = 0; i < NUM_BUFFERS; ++i) {
        greyscaleBuffers[i] = new Float32Array(640 * 480)
        for (let index = 0; index < 640*480; ++index) greyscaleBuffers[i][index] = 1
        bufferLocks[i] = false
    }

    ctx.drawImage(overlays.opening, 0, 0)
    if (showDebugInfo) {
        transport.setIsPlaying(false)
        if (params.has('time')) {
            update()
        }
    } 
}

const textures = {
    default: {
        size: 2,
        data: [1,0,0,1]
    }
}

const TUNNEL_TEX_SIZE = 40
const tunnelTex = new Float32Array(TUNNEL_TEX_SIZE * TUNNEL_TEX_SIZE)
{
    let index = 0
    for (let y = 0; y < TUNNEL_TEX_SIZE; ++y) {
        const dy = (y + 0.5) - (TUNNEL_TEX_SIZE / 2)
        const dys = dy * dy
        for (let x = 0; x < TUNNEL_TEX_SIZE; ++x) {
            const dx = (x + 0.5) - (TUNNEL_TEX_SIZE / 2)
            const dxs = dx * dx
            const d = Math.sqrt(dxs + dys)
            let v = Math.max(1 - (d / TUNNEL_TEX_SIZE * 1.25), 0)
            if (y === 0 || y === TUNNEL_TEX_SIZE-1 ||
                x === 0 || x === TUNNEL_TEX_SIZE-1) {
                v = 0
            }
            tunnelTex[index++] = v * 1
        }
    }
}
textures['square'] = {
    size: TUNNEL_TEX_SIZE,
    data: tunnelTex
}


const BALL_SIZE = 13
const ball = new Float32Array(BALL_SIZE * BALL_SIZE)
{
    let index = 0
    for (let y = 0; y < BALL_SIZE; ++y) {
        const dy = (y + 0.5) - (BALL_SIZE / 2)
        const dys = dy * dy
        for (let x = 0; x < BALL_SIZE; ++x) {
            const dx = (x + 0.5) - (BALL_SIZE / 2)
            const dxs = dx * dx
            const d = Math.sqrt(dxs + dys)
            const v = Math.max(1 - (d / BALL_SIZE * 2.5), 0)
            ball[index++] = v
        }
    }
}
textures['ball'] = {
    size: BALL_SIZE,
    data: ball
}


const HOOP_SIZE = 15
const hoop = new Float32Array(HOOP_SIZE * HOOP_SIZE)
{
    let index = 0
    for (let y = 0; y < HOOP_SIZE; ++y) {
        const dy = (y + 0.5) - (HOOP_SIZE / 2)
        const dys = dy * dy
        for (let x = 0; x < HOOP_SIZE; ++x) {
            const dx = (x + 0.5) - (HOOP_SIZE / 2)
            const dxs = dx * dx
            const d = Math.sqrt(dxs + dys)
            let v = (Math.max(1 - (d / HOOP_SIZE * 2.5), 0))
            v = 1 - Math.pow((Math.abs(v - 0.5) + 0.5), 0.5)
            hoop[index++] = v
        }
    }
}
textures['hoop'] = {
    size: HOOP_SIZE,
    data: hoop
}


const BALL_COUNT = 1000
const balls = []
const ballRng = new Math.seedrandom('balls')
for (let b = 0; b < BALL_COUNT; ++b) {
    balls.push(randomNormal(ballRng))
}

function moduleNoise (frameTime, config, syncs) {
    const strength = readAtTime(frameTime, config.strength, syncs.strength, 'float', 1)
    const mode = readAtTime(frameTime, config.mode, syncs.mode, 'string', 'absolute')

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    if (mode === 'absolute') {
        for (let i = 0; i < 640 * 480; ++i) {
            greyscaleBuffer[i] = Math.random() * strength
        }
    } else if (mode === 'additive') {
        for (let i = 0; i < 640 * 480; ++i) {
            greyscaleBuffer[i] += Math.random() * strength
        }
    }
}

const octaveData = [
   [0.030, 4.5,12.34, 0.5],
   [0.001, 2.22,1.99, 0.3],
   [0.123, 5.7,11.17, 0.2],
]
const noiseRng = new Math.seedrandom('noise')
const noiseData = new Float32Array([...Array(1000).keys()].map(x => noiseRng() * 2 - 1))
function moduleDistort (frameTime, config, syncs) {
    const strength =  readAtTime(frameTime, config.strength, syncs.strength, 'float', 1)
    const magnetDistance =  readAtTime(frameTime, config.magnetDistance, syncs.magnetDistance, 'float', 0)
    const distortDistance = readAtTime(frameTime, config.distance, syncs.distance, 'float', 20)
    const distortTime = readAtTime(frameTime, config.distortTime, syncs.distortTime, 'float', 0)
    const distortBrightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 0.5)

    if (strength > 0.1) {
        const srcBufferId = getOutputBufferId()
        const dstBufferId = lockBuffer()
        const srcBuffer = greyscaleBuffers[srcBufferId]
        const dstBuffer = greyscaleBuffers[dstBufferId]

        const magnetTime = frameTime * 1.57
        const magnetTimeAlpha = magnetTime % 1
        const magnetStrengthA = noiseData[Math.floor(frameTime) % noiseData.length]
        const magnetStrengthB = noiseData[(Math.floor(frameTime) + 1) % noiseData.length]
        const magnetStrength = (magnetStrengthA + (magnetStrengthB - magnetStrengthA) * magnetTimeAlpha) * 0.5 + 0.5

        let writeIndex = 0
        for (let y = 0; y < 480; ++y) {
            const brightness = 1 +
                Math.sin(-distortTime*1.21 + y*0.012)*distortBrightness*0.75 + 
                Math.sin(+distortTime*1.53 + y*0.003)*distortBrightness*0.25

            const readLine = 640 * y
            let distortValue = 0
            for (let i = 0; i < octaveData.length; ++i) {
                const octave = octaveData[i]
                distortValue += noiseData[Math.floor(y*octave[0] + (frameTime+octave[1])*octave[2]) % noiseData.length] * octave[3]
            }
            distortValue *= distortDistance
            distortValue += Math.pow(1-(y/480), magnetStrength * 10 + 2) * magnetDistance * magnetStrength
            distortValue *= strength
            distortValue = Math.floor(distortValue)
            if (distortValue > 0) {
                let x = 0
                const value = srcBuffer[readLine] * brightness
                for (; x < distortValue; ++x) {
                    dstBuffer[writeIndex++] = value
                }
                let readIndex = 0
                for (; x < 640; ++x) {
                    dstBuffer[writeIndex++] = srcBuffer[readLine + readIndex++] * brightness
                }
            } else {
                let readIndex = -distortValue
                let x = 0
                let value
                for (; x < 640 + distortValue; ++x) {
                    value = srcBuffer[readLine + readIndex++] * brightness
                    dstBuffer[writeIndex++] = value
                }
                for (; x < 640; ++x) {
                    dstBuffer[writeIndex++] = value
                }
            }
        }

        unlockBuffer(srcBufferId)
        setOutputBufferId(dstBufferId)
    }
}

function drawBallsNegative(balls, textureName) {
    const texture = textures[textureName]
    const texSize = texture.size
    const texData = texture.data

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    const halfBallSize = texSize / 2

    for (let i = 0; i < balls.length; ++i) {
        const [x, y, brightness, size] = balls[i]

        const range = Math.floor(texSize * size)
        const halfRange = Math.floor(range / 2)

        if (x+halfBallSize < 0 || x-halfBallSize > 640 ||
            y+halfBallSize < 0 || y-halfBallSize > 480 ||
            size <= 0) {
                continue
        }

        const iHalfRangeTimeHalfBallSize = 1 / halfRange

        for (let py = -halfRange; py <= +halfRange; ++py) {
            if (y+py >= 0 && y+py < 480) {
                let ty = Math.floor(py * iHalfRangeTimeHalfBallSize * halfBallSize + halfBallSize)
                //ty = Math.min(Math.max(ty, 0), texSize - 1)
                ty = Math.min(ty, texSize-1)
                const dy = (y + py) * 640
                for (let px = -halfRange; px <= +halfRange; ++px) {
                    if (x+px >= 0 && x+px < 640) {
                        let tx = Math.floor(px * iHalfRangeTimeHalfBallSize * halfBallSize + halfBallSize)
                        //tx = Math.min(Math.max(tx, 0), texSize - 1)
                        tx = Math.min(tx, texSize-1)
                        greyscaleBuffer[(x + px) + dy] = Math.max(0, greyscaleBuffer[(x + px) + dy] - texData[tx + ty * texSize] * brightness)
                    }
                }
            }
        }
    }
}

function transformBalls(balls3d) {
    const balls2d = []
    balls3d.forEach(ball => {
        const [pos, brightness, size] = ball
        const v = m4.transformVector(camViewPersp, pos)
        if (v[2] > 1) {
            const w = v[3]
            const scaledSize = 30*size/w
            if (scaledSize > 0) {
                balls2d.push([Math.floor(v[0]*320/w) + 320, Math.floor(v[1]*240/w) + 240, scaledSize, brightness])
            }
        }
    })
    return balls2d
}

function drawBalls(balls, textureName) {
    const texture = textures[textureName]
    const texSize = texture.size
    const texData = texture.data

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    const halfBallSize = texSize / 2
    for (let i = 0; i < balls.length; ++i) {
        let [x, y, size, brightness] = balls[i]
        if (size > 0.2) {
            brightness = Math.max(0, brightness)

            const range = Math.floor(texSize * size)
            const halfRange = Math.floor(range / 2)
            const iHalfRangeTimesHalfBallSize = 1 / halfRange * halfBallSize
            for (let py = -halfRange; py <= +halfRange; ++py) {
                if (y+py >= 0 && y+py < 480) {
                    let ty = Math.floor(py * iHalfRangeTimesHalfBallSize + halfBallSize)
                    ty = Math.min(Math.max(ty, 0), texSize - 1)
                    const dy = (y + py) * 640
                    for (let px = -halfRange; px <= +halfRange; ++px) {
                        if (x+px >= 0 && x+px < 640) {
                            let tx = Math.floor(px * iHalfRangeTimesHalfBallSize + halfBallSize)
                            tx = Math.min(Math.max(tx, 0), texSize - 1)
                            greyscaleBuffer[(x + px) + dy] += texData[tx + ty * texSize] * brightness
                        }
                    }
                }
            }
        }
    }
}

function moduleDotSphere (frameTime, config, syncs) {
    const offsetSize = readAtTime(frameTime, config.offsetSize, syncs.offsetSize, 'float', 20)
    const innerSize = readAtTime(frameTime, config.innerSize, syncs.innerSize, 'float', 50)
    const offsetPosition = readAtTime(frameTime, config.offsetPosition, syncs.offsetPosition, 'float', 0)
    const brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 1)
    const spriteSize = readAtTime(frameTime, config.spriteSize, syncs.spriteSize, 'float', 1)
    const textureName = readAtTime(frameTime, config.texture, syncs.texture, 'string', 'default')

    const balls3d = []
    for (let b = 0; b < BALL_COUNT; ++b) {
        const ba = b / BALL_COUNT
        const of = Math.pow(Math.max(0, Math.sin(offsetPosition + ba * Math.PI * 2)), 4)
        const bd = innerSize + of * offsetSize
        const dotSize = 1 - of
        let [sx, sy, sz] = balls[b]
        sx *= bd
        sy *= bd
        sz *= bd
        balls3d.push([[sx, sy, sz, 1], brightness, spriteSize * dotSize])
    }

    const balls2d = transformBalls(balls3d)
    drawBalls(balls2d, textureName)
}

const tendrils3D = []
{
    const tendrilRng = new Math.seedrandom('tendril')
    for (let tendril = 0; tendril < 50; ++tendril) {
        let pos = [0, 0, 0]
        let dir = randomNormal(tendrilRng)
        let startDistance = tendrilRng() * 20
        pos[0] += dir[0] * startDistance
        pos[1] += dir[1] * startDistance
        pos[2] += dir[2] * startDistance
        let tendrilLength = tendrilRng() * 20 + 90
        for (let i = 0; i < tendrilLength; ++i) {
            const a = 1 - (i / tendrilLength)
            const as = Math.sqrt(a)
            tendrils3D.push([[pos[0], pos[1], pos[2], 1], as*1+0.5, as*1.5+1, a * 2 * Math.PI])
            pos[0] += dir[0] * 1
            pos[1] += dir[1] * 1
            pos[2] += dir[2] * 1

            let dirMod = randomNormal(tendrilRng)
            dir[0] += dirMod[0] * 0.4 * as
            dir[1] += dirMod[1] * 0.4 * as
            dir[2] += dirMod[2] * 0.4 * as
            dir = m4.normalize(dir)
        }
    }
}

function moduleTendrils (frameTime, config, syncs) {
    const textureName = readAtTime(frameTime, config.texture, syncs.texture, 'string', 'default')
    const brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 1)
    const size = readAtTime(frameTime, config.size, syncs.size, 'float', 1)
    const pulsePosition = readAtTime(frameTime, config.pulsePosition, syncs.pulsePosition, 'float', 1)
    const pulseSize = readAtTime(frameTime, config.pulseSize, syncs.pulseSize, 'float', 1)

    const balls2d = transformBalls(tendrils3D.map(x => {
        x = x.slice()
        x[1] *= brightness
        const pulseMod = Math.abs(Math.sin(x[3] + pulsePosition))
        x[2] = x[2] * (size + pulseSize * pulseMod*pulseMod)
        return x
    }))
    drawBalls(balls2d, textureName)
}
    

const DUST_COUNT = 2000
const dust = []
const dustRnd = new Math.seedrandom('dust')
for (let i = 0; i < DUST_COUNT; ++i) {
    const x = dustRnd() * 700
    const y = dustRnd() * 700
    const s = dustRnd() * 4 + 2
    const a = dustRnd() * 2 * Math.PI
    dust.push([x, y, s, a])
}
function moduleDust (frameTime, config, syncs) {
    const brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 1)
    const size = readAtTime(frameTime, config.size, syncs.size, 'float', 1)
    const amount = readAtTime(frameTime, config.amount, syncs.amount, 'float', 1)
    const wobble = readAtTime(frameTime, config.wobble, syncs.wobble, 'float', 1)
    const position = readAtTime(frameTime, config.position, syncs.position, 'float2', [0, 0])
    const textureName = readAtTime(frameTime, config.texture, syncs.texture, 'string', 'default')

    const balls = []
    for (let i = 0; i < DUST_COUNT * amount; ++i) {
        const [x, y, s, a] = dust[i]
        const ox = Math.sin(a + wobble) * 40
        const oy = Math.cos(a + wobble) * 40
        const fx = Math.floor((x + ox + position[0] + 10000) % 800)
        const fy = Math.floor((y + oy + position[1] + 10000) % 800)
        balls.push([fx, fy, brightness, s * size])
    }
    drawBallsNegative(balls, textureName)
}

let camViewPersp = m4.identity()
function moduleOrbitCam (frameTime, config, syncs) {
    const dist = readAtTime(frameTime, config.distance, syncs.distance, 'float', 70)
    const roll = degToRad(readAtTime(frameTime, config.roll, syncs.roll, 'float', 0))
    const rotation = degToRad(readAtTime(frameTime, config.rotation, syncs.rotation, 'float', 0))
    const fov = degToRad(readAtTime(frameTime, config.fov, syncs.fov, 'float', 90))
    const lookAt = readAtTime(frameTime, config.lookAt, syncs.lookAt, 'float3', [0, 0, 0])

    const camera = m4.lookAt([lookAt[0] + Math.sin(rotation) * dist, lookAt[1], lookAt[2] + Math.cos(rotation) * dist], lookAt, [0, 1, 0])
    const rollMat = m4.zRotation(roll)
    const view = m4.multiply(m4.inverse(camera), rollMat)
    const perspective = m4.perspective(fov, 640/480, 1, 500)
    camViewPersp = m4.multiply(perspective, view)
}

function moduleDotCube (frameTime, config, syncs) {
    const halfSize = readAtTime(frameTime, config.width, syncs.width, 'int', 1)
    const spacing = readAtTime(frameTime, config.spacing, syncs.spacing, 'float', 10)
    let brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 0.2)
    let dotSize = readAtTime(frameTime, config.dotSize, syncs.dotSize, 'float', 0.5)
    const warpPos = readAtTime(frameTime, config.warpPos, syncs.warpPos, 'float3', 0.0)

    const balls3d = []
    for (let x = -halfSize; x <= halfSize; ++x) {
        const sx = x * spacing
        for (let y = -halfSize; y <= halfSize; ++y) {
            const sy = y * spacing
            for (let z = -halfSize; z <= halfSize; ++z) {
                const sz = z * spacing
                const size = 1 + (Math.abs(Math.sin(warpPos[0] + y*0.1)) * 0.5 + Math.abs(Math.sin(warpPos[1] + x*0.1)) * 0.5 + Math.abs(Math.sin(warpPos[2] + z*0.1)) * 0.5)
                balls3d.push([[sx, sy, sz, 1], brightness, size * dotSize])
            }
        }
    }

    const balls2d = transformBalls(balls3d)
    drawBalls(balls2d, 'ball')
}

let greetsBuffer1 = new Float32Array(641*481)
let greetsBuffer2 = new Float32Array(641*481)
for (let i = 0; i < 641*481; ++i) {
    greetsBuffer1[i] = 0
    greetsBuffer2[i] = 0
}

const greetData = [
    [30, 100],
    [100, 180],
    [20, 400],
    [300, 340],
    [340, 10],
    [130, 180], // ukscene
    [370, 110],
    [160, 260],
    [260, 40],
    [300, 250],
    [50, 340],
    [320, 350],
    [120, 20],
    [270, 180], // holon
    [100, 100],
    [70, 220],
]
function moduleGreets (frameTime, config, syncs) {
    const startDelay = readAtTime(frameTime, config.startDelay, syncs.startDelay, 'float', 0)
    const plasmaBase = readAtTime(frameTime, config.plasmaBase, syncs.plasmaBase, 'float', -100)
    const plasmaRange = readAtTime(frameTime, config.plasmaRange, syncs.plasmaRange, 'float', 100)
    const plasmaSteps = readAtTime(frameTime, config.plasmaSteps, syncs.plasmaSteps, 'float', 2)
    const lightBase = readAtTime(frameTime, config.lightBase, syncs.lightBase, 'float3', [320, 240, 250])
    const lightRange = readAtTime(frameTime, config.lightRange, syncs.lightRange, 'float3', [180, 180, 20])
    const lightFallOff = readAtTime(frameTime, config.lightFallOff, syncs.lightFallOff, 'float', 450)
    const lightCap = readAtTime(frameTime, config.lightCap, syncs.lightCap, 'float', 0.2)
    const cycleSpeed = readAtTime(frameTime, config.cycleSpeed, syncs.cycleSpeed, 'float', 0.8)
    const displaySpeed = readAtTime(frameTime, config.displaySpeed, syncs.displaySpeed, 'float', 1)
    const textBase = readAtTime(frameTime, config.textBase, syncs.textBase, 'float', -250)
    const textRange = readAtTime(frameTime, config.textRange, syncs.textRange, 'float', 100)
    const textHeight = readAtTime(frameTime, config.textHeight, syncs.textHeight, 'float', 125)
    const lightBrightness = readAtTime(frameTime, config.lightBrightness, syncs.lightBrightness, 'float', 1)

    let index = 0
    // paulsma
    ///*
    const totalRange = 2 + Math.PI + 2 + Math.PI
    const iTotalRange = 1 / totalRange
    const iPlasmaSteps = 1 / plasmaSteps
    for (let y = 0; y < 481; ++y) {
        const ys = Math.sin((frameTime+5.1)*-0.073 + y * 0.0042) + Math.atan((frameTime+8.2)*+0.337 + y * 0.0095)
        for (let x = 0; x < 641; ++x) {
            const xs = Math.sin((frameTime+5.3)*-0.21 + x * 0.0045) + Math.atan((frameTime+3.9)*+0.062 + x * 0.0073)
            const v = (xs + ys) * iTotalRange + 0.5
            greetsBuffer1[index++] = Math.floor(v * plasmaSteps) * iPlasmaSteps * plasmaRange + plasmaBase
        }
    }
    //*/

    // neelsma
    /*
    const totalRange = Math.PI + Math.PI + 2 + 2
    for (let y = 0; y < 481; ++y) {
        const ys = Math.atan((frameTime+5.1)*-0.073 + y * 0.0022) + Math.atan((frameTime+8.2)*+0.337 + y * 0.0095)
        for (let x = 0; x < 641; ++x) {
            const xs = Math.cos((frameTime+1.3)*-0.002 + x * 0.0045) + Math.cos((frameTime+3.9)*+0.062 + x * 0.0073)
            const v = (xs + ys) / totalRange + 0.5
            greetsBuffer1[index++] = Math.floor(v * plasmaSteps) / plasmaSteps * plasmaRange + plasmaBase
        }
    }
    */

    // Greets
    const time = frameTime * cycleSpeed;
    let greet;
    for (let i = 0; i < greetData.length; ++i) {
        const greetTime = (time - i) * displaySpeed - startDelay
        if (greetTime >= 0 && greetTime < Math.PI) {
            const greetHeight = textBase + Math.sin(greetTime) * textRange
            const gpx = greetData[i][0]
            const gpy = greetData[i][1]
            greet = greets[i % greets.length]
            for (let gy = 0; gy < greet.length; ++gy) {
                const greetLine = greet[gy]
                for (let gx = 0; gx < greetLine.length; ++gx) {
                    const pixelHeight = greetLine[gx] * textHeight
                    if (pixelHeight != 0) {
                        const ofs = gpx+gx + (gpy+gy)*641
                        greetsBuffer1[ofs] = Math.max(greetsBuffer1[ofs], greetHeight + pixelHeight)
                    }
                }
            }
        }        
    }

    /*
    // Blur
    for (let pass = 0; pass < 0; ++pass) {
        let index = 641 + 1
        for (let y = 1; y < 481-1; ++y) {
            for (let x = 1; x < 641-1; ++x) {
                const v = greetsBuffer1[index-642] + greetsBuffer1[index-641] + greetsBuffer1[index-640] +
                    greetsBuffer1[index-1] + greetsBuffer1[index] + greetsBuffer1[index+1] +
                    greetsBuffer1[index+640] + greetsBuffer1[index+641] + greetsBuffer1[index+642]
                greetsBuffer2[index++] = v / 9
            }
            index += 2
        }
        const swap = greetsBuffer1
        greetsBuffer1 = greetsBuffer2
        greetsBuffer2 = swap
    }
    */

    const lsx = lightBase[0] + Math.sin(3.01+frameTime*0.185) * lightRange[0]
    const lsy = lightBase[1] + Math.cos(1.34+frameTime*0.263) * lightRange[1]
    const lsz = lightBase[2] + Math.sin(0.99+frameTime*0.821) * lightRange[2]

    const outputBuffer = greyscaleBuffers[getOutputBufferId()]
    const iLightFallOff = 1 / lightFallOff
    let writeIndex = 0
    for (let y = 0; y < 480; ++y) {
        let readIndex = (y + 1) * 641 + 1
        for (let x = 0; x < 640; ++x) {
            const thisHeight = greetsBuffer1[readIndex]
            /*
            const a = m4.normalize([0, 1, thisHeight - greetsBuffer1[index-640]])
            const b = m4.normalize([1, 0, thisHeight - greetsBuffer1[index-1]])
            const fn = m4.normalize([
                a[1]*b[2] - a[2]*b[1],
                a[2]*b[0] - a[0]*b[2],
                a[0]*b[1] - a[1]*b[0],
            ])
            */
            const fn = m4.normalize([thisHeight - greetsBuffer1[readIndex-1], thisHeight - greetsBuffer1[readIndex-641], -1])

            const lx = lsx - x
            const ly = lsy - y
            const lz = lsz - thisHeight
            const ll = Math.sqrt(lx*lx + ly*ly + lz*lz)
            const iLl = 1 / ll
            //const li = (0.5 - (ll / lightFallOff / lightBrightness)) + 0.5 // pow smooths the lighting, but too much cpu :(
            const li = Math.min(lightCap, 1 - (ll * iLightFallOff)) * lightBrightness // pow smooths the lighting, but too much cpu :(
            //const ln = [lx / ll, ly / ll, lz / ll]
            //const v = Math.acos(fn[0]*ln[0] + fn[1]*ln[1] + fn[2]*ln[2])
            const v = Math.acos(fn[0]*lx*iLl + fn[1]*ly*iLl + fn[2]*lz*iLl)

            outputBuffer[writeIndex++] = v * li
            readIndex++
        }
    }
}

const BORDER_TEX_SIZE = 30
const borderTex = new Float32Array(BORDER_TEX_SIZE * BORDER_TEX_SIZE)
{
    let index = 0
    for (let y = 0; y < BORDER_TEX_SIZE; ++y) {
        for (let x = 0; x < BORDER_TEX_SIZE; ++x) {
            let v
            if (!(y <= 1 || y >= BORDER_TEX_SIZE-2 ||
                x <= 1 || x >= BORDER_TEX_SIZE-2)) {
                v = 0
            } else {
                v = 1
            }
            borderTex[index++] = v * 1
        }
    }
}
textures['border'] = {
    size: BORDER_TEX_SIZE,
    data: borderTex
}

function moduleCube (frameTime, config, syncs) {
    const fov = degToRad(readAtTime(frameTime, config.fov, syncs.fov, 'float', 90))
    const roll = degToRad(readAtTime(frameTime, config.roll, syncs.roll, 'float', 0))
    const pitch = degToRad(readAtTime(frameTime, config.pitch, syncs.pitch, 'float', 0))
    const yaw = degToRad(readAtTime(frameTime, config.yaw, syncs.yaw, 'float', 0))
    const brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 1)
    const textureName = readAtTime(frameTime, config.texture, syncs.texture, 'string', 'default')
    const size = readAtTime(frameTime, config.size, syncs.size, 'float', 4)
    const scale = readAtTime(frameTime, config.scale, syncs.scale, 'float3', [1, 1, 1])

    const texture = textures[textureName]
    const texSize = texture.size
    const texData = texture.data

    let m = m4.yRotation(yaw)
    m = m4.xRotate(m, pitch)
    m = m4.zRotate(m, roll)

    let cubeDist = texSize * size
    const depthBrightnessScaler = 1 / (texSize * size * .03)
    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    const iHalfWidth = 1 / 240
    const iHalfHeight = 1 / 320
    let index = 0
    for (let y = 0; y < 480; ++y) {
        for (let x = 0; x < 640; ++x) {
            const direction = [(x - 320) * iHalfWidth, (y - 240 ) * iHalfHeight, fov, 0]
            const l = m4.transformVector(m, direction)

            const dx = Math.abs(cubeDist / l[0])
            const dy = Math.abs(cubeDist / l[1])
            const dz = Math.abs(cubeDist / l[2])
            
            let u, v
            const dmin = Math.min(Math.min(dx, dy), dz)
            if (dmin === dx) {
                u = Math.abs(l[1] * dx) * scale[0]
                v = Math.abs(l[2] * dx) * scale[0]
            } else if (dmin === dy) {
                u = Math.abs(l[0] * dy) * scale[1]
                v = Math.abs(l[2] * dy) * scale[1]
            } else {
                u = Math.abs(l[0] * dz) * scale[2]
                v = Math.abs(l[1] * dz) * scale[2]
            }

            const directionNormal = m4.normalize(direction)
            const tx = directionNormal[0] * dmin
            const ty = directionNormal[1] * dmin
            const tz = directionNormal[2] * dmin
            const distance = Math.sqrt(tx*tx + ty*ty + tz*tz) * .01
            const depthBrightness = 1 - (distance * depthBrightnessScaler)

            const greyscale = texData[Math.floor((u%texSize)) + Math.floor((v%texSize)) * texSize]
            greyscaleBuffer[index++] = Math.max(0, greyscale * brightness * depthBrightness)
        }
    }
}

function moduleTunnel (frameTime, config, syncs) {
    let roll = degToRad(readAtTime(frameTime, config.roll, syncs.roll, 'float', 0)) + 1000
    const distance = degToRad(readAtTime(frameTime, config.distance, syncs.distance, 'float', 0))
    let brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 0.025)
    let textureName = readAtTime(frameTime, config.texture, syncs.texture, 'string', 'default')

    const texture = textures[textureName]
    const texSize = texture.size
    const texData = texture.data

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    let CONE_RADIUS = 256
    const origin = [0, 0, 0]
    let index = 0

    const iHalfWidth = 1 / 240
    const iHalfHeight = 1 / 320
    const iUScaler = 1 / 20000
    const iVScaler = 6 * 1 / Math.PI

    for (let y = 0; y < 480; ++y) {
        for (let x = 0; x < 640; ++x) {
            let direction = [(x - 320) * iHalfWidth, (y - 240 ) * iHalfHeight, 1.0, 1.0]
            direction = m4.transformVector(camViewPersp, direction)

            const a = (direction[0]*direction[0]) + (direction[1]*direction[1])
            const b = 2*(origin[0]*direction[0] + origin[1]*direction[1])
            const c = (origin[0]*origin[0]) + (origin[1]*origin[1]) - (CONE_RADIUS*CONE_RADIUS)
        
            const delta = Math.sqrt(b*b - 4*a*c)

            const t = (-b + delta) / (2*a)
            //const t2 = (-b - delta) / (2*a)
            //const t = Math.max(t1, t2)

            const intersect = [origin[0] + direction[0]*t, origin[1] + direction[1]*t, origin[2] + direction[2]*t]
        
            let u = Math.abs(intersect[2]*iUScaler + distance) // 400 = y scale
            let v = Math.abs(Math.atan2(intersect[1], intersect[0])*iVScaler) // 8 = x scale
            let z = 3000 / t

            let tu = Math.floor((u%1) * texSize) //ang*60 = distance
            if (direction[2] > 0) {
                tu = (texSize-1) - tu
            }
            let tv = (Math.sign(intersect[1])*v + roll*2) % 1
            tv = Math.floor(tv * texSize)
            const greyscale = texData[tu + tv * texSize]

            greyscaleBuffer[index++] = greyscale * (z*z) * brightness
        }
    }
}

function moduleBlur (frameTime, config, syncs) {
    const passes = readAtTime(frameTime, config.passes, syncs.passes, 'int', 1)

    let bufferAId = getOutputBufferId()
    let bufferBId = lockBuffer()
    const blurDivisor = 1 / 5
    for (let i = 0; i < passes; ++i) {
        const src = greyscaleBuffers[bufferAId]
        const dst = greyscaleBuffers[bufferBId]
        let p = 0

        for (let x = 0; x < 640; ++x) {
            dst[p] = src[p++]
        }

        if (i & 0) {
            for (let y = 1; y < 480-1; ++y) {
                dst[p] = src[p++]
                for (let x = 1; x < 640-1; ++x) {
                    const v = src[p - 641] +
                              //src[p - 640] +
                              src[p - 639] +
                              //src[p - 1] +
                              src[p] +
                              //src[p + 1] +
                              src[p + 639] +
                              //src[p + 640] +
                              src[p + 641]
                    dst[p++] = v * blurDivisor
                }
                dst[p] = src[p++]
            }
        } else {
            for (let y = 1; y < 480-1; ++y) {
                dst[p] = src[p++]
                for (let x = 1; x < 640-1; ++x) {
                    const v = //src[p - 641] +
                              src[p - 640] +
                              //src[p - 639] +
                              src[p - 1] +
                              src[p] +
                              src[p + 1] +
                              //src[p + 639] +
                              src[p + 640]
                              //src[p + 641]
                    dst[p++] = v * blurDivisor
                }
                dst[p] = src[p++]
            }
        }


        for (let x = 0; x < 640; ++x) {
            dst[p] = src[p++]
        }

        const swap = bufferBId
        bufferBId = bufferAId
        bufferAId = swap
    }
    setOutputBufferId(bufferAId)
    unlockBuffer(bufferBId)
}

function moduleEdge (frameTime, config) {
    const srcBufferId = getOutputBufferId()
    const bufferAId = lockBuffer()
    const bufferBId = lockBuffer()

    convolveFilter(srcBufferId, bufferAId, [-1,0,1,-2,0,2,-1,0,1])
    convolveFilter(srcBufferId, bufferBId, [-1,-2,-1,0,0,0,1,2,1])
    for (let i = 0; i < 640*480; ++i) {
        greyscaleBuffers[srcBufferId][i] = (greyscaleBuffers[bufferAId][i] + greyscaleBuffers[bufferBId][i])
    }

    unlockBuffer(bufferAId)
    unlockBuffer(bufferBId)
}


function randomNormal (rng) {
    while (true) {
        const v = [(rng() - 0.5) * 2, (rng() - 0.5) * 2, (rng() - 0.5) * 2]
        var length = Math.sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2])
        if (length <= 1) {
            return m4.normalize(v)
        }
    }
}

function hslToRgb (h, s, l) {
    var r, g, b;

    if(s == 0){
        r = g = b = l; // achromatic
    }else{
        var hue2rgb = function hue2rgb(p, q, t){
            if(t < 0) t += 1;
            if(t > 1) t -= 1;
            if(t < 1/6) return p + (q - p) * 6 * t;
            if(t < 1/2) return q;
            if(t < 2/3) return p + (q - p) * (2/3 - t) * 6;
            return p;
        }

        var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        var p = 2 * l - q;
        r = hue2rgb(p, q, h + 1/3);
        g = hue2rgb(p, q, h);
        b = hue2rgb(p, q, h - 1/3);
    }

    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

function convolveFilter (srcBufferId, dstBufferId, weights) {
    const src = greyscaleBuffers[srcBufferId]
    const dst = greyscaleBuffers[dstBufferId]
    for (let y = 1; y < 480-1; ++y) {
        for (let x = 1; x < 640-1; ++x) {
            const p = y * 640 + x
            const v = src[p - 641] * weights[0] +
                      src[p - 640] * weights[1] +
                      src[p - 639] * weights[2] +
                      src[p - 1] * weights[3] +
                      src[p] * weights[4] +
                      src[p + 1] * weights[5] +
                      src[p + 639] * weights[6] +
                      src[p + 640] * weights[7] +
                      src[p + 641] * weights[8]
            dst[p] = v
        }
    }
}


function resetBufferLocks () {
    for (let i = 0; i < NUM_BUFFERS; ++i) {
        bufferLocks[i] = false
    }
    outputBufferId = -1
}
function lockBuffer () {
    for (let i = 0; i < NUM_BUFFERS; ++i) {
        if (!bufferLocks[i]) {
            bufferLocks[i] = true
            return i
        }
    }
    throw new Error('Ran out of buffers')
}
function unlockBuffer (bufferId) {
    if (bufferLocks[bufferId]) {
        bufferLocks[bufferId] = false
    } else {
        throw new Error('Buffer was not locked during unlock')
    }
}
function setOutputBufferId (bufferId) {
    if (bufferLocks[bufferId]) {
        outputBufferId = bufferId
    } else {
        throw new Error('Buffer was not locked during set')
    }
}
function getOutputBufferId () {
    if (bufferLocks[outputBufferId]) {
        return outputBufferId
    } else if (outputBufferId === -1) {
        throw new Error('Output buffer was not set during get')
    } else {
        throw new Error('Output buffer was not locked during get')
    }
}


// https://easings.net/#
const tweeners = {
    'hold': x => 0,
    'linear': x => x,
    'easeInSine': x => 1 - Math.cos((x * Math.PI) / 2),
    'easeOutSine': x => Math.sin((x * Math.PI) / 2),
    'easeInOutSine': x => -(Math.cos(Math.PI * x) - 1) / 2,
    'easeOutBounce': x => {
        const n1 = 7.5625
        const d1 = 2.75
        if (x < 1 / d1) {
            return n1 * x * x
        } else if (x < 2 / d1) {
            return n1 * (x -= 1.5 / d1) * x + 0.75
        } else if (x < 2.5 / d1) {
            return n1 * (x -= 2.25 / d1) * x + 0.9375
        } else {
            return n1 * (x -= 2.625 / d1) * x + 0.984375
        }
    }
}

function keyValidate_int (keyData) {
    return typeof keyData === 'number'
}
function keyLerp_int (a, b, alpha) {
    return Math.floor(a + (b - a) * alpha)
}

function keyValidate_float (keyData) {
    return typeof keyData === 'number'
}
function keyLerp_float (a, b, alpha) {
    return a + (b - a) * alpha
}

function keyValidate_float2 (keyData) {
    if (typeof keyData === 'object' &&
    keyData.length === 2 &&
        typeof keyData[0] === 'number' &&
        typeof keyData[1] === 'number') {
      return true
    } else {
      return false
    }
}
function keyLerp_float2 (a, b, alpha) {
    return [
        a[0] + (b[0] - a[0]) * alpha,
        a[1] + (b[1] - a[1]) * alpha,
    ]
}

function keyValidate_float3 (keyData) {
    if (typeof keyData === 'object' &&
    keyData.length === 3 &&
        typeof keyData[0] === 'number' &&
        typeof keyData[1] === 'number' &&
        typeof keyData[2] === 'number') {
      return true
    } else {
      return false
    }
}
function keyLerp_float3 (a, b, alpha) {
    return [
        a[0] + (b[0] - a[0]) * alpha,
        a[1] + (b[1] - a[1]) * alpha,
        a[2] + (b[2] - a[2]) * alpha,
    ]
}

function keyValidate_string (keyData) {
    return typeof keyData === 'string'
}
function keyLerp_string (a, b, alpha) {
    return a
}

const allKeyFuncs = {
    int: [keyValidate_int, keyLerp_int],
    float: [keyValidate_float, keyLerp_float],
    float2: [keyValidate_float2, keyLerp_float2],
    float3: [keyValidate_float3, keyLerp_float3],
    string: [keyValidate_string, keyLerp_string]
}

function readAtTime (time, keyData, syncData, keyType, defaultValue) {
    let result = undefined
    if (keyData !== undefined) {
        const keyFuncs = allKeyFuncs[keyType]
        if (keyFuncs[0](keyData)) {
            result = keyData
        } else {
            var lastKey = keyData[0]
            if (time <= lastKey[0]) {
                result = lastKey[1]
            } else {
                for (var i in keyData) {
                    const nextKey = keyData[i];
                    if (nextKey[0] > time) {
                        const alpha = (time - lastKey[0]) / (nextKey[0] - lastKey[0])
                        let tweener = lastKey[2] ? lastKey[2] : 'linear'
                        if (tweeners[tweener] === undefined) {
                        console.warn(`Unknown tweener type '${tweener}'`)
                        tweener = 'linear'
                        }
                        const tweenedAlpha = tweeners[tweener](alpha)
                        result = keyFuncs[1](lastKey[1], nextKey[1], tweenedAlpha)
                        break
                    }
                    lastKey = nextKey
                }
                if (result === undefined) {
                    result = lastKey[1]
                }
            }
        }
    } else {
        result = defaultValue
    }

    if (syncData) {
        if (keyType === 'float') {
            syncData.forEach(sync => {
                const a = 1
                const b = sync[1]
                const alpha = readInstrumentSync(sync[0])
                result = result * (a + (b-1)*alpha)
            })
        }  else if (keyType === 'int') {
            syncData.forEach(sync => {
                const a = 1
                const b = sync[1]
                const alpha = readInstrumentSync(sync[0])
                result = Math.floor(result * (a + (b-1)*alpha))
                //console.log(result)
            })
        }
    } 

    return result
}

const modules = {
    scene: () => {},
    blur: moduleBlur,
    edge: moduleEdge,
    clear: moduleClear,
    brightSpot: moduleBrightSpot,
    orbitCam: moduleOrbitCam,
    dust: moduleDust,
    greets: moduleGreets,
    dotCube: moduleDotCube,
    dotSphere: moduleDotSphere,
    tendrils: moduleTendrils,
    cube: moduleCube,
    tunnel: moduleTunnel,
    zoom: moduleZoom,
    noise: moduleNoise,
    distort: moduleDistort,
    debugText: moduleDebugText,
    present: modulePresent,
    overlay: moduleOverlay
}

let modulesRunStrings = []
let modulesIndent = 0
function runModules (frameTime, children) {
    children.forEach(x => {
        if (x.enabled !== false) {
            const start = x.start || 0
            const duration = x.duration || 9999
            let adjustedFrameTime = frameTime - start
            if (adjustedFrameTime >= 0 && adjustedFrameTime < duration) {
                const moduleType = x.type
                if (modules[moduleType]) {
                    const modulesRunStringsIndex = modulesRunStrings.length
                    modulesRunStrings.push('.')
                    const startTime = window.performance.now()
                    const time = readAtTime(adjustedFrameTime, x.config?.time, x.syncs?.time, 'float', adjustedFrameTime)

                    modules[moduleType](time, x.config || {}, x.syncs || {})

                    if (x.children) {
                        modulesIndent++
                        runModules(time, x.children)
                        modulesIndent--
                    }

                    const endTime = window.performance.now()
                    const duration = endTime - startTime
                    modulesRunStrings[modulesRunStringsIndex] = duration.toFixed(2).padStart(5, ' ') + '  '.repeat(modulesIndent + 1) + moduleType + ':' + (x.name || '...')
                } else {
                    console.warn(`Unknown module type ${moduleType}`)
                }
            }
        }
    })
}

function openFullscreen() {
    if (canvas.requestFullscreen) {
        canvas.requestFullscreen()
    } else if (canvas.webkitRequestFullscreen) { /* Safari */
        canvas.webkitRequestFullscreen()
    } else if (canvas.msRequestFullscreen) { /* IE11 */
        canvas.msRequestFullscreen()
    }
}

function fitScreen() {
    const params = new URLSearchParams(window.location.search)
    if (params.has('dev') || params.has('capture')) return

    const aspect = 640 / 480
    const windowWidth = window.innerWidth
    const windowHeight = window.innerHeight
    const windowAspect = windowWidth / windowHeight
    if (windowAspect >= aspect) {
        canvas.style.width = windowHeight * aspect
        canvas.style.height = windowHeight
        canvas.style.top = 0
        canvas.style.left = (windowWidth - (windowHeight * aspect)) / 2
    } else {
        canvas.style.width = windowWidth
        canvas.style.height = windowWidth / aspect
        canvas.style.top = (windowHeight - (windowWidth / aspect)) / 2
        canvas.style.left = 0
    }
    canvas.style.position = 'absolute'
    canvas.style.cursor = 'none'
}

let firstUpdate = false
let currentFrameTime = 0
let lastFrameTime = 0
function update() {
    if (!firstUpdate) {
        console.log('Kick out the jams..')
        firstUpdate = true
    }

    transport.update()

    const frameTime = transport.getCurrentTime() / 60 * bpm // + 0.5 // sync fix?
    currentFrameTime = frameTime

    const startTime = window.performance.now()
    modulesRunStrings = []
    resetBufferLocks()
    setOutputBufferId(lockBuffer())
    runModules(frameTime, script)
    const endTime = window.performance.now()

    if (doPerf) {
        let renderTimings = []
        for (let i = 0; i < 50; ++i) {
            modulesRunStrings = []
            resetBufferLocks()
            setOutputBufferId(lockBuffer())
            const startTime = window.performance.now()
            runModules(frameTime, script)
            const endTime = window.performance.now()
            const renderTime = endTime - startTime
            renderTimings.push(renderTime)
        }
        const min = Math.min(...renderTimings)
        const max = Math.max(...renderTimings)
        renderTimings = renderTimings.filter(x => !(x === min || x === max))
        console.log('Max:', max.toFixed(2))
        console.log('Min:', min.toFixed(2))
        console.log('Filtered avg:', (renderTimings.reduce((p, c) => p + c, 0) / renderTimings.length).toFixed(2))
    }

    if (showDebugInfo) {
        const newFrameTime = window.performance.now()
        const totalFrameTime = newFrameTime - lastFrameTime
        lastFrameTime = newFrameTime
        const fps = 1000 / totalFrameTime

        const debugText = 'BEAT: ' + frameTime.toFixed(2) + '\n' +
            (endTime - startTime).toFixed(2) + 'ms\n' +
            fps.toFixed(2) + 'fps\n' +
            (transport.getIsPlaying() ? 'PLAYING' : 'PAUSED') + '\n' +
            '\n' + 
            modulesRunStrings.join('\n')
        document.getElementById('debug').innerText = debugText

        showTimelines(frameTime)    
    }

    if (transport.getIsPlaying()) {
        requestUpdate()
    }
}

let timelineCanvasCtx = null
let timelineCanvasBuffer = 0
function showTimelines (frameTime) {
    const numRows = Object.values(instrumentIds).length
    const rowHeight = 10
    const width = 320
    const height = numRows*rowHeight
    const playHeadY = width * 0.25

    if (!timelineCanvasCtx) {
        const canvas = document.getElementById('timeline')
        canvas.width = width
        canvas.height = height
        timelineCanvasCtx = canvas.getContext('2d', { alpha: false })
        timelineCanvasBuffer = timelineCanvasCtx.createImageData(width, height)
        for (var i = 0; i < timelineCanvasBuffer.data.length; ++i) {
            timelineCanvasBuffer.data[i] = 255
        }
    }

    // clear background
    let index = 0
    for (let i = 0; i < width * height; ++i) {
        timelineCanvasBuffer.data[index++] = 0
        timelineCanvasBuffer.data[index++] = 0
        timelineCanvasBuffer.data[index++] = 0
        index++
    }

    // syncs
    const syncsOffset = Math.floor(frameTime * 100 - playHeadY)
    let instrumentRow = 0
    Object.values(instrumentIds).forEach(instrumentData => {
        for (let x = 0; x < width; ++x) {
            const syncValue = instrumentData[3][syncsOffset + x]
            const syncHeight = Math.floor(syncValue * (rowHeight-2)) + 1
            index = (x + width*(rowHeight*(instrumentRow+1)-1)) * 4
            for (let y = 0; y < syncHeight; ++y) {
                timelineCanvasBuffer.data[index++] = 150
                timelineCanvasBuffer.data[index++] = 150
                timelineCanvasBuffer.data[index++] = 150
                index -= width*4 + 3
            }
        }
        ++instrumentRow
    })

    // bars and beats
    let beatPos = playHeadY - Math.floor((frameTime % 4) * 100)
    let beat = 0
    while (true) {
        if (beatPos >= width) break
        if (beatPos >= 0) {
            index = beatPos * 4
            const colour = beat === 0 ? 250 : 150
            for (let i = 0; i < height; ++i) {
                timelineCanvasBuffer.data[index++] = colour
                timelineCanvasBuffer.data[index++] = colour
                index += (width*4)-2
            }
        }
        beatPos += 100
        beat = (beat + 1) % 4
    }

    // playhead
    index = playHeadY*4
    for (let i = 0; i < height; ++i) {
        timelineCanvasBuffer.data[index++] = 250
        timelineCanvasBuffer.data[index++] = 50
        timelineCanvasBuffer.data[index++] = 50
        index += (width*4)-3
    }

    timelineCanvasCtx.putImageData(timelineCanvasBuffer, 0, 0)
}

// mode ref => https://developer.mozilla.org/en-US/docs/Web/API/CanvasRenderingContext2D/globalCompositeOperation
function moduleOverlay (frameTime, config, syncs) {
    const alpha = readAtTime(frameTime, config.alpha, syncs.alpha, 'float', 1)
    const position = readAtTime(frameTime, config.position, syncs.position, 'float2', [0, 0])
    const source = readAtTime(frameTime, config.source, syncs.source, 'string', '')
    const mode = readAtTime(frameTime, config.mode, syncs.mode, 'string', 'source-over')
    const sources = config.sources;

    const src = sources && Array.isArray(sources) ? sources[Math.floor(frameTime) % sources.length] : source

    if (src && alpha > 0) {
        ctx.globalAlpha = alpha
        ctx.globalCompositeOperation = mode
        ctx.drawImage(overlays[src], position[0], position[1]);
    }
}

function moduleClear (frameTime, config, syncs) {
    const value = readAtTime(frameTime, config.value, syncs.value, 'float', 0)

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    for (var i = 0; i < 640 * 480; ++i) {
        greyscaleBuffer[i] = value
    }
}

function moduleBrightSpot (frameTime, config, syncs) {
    const brightness = readAtTime(frameTime, config.brightness, syncs.brightness, 'float', 1)
    const offset = readAtTime(frameTime, config.offset, syncs.offset, 'float2', [0, 0])

    const greyscaleBuffer = greyscaleBuffers[getOutputBufferId()]
    let index = 0
    const iHalfHeight = 1 / 320
    for (var y = 0; y < 480; ++y) {
        const vy = (y - 240) * iHalfHeight + offset[1]
        const vys = vy*vy
        for (var x = 0; x < 640; ++x) {
            const vx = (x - 320) * iHalfHeight - offset[0]
            const vxs = vx*vx
            const v = 1 - Math.sqrt(vxs + vys)
            greyscaleBuffer[index++] = Math.sin(v*v*v) * brightness
        }
    }
}

function moduleZoom (frameTime, config, syncs) {
    const repeats = readAtTime(frameTime, config.repeats, syncs.repeats, 'int', 5)
    const zoom = readAtTime(frameTime, config.zoom, syncs.zoom, 'float', 1.1)
    const blend = readAtTime(frameTime, config.blend, syncs.blend, 'float', 0.25)
    const offset = readAtTime(frameTime, config.offset, syncs.offset, 'float2', [0, 0])
    //const borderColour = readAtTime(frameTime, config.borderColour, 'float', 1)

    const ox = -offset[0] * 320 + 320
    const oy = offset[1] * 240 + 240

    let bufferAId = getOutputBufferId()
    let bufferBId = lockBuffer()
    const scale = 1 / zoom
    for (let i = 0; i < repeats; ++i) {
        setOutputBufferId(bufferBId)

        const srcBuffer = greyscaleBuffers[bufferAId]
        const dstBuffer = greyscaleBuffers[bufferBId]
        let index = 0
        for (let y = 0; y < 480; ++y) {
            const sy = Math.floor(((y - 240) * scale + oy))
            //if (sy >= 0 && sy < 480) {
                const sym = sy * 640
                for (let x = 0; x < 640; ++x) {
                    const sx = Math.floor((x - 320) * scale + ox)
                    //if (sx >= 0 && sx < 640) {
                        dstBuffer[index] = srcBuffer[index] + srcBuffer[sx + sym] * blend
                        index++
                    //} else {
                    //    dstBuffer[index++] = borderColour
                    //}
                }
            //} else {
            //    for (let x = 0; x < 640; ++x) {
            //        dstBuffer[index++] = borderColour
            //    }
            //}
        }
    
        const swap = bufferBId
        bufferBId = bufferAId
        bufferAId = swap
    }
    setOutputBufferId(bufferAId)
    unlockBuffer(bufferBId)
}

function modulePresent (frameTime, config, syncs) {
    const colourFieldIndex = readAtTime(frameTime, config.colourField, syncs.colourField, 'int', 0)
    const colourField = colourFields[colourFieldIndex]
    presentSlow(getOutputBufferId(), colourField)
    //presentGreyscale(getOutputBufferId())
}

function moduleDebugText (frameTime, config, syncs) {
    if (showDebugInfo) {
        const position = readAtTime(frameTime, config.position, syncs.position, 'float2', [10, 10])
        const text = readAtTime(frameTime, config.text, syncs.text, 'string', 'xxx')
        
        ctx.globalAlpha = 1.0
        ctx.globalCompositeOperation = 'source-over'
        ctx.font = 'bold 32px sans-serif'
        ctx.textAlign = 'start'
        ctx.textBaseline = 'top'
        ctx.strokeStyle = '#000'
        ctx.strokeText(text, position[0], position[1])
        ctx.fillStyle = '#fff'
        ctx.lineWidth = 8
        ctx.fillText(text, position[0], position[1])
    }
}

var buf = new ArrayBuffer(640 * 480 * 4)
var buf8 = new Uint8ClampedArray(buf)
var data = new Uint32Array(buf)
function presentFast (buffer) {
    const greyscaleBuffer = greyscaleBuffers[buffer]
    const pixels = canvasBuffer.data

    let index = 0
    let colourIndex = 0
    let y, ys, x, greyscale, r, g, b
    for (y = 0; y < 480; ++y) {
        ys = Math.sin(1 - x * 0.00025)
        for (x = 0; x < 640; ++x) {
            greyscale = greyscaleBuffer[index]
            r = Math.min(255, colours[colourIndex++] * greyscale)
            g = Math.min(255, colours[colourIndex++] * greyscale)
            b = Math.min(255, colours[colourIndex++] * greyscale)
            data[index++] = 
                0xff000000 |
                (b << 16) |
                (g << 8) |
                r
        }
    }

    pixels.set(buf8)
    ctx.putImageData(canvasBuffer, 0, 0)
}

function presentSlow (buffer, colourField) {
    const greyscaleBuffer = greyscaleBuffers[buffer]
    const pixels = canvasBuffer.data
    let writeIndex = 0
    let colourIndex = 0
    let readIndex, greyscale
    for (readIndex = 0; readIndex < 640 * 480; ++readIndex) {
        greyscale = greyscaleBuffer[readIndex]
        pixels[writeIndex++] = colourField[colourIndex++] * greyscale
        pixels[writeIndex++] = colourField[colourIndex++] * greyscale
        pixels[writeIndex++] = colourField[colourIndex++] * greyscale
        ++writeIndex
    }

    ctx.putImageData(canvasBuffer, 0, 0)
}

function presentGreyscale (buffer) {
    const greyscaleBuffer = greyscaleBuffers[buffer]
    const pixels = canvasBuffer.data
    let readIndex = 0
    let writeIndex = 0
    for (var y = 0; y < 480; ++y) {
        const ys = Math.sin(1 - x * 0.00025)
        for (var x = 0; x < 640; ++x) {
            const greyscale = greyscaleBuffer[readIndex++]
            const r=Math.max(Math.min(greyscale*256, 255), 0)
            pixels[writeIndex++] = r
            pixels[writeIndex++] = r
            pixels[writeIndex++] = r
            ++writeIndex
        }
    }

    ctx.putImageData(canvasBuffer, 0, 0)
}
