//--------------------------------------------------------------------------------------
// Constant Buffer Variables
//--------------------------------------------------------------------------------------
Texture2D tex0 : register(t0);
SamplerState sampLinearClamp : register(s0);
SamplerState sampNearestClamp : register(s1);
SamplerState sampLinearWrap : register(s2);
SamplerState sampNearestWrap : register(s3);

cbuffer GlobalRenderData : register(b3)
{
    matrix g_matWorld;
    matrix g_matView;
    matrix g_matProjection;
};

cbuffer GlobalRenderData: register(b4)
{
    matrix g_matModel;
    float g_fTime;
    float g_fFloat1;    // Saturation
    float g_fFloat2;    // Lightness
    float g_fFloat3;    // Position
    float g_fFloat4;
    float g_fFloat5;
    float g_fFloat6;
    float g_fSeed;
};

#define PI      3.1415926535897932384626433832795
#define TWOPI   6.283185307179586476925286766559



//--------------------------------------------------------------------------------------
struct VS_INPUT
{
    float3 vecPosition : POSITION;
    float2 vecTexCoord : TEXCOORD0;
};

struct PS_INPUT
{
    float4 vecPosition : SV_POSITION;
    float2 vecTexCoord : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Support functions
//--------------------------------------------------------------------------------------
float3 RGBToHSL(float3 color)
{
    float3 hsl; // init to 0 to avoid warnings ? (and reverse if + remove first part)

    float fmin = min(min(color.r, color.g), color.b);    //Min. value of RGB
    float fmax = max(max(color.r, color.g), color.b);    //Max. value of RGB
    float delta = fmax - fmin;             //Delta RGB value

    hsl.z = (fmax + fmin) / 2.0; // Luminance

    if (delta == 0.0)		//This is a gray, no chroma...
    {
        hsl.x = 0.0;	// Hue
        hsl.y = 0.0;	// Saturation
    }
    else                                    //Chromatic data...
    {
        if (hsl.z < 0.5)
            hsl.y = delta / (fmax + fmin); // Saturation
        else
            hsl.y = delta / (2.0 - fmax - fmin); // Saturation

        float deltaR = (((fmax - color.r) / 6.0) + (delta / 2.0)) / delta;
        float deltaG = (((fmax - color.g) / 6.0) + (delta / 2.0)) / delta;
        float deltaB = (((fmax - color.b) / 6.0) + (delta / 2.0)) / delta;

        if (color.r == fmax)
            hsl.x = deltaB - deltaG; // Hue
        else if (color.g == fmax)
            hsl.x = (1.0 / 3.0) + deltaR - deltaB; // Hue
        else if (color.b == fmax)
            hsl.x = (2.0 / 3.0) + deltaG - deltaR; // Hue

        if (hsl.x < 0.0)
            hsl.x += 1.0; // Hue
        else if (hsl.x > 1.0)
            hsl.x -= 1.0; // Hue
    }

    return hsl;
}

float HueToRGB(float f1, float f2, float hue)
{
    if (hue < 0.0)
        hue += 1.0;
    else if (hue > 1.0)
        hue -= 1.0;
    float res;
    if ((6.0 * hue) < 1.0)
        res = f1 + (f2 - f1) * 6.0 * hue;
    else if ((2.0 * hue) < 1.0)
        res = f2;
    else if ((3.0 * hue) < 2.0)
        res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
    else
        res = f1;
    return res;
}

float3 HSLToRGB(float3 hsl)
{
    float3 rgb;

    if (hsl.y == 0.0)
        rgb = float3(hsl.z, hsl.z, hsl.z); // Luminance
    else
    {
        float f2;

        if (hsl.z < 0.5)
            f2 = hsl.z * (1.0 + hsl.y);
        else
            f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);

        float f1 = 2.0 * hsl.z - f2;

        rgb.r = HueToRGB(f1, f2, hsl.x + (1.0 / 3.0));
        rgb.g = HueToRGB(f1, f2, hsl.x);
        rgb.b = HueToRGB(f1, f2, hsl.x - (1.0 / 3.0));
    }

    return rgb;
}

bool IsInsideTriangle(float2 vecP0, float2 vecP1, float2 vecP2, float2 vecP)
{
    float fArea = 1.0 / 2.0 * (-vecP1.y*vecP2.x + vecP0.y*(-vecP1.x + vecP2.x) + vecP0.x*(vecP1.y - vecP2.y) + vecP1.x*vecP2.y);
    float fS = 1.0 / (2.0 * fArea)*(vecP0.y*vecP2.x - vecP0.x*vecP2.y + (vecP2.y - vecP0.y)*vecP.x + (vecP0.x - vecP2.x)*vecP.y);
    float fT = 1.0 / (2.0 * fArea)*(vecP0.x*vecP1.y - vecP0.y*vecP1.x + (vecP0.y - vecP1.y)*vecP.x + (vecP1.x - vecP0.x)*vecP.y);
    if (fS > 0 && fT > 0 && (1 - fS - fT) > 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}


//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
PS_INPUT VS( VS_INPUT input )
{
    PS_INPUT output;

    matrix matWorld = mul(g_matWorld, g_matModel);
    matrix matWorldView = mul(matWorld, g_matView);
    matrix matWorldViewProjection = mul(matWorldView, g_matProjection);

    output.vecPosition = mul(float4(input.vecPosition, 1), matWorldViewProjection);
    output.vecTexCoord = (input.vecTexCoord.xy - 0.5) * float2(1280.0 / 720.0, 1);

    return output;
}


//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
#define NUM_POINTS      30
#define POINT_SIZE      0.03
#define POINT_DISTANCE  0.58
float4 PS(PS_INPUT input) : SV_Target
{
    float3 avecCentres[NUM_POINTS];
    for (int i = 0; i < NUM_POINTS; i++)
    {
        float fDistance = (((float)i / NUM_POINTS) * 3 + g_fFloat3) % 3;
        float3 vecPos;
        if (fDistance < 1)
        {
            vecPos.xy = lerp(float2(0.0, -1.0), float2(+0.86, 0.5), fDistance);
        }
        else if (fDistance < 2)
        {
            vecPos.xy = lerp(float2(+0.86, 0.5), float2(-0.86, 0.5), fDistance - 1);
        }
        else
        {
            vecPos.xy = lerp(float2(-0.86, 0.5), float2(0.0, -1.0), fDistance - 2);
        }
        avecCentres[i].xy = vecPos.xy * POINT_DISTANCE + float2(0, 0.14);
        avecCentres[i].z = abs((fDistance % 1) - 0.5) * 2;
    }

    for (int j = 0; j < NUM_POINTS; j++)
    {
        float fDist = length(input.vecTexCoord.xy - avecCentres[j].xy) / avecCentres[j].z;
        fDist /= POINT_SIZE;
        if (fDist < 1)
        {
//            if (fDist < 0.8)
//            {
                return float4(HSLToRGB(float3((float)j / NUM_POINTS, g_fFloat1, g_fFloat2)).xyz, 1);
                return 1;
//            }
//            else
//            {
//                return float4(0, 0, 0, 1);
//            }
        }
    }

    discard;
    return 0.5;
}
