﻿#region License

// --------------------------------------------------
// Copyright © OKB. All Rights Reserved.
// 
// This software is proprietary information of OKB.
// USE IS SUBJECT TO LICENSE TERMS.
// --------------------------------------------------

#endregion

using System;
using System.Threading;

using Tasks.Models;

namespace Tasks
{
    internal class Program
    {
        public static Coordinate GetCurrentCoordinate(Map map)
        {
            Coordinate target = new Coordinate
            {
                X = (map.Road[map.CurrentPathNode].Tile_X) * map.Tile_Height,
                Y = (map.Road[map.CurrentPathNode].Tile_Y) * map.Tile_Width
            };

            target.X += (float)map.Tile_Width / 2;
            target.Y += (float)map.Tile_Height / 2;

            return target;
        }


        public static int GetMyCurrentTile(int CarId, Processor brain, Map map)
        {
            //Get the closest node to where I spawn
            int counter = 0;
            float min = 99999;
            int closest = 0;

            //Loop trough our road
            foreach (Path path in map.Road)
            {
                float dist = brain.Distance(CarId, new Coordinate
                {
                    X = path.Tile_X * map.Tile_Height,
                    Y = path.Tile_Y * map.Tile_Width
                });

                if (dist < min)
                {
                    min = dist;
                    closest = counter;
                }
                counter++;
            }

            return closest;
        }


        public static bool IsTurn(Map map, int tileId)
        {
            try
            {
                var nextRoad = map.Road[tileId];
                string nextTile = map.Tiles[nextRoad.Tile_Y][nextRoad.Tile_X];
                if (nextTile == "/" || nextTile == "\\" || nextTile == "`" || nextTile == "," || nextTile == "+")
                    return true;
            }
            catch (Exception e)
            {
                //Didn't find the next tile
            }

            return false;
        }


        private static void Main(string[] args)
        {
            if (args.Length == 0)
            {
                args = new string[1];
                args[0] = "SoccerMom";
            }

            const float driveAngle = 180;
            const float turnAngle = 1;
            const float abilityAngle = 45;
            const float maxPrediction = 1.2f;
            bool first = true;

            Communicator communicator = new Communicator();
            Processor brain = new Processor();
            Map map = new Map();
            Coordinate currentTarget = null;
            communicator.Connect(args[0]);
            brain.LoadMap(communicator.ReadResponse());
            int CarId = brain.State.Id;
            brain.Log(Logger.LogType.INFO, "Car ID", CarId.ToString());

            //We now have the map! Store it!
            if (brain.State.Map != null)
                map = brain.State.Map;

            Thread.Sleep(100);

            //Jus start forward
            communicator.SendCommand(Command.Up, false);

            while (communicator.Client.Connected)
            {
                //Get latest update
                brain.ProcessUpdate(communicator.ReadResponse());

                //Madness since we have no cars before first update
                if (first)
                {
                    //Set current node to the closest node
                    map.CurrentPathNode = GetMyCurrentTile(CarId, brain, map);

                    //Set the first target node
                    currentTarget = GetCurrentCoordinate(map);
                    brain.Log(Logger.LogType.INFO, "Set new target", currentTarget.X + " " + currentTarget.Y);

                    //Set to false so we don't do it again!
                    first = false;
                }

                //What's the distance to the target?
                brain.Log(Logger.LogType.INFO, "Distance", brain.Distance(CarId, currentTarget).ToString());
                float maxDistance = brain.Speed(brain.State.Id) / 100;
                if (maxDistance > maxPrediction)
                    maxDistance = maxPrediction;

                if (brain.Distance(CarId, currentTarget) < (float)map.Tile_Height * maxDistance)
                {
                    map.CurrentPathNode++;

                    //Restart pathing
                    if (map.Road.Count <= map.CurrentPathNode)
                        map.CurrentPathNode = 0;

                    //Set our next target
                    currentTarget = GetCurrentCoordinate(map);
                    brain.Log(Logger.LogType.INFO, "Set new target", currentTarget.X + " " + currentTarget.Y);
                }

                //Angle to the target?
                double angleToTarget = brain.AngleToTarget(CarId, currentTarget);
                int command = 0;

                brain.Log(Logger.LogType.INFO, "Angle to target", angleToTarget.ToString());

                if (Math.Abs(angleToTarget) < driveAngle)
                {
                    int tileId = GetMyCurrentTile(CarId, brain, map);

                    if (brain.Speed(CarId) < 180)
                    {
                        command = Command.Up;
                        brain.Log(Logger.LogType.INFO, "FULL", "FORCE!");
                    }
                    
                    if (brain.Speed(CarId) > 185)
                    {
                        if (IsTurn(map, tileId) || IsTurn(map, tileId + 1) || IsTurn(map, tileId + 2) || IsTurn(map, tileId + 3))
                        {
                            command = Command.Down;
                            brain.Log(Logger.LogType.INFO, "BRAKE", "NOW!");
                        }
                    }
                }

                if (angleToTarget > turnAngle)
                    command += Command.Right;
                else if (angleToTarget < turnAngle)
                    command += Command.Left;

                //Should we use ability?
                if (angleToTarget < abilityAngle)
                {
                    try
                    {
                        int tileId = GetMyCurrentTile(CarId, brain, map);

                        //Is this a straight?
                        if (!IsTurn(map, tileId) && !IsTurn(map, tileId + 1) && !IsTurn(map, tileId + 2) && !IsTurn(map, tileId + 3))
                            command += Command.Use;
                    }
                    catch (Exception e)
                    {
                        brain.Log(Logger.LogType.ERROR, "NOT STRAIGHT", angleToTarget.ToString());
                    }
                }

                //Send command
                communicator.SendCommand(command, true);

                //Delay to debug
                //Thread.Sleep(1);
            }
        }
    }
}