using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Xml;

namespace StarifficEditor
{
    public delegate void LevelChanged();

    public class LevelOptions : ICloneable
    {
        private String name = "Unnamed level";
        [CategoryAttribute("General"),
         DefaultValue("Unnamed level"),
         Description("The levels name.")]
        public String Name
        {
            get { return name; }
            set { name = value; }
        }

        private float time_bronze = 30.0f;
        [CategoryAttribute("General"),
         DefaultValue(30.0f),
         Description("The time to beat (seconds) to get bronze.")]
        public float TimeBronze
        {
            get { return time_bronze; }
            set { time_bronze = value; }
        }

        private float time_silver = 20.0f;
        [CategoryAttribute("General"),
         DefaultValue(20.0f),
         Description("The time to beat (seconds) to get silver.")]
        public float TimeSilver
        {
            get { return time_silver; }
            set { time_silver = value; }
        }

        private float time_gold = 10.0f;
        [CategoryAttribute("General"),
         DefaultValue(10.0f),
         Description("The time to beat (seconds) to get gold.")]
        public float TimeGold
        {
            get { return time_gold; }
            set { time_gold = value; }
        }

        private float gravity = 120.0f;
        [CategoryAttribute("Gameplay"),
         DefaultValue(120.0f),
         Description("The force of gravity.")]
        public float Gravity
        {
            get { return gravity; }
            set { gravity = value; }
        }

        private float screen_width = 200.0f;
        [CategoryAttribute("Visualization"),
         DefaultValue(200.0f),
         Description("The minimum screen width in level coordinates. Not thet more may be shown if the screen aspect ratio differs from that of the level screens.")]
        public float ScreenWidth
        {
            get { return screen_width; }
            set { screen_width = value; }
        }

        private float screen_height = 150.0f;
        [CategoryAttribute("Visualization"),
         DefaultValue(150.0f),
         Description("The minimum screen height in level coordinates. Not thet more may be shown if the screen aspect ratio differs from that of the level screens.")]
        public float ScreenHeight
        {
            get { return screen_height; }
            set { screen_height = value; }
        }

        public virtual void Load(XmlElement elem)
        {
            if (elem == null)
                return;
            name = TextUtil.GetString(elem["name"], "value", name);
            time_bronze = TextUtil.GetFloat(elem["time_bronze"], "value", time_bronze);
            time_silver = TextUtil.GetFloat(elem["time_silver"], "value", time_silver);
            time_gold = TextUtil.GetFloat(elem["time_gold"], "value", time_gold);
            screen_width = TextUtil.GetFloat(elem["screen_width"], "value", screen_width);
            screen_height = TextUtil.GetFloat(elem["screen_height"], "value", screen_height);
        }

        public virtual void Save(XmlElement elem)
        {
            TextUtil.AddStringElement(elem, "name", "value", name);
            TextUtil.AddStringElement(elem, "time_bronze", "value", "" + time_bronze);
            TextUtil.AddStringElement(elem, "time_silver", "value", "" + time_silver);
            TextUtil.AddStringElement(elem, "time_gold", "value", "" + time_gold);
            TextUtil.AddStringElement(elem, "screen_width", "value", "" + screen_width);
            TextUtil.AddStringElement(elem, "screen_height", "value", "" + screen_height);
        }

        public object Clone()
        {
            LevelOptions opts = (LevelOptions) MemberwiseClone();
            return opts;
        }
    }

    public class Level
    {
        public RectangleF BoundingBox
        {
            get
            {
                RectangleF bbox = new RectangleF();
                bool first = true;
                foreach (LevelObject obj in objects)
                {
                    if (first)
                    {
                        bbox = obj.BoundingBox;
                        first = false;
                    }
                    else
                        bbox = RectangleF.Union(bbox, obj.BoundingBox);
                }
                return bbox;
            }
        }

        public LevelOptions options;
        public List<LevelObject> objects;

        public event LevelChanged level_changed_event;

        public Level()
        {
            options = new LevelOptions();
            objects = new List<LevelObject>();
            Clear();
        }

        public void Clear()
        {
            options = new LevelOptions();

            objects.Clear();
            objects.Add(new LevelObjectStart());

            LevelObject end = new LevelObjectEnd();
            end.X += 30;
            objects.Add(end);

            NotifyChanges();
        }

        public LevelObject FindObject(PointF pt)
        {
            foreach (LevelObject obj in objects)
            {
                if (obj.IsPointOnObject(pt))
                    return obj;
            }
            return null;
        }

        public LevelObject[] FindObjects(PointF pt)
        {
            List<LevelObject> ret = new List<LevelObject>();

            foreach (LevelObject obj in objects)
            {
                if (obj.IsPointOnObject(pt))
                    ret.Add(obj);
            }

            return ret.ToArray();
        }

        public bool Load(String filename)
        {
            try
            {
                Clear();
                objects.Clear();

                // Load the XML
                XmlDocument doc = new XmlDocument();
                doc.Load(filename);

                // Load the options
                options.Load(doc.DocumentElement["options"]);

                // Load the objects
                LoadObjects(doc.DocumentElement["objects"]);

                NotifyChanges();
                return true;
            }
            catch (Exception)
            {
                Clear();
                return false;
            }
        }

		public bool Save(String filename)
		{
			try
			{
				XmlDocument doc = new XmlDocument();

				// Create document root
				doc.AppendChild(doc.CreateXmlDeclaration("1.0", null, null));
				XmlElement root = doc.CreateElement("level");
				doc.AppendChild(root);

                // Save the options
                if (true)
                {
                    XmlElement options_root = doc.CreateElement("options");
                    root.AppendChild(options_root);

                    options.Save(options_root);
                }

				// Save the objects
				if (true)
				{
					XmlElement objects_root = doc.CreateElement("objects");
					root.AppendChild(objects_root);

					foreach (LevelObject obj in objects)
					{
						XmlElement object_root = doc.CreateElement("object");
						objects_root.AppendChild(object_root);
						object_root.SetAttribute("type", "" + obj.Type);

						obj.Save(object_root);
					}
				}

				// Save
				doc.Save(filename);
				return true;
			}
			catch (Exception)
			{
				return false;
			}
		}

        public byte[] Export()
        {
            using (MemoryStream dest_stm = new MemoryStream())
            {
                JavaBinaryWriter dest_out = new JavaBinaryWriter(dest_stm);
                dest_out.FixedNumberScale = 1.0f / 10.0f;

                // Write the screen sizes and aspect ratio
                dest_out.WriteFixedScaled(options.ScreenWidth);
                dest_out.WriteFixedScaled(options.ScreenHeight);
                dest_out.WriteFixed(options.ScreenWidth / options.ScreenHeight);

                // Write the gravity
                dest_out.WriteFixedScaled(options.Gravity);

                // Write the times for bronze/silver/gold
                dest_out.WriteFixed(options.TimeBronze);
                dest_out.WriteFixed(options.TimeSilver);
                dest_out.WriteFixed(options.TimeGold);

                // Write the bounding box for the level
                RectangleF total_bbox = BoundingBox;
                dest_out.WriteFixedScaled(total_bbox.X);
                dest_out.WriteFixedScaled(total_bbox.Y);
                dest_out.WriteFixedScaled(total_bbox.Width);
                dest_out.WriteFixedScaled(total_bbox.Height);

                // Write the objects
                LevelObject[] obj_rev_list = objects.ToArray();
                Array.Reverse(obj_rev_list);

                dest_out.Write((ushort)objects.Count);
                foreach (LevelObject obj in obj_rev_list)
                {
                    // Write the type
                    dest_out.Write((byte)obj.Type);
                    // Export the object
                    obj.Export(dest_out);
                }

                // Create the grid
                LevelGrid grid = new LevelGrid(this);
                dest_out.WriteFixedScaled(grid.grid_x);
                dest_out.WriteFixedScaled(grid.grid_y);
                dest_out.WriteFixedScaled(grid.grid_slot_size);
                dest_out.Write((short)grid.grid_w);
                dest_out.Write((short)grid.grid_h);

                for (int y = 0; y < grid.grid_h; ++y)
                    for (int x = 0; x < grid.grid_w; ++x)
                    {
                        List<LevelObject> slot_objs = grid.grid[ x, y ];
                        dest_out.Write((short)slot_objs.Count);
                        foreach (LevelObject obj in slot_objs)
                        {
                            int idx = Array.IndexOf(obj_rev_list, obj);
                            Debug.Assert(idx >= 0);
                            dest_out.Write((short)idx);
                        }
                    }

                // Get the data
                dest_out.Flush();
                return dest_stm.ToArray();
            }
        }

		public bool Export(String filename) 
		{
			// Generate the data
			byte[] data = Export();

			// Write the file
			try
			{
				using (FileStream fout = new FileStream(filename, FileMode.Create, FileAccess.Write))
				{
					fout.Write(data, 0, data.Length);
				}
			}
			catch (Exception)
			{
				return false;
			}

			return true;
		}

        private void LoadObjects(XmlElement objects)
        {
            if (objects == null)
                return;

            foreach (XmlElement obj_elem in objects.SelectNodes("object"))
            {
                // Create the object
                LevelObject obj = null;
                try
                {
                    LevelObjectType type = (LevelObjectType)Enum.Parse(typeof(LevelObjectType), TextUtil.GetString(obj_elem,"type", null), true);
                    obj = LevelObjectFactory.CreateObject(type);
                }
                catch (Exception)
                {
                }
                if (obj == null)
                    continue;

                // Load the object data
                obj.Load(obj_elem);

                // Add the object
                this.objects.Add(obj);
            }
        }

        public void NotifyChanges()
        {
            if (level_changed_event != null)
                level_changed_event();
        }
    }
}
