// DirectXConversion.cpp / CuteElf
// Copyright (c) 1997 Samuel Marshall

#include <windows.h>
#include "ddraw.h"
#include "DXErrors.h"

#include "UserSettings.h"

// Useful macros

#ifdef _DEBUG
#define TRACE(x) \
{ char trace_buf[1024]; wsprintf(trace_buf,x);\
  OutputDebugString(trace_buf); }
#else
#define TRACE(x) {}
#endif

#define THROWHR(x)	{\
{	char strPlace[80]; \
	HRESULT hrE; \
	hrE=x; \
	wsprintf(strPlace,"%s:%i",__FILE__,__LINE__); \
	if(hrE !=DD_OK) { ShowDDHR(strPlace,hrE); return hrE; } }\
}

// Globals

HINSTANCE sg_hInst;
HWND sg_hwndMain;
int sg_iFrameCount;
DWORD dwStartTime;

LPDIRECTDRAW2 sg_dd;
LPDIRECTDRAWSURFACE2 sg_ddsPrimary,sg_ddsSecondary;
LPDIRECTDRAWPALETTE sg_ddpGlobal;

BOOL volatile sg_fThreadQuit;
BOOL volatile sg_fInCritical;

// Procedure to close down DirectX
HRESULT HrFreeDirectX()
{
	// Release all the COM objects
	if(sg_dd) 
		THROWHR(sg_dd->Release());
	sg_dd=NULL;
	
	// I think DirectDraw2 objects automatically
	// release all their surfaces...
	sg_ddsPrimary=NULL;
	
	// I think secondary buffers are "part of" primary
	// surface so don't need deleting

	TRACE("Successfully freed DirectX\n");

	return DD_OK;
}

// Procedure to get the secondary surface pointers
HRESULT WINAPI SurfacesProc(
    LPDIRECTDRAWSURFACE lpDDSurf1,
    LPDDSURFACEDESC lpDDSurfaceDesc, LPVOID lpContext)
{
	// Convert surface version... not sure this is okay
	// (help file says parameter should be a SURFACE2
	// anyway... if so this is okay... if not, I should
	// go through that QueryInterface crap)
	LPDIRECTDRAWSURFACE2 lpDDSurface=
		(LPDIRECTDRAWSURFACE2)lpDDSurf1;

	if(lpDDSurfaceDesc->ddsCaps.dwCaps | DDSCAPS_BACKBUFFER)
		sg_ddsSecondary=lpDDSurface;

	return DDENUMRET_OK;
}

// Procedure to initialise DirectX
HRESULT HrInitDirectX()
{
	// Create the DirectDraw object
	LPDIRECTDRAW lpdd;
	THROWHR(DirectDrawCreate(NULL,&lpdd,NULL));

	// Get an IDirectDraw2 out of it
	THROWHR(lpdd->QueryInterface(IID_IDirectDraw2,
		(LPVOID *)&sg_dd));

	// Release the old object
	lpdd->Release();
	
	// Set the co-operative level (we're not gonna ;)
	THROWHR(sg_dd->SetCooperativeLevel(sg_hwndMain,
		DDSCL_ALLOWMODEX | DDSCL_EXCLUSIVE |
		DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT));

	// Change screen mode
	THROWHR(sg_dd->SetDisplayMode(320,200,8,0,0));

	// Create some surfaces so we can page flip
	DDSURFACEDESC ddsd;
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
		DDSCAPS_FLIP | DDSCAPS_COMPLEX;
	ddsd.dwBackBufferCount = 1;

	// Create surfaces
	THROWHR(sg_dd->CreateSurface(&ddsd, 
		(IDirectDrawSurface**) &sg_ddsPrimary, 
		NULL));

	// Obtain back surfaces
	THROWHR(sg_ddsPrimary->EnumAttachedSurfaces(NULL,
		&SurfacesProc));

	// Create the palette (default is greyshades)
	PALETTEENTRY ape[256];
	for(int i=0;i<256;i++)
	{
		ape[i].peRed=i;
		ape[i].peGreen=i;
		ape[i].peBlue=i;
		ape[i].peFlags=0;
	}

	THROWHR(sg_dd->CreatePalette(DDPCAPS_8BIT|
		DDPCAPS_ALLOW256,ape,
		&sg_ddpGlobal,NULL));

	// Set the palette
	THROWHR(sg_ddsPrimary->SetPalette(sg_ddpGlobal));

	TRACE("Successfully initialised DirectX.\n");

	return DD_OK;
}

// Slow version of "do new frame" method
HRESULT HrNewFrame2(BYTE* pbBuffer)
{
	LPDIRECTDRAWSURFACE2 
		dds=sg_ddsSecondary;

	HRESULT hr;

	// Lock surface
	DDSURFACEDESC ddsd;
	RECT rect={0,0,320,200};
	ddsd.dwSize=sizeof(ddsd);
	hr=dds->Lock(&rect,&ddsd,DDLOCK_WAIT,NULL);
	if(hr==DDERR_SURFACELOST)
		{ THROWHR(sg_ddsPrimary->Restore());}
	else if(hr!=DD_OK) THROWHR(hr);

	// Get pointer to surface and line offset
	BYTE *bSurface=(BYTE*) ddsd.lpSurface;
	DWORD dwPitch=ddsd.lPitch-320;

	// Copy in surface from buffer
	_asm
	{
		push esi
		push edi

		mov esi,pbBuffer
		mov edi,bSurface

		mov eax,200

	loopstart:
		mov ecx,80
		rep movsd
		add edi,dwPitch
		dec eax
		jnz loopstart

		pop edi
		pop esi
	}

	// Unlock surface
	THROWHR(dds->Unlock(ddsd.lpSurface));

	// Flip frame
	hr=sg_ddsPrimary->Flip(
		(IDirectDrawSurface*)dds,DDFLIP_WAIT);

	if(hr==DDERR_SURFACELOST)
		{ THROWHR(sg_ddsPrimary->Restore()); }
	else if(hr!=DD_OK) THROWHR(hr);

	// Keep count of number of frames
	sg_iFrameCount++;

	return DD_OK;
}
HRESULT HrNewFrame(BYTE* pbBuffer)
{
	sg_fInCritical=TRUE;
	HRESULT hr=HrNewFrame2(pbBuffer);
	sg_fInCritical=FALSE;
	return hr;
}

DDSURFACEDESC sg_ddsdHrStartFrame;

// Start the faster version of the "do frame" method
HRESULT HrStartFrame2(BYTE* &pbBuffer,DWORD &dwLineOfs)
{
	LPDIRECTDRAWSURFACE2 
		dds=sg_ddsSecondary;

	HRESULT hr;

	// Lock surface
	RECT rect={0,0,320,200};
	sg_ddsdHrStartFrame.dwSize=sizeof(sg_ddsdHrStartFrame);
	hr= dds->Lock(&rect,&sg_ddsdHrStartFrame,
		DDLOCK_WAIT,NULL);
	if(hr==DDERR_SURFACELOST)
		{ THROWHR(sg_ddsPrimary->Restore());}
	else if(hr!=DD_OK) THROWHR(hr);

	// Get pointer to surface
	pbBuffer=(BYTE*) sg_ddsdHrStartFrame.lpSurface;

	// Get line offset
	dwLineOfs=sg_ddsdHrStartFrame.lPitch-320;

	return DD_OK;
}
HRESULT HrStartFrame(BYTE* &pbBuffer,DWORD &dwLineOfs)
{
	if(sg_fInCritical) 
		TRACE("HrStart/FinishFrame commands don't match!"); 
	sg_fInCritical=TRUE;
	HRESULT hr=HrStartFrame2(pbBuffer,dwLineOfs);
	return hr;
}

// Complete the faster version of the "do frame" method
HRESULT HrFinishFrame2()
{
	LPDIRECTDRAWSURFACE2 
		dds=sg_ddsSecondary;

	HRESULT hr;

	// Unlock surface
	THROWHR(dds->Unlock(sg_ddsdHrStartFrame.lpSurface));

	// Flip frame
	hr=sg_ddsPrimary->Flip(
		(IDirectDrawSurface*)dds,DDFLIP_WAIT);
	if(hr==DDERR_SURFACELOST)
		{ THROWHR(sg_ddsPrimary->Restore());}
	else if(hr!=DD_OK) THROWHR(hr);

	// Increment framecount (only used for timing)
	sg_iFrameCount++;

	return DD_OK;
}
HRESULT HrFinishFrame()
{
	if(!sg_fInCritical) 
		TRACE("HrStart/FinishFrame commands don't match!"); 
	HRESULT hr=HrFinishFrame2();
	sg_fInCritical=FALSE;
	return hr;
}

// Set the palette from a 768-byte 0-255 palette
void SetPalette(BYTE* bPalette)
{
	PALETTEENTRY ape[256];

	// Copy in palette
	for(int i=0;i<256;i++)
	{
		ape[i].peRed=bPalette[i*3];
		ape[i].peGreen=bPalette[i*3+1];
		ape[i].peBlue=bPalette[i*3+2];
		ape[i].peFlags=0;
	}

	sg_ddpGlobal->SetEntries(0,0,256,ape);
}

// Window procedure
LRESULT CALLBACK WndProc(HWND hwnd,UINT uMsg,
	WPARAM wParam,LPARAM lParam)
{
	static UINT uTimer,uTimer2;
	static HANDLE hThread;
	static BOOL fSuspended;

	DWORD dwThreadID;

	char strTemp[120];

	switch(uMsg)
	{
	case WM_CREATE:
		// On window creation, we initialise DirectX,
		// but only after a while so as to finish the
		// rest of window creation first.
		// (Translation: if you try to init DirectX
		// now instead of on the timer, it crashes :)
		uTimer=SetTimer(hwnd,1,500,NULL);
		SetFocus(hwnd);
		break;

	case WM_TIMER:
		if(wParam==1)	// Initialising timer 
		{
			KillTimer(hwnd,uTimer);

			// Init DirectX
			ShowDDHR("HrInitDirectX",HrInitDirectX());

			// Start the demo code
			hThread=CreateThread(NULL,0,&ThreadFunc,0,0,&dwThreadID);

			// Store the start time (used only for
			// fps debugmessage displayed at end)
			dwStartTime=GetTickCount();
		}
		else if(wParam==2)	// Destroying timer
			if(sg_fThreadQuit==FALSE)
		{
			// Thread has terminated.
			KillTimer(hwnd,uTimer2);

			// Close DirectX
			ShowDDHR("HrFreeDirectX",HrFreeDirectX());

			// Destroy window
			DestroyWindow(hwnd);
		}

		break;

	case WM_ACTIVATE:
		if(LOWORD(wParam)==WA_INACTIVE)
		{
			// When somebody switches to another window,
			// pause the action and reveal the cursor.
			
			// I thought we should 
			// wait until we're safely outside
			// DirectX suspends but it doesn't help.
			//while(sg_fInCritical) Sleep(10);

			if(hThread && !fSuspended)
			{
				SuspendThread(hThread);
				fSuspended=TRUE;
			}
			ShowCursor(TRUE);
			return 0;
		}

		if(LOWORD(wParam)==WA_ACTIVE)
		{
			// When somebody switches to this window,
			// hide the cursor and restart the action.
			ShowCursor(FALSE);

			if(hThread && fSuspended) 
			{
				ResumeThread(hThread);
				fSuspended=FALSE;
			}
			return 0;
		}
		break;

	case WM_CLOSE	:
		// Signal the action thread should exit, and
		// wait for it to do so.
		sg_fThreadQuit=TRUE;
		uTimer2=SetTimer(hwnd,2,50,NULL);

		// Do not destroy window yet.
		return 0;

	case WM_DESTROY:
		// Quit
		PostQuitMessage(0);
		break;
	}
	return DefWindowProc(hwnd,uMsg,wParam,lParam);
}

// Main procedure
int WINAPI WinMain(HINSTANCE hInstance,	
	HINSTANCE hPrevInstance,LPSTR lpCmdLine,int nCmdShow)
{
	LPSTR lpszClass = "DirectXDemoClass";
	sg_hInst=hInstance;

	// Create window class
	if (!hPrevInstance) 
	{ 
		WNDCLASS wc;
		wc.style = 0; 
		wc.lpfnWndProc = (WNDPROC) WndProc; 
		wc.cbClsExtra = 0; 
		wc.cbWndExtra = 0; 
		wc.hInstance = hInstance; 
		wc.hIcon = LoadIcon(hInstance, 
			SETTING_ICON); 
		wc.hCursor =NULL;
		wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
		wc.lpszMenuName =  NULL; 
		wc.lpszClassName = lpszClass; 

		if (!RegisterClass(&wc)) 
			return FALSE; 
	} 

	// Create window (note that I put it way off the
	// screen so that it never shows)
	// You need a window for DirectX programs so that
	// they show up in the taskbar, alt-ctrl-del works,
	// etc.
    sg_hwndMain = CreateWindow(lpszClass, SETTING_NAME, 
        WS_OVERLAPPEDWINDOW, 
		5000,5000, 
        100, 100, 
		(HWND) NULL, (HMENU) NULL, sg_hInst, 
		(LPVOID) NULL); 
 
	// If we couldn't create the window, give up.
    if (!sg_hwndMain) 
	{
		MessageBox(NULL,"Unable to create main program "
			"window.\n\nRestart Windows and try again.",
			SETTING_NAME " Error",
			MB_OK|MB_ICONEXCLAMATION);
        return 0; 
	}

	// Show the window minimised (it will always be
	// this way).
	ShowWindow(sg_hwndMain,SW_MINIMIZE);

	///////////////
	// Message loop
	MSG msg;
	while (GetMessage(&msg, (HWND) NULL, 0, 0)) 
	{ 
        TranslateMessage(&msg); 
        DispatchMessage(&msg); 
    } 

	// Display profiling information at end of app.
	char strBuffer[256];
	wsprintf(strBuffer,"Frames (%i) per second (%i): %i fps\n",
		sg_iFrameCount,
		(GetTickCount()-dwStartTime)/1000,
		sg_iFrameCount/((GetTickCount()-
			dwStartTime)/1000));
	TRACE(strBuffer);

    return msg.wParam; 
}

