/*==============================================================================================*/
/*   output.cc                                                                     Font3D       */
/*----------------------------------------------------------------------------------------------*/
/*                                                                                              */
/*   Copyright (c) 1994, 1995 by Todd A. Prater                                 Version 1.50    */
/*   All rights reserved.                                                                       */
/*                                                                                              */
/*==============================================================================================*/

#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>

#include "config.h"
#include "font3d.h"
#include "truetype.h"
#include "geometry.h"


#define DEGENERATE_THRESHOLD 1.0e-9


/*==============================================================================================*/
/*  OutputTriangles()                                                                           */
/*==============================================================================================*/
/*                                                                                              */
/*  SYNTAX:       void OutputTriangles(ostream& outputFile, USHORT outputFormat,                */
/*                                     USHORT triangleType, USHORT spacesToIndent,              */
/*                                     TRIANGLELIST& triangleList);                             */
/*                                                                                              */
/*  DESCRIPTION:  This function outputs a list of triangles to the stream 'outputFile'.         */
/*                The 'outputFormat' parameter specifies the output format (ie. POV, RAW,       */
/*                RIB, etc.);  the following symbolic constants are defined in config.h:        */
/*                                                                                              */
/*                     Symbol           Description                                             */
/*                 ------------------------------------------------------------------------     */
/*                     RAW..............Output raw vertex data (ie. nine space-delimited        */
/*                                      values per line, corresponding to the x-y-z coor-       */
/*                                      dinates of each vertex.)                                */
/*                     RIB..............Output RenderMan compatible triangles.                  */
/*                     POV..............Output Persistence of Vision V2.x compatible            */
/*                                      triangles.                                              */
/*                                                                                              */
/*                The 'triangleType' parameter specifies the type of triangle to output.        */
/*                The following symbolic constants are defined in config.h:                     */
/*                                                                                              */
/*                     Symbol           Description                                             */
/*                 ------------------------------------------------------------------------     */
/*                     FLAT.............Output flat triangles (only vertex data).               */
/*                     SMOOTH...........Output smooth triangles (vertices and normals).         */
/*                                                                                              */
/*                The 'spacesToIndent' parameter specifies the number of spaces to indent       */
/*                each line of output.                                                          */
/*                                                                                              */
/*  NOTES:        When adding a new output format, you must add a case to BOTH switch           */
/*                statements, even if they are both exactly the same.                           */
/*                                                                                              */
/*==============================================================================================*/

   void OutputTriangles(ostream&      outputFile,
                        USHORT        outputFormat,
                        USHORT        triangleType,
                        USHORT        spacesToIndent,
                        TRIANGLELIST& triangleList)
   {

      INT       i;
      CHARPTR   indentString;
      TRIANGLE* t;

      VECTOR    side1, side2;
      VECTOR    normal;

      indentString = new CHAR[spacesToIndent+1];
      for (i=0;i<spacesToIndent;i++) indentString[i]=' ';
      indentString[i]=0x00;

      triangleList.gotoFirst();
      for (i=0;i<triangleList.Count();i++)
      {
         t = triangleList.Current();

         side1  = t->v1 - t->v2;
         side2  = t->v3 - t->v2;
         normal = side1 ^ side2;

         if (length(normal) < DEGENERATE_THRESHOLD)
	 {
            triangleList.gotoNext();
            continue;
	 }

         if (triangleType==SMOOTH)
         {
            if (outputFormat==POV)
            {
               outputFile<<indentString<<"smooth_triangle{";
               outputFile<<'<'<<t->v1.x<<','<<t->v1.y<<','<<t->v1.z<<'>'<<',';
               outputFile<<'<'<<t->n1.x<<','<<t->n1.y<<','<<t->n1.z<<'>'<<',';
               outputFile<<'<'<<t->v2.x<<','<<t->v2.y<<','<<t->v2.z<<'>'<<',';
               outputFile<<'<'<<t->n2.x<<','<<t->n2.y<<','<<t->n2.z<<'>'<<',';
               outputFile<<'<'<<t->v3.x<<','<<t->v3.y<<','<<t->v3.z<<'>'<<',';
               outputFile<<'<'<<t->n3.x<<','<<t->n3.y<<','<<t->n3.z<<'>'<<'}'<<endl;
            }
            else if (outputFormat==RAW)
            {
               outputFile<<indentString;
               outputFile<<t->v1.x<<' '<<t->v1.y<<' '<<t->v1.z<<' ';
               outputFile<<t->v2.x<<' '<<t->v2.y<<' '<<t->v2.z<<' ';
               outputFile<<t->v3.x<<' '<<t->v3.y<<' '<<t->v3.z<<endl;
            }
            else if (outputFormat==RIB)
            {
               outputFile<<indentString<<"Polygon \"P\" [";
               outputFile<<t->v1.x<<' '<<t->v1.y<<' '<<t->v1.z<<' ';
               outputFile<<t->v2.x<<' '<<t->v2.y<<' '<<t->v2.z<<' ';
               outputFile<<t->v3.x<<' '<<t->v3.y<<' '<<t->v3.z<<" ] ";
               outputFile<<"\"N\" [";
               outputFile<<t->n1.x<<' '<<t->n1.y<<' '<<t->n1.z<<' ';
               outputFile<<t->n2.x<<' '<<t->n2.y<<' '<<t->n2.z<<' ';
               outputFile<<t->n3.x<<' '<<t->n3.y<<' '<<t->n3.z<<" ]"<<endl;
            }
            else if (outputFormat==DXF)
	    {
               outputFile<<"  0"<<endl;
               outputFile<<"3DFACE"<<endl;
               outputFile<<"  8"<<endl;
               outputFile<<0<<endl;
               outputFile<<" 10"<<endl;
               outputFile<<t->v1.x<<endl;
               outputFile<<" 20"<<endl;
               outputFile<<t->v1.y<<endl;
               outputFile<<" 30"<<endl;
               outputFile<<t->v1.z<<endl;
               outputFile<<" 11"<<endl;
               outputFile<<t->v2.x<<endl;
               outputFile<<" 21"<<endl;
               outputFile<<t->v2.y<<endl;
               outputFile<<" 31"<<endl;
               outputFile<<t->v2.z<<endl;
               outputFile<<" 12"<<endl;
               outputFile<<t->v3.x<<endl;
               outputFile<<" 22"<<endl;
               outputFile<<t->v3.y<<endl;
               outputFile<<" 32"<<endl;
               outputFile<<t->v3.z<<endl;
               outputFile<<" 13"<<endl;
               outputFile<<t->v3.x<<endl;
               outputFile<<" 23"<<endl;
               outputFile<<t->v3.y<<endl;
               outputFile<<" 33"<<endl;
               outputFile<<t->v3.z<<endl;
	    }
            else if (outputFormat==VIVID)
	    {
               outputFile<<indentString;
               outputFile<<"patch { vertex "<<t->v1.x<<" "<<t->v1.y<<" "<<t->v1.z<<" "
                                 <<"normal "<<t->n1.x<<" "<<t->n1.y<<" "<<t->n1.z<<" "
                                 <<"vertex "<<t->v2.x<<" "<<t->v2.y<<" "<<t->v2.z<<" "
                                 <<"normal "<<t->n2.x<<" "<<t->n2.y<<" "<<t->n2.z<<" "
                                 <<"vertex "<<t->v3.x<<" "<<t->v3.y<<" "<<t->v3.z<<" "
                                 <<"normal "<<t->n3.x<<" "<<t->n3.y<<" "<<t->n3.z<<" }"<<endl;
	    }
         }
         else
         {
            if (outputFormat==POV)
            {
               outputFile<<indentString<<"triangle{";
               outputFile<<'<'<<t->v1.x<<','<<t->v1.y<<','<<t->v1.z<<'>'<<',';
               outputFile<<'<'<<t->v2.x<<','<<t->v2.y<<','<<t->v2.z<<'>'<<',';
               outputFile<<'<'<<t->v3.x<<','<<t->v3.y<<','<<t->v3.z<<'>'<<'}'<<endl;
            }
            else if (outputFormat==RAW)
            {
               outputFile<<indentString;
               outputFile<<t->v1.x<<' '<<t->v1.y<<' '<<t->v1.z<<' ';
               outputFile<<t->v2.x<<' '<<t->v2.y<<' '<<t->v2.z<<' ';
               outputFile<<t->v3.x<<' '<<t->v3.y<<' '<<t->v3.z<<endl;
            }
            else if (outputFormat==RIB)
            {
               outputFile<<indentString<<"Polygon \"P\" [";
               outputFile<<t->v1.x<<' '<<t->v1.y<<' '<<t->v1.z<<' ';
               outputFile<<t->v2.x<<' '<<t->v2.y<<' '<<t->v2.z<<' ';
               outputFile<<t->v3.x<<' '<<t->v3.y<<' '<<t->v3.z<<" ]"<<endl;
            }
            else if (outputFormat==DXF)
	    {
               outputFile<<"  0"<<endl;
               outputFile<<"3DFACE"<<endl;
               outputFile<<"  8"<<endl;
               outputFile<<0<<endl;
               outputFile<<" 10"<<endl;
               outputFile<<t->v1.x<<endl;
               outputFile<<" 20"<<endl;
               outputFile<<t->v1.y<<endl;
               outputFile<<" 30"<<endl;
               outputFile<<t->v1.z<<endl;
               outputFile<<" 11"<<endl;
               outputFile<<t->v2.x<<endl;
               outputFile<<" 21"<<endl;
               outputFile<<t->v2.y<<endl;
               outputFile<<" 31"<<endl;
               outputFile<<t->v2.z<<endl;
               outputFile<<" 12"<<endl;
               outputFile<<t->v3.x<<endl;
               outputFile<<" 22"<<endl;
               outputFile<<t->v3.y<<endl;
               outputFile<<" 32"<<endl;
               outputFile<<t->v3.z<<endl;
               outputFile<<" 13"<<endl;
               outputFile<<t->v3.x<<endl;
               outputFile<<" 23"<<endl;
               outputFile<<t->v3.y<<endl;
               outputFile<<" 33"<<endl;
               outputFile<<t->v3.z<<endl;
	    }
            else if (outputFormat==VIVID)
	    {
               outputFile<<indentString;
               outputFile<<"patch { vertex "<<t->v1.x<<" "<<t->v1.y<<" "<<t->v1.z<<" "
                                 <<"normal "<<t->n1.x<<" "<<t->n1.y<<" "<<t->n1.z<<" "
                                 <<"vertex "<<t->v2.x<<" "<<t->v2.y<<" "<<t->v2.z<<" "
                                 <<"normal "<<t->n2.x<<" "<<t->n2.y<<" "<<t->n2.z<<" "
                                 <<"vertex "<<t->v3.x<<" "<<t->v3.y<<" "<<t->v3.z<<" "
                                 <<"normal "<<t->n3.x<<" "<<t->n3.y<<" "<<t->n3.z<<" }"<<endl;
	    }
         }

         triangleList.gotoNext();
      }

      delete indentString;

   }



/*==============================================================================================*/
/*  OutputObjects()                                                                             */
/*==============================================================================================*/
/*                                                                                              */
/*  SYNTAX:       USHORT OutputObjects(ostream& outputFile, TTFont& font,                       */
/*                                     Font3DOptions& options);                                 */
/*                                                                                              */
/*  DESCRIPTION:  This function outputs the triangle information that makes up a group of       */
/*                glyphs to a specified stream.  The string to generate is found in the         */
/*                Font3DOptions structure 'options', the font to use is given by 'font',        */
/*                and the output stream is 'outputFile'.                                        */
/*                                                                                              */
/*  ERRORS:       Possible error conditions upon return from this function:                     */
/*                                                                                              */
/*                   ERR_NoError...............Successful completion.                           */
/*                   ERR_OutOfMemory...........An attempt to allocate memory failed.            */
/*                   There are more...                                                          */
/*                                                                                              */
/*==============================================================================================*/

   USHORT OutputObjects(ostream&       outputFile,
                        TTFont&        font,
                        Font3DOptions& options)
   {
      INT          i;
      INT          triangleCount;
      TRIANGLELIST frontFaceTriangleList;
      TRIANGLELIST backFaceTriangleList;
      TRIANGLELIST frontBevelTriangleList;
      TRIANGLELIST backBevelTriangleList;
      TRIANGLELIST sideTriangleList;
      USHORT       success;
      USHORT       code;
      DOUBLE       startX,endX;

      DOUBLE xmin,xmax,ymin,ymax,zmin,zmax,upem,xdelta,ydelta;
      DOUBLE thisymin,thisymax;
      VECTOR offset(0.0,0.0,0.0);

      USHORT       partsCount;

      outputFile<<setprecision(options.outputPrecision);
      code = options.c;
      upem = (DOUBLE)font.UnitsPerEm();

      if (options.string==NULL)
      {
         xmin = (DOUBLE)font.GlyphXMin(font.CharacterMap(code))/(DOUBLE)upem;
         xmax = (DOUBLE)font.GlyphXMax(font.CharacterMap(code))/(DOUBLE)upem;

         if (options.xPosition==LEFT) xdelta = -xmin;
         else if (options.xPosition==CENTER) xdelta = -(xmin+xmax)/2;
         else xdelta = -xmax;

         xmin+=xdelta;
         xmax+=xdelta;

         ymin = (DOUBLE)font.GlyphYMin(font.CharacterMap(code))/(DOUBLE)upem;
         ymax = (DOUBLE)font.GlyphYMax(font.CharacterMap(code))/(DOUBLE)upem;

         if (options.yPosition==BOTTOM) ydelta = -ymin;
         else if (options.yPosition==BASELINE) ydelta=0.0;
         else if (options.yPosition==CENTER) ydelta = -(ymin+ymax)/2;
         else ydelta = -ymax;

         ymin+=ydelta;
         ymax+=ydelta;

         if (options.zPosition==FRONT) { zmin=0; zmax=options.depth; }
         else if (options.zPosition==CENTER) { zmin=-options.depth/2; zmax=options.depth/2; }
         else { zmin=-options.depth; zmax=0; }

         offset = VECTOR(xdelta,ydelta,0.0);

         success = CreateFaces(font, font.CharacterMap(code),
                               options, offset, frontFaceTriangleList,
                               backFaceTriangleList);

         if (success!=ERR_NoError)
         {
            return success;
         }

         success = CreateSides(font, font.CharacterMap(code),
                               options, offset, frontBevelTriangleList,
                               backBevelTriangleList, sideTriangleList);

         if (success!=ERR_NoError)
         {
            frontFaceTriangleList.Empty();
            backFaceTriangleList.Empty();
            return success;
         }

      }
      else
      {
         startX = font.Glyph(options.string[0])->LeftSideBearing()/upem;
         endX=0;

         ymin = (DOUBLE)font.GlyphYMin(font.CharacterMap(options.string[0]))/(DOUBLE)upem;
         ymax = (DOUBLE)font.GlyphYMax(font.CharacterMap(options.string[0]))/(DOUBLE)upem;

         for (i=0;i<options.stringLength;i++)
         {
            endX+=( font.Glyph(options.string[i])->AdvanceWidth())/upem;
            if (i<options.stringLength) 
               endX += font.Kerning(font.CharacterMap(options.string[i]),
                                    font.CharacterMap(options.string[i+1]))/upem;

            thisymin=(DOUBLE)font.GlyphYMin(font.CharacterMap(options.string[i]))/(DOUBLE)upem;
            thisymax=(DOUBLE)font.GlyphYMax(font.CharacterMap(options.string[i]))/(DOUBLE)upem;

            if (ymin>thisymin) ymin=thisymin;
            if (ymax<thisymax) ymax=thisymax;

         }

         endX -= font.Glyph(options.string[i-1])->RightSideBearing()/upem;

         xmin = startX;
         xmax = endX;

         if (options.xPosition==LEFT) xdelta = -xmin;
         else if (options.xPosition==CENTER) xdelta = -(xmin+xmax)/2;
         else xdelta = -xmax;

         xmin+=xdelta;
         xmax+=xdelta;

         if (options.yPosition==BOTTOM) ydelta = -ymin;
         else if (options.yPosition==BASELINE) ydelta = 0.0;
         else if (options.yPosition==CENTER) ydelta = -(ymin+ymax)/2;
         else ydelta = -ymax;

         ymin+=ydelta;
         ymax+=ydelta;

         if (options.zPosition==FRONT) { zmin=0; zmax=options.depth; }
         else if (options.zPosition==CENTER) { zmin=-options.depth/2; zmax=options.depth/2; }
         else { zmin=-options.depth; zmax=0; }

         offset = VECTOR(xdelta,ydelta,0.0);

         for (i=0;i<options.stringLength;i++)
         {

            success = CreateFaces(font, font.CharacterMap(options.string[i]),
                                  options, offset, frontFaceTriangleList,
                                  backFaceTriangleList);

            if (success!=ERR_NoError)
            {
               return success;
            }

            success = CreateSides(font, font.CharacterMap(options.string[i]),
                                  options, offset, frontBevelTriangleList,
                                  backBevelTriangleList, sideTriangleList);
            if (success!=ERR_NoError)
            {
               frontFaceTriangleList.Empty();
               backFaceTriangleList.Empty();
               return success;
            }

            offset.x += (  font.Glyph(options.string[i])->AdvanceWidth() )/upem; 
            if (i<options.stringLength-1) 
               offset.x += font.Kerning(font.CharacterMap(options.string[i]),
                                        font.CharacterMap(options.string[i+1]))/upem;
         }
      }


      triangleCount = 0;

      if (options.frontFaceVisible)
         triangleCount+=frontFaceTriangleList.Count();  
      if (options.backFaceVisible)
         triangleCount+=backFaceTriangleList.Count();
      if (options.frontBevelVisible
          && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         triangleCount+=frontBevelTriangleList.Count();
      if (options.backBevelVisible
          && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         triangleCount+=backBevelTriangleList.Count();
      if (options.sideVisible)
         triangleCount+=sideTriangleList.Count();



     /*---------------------------------------------------------------------------*/
     /*  Output RAW Triangles...                                                  */
     /*---------------------------------------------------------------------------*/

      if (options.outputFormat==RAW)
      {
         OutputTriangles(outputFile,RAW,FLAT,0,frontFaceTriangleList);
         OutputTriangles(outputFile,RAW,FLAT,0,backFaceTriangleList);
         OutputTriangles(outputFile,RAW,FLAT,0,frontBevelTriangleList);
         OutputTriangles(outputFile,RAW,FLAT,0,backBevelTriangleList);
         OutputTriangles(outputFile,RAW,FLAT,0,sideTriangleList);
      }

     /*---------------------------------------------------------------------------*/
     /*  Output RenderMan RIB File...                                             */
     /*---------------------------------------------------------------------------*/

      else if (options.outputFormat==RIB)
      {
         outputFile<<"#-----------------------------------------------------------"<<endl;
         outputFile<<"# This text object description was generated by"<<endl;
         outputFile<<"# Font3D Version 1.50"<<endl;
         outputFile<<"#"<<endl;
         outputFile<<"# Total number of triangles: "<<triangleCount<<endl;
         outputFile<<"#-----------------------------------------------------------"<<endl;
         outputFile<<endl;

         if (options.frontFaceVisible)
         {
            outputFile<<"# Front Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,frontFaceTriangleList);
            outputFile<<endl<<endl;
	 }
         if (options.backFaceVisible)
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,backFaceTriangleList);
            outputFile<<endl<<endl;
         }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,frontBevelTriangleList);
            outputFile<<endl<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,backBevelTriangleList);
            outputFile<<endl<<endl;
         }
         if (options.sideVisible)
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,sideTriangleList);
            outputFile<<endl<<endl;
         }
      }

     /*---------------------------------------------------------------------------*/
     /*  Output POV 2.x Triangles...                                              */
     /*---------------------------------------------------------------------------*/

      else if (options.outputFormat==POV)
      {
         outputFile<<"//-----------------------------------------------------------"<<endl;
         outputFile<<"// This text object description was generated by"<<endl;
         outputFile<<"// Font3D Version 1.50"<<endl;
         outputFile<<"// "<<endl;
         outputFile<<"// Total number of triangles: "<<triangleCount<<endl;
         outputFile<<"//-----------------------------------------------------------"<<endl;
         outputFile<<endl;


         outputFile<<"#declare "<<options.objectName<<"_XMin   = "<<xmin<<endl;
         outputFile<<"#declare "<<options.objectName<<"_YMin   = "<<ymin<<endl;
         outputFile<<"#declare "<<options.objectName<<"_ZMin   = "<<zmin<<endl;
         outputFile<<"#declare "<<options.objectName<<"_XMax   = "<<xmax<<endl;
         outputFile<<"#declare "<<options.objectName<<"_YMax   = "<<ymax<<endl;
         outputFile<<"#declare "<<options.objectName<<"_ZMax   = "<<zmin<<endl;
         outputFile<<endl;
         outputFile<<"#declare "<<options.objectName<<"_Width  = "<<(xmax-xmin)<<endl;
         outputFile<<"#declare "<<options.objectName<<"_Height = "<<(ymax-ymin)<<endl;
         outputFile<<"#declare "<<options.objectName<<"_Depth  = "<<(zmax-zmin)<<endl;
         outputFile<<endl<<endl;

         partsCount = 0;

         if (options.frontFaceVisible)
         {
            outputFile<<"#declare FF_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,frontFaceTriangleList);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.backFaceVisible)
         {
            outputFile<<"#declare BF_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,backFaceTriangleList);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         {
            outputFile<<"#declare FB_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,frontBevelTriangleList);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"#declare BB_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,backBevelTriangleList);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.sideVisible)
         {
            outputFile<<"#declare S_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,sideTriangleList);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }

         if (partsCount < 2)
         {
            outputFile<<"#declare "<<options.objectName<<" = object { ";

            if (options.frontFaceVisible)
            {
               outputFile<<"FF_"<<options.objectName<<" ";
               if (options.frontFaceTextureName != NULL)
                  outputFile<<"texture { "<<options.frontFaceTextureName<<" }";
            }
            else if (options.backFaceVisible)
            {
               outputFile<<"BF_"<<options.objectName<<" ";
               if (options.backFaceTextureName != NULL)
	          outputFile<<"texture { "<<options.backFaceTextureName<<" }";
            }
            else if (options.frontBevelVisible
                     && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
            {
               outputFile<<"FB_"<<options.objectName<<" ";
               if (options.frontBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.frontBevelTextureName<<" }";
            }
            else if (options.backBevelVisible
                     && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
            {
               outputFile<<"BB_"<<options.objectName<<" ";
               if (options.backBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.backBevelTextureName<<" }";
            }
            else if (options.sideVisible)
            {
               outputFile<<"S_"<<options.objectName<<" ";
               if (options.sideTextureName != NULL)
                  outputFile<<"texture { "<<options.sideTextureName<<" }";
            }

            outputFile<<" }"<<endl;
         }
         else
         {
            outputFile<<"#declare "<<options.objectName<<" = ";
            outputFile<<"  union {"<<endl;
            if (options.frontFaceVisible)
            {
               outputFile<<"    object { FF_"<<options.objectName<<" ";
               if (options.frontFaceTextureName != NULL)
                  outputFile<<"texture { "<<options.frontFaceTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.backFaceVisible)
            {
               outputFile<<"    object { BF_"<<options.objectName<<" ";
               if (options.backFaceTextureName != NULL)
	          outputFile<<"texture { "<<options.backFaceTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.frontBevelVisible
                && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
            {
               outputFile<<"    object { FB_"<<options.objectName<<" ";
               if (options.frontBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.frontBevelTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.backBevelVisible
                && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
            {
               outputFile<<"    object { BB_"<<options.objectName<<" ";
               if (options.backBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.backBevelTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.sideVisible)
            {
               outputFile<<"    object { S_"<<options.objectName<<" ";
               if (options.sideTextureName != NULL)
                  outputFile<<"texture { "<<options.sideTextureName<<" }";
               outputFile<<" }"<<endl;
            }

            outputFile<<" }"<<endl;
         }

      }

     /*---------------------------------------------------------------------------*/
     /*  Output AutoCad DXF Triangles...                                          */
     /*---------------------------------------------------------------------------*/

      else if (options.outputFormat==DXF)
      {
         outputFile<<"999"<<endl;
         outputFile<<"This text object description was generated by"<<endl;
         outputFile<<"999"<<endl;
         outputFile<<"Font3D Version 1.50"<<endl;
         outputFile<<"999"<<endl;
         outputFile<<"Total number of triangles: "<<triangleCount<<endl;

         if (options.frontFaceVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,frontFaceTriangleList);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.backFaceVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,backFaceTriangleList);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,frontBevelTriangleList);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,backBevelTriangleList);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
         }
         if (options.sideVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,sideTriangleList);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
         }
         outputFile<<"  0"<<endl;
         outputFile<<"EOF"<<endl;
      }

     /*---------------------------------------------------------------------------*/
     /*  Output VIVID 2.0 Triangles...                                            */
     /*---------------------------------------------------------------------------*/

      else if (options.outputFormat==VIVID)
      {
         outputFile<<"//-----------------------------------------------------------"<<endl;
         outputFile<<"// This text object description was generated by"<<endl;
         outputFile<<"// Font3D Version 1.50"<<endl;
         outputFile<<"// "<<endl;
         outputFile<<"// Total number of triangles: "<<triangleCount<<endl;
         outputFile<<"//-----------------------------------------------------------"<<endl;
         outputFile<<endl;

         if (options.frontFaceVisible)
	 {
            outputFile<<"// Front Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,frontFaceTriangleList);
            outputFile<<endl;
	 }
         if (options.backFaceVisible)
	 {
            outputFile<<"// Back Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,backFaceTriangleList);
            outputFile<<endl;
	 }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
	 {
            outputFile<<"// Front Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,frontBevelTriangleList);
            outputFile<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"// Back Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,backBevelTriangleList);
            outputFile<<endl;
	 }
         if (options.sideVisible)
	 {
            outputFile<<"// Side Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,sideTriangleList);
            outputFile<<endl;
	 }
      }


     /*---------------------------------------------------------------------------*/
     /*  Clean up...                                                              */
     /*---------------------------------------------------------------------------*/

      frontFaceTriangleList.Empty();
      backFaceTriangleList.Empty();
      frontBevelTriangleList.Empty();
      backBevelTriangleList.Empty();
      sideTriangleList.Empty();

      return ERR_NoError;

   }
