//////////////////////////////////////////////////////////////////////////////
//  BMP loading routines                                                    //
//                                                                          //
//  Note(s):                                                                //
//                                                                          //
//  Revisions:                                                              //
//    27/12/1997, Initial version, loader for 1, 4, and 8 bpp              //
//                                                                          //
//  Copyright (c) 1997 by Matjaz Trtnik aka maLi/MaLixa                     //
//  Email: mtrtnik@bigfoot.com                                              //
//  Web: http://www2.arnes.si/~ssdmalic/mali/                               //
//                                                                          //
//  The author takes no responsibility, if something in this document or    //
//  the accompanying classes causes any kind of data loss or damage to      //
//  your hardware.                                                          //
//                                                                          //
//  You can use this product strictly for *NON* commercial programs.        //
//  If you want to use it for commercial programs please contact author.    //
//                                                                          //
//  You are not permitted to distribute, sell or use any part of            //
//  this source for your software without special permision of author.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <assert.h>

#define _DEBUG_

#define BI_RGB          0    // No compression
#define BI_RLE8         1    // RLE8 compression (256 colors)
#define BI_RLE4         2    // RLE4 compression (16 colors)

#pragma pack(1)
typedef struct _BITMAPFILEHEADER {  // Offset   Size
    short   bfType;                 //      0      2
    long    bfSize;                 //      2      4
    short   bfReserved1;            //      6      2
    short   bfReserved2;            //      8      2
    long    bfOffBits;              //     10      4
} BITMAPFILEHEADER;                 // Total size: 14
#pragma pack()

#pragma pack(1)
typedef struct _BITMAPINFOHEADER {  // Offset   Size
    long    biSize;                 //      0      4
    long    biWidth;                //      4      4
    long    biHeight;               //      8      4
    short   biPlanes;               //     12      2
    short   biBitCount;             //     14      2
    long    biCompression;          //     16      4
    long    biSizeImage;            //     20      4
    long    biXPelsPerMeter;        //     24      4
    long    biYPelsPerMeter;        //     28      4
    long    biClrUsed;              //     32      4
    long    biClrImportant;         //     36      4
} BITMAPINFOHEADER;                 // Total size: 40
#pragma pack()

#pragma pack(1)
typedef struct _RGBQUAD {
    char    rgbBlue;
    char    rgbGreen;
    char    rgbRed;
    char    rgbReserved;
} RGBQUAD;
#pragma pack()

BITMAPFILEHEADER        bmfh;
BITMAPINFOHEADER        bmih;
RGBQUAD                *bmcoltab;
long                    bmnumcolors;
char                    pal[768];

void SetColor(char num, char r, char g, char b)
{
  outp(0x3c8, num);
  outp(0x3c9, r >> 2);
  outp(0x3c9, g >> 2);
  outp(0x3c9, b >> 2);
  pal[num*3+0] = r >> 2;
  pal[num*3+1] = g >> 2;
  pal[num*3+2] = b >> 2;
}

void BMPBufferFlip(long biWidth, long biHeight, char *buffer)
{
  char  byte;
  long  x, y;

  for (y = 0; y < biHeight/2; y++)
    for (x = 0; x < biWidth; x++) {
      byte = buffer[y*biWidth+x];
      buffer[y*biWidth+x] = buffer[(biHeight-y-1)*biWidth+x];
      buffer[(biHeight-y-1)*biWidth+x] = byte;
    }
}

char *LoadBMP(char *filename, long *biWidth, long *biHeight)
{
  FILE                   *f;
  char                   *buffer, byte;
  char                    byte1, byte2;
  char                    hinib, lonib;
  char                    deltax, deltay;
  long                    x, y;
  long                    size, k, i;

  f = fopen(filename,"rb");
  assert(f != NULL);

  fread(&bmfh, sizeof(BITMAPFILEHEADER), 1, f);
  fread(&bmih, sizeof(BITMAPINFOHEADER), 1, f);
  /*
  #ifdef _DEBUG_
  printf("\nType   : %c%c",bmfh.bfType,bmfh.bfType>>8);
  printf("\nSize   : %d",bmfh.bfSize);
  printf("\nRes1   : %d",bmfh.bfReserved1);
  printf("\nRes2   : %d",bmfh.bfReserved2);
  printf("\nOffBits: %d\n",bmfh.bfOffBits);
  printf("\nSize       : %d",bmih.biSize);
  printf("\nWidth      : %d",bmih.biWidth);
  printf("\nHeight     : %d",bmih.biHeight);
  printf("\nPlanes     : %d",bmih.biPlanes);
  printf("\nBitCount   : %d",bmih.biBitCount);
  printf("\nCompression: %d",bmih.biCompression);
  printf("\nSizeImage  : %d",bmih.biSizeImage);
  printf("\nXPels      : %d",bmih.biXPelsPerMeter);
  printf("\nYPels      : %d",bmih.biYPelsPerMeter);
  printf("\nClrUsed    : %d",bmih.biClrUsed);
  printf("\nClrImpor   : %d",bmih.biClrImportant);
  printf("\n");
  getch();
  #endif
  */

  bmnumcolors = (1 << bmih.biBitCount);
  //printf("\nNumber of colors: %d", bmnumcolors);

  // Bitmap size
  size = bmih.biWidth * bmih.biHeight;

  // Load color table if biBitCount < 24
  if (bmih.biBitCount != 24) {
    bmcoltab = (RGBQUAD *)calloc(bmnumcolors, sizeof(RGBQUAD));
    #ifdef _DEBUG_
    assert(bmcoltab != NULL);
    #endif
//  mv_Error("Cannot allocate memory for BMP color table");
    fread(bmcoltab, bmnumcolors*sizeof(RGBQUAD), 1, f);
  }
  else
    size *= 3;

  for (k = 0; k < bmnumcolors; k++)
    SetColor(k, bmcoltab[k].rgbRed, bmcoltab[k].rgbGreen, bmcoltab[k].rgbBlue);

  // Allocate memory for picture
  buffer = (char *)calloc(size, sizeof(char));
  #ifdef _DEBUG_
  assert(buffer != NULL);
  #endif

  fseek(f, bmfh.bfOffBits, SEEK_SET);
  switch (bmih.biCompression) {

    // RGB
    case BI_RGB:
      switch (bmih.biBitCount) {
        // 2 colors
        case 1:
          k = 0;
          for (y = 0; y < bmih.biHeight; y++) {
            for (x = 0; x < (bmih.biWidth >> 3); x++) {
              fread(&byte, sizeof(char), 1, f);
              for (i = 7; i >= 0; i--)
                buffer[k++] = ((byte & (1 << i)) >> i);
            }
            // biWidth % 8 pixels still to draw
            if ((bmih.biWidth % 8) > 0) {
              fread(&byte, sizeof(char), 1, f);
              size = 8 - (bmih.biWidth % 8);
              for (i = 7; i >= size; i--)
                buffer[k++] = ((byte & (1 << i)) >> i);
              // We read one more byte
              size = 1;
            }
            else
              size = 0;
            // Scan line must be zero-padded to end on a 32-bit boundary
            size += (bmih.biWidth >> 3);
            while (size % 4 != 0) {
              fread(&byte, sizeof(char), 1, f);
              size++;
            }
          }
        break;
        // 16 colors
        case 4:
          k = 0;
          for (y = 0; y < bmih.biHeight; y++) {
            for (x = 0; x < (bmih.biWidth >> 1); x++) {
              fread(&byte, sizeof(char), 1, f);
              buffer[k++] = ((byte >> 4) & 0x0F);
              buffer[k++] = (byte & 0x0F);
            }
            // Last pixel of row if number of pixels in row is odd
            if ((bmih.biWidth % 2) == 1) {
              fread(&byte, sizeof(char), 1, f);
              buffer[k++] = ((byte >> 4) & 0x0F);
              // We read one more byte
              size = 1;
            }
            else
              size = 0;
            // Scan line must be zero-padded to end on a 32-bit boundary
            size += (bmih.biWidth >> 1);
            while (size % 4 != 0) {
              fread(&byte, sizeof(char), 1, f);
              size++;
            }
          }
        break;
        // 256 colors
        case 8:
          k = 0;
          for (y = 0; y < bmih.biHeight; y++) {
            for (x = 0; x < bmih.biWidth; x++)
              fread(&buffer[k++], sizeof(char), 1, f);

            // Scan line must be zero-padded to end on a 32-bit boundary
            size += bmih.biWidth;
            while (size % 4 != 0) {
              fread(&byte, sizeof(char), 1, f);
              size++;
            }
          }
        break;
        // 16.7M colors
        case 24:
          assert(0);
          break;
        default:
          #ifdef _DEBUG_
          assert(0);
          #endif
      }
      break;

    // RLE8
    case BI_RLE8:
      x = 0;
      y = 0;

      while (y*bmih.biWidth+x < size) {
        fread(&byte1, sizeof(char), 1, f);
        fread(&byte2, sizeof(char), 1, f);

        // Absolute Mode
        if (byte1 == 0 && byte2 >= 0x03 && byte2 <= 0xFF) {
          for (k = 0; k < byte2; k++) {
            fread(&byte, sizeof(char), 1, f);
            buffer[y*bmih.biWidth+x++] = byte;
          }
          // Each run must be aligned on a word boundary
          if ((byte2 % 2) != 0)
            fread(&byte, sizeof(char), 1, f);
        }
        // Encoded Mode
        else {
          // Indicate an escape
          if (byte1 == 0) {
            switch (byte2) {
              // End of line
              case 0:
                x = 0;
                y++;
                break;
              // End of bitmap
              case 1:
                break;
              // Delta
              case 2:
                fread(&deltax, sizeof(char), 1, f);
                fread(&deltay, sizeof(char), 1, f);
                x += deltax;
                y += deltay;
                break;

              default:
                assert(0);
            }
          }
          else {
            for (k = 0; k < byte1; k++)
              buffer[y*bmih.biWidth+x++] = byte2;
          }
        }
      }
      break;

    // RLE4
    case BI_RLE4:
      x = 0;
      y = 0;

      while (y*bmih.biWidth+x < size) {

        fread(&byte1, sizeof(char), 1, f);
        fread(&byte2, sizeof(char), 1, f);

        // Absolute Mode
        if (byte1 == 0 && byte2 >= 0x03 && byte2 <= 0xFF) {
          byte2 >>= 1;
          for (k = 0; k < byte2; k++) {
            fread(&byte, sizeof(char), 1, f);
            hinib = ((byte >> 4) & 0x0F);
            lonib = (byte & 0x0F);
            buffer[y*bmih.biWidth+x++] = hinib;
            buffer[y*bmih.biWidth+x++] = lonib;
          }
          // Each run must be aligned on a word boundary
          if ((byte2 % 2) != 0)
            fread(&byte, sizeof(char), 1, f);
        }
        // Encoded Mode
        else {
          // Indicate an escape
          if (byte1 == 0) {
            switch (byte2) {
              // End of line
              case 0:
                x = 0;
                y++;
                break;
              // End of bitmap
              case 1:
                break;
              // Delta
              case 2:
                fread(&deltax, sizeof(char), 1, f);
                fread(&deltay, sizeof(char), 1, f);
                x += deltax;
                y += deltay;
                break;

              default:
                assert(0);
            }
          }
          else {
            hinib = ((byte2 >> 4) & 0x0F);
            lonib = (byte2 & 0x0F);

            byte2 = byte1 % 2;
            byte1 >>= 1;
            for (k = 0; k < byte1; k++) {
              buffer[y*bmih.biWidth+x++] = hinib;
              buffer[y*bmih.biWidth+x++] = lonib;
            }
            if (byte2 != 0)
              buffer[y*bmih.biWidth+x++] = hinib;
          }
        }
      }
      break;


    default:            // Should not happen
      #ifdef _DEBUG_
      assert(0);
      #endif
  }
  fclose(f);

  BMPBufferFlip(bmih.biWidth, bmih.biHeight, buffer);
  *biWidth = bmih.biWidth;
  *biHeight = bmih.biHeight;

  return buffer;
}



//  Save .BMP picture  //
//  Paramaters:
//    biFileName    = Name of file bitmap is saved into
//    biStartX      = Bitmap starting X poition
//    biStartY      = Bitmap starting Y poition
//    biWidth       = Width of bitmap
//    biHeight      = Height of bitmap
//    biBuffer      = Buffer with bitmap
//    coltab        = Bitmap color table
//    biBitCount    = Number of bits for one pixel (1, 4, 8, 24)
//    biCompression = Use RGB or RLE compression (BI_RGB, BI_RLE8, BI_RLE4)
char SaveBMP(char *biFileName, long biStartX, long biStartY,
             long biWidth, long biHeight, char *biBuffer, char *coltab,
             char  biBitCount, char biCompression)
{
  FILE  *f;
  char   byte;
  long   size, k, i;
  long   x, y;

  bmnumcolors = (1 << biBitCount);
  // True color BMP file does not have any color map data
  if (biBitCount == 24) bmnumcolors = 0;

  // BITMAPFILEHEADER
  bmfh.bfType = 0x4D42;         // 'BM'
  bmfh.bfReserved1 = 0;         // Reserved, always 0
  bmfh.bfReserved2 = 0;         // Reserved, always 0
  // bfOffBits = Byte offset from BITMAPFILEHEADER to the actual bitmap data
  bmfh.bfOffBits = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) +
                   bmnumcolors*sizeof(RGBQUAD);
  bmfh.bfSize = bmfh.bfOffBits; // File size, will be written below

  // BITMAPINFOHEADER
  bmih.biSize = sizeof(BITMAPINFOHEADER);       // Size of BITMAPINFOHEADER
  bmih.biWidth = biWidth;
  bmih.biHeight = biHeight;
  bmih.biPlanes = 1;            // Number of planes for the target device
  bmih.biBitCount = biBitCount; // Number of bits per pixel (1, 4, 8, 24)

  bmih.biSizeImage = biWidth * biHeight;  // Size of image in bytes
  switch (biBitCount) {
    // Bitmap has 2 colors
    case 1:
      bmih.biCompression = BI_RGB;        // No compression allowed
      bmih.biSizeImage >>= 3;             // 8 pixels can be represented with 1 byte
      break;

    // Bitmap has 16 colors
    case 4:
      if (biCompression == BI_RLE4)
        bmih.biCompression = BI_RLE4;     // Can have BI_RLE4 compression
      else
        bmih.biCompression = BI_RGB;      // Default compression
      bmih.biSizeImage >>= 1;             // 2 pixels can be represented with 1 byte
      break;

    // Bitmap has 256 colors
    case 8:
      if (biCompression == BI_RLE8)
        bmih.biCompression = BI_RLE8;     // Can have BI_RLE8 compression
      else
        bmih.biCompression = BI_RGB;      // Default compression
      break;

    // True color bitmap (16.7M colors)
    case 24:
      bmih.biCompression = BI_RGB;
      bmih.biSizeImage *= 3;              // 3 bytes needed for each pixel
      break;

    default:
      #ifdef _DEBUG_
      assert(0);
      #endif
  }

  bmih.biXPelsPerMeter = 0;
  bmih.biYPelsPerMeter = 0;
  bmih.biClrUsed = bmnumcolors;           // Number of color indexes in color
                                          // table actually used by the bitmap
  bmih.biClrImportant = bmnumcolors;      // All colors are important

  f = fopen(biFileName, "w+b");
  #ifdef _DEBUG_
  assert(f != NULL);
  #endif

  // Write BITMAPFILEHEADER and BITMAPINFOHEADER to file
  fwrite(&bmfh, sizeof(BITMAPFILEHEADER), 1, f);
  fwrite(&bmih, sizeof(BITMAPINFOHEADER), 1, f);

  // Color table
  if (bmnumcolors > 0) {
    bmcoltab = (RGBQUAD *)calloc(bmnumcolors, sizeof(RGBQUAD));
    #ifdef _DEBUG_
    assert(bmcoltab != NULL);
    #endif
    // Build color table from RGB
    for (k = 0; k < bmnumcolors; k++) {
      bmcoltab[k].rgbRed = coltab[k*3+0] << 2;
      bmcoltab[k].rgbGreen = coltab[k*3+1] << 2;
      bmcoltab[k].rgbBlue = coltab[k*3+2] << 2;
      bmcoltab[k].rgbReserved = 0;
    }
    fwrite(bmcoltab, bmnumcolors*sizeof(RGBQUAD), 1, f);
    free(bmcoltab);
  }

  BMPBufferFlip(biWidth, biHeight, biBuffer);
  switch (biCompression) {

    // BI_RGB
    case BI_RGB:

      switch (biBitCount) {
        // 16 colors
        //case 4:
        //  size >>= 1;
        //  for (k = 0; k < size; k++) {
        //    fread(&byte, sizeof(char), 1, f);
        //    buffer[(k << 1)] = ((byte>>4) & 0x0F);
        //    buffer[(k << 1)+1] = (byte & 0x0F);
        //  }

        // 256 colors
        case 8:
          k = 0;
          byte = 0;
          for (y = 0; y < biHeight; y++) {
            for (x = 0; x < biWidth; x++) {
              fwrite(&biBuffer[y*320+x], sizeof(char), 1, f);
              bmfh.bfSize++;
            }

            size = biWidth;
            while ((size % 4) != 0) {
              fwrite(&byte, sizeof(char), 1, f);
              bmfh.bfSize++;
              size++;
            }
          }
          break;

        default:
          #ifdef _DEBUG_
          assert(0);
          #endif
      }
      break;

    // BI_RLE8
    //case BI_RLE8:
    //  break;

    default:
      #ifdef _DEBUG_
      assert(0);
      #endif
  }

  fseek(f, 0, SEEK_SET);

  // Write BITMAPFILEHEADER and BITMAPINFOHEADER to file
  fwrite(&bmfh, sizeof(BITMAPFILEHEADER), 1, f);
  fwrite(&bmih, sizeof(BITMAPINFOHEADER), 1, f);

  fclose(f);

  return 0;
}




main(int argc, char **argv)
{
  long   x, y, width, height;
  char  *picture;
  char  *video = (char *)0xa0000;

  printf("\nBMP file viewer, Copyright (c) 1997 by Matjaz Trtnik");
  if (argc != 2) {
     printf("\nUSAGE : bmp.exe <file.bmp>");
     exit(1);
  }

  __asm {
    mov   eax,13h
    int   10h
  }

  picture = LoadBMP(argv[1], &width, &height);
  memset(video, 0, 64000);
  if (width > 320) width = 320;
  if (height > 200) height = 200;
  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++)
      video[y*320+x] = picture[y*bmih.biWidth+x];
  }
  getch();

  __asm {
    mov   eax,03h
    int   10h
  }
  return 1;
}
