;------------------------------------------------------------------------------
; some memory managers
;------------------------------------------------------------------------------

LoHeapOrg          DD    ?  ; Code32 based
LoHeapEnd          DD    ?  ; Code32 based
LoFreeList         DD    ?
HiHeapOrg          DD    ?  ; Code32 based
HiHeapEnd          DD    ?  ; Code32 based
HiFreeList         DD    ?
_errmsg HeapErr001,'HEAP001: null pointer assignment!'
_errmsg HeapErr002,'HEAP002: attempt to free not allocated block!'

;
;Lo_Init!
;  EXPLANATION  : initializes lo heap manager
;  INPUT        : none
;  OUTPUT       : none
;  SCREWED REGS : none
;

Lo_Init            PROC NEAR
                   PUSH EAX
                   PUSH EBX
                   MOV  EBX,LoHeapOrg
                   MOV  EAX,LoHeapEnd
                   MOV  [EBX.FMB_Next],EAX
                   SUB  EAX,EBX
                   MOV  [EBX.FMB_Size],EAX
                   MOV  LoFreeList,EBX
                   POP  EBX
                   POP  EAX
                   RET
Lo_Init            ENDP

;
;Lo_MaxAvail!
;  EXPLANATION  : returns size of largest block that can be allocated
;  INPUT        : none
;  OUTPUT       : eax - maxavail
;  SCREWED REGS : eax
;

Lo_MaxAvail        PROC NEAR
                   PUSH EBX

                   XOR  EAX,EAX
                   MOV  EBX,LoFreeList

@@LoopLoc:         CMP  EBX,LoHeapEnd
                   JAE  @@ToExit
                   CMP  EAX,[EBX.FMB_Size]
                   JAE  @@Skip1
                   MOV  EAX,[EBX.FMB_Size]
@@Skip1:           MOV  EBX,[EBX.FMB_Next]
                   JMP  @@LoopLoc

@@ToExit:          POP  EBX
                   RET
Lo_MaxAvail        ENDP

;
;Lo_FindBest!
;  EXPLANATION  : finds free block that fits best to EAX
;  INPUT        : EAX - size
;  OUTPUT       : EAX = 0 - no block found
;                  EAX > 0 - address of block
;                  if EAX > 0
;                     EBX = 0 - no free block before
;                     EBX > 0 - address of preceding free block
;  SCREWED REGS : EAX,EBX
;

Lo_FindBest        PROC NEAR
                   ; EAX - best block found
                   ; EBX - block before best
                   ; ECX - best block size
                   ; EDX - preferred block size
                   ; ESI - current block
                   ; EDI - block before current

                   PUSH ECX
                   PUSH EDX
                   PUSH ESI
                   PUSH EDI

                   MOV  EDX,EAX
                   XOR  EAX,EAX
                   XOR  EBX,EBX
                   MOV  ECX,0FFFFFFFFH
                   MOV  ESI,LoFreeList
                   XOR  EDI,EDI

@@LoopLoc:         CMP  ESI,LoHeapEnd
                   JAE  @@ToExit
                   CMP  EDX,[ESI.FMB_Size]
                   JA   @@Skip
                   CMP  ECX,[ESI.FMB_Size]
                   JBE  @@Skip
                   MOV  ECX,[ESI.FMB_Size]
                   MOV  EAX,ESI
                   MOV  EBX,EDI
@@Skip:            MOV  EDI,ESI
                   MOV  ESI,[ESI.FMB_Next]
                   JMP  @@LoopLoc

@@ToExit:          POP  EDI
                   POP  ESI
                   POP  EDX
                   POP  ECX
                   RET
Lo_FindBest        ENDP

;
;Lo_Alloc!
;  EXPLANATION  : allocates block of memory
;  INPUT        : EAX = size of block we want in bytes
;  OUTPUT       : EAX = 0 - no block allocated,
;                  EAX > 0 - address of allocated block
;  SCREWED REGS : EAX
;

Lo_Alloc           PROC NEAR
                   PUSH EBX
                   PUSH ECX
                   PUSH EDX
                   PUSH ESI

                   ADD  EAX,7
                   AND  AL,NOT 7
                   MOV  ECX,EAX
                   CALL Lo_FindBest
                   OR   EAX,EAX
                   JZ   @@ToExit
                   MOV  ESI,EAX
                   MOV  EDX,ECX
                   SUB  [ESI.FMB_Size],EDX
                   JZ   @@BlockFits
                   ADD  EAX,[ESI.FMB_Size]
                   JMP  @@ToExit
@@BlockFits:       OR   EBX,EBX
                   MOV  EDX,[ESI.FMB_Next]
                   JZ   @@NothingBefore
                   MOV  ESI,EBX
                   MOV  [ESI.FMB_Next],EDX
                   JMP  @@ToExit
@@NothingBefore:   MOV  LoFreeList,EDX

@@ToExit:          POP  ESI
                   POP  EDX
                   POP  ECX
                   POP  EBX
                   OR   EAX,EAX
                   RET
Lo_Alloc           ENDP

;
;Lo_Free!
;  EXPLANATION  : frees given block of memory, actually adds it to freelist
;  INPUT        : EAX - address of block
;                  ECX - size of block to free in bytes
;  OUTPUT       : none
;  SCREWED REGS : none
;

Lo_Free            PROC NEAR
                   PUSH EAX
                   PUSH EBX
                   PUSH EDX
                   PUSH ESI
                   PUSH EDI

                   ADD  ECX,7
                   AND  CL,NOT 7  ; align to 8 bytes up
                   MOV  ESI,EAX
; find free block before one we are freeing

                   MOV  EDX,EAX
                   XOR  EDI,EDI
                   MOV  EBX,LoFreeList

@@NextBlock:       CMP  EBX,EDX
                   JAE  @@FindDone
                   CMP  EBX,LoHeapEnd
                   JAE  @@FindDone
                   MOV  EDI,EBX
                   MOV  EBX,[EBX.FMB_Next]
                   JMP  @@NextBlock
@@FindDone:
                   OR   EDI,EDI
                   JZ   @@NothinkBefore
; lets check if we should glue
                   MOV  EAX,[EDI.FMB_Size]
                   ADD  EAX,EDI
                   CMP  EAX,ESI ; previous+size = current ?
                   JNZ  @@NoDnGlue
                   ADD  [EDI.FMB_Size],ECX ; just expand previous block
                   MOV  ECX,[EDI.FMB_Size] ; current block size changed slightly
                   MOV  ESI,EDI
                   JMP  @@UpGlue
; create here a block at ESI
@@NoDnGlue:        MOV  [ESI.FMB_Size],ECX
                   MOV  EAX,[EDI.FMB_Next]
                   MOV  [ESI.FMB_Next],EAX
                   MOV  [EDI.FMB_Next],ESI
                   JMP  @@UpGlue
                   
@@NothinkBefore:   MOV  EAX,LoFreeList
                   MOV  [ESI.FMB_Next],EAX
                   MOV  LoFreeList,ESI

@@UpGlue:          ; lets check if we can glue with next block
                   MOV  EAX,[ESI.FMB_Next]
                   CMP  EAX,LoHeapEnd
                   JAE  @@NoUpGlue
                   LEA  EBX,[ESI+ECX]
                   CMP  EBX,EAX
                   JNZ  @@NoUpGlue
; end of our block is adjacent to next block, do somethink about it
                   MOV  EBX,[EAX.FMB_Size]
                   MOV  EAX,[EAX.FMB_Next]
                   ADD  [ESI.FMB_Size],EBX
                   MOV  [ESI.FMB_Next],EAX
@@NoUpGlue:
@@ToExit:          POP  EDI
                   POP  ESI
                   POP  EDX
                   POP  EBX
                   POP  EAX
                   RET
Lo_Free            ENDP

;
;Hi_Init!
;  EXPLANATION  : initializes Hi heap manager
;  INPUT        : none
;  OUTPUT       : none
;  SCREWED REGS : none
;

Hi_Init            PROC NEAR
                   PUSH EAX
                   PUSH EBX
                   MOV  EBX,HiHeapOrg
                   MOV  EAX,HiHeapEnd
                   MOV  [EBX.FMB_Next],EAX
                   SUB  EAX,EBX
                   MOV  [EBX.FMB_Size],EAX
                   MOV  HiFreeList,EBX
                   POP  EBX
                   POP  EAX
                   RET
Hi_Init            ENDP

;
;Hi_MaxAvail!
;  EXPLANATION  : returns size of largest block that can be allocated
;  INPUT        : none
;  OUTPUT       : eax - maxavail
;  SCREWED REGS : eax
;

Hi_MaxAvail        PROC NEAR
                   PUSH EBX

                   XOR  EAX,EAX
                   MOV  EBX,HiFreeList

@@LoopLoc:         CMP  EBX,HiHeapEnd
                   JAE  @@ToExit
                   CMP  EAX,[EBX.FMB_Size]
                   JAE  @@Skip1
                   MOV  EAX,[EBX.FMB_Size]
@@Skip1:           MOV  EBX,[EBX.FMB_Next]
                   JMP  @@LoopLoc

@@ToExit:          POP  EBX
                   RET
Hi_MaxAvail        ENDP

;
;Hi_FindBest!
;  EXPLANATION  : finds free block that fits best to EAX
;  INPUT        : EAX - size
;  OUTPUT       : EAX = 0 - no block found
;                  EAX > 0 - address of free block
;                  if EAX > 0
;                     EBX = 0 - no free block before
;                     EBX > 0 - address of preceding free block
;  SCREWED REGS : EAX,EBX
;

Hi_FindBest        PROC NEAR
                   ; EAX - best block found
                   ; EBX - block before best
                   ; ECX - best block size
                   ; EDX - preferred block size
                   ; ESI - current block
                   ; EDI - block before current

                   PUSH ECX
                   PUSH EDX
                   PUSH ESI
                   PUSH EDI

                   MOV  EDX,EAX
                   XOR  EAX,EAX
                   XOR  EBX,EBX
                   MOV  ECX,0FFFFFFFFH
                   MOV  ESI,HiFreeList
                   XOR  EDI,EDI

@@LoopLoc:         CMP  ESI,HiHeapEnd
                   JAE  @@ToExit
                   CMP  EDX,[ESI.FMB_Size]
                   JA   @@Skip
                   CMP  ECX,[ESI.FMB_Size]
                   JBE  @@Skip
                   MOV  ECX,[ESI.FMB_Size]
                   MOV  EAX,ESI
                   MOV  EBX,EDI
@@Skip:            MOV  EDI,ESI
                   MOV  ESI,[ESI.FMB_Next]
                   JMP  @@LoopLoc

@@ToExit:          POP  EDI
                   POP  ESI
                   POP  EDX
                   POP  ECX
                   RET
Hi_FindBest        ENDP

;
;Hi_Alloc!
;  EXPLANATION  : allocates block of memory
;  INPUT        : EAX = size of block we want in bytes
;  OUTPUT       : EAX = 0 - no block allocated,
;                  EAX > 0 - address of allocated block
;  SCREWED REGS : EAX
;

Hi_Alloc           PROC NEAR
                   PUSH EBX
                   PUSH ECX
                   PUSH EDX
                   PUSH ESI

                   ADD  EAX,7
                   AND  AL,NOT 7
                   MOV  ECX,EAX
                   CALL Hi_FindBest
                   OR   EAX,EAX
                   JZ   @@ToExit
                   MOV  ESI,EAX
                   MOV  EDX,ECX
                   SUB  [ESI.FMB_Size],EDX
                   JZ   @@BlockFits
                   ADD  EAX,[ESI.FMB_Size]
                   JMP  @@ToExit
@@BlockFits:       OR   EBX,EBX
                   MOV  EDX,[ESI.FMB_Next]
                   JZ   @@NothingBefore
                   MOV  ESI,EBX
                   MOV  [ESI.FMB_Next],EDX
                   JMP  @@ToExit
@@NothingBefore:   MOV  HiFreeList,EDX

@@ToExit:          POP  ESI
                   POP  EDX
                   POP  ECX
                   POP  EBX
                   RET
Hi_Alloc           ENDP

;
;Hi_Free!
;  EXPLANATION  : frees given block of memory, actually adds it to freelist
;  INPUT        : EAX - address of block
;                  ECX - size of block to free in bytes
;  OUTPUT       : none
;  SCREWED REGS : none
;

Hi_Free            PROC NEAR
                   PUSH EAX
                   PUSH EBX
                   PUSH EDX
                   PUSH ESI
                   PUSH EDI

                   ADD  ECX,7
                   AND  CL,NOT 7  ; align to 8 bytes up
                   MOV  ESI,EAX
; find free block before one we are freeing

                   MOV  EDX,EAX
                   XOR  EDI,EDI
                   MOV  EBX,HiFreeList

@@NextBlock:       CMP  EBX,EDX
                   JAE  @@FindDone
                   CMP  EBX,HiHeapEnd
                   JAE  @@FindDone
                   MOV  EDI,EBX
                   MOV  EBX,[EBX.FMB_Next]
                   JMP  @@NextBlock
@@FindDone:
                   OR   EDI,EDI
                   JZ   @@NothingBefore
; lets check if we should glue
                   MOV  EAX,[EDI.FMB_Size]
                   ADD  EAX,EDI
                   CMP  EAX,ESI ; previous+size = current ?
                   JNZ  @@NoDnGlue
                   ADD  [EDI.FMB_Size],ECX ; just expand previous block
                   MOV  ECX,[EDI.FMB_Size] ; current block size changed slightly
                   MOV  ESI,EDI
                   JMP  @@UpGlue
; create here a block at ESI
@@NoDnGlue:        MOV  [ESI.FMB_Size],ECX
                   MOV  EAX,[EDI.FMB_Next]
                   MOV  [ESI.FMB_Next],EAX
                   MOV  [EDI.FMB_Next],ESI
                   JMP  @@UpGlue
                   
@@NothingBefore:   MOV  EAX,HiFreeList
                   MOV  [ESI.FMB_Next],EAX
                   MOV  HiFreeList,ESI

@@UpGlue:          ; lets check if we can glue with next block
                   MOV  EAX,[ESI.FMB_Next]
                   CMP  EAX,HiHeapEnd
                   JAE  @@NoUpGlue
                   LEA  EBX,[ESI+ECX]
                   CMP  EBX,EAX
                   JNZ  @@NoUpGlue
; end of our block is adjacent to next block, do something about it
                   MOV  EBX,[EAX.FMB_Size]
                   MOV  EAX,[EAX.FMB_Next]
                   ADD  [ESI.FMB_Size],EBX
                   MOV  [ESI.FMB_Next],EAX
@@NoUpGlue:
@@ToExit:          POP  EDI
                   POP  ESI
                   POP  EDX
                   POP  EBX
                   POP  EAX
                   RET
Hi_Free            ENDP

Comment #
!malloc
  Explanation  : allocates block of memory
  Expects      : EAX - amount of memory we want (it will be rounded up
                        to be multiple of 4)
  Returns      : EAX == 0 and ZF=1, allocation failed
                  EAX != 0 and ZF=0, allocation ok, EAX is pointer
                  to block of memory
  Screwed regs : none
#
malloc             PROC NEAR
                   CMP  D NullPointer,6C6C754EH
                   JNZ  HeapErr001
                   PUSH EBX
                   ADD  EAX,8
                   MOV  EBX,EAX
                   CALL Hi_Alloc
                   OR   EAX,EAX
                   JZ   @@ProcExit
                   MOV  DWORD PTR [EAX],EBX
                   NOT  EBX
                   MOV  DWORD PTR [EAX+4],EBX
                   ADD  EAX,8
@@ProcExit:        POP  EBX
                   OR   EAX,EAX
                   RET
malloc             ENDP

Comment #
!free
  Explanation  : frees memory block allocated by malloc
  Expects      : EAX - pointer to block
  Returns      : none
  Screwed regs : none
#
free               PROC NEAR
                   CMP  D NullPointer,6C6C754EH
                   JNZ  HeapErr001
                   PUSHAD
                   SUB  EAX,8
                   MOV  ECX,DWORD PTR [EAX+4]
                   NOT  ECX
                   CMP  ECX,DWORD PTR [EAX]
                   JNZ  HeapErr002
                   CALL Hi_Free
@@ProcExit:        POPAD
                   RET
free               ENDP

