/* Copyright (C) 1995 DJ Delorie, see COPYING.DJ for details */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <string.h>
#include <io.h>
#include <unistd.h>
#include <dos32.h>
#include <dir.h>

#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifdef __DOS32__
#include <crt0.h>
int _crt0_startup_flags=_CRT0_FLAG_GRAP_ALL_MEM;
#endif

unsigned char stub_bytes[] = {
#include "stub.h"
};

unsigned char small_dos32_stub[] = {
#include "verysmll.h"
};

/* this is the lzss compression (c) by Haruhiko Okumura */
#include "lzsscomp.c"

char *stubname=NULL;

#define v_printf if(verbose)printf
int verbose=0;

static unsigned long
get32(unsigned char *ptr)
{
  return ptr[0] | (ptr[1]<<8) | (ptr[2]<<16) | (ptr[3]<<24);
}

static unsigned short
get16(unsigned char *ptr)
{
  return ptr[0] | (ptr[1]<<8);
}

static void coff2exe(char *fname)
{
  char ifilename[256];
  char ofilename[256];
  int ifile;
  int ofile;
  char *ofname, *ofext;
  char buf[4096];
  int rbytes, used_temp, i, n;
  long coffset=0;
  unsigned char filehdr_buf[20];
  int max_file_size = 0;
  int coff_file_size = 0;
  int drop_last_four_bytes = 0;

  strcpy(ifilename, fname);
  strcpy(ofilename, fname);
  ofext = 0;
  for (ofname=ofilename; *ofname; ofname++)
  {
    if (strchr("/\\:", *ofname))
      ofext = 0;
    if (*ofname == '.')
      ofext = ofname;
  }
  if (ofext == 0)
    ofext = ofilename + strlen(ofilename);
  strcpy(ofext, ".exe");
  if (access(ofilename, 0) == 0)
    for (ofile=0; ofile<0xfff; ofile++)
    {
      used_temp = 1;
      sprintf(ofext, ".%03x", ofile);
      if (access(ofilename, 0))
	break;
    }
  else
    used_temp = 0;

  ifile = open(ifilename, O_RDONLY|O_BINARY);
  if (ifile < 0)
  {
    perror(fname);
    return;
  }
  while (1)
  {
    lseek(ifile, coffset, 0);
    read(ifile, buf, 6);
    if (buf[0] == 'M' && buf[1] == 'Z') /* stubbed already, skip stub */
    {
      int blocks = (unsigned char)buf[4] + (unsigned char)buf[5] * 256;
      int partial = (unsigned char)buf[2] + (unsigned char)buf[3] * 256;
      coffset += blocks * 512;
      if (partial)
	coffset += partial - 512;
    }
    else if (buf[0] == 0x4c && buf[1] == 0x01) /* it's a COFF */
    {
      break;
    }
    else
    {
      fprintf(stderr, "Warning: input file is not COFF or stubbed COFF\n");
      break;
    }
  }

  coff_file_size = lseek(ifile, 0, SEEK_END) - coffset;
  lseek(ifile, coffset, 0);

  read(ifile, filehdr_buf, 20); /* get the COFF header */
  lseek(ifile, get16(filehdr_buf+16), SEEK_CUR); /* skip optional header */
  n = get16(filehdr_buf + 2);
  for (i=0; i<n; i++) /* for each COFF section */
  {
    int addr, size, flags;
    unsigned char coffsec_buf[40];
    read(ifile, coffsec_buf, 40);
    size = get32(coffsec_buf+16);
    addr = get32(coffsec_buf+20);
    flags = get32(coffsec_buf+36);
    if (flags & 0x60) /* text or data */
    {
      int maybe_file_size = size + addr;
      if (max_file_size < maybe_file_size)
	max_file_size = maybe_file_size;
    }
  }
  if (coff_file_size == max_file_size + 4)
    /* it was built with "gcc -s" and has four too many bytes */
    drop_last_four_bytes = 1;

  lseek(ifile, coffset, 0);

  ofile = open(ofilename, O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, 0666);
  if (ofile < 0)
  {
    perror(ofname);
    return;
  }
  v_printf("stubify: %s -> %s", ifilename, ofilename);
  if (used_temp)
  {
    strcpy(ifilename, ofilename);
    strcpy(ofext, ".exe");
    v_printf(" -> %s", ofilename);
  }
  v_printf("\n");

  write(ofile, stub_bytes, sizeof(stub_bytes));
  
  while ((rbytes=read(ifile, buf, 4096)) > 0)
  {
    int wb;

    if (drop_last_four_bytes && rbytes < 4096)
      rbytes -= 4;

    wb = write(ofile, buf, rbytes);
    if (wb < 0)
    {
      perror(ofname);
      unlink(ofilename);
      close(ifile);
      close(ofile);
      exit(1);
    }
    if (wb < rbytes)
    {
      fprintf(stderr, "%s: disk full\n", ofname);
      unlink(ofilename);
      close(ifile);
      close(ofile);
      exit(1);
    }
  }

  close(ifile);
  close(ofile);

  if (used_temp)
  {
    unlink(ofilename);
    if (rename(ifilename, ofilename))
    {
      printf("rename of %s to %s failed.\n", ifilename, ofilename);
      perror("The error was");
    }
  }
}

static void generate_go32 (char *name)
{
  char ofilename[256], *ofname, *ofext=0;
  int ofile;

  strcpy(ofilename, name);
  for (ofname=ofilename; *ofname; ofname++)
  {
    if (strchr("/\\:", *ofname))
    ofext = 0;
    if (*ofname == '.')
    ofext = ofname;
  }
  if (ofext == 0)  ofext = ofilename + strlen(ofilename);
  strcpy(ofext, ".exe");
    
  ofile = open(ofilename, O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, 0666);
  if (ofile < 0)
  {
    printf("Cannot open output file to generate\n");
    perror(ofilename);
    exit(1);
  }
  v_printf("stubify: generate %s\n", name);

  write(ofile, stub_bytes, sizeof(stub_bytes));
  close(ofile);
}
/*************** DOS32 stuff *****************************************/

static int test_dos32 (char *name)
{
    dos32_header buffer;
    int dummy = open(name, O_RDONLY|O_BINARY);
    if (dummy>-1) {
       if (read(dummy,&buffer,sizeof(buffer))==sizeof(buffer))
       {
          if (buffer.dos32_id==dos32_header_id) return 1;
       }
       close(dummy);
     }
    return 0;
}

#define complain(s) fprintf(stderr,s); exit(1)

#define errorout(w,s) perror(s); complain(w)

static void dos32_to_exe (char *name,char compress, char logo)
{
    FILE  *ofile,*nfile;
    int   astub;
    unsigned long foffset;
    char newname[256];
    char buffer[512];
    dos32_header new_header;
    int  i,used_temp;
    unsigned int realread;

    strcpy((char *)&newname,name);
    for(i=strlen(newname)-1;i && newname[i]!='.';i--);
    if (!i) i=strlen(newname);
    strcpy(&newname[i],".exe");
    if (access(newname, 0) == 0)
    for (astub=0; astub<0xfff; astub++)
    {
      used_temp = 1;
      sprintf(&newname[i], ".%03x", astub);
      if (access(newname, 0)) break;
    }
    else used_temp = 0;

    ofile=fopen(name,"rb");
    if (ofile==  0) { errorout("Can't open DOS32 file\n",name); }
    nfile=fopen(newname,"wb");
    if (nfile== 0) { errorout("Can't create exe file\n",newname); }
    v_printf("stubify: %s -> %s\n",name,newname);
    if (stubname==NULL) {
       v_printf("...using small stub...\n");
       if (fwrite(&small_dos32_stub,1,sizeof(small_dos32_stub),nfile)!=
         sizeof(small_dos32_stub)) {complain("Can't write file. Disk full ?\n");}
    }
    else {
      if (*stubname==0)  stubname=searchpath("dos32.exe");
      if (stubname==NULL) {complain("dos32.exe not found\n");}
      strcpy(buffer,stubname);
      v_printf("...using %s as stub...\n",buffer);
      astub=open(buffer,O_RDONLY | O_BINARY);
      if (astub < 0) { errorout("Stub not found\n",buffer); }

      while ((realread=read(astub,&buffer,512))>0) {
        if (fwrite(&buffer,1,realread,nfile)<realread)
         { complain("Can't write file. Disk full ?\n"); }
      }

      close(astub);
    }
    
    if (fread(&new_header,1,sizeof(dos32_header),ofile)!=sizeof(dos32_header))
          { errorout("Error reading DOS32 header !\n",name); }
    if (logo)      new_header.exe_flags |= dos32_show_copyright;
    if (compress)  new_header.exe_flags |= dos32_is_compressed;
    foffset = ftell(nfile);
    if (fseek(nfile,sizeof(dos32_header),SEEK_CUR))
          { errorout("Error seeking in new file !\n",newname); }

    if (compress)    /* compress executable image if required */
    {
        long out = 0;
        v_printf("...compressing...");
        Encode(ofile,nfile,&out,new_header.image_length);
        v_printf("ratio: %ld%%\n",100-(100*out/new_header.image_length));
        if (out>new_header.image_length) printf("WARNING: size increase due to\
 compression !\n");
        new_header.exe_length -= new_header.image_length - out;
        new_header.image_length = out;
    }

   /* copy executable image or rest of file (if compressed DOS32 exe) */
   while ((realread=fread(&buffer,1,512,ofile))>0) {
        if (fwrite(&buffer,1,realread,nfile)<realread)
         { complain("Can't write file. Disk full ?\n"); }
   }

   if (fseek(nfile,foffset,SEEK_SET))
          { errorout("Error seeking in new file !\n",newname); }
   if (fwrite(&new_header,1,sizeof(dos32_header),nfile)!=sizeof(dos32_header))
          { errorout("Error writing DOS32 header !\n",newname); }

    fflush(nfile);
    fflush(ofile);
    fclose(nfile);
    fclose(ofile);

    if (used_temp) {
      strcpy(&buffer[0],(char *)&newname);
      for(i=strlen(buffer)-1;i && buffer[i]!='.';i--);
      if (!i) i=strlen(buffer);
      strcpy(&buffer[i],".exe");
      v_printf(" -> %s\n", buffer);
      if (unlink(buffer)) { errorout("Deleting of old exe failed !\n",buffer);}
      if (rename(newname, buffer))
      { errorout("Renaming of tempfile failed !\n",newname);   }
    }
}

int main(int argc, char **argv)
{
  char generate = 0, compress = 0, logo = 0;
  if (argc > 1 && strcmp(argv[1], "-v")==0)
  {
    verbose = 1;
    argv++;
    argc--;
  }
  v_printf("stubify for djgpp V2.X executables, Copyright (C) 1995 DJ Delorie\n");
  v_printf("modified for DOS32 support by r\n");
  if (argc < 2)
  {
    fprintf(stderr, "\
Usage: stubify [-g] [-sNAME] [-c] [-l] <program>  (program may be COFF\n\
stubbed .exe (GO32), COFF with .exe extension or DOS32 program. Resulting\n\
file will have an .exe extention)\n");

    fprintf(stderr, "\nThis program is NOT shareware or public domain.  It is copyrighted.\n");
    fprintf(stderr, "It is redistributable but only as part of a complete package.  If you\n");
    fprintf(stderr, "have a copy of this program, the place that you got it from is\n");
    fprintf(stderr, "responsible for making sure you are able to get it's sources as well.\n");

    exit(1);
  }
  while (argc > 1) {
    if (strcmp(argv[1], "-g") == 0) generate=1;
    else if (strstr(argv[1],"-s")==argv[1]) stubname=argv[1]+2;
    else if (strcmp("-c",argv[1])==0) compress=1;
    else if (strcmp("-l",argv[1])==0) logo=1;
    else if (generate==0) {
      if (test_dos32(argv[1])==0) coff2exe(argv[1]);
      else dos32_to_exe(argv[1],compress,logo);
    }
    else generate_go32(argv[1]);
    argc--;
    argv++;
  }
  return 0;
}

