// Routines for playing the mod, s3m and xm format
//
// $Id: player.cc,v 0.1 1998/06/08 01:18:08 pcburns Exp pcburns $
// $Log: player.cc,v $
// Revision 0.1  1998/06/08 01:18:08  pcburns
// cleaned up code for initial release
//

#include <stdlib.h>    // malloc
#ifdef PRINT_PATTERNS
#include <stdio.h>
#endif
#include <string.h>    // memset
#include <sys/types.h>
#include <sys/wait.h>
#include <math.h>
#ifdef USE_FORK
#include <signal.h>    // kill
#include <sys/stat.h>
#include <unistd.h>
#endif

#include "player.h"

// bits used for fractional part of a fixed point number
// FBITS is only used in mix() to calcluate the position
#define FBITS 12

////////////////////////////////////////////////////////////////////////////
//  constuctors and destructors
////////////////////////////////////////////////////////////////////////////

player::player()
{
  if (DSP::error)
    {
      callback_error("Error creating DSP");
      error   = 1;
      mix_buf = 0;
      chan    = 0;
    }
  else
    {
      child         = 0;
      forked_player = 0;
      error         = 0;
      tick          = 0;
      row           = 0;
      mod           = 0;
      order         = 0;
      pattern       = 0;
      pattern_break = 0;
      pattern_jump  = 0;
      NumChan       = 0;
      pattern_delay = 0;
      chan          = 0;
      mix_buf       = 0;
      set_speed(6);
      set_bpm(125);
  
      create_volume_table();
    }
}

player::~player()
{
  free(mix_buf);
  free(chan);
}

#ifdef USE_FORK
// handler to tell child to _exit
void sig_handler(int sig)
{
  switch (sig)
    {
    case SIGINT:
      break;
    case SIGUSR1: 
      _exit(0);
      break;
    case SIGUSR2:
      break;
    }
}
#endif
////////////////////////////////////////////////////////////////////////////
// tick functions
////////////////////////////////////////////////////////////////////////////
// this player doesn't use a timer & handler to calculate tick lengths
// although this could be done using setitimer and sigaction
// instead a ticks worth of buffer is played
void player::play_tick(void)
{
  if (tick == 0)
    update_row();
  else
    update_effects();

  // mix in each active channel
  mix_channels();
  // now play the buffer
  // 32 bit mixing buffer played in 8bit
  DSP::play32_8(mix_buf,mix_length);
  // increment tick
  tick++;
}

// for effects that need to be updated on tick 0
void player::update_row(void)
{
  // reset flags
  pattern_break = pattern_jump = 0;

  // need to process channels in order to satisfy pattern break and jump
  for (ubyte ch = 0; ch < mod->channels; ch++)
    {
      Note n = mod->PatternData[pattern][ch][row];
      const byte x = n.effParam >> 4;
      byte y = n.effParam & 0xf;
      // reset vibrato delta 
      chan[ch].vib_delta = 0;

      // set the new instrument
      if (n.number)
	{
	  if (n.effect != NOTE_DELAY) 
	    {
	      // check to make sure its less than mod.NumSamples ?
	      chan[ch].Number = n.number;
	      if (n.InstMode)
		{
		  chan[ch].InstMode = 1;
		  if (n.number-1 < mod->NumInst)
		    {
		      chan[ch].In = mod->in[n.number-1];
		      chan[ch].reset_parameters(&chan[ch].EV_Status);
		      //chan[ch].reset_parameters(&chan[ch].PV_Status);
		    }
		  else
		    chan[ch].In = 0;
		  callback_ch_inst(ch);
		}
	      else
		{
		  chan[ch].InstMode = 0;
		  if (n.number <= mod->NumSamples)
		    chan[ch].smp = mod->samples[n.number-1];
		  else
		    chan[ch].smp = 0;
		  callback_ch_smp(ch);
		}
	      // reset position to start of sample data
	      chan[ch].reset_channel();
	      // reset volume
	      if (chan[ch].smp)
		  chan[ch].volume = chan[ch].smp->volume;
	      else if (!chan[ch].In)
		chan[ch].volume = 0;
	      callback_ch_volume(ch);
	    }
	}

      // set the new note
      if (n.NoteNum)
	{
	  if (n.NoteNum == NOTE_OFF)
	    {
	      if (!chan[ch].InstMode) 
		{
		  chan[ch].NoteNum = 0;
		  chan[ch].period  = 0;
		  callback_ch_note(ch);
		}
	      else
		{
		  if (chan[ch].In->volume.type & 0x01)
		    {
		      chan[ch].key_off = true;
		      chan[ch].fadeout = chan[ch].In->volume.fadeout;
		    }
		  else
		    {
		      chan[ch].NoteNum = 0;
		      chan[ch].period  = 0;
		      callback_ch_note(ch);
		    }
		}
	    }
	  else
	    {
	      // if not portamento or delay note
	      if ((n.effect != PORT_TONE)      && 
		  (n.effect != PORT_VOL_SLIDE) && 
		  (n.effect != NOTE_DELAY)     && 
		  (n.volume < (VC_PORT_TONE<<4)))
		{
		  chan[ch].NoteNum = n.NoteNum;
		  chan[ch].period  = note_to_period(ch,n.NoteNum-1);
		  callback_ch_note(ch);
		  if (n.number && n.InstMode && chan[ch].In)
		    {
		      byte Num = chan[ch].In->key_samp[n.NoteNum-1];
		      int NumSamp = chan[ch].In->NumSamples;
		      if ((NumSamp > 0) && (Num < NumSamp))
			{
			  chan[ch].volume = chan[ch].In->samples[Num]->volume;
			  callback_ch_volume(ch);
			}
		    }		  
		}
	    }
	}
      
      // set the volume if there is one
      if (n.volume)
	{
	  if (n.volume < 0x42)
	    {
	      chan[ch].volume = n.volume - 1;
	      callback_ch_volume(ch);
	    }
	  else
	    {
	      // process volume column effects
	      const byte x = n.volume >> 4;
	      byte y = n.volume & 0xf;
	      switch (x)
		{
		case VC_VOL_SLIDE_DN:
		  chan[ch].vol_slide_fine = 0;
		  chan[ch].vol_slide = y;
		  break;
		case VC_VOL_SLIDE_UP:
		  chan[ch].vol_slide_fine = 0;
		  chan[ch].vol_slide = (y<<4);
		  break;
		case VC_FINE_VOL_SLIDE_DN:
		  if (!y)
		    y = chan[ch].vol_slide_fine & 0xf;
		  vol_dn(ch,y);
		  chan[ch].vol_slide_fine = 0xf0 | y;
		  chan[ch].vol_slide = 0;
		  break;
		case VC_FINE_VOL_SLIDE_UP:
		  if (!y)
		    y = chan[ch].vol_slide_fine >> 4;	    
		  vol_up(ch,y);
		  chan[ch].vol_slide_fine = 0x0f | (y<<4);
		  chan[ch].vol_slide = 0;
		  break;
		case VC_SET_VIBRATO_SPEED:
		  chan[ch].vib_spd = y;
		  break;
		case VC_VIBRATO:
		  // check if retrig vibrato
		  if (chan[ch].vib_wave <= 3)
		    chan[ch].vib_pos = 0;
		  if (x)
		    chan[ch].vib_spd = x;
		  if (y)
		    chan[ch].vib_dpth = y;	      
		  break;
		case VC_SET_PANNING:
		  break;
		case VC_PAN_SLIDE_LEFT:
		  break;
		case VC_PAN_SLIDE_RIGHT:
		  break;
		case VC_PORT_TONE:
		  if (y)
		    chan[ch].port_speed  = y << (4+mod->period_shift);
		  if (n.NoteNum)
		    chan[ch].port_period = note_to_period(ch,n.NoteNum-1); 
		  break;
		  //		default:
		  //		  break;
		}
	    }
	}
      else if (chan[ch].tremor_vol)
	{
	  chan[ch].volume = chan[ch].tremor_vol;
	  callback_ch_volume(ch);
	  chan[ch].tremor_vol = 0;
	}
      // if volume slide & effect parameter = 0
      // then check if last volume slide was a fine volume slide
      if ((n.effect == VOLUME_SLIDE) && (n.effParam == 0) 
	  && chan[ch].vol_slide_fine)
	{
	  const byte a = chan[ch].vol_slide_fine>>4;
	  const byte b = chan[ch].vol_slide_fine & 0xf;
	  if (a && b)
	    {
	      if (a == 0xf)
		{
		  n.effect = FINE_VOL_SLIDE_DN;
		  n.effParam = b;
		}
	      else
		{
		  n.effect = FINE_VOL_SLIDE_UP;
		  n.effParam = a;
		}
	    }
	}

      // process effect      
      switch (n.effect)
	{ 
	case PORT_TONE: // tone portamento -- tick based 3xx
	  // tick 0 set porta speed
	  // on other ticks -- add or subtract speed to frequency 
	  // until note is reached 
	  if (n.effParam)
	    chan[ch].port_speed  = n.effParam << mod->period_shift;
	  if (n.NoteNum)
	    chan[ch].port_period = note_to_period(ch,n.NoteNum-1); 
	  break;
	case VIBRATO: // vibrato
	case FINE_VIBRATO:
	  // check if retrig vibrato
	  if (chan[ch].vib_wave <= 3)
	    chan[ch].vib_pos = 0;
	  if (x)
	    chan[ch].vib_spd = x;
	  if (y)
	    chan[ch].vib_dpth = y;
	  break;
	case TREMOLO: // 7xx
	  // similar to vibrato except it affects volume rather than pitch
	  if (chan[ch].trem_wave <= 3)
	    chan[ch].trem_pos = 0;
	  if (x)
	    chan[ch].trem_spd = x;
	  if (y)
	    chan[ch].trem_dpth = y;
	  break;
	case PAN: // 8xx not used - panning	  
	  break;
	case SAMPLE_OFFSET: // 9xx set sample offset -- NOT TICK BASED 
	  {
	    if (!n.effParam) // remember last offset 
	      { // not implemented yet
		chan[ch].reset_channel();
		break; 
	      }
	    // sample_offset = EffParam * 0x100
	    unsigned long offset = n.effParam * 0x100;
	    if (chan[ch].InstMode)
	      {
		Sample *samp = 0;
		if (chan[ch].In) // check to make sure it exists
		  {
		    byte Num = chan[ch].In->key_samp[chan[ch].NoteNum-1]; 
		    int NumSamp = chan[ch].In->NumSamples;
		    if ((NumSamp > 0) && (Num < NumSamp))
		      {
			samp=chan[ch].In->samples[Num];
			if (offset < samp->length)
			  chan[ch].position = (offset-1)<<FBITS;
		      }
		  }
	      }
	    else
	      {
		if (chan[ch].smp) // check to make sure it exists
		  if (offset < chan[ch].smp->length)
		    chan[ch].position = (offset-1)<<FBITS;
	      }
	    break;
	  }
	case VOLUME_SLIDE: // volume slide -- tick based Axy
	  // worry about the bug in st3 where vol_slide is performed on tick 0?
	case PORT_VOL_SLIDE: // tone portamento + volume slide 5xx
	case VIBRATO_VOL_SLIDE: // 6xx
	  chan[ch].vol_slide_fine = 0;
	  chan[ch].vol_slide = n.effParam;
	  break;
	case PATTERN_JUMP: // pattern jump -- Bxx  
	  // when row is finished jump to pattern xx row 0
	  // nullifies pattern breaks
	  // just set a flag saying that a pattern jump has been found
	  if (!pattern_jump)
	    {
	      pattern_jump = 1;
	      next_row = 0;	      	      
	      next_order = n.effParam-1; //check this pattern exists
	    }
	  break;
	case VOLUME_SET: // set volume -- Cxx
	  if (n.effParam > 65)
	    chan[ch].volume = 64;
	  else
	    chan[ch].volume = n.effParam;
	  callback_ch_volume(ch);
	  break;
	case PATTERN_BREAK: // pattern break -- dxy
	  // only do this once per row and don't do it if there is a Bxx
	  // jump to next pattern row xx
	  // if xy > patternlength-1 xx = 0 
	  next_row = x*10 + y;
	  if (next_row > mod->rows[pattern]-1)
	    next_row = 0;
	  pattern_break = 1;
	  break;
	case FINE_PORT_UP: // E1y fine portamento up
	  // only do this on tick 0
	  if ((chan[ch].period -= (y<<2)) < period_least)
	    chan[ch].period = period_least;
	  break;
	case FINE_PORT_DN: // E2y fine portamento down
	  // only do this on tick 0
	  if ((chan[ch].period += (y<<2)) > period_most)
	    chan[ch].period = period_most;
	  break;
	case GLISSANDO: // E3y glissando control
	  chan[ch].glissando = (y & 0x1);
	  break;
	case VIBRATO_WAVEFORM: // E4y set vibrato waveform
	  chan[ch].vib_wave = (y & 0x7);
	  break;
	case FINETUNE_SET: // E5y set finetune
	  // tick 0 only
	  // set the Instruments fine tune to EffParam
	  if (chan[ch].smp)
	    chan[ch].smp->frequency = finetune_to_C2SPD(y);
	  break;
	case PATTERN_LOOP: // E6y pattern loop
	  if (!y) 
	    chan[ch].ploop_start = row;
	  else
	    {
	      pattern_loop = 1;
	      if (!chan[ch].ploop_count)
		chan[ch].ploop_count = y;
	      else
		chan[ch].ploop_count--;
	      // this is not right should loop after we finish 
	      if (chan[ch].ploop_count > 0)
		next_row = chan[ch].ploop_start; 
	      if (!chan[ch].ploop_count)
		pattern_loop = 0;
	    }
	  break;
	case TREMOLO_WAVEFORM: // E7y set tremelo waveform
	  chan[ch].trem_wave = (y & 0x7);
	  break;
	case FINE_VOL_SLIDE_UP: // EAy fine volume slide up 
	  // make sure volume is less than 65
	  if (!y)
	    y = chan[ch].vol_slide_fine >> 4;	    
	  vol_up(ch,y);
	  chan[ch].vol_slide_fine = 0x0f | (y<<4);
	  chan[ch].vol_slide = 0;
	  break;
	case FINE_VOL_SLIDE_DN: // EBy fine volume slide down
	  if (!y)
	    y = chan[ch].vol_slide_fine & 0xf;
	  vol_dn(ch,y);
	  chan[ch].vol_slide_fine = 0xf0 | y;
	  chan[ch].vol_slide = 0;
	  break;
	case PATTERN_DELAY: // EEy pattern delay
	  // tick 0 only -- needs to be incorporated in the tick handler
	  if (!pattern_delay)
	    if (y > speed)
	      pattern_delay = y - speed;
	  break;
	case FUNK_IT: // EFy invert loop  (play sample backwards)
	  // not supported in any player or tracker
	  break;
	case SPEED_SET: // set speed -- Fxy
	  //  EffParam 0x00 - 0x1f set speed
	  //  EffParam 0x20 - 0xff set BPM
	  if (!n.effParam) 
	    {
	      callback_end();
	    }
	  else if (n.effParam < 0x20) // set speed
	      set_speed(n.effParam);
	  else
	      set_bpm(n.effParam);
	  break;
	case XTR_FINE_PORT_DN:
	  if ((chan[ch].period += y) > period_most)
	    chan[ch].period = period_most;	  
	  break;
	case XTR_FINE_PORT_UP:
	  if ((chan[ch].period -= y) < period_least)
	    chan[ch].period = period_least;
	  break;
	case TREMOR:
	  if (n.effParam)
	    chan[ch].tremor  = n.effParam;
	  chan[ch].tremor_vol = chan[ch].volume;
	  chan[ch].tremor_count = 0;
	  do_tremor(ch);
	  break;
	case GLOBAL_VOLUME:
	  if (n.effParam < 65)
	    set_gv(n.effParam);
	  break;
	case STEREO_CONTROL:
	  break;
	case RETRIG_VOL_SLIDE:
	  if (n.effParam)
	    chan[ch].retrig = n.effParam;
	  break;
	default:
	  break;
	}      
      // process panning envelopes
      // process volume envelopes
      if (chan[ch].InstMode)
	{
	  chan[ch].EV = chan[ch].process_parameters(&chan[ch].In->volume,
						    &chan[ch].EV_Status);
	  //chan[ch].EP = chan[ch].process_parameters(&chan[ch].In->Panning, 
	  //                                          &chan[ch].PV_Status);  
	}
    }
}

// update tick based effects 
void player::update_effects(void)
{
  for (ubyte ch = 0; ch < mod->channels; ch++)
    {
      // process effect      
      const Note& n = mod->PatternData[pattern][ch][row];
      // process volume effects
      if (n.volume > 0x50)
	{
	  const byte x = n.volume >> 4;
	  switch (x)
	    {
	    case VC_VOL_SLIDE_DN:
	    case VC_VOL_SLIDE_UP:
	      do_volume_slide(ch);
	      break;
	    case VC_VIBRATO:
	      do_vibrato(ch,0);
	      break;
	    case VC_PORT_TONE:
	      if (chan[ch].port_period != chan[ch].period)
		do_portamento(ch);
	      break;
	    default:
	      break;
	    }
	}
      
      byte x = n.effParam >> 4;
      byte y = n.effParam & 0xf;
      switch (n.effect)
	{
	case ARPEGGIO: // arpeggio -- tick based 0xy
	  // tick 0 - normal note
	  // tick 1 - add x to NoteNum
	  // tick 2 - add y to NoteNum
	  if (n.effParam)
	    {
	      switch (tick % 3)
		{
		case 0:
		  chan[ch].period = note_to_period(ch,chan[ch].NoteNum-1);
		  break;
		case 1:
		  chan[ch].period = note_to_period(ch,chan[ch].NoteNum-1+x);
		  break;
		case 2:
		  chan[ch].period = note_to_period(ch,chan[ch].NoteNum-1+y);
		  break;
		}
	    }
	  break;
	case PORT_UP: // pitch slide up -- tick based 1xx
	  // tick 0 do nothing	  
	  // every other tick subtract EffParam from the amiga frequency  
	  // may need to change this for linear slides
	  if ((chan[ch].period -= (n.effParam<<2)) < period_least)
	    chan[ch].period = period_least;
	  break;
	case PORT_DOWN: // pitch slide down -- tick based 2xx
	  // tick 0 do nothing
	  // every other tick add EffParam to the amiga frequency	  
	  // need to change this for linear slides
	  if ((chan[ch].period += (n.effParam<<2)) > period_most)
	    chan[ch].period = period_most;
	  break;
	case PORT_TONE: // tone portamento -- tick based 3xx
	  // port_speed set on tick 0 or previously
	  if (chan[ch].port_period != chan[ch].period)
	    do_portamento(ch);
	  break;
	case VIBRATO: // vibrato -- tick based 4xy
	  // tick 0  - do nothing
	  // need to use tables -- this is complicated
	  do_vibrato(ch,0);
	  break;
	case PORT_VOL_SLIDE: // tone portamento + volume slide 5xx
	  // EffParam refers to the volume slide
	  do_volume_slide(ch);
	  // continue previous portamento 
	  if (chan[ch].period != chan[ch].port_period)
	    do_portamento(ch);
	  break;
	case VIBRATO_VOL_SLIDE: // 6xx
	  // similar to 5xx
	  do_volume_slide(ch);
	  do_vibrato(ch,0);
	  break;
	case TREMOLO: // 7xx tremolo
	  // similar to vibrato except it affects volume rather than pitch
	  do_tremolo(ch);
	  break;
	case PAN: // not used - panning
	  
	  break;
	case VOLUME_SLIDE: // volume slide -- tick based Axy
	  // tick 0 do nothing
	  // on other ticks
	  do_volume_slide(ch);
	  break;
	case RETRIG_NOTE: // E9y retrig note
	  if (!(tick % y))
	    {
	      chan[ch].reset_channel();
	      if (chan[ch].smp)
		{
		  chan[ch].volume = chan[ch].smp->volume;		  
		  callback_ch_volume(ch);
		}
	    }
	  break;
	case NOTE_CUT: // ECy cut note
	  if (tick == y)
	    {
	      chan[ch].volume = 0;
	      callback_ch_volume(ch);
	    }
	  break;
	case NOTE_DELAY: // EDy delay note
	  if (tick == y)
	    {
	      if (n.NoteNum)
		{
		  chan[ch].NoteNum = n.NoteNum;
		  chan[ch].period  = note_to_period(ch,n.NoteNum-1);
		  callback_ch_note(ch);
		}
	      if (n.number)
		{
		  chan[ch].Number   = n.number;
		  chan[ch].reset_channel();
		  if (chan[ch].smp)
		    {
		      chan[ch].volume  =  chan[ch].smp->volume;
		      callback_ch_volume(ch);
		    }
		}
	      if (n.volume)
		{
		  if (n.volume > 66)
		    chan[ch].volume = 64;
		  else
		    chan[ch].volume = n.volume - 1;
		  callback_ch_volume(ch);
		}
	    }
	  break;
	case FUNK_IT: // EFy invert loop  (play sample backwards)
	  // not supported in any player or tracker
	  break;
	case TREMOR:
	  do_tremor(ch);
	  break;
	case FINE_VIBRATO:
	  do_vibrato(ch,FINE_VIBRATO);
	  break;
	case GLOBAL_VOLUME_SLIDE:
	  if (!x && !y) // retrieve last global slide
	    {
	      x = chan[ch].gv_slide >> 4;
	      y = chan[ch].gv_slide & 0xf;
	    } 
	  else // remember this global volume slide
	    chan[ch].gv_slide = n.effParam;

	  if ((x>0) && (y>0))
	    break;
	  else if (x>0)
	    {
	      if ((gv+x) > 64)
		set_gv(64);
	      else
		set_gv(gv+x);
	    }
	  else
	    { 
	      if (gv > y)
		set_gv(gv-y);
	      else
		set_gv(0);
	    }
	    break;
	case STEREO_CONTROL:
	  break;
	case RETRIG_VOL_SLIDE:
	  {
	    byte ry = chan[ch].retrig >> 4;
	    byte rx = chan[ch].retrig & 0xf;
	    if (!ry)
	      break;
	    if (!(tick % ry))
	      {
		switch (rx)
		  {
		  case 0x1:
		    chan[ch].volume -= 1;
		    break;
		  case 0x2:
		    chan[ch].volume -= 2;
		    break;
		  case 0x3:
		    chan[ch].volume -= 4;
		    break;
		  case 0x4:
		    chan[ch].volume -= 8;
		    break;
		  case 0x5:
		    chan[ch].volume -= 16;
		    break;
		  case 0x6:
		    chan[ch].volume *= 2/3;
		    break;
		  case 0x7:
		    chan[ch].volume >>= 1;
		    break;
		  case 0x9:
		    chan[ch].volume += 1;
		    break;
		  case 0xa:
		    chan[ch].volume += 2;
		    break;
		  case 0xb:
		    chan[ch].volume += 4;
		    break;
		  case 0xc:
		    chan[ch].volume += 8;
		    break;
		  case 0xd:
		    chan[ch].volume += 16;
		    break;
		  case 0xe:
		    chan[ch].volume *= 3/2;
		    break;
		  case 0xf:
		    chan[ch].volume *= 2;
		    break;
		  default:
		    break;
		  }
		if (chan[ch].volume > 0x40)
		  chan[ch].volume = 0x40;
		else if (chan[ch].volume < 0)
		  chan[ch].volume = 0;
		chan[ch].reset_channel();
		callback_ch_volume(ch);
	      }
	    break;
	  }
	default:
	  break;
	}
      // process panning envelopes
      // process volume envelopes 
      if (chan[ch].InstMode)
	{
	  chan[ch].EV = chan[ch].process_parameters(&chan[ch].In->volume,
						    &chan[ch].EV_Status);
	  //chan[ch].EP = chan[ch].process_parameters(&chan[ch].In->Panning, 
	  //                                          &chan[ch].PV_Status);  
	}
    }
}

void player::vol_up(const byte ch, const byte x)
{
  chan[ch].volume += x;
  if (chan[ch].volume > 64)
    chan[ch].volume = 64;
  callback_ch_volume(ch);
}

void player::vol_dn(const byte ch, const byte y)
{
  if (chan[ch].volume < y)
    chan[ch].volume = 0;
  else
    chan[ch].volume -= y;
  callback_ch_volume(ch);
}

void player::do_volume_slide(const int ch)
{
  // if x > 0 add x to the volume
  // if y > 0 sub y from the volume
  // make sure volume is with in limits (0 - 64)
  // if ((x >  0) && (y > 0)) don't do anything
  const byte x = chan[ch].vol_slide >> 4;
  const byte y = chan[ch].vol_slide & 0xf;

  if ((x>0) && (y>0))
    return;
  else if (x>0)
    vol_up(ch,x);
  else 
    vol_dn(ch,y);
}

void player::do_portamento(const int ch)
{
  // add or subtract speed to frequency 
  // until note is reached 
  // check for glissando
  if (chan[ch].period < chan[ch].port_period)
    {
      if ((chan[ch].period += chan[ch].port_speed) > chan[ch].port_period)
	chan[ch].period = chan[ch].port_period;
    }
  else if (chan[ch].period > chan[ch].port_period)
    {
      // eeeek underflow on unsigned variables - really difficult to detect
      if ((chan[ch].period -= chan[ch].port_speed) < chan[ch].port_period)
	chan[ch].period = chan[ch].port_period;
    }
}

ubyte babs(sbyte x)
{
  if (x < 0)
    return (x + 128);
  else
    return x;
}

// vib_type = 0 for normal vibrato, or FINE_VIBRATO for you guessed it
void player::do_vibrato(const int ch, const ubyte vib_type)
{
  switch (chan[ch].vib_wave & 0x3)
    {
    case 0: // sine table
      chan[ch].vib_delta = sine_table[babs(chan[ch].vib_pos)];
      break;
    case 1: // ramp down
      {
	chan[ch].vib_delta = babs(chan[ch].vib_pos) << 3;
	if (chan[ch].vib_pos < 0)
	  chan[ch].vib_delta = -chan[ch].vib_delta;	
	break;
      }
    case 2: // square
      chan[ch].vib_delta = 128; 
      break;
    case 3: // random
      chan[ch].vib_delta = sine_table[babs(chan[ch].vib_pos)];
      break;
    }

  chan[ch].vib_delta = chan[ch].vib_delta * chan[ch].vib_dpth;

  if (vib_type == FINE_VIBRATO) 
    chan[ch].vib_delta>>=7;
  else              
    chan[ch].vib_delta>>=5;

  if (chan[ch].vib_pos < 0) 
    chan[ch].vib_delta = -chan[ch].vib_delta;
  
  if ((chan[ch].vib_pos += chan[ch].vib_spd) > 31)
    chan[ch].vib_pos -= 64;
}

void player::do_tremolo(const int ch)
{
  byte delta = 0;
  switch (chan[ch].trem_wave & 0x3)
    {
    case 0: // sine table
      delta = sine_table[chan[ch].trem_pos & 0x1f];
      break;
    case 1: // ramp down
      delta = (chan[ch].trem_pos & 0x1f) << 3;
      if (chan[ch].trem_pos & 0x80)
	delta = 255 - delta;	
      break;
    case 2: // square
      delta = 255;
      break;
    case 3: // random
      delta = sine_table[chan[ch].trem_pos & 0x1f];
      break;
    }
  delta = (delta * chan[ch].trem_dpth) >> 6;
  if (chan[ch].trem_pos & 0x80)
    {
      if (chan[ch].period < delta)
	{
	  chan[ch].volume = 0;
	  callback_ch_volume(ch);
	}
      else
	chan[ch].period -= delta;
    }
  else
    {
      if ((chan[ch].volume + delta) > 64)
	delta = 64 - chan[ch].volume;
      chan[ch].volume += (delta * gv)/64;
      callback_ch_volume(ch);
    }
  if (((chan[ch].trem_pos += chan[ch].trem_spd) & 0x7f) > 31)
    chan[ch].trem_pos = (chan[ch].trem_pos & 0x1f) | 0x80;
}

void player::do_tremor(const byte ch)
{
  const byte x = (chan[ch].tremor >> 4)  + 1;
  const byte y = (chan[ch].tremor & 0xf) + 1;
  
  if (chan[ch].tremor_count >= y)
    {
      chan[ch].volume = 0;
      callback_ch_volume(ch);
    }
  chan[ch].tremor_count = chan[ch].tremor_count % (x + y);
}

void channel::reset_channel(void)
{
  position = 0;
  key_off = false;
  reverse = 0;
  fadevol = 0xffff;
  fadeout = 0;
  if (InstMode)
    {
      EV = 64;
      reset_parameters(&EV_Status);
      //PV = ?;
      //reset_parameters(&PV_Status);
    }
}

// this should work for both panning envelopes and volume envelopes
ubyte channel::process_parameters(Parameters *p, ParamStatus *t)
{
  if (!(p->type & p->ENABLED))
     return 64; // not enabled

  sdword di = t->Position - p->envelope[2*t->Last_Point];
  ubyte value;

  if (!di) // no need to interpolate
    value = p->envelope[2*t->Last_Point+1];
  else
    { // interpolate between the next point and the last point
      const sdword dy = p->envelope[2 * t->Next_Point + 1] -
	p->envelope[2 * t->Last_Point + 1];

      const sdword dx = p->envelope[2 * t->Next_Point] - 
	p->envelope[2 * t->Last_Point];
      
      if (!dx)
	value = p->envelope[2 * t->Next_Point + 1];
      else
	value = (ubyte)(p->envelope[2 * t->Last_Point + 1] + di * dy / dx);
    }
  
  if ((~p->type & p->SUSTAIN) || key_off || 
      (t->Position != p->envelope[2*p->sustain]))
    { // not sustained
      t->Position++;
      if (t->Position >= p->envelope[2*t->Next_Point])
	{
	  t->Last_Point = t->Next_Point;
	  t->Next_Point++;
	  if (p->type & p->LOOP)
	    {
	      if (t->Next_Point > p->loop_end)
		{
		  t->Last_Point = p->loop_start;
		  t->Next_Point = t->Last_Point+1;
		  t->Position = p->envelope[2*t->Last_Point];
		}
	    }
	  else
	    {
	      if (t->Next_Point > p->points)
		{
		  t->Next_Point--;
		  t->Position--;		
		}
	    }
	}	
    }
  return value;
}

void channel::reset_parameters(ParamStatus *t)
{
  t->Position   = 0;
  t->Last_Point = 0;
  t->Next_Point = 1;
}

////////////////////////////////////////////////////////////////////////////
// module playing functions
////////////////////////////////////////////////////////////////////////////
void player::play_pattern(int pattern, uword row)
{
  if (row > mod->rows[pattern])
    this->row = 0;
  else
    this->row = row;
  this->pattern = pattern;
  // FT2 rows are from 1 - 256 so can't have zero length pattern
  // well use udword size row for really long patterns
  while (this->row < mod->rows[pattern])
    {
      tick = 0;
      while (1) // play a row
	{	
	  play_tick();	  
	  if (tick >= speed)
	    {
	      if (!pattern_delay)
		break;
	      else
		pattern_delay--;
	    }
	}
      if (pattern_break || pattern_jump)
	{
	  if (pattern_jump)
	    this->order = next_order;
	  this->row = next_row;
	  return;
	}
      if (pattern_loop)
	{
	  this->row = next_row - 1;
	  pattern_loop = 0;
	}
      this->row++;
    }
  this->row = 0;
}

void player::play(void)
{
  // check that a mod has been loaded
  if (!mod) 
    {
      callback_error("No MOD loaded");
      error = 1;
      return;
    }
  // linux uses copy on write
#ifdef USE_FORK

  if (child) // send SIGCONT so child will start if paused
    {
      if (paused)
	if (kill(child,SIGCONT))
	  callback_error("kill");
      return;
    }
  // fork
  wait(0);  
  if ((child = fork()) == -1) 
    {
      callback_error("fork");
      return;
    }
  if (child == 0) 
    { 
      forked_player = 1;
      //  set up handler for SIGINT ?
      struct sigaction sa;
      sa.sa_handler = sig_handler;
      sa.sa_flags |= SA_RESTART;
      sigaction(SIGUSR1, &sa,0);
      // sleep for a bit, so parent has time to set up stuff for callback's
      usleep(100);
#endif
      // make sure the parents display is correct
      callback_speed();
      callback_bpm();
      // check to make sure mix_buf is allocated
      if (!mix_buf)
	{
	  callback_error("No mix buffer allocated");
	  error = 1;
	  return;
	}

      // play the module, then exit
      // go through the order;
      order = 0;
      pattern = mod->order[order];
      // highest pattern number is mod->NumPatterns
      while (pattern < mod->NumPatterns) // this is not quite right
	{                                
	  callback_order();
	  play_pattern(pattern,row);
	  order++;
	  if (order == mod->SongLength) 
	    {
	      // end ?
	      callback_end();
	      
	      // reset variables to initial values
	      set_gv(mod->global_volume);
	      set_speed(mod->speed);
	      set_bpm(mod->bpm);

	      if (mod->restart >= mod->SongLength)
		order = 0;
	       else
		order = mod->restart;
	      pattern = mod->order[order];
	    }
	  else
	    pattern = mod->order[order];
	}
#ifdef USE_FORK
      _exit(0);
    }
#endif      
}

void player::stop(void)
{
  // i dont know how !
#ifdef USE_FORK
  if (child > 0) 
    {
      // send a SIGCONT -- so the child receives the SIGUSR1 if its stopped
      if (kill(child,SIGCONT))
	callback_error("kill");
      if (kill(child, SIGUSR1)) 
	{
	  callback_error("kill");
	  return;
	}
      wait(0);
      child = 0;
    }
  else 
    {
      //      if (forked_player)// I'm the child !	
      //_exit(0);
    }
#endif
}

void player::pause(void)
{
#ifdef USE_FORK
  if (child > 0) 
    {
      if (kill(child, SIGSTOP)) 
	callback_error("kill");
      paused = 1;
    }
#endif
}


/////////////////////////////////////////////////////////////////////////////
// get and set functions 
/////////////////////////////////////////////////////////////////////////////

void player::release_mod(void)
{
  mod = 0;
  free(chan);
  chan = 0;
}

void player::set_mod(Module *new_mod)
{
  if (this->mod==new_mod)
    return;
  if (this->mod) 
    release_mod();
  this->mod = new_mod; 
  set_bpm(mod->bpm);
  set_speed(mod->speed);
  set_gv(mod->global_volume);
  order = 0;
  callback_order();
  chan = (channel *)calloc(mod->channels, sizeof(channel));
  if (!chan)
    { 
      callback_error("Can't allocate memory for channels");
      error = 1; // consider throwing an exception ?
    }
  for (int x =0; x < mod->channels; x++)
    chan[x].fadevol = 0xffff;
  if (mod->linear)
    {    
      period_most   = 7680;
      period_least  = 64;
      create_linear_freq_table();
    }
  else
    {
      period_most   = 27392;
      period_least  = 14;  // 9?
    }
}


//bpm should be greater than 32
void player::set_bpm(const int bpm)
{  
  this->bpm = bpm;

  //  mix_length = mixing_rate / tick_frequency;
  this->mix_length = (5L * get_rate()) / (bpm<<1);
  if (!this->mix_buf || (mix_length > buf_length))
    {
      // allocate 32 bit mixing buffer -- remember: multiply by 2 if stereo  
      this->buf_length = mix_length;
      
      this->mix_buf = (sdword *)malloc(buf_length*sizeof(udword));
      if (!this->mix_buf)
	{
	  callback_error("Can't allocate memory for mix buffer");
	  error = 1;
	}
    }
  callback_bpm(); // notify user interface
}

void player::set_speed(const int speed)
{
  this->speed = speed;
  callback_speed();
}

void player::set_gv(const int gv)
{
  this->gv = gv;
  callback_gv();
}

Module *player::get_mod(void)
{
  return mod;
}

ubyte player::get_bpm(void)
{
  return bpm;
}

ubyte player::get_speed(void)
{
  return speed;
}

// make sure NoteNum is the real note number
// the notes in the pattern data are stored as 1 + the real note number
// empty notes are stored as 0
sdword player::note_to_period(byte ch, byte NoteNum)  
{
  Sample *samp = 0;
  if (chan[ch].InstMode)
    {      
      if (chan[ch].In) // make sure sample exists
	{
	  const ubyte& Num = chan[ch].In->key_samp[chan[ch].NoteNum-1]; 
	  const uword& NumSamp = chan[ch].In->NumSamples;
	  if ((NumSamp > 0) || (Num < NumSamp))
	    samp=chan[ch].In->samples[Num];
	}
    }
  else
    samp = chan[ch].smp;

  if (!samp) 
    return 0;

  if (mod->linear)
    return (7680-(word(NoteNum+samp->relative_note)<<6) -
	    (sbyte(samp->finetune)/2));
    
  
  return (udword)period_table[NoteNum];
}

/////////////////////////////////////////////////////////////////////////////
// interface functions -- mostly empty
/////////////////////////////////////////////////////////////////////////////
void player::callback_order(void)
{
#ifdef PRINT_PATTERNS
  printf("[%.2X : %.2X] ", order, pattern);
  fflush(stdout);
#endif
}
void player::callback_bpm(void)
{
}
void player::callback_speed(void)
{
}
void player::callback_gv(void)
{
}
void player::callback_end(void)
{
}
void player::callback_ch_volume(int)
{
}
void player::callback_ch_note(int)
{
}
void player::callback_ch_smp(int)
{
}
void player::callback_ch_inst(int)
{
}

void player::callback_error(const char *)
{
}

/////////////////////////////////////////////////////////////////////////////
// mixing functions
/////////////////////////////////////////////////////////////////////////////
// clear the mix_buf before doing this

void player::mix_channels(void)
{
  long rate = DSP::get_rate();

  // clear the buffer
  //  memset(mix_buf,0,mix_length*4); // change 4 to mix_size ?
  for (int p=mix_length;p--;)
    mix_buf[p] = 0;// not sure which  

  for (ubyte ch = mod->channels; ch--;)
    {
      if (((chan[ch].smp&&chan[ch].smp->data)||(chan[ch].In))&&chan[ch].period)
	{
	  sdword mix_period = chan[ch].period + chan[ch].vib_delta;
	  if (mix_period > period_most)
	    mix_period = period_most;
	  else if (mix_period < period_least)
	    mix_period = period_least;

	  Sample *samp;
	  if (chan[ch].InstMode)
	    {
	      if (chan[ch].In)
		{
		  byte Num = chan[ch].In->key_samp[chan[ch].NoteNum-1];
		  int NumSamp = chan[ch].In->NumSamples;
		  if ((NumSamp==0) || (Num > NumSamp))
		    continue; // no sample to play
		  samp = chan[ch].In->samples[Num];	  
		}
	      else
		continue;
	    }
	  else
	    samp = chan[ch].smp;
	  // scale = sample_frequency in HZ / mix_rate
	  udword scale;
	  if (mod->linear)
	    {
	      scale = (linear_freq[mix_period] << FBITS) / rate;
	    }
	  else
	    {
	      scale = (samp->frequency<<FBITS) / rate;
	      scale = (scale*1712) / mix_period;
	    }
	  udword end;
	  udword playlength;
	  udword looplength = 0;

	  if (samp->flags & samp->SMP_LOOP) 
	    {
	      playlength = mix_length;
	      looplength =  samp->loop_end - samp->loop_start;
	      end = samp->loop_end-1;
	    }
	  else
	    {
	      // playlength = scaled length of sample left to play
	      playlength = (samp->length<<FBITS)-chan[ch].position;
	      playlength = playlength / scale;
	      if (playlength > mix_length)
		playlength = mix_length;
	      end = samp->length-1;
	    }
	  
	  sdword *buffer = mix_buf;
	  // calculate mix volume for this channel
	  ubyte mix_volume = (chan[ch].volume*gv)/64;
	  // Is the envelope volume only valid when the volume 
	  // is not set explicitly ?
	  // Might need to change this.
	  if (chan[ch].InstMode) 
	    {
	      mix_volume = (mix_volume*chan[ch].EV)/64;
	    }
	  if (chan[ch].key_off)
	    {
	      if (chan[ch].fadevol > chan[ch].fadeout)
		{
		  chan[ch].fadevol -= chan[ch].fadeout; 
		  mix_volume = ubyte(udword(mix_volume * chan[ch].fadevol)
				     / 0x10000);
		}
	      else
		{
		  chan[ch].fadevol = 0;
		  mix_volume = 0;
		}
	    }

	  if (!mix_volume)
	    {
	      // no volume don't bother mixing -- just update position
	      chan[ch].position += playlength<<FBITS;
	      // need to update this for ping pong loops
	      if (samp->flags & samp->SMP_LOOP) // specially for tiny loops
		while ((chan[ch].position>>FBITS) > end) 
		  chan[ch].position -= (looplength<<FBITS);
	    }
	  else
	    {	 
	      byte *start = samp->data;
	      mix_volume--;
	      
	      if (samp->flags & Sample::SMP_16BIT)
		{
		  while (playlength--)
		    {
		      check_for_end(ch,end,looplength,samp->flags);
		      
#if 1
      sword value =  *((sword *)start + (chan[ch].position>>FBITS));
      *buffer++ += ((value * mix_volume / 64)<<16) / mod->channels;
#else		      
		      ubyte bu = *(start+(chan[ch].position>>FBITS)*2+1)^0x80;
		      ubyte bl = *(start+(chan[ch].position>>FBITS)*2); 
		      ubyte carry;

		      if (bu & 0x80)
			carry = ~bu & 1;
		      else
			carry = bu & 1;

		      *buffer++ += (sdword)(((volume_table[mix_volume][bu]<<24) | ((ubyte)volume_table[mix_volume][bl]<<16)) + (carry<<23)) / mod->channels;
#endif		      
		      update_position(ch,scale,samp->loop_start);
		    }
		}
	      else // 8bit sample mixing
		{
		  while (playlength--)
		    {
		      check_for_end(ch,end,looplength,samp->flags);

		      *buffer++ += (sdword)(volume_table[mix_volume][*(start+(chan[ch].position>>FBITS))]<<24) / mod->channels;

		      update_position(ch,scale,samp->loop_start);
		    }
		}
	    }
          if ((samp->flags&~Sample::SMP_LOOP) && ((chan[ch].position>>FBITS) > end))
	    {
	      if (chan[ch].InstMode)
		{
		  chan[ch].In = 0;
		  chan[ch].InstMode = 0;
		  callback_ch_inst(ch);
		}
	      else
		{
		  chan[ch].smp = 0;
		  callback_ch_smp(ch);
		}
	      chan[ch].reset_channel();
	    }
	}
    }
}

void player::update_position(int ch, udword scale, udword loop_start)
{
  if (chan[ch].reverse) 
    {
      if (chan[ch].position > (loop_start<<FBITS)+scale) 
	{
	  chan[ch].position -= scale;
	}
      else
	{			    
	  chan[ch].reverse = 0;
	  chan[ch].position= 2*(loop_start<<FBITS) - chan[ch].position + scale;
	}
    }
  else 
    chan[ch].position += scale;  
}

void player::check_for_end(int ch, udword end, udword looplength, ubyte flags)
{
  if ((chan[ch].position>>FBITS) > end)
    {
      if (flags & Sample::SMP_BIDI)
	{
	  chan[ch].reverse = 1;
	  chan[ch].position= 2 * (end << FBITS) - chan[ch].position;
	}
      else if (flags & Sample::SMP_LOOP)
	while ((chan[ch].position>>FBITS) > end)
	  chan[ch].position -= (looplength<<FBITS);
    } 
}

// functions for initialisation

void player::create_volume_table(void)
{
  for (int i = 1; i < 65; i++) 
    {
      for (int j = 0; j < 256; j++)
	volume_table[i-1][j] = (sbyte)((double)i*double(j-128)/(double)64.0);
    }
}

int player::create_linear_freq_table(void)
{
  linear_freq = (udword *)malloc(sizeof(udword)*7681);

  if (!linear_freq)
    return 1;

  for(int i=0; i < 7681; i++)
    linear_freq[i] = udword(8363.0 * pow(2, float(4608-i) / 768.0));

  return 0;
}

