#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#include "fmod.h"

/****************************************************************************
 *    Name : AllocInst                                                      *
 * Purpose : To allocate memory for the requested instrument number         *
 *  Passed : UBYTE instno - The instrument number we want allocated and     *
 *                          initialized.                                    *
 * Returns : an errorcode                                                   *
 *  Locals : -                                                              *
 *   Notes :                                                                *
 ****************************************************************************/
UBYTE AllocInst(UBYTE instno) {
	if ((inst[instno] = (SampleHDR *)calloc(sizeof(SampleHDR),1)) == NULL) {
		return ERR_OUT_OF_MEM;
	}
	inst[instno]->middlec=8363;
	return 0;			// no error
}


/****************************************************************************
 *    Name : LoadSample                                                     *
 * Purpose : To Load a sample from a file and upload it to GUS dram         *
 *  Passed : UBYTE number - number of the sample we want to store it as     *
 *           UBYTE type - the value to XOR each byte of the sample with     *
 *                        if it is 255, then the values are DELTA VALUES    *
 * Returns : an errorcode                                                   *
 *  Locals : char *samplebuf - buffer to hold 64kb of sampledata            *
 *           UBYTE count - number of 64kb blocks to write                   *
 *			 UWORD dcount - count variable for deltas                       *
 *           UWORD remainder - 16 value containing anything over 64kb       *
 *           UDWORD samplelength - length of sample data to read            *
 *           static UDWORD gusoff - position in gus dram to poke the sample *
 *   Notes : This version now supports any sized sample, Deltas values and  *
 *           16 bit samples, as long as they fit on the GUS                 *
 ****************************************************************************/
UBYTE LoadSample(UBYTE instno, FILE *fp, UBYTE type) {
	char *samplebuf;
	UWORD count, numbytes, oldval=0, newval;
	UDWORD samplelen, remainder, nextseekpos;
	static UDWORD gusoff;

	// Reset GUS position to 0 if it is the 1st sample (mainly for wildcards
	// when multiple songs are being loaded one after the other).
	if (instno == 0) gusoff = 0;

	// If no sample data then return.
	if (!inst[instno]->length) return 0;

	// Remember where to seek too for the end of the sample, as looping
	// samples are clipped and only up to the loopend of the sample is loaded
	nextseekpos = ftell(fp) + inst[instno]->length;

	// Only load as much as we need - ie set sample length to loop end.
	if (inst[instno]->mode & 8) {
		inst[instno]->length =
		inst[instno]->loopend;
	}
	samplelen = inst[instno]->length;

	// Set sample's GUS position in DRAM (32 byte aligned).
	gusoff += 31;
	gusoff &= -32;
	if ((gusoff+samplelen) > gusdram) return ERR_GUS_DRAM_FULL;
	inst[instno]->offset = gusoff;

	// find out how much memory to allocate for a sample buffer.
	if (!(samplebuf =(char *)malloc((samplelen > 65535 ? 65535:samplelen)))) {
		return ERR_OUT_OF_MEM;
	}
	remainder = samplelen;

	while (remainder) {
		numbytes = (remainder > 65535 ? 65535 : remainder);
		remainder -= numbytes;

		fread(samplebuf, numbytes, 1, fp);
		if (type==255) {
			if (inst[instno]->mode & 4) {
				// make a new pointer to the sample buffer using words(16bit)
				WORD *samplebuf16 = (WORD *)samplebuf;
				for (count=0; count < (numbytes >> 1); count++) {
					newval = samplebuf16[count] + oldval;
					samplebuf16[count] = newval;
					oldval = newval;
				}
			}
			else {
				for (count=0; count < numbytes; count++) {
					newval = samplebuf[count] + (UBYTE)oldval;
					samplebuf[count]= (UBYTE)newval;
					oldval = (UBYTE)newval;
				}
			}
			GUSUpload(gusoff, samplebuf, numbytes, 0);
		}
		else GUSUpload(gusoff, samplebuf, numbytes, type);

		gusoff+=numbytes;
	}

	// now it's uploaded move filepointer to end of sample in file
	fseek(fp, nextseekpos, SEEK_SET);

	// because GUS seems to reference 16bit data in terms of words and not
	// bytes.. the actuall stored values should be divided by 2.
	if (inst[instno]->mode & 4) {
		inst[instno]->length >>= 1;
		inst[instno]->loopstart >>= 1;
		inst[instno]->loopend >>= 1;
		inst[instno]->offset >>= 1;
	}

	// free sample buffer
	free(samplebuf);

	// Fix end of sample GUS clicks
	// looping (write first byte of sample loop to end of sample)
	if ( (inst[instno]->mode & 8) && !(inst[instno]->mode & 16)) {
		GUSPoke(gusoff, GUSPeek(gusoff-inst[instno]->length+
									   inst[instno]->loopstart));
	}
	// non looping (write a 0 to the end)
	else GUSPoke(gusoff, 0);
	gusoff++;

	return 0;	// no error
}


/****************************************************************************
 *    Name : LoadS3M                                                        *
 * Purpose : To Load an S3M file into memory                                *
 *  Passed : FILE *handle - the handle of the file to be loaded             *
 * Returns : -                                                              *
 *  Locals : UBYTE count - counter variable for instruments, patterns etc.  *
 *           UBYTE temp - holds temporary calculations etc.                 *
 *           UBYTE remap[32] - remapped channel values (sorted)             *
 *			 UBYTE channel - channel number to reference while unpacking    *
 *                           pattern data.                                  *
 *			 UBYTE st3pan - the default panning byte from S3M header        *
 *			 UBYTE instflag - stores flags about instrument, ie 16bits,loop *
 *			 UBYTE mastervol - mastervolume byte from S3M header            *
 *			 UBYTE ERR - Holds the error code if there is an error          *
 *			 UWORD temp16 - Another temporary value, but this one holds more*
 *			 UWORD songlength - Number of orders the header says there is.  *
 *			 UWORD patterns - Number of patterns the header says there is.  *
 *			 UWORD offset - Temporary variable to store calculation of      *
 *                          offset in pattern buffer, according to row etc. *
 *			 UWORD parapoint[355] - Physical offsets of pattern and         *
 *                                  instrument data, divided by 16.         *
 *			 UWORD samplepoint[99] - Parapointer type offsets of sample data*
 *			 Note dummy - Place to read crap into when data should be       *
 *                        discarded.  (ie when referenced channel is outside*
 *                        of the actual number of channels)                 *
 ****************************************************************************/
UBYTE LoadS3M(FILE *handle) {
	UBYTE count, temp, remap[32], channel, st3pan, instflag, mastervol, ERR=0;
	UWORD temp16, songlength, patterns, offset;
	UWORD parapoint[355], samplepoint[99];
	Note dummy;

	FM.restart = 0;
	memset(FM.patlen, 64, 256);             // set each pattern to have 64 rows
	// Blank comment data
	if ((SongMSG = (char *)calloc(1,1)) == NULL) return ERR_OUT_OF_MEM;
	SongMSG[0] = -1;

	fseek(handle, 0, SEEK_SET);             // start at the beginning.
	fread(FM.name, 28, 1, handle);          // read in module name.
	fgetc(handle);                          // 1A byte.  Ignore as some s3m's
											// dont actually have it.
	fseek(handle, 0x20, SEEK_SET);
	fread(&songlength, 2, 1, handle);       // get song length
	fread(&FM.numinsts, 2, 1, handle);      // get number of instruments
	fread(&patterns, 2, 1, handle);         // get number of patterns
	fread(&temp16, 2, 1, handle);           // Flags
	if (temp16 & 64) FM.flags = 1;          // fast volume slides?

	fread(&temp16, 2, 1, handle);           // Created With tracker/version
	if ((temp16 & 0xFFF) == 300) FM.flags = 1;      // fast volume slides?

	// put version info in type string.
	sprintf(FM.type, "S3M%d.%02X", (temp16 & 0xFFF) >> 8, (temp16 & 0xFF));

	// seek to offset 030h
	fseek(handle, 0x30, SEEK_SET);
	globalvol = fgetc(handle);           // set global volume
	speed = fgetc(handle);               // set default speed
	bpm = fgetc(handle);                 // set default bpm
	mastervol = fgetc(handle);           // st3 mastervol (check bit 8 later)
	fgetc(handle);                       // skip ultraclick removal (duhh)
	st3pan = fgetc(handle);              // default panning flag

	// Find the number of channels and remap the used channels linearly
	fseek(handle, 0x40, SEEK_SET);

	FM.channels=0;
	memset(remap, 255, 32);
	for (count=0; count<32; count++) {
		temp = fgetc(handle);
		if (temp < 16) {
			remap[count] = FM.channels;
			if (temp <= 7) panval[FM.channels] = defpan;
			else panval[FM.channels] = 0xF0-defpan;
			FM.channels++;
		}
	}

	// Load order data & calculate number of physical patterns
	fseek(handle, 0x60, SEEK_SET);

	fread(FM.order,songlength,1,handle);
	FM.songlength = 0;
	FM.numpats =0;
	for (count=0; count<songlength; count++) {
		FM.order[FM.songlength]=FM.order[count];
		if (FM.order[count]<254) {
			FM.songlength++;
			if (FM.order[count] > FM.numpats) FM.numpats=FM.order[count];
		}
	}

	// Load in instrument and pattern parapointers
	fread(parapoint,2,FM.numinsts+patterns,handle);

	// check for default panning if the panning flag is set (252 = 0xFC :)
	if (st3pan == 252) {
		for (count=0; count<32; count++) {
			temp = fgetc(handle);
			if (temp & 0x10) panval[remap[count]] = ((temp & 0xF)<<4);
		}
	}

	// if stereo flag is not set then make song mono
	if (!(mastervol&128)) for (count=0;count<32;count++) panval[count]=0x80;

	// Load instrument information
	for (count=0; count<FM.numinsts; count++) {

		// allocate memory for the instrument
		ERR = AllocInst(count);
		if (ERR) return ERR;

		// jump to instrument parapointer.
		fseek(handle, (UDWORD)((parapoint[count]) << 4), SEEK_SET);
		fseek(handle, 13, SEEK_CUR);                                            // skip filename

		// find parapointer to actual sample data (3 bytes)
		temp = fgetc(handle);				// read high byte
		fread(&temp16, 2, 1, handle);		// read lower word
		samplepoint[count] = ((UDWORD)temp << 16) + temp16;

		// get rest of inforamtion
		fread(&inst[count]->length, 2, 1, handle);              // get length
		fseek(handle, 2, SEEK_CUR);
		fread(&inst[count]->loopstart, 2, 1, handle);           // loop start
		fseek(handle, 2, SEEK_CUR);
		fread(&inst[count]->loopend, 2, 1, handle);             // loop end
		fseek(handle, 2, SEEK_CUR);
		inst[count]->volume = fgetc(handle);                    // volume
		fseek(handle, 2, SEEK_CUR);
		instflag = fgetc(handle);                                                   // loop/16bit
		fread(&inst[count]->middlec, 2, 1, handle);             // C2SPD
		fseek(handle, 14, SEEK_CUR);
		fread(inst[count]->name, 1, 28, handle);				// Inst name
		fseek(handle, 4, SEEK_CUR);								// 'SCRS'

		// fix it so sample length = loopend
		// also check that loopend doesnt exeed length of sample, and that
		// any samples with a loop length of 2 or less have no loop.
		inst[count]->mode = 0;
		if(instflag&1) {
			inst[count]->mode = 8;
			if (inst[count]->loopend > inst[count]->length)
				inst[count]->loopend = inst[count]->length;
			if (inst[count]->loopend <= 2) inst[count]->mode=0;
			inst[count]->length = inst[count]->loopend;
		}
		if(instflag&4) inst[count]->mode += 4;
	}

	// Load the pattern data

	for (count=0; count<=FM.numpats; count++) {

		// jump to pattern parapointer +2, skipping length bytes.
		fseek(handle,(((UDWORD)parapoint[FM.numinsts+count])<<4)+2,SEEK_SET);

		// calculate size of and allocate memory for pattern
		if ((patbuff[count] = (char *)calloc(FM.channels*320L,1))==NULL) {
			while (count > 0) {
				count--;
				free(patbuff[count]);
			}
			for (count=0; count<FM.numinsts; count++) free(inst[count]);
			return ERR_OUT_OF_MEM;
		}

		// wipe out pattern data for this pattern
		dummy.note=NONOTE;
		dummy.number=0;
		dummy.volume=NOVOL;
		dummy.effect=0;
		dummy.eparm=0;
		for (offset=0; offset< (FM.channels*320); offset+=5) {
			current = (Note *)(patbuff[count]+offset);
			memcpy(current, &dummy,5);
		}

		// unpack one pattern (pattern number 'count')
		row=0;
		while(row<64){
			temp=fgetc(handle);

			// if it not 0, unpack note, if it is 0 then end of row.
			if(temp) {
				channel=remap[temp & 31];

				if(channel<FM.channels){
					offset = ((UDWORD)FM.channels*5L*row)+   // row offset
							 (channel*5);                   // column offset
					current = (Note *)(patbuff[count]+offset);
				}
				else current = &dummy;  // if it has told us to use a channel
										// outside of the number of channels,
										// then load it into a dummy.

				// if there is a note
				if(temp & 32) {
					temp16 = fgetc(handle);

					// convert s3m note to internal type note

					switch(temp16) {
						case 255 : current->note=NONOTE;
								   break;
						case 254 : current->note=KEYOFF;
								   break;
						 default : current->note=((temp16>>4)*12)+(temp16&0xf);
					};
					current->number= fgetc(handle);
				}

				// if there is a volume byte
				if(temp & 64) current->volume=fgetc(handle);

				// if there is an effect
				if(temp & 128) {

					current->effect = fgetc(handle);
					current->eparm = fgetc(handle);

					// convert S3M effects to FMOD effects (they are same but
					// just use different numbers - ie more .MOD like)
					switch(current->effect) {
						case 0: current->eparm  = 0; // if there is no effect
								break; //but there is a param-dont want arp
						case 1: current->effect = 0x10;   // Axx set speed to xx
								break;
						case 2: current->effect = 0xB;    // Bxx pattern jump
								break;
						case 3: current->effect = 0xD;    // Cxx patternbreak
								break;
						case 4: current->effect = 0x11;   // Dxy volumeslide
								break;
						case 5: current->effect = 0x12;   // Exy portamento down
								break;
						case 6: current->effect = 0x13;   // Fxy portamento up
								break;
						case 7: current->effect = 0x3; 	  // Gxx Porta to note
								break;
						case 8: current->effect = 0x4; 	  // Hxy vibrato
								break;
						case 9: current->effect = 0x14;   // Ixy tremor
								break;
						case 0xa: current->effect = 0; 	  // Jxy arpeggio
								break;
						case 0xb: current->effect = 0x15;  // Kxy Vib+Volslide
								break;
						case 0xc: current->effect = 0x16;  // Lxy Port+Volslide
								break;
						case 0xf: current->effect = 0x9;  // Oxx sample offset
								break;
						case 0x11: current->effect = 0x17;// Qxy Retrig+volslide
								break;
						case 0x12: current->effect = 0x7; // Rxy tremolo
								break;
						case 0x13: // Sxx special commands
								current->effect = 0xE;
								switch(current->eparm >> 4) {
									// these effects need to have
									// their effect numbers changed
									case 0x1: current->eparm = 0x30+  // S1x
											  (current->eparm & 0xF);
											  break;
									case 0x2: current->eparm = 0x50+  // S2x
											  (current->eparm & 0xF);
											  break;
									case 0x3: current->eparm = 0x40+  // S3x
											  (current->eparm & 0xF);
											  break;
									case 0x4: current->eparm = 0x70+  // S4x
											  (current->eparm & 0xF);
											  break;
									case 0xa: current->effect = 0x1A; // SAx
											  current->eparm &= 0xF;
											  break;
									case 0xb: current->eparm = 0x60+  // SBx
											  (current->eparm & 0xF);
											  break;
								};
								break;
						case 0x14: current->effect = 0xF; // Txx tempo
								break;
						case 0x15: current->effect = 0x18;// Uxy fine vibrato
								break;
						case 0x16: current->effect = 0x19;// Vxx global volume
								break;
						case 0x18: current->effect = 0x8; // Xxx amiga command 8xx
								break;
					};
				}
			}
			else row++;
		}
	}

	// Load Sample Data
	for (count = 0; count < FM.numinsts; count++) {
		fseek(handle, (UDWORD)(samplepoint[count]) << 4, SEEK_SET);
		ERR = LoadSample(count, handle, 128);
		if (ERR) return ERR;
	}

	// no errors
	return 0;
}


/****************************************************************************
 *    Name : LoadFAR                                                        *
 * Purpose : To load in and convert a FAR music file to the internal format *
 *  Passed : FILE *handle - the handle to the FAR file opened.              *
 * Returns :                                                                *
 *  Locals : UBYTE samplemap[8] - 64 bits which are set on or off according *
 *                                to which samples are used or not          *
 *			 UBYTE temp8 - an 8 bit temporary variable to hold calculations *
 *			 UBYTE rows - number of rows in the pattern being read          *
 *			 UBYTE ERR - place to hold error code if a function returns one *
 *			 UWORD count - a counter variable                               *
 *			 UWORD count2 - another counter variable                        *
 *			 UWORD patsize[256] - pattern sizes in FAR, in bytes I think    *
 *			 UWORD STlen - Size of song text                                *
 *			 UWORD HDRlen - Size of main header                             *
 *			 UWORD offset - offset in pattern buffer according to row etc.  *
 *   Notes :                                                                *
 ****************************************************************************/
UBYTE LoadFAR(FILE *handle) {
	UBYTE samplemap[8], temp8, rows, ERR=0;
	UWORD count, count2, patsize[256], STlen=0, HDRlen=0, offset;

	FM.channels=16;
	fseek(handle, 4L, SEEK_SET);
	fread(FM.name, 40, 1, handle);            // song name
	fseek(handle, 3L, SEEK_CUR);              // skip 3 bytes
	fread(&HDRlen, 2, 1, handle);             // remaining length of header
	fseek(handle, 26L, SEEK_CUR);             // skip next 25 bytes

	speed = fgetc(handle);
	bpm = 99;

	for (count=0; count<16; count++) panval[count]=((UWORD)fgetc(handle)<<4);
	fseek(handle, 4L, SEEK_CUR);

	fread(&STlen, 2, 1, handle);              // song text length
	if ((SongMSG = (char *)calloc(1,1)) == NULL) return ERR_OUT_OF_MEM;
	fseek(handle, STlen, SEEK_CUR);			  // skip song message
	SongMSG[0] = -1;			     		  // terminate with null byte

	fread(FM.order, 256, 1, handle);          // read in orders
	fgetc(handle);
	FM.songlength = fgetc(handle);
	FM.restart = fgetc(handle);
	fread(patsize, 256, 2, handle);          // read in pattern sizes

	// find number of patterns
	FM.numpats=0;
	for(count=0;count<256;count++){
		if(patsize[count] && (count+1)>FM.numpats) FM.numpats=count;
	}

	// skip up to pattern data
	fseek(handle, HDRlen-(869+STlen), SEEK_CUR);

	// load pattern data
	for (count=0; count<= FM.numpats; count++) {
		rows=0;
		if (patsize[count]) {
			rows = fgetc(handle);   		// length of pattern in rows
			fgetc(handle);                  // pattern tempo, ignored

			// allocate memory for the pattern data
			if ((patbuff[count] = (char *)calloc(80*(rows+2),1)) == NULL) {
				while (count > 0) {
					count--;
					free(patbuff[count]);
				}
				return ERR_OUT_OF_MEM;
			}
			offset=0;

			// load in pattern data
			for (count2=0; count2 < ((patsize[count]-2)>>2); count2++) {
				current = (Note *)(patbuff[count]+offset);
				// note
				current -> note = fgetc(handle) + 35;
				if (current->note == 35) current->note=NONOTE;
				// instrument
				temp8= fgetc(handle);
				if (temp8) current -> number = temp8+1;
				else current->number=0;
				// volume
				current -> volume = ((fgetc(handle) & 0xF)<<2);
				if (!current->volume) current->volume=255;
				// effect
				temp8 = fgetc(handle);
				switch (temp8 >> 4) {
					case 0xf : current->effect = (temp8 >> 4);
							   current->eparm = (temp8 & 0xF);
							   break;
					 default : current->effect = 0;
							   current->eparm = 0;
				};
				offset+=5;
			}
		}
		FM.patlen[count] = rows+2;
	}

	// Load sample data

	FM.numinsts=64;
	fread(samplemap, 8, 1, handle);          // read in stupid sample map
	for (count=0; count<FM.numinsts; count++) {
		// allocate memory for the instrument
		ERR = AllocInst(count);
		if (ERR) return ERR;

		if (samplemap[count/8] & (1<<(count%8))) {
			fread(inst[count]->name, 28, 1, handle);
			inst[count]->name[27]=0;                        // null last byte
			fseek(handle, 4, SEEK_CUR);
			fread(&inst[count]->length, 4, 1, handle);
			inst[count]->middlec = FinetoHz(fgetc(handle));
			inst[count]->volume = (fgetc(handle) << 2);
			fread(&inst[count]->loopstart, 4, 1, handle);
			fread(&inst[count]->loopend, 4, 1, handle);
			temp8 = fgetc(handle);
			if (temp8 & 1) inst[count]->mode = 4;   		// 16 bit
			else inst[count]->mode =0;                      // 8 bit
			temp8 = fgetc(handle);
			if (temp8) inst[count]->mode += 8;              // looping
			if (inst[count]->loopend > inst[count]->length)
				inst[count]->loopend = inst[count]->length;

			ERR = LoadSample(count, handle, 0);
			if (ERR) return ERR;
		}
	}

	return 0;		// no error
}


/****************************************************************************
 *    Name : Load669                                                        *
 * Purpose : To load in and convert a 669 music file to the internal format *
 *  Passed : FILE *handle - the handle to the 669 file opened.              *
 * Returns : the error code                                                 *
 *  Locals : UBYTE count - counter variable                                 *
 *           UBYTE temp3 - place to read in the 3 bytes worth of note data  *
 *           UBYTE ERR - place to store error code if a function returns one*
 *           UWORD offset - offset in pattern buffer according to row etc.  *
 *           UDWORD count2 - a 32 bit counting variable.                    *
 *   Notes : ARGH why did i implement such a stupid fucking format!?        *
 ****************************************************************************/
UBYTE Load669(FILE *handle) {
	UBYTE count, temp[3], ERR=0;
	UWORD offset;
	UDWORD count2;

	bpm = 80;
	FM.channels=8;

	// Blank comment data
	if ((SongMSG = (char *)calloc(1,1)) == NULL) return ERR_OUT_OF_MEM;
	SongMSG[0] = -1;

	// set default panning positions LRLRLRLR
	for (count=0; count<8; count++) {
		if (count%2 == 0) panval[count]=defpan;
		else panval[count]= 0xF0-defpan;
	}

	fseek(handle, 2, SEEK_SET);
	fread(FM.name, 28, 1, handle);            		       // read in info
	fseek(handle, 0x6e, SEEK_SET);

	FM.numinsts = fgetc(handle);
	FM.numpats = fgetc(handle)-1;
	FM.restart = fgetc(handle);                                                     // loop order number
	fread(FM.order, 128, 1, handle);                       // order list
	for(count=0;count<128;count++){
		if(FM.order[count]==0xff) break;
	}
	FM.songlength=count;

	speed = fgetc(handle);                                                          // load first tempo
	for (count=0; count< 127; count++) fgetc(handle);       // tempo list
	for (count=0; count< 128; count++) FM.patlen[count] = fgetc(handle)+1;
	for (count=0; count< FM.numinsts; count++) {

		// allocate memory for the instrument
		ERR = AllocInst(count);
		if (ERR) return ERR;

		fread(inst[count]->name, 13, 1, handle);			// sample name
		fread(&inst[count]->length, 4, 1, handle);      	// length
		fread(&inst[count]->loopstart, 4, 1, handle);   	// loop start
		fread(&inst[count]->loopend, 4, 1, handle);     	// loop end
		inst[count]->middlec = 8363;
		inst[count]->volume = 64;

		if (inst[count]->loopend > 2) inst[count]->mode=8;
		else inst[count]->mode = 0;
		if (inst[count]->loopend > inst[count]->length) {
			inst[count]->loopend = inst[count]->length;
			inst[count]->mode = 0;
		}
	}

	// load pattern data

	for (count=0; count <= FM.numpats; count++) {

		// calculate and allocate memory for the pattern data
		if ((patbuff[count] = (char *)calloc(2560,1)) == NULL) {
			while (count > 0) {
				count--;
				free(patbuff[count]);
			}
			for (count=0; count<FM.numinsts; count++) free(inst[count]);
			return ERR_OUT_OF_MEM;
		}
		offset =0;

		for (count2=0; count2 < (FM.channels*64); count2++) {
			// point our little note structure to patbuff
			current = (Note *)(patbuff[count] + offset);
			// load up 4 bytes of note information from file
			fread(temp, 3, 1, handle);
			current -> note = (temp[0] >> 2);
			current -> number =  ((temp[0] & 3)<<4) + (temp[1] >> 4)+1;

			if (temp[0] == 0xFE) {
				current -> note = NONOTE;
				current -> number = 0;
				current -> volume = (temp[1] & 0xF) << 2;
			}
			else if (temp[0] == 0xFF) {
				current -> note = NONOTE;
				current -> number = 0;
				current -> volume = NOVOL;
			}
			else {
				current -> note+= 24;
				if (current->note == 24) current->note=NONOTE;
				current -> volume = (temp[1] & 0xF) << 2;
			}
			if (temp[2] == 0xFF) { current->effect=0; current->eparm=0;}
			else {
				current -> eparm  = temp[2] & 0xF;
				switch(temp[2] >> 4) {
					case 0: current -> effect = 1;          // porta up
							current -> eparm = 0;           // disabled
							break;
					case 1: current -> effect = 2;          // porta down
							current -> eparm =0;            // disabled
							break;
					case 2: current -> effect = 3;          // porta to
							current -> eparm =0;            // disabled
							break;
					case 3: current -> effect = 0xE;        // freq adjust
							current -> eparm += 0x10;       // make it E1
							// what a stupid effect!
							break;
					case 4: current -> effect = 4;          // vibrato
							current -> eparm = 0;       // disabled
							break;
					case 5: current -> effect = 0xF;        // set speed
							break;
					case 6: current -> eparm = 0;
							break;
					case 7: current -> eparm = 0;
							break;
				};
			}
			offset += 5;
		}
	}

	// Load sample data
	for (count = 0; count < FM.numinsts; count++) {
		ERR = LoadSample(count, handle, 128);
		if (ERR) return ERR;
	}

	return 0;	// no errors
}



/****************************************************************************
 *    Name : LoadMTM                                                        *
 * Purpose : To load Multitracker .MTM files into memory.                   *
 *  Passed : FILE *handle - the handle of the file being loaded             *
 * Returns : -                                                              *
 *  Locals : UBYTE temp[3] - 3 byte buffer to hold 1 MTM style note			*
 *			 UBYTE count3 - an 8 bit counter variable                       *
 *			 UBYTE ERR - place to store error code if a function returns one*
 *			 UWORD tracks - number of MTM 'tracks' the file contains        *
 *			 UWORD trackseq - word containg the track number to allocate    *
 *                            to the current pattern and channel number     *
 *			 UWORD count - looks like another counter variable - 16bits     *
 *			 UWORD commentlen - size of comment - song message              *
 *			 UDWORD count2 - whats this.. another counter! wee.. 32bits     *
 *			 char *track[500] - pointers to all MTM tracks.. 500 should be  *
 *                              enough for a song.                          *
 ****************************************************************************/
UBYTE LoadMTM(FILE *handle) {
	UBYTE temp[3], count3, ERR=0;
	UWORD tracks, trackseq, count, commentlen;
	UDWORD count2;
	char *track[500];

	// set some default information for this format

	speed = 6;
	bpm = 125;
	FM.restart=0;
	memset(FM.patlen, 64, 256);         // set each pattern to have 64 rows

	// read header information

	fseek(handle, 4L, SEEK_SET);
	fread(FM.name, 20, 1, handle);              // song name
	fread(&tracks, 2, 1, handle);               // number of tracks
	FM.numpats = fgetc(handle);                 // number of patterns
	FM.songlength = fgetc(handle)+1;            // songlength
	fread(&commentlen, 2, 1, handle);           // length of comments field
	FM.numinsts = fgetc(handle);                // number of instruments
	fgetc(handle);                              // attribute byte
	fgetc(handle);                              // beats per track (?)
	FM.channels = fgetc(handle);                // number of channels

	// read and set panning positions

	for (count=0; count<32; count++) panval[count] = (fgetc(handle) << 4);

	// instrument information

	for (count=0; count<FM.numinsts; count++) {

		// allocate memory for the instrument
		ERR = AllocInst(count);
		if (ERR) return ERR;

		fread(inst[count]->name, 22, 1, handle);        // sample name
		fread(&inst[count]->length, 4, 1, handle);      // sample length
		fread(&inst[count]->loopstart, 4, 1, handle);   // sample loop start
		fread(&inst[count]->loopend, 4, 1, handle);     // sample loop end
		inst[count]->middlec = FinetoHz(fgetc(handle)); // sample finetune
		inst[count]->volume = fgetc(handle);            // sample default vol

		if (inst[count]->loopend > inst[count]->length)
			inst[count]->loopend = inst[count]->length;
		if (inst[count]->loopend > 2) inst[count]->mode=8;
		else inst[count]->mode = 0;
		if (fgetc(handle) == 1) inst[count]->mode+=4;   // attribute: 8/16bit
	}

	// read order list

	fread(FM.order, 128, 1, handle);

	// try to allocate memory for tracks

	for (count=0; count<tracks; count++) {
		if ((track[count] = (char *)calloc(192,1)) == NULL) {
			while (count > 0) {
				count--;
				free(track[count]);
			}
			for (count=0; count<FM.numinsts; count++) free(inst[count]);
			return ERR_OUT_OF_MEM;
		}
	}

	// load track data into track buffer

	for (count=0; count<tracks; count++) fread(track[count], 192, 1, handle);

	// first try and allocate memory for all patterns

	for (count=0; count<=FM.numpats; count++) {
		if ((patbuff[count] = (char *)calloc(320L * FM.channels,1)) == NULL) {
			while (count > 0) {
				count--;
				free(patbuff[count]);
			}
			for (count=0; count<tracks; count++) free(track[count]);
			for (count=0; count<FM.numinsts; count++) free(inst[count]);
			return ERR_OUT_OF_MEM;
		}
	}

	// now convert the track data to our pattern data

	for (count=0; count<=FM.numpats; count++) {
		for (count2=0; count2<32; count2++) {
			fread(&trackseq, 2, 1, handle);

			if (count2 >= FM.channels) continue;  // <- bugfix

			if (trackseq > 0) {

				// copy track 'trackseq' into patterndata
				for (count3 = 0; count3 < 64; count3++) {

					// set pointer at right offset in the pattern buffer
					current = (Note *)(patbuff[count]+(5L*count3*FM.channels)+
									  (count2*5));

					//	        BYTE 0   BYTE 1   BYTE 2
					//	       ppppppii iiiieeee aaaaaaaa
					//	      
					//	       p = pitch value (0=no pitch stated)
					//	       i = instrument number (0=no instrument number)
					//	       e = effect number
					//	       a = effect argument

					// copy 3 bytes from track data into a 3 byte buffer
					memcpy(temp, track[trackseq-1]+(count3*3), 3);

					// get info from 3 bytes of mtm note data
					current->note = temp[0] >> 2;
					if (!current->note) current->note = NONOTE;
					else current->note+=24;

					current->number = ((temp[0] & 0x3) << 4) + (temp[1] >> 4);
					current->effect = (temp[1] & 0xF);
					current->eparm = temp[2];

					// mtm doesnt have a volume byte so use novolume value
					current -> volume = NOVOL;

					// mtm vol slide fix
					if(current->effect==0xa && current->eparm&0xf0)
								current->eparm &= 0xf0;
				}
			}
			else {
				// only write to the buffer if the 0 track is in our pattern
				if (count2 < FM.channels) {
					for (count3 =0; count3 < 64; count3++) {
						// lookup right destination note in pattern buffer
						current=(Note *)(patbuff[count]+(count3*FM.channels*5)+
										 (count2*5));
						current -> note = NONOTE;
						current -> number =0;
						current -> volume =NOVOL;
						current -> effect =0;
						current -> eparm =0;
					}
				}
			}
		}
	}

	// free track data
	for (count=0; count<tracks; count++) free(track[count]);

	// Load song message
	if ((SongMSG = (char *)calloc(commentlen+1,1)) == NULL) {
		return ERR_OUT_OF_MEM;
	}
	fread(SongMSG, commentlen, 1, handle);
	SongMSG[commentlen] = -1;

	// Load sample data (xor with 128)
	for (count = 0; count < FM.numinsts; count++) {
		ERR = LoadSample(count, handle, 128);
		if (ERR) return ERR;
	}

	return 0;	// no errors
}


/****************************************************************************
 *    Name : LoadMOD                                                        *
 * Purpose : To load M.K. or FastTracker type .MOD files into memory.       *
 *  Passed : FILE *handle - the handle of the file being loaded             *
 * Returns : -                                                              *
 *  Locals : UBYTE temp[4] - 4byte temporary buffer.                        *
 *           UBYTE count, count3 - guess.                                   *
 *           UBYTE ERR - place to store error code if a function returns one*
 *           UWORD period - the amiga period stored in the file             *
 *           UWORD offset - offset in pattern buffer according to row etc.  *
 *           UWORD count2 - must be a 16bit counter variable                *
 ****************************************************************************/
UBYTE LoadMOD(FILE *handle) {
	UBYTE temp[4], count, count3, ERR=0;
	UWORD period, offset, count2;

	// set a few default values for this format

	FM.numinsts = 31;
	speed = 6;
	bpm = 125;
	FM.numpats = 0;                     	// currently 0, to be set later.
	FM.restart = 0;
	memset(FM.patlen, 64, 256);             // set each pattern to have 64 rows
	for (count=0; count<32; count++) {      // set panning values LRRL etc
		switch(count%4) {
			case 0:
			case 3: panval[count]=defpan;
					break;
		   default: panval[count]=0xF0-defpan;
		};
	}
	// Blank comment data
	if ((SongMSG = (char *)calloc(1,1)) == NULL) return ERR_OUT_OF_MEM;
	SongMSG[0] = -1;

	// start at the beginning of the file and read in module name.

	fseek(handle, 0, SEEK_SET);
	fread(FM.name, 20, 1, handle);

	// load instrument headers

	for (count=0; count<FM.numinsts; count++) {

		// allocate memory for the instrument
		ERR = AllocInst(count);
		if (ERR) return ERR;

		// read sample name, and fix up bad characters by making them spaces
		fread(inst[count]->name, 22, 1, handle);
		for (count2=0; count2<28; count2++)
			if (inst[count]->name[count2]<32) inst[count]->name[count2]=0;

		// read rest of information.
		inst[count]->length = ((UDWORD)(fgetc(handle) << 8)+fgetc(handle))*2;
		inst[count]->middlec = FinetoHz(fgetc(handle));
		inst[count]->volume = fgetc(handle);
		inst[count]->loopstart = ((UDWORD)(fgetc(handle)<<8)+fgetc(handle))*2;
		inst[count]->loopend = ((UDWORD)(fgetc(handle) << 8)+fgetc(handle))*2
								  + inst[count]->loopstart;

		if (inst[count]->loopend > inst[count]->length)
			inst[count]->loopend = inst[count]->length;
		if (inst[count]->loopend > 2) inst[count]->mode=8;
		else inst[count]->mode = 0;
	}

	// load order data

	FM.songlength = fgetc(handle);                  // number of pats.
	fgetc(handle);                     				// Restart pos.  Usually
													// 127 so it isnt reliable
	fread(FM.order, 128, 1, handle);                // Read in 128 orders
	for (count=0; count<128; count++)
		if (FM.order[count] > FM.numpats) FM.numpats = FM.order[count];
	fread(temp, 4, 1, handle);         				// MOD signature, skip.

	// load pattern data

	for (count=0; count <= FM.numpats; count++) {
		// allocate memory for 1 pattern
		if ((patbuff[count] = (char *)calloc(FM.channels*320,1)) == NULL) {
			while (count > 0) {
				count--;
				free(patbuff[count]);
			}
			for (count=0; count<FM.numinsts; count++) free(inst[count]);
			return ERR_OUT_OF_MEM;
		}
		offset =0;

		for (count2=0; count2 < (FM.channels*64); count2++) {
			// point our little note structure to patbuff
			current = (Note *)(patbuff[count] + offset);
			// load up 4 bytes of note information from file
			fread(temp, 4, 1, handle);
			// store sample number
			current -> number = ((temp[0] & 0xF0) + (temp[2] >> 4));
			// store note
			period = ((temp[0] & 0xF) << 8) + temp[1];
			// convert the amiga period to a note number
			current->note=NONOTE;
			for (count3=0; count3<108; count3++) {
				if (period >= periodtab[count3+24]) {
					current->note = count3;
					count3=109;                      // break from loop
				}
			}
			// store volume byte (mods dont have one so NOVOL = empty)
			current -> volume = NOVOL;
			// store effects and arguments */
			current -> effect = temp[2] & 0xF; // Effect number
			current -> eparm = temp[3];        // parameter
			offset += 5;
		}
	}

	// Load sample data
	for (count = 0; count < FM.numinsts; count++) {
		ERR = LoadSample(count, handle, 0);
		if (ERR) return ERR;
	}

	return 0;
}


/****************************************************************************
 *    Name : LoadSong                                                       *
 * Purpose : To perform autoextension detection, detect what type of tune it*
 *           actually is, and call the relevant loading procedure.          *
 *  Passed : char *path - the pathname of the file to load                  *
 *           char *filename - the actuall filename, could also contain a    *
 *                            path.                                         *
 *           UBYTE wild - whether wildcards are being used, if it is true,  *
 *                        then it will not try and autodetect extensions    *
 * Returns : 0 - if loading was unsucessfull                                *
 * Returns : 1 - if loading was sucessfull                                  *
 *  Locals : FILE *handle - file handle of the song to open                 *
 *           UBYTE temp[20] - temporary variable to hold magic letters      *
 *           UBYTE count - counter variable for various purposes            *
 *           UBYTE ERR - temporary to store error code before returning it  *
 *           char *newname - variable to hold filename, usually only stem so*
 *                           extensions can be added to it later.           *
 *           UWORD gushz[18] - Lookup table for GUS mixing rates, according *
 *                             to how many channels are found to be used    *
 *   Notes : HEh im not sure how I coded this but it seems to work          *
 ****************************************************************************/
UBYTE LoadSong(char *path, char *filename, UBYTE wild) {
	FILE *handle;
	char magic[4];
	UBYTE count, ERR=0;
	char *newname;
	UWORD gushz[18] = {
		41160,38587,36317,34300,32494,30870,29400,28063,26843,
		25725,24696,23746,22866,22050,21289,20580,19916,19293
	};

	if ((newname = (char *)calloc(255,1)) == NULL) return ERR_OUT_OF_MEM;

	strcpy(newname, filename);
	if (!wild) goto autodetect;

	// get filename if we are using wildcards
	for (count=strlen(path); count>0; count--) {
		if (path[count]=='\\') {
			strncpy(newname, path, count+1);
			newname[count+1]=0;
			strcat(newname, filename);
			if ((handle = fopen(newname, "rb")) == NULL) {
				free(newname);
				return ERR_NO_FILE;
			}
			goto foundext;
		}
	}

autodetect:
	// else cut up the filename and try to find an extension for it
	if (strrchr(filename, '\\')) strcpy(newname, strrchr(filename, '\\')+1);
	if (strchr(newname, '.') == NULL) {
		strcpy(newname, filename);
		strcat(newname,".mod");
		if ((handle = fopen(newname, "rb")) != NULL) goto foundext;
		strcpy(newname, filename);
		strcat(newname,".mtm");
		if ((handle = fopen(newname, "rb")) != NULL) goto foundext;
		strcpy(newname, filename);
		strcat(newname,".s3m");
		if ((handle = fopen(newname, "rb")) != NULL) goto foundext;
		strcpy(newname, filename);
		strcat(newname,".669");
		if ((handle = fopen(newname, "rb")) != NULL) goto foundext;
		strcpy(newname, filename);
		strcat(newname,".far");
		if ((handle = fopen(newname, "rb")) != NULL) goto foundext;

		free(newname);
		return ERR_NO_FILE;
	}
	else if ((handle = fopen(filename, "rb")) == NULL) {
		free(newname);
		return ERR_NO_FILE;
	}
foundext:
	free(newname);
	fseek(handle, 0, SEEK_END);
	filelen = ftell(handle);

	// clear name and type strings to 0's
	memset(FM.name, 0, 28);
	memset(FM.type, 0, 8);

	// VERIFICATION (Start looking at ID strings)

	// skip to offset 1080
	fseek(handle, 1080, SEEK_SET);
	fread(magic, 4, 1, handle);

	// 4 channel type .MOD
	if (!strncmp(magic,"M.K.",4) ||
		!strncmp(magic,"FLT4",4) ||
		!strncmp(magic,"M!K!",4)) {
		FM.channels=4;
		strncpy(FM.type, magic, 4);
		ERR=LoadMOD(handle);
	}
	// Fast Tracker .MOD (1 to 9 channels)
	else if (!strncmp(magic+1,"CHN",3)) {
		FM.channels = magic[0] - '0';
		if (FM.channels < 1 || FM.channels > 9) ERR = ERR_UNKNOWN_FORMAT;
		else {
			strncpy(FM.type, magic, 4);
			ERR=LoadMOD(handle);
		}
	}
	// Fast Tracker 2 / TakeTracker .MOD (10 channels and more)
	else if (!strncmp(magic+2,"CH",2)) {
		FM.channels = ((magic[0] - '0') * 10) + (magic[1] - '0');
		strncpy(FM.type, magic, 2);
		strncpy(FM.type+2, "CH",2);
		ERR = LoadMOD(handle);
	}
	// Now read from offset 0
	else {
		fseek(handle, 0L, SEEK_SET);
		fread(magic, 4, 1, handle);

		// MTM
		if (!strncmp(magic, "MTM" ,3)) {
			strcpy(FM.type, "MTM");
			ERR = LoadMTM(handle);
		}
		// 669
		else if (!strncmp(magic, "if"  ,2)) {
			strcpy(FM.type, "669");
			ERR = Load669(handle);
		}
		// Extended 669
		else if (!strncmp(magic, "JN"  ,2)) {
			strcpy(FM.type, "e669");
			ERR = Load669(handle);
		}
		// Farandole Tracker
		else if (!strncmp(magic, "FAR",4)) {
			strcpy(FM.type, "FAR");
			ERR = LoadFAR(handle);
		}
		// Now try at offset 02Ch
		else {
			fseek(handle, 0x2C, SEEK_SET);
			fread(magic, 4, 1, handle);

			// S3M
			if (!strncmp(magic, "SCRM",4)) {
				strcpy(FM.type, "S3M");
				ERR = LoadS3M(handle);}
		}
	}

	fclose(handle);

	// if any loaders returned an error, it will be picked up here.
	if (ERR) return ERR;

	// if no type was stored in the type string, then return an error
	if (FM.type[0]==0) return ERR_UNKNOWN_FORMAT;

	// ok we have channels, now get our gus rate divisor
	if (FM.channels > 14) divisor = gushz[FM.channels-15];
	else divisor = 44100;

	return 0;	// no errors!
}

