#include <dos.h>
#include "fmod.h"

#define ADDR_HIGH(x) ((unsigned int)((unsigned int)((x>>7L)&0x1fffL)))
#define ADDR_LOW(x)  ((unsigned int)((unsigned int)((x&0x7fL)<<9L)))

// UltraSound Ports
#define StatusPort              0x6
#define SelectVoice             0x102
#define CommandPort             0x103
#define DataLowPort             0x104
#define DataHighPort            0x105
#define DRAMIOPort              0x107

// UltraSound Commands

#define SetVoiceFreq            0x01
#define VolRampRate             0x06
#define VolRampStart            0x07
#define VolRampEnd              0x08
#define SetVolume               0x09
#define SampleStartLo           0x0A
#define SampleStartHi           0x0B
#define VoiceBalance            0x0C
#define SetVolumeCtrl			0x0D
#define VoicesActive            0x0E
#define DMACtrl                 0x41
#define DRAMAddrLo              0x43
#define DRAMAddrHi              0x44
#define TimerCtrl               0x45
#define SampleCtrl              0x49
#define Initialize              0x4C
#define GetVolumeCtrl			0x8D

UWORD Base;
UDWORD gusdram;
UWORD GUSvol[65] = {
	0x1500,
	0xA0DE,0xAB52,0xB2BD,0xB87E,0xBD31,0xC12B,0xC49C,0xC7A5,
	0xCA5D,0xCCD2,0xCF10,0xD120,0xD309,0xD4D1,0xD67B,0xD80B,
	0xD984,0xDAE9,0xDC3B,0xDD7D,0xDEB0,0xDFD6,0xE0EF,0xE1FC,
	0xE2FF,0xE3F8,0xE4E8,0xE5D0,0xE6AF,0xE788,0xE859,0xE924,
	0xE9E9,0xEAA9,0xEB63,0xEC18,0xECC8,0xED73,0xEE1A,0xEEBD,
	0xEF5C,0xEFF7,0xF08F,0xF123,0xF1B5,0xF242,0xF2CD,0xF356,
	0xF3DB,0xF45E,0xF4DE,0xF55B,0xF5D7,0xF650,0xF6C7,0xF73C,
	0xF7AE,0xF81F,0xF88E,0xF8FB,0xF967,0xF9D0,0xFA38,0xFA9E
};


/****************************************************************************
 *    Name : GUSDELAY 													    *
 * Purpose : Macro to wait a little bit          							*
 *  Passed : -                                                              *
 * Returns : -                                                              *
 ****************************************************************************/
#define GUSDELAY inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort); \
				 inp(Base+DRAMIOPort);


/****************************************************************************
 *    Name : GUSPeek                                                        *
 * Purpose : Gets the value of the byte at the specified GUS DRAM location  *
 *  Passed : UDWORD Loc - The point to peek at                              *
 * Returns : The byte stored at the specified position.                     *
 ****************************************************************************/
UBYTE GUSPeek(UDWORD Loc) {
	outp(Base+CommandPort, DRAMAddrLo);
	outpw(Base+DataLowPort, Loc);

	outp(Base+CommandPort, DRAMAddrHi);
	outp(Base+DataHighPort, (Loc>>16)&0xff);

	return inp(Base+DRAMIOPort);
}


/****************************************************************************
 *    Name : GUSPoke                                                        *
 * Purpose : Pokes 1 byte, specified in the parameter, to a position also   *
 *           specified in the GUS DRAM                                      *
 *  Passed : UDWORD Loc - The point to poke the particular byte             *
 *           UBYTE B - The byte to poke                                     *
 * Returns : -                                                              *
 ****************************************************************************/
void GUSPoke(UDWORD Loc, UBYTE B) {
	outp(Base+CommandPort, DRAMAddrLo);
	outpw(Base+DataLowPort, Loc);

	outp(Base+CommandPort, DRAMAddrHi);
	outp(Base+DataHighPort, (Loc>>16)&0xff);

	outp(Base+DRAMIOPort, B);
}


/****************************************************************************
 *    Name : GUSUpload   													*
 * Purpose : To upload a chunk of data to the GUS quickly, as long as it is	*
 *           smaller than 64kb.                                             *
 *  Passed : UDWORD Loc - the position in the GUS DRAM to start uploading   *
 *                        the data too.                                     *
 *           char *buffer - pointer to sample data.                         *
 *           UWORD buflength - 16bit size variable, containing the number of*
 *                             bytes to upload.                             *
 *           UBYTE xorval - value to XOR the sample data by, usually for    *
 *                          converting between signed and unsigned data     *
 * Returns : -                                                              *
 *   Notes : 3 seconds faster on when.s3m than a normal "for loop".         *
 *           translated directly from the .ASM version, and just as fast.   *
 ****************************************************************************/
void GUSUpload(UDWORD Loc, char *buffer, UWORD buflength, UBYTE xorval) {
	register UWORD si, bx=0;
	register UDWORD di;

	si = ((Loc >> 16) & 0xFF);
	di = (UWORD)Loc;

	outp(Base+CommandPort, DRAMAddrHi);
	outp(Base+DataHighPort, si);
MainLoop:
	outp(Base+CommandPort, DRAMAddrLo);
	outpw(Base+DataLowPort, (UWORD)di);
	outp(Base+DRAMIOPort, *(buffer+bx)+xorval);
	bx++;
	di++;
	if (di <= 65535) goto DoLoop;

	di=0;
	si++;
	outp(Base+CommandPort, DRAMAddrHi);
	outp(Base+DataHighPort, si);
DoLoop:
	if (buflength == 0) return;
	buflength --;
	goto MainLoop;
}


/****************************************************************************
 *    Name : GUSReset														*
 * Purpose : To reset the gus and tell it the number of voices to use       *
 *  Passed : UBYTE NumVoices - The number of voices to initialize with      *
 * Returns : -                                                              *
 ****************************************************************************/
void GUSReset(UBYTE NumVoices) {
	register UBYTE count;

	// If it is less than 14 voices then force it to be 14 voices.
	if (NumVoices < 14) NumVoices = 14;

	// disable line out
	outp(Base, 2);

	// Poke a 0 to dram position 0
	GUSPoke(0,0);

	// clear end bit - master reset
	outp(Base+CommandPort, Initialize);
	outp(Base+DataHighPort, 0);

	for (count=0; count<10; count++) GUSDELAY

	// now set end bit - master run
	outp(Base+CommandPort, Initialize);
	outp(Base+DataHighPort, 1);

	// Reset the MIDI IRQ port
	outp(Base+0x100, 3);

	for (count=0; count<10; count++) GUSDELAY

	// reset
	outp(Base+0x100, 0);

	// CLEAR INTERRUPTS

	outp(Base+CommandPort, DMACtrl);
	outp(Base+DataHighPort, 0);

	outp(Base+CommandPort, TimerCtrl);
	outp(Base+DataHighPort, 0);

	outp(Base+CommandPort, SampleCtrl);
	outp(Base+DataHighPort, 0);

	// SET NUMBER OF VOICES
	outp(Base+CommandPort, VoicesActive);
	outp(Base+DataHighPort, ((NumVoices-1) | 0xC0));

	// CLEAR IRQ's - reading status port clears them
	inp(Base+StatusPort);

	outp(Base+CommandPort, DMACtrl);
	inp(Base+DataHighPort);

	outp(Base+CommandPort, 0x49);
	inp(Base+DataHighPort);

	outp(Base+CommandPort, 0x8F);
	inp(Base+DataHighPort);

	for (count=0; count<32; count++) {
		outp(Base+SelectVoice, count);
		// stop voice
		outp(Base+CommandPort, 0);
		outp(Base+DataHighPort, 3);
		// stop ramp
		outp(Base+CommandPort, 0xD);
		outp(Base+DataHighPort, 3);

		GUSDELAY
	}

	// read irq port
	inp(Base+6);

	outp(Base+CommandPort, DMACtrl);
	inp(Base+DataHighPort);

	outp(Base+CommandPort, SampleCtrl);
	inp(Base+DataHighPort);

	outp(Base+CommandPort, 0x8F);
	inp(Base+DataHighPort);

	// RESET AND ENABLE DACS
	outp(Base+CommandPort, Initialize);
	outp(Base+DataHighPort, 7);

	// enable line out
	outp(Base, 0);
}


/****************************************************************************
 *    Name : GUSFindMem                                 					*
 * Purpose : To return the size of the GUS's DRAM banks in bytes            *
 *  Passed : -                                                              *
 * Returns : The size of the GUS DRAM.                                      *
 ****************************************************************************/
UDWORD GUSFindMem() {
	// First reset the GUS to 14 channels.  The dram gets corrupted sometimes
	// if you dont, ie if you ran ST3, then fmod.  Well it doesnt now..
	// 14 channels is used here, but dont worry as the GUS will be reset
	// again later to the correct number of channels in fmod.cpp
	// (with the call to GUSReset(FM.channels))
	GUSReset(14);
	outp(Base, 2);


	GUSPoke(0x40000, 0xAA);
	if (GUSPeek(0x40000) != 0xAA) return 0x3FFFF;

	GUSPoke(0x80000, 0xAA);
	if (GUSPeek(0x80000) != 0xAA) return 0x8FFFF;

	GUSPoke(0xC0000, 0xAA);
	if (GUSPeek(0xC0000) != 0xAA) return 0xCFFFF;
	else return 0xFFFFF;
}


/****************************************************************************
 *    Name : GUSSetVolume													*
 * Purpose : To set the volume of a voice Linearly from 0-64                *
 *  Passed : UBYTE Voice - the voice to set the volume for                  *
 *           UBYTE Volume - the volume to set the voice too, from 0-64      *
 * Returns : -                                                              *
 *   Notes : This function is passed a MOD type volume from 0-64, so it     *
 *           looks up the volume table before it sets.  It also controls    *
 *           muting, and if a channel is muted it will set the volume to 0  *
 ****************************************************************************/
void GUSSetVolume(UBYTE Voice, UBYTE Volume) {
	if (mute[Voice]) Volume=0;

	outp(Base+SelectVoice, Voice);
	outp(Base+CommandPort, SetVolume);

	outpw(Base+DataLowPort, GUSvol[Volume]);
	GUSDELAY
	outpw(Base+DataLowPort, GUSvol[Volume]);
}


/****************************************************************************
 *    Name : GUSSetBalance  												*
 * Purpose : To position the GUS Voice from left to right according to the  *
 *           parameter. 													*
 *  Passed : UBYTE Voice - the voice to set the balance for                 *
 *           UBYTE Balance - the position to set the voice - left or right. *
 * Returns : -                                                              *
 *   Notes : 'Balance' is a 0-255 value, but as the GUS only allows 0-15    *
 *           position panning, then it is shifted right 4 bits.             *
 ****************************************************************************/
void GUSSetBalance(UBYTE Voice, UBYTE Balance) {
	outp(Base+SelectVoice, Voice);

	outp(Base+CommandPort, VoiceBalance);
	outp(Base+DataHighPort, Balance >> 4);
}


/****************************************************************************
 *    Name : GUSSetFreq                                 					*
 * Purpose : To set the frequency of a voice, using GUS frequencies only 	*
 *  Passed : UBYTE Voice - the voice to set the frequency for               *
 *           UWORD Freq - the GUS value to set the frequency too.           *
 * Returns : -                                                              *
 *   Notes : The frequency here is a GUS value.  It was converted from an   *
 *           amigaval->hz->GUSfreq in PLAYER.CPP in function GUSfreq        *
 ****************************************************************************/
void GUSSetFreq(UBYTE Voice, UWORD Freq) {
	outp(Base+SelectVoice, Voice);

	outp(Base+CommandPort, SetVoiceFreq);
	outpw(Base+DataLowPort, Freq);
}


/****************************************************************************
 *    Name : GUSStopVoice													*
 * Purpose : Stops a voice playing dead in it's tracks						*
 *  Passed : UBYTE Voice - the voice to stop                                *
 * Returns : -                                                              *
 *  Locals : UBYTE x - a temporary value holding result of an inport.       *
 *   Notes : Thanks to Brad Thomas for finding the cause of a bug and       *
 *           narrowing it down to this function                             *
 ****************************************************************************/
void GUSStopVoice(UBYTE Voice) {
	UBYTE x;

	outp(Base+SelectVoice, Voice);

	outp(Base+CommandPort, 0x80);
	x = inp(Base+DataHighPort); // bh

	outp(Base+CommandPort, 0);
	outp(Base+DataHighPort, (x | 3));
	GUSDELAY
	outp(Base+CommandPort, 0);
	outp(Base+DataHighPort, ((x & 0xDF) | 3));
}


/****************************************************************************
 *    Name : GUSRamp                                                        *
 * Purpose : To ramp a voice's volume up or down quickly, from the current  *
 *           volume to the desired volume                                   *
 *  Passed : UBYTE mode - 0x40 to ramp down, or 0 to ramp up.               *
 *           UWORD ramppoint - volume to ramp too.                          *
 * Returns : -                                                              *
 *   Notes : Ramping is not implemented due to my laziness.  This function  *
 *           works I think, but I couldnt get it to remove clicks, most     *
 *           likely becuase I didnt wait for the ramps to end (?)           *
 ****************************************************************************
void GUSRamp(UBYTE mode, UWORD ramppoint) {
	UBYTE vmode;

	// don't let bad bits through ...
	mode &= ~(0x80|0x4|0x2|0x1);

	// Stop any previous ramps
	outp(Base+CommandPort,GetVolumeCtrl);
	vmode = inp(Base+DataHighPort);			// keep any previous bits...
	vmode |= (0x1|0x2);						// ... but add the stop ramp bits
	outp(Base+CommandPort,SetVolumeCtrl);
	outp(Base+DataHighPort,vmode);
	GUSDELAY();
	outp(Base+DataHighPort,vmode);

	// set ramp rate
	outp(Base+CommandPort,VolRampRate);
	outp(Base+DataHighPort, 0x3F);			// 3F = fastest ramp possible
	// set ramp start
	outp(Base+CommandPort,VolRampStart);
	outp(Base+DataHighPort, 0);				// silence
	// set ramp end
	outp(Base+CommandPort,VolRampEnd);
	outp(Base+DataHighPort,(UBYTE)ramppoint>>4);

	// set volume to start of ramp
	outp(Base+CommandPort,SetVolume);
	outpw(Base+DataLowPort,ramppoint<<4);

	// Start the ramp
	outp(Base+CommandPort, SetVolumeCtrl);
	outp(Base+DataHighPort,mode);
	GUSDELAY();
	outp(Base+DataHighPort,mode);
}
*/


/****************************************************************************
 *    Name : GUSPlayVoice													*
 * Purpose : To play a part of the GUS DRAM on a certain voice				*
 *  Passed : UBYTE Voice - the voice to play the sound through              *
 *           UBYTE Mode - the mode to play the sound as, ie. looping/16bit/.*
 *			 UDWORD begin - the start of the sample                         *
 *           UDWORD start - this is actually the beginning of a sample loop *
 *                          and if it is not a looping sample then begin and*
 *                          start should be the same.                       *
 *           UDWORD end - the end of the sample
 * Returns : -                                                              *
 ****************************************************************************/
void GUSPlayVoice(UBYTE Voice,UBYTE Mode,UDWORD begin,UDWORD start,UDWORD end) {

	GUSStopVoice(Voice);

	outp(Base+CommandPort, 0x0b);
	outpw(Base+DataLowPort, ADDR_LOW(begin));
	outp(Base+CommandPort, 0x0a);
	outpw(Base+DataLowPort, ADDR_HIGH(begin));

	outp(Base+CommandPort, 0x03);
	outpw(Base+DataLowPort, ADDR_LOW(start));
	outp(Base+CommandPort, 0x02);
	outpw(Base+DataLowPort, ADDR_HIGH(start));

	outp(Base+CommandPort, 0x05);
	outpw(Base+DataLowPort, ADDR_LOW(end));
	outp(Base+CommandPort, 0x04);
	outpw(Base+DataLowPort, ADDR_HIGH(end));

	outp(Base+CommandPort, 0);
	outp(Base+DataHighPort, Mode);
}
