// Panard-Vision
// 3D real time engine
// (C) 1997, Olivier Brunet
// Email: bruneto@efrei.fr
//
//
// Before using this library consult the LICENSE file


#ifndef _PVISION_H_
#define _PVISION_H_

#define __PV_VERSION__ 95         // Under 100 means, huuuu beta ? :)

//#include <unistd.h>             // NULL, SEEK_SET ... on SunOS
#include "config.h"


// Assembly code & HLine routines calling convention is __cdecl
#ifdef  __GNUC__
#define __cdecl     // GCC doesn't know about __cdecl modifier
#endif

#ifndef inline
#define inline
#endif

#ifndef PI
#define PI 3.141592654
#endif

#ifndef max
#define max(a,b) (a>b?a:b)
#endif

#ifndef min
#define min(a,b) (a<b?a:b)
#endif

// Float MiniMax
#ifndef MAXFLOAT
#include <float.h>
#define MINFLOAT -FLT_MAX
#define MAXFLOAT FLT_MAX
#endif

// (not) Useful macros
#define ISFLAGSET(f,ftt) (f&ftt)
#define ADDFLAG(f,ftt) (f|=ftt)
#define REMOVEFLAG(f,ftt) (f&=~ftt)

// Nothing interesting here
extern char *PVISION_DATE;
extern char *PVISION_TIME;
extern char *PVISION_VERSION;

//------------------------------------------ Panard Vision working modes
// PV_Mode, PV_SetMode()

#define PVM_PALETIZED8      1
#define PVM_RGB16           2
#define PVM_RGB             4
#define PVM_SBUFFER         8
#define PVM_MIPMAPPING      16

/***************************************************************************
********************************* STRUCTS & FLAGS **************************
****************************************************************************/
//------------------------------------------ Basics

typedef struct _Quat
{
    float w,x,y,z;
} Quat;

typedef int PVFLAGS;                      // 32 bits

typedef char PVD8;
typedef unsigned char UPVD8;

typedef short PVD16;
typedef unsigned short UPVD16;

typedef int PVD32;
typedef unsigned int UPVD32;

// Matrices
typedef float Mat3x3[3][3];
typedef float Mat4x3[4][3];             // La 4eme ligne est le produit des 2 colonnes
                                        // Pour faire les rotations en 6 muls
typedef struct _Point
{
    float xf,yf,zf,bf;
} Point;

typedef struct _RGB
{
   UPVD8 r,g,b;
} RGB;

typedef struct _RGBF
{
   float r,g,b;
} RGBF;

//------------------------------------------ Camera

typedef struct _Camera
{
    char *Name;
    float fieldofview;
    unsigned Height,Width;
    float roll,pitch,yaw;
    float xscreenscale,yscreenscale,fscale;
    Point pos;
    Point target;
    Mat3x3 Matrix;                // [0][x]=vright [1][x]=vup [2][x]=vpn
} PVCam;

//------------------------------------------ Light

typedef enum _PVLightType {PVL_DIRECTIONAL,PVL_PARALLEL,PVL_INFINITEPOINT,PVL_POINT,PVL_SPOT,PVL_USER_LIGHT} PVLightType;

#define LIGHT_NORMAL 0
#define LIGHT_FORGET 1                  // Switch off the light

typedef struct _TLight
{
    char *Name;
    PVLightType Type;
    PVFLAGS Flags;
    RGBF Color;
    float Intensity;
    float Range;
    float Attenuation0,Attenuation1,Attenuation2;
    float FallOff,Theta,Phi;
    float cosTh,cosPh;
    Point Position;
    Point Direction;
    Point Target;
    struct _TLight *Next;
    void *UserData;
} PVLight;

//--------------------------------------------- Material
// Material Flags
// Shading
#define NOTHING         0
#define FLAT            1
#define GOURAUD         2
#define PHONG           4
#define BUMP            8
#define U_PHONG         128
#define U_BUMP          256
#define SHADE_MASK      (1+2+4+8+128+256)

// rendering
#define MAPPING                     16
#define PERSPECTIVE_MAPPING         32
#define AMBIANT_MAPPING             64
#define RENDER_MASK              (16+32+64)

#define AUTOMATIC                   512           // Used with PERSPECTIVE_MAPPING
#define USER_FX                     1024          // Used to create special effects handled by user routines

// Texture Flags
#define TEXTURE_NONE            0
#define TEXTURE_PALETIZED8      1
#define TEXTURE_RGB             2
#define TEXTURE_MIPMAP          4

// added after materials compilation
#define TEXTURE_RGBDIRECT       8
#define TEXTUREQUANTIZED        64
#define TEXTURE_PROCESSED       256

typedef struct _Texture
{
    unsigned Width,Height;
    unsigned ShiftHeight,ShiftWidth;
    UPVD8 *Texture;
} PVTexture;

typedef struct _BumpInfo
{
    PVD8 nx,ny;
}PVBumpInfo;

typedef struct _BumpMap
{
    unsigned Width,Height;
    PVBumpInfo *Map;
} PVBumpMap;

typedef struct _Mat
{
    char *Name;
    PVFLAGS Type;

    unsigned char NbrMipMaps;
    PVFLAGS TextureFlags;
    RGB *Pal;
    PVTexture *Tex;

    PVTexture AuxiliaryTexture;

    PVBumpMap BumpMap;

    UPVD8  StartingColorIndex,EndingColorIndex;           // for pure colors

    RGBF Diffuse,Specular,Emissive;
    unsigned SpecularPower;

    UPVD8 *PaletteTranscodeTable;
    UPVD16 *RGB16TranscodeTable;

    RGBF Color;

    PVD32 UserData;                                       // Used to handle special FX by users routines

    struct _Mat *Next;
} PVMaterial;

//------------------------------------------ Mesh

#define MESH_FORGET          1          // Hide mesh

// Flags used to indicate wrapped mapping on faces
#define U_WRAP          (1<<30)
#define V_WRAP          (1<<31)

typedef struct _PVVertex3D
{
    float xf,yf,zf,bf;
    Point Normal;
    int Flags;
} PVVertex;

typedef struct _PVMapInfo
{
    float u,v;
    float AmbiantU,AmbiantV;
    float InvertZ;
} PVMapInfo;

typedef struct _PVScreenPoint
{
    float xf,yf;
} PVScreenPoint;

typedef struct _ShadingInfo
{
    unsigned Shade;
    RGBF Color;
} PVShadeInfo;

typedef struct _Box
{
    unsigned int NbrFaces;
    Point p0,t0;
    unsigned *FaceIndex;
    struct _Box *Next;
} PVBox;

typedef struct _PVFace
{
    char *Material;
    PVMaterial *MaterialInfo;
    void (PVAPI * Filler)(struct _PVFace *f);
    PVFLAGS Flags;
    unsigned int V[3];
    int ZAvg;
    float DistanceToCam;
    struct _PVMesh *Father;
    Point Normal;
} PVFace;

typedef struct _PVClippedFace
{
    unsigned int a,b;                      // Contient les points originaux
} PVClippedFace;

typedef struct _PVMesh
{
        char *Name;
        PVFLAGS Flags;
        unsigned int NbrFaces,NbrVertexes,NbrVisibles,NbrBoxes;
        Point Pivot;

        char *VertexVisible;
        unsigned nbrclipf,nbrclipv;          // Nombre de faces clipps
        PVClippedFace *ClippedOrg;           // pt originaux

        Point OldPos,OldPosv,Pos;
        Mat3x3 OldMatrix,OldMatrixv,Matrix;
        float ScaleX,ScaleY,ScaleZ;

        PVVertex *Vertex;
        PVFace *Face;

        PVMapInfo *Mapping;

        Point         *Rotated;
        PVScreenPoint  *Projected;

        PVFace          **Visible;

        PVShadeInfo *Shading;

        PVBox         *Box;

        void        *UserData;

        struct _PVMesh *Next;
} PVMesh;


//------------------------------------------------- World

typedef struct _PVWorld
{
    unsigned     NbrFaces,NbrVertexes,NbrVisibles;
    unsigned     NbrObjs;

    RGBF AmbiantLight;

    PVMaterial *Materials;

    PVFace    **Visible;

    PVMesh *Objs;
    PVLight *Lights;

    UPVD8 ReservedColors;
    RGB *Global256Palette;
} PVWorld;

/***************************************************************************
********************************* API **************************************
****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

// Math stuffs
void PVAPI SetupMatrix4x3(Mat4x3 Matrix,float AngleX,float AngleY,float AngleZ);
void PVAPI RotatePoint(Point *v,Mat4x3 m,Point *p);
void PVAPI RotatePoint3x3(Point *v,Mat3x3 *m,Point *p);
void PVAPI RotateInvertPoint(Point *v,Mat3x3 m,Point *p);
void PVAPI OrthonormalizeMatrix(Mat3x3 m);
void PVAPI OrientationToQuat(float x,float y,float z,float a,Quat *q);
void PVAPI QuatMul(Quat *q1,Quat *q2,Quat *q3);
void PVAPI QuatSLERP(Quat *q1,Quat *q2,float t,Quat *q);
void PVAPI QuatToMatrix(Quat *q,Mat3x3 Matrix);
void PVAPI MatrixToQuat(Mat3x3 Matrix,Quat *q);

// Misc
void PVAPI InitPVision(void);
int PVAPI PV_SetMode(PVFLAGS mode);
int PVAPI PV_SetClipLimit(unsigned int minx,unsigned int maxx, unsigned int miny, unsigned int maxy, unsigned int ScanLine,float centerx,float centery);
void PVAPI PV_GarbageCollector(PVWorld *w);
void PVAPI PV_AssignCam(PVCam *c);
PVCam * PVAPI PV_GetCurrentCam(void);
void PVAPI PV_Fatal(char *s,unsigned int t);      // Exit!

// PVWorld
int PVAPI PV_ComputeWorld(PVWorld *w);
PVWorld * PVAPI PV_CreateWorld(void);
void PVAPI PV_KillWorld(PVWorld *w);
int PVAPI SortWorld(PVWorld *w);
int PVAPI PV_AffineSort(PVWorld *w,float camx,float camy,float camz);
void PVAPI PV_RenderWorld(PVWorld *w,UPVD8 *Scr);
void PVAPI PV_AddMaterial(PVWorld *w,PVMaterial *m);
PVMaterial * PVAPI PV_GetMaterialPtrByName(char *name,PVWorld *z);
void PVAPI PV_AddMesh(PVWorld *w,PVMesh *no);
int PVAPI PV_RemoveMeshByPtr(PVWorld *,PVMesh *);
int PVAPI PV_RemoveMeshByName(PVWorld *,char *);
PVMesh * PVAPI PV_GetMeshPtr(PVWorld *w,char *s);
int PVAPI PV_WorldSetAmbiantLight(PVWorld *w,RGBF a);
void PVAPI PV_AddLight(PVWorld *w,PVLight *l);
int PVAPI PV_RemoveLightByName(PVWorld *w,char *s);
int PVAPI PV_RemoveLightByPtr(PVWorld *w,PVLight *o);
PVLight * PVAPI PV_GetLightPtr(PVWorld *w,char *s);
int PVAPI PV_CompileMeshes(PVWorld *z);
void PVAPI PV_ScaleWorld(PVWorld *w,float ax,float ay,float az);
int PVAPI PV_CompileMaterials(PVWorld *w,UPVD8 (*CalcFunc)(RGB *pal,PVMaterial *m,unsigned col,unsigned lightlevel,RGBF ambiant,unsigned reserved),UPVD16 (*CalcFunc16)(PVMaterial *m,unsigned col,unsigned lightlevel,RGBF ambiant));

// Camera
PVCam * PVAPI PV_CreateCam(char *n);
void PVAPI PV_RAZCam(PVCam *c);
void PVAPI PV_ComputeCam(PVCam *c);
void PVAPI PV_CamAhead(PVCam *c,float a);
int PVAPI PV_SetCamName(PVCam *c,char *n);
void PVAPI PV_SetCamFieldOfView(PVCam *c,float f);
void PVAPI PV_SetCamAngles(PVCam *c,float yax,float pitch,float roll);
void PVAPI PV_SetCamPos(PVCam *c,float x,float y, float z);
void PVAPI PV_SetCamTarget(PVCam *ca,float x,float y, float z);
void PVAPI PV_SetCamRoll(PVCam *c,float roll);

// Light
PVLight * PVAPI PV_CreateLight(PVLightType f,char *name);
int PVAPI PV_SetLightName(PVLight *l,char *n);
void PVAPI PV_KillLight(PVLight *l);
void PVAPI PV_SetLightDirection(PVLight *l,float x,float y,float z);
void PVAPI PV_SetLightPosition(PVLight *l,float x,float y,float z);
void PVAPI PV_SetLightTarget(PVLight *l,float x,float y,float z);
int PVAPI PV_SetLightIntensity(PVLight *l,float x);
void PVAPI PV_SetLightFlags(PVLight *l,PVFLAGS x);
void PVAPI PV_SetLightType(PVLight *l,PVLightType x);
void PVAPI PV_SetLightRange(PVLight *l,float x);
void PVAPI PV_SetLightAttenuation(PVLight *l,float a0,float a1,float a2);
void PVAPI PV_SetLightSpot(PVLight *l,float Theta,float Phi,float Falloff);
void PVAPI PV_RecomputeLight(void);
void PVAPI PV_NoRecomputeLight(void);
int PVAPI PV_LightRecomputeNeeded(void);

// Mesh
PVMesh * PVAPI PV_CreateMesh(void);
void PVAPI PV_KillMesh(PVMesh *no);
void PVAPI PV_MeshCompileVertexes(PVMesh *o);
void PVAPI PV_MeshTranslateVertexes(PVMesh *o,float x,float y,float z);
void PVAPI PV_MeshRotateVertexes(PVMesh *o,Mat3x3 m);
void PVAPI PV_MeshScaleVertexes(PVMesh *o,float a,float b, float c);
int PVAPI PV_MeshNormCalc(PVMesh *o);
void PVAPI PV_MeshBuildBoxes(PVMesh *o);
void PVAPI PV_MeshSetupMatrix(PVMesh *o,float AngleX,float AngleY,float AngleZ);
void PVAPI PV_MeshSetFlags(PVMesh *o,PVFLAGS f);
int PVAPI PV_MeshRAZShade(PVMesh *o);
void PVAPI PV_MeshProcessClippedVertex(PVMesh *o);

// Material
PVMaterial * PVAPI PV_CreateMaterial(char *n,PVFLAGS type,PVFLAGS flags,unsigned char nbrmipmaps);
void PVAPI PV_KillMaterial(PVMaterial *m);
void PVAPI PV_SetMaterialTextureFlags(PVMaterial *m,PVFLAGS f);
void PVAPI PV_SetMaterialType(PVMaterial *m,PVFLAGS f);
int PVAPI PV_SetMaterialTexture(PVMaterial *m,unsigned width,unsigned height,UPVD8 *texture,RGB *pal);
int PVAPI PV_SetMaterialAuxiliaryTexture(PVMaterial *m,unsigned width,unsigned height,UPVD8 *texture);
void PVAPI PV_SetMaterialPureColorsIndex(PVMaterial *m,UPVD8 start,UPVD8 end);
int PVAPI PV_SetMaterialLightInfo(PVMaterial *m,RGBF Emissive,RGBF Diffuse,RGBF Speculer,unsigned specularpower);
int PVAPI PV_SetMaterialPhongPalette(PVMaterial *t,RGBF ambiant);
int PVAPI PV_SetMaterialColor(PVMaterial *m,float r,float g,float b);

UPVD8 PVAPI PV_LookClosestColor(RGB *pal,RGB c,unsigned reserved);
int PVAPI PV_CreateMipMapTextures(PVMaterial *m,RGB *WorldPal,unsigned reserved);
int PVAPI PV_BuildBumpMap(PVMaterial *m,UPVD8 *Texture,unsigned width,unsigned height,float scale);
PVD32 PVAPI PV_MakeNormalizedColorRGB(RGB c);
RGB PVAPI PV_DecomposeNormalizedColor(UPVD32 c);
int PVAPI PV_SetRGBIndexingMode(UPVD8 RMaskSize,UPVD8 GMaskSize,UPVD8 BMaskSize,
                            UPVD8 RFieldP,UPVD8 GFieldP, UPVD8 BFieldP,UPVD8 AMaskS);
void PVAPI PV_SetGamma(float GammaR,float GammaG,float GammaB);
int PVAPI PV_CreateMaterialLightTranscodeTable16(PVMaterial *mat,unsigned TexNumber,UPVD16 (*CalcFunc)(PVMaterial *m,unsigned col,unsigned lightlevel,RGBF ambiant),RGBF ambiant);
int PVAPI PV_CreateMaterialLightTranscodeTable(PVMaterial *mat,unsigned TexNumber,RGB *pal,UPVD8 (*CalcFunc)(RGB *pal,PVMaterial *m,unsigned col,unsigned lightlevel,RGBF ambiant,unsigned reserved),RGBF ambiant,unsigned reserved);
int PVAPI PV_CreateGlobal256Palette(PVMaterial *z,UPVD8 ReservedColors,RGB **finalpal);
int PVAPI PV_ConvertMaterial(PVMaterial *m);


#ifdef __cplusplus
}
#endif


// User Defined Fonctions, default is provided
extern void (* PVAPI PV_UserCleanUp)(unsigned t);              // Called when a fatal error occur. T is the error code
extern void (* PVAPI PV_UserLight)(PVMesh *,PVLight *);        // Called to render an user light
extern void (* PVAPI PV_UserMeshCleanUp)(PVMesh *no);          // Called when a mesh is killed
extern int (* PVAPI PV_UserCompileMaterials)(PVMesh *m,PVMaterial *mat,unsigned FaceNumber); // Used to add special FX to PV
                                                                      // return 0 if no errors
                                                                      // return non zero if fatal error occured


// RGB Stuffs (use PV_SetRGBIndexingMode to set this up)
extern UPVD8 RMaskSize,GMaskSize,BMaskSize;
extern UPVD8 PixelSize;
extern UPVD32 RedMask,GreenMask,BlueMask;
extern unsigned RedFact,GreenFact,BlueFact;
extern UPVD8 RFieldPos,GFieldPos,BFieldPos;

// PV Working mode
extern PVFLAGS PV_Mode;                 // PV_SetMode()
extern float PV_BACKCLIPPLANE;          // -8000
extern float PV_FRONTCLIPPLANE;         // -0.5

/***************************************************************************
********************************* Error codes ******************************
****************************************************************************/

#define COOL                    0
#define FILE_IOERROR            1
#define ALREADY_ASSIGNED        2
#define NO_MEMORY               3
#define BIZAR_ERROR             5
#define ARG_INVALID             6
#define MESH_NOT_FOUND          7
#define MATERIAL_NOT_REGISTERED 8
#define INVALID_CAM             9
#define INVALID_TEXTURE         10
#define NO_MATERIALS            11
#define LIGHT_NOT_FOUND         20



/***************************************************************************
********************************* GENERIC FILLER API ***********************
****************************************************************************/

#define MAX_INCREMENT 10    // minus 1 ie 9

////////////////////////////////////////////////////////////////////////////

typedef struct _FillerUserFunct {
    int(*InitFunc1)(void);
    void(*InitFunc2)(void);
} FillerUserFunct;

typedef struct _Edge
{
    float Slope;
    int iSlope;

    float X;
    int iX;

    int ESlope;
    float FSlope,FractionnalTerm;

    float XPreStep,YPreStep;
} Edge;

////////////////////////////////////////////////////////////////////////////

extern void (__cdecl *HLineRoutine)(unsigned, unsigned);  // Scanline filler

extern unsigned GenFill_NbrInc,GenFill_NbrIncF;
extern unsigned GenFill_p1,GenFill_p2,GenFill_p3;
extern float GenFill_GradientX[MAX_INCREMENT];
extern float GenFill_GradientY[MAX_INCREMENT];
extern int GenFill_iGradientX[MAX_INCREMENT];
extern float GenFill_InitialValues[3][MAX_INCREMENT];
extern PVFace *GenFill_CurrentFace;

extern unsigned GenFill_CurrentLeftVal[MAX_INCREMENT];
extern float GenFill_CurrentLeftValF[MAX_INCREMENT];
extern unsigned GenFill_Modifier[MAX_INCREMENT];


extern unsigned GenFill_Height;
extern unsigned GenFill_CurrentY;

// Screen Layout, used by fillers
// Use PV_SetClipLimit
extern int ClipMinX;
extern int ClipMinY;
extern int ClipMaxX;
extern int ClipMaxY;

extern unsigned int ScanLength;     // Scanline length
extern UPVD8 *TriFill_BufOfs;       // Drawing Buffer

////////////////////////////////////////////////////////////////////////////

#ifdef __cplusplus
extern "C" {
#endif

void PVAPI GenFiller(PVFace *f,FillerUserFunct *FUT);
void PVAPI GenFill_Wrap(unsigned i);
unsigned PVAPI GetMipMap(unsigned nbrmips,float mu,float mv);   // Linear gradients
unsigned PVAPI GetMipMapIndex(float d);             // Perspective
unsigned PVAPI IsWrapped(unsigned i);

#ifdef __cplusplus
}
#endif

/***************************************************************************
********************** GENERIC PERSPECTIVE SCANLINE FILLER *****************
****************************************************************************/

#define AFFINE_LENGTH 16

extern int GenPHLine_NbrIncF2F;
extern float GradientXAff[MAX_INCREMENT];
extern void (__cdecl *SLineRoutine)(unsigned, unsigned);
extern void (__cdecl *HLPInitFunc)(void);

#ifdef __cplusplus
extern "C" {
#endif

void __cdecl GenPHLine(unsigned deb,unsigned fin);
void PVAPI PV_SetAutomaticPerspectiveRatio(float r);
float PVAPI PV_GetAutomaticPerspectiveRatio(float r);
int PVAPI PerspectiveNeeded(float z1,float z2,float z3);

#ifdef __cplusplus
}
#endif



#endif
