(*

                     Inertia Realtime 3D Rendering Engine
     Copyright (c) 1996, Alex Chalfin, Jeroen Bouwens. All Rights Reserved.
           DISTRIBUTION OF THIS SOURCE CODE IS STRICTLY PROHIBITED

*)

Program GMV;
{$N+,E+,X+}
{ Gfxfx Mesh Viewer }

Uses
  Crt,     { Included for keyboard reading      }
  Dos,     { Included for file finding routines }
  Mouse,   { Mouse reading routines             }
  Types,   { Inertia data types                 }
  Inertia, { Inertia 3d engine                  }
  Font13h, { Cool 7x3 graphics font             }
  Mode13h; { A simple mode 13h graphics unit    }

Const
  DataSet : String = 'Copprenv';

  MENUCOLOR  = 254;
  MENUCOLOR2 = 253;
  MENUTEXT   = 255;
  MAXFILES   = 256;
  F1 = #59; F2 = #60; F3 = #61; F4 = #62;
  F5 = #63; F6 = #64; F7 = #65; F8 = #66;
  F9 = #67; F10 = #68;
  HelpText : Array[0..15] of String = (
  'GFXFX Mesh Viewer  v0.99',
  ' ',
  'Keys while running:',
  ' F1 - Select AMBIENT shading',
  ' F2 - Select LAMBERT shading',
  ' F3 - Select GOURAUD shading',
  ' F4 - Select PHONG shading',
  ' F5 - Select SMOOTH surface model',
  ' F6 - Select TEXTURE surface model',
  ' F7 - Select REFLECTION surface model',
  ' T  - Toggle object transparency',
  ' H  - Toggle help screen',
  ' S  - Object stats',
  ' P  - PCX Screen dump',
  ' ESC - Quit',
  '');

Type
  pString = ^String;
  tFileList = Array[0..MAXFILES-1] of pString;
  fVector3d = Record
    x, y, z : Real;
  End;

Var
  MemStart, MaxMem : Longint;
  OldExitProc : Pointer;
  MenuLocX1, MenuLocY1, MenuLocX2, MenuLocY2 : Integer;
  ScrollButton1X1, ScrollButton1Y1, ScrollButton1X2, ScrollButton1Y2,
  ScrollButton2X1, ScrollButton2Y1, ScrollButton2X2, ScrollButton2Y2 : Integer;
  TopSelect, BottomSelect : Integer;
  CurrentSelection : Integer;
  FileList : tFileList;
  NumFiles : Integer;
  NewSelection : Boolean;
  View : ViewObject;
  Obj : VectorObject;
  Surface : SurfaceType;
  Shade : ShadeType;
  HighLight : Integer;
  Xaxis, Yaxis, Zaxis : FVector3d;

Procedure GetFileList;

Var
  DirInfo : SearchRec;
  i, j : Integer;
  temp : pString;

Begin
  NumFiles := 0;
  FindFirst('*.GVO', AnyFile, DirInfo);
  While (DOSError = 0) And (NumFiles < 256) do
    Begin
      New(FileList[NumFiles]);
      FileList[NumFiles]^ := DirInfo.Name;
      NumFiles := NumFiles + 1;
      FindNext(DirInfo);
    End;
  CurrentSelection := 0;
  TopSelect := 0;
  If (NumFiles > 9)
    Then BottomSelect := 9
    Else BottomSelect := (NumFiles - 1);

  { Sort the file entries }
  For i := 0 to (NumFiles - 1) do
    Begin
      For j := (NumFiles-2) downto i do
        Begin
          If (FileList[j]^ > FileList[j+1]^)
            Then Begin
              Temp := FileList[j];
              FileList[j] := FileList[j+1];
              FileList[j+1] := Temp;
            End;
        End;
    End;
End;

Procedure Quit(S : String);

Begin
  Writeln(S);
  Halt(0);
End;

Procedure Quit03(S : String);

Begin
  SetMode($03);
  Writeln(S);
  Halt(0);
End;

Procedure Initialize;

Var
  x, y, b : Integer;

Begin
  If (InitMouse <> -1) Then Quit('A mouse is required to run this program.');
  GetFileList;
  Writeln;
  If (NumFiles = 0) Then Quit('There are no GVO files to view!');
  If LoadIntensityTable(DataSet + '.It', 0) <> 0 Then Quit('Error Loading Intensity Table');
  If LoadTransparencyTable('Copprenv.tt', 0) <> 0 Then Quit('Error Loading Transparency Table');
  If LoadPCXTexture(1, DataSet + '.PCX', 0) <> 0 Then Quit('Error Loading Texture');
  If MakePhongMap(0) <> 0 Then Quit('Error making phong buffer');
  If InitView(View, 3000) <> 0 Then Quit('Error Initializing View');

  For x := 0 to 15 do
    Writeln(HelpText[x]);
  Writeln;
  Writeln('Use your mouse to rotate an object. Right click to bring up the object');
  Writeln('selection menu.');
  Writeln;
  Writeln('Press any key to continue.');
  While Not(KeyPressed) do;
  While KeyPressed do ReadKey;
  GetTexturePalette(1);
  PushLightSource(View, 0, 0, -10);
  Xaxis.x := 1; Xaxis.y := 0; Xaxis.z := 0;
  Yaxis.x := 0; Yaxis.y := 1; Yaxis.z := 0;
  Zaxis.x := 0; Zaxis.y := 0; Zaxis.z := 1;
End;

Procedure SolidRectangle(x1, y1, x2, y2 : Integer; Color : Byte);

Var
  vSeg, vOfs : Word;

Begin
  vSeg := Seg(VirtualPage^);
  vOfs := Ofs(VirtualPage^);
  Inc(vOfs, y1*320+x1);
  While (y1 < y2) do
    Begin
      FillChar(Mem[vSeg:vOfs], x2-x1, Color);
      Inc(vOfs, 320);
      Inc(y1);
    End;
End;

Procedure RenderMenu(x, y, mx, my : Integer);

Const
  MenuWidth = 13*6;
  dMaxFiles = 10;

Var
  MenuHeight : Integer;
  dMenuHeight : Integer;
  x2, y2 : Integer;
  b1, b2 : Integer;
  i, Range : Integer;


Begin
  { figure out how many files to display }
  If (NumFiles < dMaxFiles)
    Then MenuHeight := NumFiles
    Else MenuHeight := dMaxFiles;

  dMenuHeight := MenuHeight * 8;

  { calculate the x and y parameters of the menu}
  If (x + MenuWidth) > 319 Then x := x - ((x + MenuWidth) - 319);
  If (y + dMenuHeight) > 199 Then y := y - ((y + dMenuHeight) - 199);
  MenuLocX1 := x;
  MenuLocY1 := y;
  MenuLocX2 := x + MenuWidth;
  MenuLocY2 := y + dMenuHeight+1;

  { draw the background menu image }
  SolidRectangle(MenuLocX1, MenuLocY1, MenuLocX2, MenuLocY2, MENUCOLOR);

  { draw the scrollbar }
  ScrollButton1X1 := MenuLocX1+2;
  ScrollButton1Y1 := MenuLocY1+2;
  ScrollButton1X2 := MenuLocX1+6;
  ScrollButton1Y2 := MenuLocY1+5;
  ScrollButton2X1 := MenuLocX1+2;
  ScrollButton2Y1 := MenuLocY2-5;
  ScrollButton2X2 := MenuLocX1+5;
  ScrollButton2Y2 := MenuLocY2+2;

  SolidRectangle(MenuLocX1+1, MenuLocY1+1, MenuLocX1+6, MenuLocY2-1, MENUCOLOR2);
  SolidRectangle(MenuLocX1+2, MenuLocY1+2, MenuLocX1+5, MenuLocY1+5, MENUCOLOR);
  SolidRectangle(MenuLocX1+2, MenuLocY2-5, MenuLocX1+5, MenuLocY2-2, MENUCOLOR);

  Range := (ScrollButton2Y1 - 1) - (ScrollButton1Y2 + 1);
  b1 := (ScrollButton1Y2 + 1) + Round(Range * (TopSelect / NumFiles));
  b2 := (ScrollButton1Y2 + 1) + Round(Range * ((BottomSelect+1) / NumFiles));
  SolidRectangle(MenuLocX1+2, b1, MenuLocX1+5, b2, MenuColor);

  { draw the text }
  For i := TopSelect to BottomSelect do
    Begin
      If ((mx > ScrollButton2X2) And
          (mx <= MenuLocX2)) And
         ((my >= (MenuLocY1+1+((i-TopSelect)*8))) And (my <= (MenuLocY1+1+((i-TopSelect)*8)+7)))
           Then Begin
             SolidRectangle(ScrollButton2X2+2, MenuLocY1+1+((i-TopSelect)*8-1),
                          MenuLocX2-1, MenuLocY1+1+((i-TopSelect)*8+7), MENUCOLOR2);
             HighLight := i;
             WriteString(MenuLocX1+8, MenuLocY1+1+((i-TopSelect)*8), FileList[i]^, MENUTEXT, VirtualPage);
           End
         Else Begin
           WriteString(MenuLocX1+8, MenuLocY1+1+((i-TopSelect)*8), FileList[i]^, MENUCOLOR2, VirtualPage);
           WriteString(MenuLocX1+9, MenuLocY1+2+((i-TopSelect)*8), FileList[i]^, MENUTEXT, VirtualPage);
         End;

    End;
End;

Procedure SelectAction(x, y : Integer);

Begin
  If (x >= ScrollButton1X2) And (x <= MenuLocX2) And
     (y >= MenuLocY1) And (y <= MenuLocY2)
     Then Begin
       CurrentSelection := highlight;
       NewSelection := True;
     End;

  If (x >= ScrollButton1X1) And (x <= ScrollButton1X2) And
     (y >= ScrollButton1Y1) And (y <= ScrollButton1Y2)
     Then Begin
       If TopSelect > 0
         Then Begin
           Dec(TopSelect);
           Dec(BottomSelect);
         End;
     End;

  If (x >= ScrollButton2X1) And (x <= ScrollButton2X2) And
     (y >= ScrollButton2Y1) And (y <= ScrollButton2Y2)
     Then Begin
       If BottomSelect < (NumFiles - 1)
         Then Begin
           Inc(TopSelect);
           Inc(BottomSelect);
         End;
     End;
  WaitBClear;
End;

Procedure DisplayHelp;

Var
  i : Integer;

Begin
  For i := 0 to 15 do
    WriteString(0, i*8, HelpText[i], MENUTEXT, VirtualPage);
End;

{$F+}
Procedure FakeExtenderHalt;

Begin
  SetMode($03);
  Writeln('An Error has occured in this application.');
  Writeln(' Object:  ', FileList[CurrentSelection]^);
  Writeln(' Shade:   ', Ord(Shade));
  Writeln(' Surface: ', Ord(Surface));
  Writeln(' Orientation: ');
  Writeln('     |',Obj.Orientation[0]:7, Obj.Orientation[1]:7, Obj.Orientation[2]:7,'|');
  Writeln('     |',Obj.Orientation[3]:7, Obj.Orientation[4]:7, Obj.Orientation[5]:7,'|');
  Writeln('     |',Obj.Orientation[6]:7, Obj.Orientation[7]:7, Obj.Orientation[8]:7,'|');
  Writeln;
  Writeln('Sorry for the inconvience. Bug fixes are in the works.');
  Writeln;

  ExitProc := OldExitProc;
End;
{$F-}

Procedure AxisRotate(Var v : fVector3d; A : Real; x, y, z : Real);

Var
  cosa, sina : Real;
  dot, dx, dy, dz : Real;
  rx, ry, rz : Real;
  cx, cy, cz : Real;

Begin
  cosa := cos(a);
  sina := sin(a);
  dot := (x * v.x + y * v.y + z * v.z);
  dx := dot * x * (1 - cosa);
  dy := dot * y * (1 - cosa);
  dz := dot * z * (1 - cosa);
  rx := v.x * cosa;
  ry := v.y * cosa;
  rz := v.z * cosa;
  cx := (y * v.z - z * v.y)*sina;
  cy := (z * v.x - x * v.z)*sina;
  cz := (x * v.y - y * v.x)*sina;

  v.x := rx + dx + cx;
  v.y := ry + dy + cy;
  v.z := rz + dz + cz;
End;

Procedure FixAxis;

Var
  dot1, dot2 : Real;
  len : Real;

Begin
  dot1 := xaxis.x * yaxis.x + xaxis.y * yaxis.y + xaxis.z * yaxis.z;
  dot2 := xaxis.x * zaxis.x + xaxis.y * zaxis.y + xaxis.z * zaxis.z;
  xaxis.x := xaxis.x - dot1 * yaxis.x;
  xaxis.y := xaxis.y - dot1 * yaxis.y;
  xaxis.z := xaxis.z - dot1 * yaxis.z;
  xaxis.x := xaxis.x - dot2 * zaxis.x;
  xaxis.y := xaxis.y - dot2 * zaxis.y;
  xaxis.z := xaxis.z - dot2 * zaxis.z;
  len := 1/sqrt(sqr(xaxis.x) + sqr(xaxis.y) + sqr(xaxis.z));
  xaxis.x := xaxis.x*len;
  xaxis.y := xaxis.y*len;
  xaxis.z := xaxis.z*len;

  dot1 := xaxis.x * yaxis.x + xaxis.y * yaxis.y + xaxis.z * yaxis.z;
  dot2 := yaxis.x * zaxis.x + yaxis.y * zaxis.y + yaxis.z * zaxis.z;
  yaxis.x := yaxis.x - dot1 * xaxis.x;
  yaxis.y := yaxis.y - dot1 * xaxis.y;
  yaxis.z := yaxis.z - dot1 * xaxis.z;
  yaxis.x := yaxis.x - dot2 * zaxis.x;
  yaxis.y := yaxis.y - dot2 * zaxis.y;
  yaxis.z := yaxis.z - dot2 * zaxis.z;
  len := 1/sqrt(sqr(yaxis.x) + sqr(yaxis.y) + sqr(yaxis.z));
  yaxis.x := yaxis.x*len;
  yaxis.y := yaxis.y*len;
  yaxis.z := yaxis.z*len;

  zaxis.x := xaxis.y*yaxis.z - xaxis.z*yaxis.y;
  zaxis.y := xaxis.z*yaxis.x - xaxis.x*yaxis.z;
  zaxis.z := xaxis.x*yaxis.y - xaxis.y*yaxis.x;
End;

Procedure SetObjectMatrix(x, y, z : Integer);

Var
  phi, theta : Real;

Begin
  phi := (x And 1023)*2*Pi/1024;
  theta := (y And 1023)*2*Pi/1024;
  AxisRotate(XAxis, theta, xaxis.x, xaxis.y, xaxis.z);
  AxisRotate(YAxis, theta, xaxis.x, xaxis.y, xaxis.z);
  AxisRotate(ZAxis, theta, xaxis.x, xaxis.y, xaxis.z);

  AxisRotate(XAxis, phi, yaxis.x, yaxis.y, yaxis.z);
  AxisRotate(YAxis, phi, yaxis.x, yaxis.y, yaxis.z);
  AxisRotate(ZAxis, phi, yaxis.x, yaxis.y, yaxis.z);

  phi := (z And 1023)*2*Pi/1024;
  AxisRotate(XAxis, phi, zaxis.x, zaxis.y, zaxis.z);
  AxisRotate(YAxis, phi, zaxis.x, zaxis.y, zaxis.z);
  AxisRotate(ZAxis, phi, zaxis.x, zaxis.y, zaxis.z);

  FixAxis;

  Obj.Orientation[0] := Round(Xaxis.x * FIXONE);
  Obj.Orientation[3] := Round(Xaxis.y * FIXONE);
  Obj.Orientation[6] := Round(Xaxis.z * FIXONE);

  Obj.Orientation[1] := Round(Yaxis.x * FIXONE);
  Obj.Orientation[4] := Round(Yaxis.y * FIXONE);
  Obj.Orientation[7] := Round(Yaxis.z * FIXONE);

  Obj.Orientation[2] := Round(Zaxis.x * FIXONE);
  Obj.Orientation[5] := Round(Zaxis.y * FIXONE);
  Obj.Orientation[8] := Round(Zaxis.z * FIXONE);
End;

Procedure Action;

Var
  x, y, b : Integer;
  InMenu : Boolean;
  mLocx, mLocy : Integer;
  rotx, roty : Integer;
  dx, dy : Integer;
  retcode : Integer;
  ch : Char;
  Fini : Boolean;
  ViewHelp : Boolean;
  Trans : Boolean;
  MemMessage : Boolean;
  ZLoc : Longint;
  zscale : Real;
  stats : boolean;
  s1, s2, s3, s4, s5 : String;
  lastdraw : Word;
  fps : Real;
  st, et, frame : Longint;

Begin
  SetMode($13);
  SetPalette(GlobalPalette);
  SetRGB(MENUCOLOR, 45, 45, 45);
  SetRGB(MENUCOLOR2, 32, 32, 32);

  OldExitProc := ExitProc;
  ExitProc := @FakeExtenderHalt;

  InMenu := False;
  NewSelection := False;
  InitVectorObject(Obj);
  dx := 0;
  dy := 0;
  Fini := False;
  Surface := SMOOTH;
  Shade := AMBIENT;
  ViewHelp := False;
  Trans := False;
  MemMessage := False;
  stats := False;
  st := MemL[Seg0040:$6c];
  frame := 0;
  MaxMem := MemAvail;
  MemStart := MemAvail;
  Repeat
    If KeyPressed
      Then Begin
        st := MemL[Seg0040:$6c];
        frame := 0;
        ch := Readkey;
        If (ch = #0)
          Then Begin
            Case ReadKey of
              F1 :  Shade := Ambient;
              F2 :  Shade := Lambert;
              F3 :  Shade := Gouraud;
              F4 :  Shade := Phong;
              F5 :  Surface := Smooth;
              F6 :  Surface := Texture;
              F7 :  Surface := Reflection;
            End;
          End
          Else Begin
            Case UpCase(Ch) of
              #27 : Fini := True;
              'H' : ViewHelp := Not(ViewHelp);
              'T' : Trans := Not(Trans);
              'S' : Stats := Not(Stats);
              'P' : ScreenCapture;
            End;
          End;
        SetSurface(Obj, Surface);
        SetShading(Obj, Shade);
        SetCull(Obj, Not(Trans));
        SetObjTrans(Obj, Trans);
      End;

    AddtoRenderList(View, Obj);
    lastdraw := view.numinsortlist;
    Render(View, VirtualPage);

    ReadMickey(rotx, roty);

    If Not(InMenu) And LeftButtonPressed
      Then Begin
        Inc(ZLoc, Longint(Round(roty*zscale)));
        SetAbsoluteLocation(Obj, 0, 0, ZLoc);
        SetObjectMatrix(0, 0, rotx);
      End
      Else Begin
        SetObjectMatrix(rotx, roty, 0);
        ReadMouse(x, y, b);
      End;

    If InMenu Then RenderMenu(mLocx, mLocy, x, y);
    If ViewHelp Then DisplayHelp;
    If (b = 2) Then Begin WaitBClear; InMenu := Not(InMenu); mLocx := x; mLocy := y; End;
    If (InMenu And (b = 1)) Then SelectAction(x, y);
    If NewSelection
      Then Begin
        { Load a new object }
        FreeVectorObject(Obj);
        MemMessage := False;
        InMenu := Not(InMenu);
        NewSelection := False;
        RetCode := LoadGVO(Obj, FileList[CurrentSelection]^, 0);
        If (RetCode <> 0)
          Then Begin
            If RetCode <> I_MEMERR
              Then Quit03('Error loading GVO object! ' + FileList[CurrentSelection]^)
              Else MemMessage := True;
          End;
        SetBaseColor(Obj, 36);
        SetObjectTexture(Obj, 1);
        SetAbsoluteLocation(Obj, 0, 0, 5*Obj.Radius);
        SetMaxZ(Obj, 20*Obj.Radius);
        SetMinZ(Obj, Obj.Radius Div 4);
        ZLoc := 5*Obj.Radius;
        ZScale := Obj.Radius / 20;
        SetSurface(Obj, Surface);
        SetShading(Obj, Shade);
        SetCull(Obj, Not(Trans));
        SetObjTrans(Obj, Trans);
        st := MemL[Seg0040:$6c];
        frame := 0;
        MaxMem := MemAvail;
        SetObjectPhongMap(Obj, 0);
      End;

    If Stats
      Then Begin
        If Trans Then WriteString(95, 180, 'Transparent', MENUTEXT, VirtualPage);
        Case Shade of
          AMBIENT : WriteString(0, 180, 'Ambient', MENUTEXT, VirtualPage);
          Lambert : WriteString(0, 180, 'Lambert', MENUTEXT, VirtualPage);
          Gouraud : WriteString(0, 180, 'Gouraud', MENUTEXT, VirtualPage);
          Phong   : WriteString(0, 180, 'Phong', MENUTEXT, VirtualPage);
        End;

        Case Surface of
          SMOOTH     : WriteString(40, 180, 'Smooth', MENUTEXT, VirtualPage);
          TEXTURE    : WriteString(40, 180, 'Texture', MENUTEXT, VirtualPage);
          Reflection : WriteString(40, 180, 'Reflection', MENUTEXT, VirtualPage);
        End;

        et := MemL[Seg0040:$6c];
        if (et - st) = 0
          Then fps := 0
          Else fps := (frame * 18.2) / (et-st);
        Str(Obj.Num_verticies, s1);
        Str(Obj.Num_polygons, s2);
        Str(lastdraw, s3);
        Str(MemStart-MaxMem, s4);
        Str(Round(fps), s5);
        WriteString(0, 190, 'Verts: '+s1+' Faces: '+s2+' Drawn: '+s3+
                            ' Mem: '+s4+' FPS: '+s5, MENUTEXT, VirtualPage);
      End;

    If MemMessage Then WriteString(0, 190, 'Not enough memory for object!', MENUTEXT, VirtualPage);

    PutMouseCursor(x-4, y-4);
    frame := frame + 1;

    Flip;

  Until Fini;
  ExitProc := OldExitProc;
  SetMode($03);
End;


Begin
  Initialize;
  Action;
  Writeln('This program was written using the Inertia realtime 3d rendering');
  Writeln('system, a subset of the GFXFX3 project.');
  Writeln;
  Writeln('Program code: Alex Chalfin');
  Writeln('Engine code:  Alex Chalfin, Jeroen Bouwens');
  Writeln('GFXFX3 team:  Jeroen Bouwens, Alex Chalfin, Bas van Gaalen, Sean Palmer');
  Writeln;
End.