#ifndef __OPENGLINTERFACEC_H__
#define __OPENGLINTERFACEC_H__

#include "PajaTypes.h"
#include "DeviceInterfaceI.h"
#include "Vector2C.h"
#include "BBox2C.h"
#include <gl\gl.h>


namespace PajaSystem {

	//! The ID of the OpenGL interface, use this ID to request the interface.
	const PajaTypes::int32	INTERFACE_OPENGL = 0x1000;


	//! Device interface for OpenGL
	/*!	The purpose of the device interface for OpenGL is to setup
		the area where an effect can draw and to convert coordinates
		from screen (pixel) coordinates to the layout.
	*/
	class OpenGLInterfaceC : public DeviceInterfaceI
	{
	public:
		//! Defautl constructor (used internally).
		OpenGLInterfaceC( PajaTypes::int32 i32PosX, PajaTypes::int32 i32PosY, PajaTypes::int32 i32Width, PajaTypes::int32 i32Height, const PajaTypes::BBox2C& rViewport, const PajaTypes::BBox2C& rLayout );
		//! Default destructor.
		virtual ~OpenGLInterfaceC();

		//! Returns the type of the interface (INTERFACE_OPENGL).
		virtual PajaTypes::uint32	get_type() const;

		//! Sets perspective viewport and projection matrix.
		/*!	\param rBBox the bounding box of the effect.
			\param f32FOV FOV on x-axis.
			\param f32Aspect Aspect ration of the viewport.
			\param f32ZNear Near clipping plane.
			\param f32ZFar Far clipping plane.

			This method is a replacement for gluPerspective. Note that the FOV is different.
			Scrissor testing is enabled and the bounding box is used to define the scissor box.
		*/
		virtual void				set_perspective( const PajaTypes::BBox2C& rBBox, PajaTypes::float32 f32FOV, PajaTypes::float32 f32Aspect,
										PajaTypes::float32 f32ZNear, PajaTypes::float32 f32ZFar );

		//! Sets orthographic viewport and projection matrix.
		/*!	\param rBBox the bounding box of the effect.
			\param f32Width Width of the viewport.
			\param f32Height Height of the viewport.
			\param f32ZNear Near clipping plane.
			\param f32ZFar Far clipping plane.

			This method is a replacement for glOrtho.
			Scrissor testing is enabled and the bounding box is used to define the scissor box.
		*/
		virtual void				set_ortho( const PajaTypes::BBox2C& rBBox,
										PajaTypes::float32 f32Left, PajaTypes::float32 f32Right,
										PajaTypes::float32 f32Top, PajaTypes::float32 f32Bottom,
										PajaTypes::float32 f32ZNear = -1, PajaTypes::float32 f32ZFar = 1 );

		//! Converts positions in screen pixels to the layout coordinate system.
		virtual PajaTypes::Vector2C	client_to_layout( const PajaTypes::Vector2C& rVec );
		//! Converts positions in layout coordinate system uints to the screen pixels.
		virtual PajaTypes::Vector2C	layout_to_client( const PajaTypes::Vector2C& rVec );
		//! Converts delta values of screen pixels to the layout coordinate system.
		virtual PajaTypes::Vector2C	delta_client_to_layout( const PajaTypes::Vector2C& rVec );
		//! Converts delta values layout coordinate system uints to the screen pixels.
		virtual PajaTypes::Vector2C	delta_layout_to_client( const PajaTypes::Vector2C& rVec );

		//! Returns the viewport (visible are of the demo).
		virtual PajaTypes::BBox2C	get_viewport();
		//! Returns the width of the screen in pixels.
		virtual PajaTypes::int32	get_width();
		//! Returns the height of the screen in pixels.
		virtual PajaTypes::int32	get_height();


	private:
		//! Sets scrissoring to a spcified area.
		void						set_scissor( const PajaTypes::BBox2C& rBBox );

		PajaTypes::BBox2C	m_rViewport;			// The viewport size in paja (include pan and zoom).
		PajaTypes::BBox2C	m_rLayout;				// Layout bbox.
		PajaTypes::int32	m_i32Width;				// Actual screen size in pixels.
		PajaTypes::int32	m_i32Height;
		PajaTypes::int32	m_i32PosX;				// Position of lower left corner in pixels.
		PajaTypes::int32	m_i32PosY;
		PajaTypes::float64	m_f64LayoutAspectX;		// Conversion variables.
		PajaTypes::float64	m_f64LayoutAspectY;
		PajaTypes::float64	m_f64ClientAspectX;
		PajaTypes::float64	m_f64ClientAspectY;
	};

};

#endif // __OPENGLINTERFACEC_H__

