#ifndef __DEVICECONTEXTC_H__
#define __DEVICECONTEXTC_H__

#pragma warning( disable : 4786 )		// long names generated by STL

#include "PajaTypes.h"
#include "DeviceInterfaceI.h"
#include <vector>


namespace PajaSystem {

	//! Device context class.
	/*!	The purpose of the device context class is to gather all
		the device interfaces available to the effects. The interfaces
		may range from renderin APIs (such as OpenGL, or Direct3D) to
		music system APIs. The interface holds just enough information
		to use the API and some helper methods to help the use of
		the API under the editor. Device context simply holds these
		interfaces. These interfaces can be queried from the context. 

		This class is implemented by the system. 
	*/
	class DeviceContextC
	{
	public:
		//! Default constructor.
		/*!	Initialises the structures used by the device context. Used internally. */
		DeviceContextC();

		//! Default destructor.
		/*! The destructor. Releases all resgitered interfaces. Used internally. */
		virtual ~DeviceContextC();

		//! Query specified device interface.
		/*!	\param i32IFace ID of the interface to query. 
			\return Pointer to existing interface. If no interface was found the return value is NULL.

			This is the only method user needs to call from this class. It makes a query to
			the list inside the device context to see if the queried interface
			is available. If it is available pointer to the interface will be
			returned, otherwise the return value is NULL. 

			The IDs for the interfaces are defined in the header files of the interfaces. 

			Example:
			\code
			void
			TestEffectC::do_frame( DeviceContextC* pContext )
			{
				OpenGLInterfaceC* pInterface = (OpenGLInterfaceC*)pContext->query_interface( INTERFACE_OPENGL );
				if( !pInterface ) {
					// No interface available... bail out!
					return;
				}
				// do the effect...
			}
			\endcode
		*/
		virtual DeviceInterfaceI*	query_interface( PajaTypes::int32 i32Iface );

		//! Register device interface
		/*! The system uses this method internally to register device interfaces to the device context. */
		virtual void				register_interface( PajaTypes::int32 i32Iface, DeviceInterfaceI* pInterface );

	private:

		struct InterfaceItemS {
			PajaTypes::int32	m_i32Iface;
			DeviceInterfaceI*	m_pInterface;
		};

		std::vector<InterfaceItemS>	m_rInterfaces;
	};

}; // namespace

#endif // __DEVICECONTEXTC_H__