//	obj3d.cpp

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include "triangle.h"
#include "polygons.h"
#include "vectors.h"
#include "lightsrc.h"
#include "cameras.h"
#include "obj3d.h"


// fast float to int conversion
#ifndef DTOI_MAGIK
#define DTOI_MAGIK ((((65536.0 * 65536.0 * 16) + (65536.0 * 0.5)) * 65536.0))

int dtoi(double n) {
	double temp = DTOI_MAGIK + n;
	return ((*(int *)&temp) - 0x80000000);
}
#endif

#define INV65536 0.00001525878906


// radix sort
void radix (int byte, long N, TPolygon **source, TPolygon **dest) {
	long count[256];
	long index[256];
	memset (count, 0, sizeof (count));
	for ( int i=0; i<N; i++ ) count[((source[i]->zorder)>>(byte<<3))&0xff]++;
	index[0]=0;
	for ( i=1; i<256; i++ ) index[i]=index[i-1]+count[i-1];
	for ( i=0; i<N; i++ ) dest[index[((source[i]->zorder)>>(byte<<3))&0xff]++] = source[i];
}

void radixsort (TPolygon **source, long N) {
	TPolygon **temp = (TPolygon **)malloc(sizeof(TPolygon *) * N);
	radix (0, N, source, temp);
	radix (1, N, temp, source);
	radix (2, N, source, temp);
	radix (3, N, temp, source);
	free(temp);
}



// TPolygon implementation
TPolygon::TPolygon() {
	Vertex = (TVertex **)malloc(MaxPolyVertex * sizeof(TVertex *));
	NumVertex = 0;
	normal.SetP(0, 0, 0);
}

TPolygon::~TPolygon() {
	free(Vertex);
	NumVertex = 0;
}

void TPolygon::AddVertex(TVertex *v) {
	if (NumVertex < MaxPolyVertex) Vertex[NumVertex++] = v;
}

void TPolygon::Draw(unsigned int where) {
	StartCol = cstart;
	WidthCol = crange;
	TextureOffset = texture;
	GlobalStyle = style;
	if ((style % 10) == nc_flatshaded)
		StartCol = cstart + dtoi((DotProduct(normal, Light) + 1.0) * crange);
        for (int v = 2; v < NumVertex; v++)
                DrawTriangle(Vertex[0], Vertex[v - 1], Vertex[v], where);
}



void TPolygon::CalcNormal() {
	TVector a = (*Vertex[0]) - (*Vertex[1]);
	TVector b = (*Vertex[2]) - (*Vertex[1]);
	normal = b * a;
	normal.Normalize();
}

int TPolygon::Visible() {
	for (int i = 0; i < NumVertex; i++) if (Vertex[i]->tag) return 0;
	return (( Vertex[0]->vx * normal.x + Vertex[0]->vy * normal.y +
			  Vertex[0]->vz * normal.z) <= 0.0);
}                                                                           


void TPolygon::CalcZOrder() {
	double z = 0.0;
	for (int v = 0; v < NumVertex; v++) z += Vertex[v]->zorder;
	zorder = dtoi(2147483647.0 - z * 1048576.0 / NumVertex);
}


// TObject3D implementation

TObject3D::TObject3D() {
	Vertex = (TVertex **)malloc(MaxObjectVertex * sizeof(TVertex *));
	Poly = (TPolygon **)malloc(MaxObjectPoly * sizeof(TPolygon *));
	NumVertex = NumPolygons = 0;
	for (int v = 0; v < MaxObjectVertex; v++) Vertex[v] = NULL;
	for (int p = 0; p < MaxObjectPoly; p++) Poly[p] = NULL;
        origin.SetP(0, 0, 0);
}


TObject3D::~TObject3D() {
	for (int v = 0; v < NumVertex; v++) if (Vertex[v]) delete Vertex[v];
	for (int p = 0; p < NumPolygons; p++) if (Poly[p]) delete Poly[p];
	free(Vertex);
	free(Poly);
	NumVertex = 0;
	NumPolygons = 0;
}


void TObject3D::AddVertex(double nx, double ny, double nz) {
	if (NumVertex < MaxObjectVertex) 
		Vertex[NumVertex++] = new TVertex(nx, ny, nz);
}

void TObject3D::AddVertexUV(double nx, double ny, double nz, double nu, double nv) {
	if (NumVertex < MaxObjectVertex) {
		Vertex[NumVertex] = new TVertex(nx, ny, nz);
		Vertex[NumVertex]->u = nu;
		Vertex[NumVertex++]->v = nv;
	}
}

void TObject3D::AddPolygon(TPolygon *poly) {
	if ((poly) && (NumPolygons < MaxObjectPoly)) Poly[NumPolygons++] = poly;
}

void TObject3D::CalcNormals() {
	int v, p;
	for (p = 0; p < NumPolygons; p++) Poly[p]->CalcNormal();
	for (v = 0; v < NumVertex; v++) {
		Vertex[v]->normal.SetP(0, 0, 0);
		for (p = 0; p < NumPolygons; p++) 
			for (int n = 0; n < Poly[p]->NumVertex; n++)
				if (Vertex[v] == Poly[p]->Vertex[n])
					Vertex[v]->normal += Poly[p]->normal;
		Vertex[v]->normal.Normalize();
	}
}

void TObject3D::MapEnviroment() {
	for (int v = 0; v < NumVertex; v++) Vertex[v]->CalcEnvUV();
}

void TObject3D::Traslate(double nx, double ny, double nz) {
	for (int v = 0; v < NumVertex; v++) Vertex[v]->Traslate(nx, ny, nz);
}

void TObject3D::Rotate(int ax, int ay, int az) {
	for (int v = 0; v < NumVertex; v++) Vertex[v]->VRotate(ax, ay, az);
	for (int p = 0; p < NumPolygons; p++) Poly[p]->normal.Rotate(ax, ay, az);
}

void TObject3D::FlatRotate(int ax, int ay, int az) {
	for (int v = 0; v < NumVertex; v++) Vertex[v]->Rotate(ax, ay, az);
	for (int p = 0; p < NumPolygons; p++) Poly[p]->normal.Rotate(ax, ay, az);
}



void TObject3D::Scale(double sx, double sy, double sz) {
	for (int v = 0; v < NumVertex; v++) Vertex[v]->Scale(sx, sy, sz);
}

void TObject3D::Draw(unsigned int where) {
	TPolygon **visiblepoly;
	int n = 0;
	TransformView();
	visiblepoly = (TPolygon **)malloc(sizeof(TPolygon *) * NumPolygons);
	
	if ((Poly[0]->style % 10) == nc_gouraudshaded)
		for (int v = 0; v < NumVertex; v++)
			Vertex[v]->c = dtoi((DotProduct(Vertex[v]->normal, Light) + 1.0) * Poly[0]->crange) + Poly[0]->cstart;
			
	for (int p = 0; p < NumPolygons; p++) 
		if (Poly[p]->Visible()) {
			Poly[p]->CalcZOrder();
			visiblepoly[n++] = Poly[p];
		}
	if (n > 0) radixsort(visiblepoly, n);
//        for (p = 0; p < n; p++) visiblepoly[p]->Draw(where);
        for (p = 0; p < n; p++) DrawPolygon(visiblepoly[p], where);
	free(visiblepoly);
}


void TObject3D::TransformView() {
        TVector v(0, 0, 0);
	for (int i = 0; i < NumVertex; i++) {
		Vertex[i]->vx = v.x = Vertex[i]->x + origin.x - camera->location.x;
		Vertex[i]->vy = v.y = Vertex[i]->y + origin.y - camera->location.y;
		Vertex[i]->vz = v.z = Vertex[i]->z + origin.z - camera->location.z;
		v.Rotate(0, -camera->ay, 0);
		v.Rotate(-camera->ax, 0, -camera->az);
		if (v.z > 0.0) {
                        if (v.z < 1.0) v.z = 0.01;
			Vertex[i]->tag = 0;
			Vertex[i]->x2d = ScreenCenterX + dtoi(v.x * PerspectiveX / v.z);
                        Vertex[i]->y2d = ScreenCenterY - dtoi(v.y * PerspectiveY / v.z);
			Vertex[i]->zorder = v.z;
		}
		else Vertex[i]->tag = 1;
	}
}


void TObject3D::SphericalMap(double scale) {
	double phi, theta, xzdist;
	for (int i = 0; i < NumVertex; i++) {
		xzdist = sqrt(Vertex[i]->x * Vertex[i]->x +
		              Vertex[i]->z * Vertex[i]->z); 
		theta = atan2(Vertex[i]->x, Vertex[i]->z);
		phi = atan2(Vertex[i]->y, xzdist);
                Vertex[i]->u = scale * (theta + Pi) / Pi / 2.0;
                Vertex[i]->v = scale * (phi + Pi) / Pi / 2.0;
	}
}


void TObject3D::FlipFaces() {
        TVertex *temp;
        for (int i = 0; i < NumPolygons; i++) {
                temp = Poly[i]->Vertex[0];
                Poly[i]->Vertex[0] = Poly[i]->Vertex[2];
                Poly[i]->Vertex[2] = temp;
        }
}


void TObject3D::Centre() {
	TVector c(0, 0, 0);
	for (int i = 0; i < NumVertex; i++) c += *Vertex[i];
	c = c * (1.0 / NumVertex);
	Traslate(-c.x, -c.y, -c.z);
}

void TObject3D::FitSphere(double radius) {
	double max = 0;
	for (int i = 0; i < NumVertex; i++)
		if (Vertex[i]->Magnitude() > max) max = Vertex[i]->Magnitude();
	double s = radius / max;
	Scale(s, s, s);
}

void TObject3D::SetColor(int cstart, int crange) {
	for (int i = 0; i < NumPolygons; i++) {
		Poly[i]->cstart = cstart;
		Poly[i]->crange = crange;
	}
}

void TObject3D::SetStyle(int style) {
	for (int i = 0; i < NumPolygons; i++) Poly[i]->style = style;
}

void TObject3D::SetTexture(unsigned int texture) {
	for (int i = 0; i < NumPolygons; i++) Poly[i]->texture = texture;
}



int fgetint(FILE *file) {
	int b1 = (int)fgetc(file);
	int b2 = ((int)fgetc(file)) << 8;
	int b3 = ((int)fgetc(file)) << 16;
	int b4 = ((int)fgetc(file)) << 24;
	return b1 | b2 | b3 | b4;
}

void Load3DObject(char *filename, TObject3D *obj) {
	FILE *file;
	int i, j, nv, np, x, y, z, u, v;
	TPolygon *p;
	
	if ((file = fopen(filename, "rb")) == NULL) return;
	if ((fgetint(file) & 0xFFFF0000) != 0x1FAC0000) {
		fclose(file);
		return;
	}
	for (i = 0; i < obj->NumVertex; i++) 
		if (obj->Vertex[i]) { delete obj->Vertex[i]; obj->Vertex[i] = NULL; }
	for (i = 0; i < obj->NumPolygons; i++) 
		if (obj->Poly[i]) { delete obj->Poly[i]; obj->Poly[i] = NULL; }
	nv = fgetint(file);
	np = fgetint(file);
	for (i = 0; i < 5; i++) fgetint(file);
	for (i = 0; i < nv; i++) {
		x = fgetint(file);
		y = fgetint(file);
		z = fgetint(file);
		u = fgetint(file);
		v = fgetint(file);
		obj->AddVertexUV(x / 65536.0, y / 65536.0, z / 65536.0, u / 65536.0, v / 65536.0);
	}
	for (i = 0; i < np; i++) {
		p = new TPolygon();
		u = fgetint(file);
		for (j = 0; j < u; j++) {
			v = fgetint(file);
			if (obj->Vertex[v]) p->AddVertex(obj->Vertex[v]);
		}
		obj->AddPolygon(p);
	}
	fclose(file);
	obj->CalcNormals();
}

void Read3DObject(int *vkx, TObject3D *obj) {
	int i, nv, np, x, y, z, u, v;
        int *data = vkx;
	TPolygon *p;        
	
	for (i = 0; i < obj->NumVertex; i++) 
		if (obj->Vertex[i]) { delete obj->Vertex[i]; obj->Vertex[i] = NULL; }
	for (i = 0; i < obj->NumPolygons; i++) 
		if (obj->Poly[i]) { delete obj->Poly[i]; obj->Poly[i] = NULL; }
	obj->NumVertex = 0;
	obj->NumPolygons = 0;
        nv = *data++;
        np = *data++;
	for (i = 0; i < nv; i++) {
                x = *data++;
                y = *data++;
                z = *data++;
                u = *data++;
                v = *data++;
                obj->AddVertexUV(x * INV65536, y * INV65536, z * INV65536,
                                 u * INV65536, v * INV65536);
	}
	for (i = 0; i < np; i++) {
		p = new TPolygon();
                v = *data++;
                if (obj->Vertex[v]) p->AddVertex(obj->Vertex[v]); //else e(i, v, 0);
                v = *data++;
                if (obj->Vertex[v]) p->AddVertex(obj->Vertex[v]); //else e(i, v, 1);
                v = *data++;
                if (obj->Vertex[v]) p->AddVertex(obj->Vertex[v]); //else e(i, v, 2);
		obj->AddPolygon(p);
	}
	obj->CalcNormals();
}


void InitializeEngine() {
	CalcTrigTables();
	Light.SetP(0, 0, -1);
	camera = new TCamera();
}
