window.CanvasWrapper = (function() {
	'use strict';
	var _canvas, _config, _canvasInnerWrapper, _canvasOuterWrapper;

	var CanvasWrapper = {
		setup: function(canvas, config) {
			_canvas = canvas;
			_canvas.id = 'goo';
			_config = config || {};
		},

		add: function() {
			_canvas.tabIndex = 0;
			_canvasOuterWrapper = document.getElementById('canvas-outer');
			_canvasInnerWrapper = document.getElementById('canvas-inner');
			_canvasInnerWrapper.appendChild(_canvas);

			// Make sure the canvas is updated when the window is resized.
			window.addEventListener('resize', CanvasWrapper.resize);
			window.addEventListener('orientationchange', CanvasWrapper.resize);

			_canvas.addEventListener('mousedown', CanvasWrapper.uninterruptedPan, true);
			_canvas.addEventListener('mouseup', CanvasWrapper.allowSelection, true);

			function prevent(e) {
				e.preventDefault();
				e.stopPropagation();
			}

			_canvas.addEventListener('mousewheel', prevent, true);
			_canvas.addEventListener('DOMMouseScroll', prevent, true);

			CanvasWrapper.resize();
		},

		hide: function() {
			_canvas.style.visibility = 'hidden';
		},

		show: function() {
			_canvas.style.visibility = 'visible';
		},

		/**
		 * Resizes the camera according to the configuration included in the
		 * bundle.
		 */
		resize: function() {
			if (_config.mode === 'Resolution' && _config.resolution) {
				setSize(_config.resolution.width, _config.resolution.height);
			} else if (_config.mode == 'AspectRatio' && _config.aspect) {
				var aspectWidth = _config.aspect.width;
				var aspectHeight = _config.aspect.height;

				var ratio = aspectWidth / aspectHeight;
				var windowAspectRatio = _canvasOuterWrapper.offsetWidth / _canvasOuterWrapper.offsetHeight;

				var width, height;

				// Top/Bottom letterbox
				if (ratio > windowAspectRatio) {
					width = _canvasOuterWrapper.offsetWidth;
					height = width / ratio;
				// Left/Right letterbox
				} else {
					height = _canvasOuterWrapper.offsetHeight;
					width = height * ratio;
				}

				setSize(width, height);
			} else {
				setSize();
			}
		},

		uninterruptedPan: function() {
			var htmlEntities = document.querySelectorAll('.goo-entity');
			for (var i = 0; i < htmlEntities.length; i++) {
				htmlEntities[i].classList.add('no-pointer');
			}
		},

		allowSelection: function() {
			var htmlEntities = document.querySelectorAll('.goo-entity');
			for (var i = 0; i < htmlEntities.length; i++) {
				htmlEntities[i].classList.remove('no-pointer');
			}
		}
	};



	function setSize(width, height) {
		var classList = _canvasOuterWrapper.classList;

		if (width && height) {
			classList.add('custom');
			_canvasInnerWrapper.style.width = width + 'px';
			_canvasInnerWrapper.style.height = height + 'px';

			if (window.innerWidth > width) {
				classList.add('center-h');
			} else {
				classList.remove('center-h');
			}

			if (window.innerHeight > height) {
				classList.add('center-v');
			} else {
				classList.remove('center-v');
			}
		} else {
			classList.remove('custom');

			_canvasInnerWrapper.style.width = '';
			_canvasInnerWrapper.style.height = '';
		}
	};


	return CanvasWrapper;
})();

/* Webglsupport.js */

window.WebGLSupport = (function() {

	'use strict';

	var WebGLSupport = {
		ERRORS: {
			NO_ERROR: 'no error',
			NO_RENDERING_CONTEXT: 'noRenderingContext',
			WEBGL_DISABLED: 'webglDisabled'
		},

		BROWSERS: {
			CHROME: 'chrome',
			FIREFOX: 'firefox',
			OPERA: 'opera',
			IOS: 'ios',
			IE: 'ie',
			LUDEI: 'ludei',
			SAFARI: 'safari'
		},

		/*
		* Checks for a present webgl rendering context and if webgl is
		* disabled.
		*
		* The client's browser is checked and returned as well.
		*/
		check: function() {

			var result = {
				error: null,
				browser: null
			};

			result.browser = this.getBrowser();

			if (!window.WebGLRenderingContext) {
				result.error = this.ERRORS.NO_RENDERING_CONTEXT;
			} else if (isWebGLDisabled() === true) {
				result.error = this.ERRORS.WEBGL_DISABLED;
			} else {
				result.error = this.ERRORS.NO_ERROR;
			}

			return result;
		},

		getBrowser: function() {

			var iOS = /(iPad|iPhone|iPod)/g.test(navigator.userAgent);
			var isOpera = !!window.opera || navigator.userAgent.indexOf(' OPR/') >= 0;
			var isChrome = !!window.chrome;
			var isFirefox = typeof InstallTrigger !== 'undefined';
			var isSafari =  navigator.userAgent.indexOf('Safari') >= 0;
			var isIE = document.documentMode || false;
			var isCocoonJS = navigator.appName === "Ludei CocoonJS";

			if (!(isFirefox || isChrome)) {
				console.warn('Je vous conseil Chrome ou Firefox !.');
			}

			var browser = null;

			if (iOS === true) {
				browser = this.BROWSERS.IOS;
			} else if (isOpera === true) {
				browser = this.BROWSERS.OPERA;
			} else if (isChrome === true) {
				browser = this.BROWSERS.CHROME;
			} else if (isFirefox === true) {
				browser = this.BROWSERS.FIREFOX;
			} else if (isSafari === true) {
				browser = this.BROWSERS.SAFARI;
			} else if (isIE === true) {
				browser = this.BROWSERS.IE;
			} else if (isCocoonJS === true) {
				browser = this.BROWSERS.LUDEI;
			} else {
				console.warn('Navigateur impossible à déterminer...');
			}

			return browser;
		}

	};

	function isWebGLDisabled() {
		var gl = null;
		var canvas = document.createElement('canvas');

		try {
			gl = canvas.getContext('webgl');
		}
		catch (_error) {
			console.error(_error);
			gl = null;
		}

		if (gl === null) {
			try {
				gl = canvas.getContext('experimental-webgl');
			}
			catch (_error) {
				console.error(_error);
				gl = null;
			}
		}

		if (gl === null && window.WebGLRenderingContext) {
			return true;
		} else {
			return false;
		}
	};

	return WebGLSupport;
})();


/* GSRVMock.js */

window.gsrv = {
	track: function(){},
	exit: function(){}
}
