
#include "gl3w/include/GL3/gl3w.h"
#include <GL/glext.h>
#include <SDL/SDL.h>
#include <bass.h>

#include "Orion.hpp"
#include "Shader.hpp"
#include "BVH.hpp"

#include "materials.hpp"
#include "nodes.hpp"
#include "meshes.hpp"
#include "voxels.hpp"
#include "sky_mesh.hpp"
#include "markers.hpp"

#include <map>
#include <algorithm>
#include <stack>

#include "shader_data.hpp"

#include <IL/il.h>

#define RAYTRACE        1
#define DEVMODE         0
#define ENCODE_SHADERS  0
#define CAPTURE         0

#define SET_UNIFORM(p, n, c, ...) { static const GLint loc__=glGetUniformLocation(p.name,#n); assert(loc__>=0); c(loc__,__VA_ARGS__); }

static const double thumps[]={
   2*60+14.45735,
   2*60+16.07789,
   2*60+17.69104,
   2*60+18.07866,
   2*60+18.49578,
   2*60+18.88707,
   2*60+19.29683,
   2*60+19.48878,
   2*60+19.69181,
   2*60+19.90592,
   2*60+20.09787,
   2*60+20.19753,
   2*60+20.29351,
   2*60+20.38211,
   2*60+20.47807
   };
static const int num_thumps=sizeof(thumps)/sizeof(thumps[0]);

FILE* g_logfile = NULL;

namespace shaders
{
struct Symbol
{
   bool terminal;
   bool pl;
   unsigned char c;
   uint32 left;
   uint32 right;
   uint32 parent;
   uint32 f;
   uint32 uid;
   bool operator<(const Symbol& s) const { return f > s.f; }
   Symbol(){ c='\0'; f=0; left=0; right=0; parent=0; terminal=true;; pl=false; uid=0; }
};

Symbol root;
std::vector<Symbol> heap;
std::vector<Symbol> dict;
Symbol** uid_to_symbol;

static Symbol* findSymbol(uint32 uid)
{
   size_t sz=dict.size();
   for(size_t i=0;i<sz;++i)
   {
      if(dict[i].uid==uid)
         return &dict[i];
   }
   return NULL;
}
static int findSymbolIndex(uint32 uid)
{
   size_t sz=dict.size();
   for(size_t i=0;i<sz;++i)
   {
      if(dict[i].uid==uid)
         return i;
   }
   return -1;
}

static void initShaderSymbolDict()
{
   uint32 uid=0;

   heap.clear();
   dict.clear();

   for(int i=0;i<shader_symbol_count;++i)
   {
      Symbol s;
      s.uid=++uid;
      s.c=shader_symbols[i].character;
      s.f=shader_symbols[i].frequency;
      heap.push_back(s);
   }

   std::make_heap(heap.begin(),heap.end());

   while(!heap.empty())
   {
      Symbol s0=heap.front();
      dict.push_back(s0);
      std::pop_heap(heap.begin(),heap.end());
      heap.pop_back();
      if(heap.empty())
      {
         // this is the root
         root = s0;
         break;
      }
      Symbol s1=heap.front();
      dict.push_back(s1);
      std::pop_heap(heap.begin(),heap.end());
      heap.pop_back();

      Symbol ns;
      ns.uid=++uid;
      ns.terminal=false;
      ns.left=s0.uid;
      ns.right=s1.uid;
      ns.f=s0.f+s1.f;
      findSymbol(ns.left)->parent=ns.uid;
      findSymbol(ns.left)->pl=true;
      findSymbol(ns.right)->parent=ns.uid;
      findSymbol(ns.right)->pl=false;

      heap.push_back(ns);
      std::push_heap(heap.begin(),heap.end());
   }

   assert(!root.terminal);
   dict.push_back(root);

   uid_to_symbol=new Symbol*[dict.size()];
//   fprintf(outc,"const ShaderDictEntry shader_dictionary[%d]={\n",dict.size());
   for(size_t i=0;i<dict.size();++i)
   {
      uid_to_symbol[dict[i].uid]=&dict[i];
      //log("%02x uid:%d f:%d l:%d r:%d\n",dict[i].c,dict[i].uid,dict[i].f,dict[i].left,dict[i].right);
  //    fprintf(outc,"{ %d, %d, %d },\n",dict[i].c,findSymbolIndex(dict[i].left),findSymbolIndex(dict[i].right));
   }
   //fprintf(outc,"};\n");

}

static std::string decodeShader(const unsigned char* encoded, const long unsigned int encoded_bit_count)
{
   if(encoded==NULL || encoded_bit_count==0)
      return std::string("");

   std::string decoded;
   uint32 unencoded_bit_count=0;
   while(unencoded_bit_count<encoded_bit_count)
   {
      Symbol* s=&root;

      while(!s->terminal)
      {
         int b=encoded[unencoded_bit_count/8]&(1<<(unencoded_bit_count&7));
         if(b)
            s=uid_to_symbol[s->right];
            //s=findSymbol(s->right);
         else
            s=uid_to_symbol[s->left];
            //s=findSymbol(s->left);
         assert(s);
         ++unencoded_bit_count;
      }

      decoded.push_back(s->c);
      //log("%c",s->c);
   }
   return decoded;
}

}

/* GIMP RGBA C-Source image dump (2015.c) */

static const struct {
  unsigned long int  	 width;
  unsigned long int  height;
  unsigned long int  	 bytes_per_pixel; /* 2:RGB16, 3:RGB, 4:RGBA */
  unsigned char 	 pixel_data[188 * 72 * 4 + 1];
} image_2015 = {
  188, 72, 4,
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0w\0\0\0w\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0w\0\0\0w\0\0\0\252"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\0\0\0"
  "w\0\0\0w\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  """3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\314\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0D\0\0\0\252\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\210\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\273\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0"
  "\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0\0\0D\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0\0\0""3\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\210\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\356\0\0\0""3\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\0\0\0\21\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\21\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0"
  "\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\314\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\356\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\273\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\231\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\"\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0D\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\335\0\0\0w\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0w"
  "\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\"\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0\"\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\0\0\0D\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\335\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\0\0\0\21\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\356\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\231"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\273\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\0\0\0w\0\0\0""3\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\335\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\356\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\21\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\273\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\0\0\0""3\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0"
  "\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\21\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0U\0\0\0w\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\273\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\356\0\0\0\"\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\231\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\273\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\0\0\0\""
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\335\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356"
  "\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\273\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\356"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\356\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\335\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0""3\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\335\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0f\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0w\0\0\0\335\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\335\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\0"
  "\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0"
  "\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0\0\0f\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\"\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\335\0\0\0w\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\356"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\231\0\0\0w\0\0\0""3\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\231\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\273\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\356\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\252\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0D\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\356\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\356"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\231\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  """3\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0f\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0""3"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\0\0\0D\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\"\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0w\0\0\0w\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0w\0\0\0\210\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0D\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\210"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0f\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\21\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\356\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\0\0\0\"\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0D\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\314\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\356\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0w\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\210\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\252\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0f\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\273\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\0\0\0\273\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\210\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\231\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\356\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0U\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0f\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\0\0\0\21\0\0\0\314\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\0\0\0\"\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0""3\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0""3\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\356\0\0\0U\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\356\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0"
  "\252\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0w\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0U\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\21\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\314\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0U\0\0\0\314\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0"
  "\0\0w\0\0\0\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\210"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\335\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\356\0\0\0w\0\0\0"
  "\21\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\0\0\0\231\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\210\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\21\0\0\0w"
  "\0\0\0w\0\0\0w\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\356\0\0\0w\0\0\0w\0\0\0w\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\252\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\252\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\0\0\0\335\0\0\0\377\0\0\0\377"
  "\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0"
  "\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0"
  "\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377\0\0\0\377"
  "\0\0\0\356\0\0\0w\0\0\0w\0\0\0\"\377\377\377\0\377\377\377\0\377\377\377"
  "\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0"
  "\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377"
  "\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377"
  "\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377\0\377\377\377"
  "\0",
};



template<typename T>
T clamp(const T& x, const T& x0, const T& x1)
{
   return std::min(x1, std::max(x0, x));
}

template<>
Vec3 clamp<Vec3>(const Vec3& x, const Vec3& x0, const Vec3& x1)
{
   return Vec3(clamp(x.x, x0.x, x1.x), clamp(x.y, x0.y, x1.y), clamp(x.z, x0.z, x1.z));
}

template<typename T>
T mix(const T& a, const T& b, float x)
{
   return a * (1.0f - x) + b * x;
}

template<>
Vec3 mix<Vec3>(const Vec3& a, const Vec3& b, float x)
{
   return Vec3(mix(a.x, b.x, x), mix(a.y, b.y, x), mix(a.z, b.z, x));
}


template<typename T>
T cubic(const T& x)
{
   return (3 - 2 * x) * x * x;
}

template<typename T>
T smoothstep(const T& a, const T& b, const T& c)
{
   return cubic(clamp((c - a) / (b - a), T(0), T(1)));
}


struct ShaderProgram
{
    void uniform1i(const char* uniform_name, GLint x);
    void uniform2i(const char* uniform_name, GLint x, GLint y);

    void uniform4f(const char* uniform_name, GLfloat x, GLfloat y, GLfloat z, GLfloat w);
    void uniform3f(const char* uniform_name, GLfloat x, GLfloat y, GLfloat z);
    void uniform2f(const char* uniform_name, GLfloat x, GLfloat y);
    void uniform1f(const char* uniform_name, GLfloat x);

    void uniformMatrix3fv(const char* uniform_name, GLsizei count, GLboolean transpose, const GLfloat* value);
    void uniformMatrix4fv(const char* uniform_name, GLsizei count, GLboolean transpose, const GLfloat* value);

    void use() const;

    GLuint name;
    ProgramTransformProperties transform_properties;
};

struct Framebuffer
{
    Framebuffer()
    {
        fbo = 0;
        depth_rbo = 0;
        num_texs = 0;
        internal_depth = false;
        width = 0;
        height = 0;
    }

    void bindForDraw() const;
    void bindForRead(uint32 buffer) const;

    static const uint32 max_texs = 8;
    GLuint fbo;
    GLuint depth_rbo;
    GLuint texs[max_texs];
    GLenum tex_formats[max_texs];
    uint32 num_texs;
    bool internal_depth;
    uint32 width, height;
};

struct Sphere
{
   Sphere(): material(0)
   {
      appear_time=0;
      disappear_time=0;
      base_scale=0;
   }

   Mat4 evaluate(double t) const;
   bool active(double t) const;

   double appear_time, disappear_time;
   Vec3 location;
   Vec3 direction;
   Vec3 euler_angles;
   float scale_asr[3];
   float base_scale;
   uint32 material;
};

struct Debris
{
   Debris(): material(0)
   {
      appear_time=0;
      disappear_time=0;
      base_scale=0;
   }

   Mat4 evaluate(double t) const;
   bool active(double t) const;

   double appear_time, disappear_time;
   float base_scale;
   Vec3 initial_location;
   Vec3 initial_velocity;
   Vec3 acceleration;
   uint32 material;
};

namespace
{
   double g_rocket_linear_z_vel = 0;
   double g_time = 0;
   const double g_particles_activate_time = 77;
   const double g_particles_deactivate_time = 120;
   const double g_big_shadowmap_activate_time = 145;
   const double g_blossom_time = 1*60+47;
   const double g_explosion_time = 2*60+46.5;
   const double g_subdiv_transformation_time = 2*60+8;
   const double g_rocket_liftoff_time = 2*60+11 + 10.0;
   const double g_info_appear_time = 2*60+55;//2*60+49.5;
   ShaderProgram particle_init_program;
   ShaderProgram particle_update_program;
   ShaderProgram particle_draw_program;
   ShaderProgram particle_plot_program;
   ShaderProgram shadow_program;
   ShaderProgram secondary_program;
   ShaderProgram motionblur_program;
   ShaderProgram primary_program;
   ShaderProgram sky_program;
   ShaderProgram raytrace_program;
   ShaderProgram blur_program;
   ShaderProgram post_program;
   ShaderProgram paper_program;
   ShaderProgram chunk_program;
   ShaderProgram chunk_update_program;
   ShaderProgram chunk_shadow_program;
   ShaderProgram crystal_program;
   ShaderProgram sphere_program;
   ShaderProgram bloom_program;
   ShaderProgram light_program;
   ShaderProgram subdiv_program;
   ShaderProgram skybox_program;
   //ShaderProgram smoketrail_program;
   ShaderProgram info_program;
   ShaderProgram staticparticle_program;
   ShaderProgram skyscraper_program;
   Framebuffer staticparticle_framebuffer;
   Framebuffer particle_framebuffers[2];
   Framebuffer blur_framebuffer;
   Framebuffer primary_framebuffer;
   Framebuffer motionblur_framebuffer;
   Framebuffer final_framebuffer;
   Framebuffer crystal_framebuffer;
   Framebuffer bloom_framebuffers[2];
   uint32 g_shadow_voxel_texture_width = 0, g_shadow_voxel_texture_height = 0;
   uint32 particle_shadow_frame_counter = 0;
   GLuint particle_velocity_map = 0;
   GLuint g_cube_vbo = 0, g_cube_ebo = 0;
   Mat4 world_to_voxel;
   Vec3 voxel_texture_size;
   const Vec3 voxel_box_min(-57.813171, 24.264435, -108.505753);
   const Vec3 voxel_box_max(70.660545, 50.578312, 46.626251);
   GLuint collision_voxel_texture = 0;
   GLuint shadow_voxel_textures[2] = { 0, 0 };
   GLuint shadow_voxel_fbos[2] = { 0, 0 };
   uint32 g_texture_memory_usage = 0;
   uint32 g_buffer_memory_usage = 0;
   GLuint g_chunk_textures[2] = { 0, 0 };
   GLuint g_chunk_fbos[2] = { 0, 0 };
   uint32 g_chunk_count = 0;
   GLuint g_chunk_vbo = 0;
   GLuint g_chunk_ebo = 0;
   uint32 g_chunk_vertex_data_length = 0;
   GLuint g_chunk_texture_size = 0;
   Mat4 g_world_to_revision_logo_sdf;
   const Uint32 chunk_update_interval = 30;
   const Uint32 particle_update_interval = 13;
   uint32 g_material_chunk_counts[num_materials] = { 0 };
   std::vector<Vec3> g_blossom_locations;
   const uint32 g_subdiv_max_vertices = 10000 * 4 * 4;
   const uint32 g_subdiv_max_indices = 9000 * 3 * 4 * 4;
   const uint32 g_staticparticles_size = 512;
   GLfloat g_subdiv_scratch_vertices[2][g_subdiv_max_vertices * 3];
   uint32 g_subdiv_scratch_indices[2][g_subdiv_max_indices];
   uint32 g_subdiv_scratch_triangle_materials[2][g_subdiv_max_indices / 3];
   GLuint g_subdiv_vbo = 0, g_subdiv_ebo = 0;
   GLuint g_final_subdiv_triangle_count = 0;
   uint32 g_material_subdiv_counts[num_materials * num_materials] = { 0 };
   GLuint g_sky_cubemap = 0;
   uint32 g_sky_cubemap_size = 2048;
/*
   GLuint g_smoketrail_grid_vbo = 0;
   GLuint g_smoketrail_grid_ebo = 0;
   uint32 g_smoketrail_index_count = 0;
*/
   GLuint g_stars_texture = 0;
   const uint32 skyscraper_cubemap_size = 1024;
   GLuint skyscraper_cubemap_texs[skyscraper_node_count];
   GLuint g_sphere_vbo = 0;
   GLuint g_sphere_ebo = 0;
   GLuint g_sphere_index_count = 0;
   std::vector<Sphere> g_spheres;
   std::vector<Debris> g_debris;
   GLuint g_2015_tex = 0;
   GLuint g_info_vbo = 0, g_info_ebo = 0;
   GLuint g_info_index_count=0;
}

static void renderDebris(const CameraConfig& cc,const ProgramTransformProperties& transform_properties,const double t)
{
    if(transform_properties.world_to_view_matrix_location >= 0)
      glUniformMatrix4fv(transform_properties.world_to_view_matrix_location, 1, GL_FALSE, cc.world_to_view.e);

    if(transform_properties.view_to_clip_matrix_location >= 0)
      glUniformMatrix4fv(transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, cc.view_to_clip.e);

    assert(g_sphere_vbo != 0);
    assert(g_sphere_ebo != 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_sphere_vbo);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_sphere_ebo);

    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

   for(size_t i=0;i<g_debris.size();++i)
   {
      const Debris& d=g_debris[i];

      if(!d.active(t))
         continue;

      const Mat4 global_xfrm = d.evaluate(t);
      const Mat4 previous_global_xfrm = d.evaluate(t-1.0/24.0);

     if(transform_properties.object_to_world_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_world_matrix_location, 1, GL_FALSE, global_xfrm.e);

     if(transform_properties.object_to_view_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (cc.world_to_view * global_xfrm).e);

     if(transform_properties.object_to_clip_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_clip_matrix_location, 1, GL_FALSE, (cc.world_to_clip * global_xfrm).e);

     if(transform_properties.previous_object_to_clip_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.previous_object_to_clip_matrix_location, 1, GL_FALSE, (cc.previous_world_to_clip * previous_global_xfrm).e);

     if(transform_properties.material_index_location >= 0)
         glUniform1f(transform_properties.material_index_location, float((d.material + 1) + 0.5) / (float)((num_materials + 1) * 2));

      assert(g_sphere_index_count > 0);
      glDrawElements(GL_TRIANGLES, g_sphere_index_count, GL_UNSIGNED_BYTE, (const GLvoid*)0);

      CHECK_FOR_GL_ERRORS("end of single debris render");
    }

    glDisableVertexAttribArray(0);
}

bool Debris::active(double t) const
{
   return t >= appear_time && t < disappear_time;
}

Mat4 Debris::evaluate(double t) const
{
   t-=appear_time;
   Vec3 p=initial_location+(initial_velocity+(acceleration*t))*t;
   return Mat4::translation(p) * Mat4::scale(Vec3(base_scale));
}


static void renderSpheres(const CameraConfig& cc,const ProgramTransformProperties& transform_properties,const double t)
{
    if(transform_properties.world_to_view_matrix_location >= 0)
      glUniformMatrix4fv(transform_properties.world_to_view_matrix_location, 1, GL_FALSE, cc.world_to_view.e);

    if(transform_properties.view_to_clip_matrix_location >= 0)
      glUniformMatrix4fv(transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, cc.view_to_clip.e);

    assert(g_sphere_vbo != 0);
    assert(g_sphere_ebo != 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_sphere_vbo);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_sphere_ebo);

    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

   for(size_t i=0;i<g_spheres.size();++i)
   {
      const Sphere& s=g_spheres[i];

      if(!s.active(t))
         continue;

      const Mat4 global_xfrm = s.evaluate(t);
      const Mat4 previous_global_xfrm = s.evaluate(t-1.0/24.0);

     if(transform_properties.object_to_world_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_world_matrix_location, 1, GL_FALSE, global_xfrm.e);

     if(transform_properties.object_to_view_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (cc.world_to_view * global_xfrm).e);

     if(transform_properties.object_to_clip_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.object_to_clip_matrix_location, 1, GL_FALSE, (cc.world_to_clip * global_xfrm).e);

     if(transform_properties.previous_object_to_clip_matrix_location >= 0)
         glUniformMatrix4fv(transform_properties.previous_object_to_clip_matrix_location, 1, GL_FALSE, (cc.previous_world_to_clip * previous_global_xfrm).e);

     if(transform_properties.material_index_location >= 0)
         glUniform1f(transform_properties.material_index_location, float((s.material + 1) + 0.5) / (float)((num_materials + 1) * 2));

      assert(g_sphere_index_count > 0);
      glDrawElements(GL_TRIANGLES, g_sphere_index_count, GL_UNSIGNED_BYTE, (const GLvoid*)0);

      CHECK_FOR_GL_ERRORS("end of single sphere render");
    }

    glDisableVertexAttribArray(0);
}

bool Sphere::active(double t) const
{
   return t >= appear_time && t < disappear_time;
}



/*
static void createSmokeTrailGridVBO()
{
   const uint32 w = 20, h = 20;
   std::vector<GLfloat> vertices(w*h*2);
   std::vector<GLushort> indices((w-1)*(h-1)*2*3);
   g_smoketrail_index_count=indices.size();
   for(uint32 i=0;i<w;++i)
      for(uint32 j=0;j<h;++j)
      {
         const uint32 a=i+j*w;
         vertices[a*2+0]=GLfloat(i)/GLfloat(w-1);
         vertices[a*2+1]=GLfloat(j)/GLfloat(h-1);
         if(i<(w-1) && j<(h-1))
         {
            const uint32 b=i+j*(w-1);
            indices[b*3*2+0]=a;
            indices[b*3*2+1]=a+1+w;
            indices[b*3*2+2]=a+1;

            indices[b*3*2+3]=a;
            indices[b*3*2+4]=a+w;
            indices[b*3*2+5]=a+1+w;
         }
      }
   glGenBuffers(1, &g_smoketrail_grid_vbo);
   glGenBuffers(1, &g_smoketrail_grid_ebo);
   glBindBuffer(GL_ARRAY_BUFFER, g_smoketrail_grid_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_smoketrail_grid_ebo);
   glBufferData(GL_ARRAY_BUFFER,vertices.size()*sizeof(vertices[0]),&vertices[0],GL_STATIC_DRAW);
   glBufferData(GL_ELEMENT_ARRAY_BUFFER,indices.size()*sizeof(indices[0]),&indices[0],GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("createSmokeTrailGridVBO");
}
*/

// 90-degree field-of-view
static const Mat4 g_90_degree_frustum=Mat4(	1.0f, 0.0f, 0.0f, 0.0f,
                                    0.0f, 1.0f, 0.0f, 0.0f,
                                    0.0f, 0.0f, -1.00004f, -1.0f,
                                    0.0f, 0.0f, -0.2f, 0.0f ).transpose();

// a viewing matrix for each of the 6 cubemap faces
static const Mat4 g_cubemap_configs[6]={Mat4(0.0f,  0.0f,-1.0f,0.0f,
                                             0.0f,  1.0f,  0.0f, 0.0f,
                                             -1.0f, 0.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, 0.0f,1.0f).transpose(),

                                    Mat4(	-1.0f, 0.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, -1.0f, 0.0f,
                                             0.0f,  1.0f,  0.0f, 0.0f,
                                             0.0f,  0.0f,  0.0f,1.0f).transpose(),

                                    Mat4(	1.0f,  0.0f,  0.0f, 0.0f,
                                             0.0f,  1.0f,  0.0f, 0.0f,
                                             0.0f,  0.0f, -1.0f, 0.0f,
                                             0.0f,  0.0f, 0.0f,1.0f).transpose(),

                                    Mat4(	0.0f,  0.0f, 1.0f, 0.0f,
                                             0.0f,  1.0f, 0.0f, 0.0f,
                                             1.0f,  0.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, 0.0f,1.0f).transpose(),


                                    Mat4(  -1.0f, 0.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, 1.0f, 0.0f,
                                             0.0f, -1.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, 0.0f,1.0f).transpose(),

                                    Mat4(	-1.0f, 0.0f, 0.0f, 0.0f,
                                             0.0f,  1.0f, 0.0f, 0.0f,
                                             0.0f,  0.0f, 1.0f, 0.0f,
                                             0.0f,  0.0f,0.0f,1.0f).transpose()  };

static void attachCubemapFace(GLenum target, GLenum attachment, const uint32 face, GLuint texture_name, GLint level)
{
	static const GLenum textargets[6]=
	{
		GL_TEXTURE_CUBE_MAP_NEGATIVE_X_EXT,
		GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT,
		GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT,
		GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT,
		GL_TEXTURE_CUBE_MAP_POSITIVE_Y_EXT,
		GL_TEXTURE_CUBE_MAP_POSITIVE_Z_EXT
	};

   assert(face < 6);
	glFramebufferTexture2D(target, attachment, textargets[face], texture_name, level);
   CHECK_FOR_GL_ERRORS("attachCubemapFace");
}


static void my_glBufferData(GLenum target, GLsizeiptr size, const GLvoid* data, GLenum usage)
{
   glBufferData(target, size, data, usage);
   g_buffer_memory_usage += size;
   CHECK_FOR_GL_ERRORS("glBufferData");
}

static void my_glTexStorage1D(GLenum target, GLsizei levels, GLenum internalformat, GLsizei width)
{
   glTexStorage1D(target, levels, internalformat, width);
   g_texture_memory_usage += getBytesForInternalFormat(internalformat) * width;
   CHECK_FOR_GL_ERRORS("glTexStorage1D");
}

static void my_glTexStorage2D(GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height)
{
   glTexStorage2D(target, levels, internalformat, width, height);
   g_texture_memory_usage += getBytesForInternalFormat(internalformat) * width * height;
   CHECK_FOR_GL_ERRORS("glTexStorage2D");
}

static void my_glTexStorage3D(GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth)
{
   glTexStorage3D(target, levels, internalformat, width, height, depth);
   g_texture_memory_usage += getBytesForInternalFormat(internalformat) * width * height * depth;
   CHECK_FOR_GL_ERRORS("glTexStorage3D");
}

void loadVoxels()
{
   const Uint32 start_time=SDL_GetTicks();

   long int xsizp2 = 1, ysizp2 = 1, zsizp2 = 1;

   while(xsizp2 < voxels_xsiz)
      xsizp2 <<= 1;

   while(ysizp2 < voxels_ysiz)
      ysizp2 <<= 1;

   while(zsizp2 < voxels_zsiz)
      zsizp2 <<= 1;

   GLubyte* pix=new GLubyte[xsizp2*ysizp2*zsizp2];
   memset(pix,0,xsizp2*ysizp2*zsizp2);
   unsigned char* pix_set=new unsigned char[(xsizp2*ysizp2*zsizp2+7)/8];
   memset(pix_set,0,(xsizp2*ysizp2*zsizp2+7)/8);


   long int idx=0;
   for(int x=0;x<voxels_xsiz;x++)
      for(int y=0;y<voxels_ysiz;y++)
      {
         for(int z=0;z<voxel_column_lengths[x*voxels_ysiz+y];++z)
         {
            int z2=voxel_heights[idx];
            pix[x+(y+z2*ysizp2)*xsizp2]=255;
            ++idx;

            for(int k=-1;k<=+1;++k)
               for(int j=-1;j<=+1;++j)
                  for(int i=-1;i<=+1;++i)
                  {
                     int x2=x+i;
                     int y2=y+j;
                     int z3=z2+k;
                     if(x2<0 || y2<0 || z3<0 || x2>=xsizp2 || y2>=ysizp2 || z3>=zsizp2)
                        continue;
                     const uint32 id=x2+(y2+z3*ysizp2)*xsizp2;
                     pix_set[id/8]|=1<<(id&7);
                  }
         }
      }


   GLubyte* pix2=new GLubyte[xsizp2*ysizp2*zsizp2];
   memset(pix2,0,xsizp2*ysizp2*zsizp2);

   for(int z=1;z<zsizp2-1;++z)
      for(int y=1;y<ysizp2-1;++y)
         for(int x=1;x<xsizp2-1;++x)
         {
            const uint32 id=x+(y+z*ysizp2)*xsizp2;
            if(pix_set[id/8]&(1<<(id&7)))
            {
               int c=0;
               for(int k=-1;k<=+1;++k)
                  for(int j=-1;j<=+1;++j)
                     for(int i=-1;i<=+1;++i)
                     {
                        int x2=x+i,y2=y+j,z2=z+k;
                        c += pix[x2+(y2+z2*ysizp2)*xsizp2];
                     }
               pix2[x+(y+z*ysizp2)*xsizp2]=std::min(255,std::max(0,c / (3 * 3 * 3)));
            }
         }


   glGenTextures(1,&collision_voxel_texture);
   glBindTexture(GL_TEXTURE_3D, collision_voxel_texture);
   glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER);
   glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER);
   glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_BORDER);
   my_glTexStorage3D(GL_TEXTURE_3D, 1, GL_R8, xsizp2, ysizp2, zsizp2);
   glTexSubImage3D(GL_TEXTURE_3D,0,0,0,0,xsizp2, ysizp2, zsizp2,GL_RED,GL_UNSIGNED_BYTE,pix2);
   glBindTexture(GL_TEXTURE_3D, 0);
   CHECK_FOR_GL_ERRORS("creating collision voxel texture");

   glGenFramebuffers(2, shadow_voxel_fbos);
   glGenTextures(2, shadow_voxel_textures);

   g_shadow_voxel_texture_width = xsizp2;
   g_shadow_voxel_texture_height = ysizp2;

   for(uint32 i = 0; i < 2; ++i)
   {
      glBindFramebuffer(GL_FRAMEBUFFER,shadow_voxel_fbos[i]);
      glBindTexture(GL_TEXTURE_3D, shadow_voxel_textures[i]);
      glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_REPEAT);
      my_glTexStorage3D(GL_TEXTURE_3D, 1, GL_R8, xsizp2, ysizp2, zsizp2);
      CHECK_FOR_GL_ERRORS("creating shadow voxel texture");
      glBindTexture(GL_TEXTURE_3D, 0);
      glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, shadow_voxel_textures[i], 0);
      GLenum b[1] = { GL_COLOR_ATTACHMENT0 };
      glDrawBuffers(1, b);
      glClearColor(0,0,0,0);
      glViewport(0, 0, xsizp2, ysizp2);
      glClear(GL_COLOR_BUFFER_BIT);
      CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "shadow voxels");
      assert(glCheckFramebufferStatus(GL_FRAMEBUFFER) == GL_FRAMEBUFFER_COMPLETE);
      glBindFramebuffer(GL_FRAMEBUFFER,0);
      CHECK_FOR_GL_ERRORS("creating shadow voxel FBO");
   }

   delete[] pix_set;
   pix_set=NULL;
   delete[] pix;
   pix=NULL;
   delete[] pix2;
   pix2=NULL;

   static const Mat4 m(1.0, 0.0, 0.0, 0.0,
                       0.0, 0.0, 1.0, 0.0,
                       0.0,-1.0, 0.0, 1.0,
                       0.0, 0.0, 0.0, 1.0);

   voxel_texture_size = Vec3(xsizp2, ysizp2, zsizp2);
   world_to_voxel = Mat4::scale(Vec3(Vec3(voxels_xsiz, voxels_ysiz, voxels_zsiz) / voxel_texture_size)) * m * Mat4::scale(Vec3(Vec3(1,1,1)/(voxel_box_max - voxel_box_min))) * Mat4::translation(Vec3(-voxel_box_min));

   log("loadVoxels: %dms\n",SDL_GetTicks()-start_time);
}


void ShaderProgram::uniform1i(const char* uniform_name, GLint x)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform1i(name, loc, x);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform1i");
}

void ShaderProgram::uniform2i(const char* uniform_name, GLint x, GLint y)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform2i(name, loc, x, y);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform2i");
}


void ShaderProgram::uniform4f(const char* uniform_name, GLfloat x, GLfloat y, GLfloat z, GLfloat w)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform4f(name, loc, x, y, z, w);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform4f");
}

void ShaderProgram::uniform3f(const char* uniform_name, GLfloat x, GLfloat y, GLfloat z)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform3f(name, loc, x, y, z);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform3f");
}

void ShaderProgram::uniform2f(const char* uniform_name, GLfloat x, GLfloat y)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform2f(name, loc, x, y);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform2f");
}

void ShaderProgram::uniform1f(const char* uniform_name, GLfloat x)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniform1f(name, loc, x);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniform1f");
}

void ShaderProgram::uniformMatrix4fv(const char* uniform_name, GLsizei count, GLboolean transpose, const GLfloat* value)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniformMatrix4fv(name, loc, count, transpose, value);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniformMatrix4fv");
}

void ShaderProgram::uniformMatrix3fv(const char* uniform_name, GLsizei count, GLboolean transpose, const GLfloat* value)
{
   GLint loc=glGetUniformLocation(name, uniform_name);
   if(loc<0)
      return;
   glProgramUniformMatrix3fv(name, loc, count, transpose, value);
   CHECK_FOR_GL_ERRORS("ShaderProgram::uniformMatrix3fv");
}


void ShaderProgram::use() const
{
   glUseProgram(name);
   CHECK_FOR_GL_ERRORS("ShaderProgram::use");
}



void Framebuffer::bindForDraw() const
{
    assert(fbo != 0);
    glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo);
    CHECK_FOR_GL_ERRORS("Framebuffer::bindForDraw: glBindFramebuffer");
    GLenum draw_buffers[max_texs];
    for(uint32 i = 0; i < num_texs; ++i)
        draw_buffers[i] = GL_COLOR_ATTACHMENT0 + i;
    glDrawBuffers(num_texs, draw_buffers);
    CHECK_FOR_GL_ERRORS("Framebuffer::bindForDraw: glDrawBuffers");
    glViewport(0, 0, width, height);
    CHECK_FOR_GL_ERRORS("Framebuffer::bindForDraw: glViewport");
}

void Framebuffer::bindForRead(uint32 buffer) const
{
    assert(fbo != 0);
    glBindFramebuffer(GL_READ_FRAMEBUFFER, fbo);
    GLenum draw_buffers[max_texs];
    for(uint32 i = 0; i < num_texs; ++i)
        draw_buffers[i] = GL_COLOR_ATTACHMENT0 + i;
    glReadBuffer(draw_buffers[buffer]);
    CHECK_FOR_GL_ERRORS("Framebuffer::bindForRead");
}

namespace
{
   int      g_window_width = 1280, g_window_height = 720;
   //int      g_window_width = 1920*2, g_window_height = 1080*2;
   GLuint   g_noise_tex = 0;
   GLuint   g_noise_tex2 = 0;
   Orion*   g_orion = NULL;
   GLuint   g_generic_depth_rbo = 0;
   GLuint   g_quad_vbo = 0;
   double  g_current_rocket_z_vel = 0;
}


Mat4 Sphere::evaluate(double t) const
{
   const float t3=(t-appear_time)/(disappear_time-appear_time);
   const float t2=(scale_asr[0]+scale_asr[1]+scale_asr[2])*t3;
   float scale=(smoothstep(0.0f,scale_asr[0],t2)-smoothstep(scale_asr[0]+scale_asr[1],scale_asr[0]+scale_asr[1]+scale_asr[2],t2))*base_scale;
   const Mat4 rot=Mat4::rotation(euler_angles.z,Vec3(0,0,1))*Mat4::rotation(euler_angles.y,Vec3(0,1,0))*Mat4::rotation(euler_angles.x,Vec3(1,0,0));
   return Mat4::translation(Vec3(location + direction * t3)) * Mat4::scale(Vec3(scale)) * rot;
}


static void createQuadVBO()
{
    static const GLfloat verts[8] = { +1.0f,  -1.0f, +1.0f, +1.0f,  -1.0f,  -1.0f,  -1.0f, +1.0f };
    glGenBuffers(1, &g_quad_vbo);
    glBindBuffer(GL_ARRAY_BUFFER, g_quad_vbo);
    my_glBufferData(GL_ARRAY_BUFFER, sizeof(verts), verts, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

static void renderQuad()
{
    glBindBuffer(GL_ARRAY_BUFFER, g_quad_vbo);
    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, 0);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    glDisableVertexAttribArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

static void renderQuadGrid()
{
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glEnableVertexAttribArray(0);
   static const int s=256;
   for(int y=0;y<s;++y)
      for(int x=0;x<s;++x)
      {
         GLfloat vs[8]={float(x+0)/float(s)*2.0f-1.0f, float(y+0)/float(s)*2.0f-1.0f,
                        float(x+1)/float(s)*2.0f-1.0f, float(y+0)/float(s)*2.0f-1.0f,
                        float(x+0)/float(s)*2.0f-1.0f, float(y+1)/float(s)*2.0f-1.0f,
                        float(x+1)/float(s)*2.0f-1.0f, float(y+1)/float(s)*2.0f-1.0f};
         glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, vs);
         glDrawArrays(GL_TRIANGLE_STRIP,0,4);
      }
   glDisableVertexAttribArray(0);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
}

static void initFramebuffer(Framebuffer& fb)
{
    glGenFramebuffers(1, &fb.fbo);
    glBindFramebuffer(GL_FRAMEBUFFER, fb.fbo);
    glGenTextures(fb.num_texs, fb.texs);
    for(uint32 i = 0; i < fb.num_texs; ++i)
    {
        glBindTexture(GL_TEXTURE_2D, fb.texs[i]);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        my_glTexStorage2D(GL_TEXTURE_2D, 1, fb.tex_formats[i], fb.width, fb.height);
        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + i, GL_TEXTURE_2D, fb.texs[i], 0);
    }
    if(fb.internal_depth)
    {
        glGenRenderbuffers(1, &fb.depth_rbo);
        glBindRenderbuffer(GL_RENDERBUFFER, fb.depth_rbo);
        glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT32, fb.width, fb.height);
    }
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, fb.depth_rbo);
    CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "generic framebuffer");
    GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    assert(status == GL_FRAMEBUFFER_COMPLETE);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    CHECK_FOR_GL_ERRORS("framebuffer initialisation");
}


static void bindBackbuffer(GLenum binding_point)
{
    glBindFramebuffer(binding_point, 0);
    if(binding_point == GL_DRAW_FRAMEBUFFER)
    {
        glDrawBuffer(GL_BACK);
        glViewport(0, 0, g_window_width, g_window_height);
    }
    else
    {
        glReadBuffer(GL_BACK);
    }
    CHECK_FOR_GL_ERRORS("binding backbuffer");
}

static void createInfoVBO()
{
   static const GLfloat sw=GLfloat(image_2015.width)/1920.0f;
   static const GLfloat sh=GLfloat(image_2015.height)/1080.0f;
   static const GLfloat y=-0.8;
   static const GLfloat vertices[4*4]={-sw, -sh+y, 0.0f, 1.0f,
                                       +sw, -sh+y, 1.0f, 1.0f,
                                       -sw, +sh+y, 0.0f, 0.0f,
                                       +sw, +sh+y, 1.0f, 0.0f, };
   static GLubyte indices[]={0,1,2,2,1,3};
   g_info_index_count=sizeof(indices)/sizeof(indices[0]);
   assert((g_info_index_count%3)==0);
   glGenBuffers(1, &g_info_vbo);
   glGenBuffers(1, &g_info_ebo);
   glBindBuffer(GL_ARRAY_BUFFER, g_info_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_info_ebo);
   my_glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
   my_glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("creating info VBO");
}

static void createDebris()
{
   assert(g_orion);
   assert(g_orion->entities.size()==node_count);

   Viewport vp;
   g_orion->updateAnimations(0, vp);
   for(uint32 j=0;j<skyscraper_node_count;++j)
   {
      uint32 n = skyscraper_nodes[j];

      assert(n < g_orion->entities.size());
      const Mat4 global_xfrm = g_orion->getEntityGlobalTransformationForTime(n, 0);
      Vec3 p = global_xfrm.column(3);
      Ran ran(65423);
      const double t0 = g_orion->getFirstKeyTime(g_orion->getChannelIndexForEntity(g_orion->entities[n]));
      const double t1 = g_orion->getLastKeyTime(g_orion->getChannelIndexForEntity(g_orion->entities[n]));
      assert(t1>t0);
      assert(t0<1e8);
      assert(t1>=0);
      p.y+=5;

      for(int i=0;i<50;++i)
      {
         Debris d;
         float to=ran.doub()*4-3.2;
         d.appear_time=t0+2+to;
         d.disappear_time=t1+10+to+ran.doub()*0.5;
         float a=ran.doub()*2*M_PI;
         d.initial_location=p;
         float r=(0.1+ran.doub()*0.5)*1.5;
         d.initial_velocity=Vec3(cos(a)*r,6,sin(a)*r)*mix(3,5,ran.doub())*1.5f;
         d.acceleration=Vec3(0.0,-10,0.0);
         d.base_scale=mix(0.4,1.0,ran.doub());
         d.material=mound_material;
         g_debris.push_back(d);
      }
   }
}

static void createCubeVBO()
{
   GLfloat vertices[8 * 3];
   GLubyte indices[6 * 2 * 3];

   for(GLuint z = 0; z < 2; ++z)
      for(GLuint y = 0; y < 2; ++y)
         for(GLuint x = 0; x < 2; ++x)
         {
            GLuint i = x+(y+z*2)*2;
            vertices[i * 3 + 1] = y ? +1.0f : -1.0f;
            vertices[i * 3 + 0] = x ? +1.0f : -1.0f;
            vertices[i * 3 + 2] = z ? +1.0f : -1.0f;
         }

   GLuint i2 = 0;

   for(GLuint axis = 0; axis < 3; ++axis)
      for(GLuint side = 0; side < 2; ++side)
      {
         GLuint s_step = 1 << ((axis + 1) % 3),
               t_step = 1 << ((axis + 2) % 3),
               u_step = side << axis;

         int j = i2;

         indices[i2++] = 1 * s_step + 0 * t_step + u_step;
         indices[i2++] = 0 * s_step + 0 * t_step + u_step;
         indices[i2++] = 0 * s_step + 1 * t_step + u_step;

         indices[i2++] = 0 * s_step + 1 * t_step + u_step;
         indices[i2++] = 1 * s_step + 1 * t_step + u_step;
         indices[i2++] = 1 * s_step + 0 * t_step + u_step;

         if(side)
         {
            int k = i2;
            for(; j < i2 - 3; ++j)
               std::swap(indices[j], indices[--k]);
         }
      }

   glGenBuffers(1, &g_cube_vbo);
   glGenBuffers(1, &g_cube_ebo);
   glBindBuffer(GL_ARRAY_BUFFER, g_cube_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_cube_ebo);
   my_glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
   my_glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("creating cube VBO");
}

static void makeVertex(GLfloat* vertices, GLuint idx, GLfloat x,GLfloat y,GLfloat z)
{
   vertices[idx*3+0]=x;
   vertices[idx*3+1]=y;
   vertices[idx*3+2]=z;
}

static void makeTriangle(GLubyte* indices, GLuint idx, GLubyte x,GLubyte y,GLubyte z)
{
   indices[idx*3+0]=x;
   indices[idx*3+1]=y;
   indices[idx*3+2]=z;
}
/*
static void createSpheres()
{
   assert(g_orion);
   assert(g_orion->entities.size()==node_count);
   assert(rocket_node_index < g_orion->entities.size());
   Ran ran(65423);

   const float blastoff_smoke_scale=1;
   const float blastoff_smoke_spread=2;

   Viewport vp;
   g_orion->updateAnimations(0, vp);

   const double tt=1064.0/24.0;

   {
      const Vec3 p = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, 0.0).column(3);

      // blast off smoke 1
      for(int i=0;i<10;++i)
      {
         Sphere s;
         s.appear_time=tt-10;
         s.disappear_time=tt+5+ran.doub()*2;
         s.material=0;
         s.scale_asr[0]=1;
         s.scale_asr[1]=10;
         s.scale_asr[2]=1;
         s.base_scale=1.5*blastoff_smoke_scale;
         s.location=p;
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f*blastoff_smoke_spread;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 2
      for(int i=0;i<5;++i)
      {
         Sphere s;
         s.appear_time=tt-10;
         s.disappear_time=tt+5+ran.doub()*2;
         s.material=0;
         s.scale_asr[0]=8;
         s.scale_asr[1]=10;
         s.scale_asr[2]=4;
         s.base_scale=0.6*blastoff_smoke_scale;
         s.location=p+Vec3(0,0.15,0);
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 3
      for(int i=0;i<10;++i)
      {
         Sphere s;
         s.appear_time=(tt-10)+i*2;
         s.disappear_time=tt+5+ran.doub()*2+i*2;
         s.material=0;
         s.scale_asr[0]=1;
         s.scale_asr[1]=10;
         s.scale_asr[2]=1;
         s.base_scale=1.5;
         s.location=p;
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 4
      for(int i=0;i<5;++i)
      {
         Sphere s;
         s.appear_time=(tt-10)+i;
         s.disappear_time=tt+5+ran.doub()*2+i;
         s.material=0;
         s.scale_asr[0]=8;
         s.scale_asr[1]=10;
         s.scale_asr[2]=4;
         s.base_scale=0.6;
         s.location=p+Vec3(0,0.15,0);
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }

      for(int i=0;i<5;++i)
      {
         {
            Sphere s;
            s.appear_time=tt-3;
            s.disappear_time=tt+5+ran.doub()*2;
            s.material=0;
            s.scale_asr[0]=1;
            s.scale_asr[1]=10;
            s.scale_asr[2]=1;
            s.base_scale=1.5*blastoff_smoke_scale;
            s.location=p;
            float a=ran.doub()*2*M_PI;
            s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f*blastoff_smoke_spread;
            s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
            g_spheres.push_back(s);
         }
         {
            Sphere s;
            s.appear_time=tt-3;
            s.disappear_time=tt+5+ran.doub()*2;
            s.material=0;
            s.scale_asr[0]=8;
            s.scale_asr[1]=10;
            s.scale_asr[2]=4;
            s.base_scale=0.6*blastoff_smoke_scale;
            s.location=p+Vec3(0,0.15,0);
            float a=ran.doub()*2*M_PI;
            s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
            s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
            g_spheres.push_back(s);
         }
      }
   }

   {
      const Vec3 p = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, 0.0).column(3);

      // blast off smoke 1
      for(int i=0;i<10;++i)
      {
         Sphere s;
         s.appear_time=tt-21;
         s.disappear_time=tt+5+ran.doub()*2;
         s.material=0;
         s.scale_asr[0]=1;
         s.scale_asr[1]=10;
         s.scale_asr[2]=1;
         s.base_scale=1.5*blastoff_smoke_scale;
         s.location=p;
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f*blastoff_smoke_spread;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 2
      for(int i=0;i<5;++i)
      {
         Sphere s;
         s.appear_time=tt-21;
         s.disappear_time=tt+5+ran.doub()*2;
         s.material=0;
         s.scale_asr[0]=8;
         s.scale_asr[1]=10;
         s.scale_asr[2]=4;
         s.base_scale=0.6*blastoff_smoke_scale;
         s.location=p+Vec3(0,0.15,0);
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 3
      for(int i=0;i<10;++i)
      {
         Sphere s;
         s.appear_time=(tt-21)+i*2;
         s.disappear_time=tt+5+ran.doub()*2+i*2;
         s.material=0;
         s.scale_asr[0]=1;
         s.scale_asr[1]=10;
         s.scale_asr[2]=1;
         s.base_scale=1.5;
         s.location=p;
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
      // blast off smoke 4
      for(int i=0;i<5;++i)
      {
         Sphere s;
         s.appear_time=(tt-21)+i;
         s.disappear_time=tt+5+ran.doub()*2+i;
         s.material=0;
         s.scale_asr[0]=8;
         s.scale_asr[1]=10;
         s.scale_asr[2]=4;
         s.base_scale=0.6;
         s.location=p+Vec3(0,0.15,0);
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }

   }

   const Vec3 p0 = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, tt + 0.0).column(3);
   const Vec3 p1 = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, tt + 1.0).column(3);

   float smoke_column_scale=1.5;
   // smoke column
   for(int i=0;i<1100;++i)
   {
      Sphere s;
      Vec3 p;

      const double tofs=float(i)*.03;

      s.appear_time=tt+tofs;
      s.disappear_time=s.appear_time+2;

      if(i<0)
      {
         p=p0+(p1-p0)*tofs;
      }
      else
      {
         const Mat4 rocket_global_xfrm = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, s.appear_time);
         p = rocket_global_xfrm .column(3);
      }

      s.material=0;
      s.scale_asr[0]=4;
      s.scale_asr[1]=10;
      s.scale_asr[2]=3;
      s.base_scale=mix(0.9,1.1,ran.doub())*smoke_column_scale;
      s.location=p+Vec3(0,2,0);
      float a=ran.doub()*2*M_PI;
      s.direction=Vec3(cos(a),sin(a),0)*mix(0.5,1.0,ran.doub())+Vec3(0,-0.1,0);
      s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
      g_spheres.push_back(s);
   }

}
*/

static void createSpheres()
{
   assert(g_orion);
   assert(g_orion->entities.size()==node_count);
   assert(rocket_node_index < g_orion->entities.size());
   const Mat4 rocket_global_xfrm = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, 0);
   const Vec3 p = rocket_global_xfrm .column(3);
   Ran ran(65423);

   for(int i=0;i<10;++i)
   {
      Sphere s;
      s.appear_time=g_rocket_liftoff_time-3;
      s.disappear_time=g_rocket_liftoff_time+5+ran.doub()*2;
      s.material=0;
      s.scale_asr[0]=1;
      s.scale_asr[1]=10;
      s.scale_asr[2]=1;
      s.base_scale=1.5;
      s.location=p;
      float a=ran.doub()*2*M_PI;
      s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*1.5f;
      s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
      g_spheres.push_back(s);
   }
   for(int i=0;i<5;++i)
   {
      Sphere s;
      s.appear_time=g_rocket_liftoff_time-3;
      s.disappear_time=g_rocket_liftoff_time+5+ran.doub()*2;
      s.material=0;
      s.scale_asr[0]=8;
      s.scale_asr[1]=10;
      s.scale_asr[2]=4;
      s.base_scale=0.6;
      s.location=p+Vec3(0,0.15,0);
      float a=ran.doub()*2*M_PI;
      s.direction=Vec3(cos(a),0,sin(a))*mix(3,5,ran.doub())*2.0f;
      s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
      g_spheres.push_back(s);
   }
   for(int i=0;i<200;++i)
   {
      Sphere s;
      s.appear_time=g_rocket_liftoff_time+i*0.115;
      s.disappear_time=s.appear_time+4;
      s.material=0;
      s.scale_asr[0]=1;
      s.scale_asr[1]=10;
      s.scale_asr[2]=3;
      s.base_scale=mix(0.9,1.1,ran.doub());
      s.location=p+Vec3(0,float(i)*0.75+1.0,0);
      float a=ran.doub()*2*M_PI;
      s.direction=Vec3(cos(a),0,sin(a))*mix(0.5,1.0,ran.doub())+Vec3(0,-0.1,0);
      s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
      g_spheres.push_back(s);
   }
   for(int i=0;i<200;++i)
   {
      //if(i>10)
      {
         Sphere s;
         s.appear_time=(g_rocket_liftoff_time+14)+i*0.115;
         s.disappear_time=s.appear_time+4;
         s.material=0;
         s.scale_asr[0]=1;
         s.scale_asr[1]=10;
         s.scale_asr[2]=10;
         s.base_scale=mix(0.9,1.1,ran.doub());
         s.location=p+Vec3(0,float(i)*0.75+1.0,0);
         float a=ran.doub()*2*M_PI;
         s.direction=Vec3(cos(a),0,sin(a))*mix(0.5,1.0,ran.doub())+Vec3(0,-0.1,0);
         s.euler_angles=Vec3(ran.doub()*2*M_PI,ran.doub()*2*M_PI,ran.doub()*2*M_PI);
         g_spheres.push_back(s);
      }
   }
}

static void createSphereVBO()
{
   const uint32 vcount = 12;
   const uint32 tcount = 20;

   GLfloat vertices[vcount * 3];
   GLubyte indices[tcount * 3];

   g_sphere_index_count = tcount*3;

	GLfloat a2 = 0.5;
	GLfloat h  = (1.0+std::sqrt(5.0))/4.0;

	makeVertex(vertices,  0, 0.0,+a2,-h);
	makeVertex(vertices,  1,0.0,-a2,-h);
	makeVertex(vertices,  2,+h,0.0,-a2);
	makeVertex(vertices,  3,+h,0.0,+a2);
	makeVertex(vertices,  4,0.0,+a2,+h);
	makeVertex(vertices,  5,0.0,-a2,+h);
	makeVertex(vertices,  6,-h,0.0,+a2);
	makeVertex(vertices,  7,-h,0.0,-a2);
	makeVertex(vertices,  8,+a2,+h,0.0);
	makeVertex(vertices,  9,-a2,+h,0.0);
	makeVertex(vertices, 10,-a2,-h,0.0);
	makeVertex(vertices, 11,+a2,-h,0.0);

	makeTriangle(indices, 0, 9,8,0);
	makeTriangle(indices,  1,8,2,0);
	makeTriangle(indices,  2,8,3,2);
	makeTriangle(indices,  3,8,4,3);
	makeTriangle(indices,  4,8,9,4);
	makeTriangle(indices,  5,4,9,6);
	makeTriangle(indices,  6,6,9,7);
	makeTriangle(indices,  7,9,0,7);

	makeTriangle(indices,  8,11,10,1);
	makeTriangle(indices,  9,2 ,11,1);
	makeTriangle(indices, 10,2 ,3 ,11);
	makeTriangle(indices, 11,3 ,5 ,11);
	makeTriangle(indices, 12,5 ,10,11);
	makeTriangle(indices, 13,5 ,6 ,10);
	makeTriangle(indices, 14,6 ,7 ,10);
	makeTriangle(indices, 15,7 ,1 ,10);

	makeTriangle(indices, 16,1,7,0);
	makeTriangle(indices, 17,2,1,0);
	makeTriangle(indices, 18,4,5,3);
	makeTriangle(indices, 19,4,6,5);

   glGenBuffers(1, &g_sphere_vbo);
   glGenBuffers(1, &g_sphere_ebo);
   glBindBuffer(GL_ARRAY_BUFFER, g_sphere_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_sphere_ebo);
   my_glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
   my_glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("creating sphere VBO");
}

static uint32 subdivideMesh(const GLfloat* source_vertices, const uint32* source_indices, const uint32* source_materials, uint32 source_vertex_count, uint32 source_index_count,
                          GLfloat* dest_vertices, uint32* dest_indices, uint32* dest_materials, uint32* dest_vertex_count, uint32* dest_index_count, uint32 target_triangle_count, float max_triangle_area)
{
   assert(dest_vertex_count);
   assert(dest_index_count);
   assert(dest_vertices);
   assert(dest_indices);
   assert(source_vertices);
   assert(source_indices);
   const uint32 source_triangle_count = source_index_count / 3;
   std::map<uint32, uint32> edge_to_vertex;
   std::map<uint32, uint32> vertex_map;
   std::vector<float> triangle_areas(source_triangle_count);
   *dest_vertex_count = 0;
   *dest_index_count = 0;
   for(uint32 i = 0; i < source_index_count; i += 3)
   {
      const uint32* tri = source_indices + i;
      for(uint32 j = 0; j < 3; ++j)
      {
         uint32 va = source_indices[i + j];
         uint32 vb = source_indices[i + ((j + 1) % 3)];
         assert(va != vb);
         assert(va < 65536);
         assert(vb < 65536);

         if(vertex_map.find(va) == vertex_map.end())
         {
            for(uint32 k = 0; k < 3; ++k)
               dest_vertices[(*dest_vertex_count) * 3 + k] = source_vertices[va * 3 + k];
            vertex_map[va] = (*dest_vertex_count);
            ++(*dest_vertex_count);
         }

         if(va > vb)
            std::swap(va, vb);
         const uint32 edge_idx = va | (vb << 16);
         if(edge_to_vertex.find(edge_idx) == edge_to_vertex.end())
         {
            edge_to_vertex[edge_idx] = (*dest_vertex_count);
            for(uint32 k = 0; k < 3; ++k)
               dest_vertices[(*dest_vertex_count) * 3 + k] = mix(source_vertices[va * 3 + k], source_vertices[vb * 3 + k], 0.5f);
            ++(*dest_vertex_count);
         }
      }

      const Vec3 pa = Vec3(source_vertices[tri[0]*3+0],source_vertices[tri[0]*3+1],source_vertices[tri[0]*3+2]);
      const Vec3 pb = Vec3(source_vertices[tri[1]*3+0],source_vertices[tri[1]*3+1],source_vertices[tri[1]*3+2]);
      const Vec3 pc = Vec3(source_vertices[tri[2]*3+0],source_vertices[tri[2]*3+1],source_vertices[tri[2]*3+2]);
      triangle_areas[i / 3] = ((pb - pa).cross(pc - pa)).length() * 0.5f;
   }

   assert(source_triangle_count < target_triangle_count || target_triangle_count == 0);
   uint32 triangles_to_add = (target_triangle_count == 0) ? 0 : (target_triangle_count - source_triangle_count);
   uint32 num_subdivided = 0;
   for(uint32 i = 0; i < source_index_count; i += 3)
   {
      const uint32* tri = source_indices + i;
      const uint32 e0 = std::min(tri[0], tri[1]) | (std::max(tri[0], tri[1]) << 16);
      const uint32 e1 = std::min(tri[1], tri[2]) | (std::max(tri[1], tri[2]) << 16);
      const uint32 e2 = std::min(tri[2], tri[0]) | (std::max(tri[2], tri[0]) << 16);
      assert(edge_to_vertex.find(e0) != edge_to_vertex.end());
      assert(edge_to_vertex.find(e1) != edge_to_vertex.end());
      assert(edge_to_vertex.find(e2) != edge_to_vertex.end());
      assert(vertex_map.find(tri[0]) != vertex_map.end());
      assert(vertex_map.find(tri[1]) != vertex_map.end());
      assert(vertex_map.find(tri[2]) != vertex_map.end());

      const uint32 mat = source_materials[i / 3];

      if(triangles_to_add >= 3 || triangle_areas[i / 3] > max_triangle_area)
      {
         // make a triforce
         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = vertex_map[tri[0]];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e0];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e0];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[1]];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e1];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e1];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[2]];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e0];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e1];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];

         if(triangles_to_add > 0)
            triangles_to_add -= 3;

         ++num_subdivided;
      }
      else if(triangles_to_add >= 2)
      {
         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = vertex_map[tri[0]];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e0];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e0];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[1]];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = vertex_map[tri[1]];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[2]];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e2];

         if(triangles_to_add > 0)
            triangles_to_add -= 2;

         ++num_subdivided;
      }
      else if(triangles_to_add >= 1)
      {
         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = vertex_map[tri[0]];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[1]];
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e1];

         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = edge_to_vertex[e1];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[2]];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[0]];

         if(triangles_to_add > 0)
            triangles_to_add -= 1;

         ++num_subdivided;
      }
      else
      {
         dest_materials[(*dest_index_count) / 3] = mat;
         dest_indices[(*dest_index_count)++] = vertex_map[tri[0]];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[1]];
         dest_indices[(*dest_index_count)++] = vertex_map[tri[2]];
      }
   }

   return num_subdivided;
}

static void bakeNode(uint32 node_index, GLfloat* dest_vertices, uint32* dest_indices, uint32* dest_materials, uint32* dest_vertex_count, uint32* dest_index_count)
{
   assert(dest_vertex_count);
   assert(dest_index_count);
   assert(dest_vertices);
   assert(dest_indices);
   (*dest_vertex_count) = 0;
   (*dest_index_count) = 0;

    std::stack<uint32> node_stack;
    node_stack.push(node_index);
    while(!node_stack.empty())
    {
        const uint32 i = node_stack.top();
        node_stack.pop();

         const uint32 child_count = node_child_offsets[i + 1] - node_child_offsets[i];
         const uint32 mesh_count = node_mesh_offsets[i + 1] - node_mesh_offsets[i];

         if(mesh_count > 0)
         {
            const Mat4& global_xfrm = g_orion->getEntityGlobalTransformationForTime(i, 0);

            std::map<uint32, uint32> vertex_map;

            for(uint32 j = 0; j < mesh_count; ++j)
            {
               const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[i] + j];
               assert(mesh_index < g_orion->mesh_cache.size());

               const MeshCacheItem* m = g_orion->mesh_cache[mesh_index];
               assert(m);

               if(m->count==0)
                  continue;

               const uint32 vertex_stride = 6;

               assert((m->count % 3) == 0);

               for(uint32 k=0;k<m->count;++k)
               {
                  const uint32 idx = indices[mesh_index_bases[mesh_index] + k];
                  dest_materials[(*dest_index_count) / 3] = mesh_materials[mesh_index];
                  if(vertex_map.find(idx) == vertex_map.end())
                  {
                     const GLfloat* v = vertices + idx * vertex_stride;
                     const Vec3 vertex = global_xfrm * Vec3(v[0], v[1], v[2]);
                     assert((*dest_vertex_count) < g_subdiv_max_vertices);
                     dest_vertices[(*dest_vertex_count) * 3 + 0] = vertex.x;
                     dest_vertices[(*dest_vertex_count) * 3 + 1] = vertex.y;
                     dest_vertices[(*dest_vertex_count) * 3 + 2] = vertex.z;
                     vertex_map[idx] = (*dest_vertex_count);
                     assert((*dest_index_count) < g_subdiv_max_indices);
                     dest_indices[(*dest_index_count)] = (*dest_vertex_count);
                     ++(*dest_index_count);
                     ++(*dest_vertex_count);
                  }
                  else
                  {
                     assert((*dest_index_count) < g_subdiv_max_indices);
                     dest_indices[(*dest_index_count)] = vertex_map[idx];
                     ++(*dest_index_count);
                  }
               }
            }

         }

        for(uint32 j = 0; j < child_count; ++j)
        {
            const uint32 c = node_child_indices[node_child_offsets[i] + j];
            node_stack.push(c);
        }
    }

}

static int subdiv_material_comp_func(const void* a, const void* b)
{
   const GLushort* ia = (const GLushort*)a;
   const GLushort* ib = (const GLushort*)b;
   return ia[6] - ib[6];
}

static void createSubdividedVillage()
{
   uint32 new_vertex_count = 0, new_index_count = 0;
   GLushort* indices = NULL;
   uint32 final_triangle_count = 0;
   uint32 first_mesh_vertex_count = 0;
   uint32 final_vertex_count = 0;
   GLfloat* vertices = new GLfloat[g_subdiv_max_vertices * 3];

   {

      bakeNode(village_node_index, g_subdiv_scratch_vertices[0], g_subdiv_scratch_indices[0], g_subdiv_scratch_triangle_materials[0], &new_vertex_count, &new_index_count);

/*
      {
         FILE* obj=fopen("subdiv0.obj","w");
         for(uint32 i=0;i<new_vertex_count;++i)
            fprintf(obj,"v %f %f %f\n",g_subdiv_scratch_vertices[0][i*3+0],g_subdiv_scratch_vertices[0][i*3+1],g_subdiv_scratch_vertices[0][i*3+2]);
         for(uint32 i=0;i<new_index_count;i+=3)
            fprintf(obj,"f %d %d %d\n",g_subdiv_scratch_indices[0][i+0]+1,g_subdiv_scratch_indices[0][i+1]+1,g_subdiv_scratch_indices[0][i+2]+1);
         fclose(obj);
      }
*/

      uint32 flipper = 0;
      uint32 n = 0;

      do
      {
         n = subdivideMesh(g_subdiv_scratch_vertices[flipper], g_subdiv_scratch_indices[flipper], g_subdiv_scratch_triangle_materials[flipper], new_vertex_count, new_index_count,
                       g_subdiv_scratch_vertices[flipper^1], g_subdiv_scratch_indices[flipper^1], g_subdiv_scratch_triangle_materials[flipper^1], &new_vertex_count, &new_index_count, 0, 0.1f);
         flipper ^= 1;
      } while(n > 0);

      assert((new_index_count % 3) == 0);
      final_triangle_count = new_index_count / 3;
      indices = new GLushort[final_triangle_count * 7];

      for(uint32 i=0;i<final_triangle_count;++i)
      {
         assert(g_subdiv_scratch_indices[flipper][i*3+0] < 65536);
         assert(g_subdiv_scratch_indices[flipper][i*3+1] < 65536);
         assert(g_subdiv_scratch_indices[flipper][i*3+2] < 65536);
         indices[i * 7 + 0] = g_subdiv_scratch_indices[flipper][i*3+0];
         indices[i * 7 + 2] = g_subdiv_scratch_indices[flipper][i*3+1];
         indices[i * 7 + 4] = g_subdiv_scratch_indices[flipper][i*3+2];
         indices[i * 7 + 6] = g_subdiv_scratch_triangle_materials[flipper][i];
      }

      first_mesh_vertex_count = new_vertex_count;
      memcpy(vertices, g_subdiv_scratch_vertices[flipper], first_mesh_vertex_count * 3 * sizeof(GLfloat));

/*
      {
         FILE* obj=fopen("subdiv1.obj","w");
         for(uint32 i=0;i<new_vertex_count;++i)
            fprintf(obj,"v %f %f %f\n",g_subdiv_scratch_vertices[flipper][i*3+0],g_subdiv_scratch_vertices[flipper][i*3+1],g_subdiv_scratch_vertices[flipper][i*3+2]);
         for(uint32 i=0;i<new_index_count;i+=3)
            fprintf(obj,"f %d %d %d\n",g_subdiv_scratch_indices[flipper][i+0]+1,g_subdiv_scratch_indices[flipper][i+1]+1,g_subdiv_scratch_indices[flipper][i+2]+1);
         fclose(obj);
      }
*/

   }

   // shuffle
   {
      Ran ran(77812);
      for(uint32 i=0;i<final_triangle_count;++i)
      {
         const uint32 a = ran.int32() % final_triangle_count;
         for(uint32 j=0;j<7;++j)
            std::swap(indices[a * 7 + j], indices[i * 7 + j]);
      }
   }

   {
      bakeNode(rocket_node_index, g_subdiv_scratch_vertices[0], g_subdiv_scratch_indices[0], g_subdiv_scratch_triangle_materials[0], &new_vertex_count, &new_index_count);

/*
      {
         FILE* obj=fopen("subdiv0.obj","w");
         for(uint32 i=0;i<new_vertex_count;++i)
            fprintf(obj,"v %f %f %f\n",g_subdiv_scratch_vertices[0][i*3+0],g_subdiv_scratch_vertices[0][i*3+1],g_subdiv_scratch_vertices[0][i*3+2]);
         for(uint32 i=0;i<new_index_count;i+=3)
            fprintf(obj,"f %d %d %d\n",g_subdiv_scratch_indices[0][i+0]+1,g_subdiv_scratch_indices[0][i+1]+1,g_subdiv_scratch_indices[0][i+2]+1);
         fclose(obj);
      }
*/
      assert(final_triangle_count > 0);

      uint32 flipper = 0;
      uint32 n = 0;

      do
      {
         n = subdivideMesh(g_subdiv_scratch_vertices[flipper], g_subdiv_scratch_indices[flipper], g_subdiv_scratch_triangle_materials[flipper], new_vertex_count, new_index_count,
                       g_subdiv_scratch_vertices[flipper^1], g_subdiv_scratch_indices[flipper^1], g_subdiv_scratch_triangle_materials[flipper^1], &new_vertex_count, &new_index_count, final_triangle_count, 1e2f);
         flipper ^= 1;
      } while(n > 0 && (new_index_count / 3) < final_triangle_count);

      //log("(new_index_count / 3) = %d\n",(new_index_count / 3));
      //log("final_triangle_count = %d\n",final_triangle_count);

      assert((new_index_count % 3) == 0);
      assert((new_index_count / 3) == final_triangle_count);
      assert(indices);

      for(uint32 i=0;i<final_triangle_count;++i)
      {
         const uint32 a = g_subdiv_scratch_indices[flipper][i*3+0] + first_mesh_vertex_count;
         const uint32 b = g_subdiv_scratch_indices[flipper][i*3+1] + first_mesh_vertex_count;
         const uint32 c = g_subdiv_scratch_indices[flipper][i*3+2] + first_mesh_vertex_count;
         assert(a < 65536);
         assert(b < 65536);
         assert(c < 65536);
         indices[i * 7 + 1] = a;
         indices[i * 7 + 3] = b;
         indices[i * 7 + 5] = c;
         indices[i * 7 + 6] += g_subdiv_scratch_triangle_materials[flipper][i] * num_materials;
         g_material_subdiv_counts[indices[i * 7 + 6]] += 1;
      }

      assert((first_mesh_vertex_count + new_vertex_count) < g_subdiv_max_vertices);
      memcpy(vertices + first_mesh_vertex_count * 3, g_subdiv_scratch_vertices[flipper], new_vertex_count * 3 * sizeof(GLfloat));
      final_vertex_count = first_mesh_vertex_count + new_vertex_count;

/*
      {
         FILE* obj=fopen("subdiv1.obj","w");
         for(uint32 i=0;i<new_vertex_count;++i)
            fprintf(obj,"v %f %f %f\n",g_subdiv_scratch_vertices[flipper][i*3+0],g_subdiv_scratch_vertices[flipper][i*3+1],g_subdiv_scratch_vertices[flipper][i*3+2]);
         for(uint32 i=0;i<new_index_count;i+=3)
            fprintf(obj,"f %d %d %d\n",g_subdiv_scratch_indices[flipper][i+0]+1,g_subdiv_scratch_indices[flipper][i+1]+1,g_subdiv_scratch_indices[flipper][i+2]+1);
         fclose(obj);
      }
*/
  }


   qsort(indices, final_triangle_count, sizeof(indices[0]) * 7, subdiv_material_comp_func);

   // in-place removal of material index
   for(uint32 i=0;i<final_triangle_count;++i)
      for(uint32 j=0;j<6;++j)
         indices[i*6+j] = indices[i*7+j];


   g_final_subdiv_triangle_count = final_triangle_count;

   glGenBuffers(1, &g_subdiv_vbo);
   glGenBuffers(1, &g_subdiv_ebo);
   glBindBuffer(GL_ARRAY_BUFFER, g_subdiv_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_subdiv_ebo);
   my_glBufferData(GL_ARRAY_BUFFER, final_vertex_count * 3 * sizeof(GLfloat), vertices, GL_STATIC_DRAW);
   my_glBufferData(GL_ELEMENT_ARRAY_BUFFER, g_final_subdiv_triangle_count * 6 * sizeof(indices[0]), indices, GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("creating subdiv buffers");

   delete[] vertices;
   delete[] indices;
}

static void renderSubdiv()
{
   glBindBuffer(GL_ARRAY_BUFFER, g_subdiv_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_subdiv_ebo);
   glEnableVertexAttribArray(0);
   glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,0,(const GLvoid*)0);
   glDrawElements(GL_TRIANGLES_ADJACENCY,g_final_subdiv_triangle_count * 6,GL_UNSIGNED_SHORT,(const GLvoid*)0);
   glDisableVertexAttribArray(0);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   CHECK_FOR_GL_ERRORS("rendering subdived meshes");
}


static void createBlossoms()
{
   Ran ran(4323);
   const uint32 i=blossom_node_index;
   const uint32 child_count = node_child_offsets[i + 1] - node_child_offsets[i];
   for(uint32 j = 0; j < child_count; ++j)
   {
      const uint32 c = node_child_indices[node_child_offsets[i] + j];
      const uint32 mesh_count = node_mesh_offsets[c + 1] - node_mesh_offsets[c];

      if(mesh_count==0)
         continue;

      const Mat4 global_xfrm = g_orion->getEntityGlobalTransformationForTime(c, g_blossom_time);

      std::map<GLushort, bool> visited;
      for(uint32 j = 0; j < mesh_count; ++j)
      {
         const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[c] + j];
         assert(mesh_index < g_orion->mesh_cache.size());
         const MeshCacheItem* m = g_orion->mesh_cache[mesh_index];
         assert(m);

         if(m->count==0)
            continue;

         const uint32 vertex_stride = 6;

         for(uint32 k=0;k<m->count;++k)
         {
            if(ran.doub()<0.01)
            {
               const GLushort idx = indices[mesh_index_bases[mesh_index] + k];
               if(visited[idx])
                  continue;
               visited[idx]=true;
               const GLfloat* v = vertices + idx * vertex_stride;
               if(v[2]>-10)
               {
                  const Vec3 vertex = global_xfrm * Vec3(v[0], v[1], v[2]);
                  //g_blossom_locations.push_back(vertex.x);
                  //g_blossom_locations.push_back(vertex.y);
                  //g_blossom_locations.push_back(vertex.z);
                  g_blossom_locations.push_back(vertex);
               }
            }
         }
      }
   }
   std::random_shuffle(g_blossom_locations.begin(),g_blossom_locations.end());
}


static void createNoiseTexture()
{
    {
       static const int w = 32, h = 32;
       glGenTextures(1, &g_noise_tex);
       glBindTexture(GL_TEXTURE_2D, g_noise_tex);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
       my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_RGBA8, w, h);
       GLubyte pix[w*h*4];
       Ran r(314159);
       for(int i=0;i<w*h*4;++i)
          pix[i]=r.int32()&255;
       glTexSubImage2D(GL_TEXTURE_2D,0,0,0,w,h,GL_RGBA,GL_UNSIGNED_BYTE,pix);
       glBindTexture(GL_TEXTURE_2D, 0);
    }

    {
       static const int num_levels = 10;
       int w = 1<<(num_levels-1), h = w;
       glGenTextures(1, &g_noise_tex2);
       glBindTexture(GL_TEXTURE_2D, g_noise_tex2);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 4);
       my_glTexStorage2D(GL_TEXTURE_2D, num_levels, GL_RGBA8, w, h);
       uint32 pix[w*h];
       Ran r(314159);
       for(int i=0;i<w*h;++i)
          pix[i]=r.int32() & 0xffffff;

       glTexSubImage2D(GL_TEXTURE_2D,0,0,0,w,h,GL_RGBA,GL_UNSIGNED_BYTE,pix);

      for(int l=1;l<num_levels;++l)
      {

       for(int y=0;y<h;y+=2)
         for(int x=0;x<w;x+=2)
         {
            pix[(y/2)*w/2+x/2]=((pix[(y+0)*w+(x+0)]&0xfcfcfc) +
                                (pix[(y+0)*w+(x+1)]&0xfcfcfc) +
                                (pix[(y+1)*w+(x+0)]&0xfcfcfc) +
                                (pix[(y+1)*w+(x+1)]&0xfcfcfc)) >> 2;
         }

       glTexSubImage2D(GL_TEXTURE_2D,l,0,0,w/2,h/2,GL_RGBA,GL_UNSIGNED_BYTE,pix);
       w/=2;
       h/=2;
      }

       glBindTexture(GL_TEXTURE_2D, 0);
    }
}

static void setTextureUnit(GLint unit, GLenum binding_point, GLuint name)
{
    assert(name != 0);
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(binding_point, name);
}

static const float particle_velocity_map_seconds = 30.0f;

static Vec3 particleVelocity(float t)
{
  return mix<Vec3>(mix<Vec3>(mix<Vec3>(mix<Vec3>(Vec3(0.02,0.0,0.05) * 0.7, Vec3(0.05,0.0,0.02)*1.6,clamp((t-4.0)/20.0,0.0,1.0)),Vec3(0.02,0.0,-0.05),clamp((t-7.0)/9.0,0.0,1.0)) ,  Vec3(0.08,0.0,0.02)*0.75, clamp((t-13.0)/5.0,0.0,1.0)),
                    Vec3(0.6,0.01,0.02)*0.5, clamp((t-21.0)/4.0,0.0,1.0)) * (1.0-smoothstep(20.0f,25.0f,t));
}

static void createParticleVelocityMap()
{
   static const uint32 num_samples = 2048;
   glGenTextures(1,&particle_velocity_map);
   glBindTexture(GL_TEXTURE_1D, particle_velocity_map);
   glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   my_glTexStorage1D(GL_TEXTURE_1D, 1, GL_RGB16F, num_samples);
   GLfloat samples[num_samples * 3];
   for(uint32 i = 0; i < num_samples; ++i)
   {
      Vec3 s = particleVelocity(float(i) / float(num_samples) * float(particle_velocity_map_seconds));
      samples[i * 3 + 0] = s.x;
      samples[i * 3 + 1] = s.y;
      samples[i * 3 + 2] = s.z;
   }
   glTexSubImage1D(GL_TEXTURE_1D, 0, 0, num_samples, GL_RGB, GL_FLOAT, samples);
   glBindTexture(GL_TEXTURE_1D, 0);
   CHECK_FOR_GL_ERRORS("creating particle velocity map");
}

static int chunk_material_comp_func(const void* a, const void* b)
{
   const GLuint* ia = (const GLuint*)a;
   const GLuint* ib = (const GLuint*)b;
   return ia[24] - ib[24];
}

void createExplosionChunks()
{
   assert(g_orion);
   assert(g_orion->entities.size()==node_count);
   std::vector<GLfloat> chunk_vertices;
   std::vector<GLuint> chunk_indices;
   std::vector<GLfloat> chunk_initial_locations;

   for(uint32 cn = 0; cn < chunked_node_count; ++cn)
   {
      const uint32 i = chunked_nodes[cn];
      const Entity* ent = g_orion->entities[i];

      const uint32 mesh_count = node_mesh_offsets[i + 1] - node_mesh_offsets[i];

      if(mesh_count==0)
         continue;

      const Mat4 global_xfrm = g_orion->getEntityGlobalTransformationForTime(i, g_explosion_time);

      for(uint32 j = 0; j < mesh_count; ++j)
      {
         const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[i] + j];
         assert(mesh_index < g_orion->mesh_cache.size());
         const MeshCacheItem* m = g_orion->mesh_cache[mesh_index];
         assert(m);

         const uint32 triangle_count = m->count / 3;

         if(triangle_count==0)
            continue;

         const uint32 vertex_stride = 6;

         for(uint32 k=0;k<triangle_count;++k)
         {
            const GLushort* tri = indices + mesh_index_bases[mesh_index] + k * 3;
            Vec3 ns[3];
            Vec3 vs[3];
            for(uint32 m=0;m<3;++m)
            {
               const GLfloat* nd = vertices + tri[m] * vertex_stride + 3;
               const GLfloat* vd = vertices + tri[m] * vertex_stride + 0;
               ns[m] = Vec3(nd[0], nd[1], nd[2]);
               vs[m] = Vec3(vd[0], vd[1], vd[2]);
            }
            // transform to world space
            for(uint32 m=0;m<3;++m)
            {
               global_xfrm.transform(vs[m]);
               global_xfrm.transform3x3(ns[m]);
               ns[m]=ns[m]/sqrt(ns[m].lengthSquared());
            }
            const float e=2.0f;
            // create new vertices
            Vec3 vs2[6]={
               vs[0],
               vs[1],
               vs[2],
               vs[0]-ns[0]*e,
               vs[1]-ns[1]*e,
               vs[2]-ns[2]*e,
            };
            // transform to chunk space
            const Vec3 avg=(vs[0]+vs[1]+vs[2])/3.0f;

            // remove unwanted chunks
            if(avg.x<-60 || avg.x>+60 || avg.z<-75 || avg.z>+80)
               continue;

            const Mat4 world_to_chunk=Mat4::translation(Vec3(-avg));
            for(uint32 m=0;m<6;++m)
               world_to_chunk.transform(vs2[m]);

            const uint32 material = mesh_materials[mesh_index];
            assert(material < num_materials);

            g_material_chunk_counts[material]++;

            const uint32 base = chunk_vertices.size() / 3;
            // main faces
            chunk_indices.push_back(base + 0);
            chunk_indices.push_back(base + 1);
            chunk_indices.push_back(base + 2);
            chunk_indices.push_back(base + 5);
            chunk_indices.push_back(base + 4);
            chunk_indices.push_back(base + 3);
            for(uint32 m=0;m<6;++m)
            {
               chunk_vertices.push_back(vs2[m].x);
               chunk_vertices.push_back(vs2[m].y);
               chunk_vertices.push_back(vs2[m].z);
            }
            // edge faces
            for(uint32 m=0;m<3;++m)
            {
               const uint32 m2=(m+1)%3;

               chunk_indices.push_back(base + m);
               chunk_indices.push_back(base + m + 3);
               chunk_indices.push_back(base + m2 + 3);

               chunk_indices.push_back(base + m);
               chunk_indices.push_back(base + m2 + 3);
               chunk_indices.push_back(base + m2);
            }
            chunk_indices.push_back(material);
            chunk_initial_locations.push_back(avg.x);
            chunk_initial_locations.push_back(avg.y);
            chunk_initial_locations.push_back(avg.z);
            //log("%f, %f, %f\n",avg.x,avg.y,avg.z);
         }
      }
   }
   const uint32 chunk_count=chunk_initial_locations.size()/3;
   log("%d chunks\n",chunk_count);
   g_chunk_count=chunk_count;
   uint32 chunk_texture_size=1;
   while(chunk_texture_size*chunk_texture_size<chunk_count)
      chunk_texture_size<<=1;
   log("chunk_texture_size = %d\n",chunk_texture_size);
   const uint32 chunk_texture_data_length=chunk_texture_size*chunk_texture_size*3;
   glGenTextures(2,g_chunk_textures);
   glGenFramebuffers(2,g_chunk_fbos);
   while(chunk_initial_locations.size()<chunk_texture_data_length)
   {
      chunk_initial_locations.push_back(0);
   }
   assert(chunk_initial_locations.size()==chunk_texture_data_length);
   assert(chunk_indices.size()==chunk_count*25);

   qsort(&chunk_indices[0], chunk_indices.size() / 25, sizeof(chunk_indices[0]) * 25, chunk_material_comp_func);

   // in-place removal of material index
   for(uint i=0;i<chunk_count;++i)
      for(uint j=0;j<24;++j)
         chunk_indices[i*24+j]=chunk_indices[i*25+j];

   g_chunk_texture_size=chunk_texture_size;
   for(uint i=0;i<2;++i)
   {
      glBindTexture(GL_TEXTURE_2D,g_chunk_textures[i]);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
      my_glTexStorage2D(GL_TEXTURE_2D,1,GL_RGB32F,chunk_texture_size,chunk_texture_size);
      glTexSubImage2D(GL_TEXTURE_2D,0,0,0,chunk_texture_size,chunk_texture_size,GL_RGB,GL_FLOAT,&chunk_initial_locations[0]);
      glBindTexture(GL_TEXTURE_2D,0);
      glBindFramebuffer(GL_FRAMEBUFFER,g_chunk_fbos[i]);
      glFramebufferTexture2D(GL_FRAMEBUFFER,GL_COLOR_ATTACHMENT0,GL_TEXTURE_2D,g_chunk_textures[i],0);
      CHECK_FRAMEBUFFER(GL_FRAMEBUFFER,"chunks");
      glBindFramebuffer(GL_FRAMEBUFFER,0);
   }
   std::vector<GLfloat> texcoords;
   for(uint32 i=0;i<chunk_count;++i)
   {
      for(uint j=0;j<6;++j)
      {
         texcoords.push_back((GLfloat(i % chunk_texture_size)+0.5f)/GLfloat(chunk_texture_size));
         texcoords.push_back((GLfloat(i / chunk_texture_size)+0.5f)/GLfloat(chunk_texture_size));
      }
   }
   glGenBuffers(1,&g_chunk_vbo);
   glGenBuffers(1,&g_chunk_ebo);
   glBindBuffer(GL_ARRAY_BUFFER,g_chunk_vbo);
   assert((chunk_vertices.size()/3)==(texcoords.size()/2));
   g_chunk_vertex_data_length=chunk_vertices.size()*sizeof(chunk_vertices[0]);
   my_glBufferData(GL_ARRAY_BUFFER,g_chunk_vertex_data_length+texcoords.size()*sizeof(texcoords[0]),NULL,GL_STATIC_DRAW);
   glBufferSubData(GL_ARRAY_BUFFER,0,g_chunk_vertex_data_length,&chunk_vertices[0]);
   glBufferSubData(GL_ARRAY_BUFFER,g_chunk_vertex_data_length,texcoords.size()*sizeof(texcoords[0]),&texcoords[0]);
   glBindBuffer(GL_ARRAY_BUFFER,0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,g_chunk_ebo);
   my_glBufferData(GL_ELEMENT_ARRAY_BUFFER,chunk_indices.size()*sizeof(chunk_indices[0]),&chunk_indices[0],GL_STATIC_DRAW);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,0);
   CHECK_FOR_GL_ERRORS("creating chunk arrays");


   // set up the collision
   {
      uint32 i = revision_logo_node_index;
      const uint32 mesh_count = node_mesh_offsets[i + 1] - node_mesh_offsets[i];
      assert(mesh_count==1);
      const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[i]];
      assert(i < g_orion->entities.size());
      const Entity* ent = g_orion->entities[i];
      const Mat4 global_xfrm = g_orion->getEntityGlobalTransformationForTime(i, g_explosion_time);
      const Vec3 box_min(bounding_boxes[mesh_index*6+0],bounding_boxes[mesh_index*6+1],bounding_boxes[mesh_index*6+2]);
      const Vec3 box_max(bounding_boxes[mesh_index*6+3],bounding_boxes[mesh_index*6+4],bounding_boxes[mesh_index*6+5]);
      g_world_to_revision_logo_sdf = Mat4::translation(Vec3(-0.84,0.84,-1.0)) * Mat4::scale(Vec3(0.91+0.84,-2.0*0.84,2.0)) * Mat4::scale(Vec3(Vec3(1)/(box_max-box_min))) * Mat4::translation(Vec3(-box_min)) * global_xfrm.inverse();
   }
}

static void renderChunks()
{
   glBindBuffer(GL_ARRAY_BUFFER,g_chunk_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,g_chunk_ebo);
   glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,0,(const GLvoid*)0);
   glVertexAttribPointer(1,2,GL_FLOAT,GL_FALSE,0,(const GLvoid*)g_chunk_vertex_data_length);
   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);
   glDrawElements(GL_TRIANGLES,g_chunk_count*24,GL_UNSIGNED_INT,(const GLvoid*)0);
   glDisableVertexAttribArray(0);
   glDisableVertexAttribArray(1);
   glBindBuffer(GL_ARRAY_BUFFER,0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,0);
   CHECK_FOR_GL_ERRORS("rendering chunks");
}

static void create2015Texture()
{
   glGenTextures(1,&g_2015_tex);
   glBindTexture(GL_TEXTURE_2D, g_2015_tex);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
   my_glTexStorage2D(GL_TEXTURE_2D, 1,GL_RGBA8,image_2015.width,image_2015.height);
   glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, image_2015.width,image_2015.height,GL_RGBA,GL_UNSIGNED_BYTE,image_2015.pixel_data);
   glBindTexture(GL_TEXTURE_2D, 0);
   CHECK_FOR_GL_ERRORS("create2015Texture");
}


static void renderInfo()
{
   assert(g_info_vbo != 0);
   assert(g_info_ebo != 0);
   assert(g_2015_tex != 0);
   glEnable(GL_BLEND);
   glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_ALPHA);
   setTextureUnit(0,GL_TEXTURE_2D,g_2015_tex);
   glBindBuffer(GL_ARRAY_BUFFER, g_info_vbo);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_info_ebo);
   glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,sizeof(GLfloat)*4,(const GLvoid*)0);
   glVertexAttribPointer(1,2,GL_FLOAT,GL_FALSE,sizeof(GLfloat)*4,(const GLvoid*)(sizeof(GLfloat)*2));
   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);
   assert(g_info_index_count != 0);
   glDrawElements(GL_TRIANGLES,g_info_index_count,GL_UNSIGNED_BYTE,(const GLvoid*)0);
   glDisableVertexAttribArray(0);
   glDisableVertexAttribArray(1);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
   glDisable(GL_BLEND);
}

static void renderSkyBox()
{
   //glEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS);

   GLuint temp_texs[2] = { 0, 0 };
   glGenTextures(2, temp_texs);
   for(uint32 i=0;i<2;++i)
   {
      glBindTexture(GL_TEXTURE_2D, temp_texs[i]);
      my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_R8, g_sky_cubemap_size, g_sky_cubemap_size);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
   }

   glGenTextures(1, &g_sky_cubemap);
   glBindTexture(GL_TEXTURE_CUBE_MAP, g_sky_cubemap);
   my_glTexStorage2D(GL_TEXTURE_CUBE_MAP, 1, GL_R8, g_sky_cubemap_size, g_sky_cubemap_size);
   glTexParameteri(GL_TEXTURE_CUBE_MAP,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
   glTexParameteri(GL_TEXTURE_CUBE_MAP,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

   GLuint fbo = 0, rbo = 0;

   glGenFramebuffers(1, &fbo);
   glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo);
	glGenRenderbuffers(1, &rbo);
	glBindRenderbuffer(GL_RENDERBUFFER, rbo);
	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT32, g_sky_cubemap_size, g_sky_cubemap_size);
   glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, rbo);

   skybox_program.use();
   const float znear = 1.0f / 64.0f;
   glUniformMatrix4fv(skybox_program.transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, Mat4::frustum(-znear, +znear, -znear, +znear, znear, 256.0f).e);

   const Mat4 object_matrix = Mat4::translation(Vec3(0,-3,0));
   const float blur_radius = 0.0005;

   for(uint32 i=0;i<6;++i)
   {
      skybox_program.use();
      glUniformMatrix4fv(skybox_program.transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (g_cubemap_configs[i] * object_matrix).e);

      glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, temp_texs[0], 0);
      {
         const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
         glDrawBuffers(1, buffers);
      }
      CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "renderSkyBox: phase 1");
      glViewport(0, 0, g_sky_cubemap_size, g_sky_cubemap_size);
      glDepthMask(GL_TRUE);
      glColorMask(GL_TRUE, GL_FALSE, GL_FALSE, GL_FALSE);
      glEnable(GL_DEPTH_TEST);
      glDisable(GL_BLEND);
      glClearColor(0, 0, 0, 0);
      glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
      glBindBuffer(GL_ARRAY_BUFFER, 0);
      glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
      glEnableVertexAttribArray(0);
      glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, skymesh::vertices);
      glDrawElements(GL_TRIANGLES, skymesh::num_indices, GL_UNSIGNED_SHORT, skymesh::indices);
      glDisableVertexAttribArray(0);

      blur_program.use();
      blur_program.uniform1f("peripheral",0);
      blur_program.uniform2f("direction",blur_radius,0);
      glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, temp_texs[1], 0);
      {
         const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
         glDrawBuffers(1, buffers);
      }
      CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "renderSkyBox: phase 2");
      glDisable(GL_DEPTH_TEST);
      glDepthMask(GL_FALSE);
      glColorMask(GL_TRUE,GL_FALSE,GL_FALSE,GL_FALSE);
      setTextureUnit(0, GL_TEXTURE_2D, temp_texs[0]);
      setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
      glViewport(0, 0, g_sky_cubemap_size, g_sky_cubemap_size);
      renderQuad();

      blur_program.uniform2f("direction", 0.0, blur_radius);
      attachCubemapFace(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, i, g_sky_cubemap, 0);
      {
         const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
         glDrawBuffers(1, buffers);
      }
      CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "renderSkyBox: phase 3");
      glDisable(GL_DEPTH_TEST);
      glDepthMask(GL_FALSE);
      glColorMask(GL_TRUE,GL_FALSE,GL_FALSE,GL_FALSE);
      setTextureUnit(0, GL_TEXTURE_2D, temp_texs[1]);
      setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
      glViewport(0, 0, g_sky_cubemap_size, g_sky_cubemap_size);
      renderQuad();
   }

   glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);

   glBindTexture(GL_TEXTURE_2D, 0);
   glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
   glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
	glBindRenderbuffer(GL_RENDERBUFFER, 0);
   glDeleteFramebuffers(1, &fbo);
	glDeleteRenderbuffers(1, &rbo);
   glDeleteTextures(2, temp_texs);
   CHECK_FOR_GL_ERRORS("renderSkyBox");
}

static void createStaticParticles()
{
   const uint32 n = 100;

   GLfloat* init_pos_data=new GLfloat[g_staticparticles_size*g_staticparticles_size*3];

   Ran ran(3332);

   for(uint32 i=0;i<g_staticparticles_size*g_staticparticles_size;++i)
   {
      init_pos_data[i*3+0]=-35.3329+(ran.doub()-0.5)*5;
      init_pos_data[i*3+1]=34.5637;
      init_pos_data[i*3+2]=-3.0263+(ran.doub()-0.5)*5;
   }

   setTextureUnit(0, GL_TEXTURE_2D, staticparticle_framebuffer.texs[0]);
   glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, g_staticparticles_size, g_staticparticles_size, GL_RGB, GL_FLOAT, init_pos_data);

   for(uint32 i = 0; i < n; ++i)
   {
      staticparticle_framebuffer.bindForDraw();
      glDisable(GL_DEPTH_TEST);
      glDepthMask(GL_FALSE);
      glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
      setTextureUnit(0, GL_TEXTURE_2D, staticparticle_framebuffer.texs[0]);
      assert(collision_voxel_texture != 0);
      setTextureUnit(1, GL_TEXTURE_3D, collision_voxel_texture);
      setTextureUnit(2, GL_TEXTURE_2D, g_noise_tex2);
      particle_update_program.use();
      glViewport(0,0,g_staticparticles_size,g_staticparticles_size);
      renderQuad();
   }

   delete[] init_pos_data;
   init_pos_data = NULL;

   CHECK_FOR_GL_ERRORS("createStaticParticles");
}

static void createStarsTexture()
{
   glGenTextures(1, &g_stars_texture);
   assert(g_stars_texture);
   glBindTexture(GL_TEXTURE_2D, g_stars_texture);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   const int w = 1024, h = 1024;
   GLubyte* data = new GLubyte[w*h];
   std::fill(data, data+w*h, 0);
   const int num_stars = 2000;

   Ran ran(100);
   for(int i=0;i<num_stars;++i)
   {
      int x=ran.int32()%w,y=ran.int32()%h;
      data[x+y*w]=100+155*ran.doub();
   }

/*
   GLubyte* data2 = new GLubyte[w*h];

   for(int y=0;y<h;++y)
   {
      for(int x=0;x<w;++x)
      {
         int a=0;
         for(int k=-1;k<2;++k)
            for(int j=-1;j<2;++j)
            {
               int x=(x+j)%w,y=(y+k)%h;
               a+=data[x+y*w];
            }
         data2[x+y*w]=a/9;
      }
   }
   std::swap(data,data2);
*/

   my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_R8, w, h);
   glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, w, h, GL_RED, GL_UNSIGNED_BYTE, data);
   delete[] data;
   //delete[] data2;
   data = NULL;
   //data2 = NULL;
   glBindTexture(GL_TEXTURE_2D, 0);
}

static void createSkyscraperCubeMaps()
{
   //glEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS);
   glGenTextures(skyscraper_node_count, skyscraper_cubemap_texs);
   Viewport vp;
   g_orion->updateAnimations(g_explosion_time, vp);
   for(uint32 j=0;j<skyscraper_node_count;++j)
   {
      uint32 n = skyscraper_nodes[j];

      g_orion->entities[n]->invisible=true;
      const Mat4 global_xfrm = g_orion->getEntityGlobalTransformationForTime(n, g_explosion_time);

      GLuint temp_texs[2] = { 0, 0 };
      glGenTextures(2, temp_texs);
      for(uint32 i=0;i<2;++i)
      {
         glBindTexture(GL_TEXTURE_2D, temp_texs[i]);
         my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_R8, skyscraper_cubemap_size, skyscraper_cubemap_size);
         glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
         glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
         glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
         glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
      }

      glBindTexture(GL_TEXTURE_CUBE_MAP, skyscraper_cubemap_texs[j]);
      my_glTexStorage2D(GL_TEXTURE_CUBE_MAP, 1, GL_R8, skyscraper_cubemap_size, skyscraper_cubemap_size);
      glTexParameteri(GL_TEXTURE_CUBE_MAP,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
      glTexParameteri(GL_TEXTURE_CUBE_MAP,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

      GLuint fbo = 0, rbo = 0;

      glGenFramebuffers(1, &fbo);
      glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo);
      glGenRenderbuffers(1, &rbo);
      glBindRenderbuffer(GL_RENDERBUFFER, rbo);
      glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT32, skyscraper_cubemap_size, skyscraper_cubemap_size);
      glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, rbo);

      CameraConfig cc;
      const float znear = 1.0f / 8.0f;
      cc.view_to_clip = Mat4::frustum(-znear, +znear, -znear, +znear, znear, 2048.0f);

      skybox_program.use();

      const float blur_radius = 0.01;

      for(uint32 i=0;i<6;++i)
      {
         skybox_program.use();
         cc.world_to_view = g_cubemap_configs[i] * Mat4::translation(Vec3(-global_xfrm.column(3)));
         cc.world_to_clip = cc.view_to_clip * cc.world_to_view;

         glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, temp_texs[0], 0);
         {
            const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
            glDrawBuffers(1, buffers);
         }
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "createSkyscraperCubeMaps: phase 1");
         glViewport(0, 0, skyscraper_cubemap_size, skyscraper_cubemap_size);
         glDepthMask(GL_TRUE);
         glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_FALSE);
         glEnable(GL_DEPTH_TEST);
         glDisable(GL_BLEND);
         glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
         g_orion->renderEntities(cc, skybox_program.transform_properties, false);
         renderSpheres(cc,skybox_program.transform_properties,g_time);
         //glDrawElements(GL_TRIANGLES, skymesh::num_indices, GL_UNSIGNED_SHORT, skymesh::indices);

         blur_program.use();
         blur_program.uniform1f("peripheral",1);
         blur_program.uniform2f("direction",blur_radius,0);
         glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, temp_texs[1], 0);
         {
            const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
            glDrawBuffers(1, buffers);
         }
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "createSkyscraperCubeMaps: phase 2");
         glDisable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
         setTextureUnit(0, GL_TEXTURE_2D, temp_texs[0]);
         setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
         glViewport(0, 0, skyscraper_cubemap_size, skyscraper_cubemap_size);
         renderQuad();

         blur_program.uniform2f("direction", 0.0, blur_radius);
         attachCubemapFace(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, i, skyscraper_cubemap_texs[j], 0);
         {
            const GLenum buffers[1] = { GL_COLOR_ATTACHMENT0 };
            glDrawBuffers(1, buffers);
         }
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "createSkyscraperCubeMaps: phase 3");
         glDisable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
         setTextureUnit(0, GL_TEXTURE_2D, temp_texs[1]);
         setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
         glViewport(0, 0, skyscraper_cubemap_size, skyscraper_cubemap_size);
         renderQuad();
      }

      glBindTexture(GL_TEXTURE_2D, 0);
      glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
      glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
      glBindRenderbuffer(GL_RENDERBUFFER, 0);
      glDeleteFramebuffers(1, &fbo);
      glDeleteRenderbuffers(1, &rbo);
      glDeleteTextures(2, temp_texs);
      CHECK_FOR_GL_ERRORS("createSkyscraperCubeMaps");
   }
}

static const int NULL_bitcount=0;
#if DEVMODE
#define CREATE_PROGRAM(vs, gs, fs) createProgram(#vs, #gs, #fs);
#else
#define CREATE_PROGRAM(vs, gs, fs) createProgramFromSourceStrings(shaders::decodeShader(vs,vs##_bitcount),shaders::decodeShader(gs,gs##_bitcount),shaders::decodeShader(fs,fs##_bitcount));
#endif

int main(int argc, char *argv[])
{
#if CAPTURE
    ilInit();
#endif

    g_logfile = fopen("log.txt", "w");
    if(!g_logfile)
      return -2;
/*
   {
      Ran ran(100);
      int n=32;
      log("const vec2 samples[%d] = vec2[%d](\n",n,n);
      for(int i=0;i<n;++i)
      {
         float u,v;
         do{u=ran.doub()*2.0-1.0;v=ran.doub()*2.0-1.0;}while((u*u+v*v)>1.0);
         log("  vec2(%f, %f)%s\n",u,v,i==(n-1)?"":",");
      }
      log(");\n");
   }
*/

   if(SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_AUDIO))
   {
      const char* errorstring = SDL_GetError();
      if(errorstring)
         log("SDL_Init failed: '%s'.\n",errorstring);
      else
         log("SDL_Init failed and didn't provide an error message.\n");
      return -1;
   }

   SDL_WM_SetCaption("", NULL);

#define my_SDL_GL_SetAttribute(p,v) \
      if(SDL_GL_SetAttribute(p, v))\
      {\
         const char* errorstring = SDL_GetError();\
         if(errorstring)\
            log("SDL_GL_SetAttribute(" #p ") failed: '%s'.\n",errorstring);\
         else\
            log("SDL_GL_SetAttribute(" #p ") failed and didn't provide an error message.\n");\
         SDL_Quit();\
         return -1;\
      }

   my_SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
   my_SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
   my_SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
   my_SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 8);
   my_SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
   my_SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, true);
   my_SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);

   SDL_Surface* surf = NULL;

#if DEVMODE
   int fullscreen = 0;
#else
   int fullscreen = 1;
#endif

   int arg_start_frame=0;
   int arg_end_frame=0x800000;
   int arg_motion_blur=0;
/*
   if(argc>1)
   {
      arg_start_frame = atoi(argv[1]);
   }

   if(argc>2)
   {
      arg_end_frame = atoi(argv[2]);
   }

   if(argc>3)
   {
      arg_motion_blur = atoi(argv[3]);
   }
*/

   if(argc>1)
   {
      g_window_width = atoi(argv[1]);
      if(g_window_width <= 0)
      {
         log("Invalid window width.\n");
         SDL_Quit();
         return -1;
      }
   }

   if(argc>2)
   {
      g_window_height = atoi(argv[2]);
      if(g_window_height <= 0)
      {
         log("Invalid window height.\n");
         SDL_Quit();
         return -1;
      }
   }

   if(argc>3)
   {
      fullscreen = atoi(argv[3]);
   }

   if(!(surf = SDL_SetVideoMode(g_window_width, g_window_height, 0, SDL_DOUBLEBUF | SDL_HWSURFACE | SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : 0))))
   {
      const char* errorstring = SDL_GetError();
      if(errorstring)
         log("SDL_SetVideoMode failed: '%s'.\n",errorstring);
      else
         log("SDL_SetVideoMode failed and didn't provide an error message.\n");
      SDL_Quit();
      return -1;
   }

   g_window_width=surf->w;
   g_window_height=surf->h;

   log("Window dimensions = %d x %d.\n",g_window_width,g_window_height);
   log("Window fullscreen = %s.\n",fullscreen?"yes":"no");

#if DEVMODE
   SDL_ShowCursor(SDL_TRUE);
#else
   SDL_ShowCursor(SDL_FALSE);
#endif

   if(gl3wInit())
   {
      log("OpenGL init failed.\n");
      return -1;
   }

   if(!gl3wIsSupported(4, 0))
   {
      log("OpenGL 4.0 is not supported.\n");
      return -1;
   }

   CHECK_FOR_GL_ERRORS("OpenGL initialisation");

   {
      const GLubyte* gl_version = glGetString(GL_VERSION);
      const GLubyte* gl_vendor = glGetString(GL_VENDOR);
      const GLubyte* gl_renderer = glGetString(GL_RENDERER);
      const GLubyte* gl_extensions = glGetString(GL_EXTENSIONS);

      CHECK_FOR_GL_ERRORS("OpenGL string queries");

      log("GL_VERSION = '%s'\n", gl_version);
      log("GL_VENDOR = '%s'\n", gl_vendor);
      log("GL_RENDERER = '%s'\n", gl_renderer);
      //log("GL_EXTENSIONS = '%s'\n", gl_extensions);
   }

   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

    glGenRenderbuffers(1, &g_generic_depth_rbo);
    glBindRenderbuffer(GL_RENDERBUFFER, g_generic_depth_rbo);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT32, g_window_width, g_window_height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    CHECK_FOR_GL_ERRORS("creating generic depth renderbuffer");

   loadVoxels();

   createNoiseTexture();
   createQuadVBO();
   createCubeVBO();
   createSphereVBO();
   createInfoVBO();
   //createSmokeTrailGridVBO();

   g_orion = new Orion;
   {
      const Uint32 st=SDL_GetTicks();
   g_orion->uploadMeshes();
      log("uploadMeshes: %dms\n",SDL_GetTicks()-st);
   }
   {
      const Uint32 st=SDL_GetTicks();
   g_orion->createEntities();
      log("createEntities: %dms\n",SDL_GetTicks()-st);
   }

   static const int smooth_noise_length=512;
   static float smooth_noise_table[smooth_noise_length];
   {
      Ran ran(84892);
      for(int i=0;i<smooth_noise_length;++i)
      {
         smooth_noise_table[i]=ran.doub();
      }
   }

   createSpheres();
   shaders::initShaderSymbolDict();
   createDebris();

/*
   log("DECODING TEST\n");
   log("%s\n",shaders::decodeShader(raytrace_fragment_glsl, raytrace_fragment_glsl_bitcount).c_str());
*/

    const Uint32 shader_load_start_time=SDL_GetTicks();
#if ENCODE_SHADERS
    particle_update_program.name = createProgram("particle_update_vertex.glsl", NULL, "particle_update_fragment.glsl");
#else
    particle_update_program.name = CREATE_PROGRAM(particle_update_vertex_glsl, NULL, particle_update_fragment_glsl);
#endif
    particle_update_program.uniform1i("particle_tex", 0);
    particle_update_program.uniform1i("collision_map", 1);
    particle_update_program.uniformMatrix4fv("world_to_voxel", 1, GL_FALSE, world_to_voxel.e);
    particle_update_program.uniform3f("tsiz", voxel_texture_size.x, voxel_texture_size.y, voxel_texture_size.z);
    CHECK_FOR_GL_ERRORS("creating particle update program");

#if ENCODE_SHADERS
    particle_init_program.name = createProgram("particle_init_vertex.glsl", NULL, "particle_init_fragment.glsl");
#else
    particle_init_program.name = CREATE_PROGRAM(particle_init_vertex_glsl, NULL, particle_init_fragment_glsl);
#endif
    particle_init_program.uniform1i("particle_tex", 0);
    particle_init_program.uniform1i("collision_map", 1);
    particle_init_program.uniform1i("noise_tex", 2);
    particle_init_program.uniform1i("velocity_map", 3);
    SET_UNIFORM(particle_init_program, velocity_map_seconds, glUniform1f, particle_velocity_map_seconds);
    CHECK_FOR_GL_ERRORS("creating particle init program");

#if ENCODE_SHADERS
    particle_draw_program.name = createProgram("particle_draw_vertex.glsl", NULL, "particle_draw_fragment.glsl");
#else
    particle_draw_program.name = CREATE_PROGRAM(particle_draw_vertex_glsl, NULL, particle_draw_fragment_glsl);
#endif
    particle_draw_program.uniform1i("particle_tex", 0);
    particle_draw_program.uniform1i("particle_tex2", 1);
    particle_draw_program.uniform1i("shadow_texture0", 2);
    particle_draw_program.uniform1i("collision_map", 3);
    particle_draw_program.uniform1i("shadow_texture1", 4);
    particle_draw_program.uniformMatrix4fv("world_to_voxel", 1, GL_FALSE, world_to_voxel.e);
    particle_draw_program.uniform3f("tsiz", voxel_texture_size.x, voxel_texture_size.y, voxel_texture_size.z);
    CHECK_FOR_GL_ERRORS("creating particle draw program");

#if ENCODE_SHADERS
    particle_plot_program.name = createProgram("particle_plot_vertex.glsl", "particle_plot_geometry.glsl", "particle_plot_fragment.glsl");
#else
    particle_plot_program.name = CREATE_PROGRAM(particle_plot_vertex_glsl, particle_plot_geometry_glsl, particle_plot_fragment_glsl);
#endif
    particle_plot_program.uniform1i("particle_tex", 0);
    particle_plot_program.uniform1i("particle_tex2", 1);
    particle_plot_program.uniformMatrix4fv("world_to_voxel", 1, GL_FALSE, world_to_voxel.e);
    particle_plot_program.uniform3f("tsiz", voxel_texture_size.x, voxel_texture_size.y, voxel_texture_size.z);
    CHECK_FOR_GL_ERRORS("creating particle plot program");

    for(uint32 i = 0; i < 2; ++i)
    {
       particle_framebuffers[i].num_texs = 1;
       particle_framebuffers[i].tex_formats[0] = GL_RGB32F;
       particle_framebuffers[i].width = 2048;
       particle_framebuffers[i].height = 2048;
       particle_framebuffers[i].internal_depth = false;
       initFramebuffer(particle_framebuffers[i]);
      CHECK_FOR_GL_ERRORS("creating particle data framebuffer");
    }

#if ENCODE_SHADERS
    shadow_program.name = createProgram("shadowbuffer_vertex.glsl", NULL, "shadowbuffer_fragment.glsl");
#else
    shadow_program.name = CREATE_PROGRAM(shadowbuffer_vertex_glsl, NULL, shadowbuffer_fragment_glsl);
#endif
    shadow_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(shadow_program.name, "transformation");
    CHECK_FOR_GL_ERRORS("creating shadow buffer render program");

#if ENCODE_SHADERS
    secondary_program.name = createProgram("secondary_vertex.glsl", NULL, "secondary_fragment.glsl");
#else
    secondary_program.name = CREATE_PROGRAM(secondary_vertex_glsl, NULL, secondary_fragment_glsl);
#endif
    secondary_program.uniform1i("fb_position", 1);
    secondary_program.uniform1i("fb_normal", 2);
    secondary_program.uniform1i("material_palette", 3);
    secondary_program.uniform1i("fb_velocity", 4);
    secondary_program.uniform1i("crystal_tex", 5);
    secondary_program.uniform1i("noise_tex", 6);
    secondary_program.uniform1i("skybox_tex", 7);
    secondary_program.uniform1i("stars_tex", 8);
    CHECK_FOR_GL_ERRORS("creating secondary program");

#if ENCODE_SHADERS
    motionblur_program.name = createProgram("motionblur_vertex.glsl", NULL, "motionblur_fragment.glsl");
#else
    motionblur_program.name = CREATE_PROGRAM(motionblur_vertex_glsl, NULL, motionblur_fragment_glsl);
#endif
    motionblur_program.uniform1i("fb_final", 0);
    motionblur_program.uniform1i("fb_position", 1);
    motionblur_program.uniform1i("fb_velocity", 2);
    CHECK_FOR_GL_ERRORS("creating motion blur program");

#if ENCODE_SHADERS
    light_program.name = createProgram("light_vertex.glsl", NULL, "light_fragment.glsl");
#else
    light_program.name = CREATE_PROGRAM(light_vertex_glsl, NULL, light_fragment_glsl);
#endif
    light_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(light_program.name, "modelview");
    light_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(light_program.name, "projection");
    light_program.uniform1i("fb_position", 0);
    light_program.uniform1i("fb_normal", 1);
    light_program.uniform1i("noise_tex", 2);
    CHECK_FOR_GL_ERRORS("creating light program");

#if ENCODE_SHADERS
    primary_program.name = createProgram("primary_vertex.glsl", "primary_geometry.glsl", "primary_fragment.glsl");
#else
    primary_program.name = CREATE_PROGRAM(primary_vertex_glsl, primary_geometry_glsl, primary_fragment_glsl);
#endif
    primary_program.transform_properties.object_to_world_matrix_location = glGetUniformLocation(primary_program.name, "object_to_world");
    primary_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(primary_program.name, "modelview");
    primary_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(primary_program.name, "projection");
    primary_program.transform_properties.material_index_location = glGetUniformLocation(primary_program.name, "material");
    primary_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(primary_program.name, "object_to_clip");
    primary_program.transform_properties.previous_object_to_clip_matrix_location = glGetUniformLocation(primary_program.name, "previous_object_to_clip");
    primary_program.transform_properties.normal_matrix_location = glGetUniformLocation(primary_program.name, "normal_matrix");
    primary_program.uniform1f("material", 1);
    primary_program.uniform1i("noise_tex", 0);
    primary_program.uniform1i("shadowbuffers", 1);
    primary_program.uniform1i("shadow_voxel_texture0", 3);
    primary_program.uniform1i("paper_texture", 4);
    primary_program.uniform1i("shadow_voxel_texture1", 5);
    primary_program.uniformMatrix4fv("world_to_voxel", 1, GL_FALSE, world_to_voxel.e);
    primary_program.uniform3f("tsiz", voxel_texture_size.x, voxel_texture_size.y, voxel_texture_size.z);
    CHECK_FOR_GL_ERRORS("creating primary program");

#if ENCODE_SHADERS
    subdiv_program.name = createProgram("subdiv_vertex.glsl", "subdiv_geometry.glsl", "subdiv_fragment.glsl");
#else
    subdiv_program.name = CREATE_PROGRAM(subdiv_vertex_glsl, subdiv_geometry_glsl, subdiv_fragment_glsl);
#endif
    subdiv_program.transform_properties.object_to_world_matrix_location = glGetUniformLocation(subdiv_program.name, "object_to_world");
    subdiv_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(subdiv_program.name, "modelview");
    subdiv_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(subdiv_program.name, "projection");
    subdiv_program.transform_properties.material_index_location = glGetUniformLocation(subdiv_program.name, "material");
    subdiv_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(subdiv_program.name, "object_to_clip");
    subdiv_program.transform_properties.previous_object_to_clip_matrix_location = glGetUniformLocation(subdiv_program.name, "previous_object_to_clip");
    subdiv_program.uniform1f("material", 1);
    subdiv_program.uniform1i("noise_tex", 0);
    subdiv_program.uniform1i("shadowbuffers", 1);
    CHECK_FOR_GL_ERRORS("creating subdiv program");

#if ENCODE_SHADERS
    skybox_program.name = createProgram("skybox_vertex.glsl", "skybox_geometry.glsl", "skybox_fragment.glsl");
#else
    skybox_program.name = CREATE_PROGRAM(skybox_vertex_glsl, skybox_geometry_glsl, skybox_fragment_glsl);
#endif
    skybox_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(skybox_program.name, "modelview");
    skybox_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(skybox_program.name, "projection");
    CHECK_FOR_GL_ERRORS("creating skybox program");

/*
    smoketrail_program.name = createProgram("smoketrail_vertex.glsl", "smoketrail_geometry.glsl", "smoketrail_fragment.glsl");
    smoketrail_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(smoketrail_program.name, "modelview");
    smoketrail_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(smoketrail_program.name, "projection");
    smoketrail_program.transform_properties.material_index_location = glGetUniformLocation(smoketrail_program.name, "material");
    CHECK_FOR_GL_ERRORS("creating smoketrail program");
*/
#if ENCODE_SHADERS
    staticparticle_program.name = createProgram("staticparticle_vertex.glsl", NULL, "staticparticle_fragment.glsl");
#else
    staticparticle_program.name = CREATE_PROGRAM(staticparticle_vertex_glsl, NULL, staticparticle_fragment_glsl);
#endif
    CHECK_FOR_GL_ERRORS("creating staticparticle program");

    staticparticle_framebuffer.num_texs = 1;
    staticparticle_framebuffer.tex_formats[0] = GL_RGB32F;
    staticparticle_framebuffer.width = g_staticparticles_size;
    staticparticle_framebuffer.height = g_staticparticles_size;
    initFramebuffer(staticparticle_framebuffer);
    CHECK_FOR_GL_ERRORS("creating staticparticle framebuffer");

#if ENCODE_SHADERS
    crystal_program.name = createProgram("crystal_vertex.glsl", "crystal_geometry.glsl", "crystal_fragment.glsl");
#else
    crystal_program.name = CREATE_PROGRAM(crystal_vertex_glsl, crystal_geometry_glsl, crystal_fragment_glsl);
#endif
    crystal_program.uniform1i("noise_tex", 0);
    CHECK_FOR_GL_ERRORS("creating crystal program");

#if ENCODE_SHADERS
    sphere_program.name = createProgram("sphere_vertex.glsl", "sphere_geometry.glsl", "sphere_fragment.glsl");
#else
    sphere_program.name = CREATE_PROGRAM(sphere_vertex_glsl, sphere_geometry_glsl, sphere_fragment_glsl);
#endif
    CHECK_FOR_GL_ERRORS("creating sphere program");

#if ENCODE_SHADERS
    skyscraper_program.name = createProgram("skyscraper_vertex.glsl", "skyscraper_geometry.glsl", "skyscraper_fragment.glsl");
#else
    skyscraper_program.name = CREATE_PROGRAM(skyscraper_vertex_glsl, skyscraper_geometry_glsl, skyscraper_fragment_glsl);
#endif
    skyscraper_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(skyscraper_program.name, "modelview");
    skyscraper_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(skyscraper_program.name, "projection");
    skyscraper_program.transform_properties.object_to_world_matrix_location = glGetUniformLocation(skyscraper_program.name, "object_to_world");
    skyscraper_program.uniform1i("noise_tex", 0);
    skyscraper_program.uniform1i("cubemap_tex", 1);
    skyscraper_program.uniform1i("noise_tex2", 2);
    CHECK_FOR_GL_ERRORS("creating skyscraper program");

#if ENCODE_SHADERS
    chunk_program.name = createProgram("chunk_vertex.glsl", "chunk_geometry.glsl", "chunk_fragment.glsl");
#else
    chunk_program.name = CREATE_PROGRAM(chunk_vertex_glsl, chunk_geometry_glsl, chunk_fragment_glsl);
#endif
    chunk_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(chunk_program.name, "modelview");
    chunk_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(chunk_program.name, "projection");
    chunk_program.transform_properties.material_index_location = glGetUniformLocation(chunk_program.name, "material");
    chunk_program.uniform1i("noise_tex", 0);
    chunk_program.uniform1i("shadowbuffers", 1);
    chunk_program.uniform1i("paper_texture", 2);
    chunk_program.uniform1i("chunk_texture0", 3);
    chunk_program.uniform1i("chunk_texture1", 4);
    CHECK_FOR_GL_ERRORS("creating chunk program");

#if ENCODE_SHADERS
    chunk_shadow_program.name = createProgram("chunk_shadow_vertex.glsl", NULL, "chunk_shadow_fragment.glsl");
#else
    chunk_shadow_program.name = CREATE_PROGRAM(chunk_shadow_vertex_glsl, NULL, chunk_shadow_fragment_glsl);
#endif
    chunk_shadow_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(chunk_shadow_program.name, "transformation");
    chunk_shadow_program.uniform1i("chunk_texture0", 0);
    chunk_shadow_program.uniform1i("chunk_texture1", 1);
    CHECK_FOR_GL_ERRORS("creating chunk shadow program");

#if ENCODE_SHADERS
    chunk_update_program.name = createProgram("chunk_update_vertex.glsl", NULL, "chunk_update_fragment.glsl");
#else
    chunk_update_program.name = CREATE_PROGRAM(chunk_update_vertex_glsl, NULL, chunk_update_fragment_glsl);
#endif
    chunk_update_program.uniform1i("chunk_tex", 0);
    chunk_update_program.uniform1i("noise_tex", 1);
    CHECK_FOR_GL_ERRORS("creating chunk update program");

#if ENCODE_SHADERS
    sky_program.name = createProgram("sky_vertex.glsl", NULL, "sky_fragment.glsl");
#else
    sky_program.name = CREATE_PROGRAM(sky_vertex_glsl, NULL, sky_fragment_glsl);
#endif
    sky_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(sky_program.name, "modelview");
    sky_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(sky_program.name, "projection");
    sky_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(sky_program.name, "object_to_clip");
    sky_program.transform_properties.previous_object_to_clip_matrix_location = glGetUniformLocation(sky_program.name, "previous_object_to_clip");
    CHECK_FOR_GL_ERRORS("creating sky program");

#if ENCODE_SHADERS
    raytrace_program.name = createProgram("raytrace_vertex.glsl", "raytrace_geometry.glsl", "raytrace_fragment.glsl");
#else
    raytrace_program.name = CREATE_PROGRAM(raytrace_vertex_glsl, raytrace_geometry_glsl, raytrace_fragment_glsl);
#endif
    raytrace_program.transform_properties.object_to_view_matrix_location = glGetUniformLocation(raytrace_program.name, "modelview");
    raytrace_program.transform_properties.view_to_clip_matrix_location = glGetUniformLocation(raytrace_program.name, "projection");
    raytrace_program.transform_properties.object_to_clip_matrix_location = glGetUniformLocation(raytrace_program.name, "object_to_clip");
    raytrace_program.transform_properties.previous_object_to_clip_matrix_location = glGetUniformLocation(raytrace_program.name, "previous_object_to_clip");
    raytrace_program.transform_properties.object_to_world_matrix_location = glGetUniformLocation(raytrace_program.name, "object_to_world");
    raytrace_program.uniform1i("node0", 0);
    raytrace_program.uniform1i("node1", 1);
    raytrace_program.uniform1i("edge0", 2);
    raytrace_program.uniform1i("edge1", 3);
    raytrace_program.uniform1i("triangle", 4);
    raytrace_program.uniform1i("noise_tex", 5);
    CHECK_FOR_GL_ERRORS("creating raytrace program");


#if ENCODE_SHADERS
    blur_program.name = createProgram("blur_vertex.glsl", NULL, "blur_fragment.glsl");
#else
    blur_program.name = CREATE_PROGRAM(blur_vertex_glsl, NULL, blur_fragment_glsl);
#endif
    blur_program.uniform1i("fb_final", 0);
    blur_program.uniform1i("noise_tex", 1);
    CHECK_FOR_GL_ERRORS("creating blur program");

#if ENCODE_SHADERS
    bloom_program.name = createProgram("bloom_vertex.glsl", NULL, "bloom_fragment.glsl");
#else
    bloom_program.name = CREATE_PROGRAM(bloom_vertex_glsl, NULL, bloom_fragment_glsl);
#endif
    bloom_program.uniform1i("source_tex", 0);
    bloom_program.uniform1i("noise_tex", 1);
    CHECK_FOR_GL_ERRORS("creating bloom program");

    for(uint32 i=0;i<2;++i)
    {
       bloom_framebuffers[i].num_texs = 1;
       bloom_framebuffers[i].tex_formats[0] = GL_RGBA32F;
       bloom_framebuffers[i].width = g_window_width / 2;
       bloom_framebuffers[i].height = g_window_height / 2;
       initFramebuffer(bloom_framebuffers[i]);
       CHECK_FOR_GL_ERRORS("creating bloom framebuffer");
    }

    blur_framebuffer.num_texs = 1;
    blur_framebuffer.tex_formats[0] = GL_RGBA32F;
    blur_framebuffer.width = g_window_width;
    blur_framebuffer.height = g_window_height;
    initFramebuffer(blur_framebuffer);
    CHECK_FOR_GL_ERRORS("creating blur framebuffer");

    primary_framebuffer.num_texs = 3;
    primary_framebuffer.tex_formats[0] = GL_RGBA32F;
    primary_framebuffer.tex_formats[1] = GL_RGBA16;
    primary_framebuffer.tex_formats[2] = GL_RGB16;
    primary_framebuffer.depth_rbo = g_generic_depth_rbo;
    primary_framebuffer.width = g_window_width;
    primary_framebuffer.height = g_window_height;
    initFramebuffer(primary_framebuffer);
    CHECK_FOR_GL_ERRORS("creating primary framebuffer");

    motionblur_framebuffer.num_texs = 1;
    motionblur_framebuffer.tex_formats[0] = GL_RGBA32F;
    motionblur_framebuffer.width = g_window_width;
    motionblur_framebuffer.height = g_window_height;
    initFramebuffer(motionblur_framebuffer);
    CHECK_FOR_GL_ERRORS("creating motion blur framebuffer");

    final_framebuffer.num_texs = 1;
    final_framebuffer.tex_formats[0] = GL_RGBA32F;
    final_framebuffer.width = g_window_width;
    final_framebuffer.height = g_window_height;
    initFramebuffer(final_framebuffer);
    CHECK_FOR_GL_ERRORS("creating final framebuffer");

    crystal_framebuffer.num_texs = 1;
    crystal_framebuffer.tex_formats[0] = GL_RGB32F;
    crystal_framebuffer.width = g_window_width;
    crystal_framebuffer.height = g_window_height;
    initFramebuffer(crystal_framebuffer);
    CHECK_FOR_GL_ERRORS("creating crystal framebuffer");

#if ENCODE_SHADERS
    post_program.name = createProgram("post_vertex.glsl", NULL, "post_fragment.glsl");
#else
    post_program.name = CREATE_PROGRAM(post_vertex_glsl, NULL, post_fragment_glsl);
#endif
    post_program.uniform1i("fb_final", 0);
    post_program.uniform1i("noise_tex", 1);
    CHECK_FOR_GL_ERRORS("creating postprocess program");

#if ENCODE_SHADERS
    paper_program.name = createProgram("paper_vertex.glsl", NULL, "paper_fragment.glsl");
#else
    paper_program.name = CREATE_PROGRAM(paper_vertex_glsl, NULL, paper_fragment_glsl);
#endif
    CHECK_FOR_GL_ERRORS("creating paper texture precalc program");

#if ENCODE_SHADERS
    info_program.name = createProgram("info_vertex.glsl", NULL, "info_fragment.glsl");
#else
    info_program.name = CREATE_PROGRAM(info_vertex_glsl, NULL, info_fragment_glsl);
#endif
    info_program.uniform1i("tex", 0);
    CHECK_FOR_GL_ERRORS("creating info program");

   log("loading shaders: %dms\n",SDL_GetTicks()-shader_load_start_time);

   static const uint32 paper_texture_width = 1024;
   static const uint32 paper_texture_height = paper_texture_width;
   GLuint paper_texture = 0;
   {
      const Uint32 st=SDL_GetTicks();
      glGenTextures(1, &paper_texture);
      glBindTexture(GL_TEXTURE_2D, paper_texture);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
      my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_RG8, paper_texture_width, paper_texture_height);
      glBindTexture(GL_TEXTURE_2D, 0);
      GLuint temp_paper_fbo = 0;
      glGenFramebuffers(1, &temp_paper_fbo);
      glBindFramebuffer(GL_DRAW_FRAMEBUFFER,temp_paper_fbo);
      glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, paper_texture, 0);
      CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "paper texture precalc");
      assert(glCheckFramebufferStatus(GL_FRAMEBUFFER) == GL_FRAMEBUFFER_COMPLETE);
      glViewport(0,0,paper_texture_width,paper_texture_height);
      {
       const GLenum draw_buffers[] = { GL_COLOR_ATTACHMENT0 };
       glDrawBuffers(sizeof(draw_buffers)/sizeof(draw_buffers[0]), draw_buffers);
      }
      paper_program.use();
      glDisable(GL_DEPTH_TEST);
      glDepthMask(GL_FALSE);
      glColorMask(GL_TRUE, GL_TRUE, GL_FALSE, GL_FALSE);
      renderQuadGrid();
      glDeleteFramebuffers(1, &temp_paper_fbo);
      log("paper texture: %dms\n",SDL_GetTicks()-st);
   }
    CHECK_FOR_GL_ERRORS("precalculating paper texture");


   create2015Texture();

  {
      const Uint32 st=SDL_GetTicks();
   renderSkyBox();
      log("renderSkyBox: %dms\n",SDL_GetTicks()-st);
   }

   {
      const Uint32 st=SDL_GetTicks();
   createSkyscraperCubeMaps();
      log("createSkyscraperCubeMaps: %dms\n",SDL_GetTicks()-st);
   }

   createStarsTexture();

//   static const uint32 num_shadowbuffers = 5;
//   static const uint32 shadowbuffer_sizes[2] = { 16, 4096 };
   static const uint32 num_shadowbuffers = 4;
   static const uint32 shadowbuffer_sizes[2] = { 2048, 4096 };
   GLuint shadowbuffer_fbos[2] = { 0, 0 };
   GLuint shadowbuffers_texs[2] = { 0, 0 };
   glGenTextures(2, shadowbuffers_texs);
   glGenFramebuffers(2, shadowbuffer_fbos);
   for(uint32 i = 0; i < 2; ++i)
   {
      glBindTexture(GL_TEXTURE_2D_ARRAY, shadowbuffers_texs[i]);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER);
      GLfloat border[4]={1.0f,1.0f,1.0f,1.0f};
      glTexParameterfv(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_BORDER_COLOR, border);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_COMPARE_FUNC, GL_LESS);
      my_glTexStorage3D(GL_TEXTURE_2D_ARRAY, 1, GL_DEPTH_COMPONENT32, shadowbuffer_sizes[i], shadowbuffer_sizes[i], num_shadowbuffers);
   }
   glBindTexture(GL_TEXTURE_2D_ARRAY, 0);
   CHECK_FOR_GL_ERRORS("creating shadow buffers");

   GLuint material_palette_tex = 0;
   {
      glGenTextures(1, &material_palette_tex);
      glBindTexture(GL_TEXTURE_2D, material_palette_tex);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
      my_glTexStorage2D(GL_TEXTURE_2D, 1, GL_RGBA8, 4, (num_materials + 1) * 2);
      Ran ran(12341);
      GLubyte* pix = new GLubyte[(num_materials + 1) * 2 * 4 * 4];
      memset(pix, 0, (num_materials + 1) * 2 * 4 * 4);
      for(uint32 i = 1; i < (num_materials + 1) * 2; ++i)
      {
         uint32 mi = (i - 1) % num_materials;
         float rf = material_data[mi * 4 + 0];
         float gf = material_data[mi * 4 + 1];
         float bf = material_data[mi * 4 + 2];
         if((i - 1) / num_materials == 1)
         {
            rf *= 0.5;
            gf *= 0.5;
         }
         unsigned long int r = std::max(0.0f, std::min(1.0f, rf)) * 255;
         unsigned long int g = std::max(0.0f, std::min(1.0f, gf)) * 255;
         unsigned long int b = std::max(0.0f, std::min(1.0f, bf)) * 255;
         pix[(i * 4 + 0) * 4 + 0] = r;
         pix[(i * 4 + 0) * 4 + 1] = g;
         pix[(i * 4 + 0) * 4 + 2] = b;
         if(material_data[mi * 4 + 3] > 2.5f)
         {
            pix[(i * 4 + 1) * 4 + 2] = 255;
         }
         else if(material_data[mi * 4 + 3] > 1.5f)
         {
            pix[(i * 4 + 1) * 4 + 1] = 255;
         }
         else if(material_data[mi * 4 + 3] > 0.5f)
         {
            pix[(i * 4 + 1) * 4 + 0] = 255;
         }
      }
       pix[1 * 4 + 0] = 0xffffff;
      glTexSubImage2D(GL_TEXTURE_2D,0,0,0,4,(num_materials + 1) * 2,GL_RGBA,GL_UNSIGNED_BYTE,pix);
      delete[] pix;
      glBindTexture(GL_TEXTURE_2D, 0);
   }
    CHECK_FOR_GL_ERRORS("creating material palette texture");

   uint32 particle_framebuffer_flipper = 0;
   uint32 chunk_framebuffer_flipper = 0;

   GLuint particle_coord_vbo = 0;
   glGenBuffers(1, &particle_coord_vbo);
   glBindBuffer(GL_ARRAY_BUFFER,particle_coord_vbo);
         std::vector<GLfloat> pvs(particle_framebuffers[0].width*particle_framebuffers[0].height*2);
         for(long y=0;y<particle_framebuffers[0].height;++y)
            for(long x=0;x<particle_framebuffers[0].width;++x)
            {
               pvs[(x+y*particle_framebuffers[0].width)*2+0]=(GLfloat(x)+0.5f)/GLfloat(particle_framebuffers[0].width);
               pvs[(x+y*particle_framebuffers[0].width)*2+1]=(GLfloat(y)+0.5f)/GLfloat(particle_framebuffers[0].height);
            }
   my_glBufferData(GL_ARRAY_BUFFER, particle_framebuffers[0].width*particle_framebuffers[0].height*2*sizeof(GLfloat), &pvs[0], GL_STATIC_DRAW);
   glBindBuffer(GL_ARRAY_BUFFER,0);
    CHECK_FOR_GL_ERRORS("creating particle update VBO");


   {
      const Uint32 st=SDL_GetTicks();
   createParticleVelocityMap();
      log("createParticleVelocityMap: %dms\n",SDL_GetTicks()-st);
   }

   {
      const Uint32 st=SDL_GetTicks();
   createExplosionChunks();
      log("createExplosionChunks: %dms\n",SDL_GetTicks()-st);
   }

   {
      const Uint32 st=SDL_GetTicks();
   createBlossoms();
      log("createBlossoms: %dms\n",SDL_GetTicks()-st);
   }

   {
      const Uint32 st=SDL_GetTicks();
   createSubdividedVillage();
      log("createSubdividedVillage: %dms\n",SDL_GetTicks()-st);
   }

   createStaticParticles();

#if RAYTRACE
   std::vector<BVHData> raytrace_bvh_data(raytraced_surface_node_count);
   for(uint32 i = 0; i < raytraced_surface_node_count; ++i)
   {
      const uint32 sidx = raytraced_surface_nodes[i * 2 + 0];
      const uint32 gidx = raytraced_surface_nodes[i * 2 + 1];
      Entity* surface_ent = g_orion->entities[sidx];
      assert(surface_ent);
      Entity* ground_ent = g_orion->entities[gidx];
      assert(ground_ent);
      surface_ent->raytracer = true;

      const uint32 s_mesh_count = node_mesh_offsets[sidx + 1] - node_mesh_offsets[sidx];
      const uint32 g_mesh_count = node_mesh_offsets[gidx + 1] - node_mesh_offsets[gidx];

      assert(s_mesh_count == 1);
      assert(g_mesh_count == 1);

      const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[gidx]];
      const MeshCacheItem* m = g_orion->mesh_cache[mesh_index];

      raytrace_bvh_data[i] = createBVH(vertices, 6, indices + mesh_index_bases[mesh_index], m->count / 3, 5, 3);
   }
#endif

   glEnable(GL_CULL_FACE);

     static const Mat4 bias_matrix(
         0.5, 0.0, 0.0, 0.5,
         0.0, 0.5, 0.0, 0.5,
         0.0, 0.0, 0.5, 0.5,
         0.0, 0.0, 0.0, 1.0
         );

   assert(g_orion);
   //const Mat4 smoketrail_object_matrix = Mat4::translation(Vec3(0,-4.0,0)) * g_orion->getEntityGlobalTransformationForTime(rocket_node_index, 0);

     const float znear = 1.0f / 16.0f;
     const float aspect_ratio = float(g_window_height) / float(g_window_width);


   SDL_Event event;
   bool finished = false;

   // absorb any events accumulated during startup
   while(SDL_PollEvent(&event))
      ;

#if ENCODE_SHADERS
   {
      extern void encodeShaderSources();
      encodeShaderSources();
   }
#endif

    if(!BASS_Init(-1, 44100, 0, 0, NULL))
    {
      log("BASS_Init failed.\n");
      SDL_Quit();
      return -1;
    }

    HSTREAM audio_stream = BASS_StreamCreateFile(FALSE, "lug00ber-wavetracker.mp3", 0, 0, BASS_STREAM_PRESCAN);
    assert(audio_stream != 0);
    if(audio_stream==0)
    {
      log("BASS_StreamCreateFile failed.\n");
      SDL_Quit();
      return -1;
    }

   BASS_ChannelSetPosition(audio_stream, 0, BASS_POS_BYTE);
   BASS_ChannelPlay(audio_stream, TRUE);

#if DEVMODE
   //BASS_ChannelSetAttribute(audio_stream, BASS_ATTRIB_VOL, 0);
#endif

   const double audio_stream_length=BASS_ChannelBytes2Seconds(audio_stream, BASS_ChannelGetLength(audio_stream, BASS_POS_BYTE));

   log("Texture memory usage estimation: %d bytes.\n",g_texture_memory_usage);
   log("Buffer memory usage estimation: %d bytes.\n",g_buffer_memory_usage);

   Uint32 chunks_start_time = 0;
   Uint32 next_chunk_logic_update = 0;

#if CAPTURE
   const Uint32 mb_frame_count=std::max(1,arg_motion_blur);
   Uint32 capture_frame_rate=60*mb_frame_count;
   unsigned long int* accumulated_mb_frames=new unsigned long int[g_window_width*g_window_height*3];
   memset(accumulated_mb_frames,0,g_window_width*g_window_height*3*sizeof(unsigned long int));
   Uint32 capture_frame = arg_start_frame*mb_frame_count;//Uint32(497.0*double(capture_frame_rate)/30.0);
   Uint32 local_capture_frame = 0;
#endif

    while(!finished)
    {
#if CAPTURE
        const double capture_time = floor(double(capture_frame/mb_frame_count))*double(mb_frame_count) * 1.0 / double(capture_frame_rate) + double(capture_frame%mb_frame_count)/double(mb_frame_count) * 1.0 / 24.0;
#endif

        const Uint32 start_time = SDL_GetTicks();

#if CAPTURE
        const Uint32 ct = Uint32(capture_time * 1000);
#else
        const Uint32 ct = start_time ;
#endif

        while(SDL_PollEvent(&event))
        {
            if(finished)
                break;

            switch(event.type)
            {
                case SDL_KEYDOWN:
                    if(event.key.keysym.sym == SDLK_ESCAPE)
                    finished = true;
                    break;
            }
        }

#if !CAPTURE
        if(BASS_ChannelIsActive(audio_stream)!=BASS_ACTIVE_PLAYING)
           break;
#endif

        if(finished)
            break;

        int mouse_x=0,mouse_y=0;
        const Uint32 mouse_buttons=SDL_GetMouseState(&mouse_x,&mouse_y);

#if DEVMODE
        if(mouse_buttons & SDL_BUTTON(SDL_BUTTON_LEFT))
        {
            const double t = double(mouse_x)/double(g_window_width)*audio_stream_length;
            BASS_ChannelSetPosition(audio_stream,BASS_ChannelSeconds2Bytes(audio_stream,t),BASS_POS_BYTE);
        }
#endif

#if CAPTURE
        if(capture_frame>arg_end_frame*mb_frame_count)
            break;

        g_time = capture_time;
        if(g_time > 1825.0/24.0)
          break;
#else
        const QWORD pos = BASS_ChannelGetPosition(audio_stream, BASS_POS_BYTE);
        g_time = std::max(0.0, BASS_ChannelBytes2Seconds(audio_stream, pos));
#endif
         //g_time = double(capture_frame) * 1.0/50.0;
        //g_time = 67.0;
        //g_time = 82.0;
        //g_time = g_particles_activate_time + g_time - 20;
        //g_time = 140 + g_time;
        //g_time+=10;
         //g_time = 118;
        //g_time+=55;
         //g_time+=145;
         //g_time += g_particles_activate_time - 8;
         //g_time+=15;
         //g_time=48+g_time;
         //g_time=2*60+8;
         //g_time+=g_blossom_time-4;//fmodf(g_time,8.0);
         //g_time+=g_particles_activate_time+10;
         //g_time+=33;
         //g_time=10;
         //g_time+=2*60+12;
         //g_time+=g_subdiv_transformation_time-1;
         //g_time+=1*60+57;
         //g_time+=g_info_appear_time;
         //g_time+=44;
         //g_time+=g_rocket_liftoff_time-12;
         //g_time+=g_info_appear_time-15;


        if(g_time < g_subdiv_transformation_time)
        {
            g_orion->entities[rocket_node_index]->invisible=true;
        }
        else
        {
         const uint32 i=village_node_index;
         const uint32 child_count = node_child_offsets[i + 1] - node_child_offsets[i];
         for(uint32 j = 0; j < child_count; ++j)
         {
            const uint32 c = node_child_indices[node_child_offsets[i] + j];
            g_orion->entities[c]->invisible=true;
         }
        }


        float jitter=0,jitter2=0;
        {
            float s=20;
            int noise_frame=int(g_time*s);
            float a=smooth_noise_table[(noise_frame+0)&(smooth_noise_length-1)];
            float b=smooth_noise_table[(noise_frame+1)&(smooth_noise_length-1)];
            float t=smoothstep(0.0f,1.0f,fmodf(g_time*s,1.0f));
            jitter=a*(1-t)+b*t;
            float a2=smooth_noise_table[(noise_frame+120)&(smooth_noise_length-1)];
            float b2=smooth_noise_table[(noise_frame+121)&(smooth_noise_length-1)];
            jitter2=a2*(1-t)+b2*t;
        }

        const Mat4 rocket_jitter_xfrm = (g_time >= g_rocket_liftoff_time) ? Mat4::translation(Vec3(0,jitter*0.1,0)) : Mat4::identity();
        const float rocket_height = ((uint32)std::floor(g_time * 24.0) >= 3765) ? (g_time-(g_rocket_liftoff_time+14))*6.5 : (g_time-g_rocket_liftoff_time)*6.5;
        const Mat4 rocket_object_matrix = (g_time >= g_rocket_liftoff_time) ? Mat4::translation(Vec3(0,rocket_height,0)) : Mat4::identity();

        const GLuint svt0 = shadow_voxel_textures[((particle_shadow_frame_counter / 8) & 1) ^ 1];
        const GLuint svt1 = shadow_voxel_textures[((particle_shadow_frame_counter / 8) & 1)];
        const float shadow_voxel_tween = float(particle_shadow_frame_counter & 7) / 8.0f;
        assert(svt0 != 0);
        assert(svt1 != 0);

        Viewport vp;
        g_orion->updateAnimations(g_time, vp);

         {
            const double e = 1e-2;
            const Vec3 p0 = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, g_time + e * 0.0).column(3);
            const Vec3 p1 = g_orion->getEntityGlobalTransformationForTime(rocket_node_index, g_time + e * 1.0).column(3);
            g_current_rocket_z_vel = (p1.y - p0.y) / e;
         }

         //log("time = %d\n",(int)std::floor(g_time * 24.0));
         //log("vp.camera_entity_index = %d\n",vp.camera_entity_index);

        assert(vp.camera_entity_index < g_orion->entities.size());

        const Entity* camera_ent = g_orion->entities[vp.camera_entity_index];
        const Entity* rocket_ent = g_orion->entities[rocket_node_index];
        assert(rocket_ent);

        assert(camera_ent != NULL);

        CameraConfig view_cc;

        view_cc.world_to_view = camera_ent->global_xfrm.inverse();
        Mat4 previous_world_to_view = camera_ent->previous_global_xfrm.inverse();

         assert(g_orion);
         double camera_xfov = g_orion->getEntityXFovForTime(vp.camera_entity_index, g_time);
         if(camera_xfov==0)
            camera_xfov=60;

        for(int rep = -1; rep < 1; ++rep)
        {
           const double adjusted_time = g_time - rep * 1.0 / 24.0;
           const uint32 frame = (uint32)std::floor(g_time * 24.0);
           Mat4& matrix = rep ? previous_world_to_view : view_cc.world_to_view;
/*
           uint32 marker0_frame = 0;
           uint32 last_marker = 0;
           for(uint32 i = 0; i < marker_count; ++i)
           {
               const Marker& m = markers[i];
               if(m.frame <= frame && m.node_index != 0 && m.frame >= marker0_frame)
               {
                   marker0_frame = m.frame;
                   last_marker = i;
               }
           }*/

           if(adjusted_time >= g_rocket_liftoff_time && frame < 3474)
           {
              float jitter_attenuation=1.0f/(1.0f+2.0f*std::max(0.0,adjusted_time-g_rocket_liftoff_time))*0.2f;
              float jitter_x=0,jitter_y=0;
              {
                  float s=23;
                  int noise_frame=int(std::floor(adjusted_time*s));
                  float a=smooth_noise_table[(noise_frame+0)&(smooth_noise_length-1)];
                  float b=smooth_noise_table[(noise_frame+1)&(smooth_noise_length-1)];
                  float t=smoothstep(0.0f,1.0f,fmodf(adjusted_time*s,1.0f));
                  jitter_x=a*(1.0f-t)+b*t;
                  float a2=smooth_noise_table[(noise_frame+120)&(smooth_noise_length-1)];
                  float b2=smooth_noise_table[(noise_frame+121)&(smooth_noise_length-1)];
                  jitter_y=a2*(1.0f-t)+b2*t;
              }
              matrix = Mat4::translation(Vec3((jitter_x-0.5f)*jitter_attenuation, (jitter_y-0.5f)*jitter_attenuation,0)) * matrix;
           }
           else if(frame >= 3474 && frame < 3765)
           {
              const float tt=(adjusted_time - 3474.0/24.0)*0.2;
              matrix = Mat4::translation(Vec3(0,0,-30)) * Mat4::rotation(M_PI*-0.1 - M_PI*0.1*tt,Vec3(1,0,0)) * Mat4::rotation(M_PI*-1.25+adjusted_time*0.1,Vec3(0,0,1)) * Mat4::translation(Vec3(0,0,-8)) *
                                          (rocket_object_matrix * rocket_ent->global_xfrm).inverse();
           }

        }

        float transition_fade=1;

           // find the nearest marker
          {
              const uint32 frame = (uint32)std::floor(g_time * 24.0);
              uint32 marker0_frame = 0;
              uint32 marker1_frame = ~0U;
              uint32 last_marker = 0;
              for(uint32 i = 0; i < marker_count; ++i)
              {
                  const Marker& m = markers[i];
                  if(m.frame <= frame && m.node_index != 0 && m.frame >= marker0_frame)
                  {
                      marker0_frame = m.frame;
                      last_marker = i;
                  }
              }
              for(uint32 i = 0; i < marker_count; ++i)
              {
                  const Marker& m = markers[i];
                  if(m.frame > frame && m.node_index != 0 && m.frame <= marker1_frame)
                  {
                      marker1_frame = m.frame;
                  }
              }

             assert(marker0_frame <= frame);
             assert(marker1_frame >= frame);
             assert(marker0_frame <= marker1_frame);
           if(g_time < g_particles_activate_time)
           {
              if(last_marker >= 3)
              {
                 transition_fade=smoothstep(0.0, 1.0, g_time - marker0_frame * 1.0 / 24.0) - smoothstep(-1.0, 0.0, g_time - marker1_frame * 1.0 / 24.0);
              }
              else
              {
                 transition_fade=smoothstep(0.0, 10.0, g_time - marker0_frame * 1.0 / 24.0) - smoothstep(-2.0, 0.0, g_time - marker1_frame * 1.0 / 24.0);
              }
           }
           transition_fade=mix(transition_fade, 5.0f, smoothstep(60.0+39.0, 60.0+41.0, g_time) - smoothstep(60.0+41.0, 60.0+43.0, g_time)); // second cave exit
           transition_fade=mix(transition_fade, 0.0f, smoothstep(2*60.0+59.0, 3*60.0+8.0, g_time)); // ending
           if(g_time<g_explosion_time)
              for(int th=0;th<num_thumps;++th)
              {
                  if(g_time>=(thumps[th]-.1))
                     transition_fade=mix(1.5,1.0,std::min(1.,(g_time-(thumps[th]-.1))/.1));
              }
         }

        view_cc.world_to_view = Mat4::scale(Vec3(1.0f/view_cc.world_to_view.column(0).length(),
                                                 1.0f/view_cc.world_to_view.column(1).length(),
                                                 1.0f/view_cc.world_to_view.column(2).length())) * view_cc.world_to_view;

        previous_world_to_view = Mat4::scale(Vec3(1.0f/previous_world_to_view.column(0).length(),
                                                 1.0f/previous_world_to_view.column(1).length(),
                                                 1.0f/previous_world_to_view.column(2).length())) * previous_world_to_view;

        const float tan_half_fov = tan(camera_xfov / 2.0 * M_PI / 180.0);
        {
           const float fl = -tan_half_fov * znear, fr = +tan_half_fov * znear, fb = -aspect_ratio * tan_half_fov * znear, ft = +aspect_ratio * tan_half_fov * znear;
           //const Mat4 cam_perspective = Mat4::translation(Vec3(1.0,1.0,0.0)) * Mat4::scale(Vec3(2.0,2.0,1.0)) * Mat4::frustum(fl, fr, fb, ft, znear, 1024.0f);
           const Mat4 cam_perspective = Mat4::frustum(fl, fr, fb, ft, znear, 1024.0f);

           const Entity* logo_ent = g_orion->entities[revision_logo_node_index];
           assert(logo_ent);

            const uint32 logo_mesh_count = node_mesh_offsets[revision_logo_node_index + 1] - node_mesh_offsets[revision_logo_node_index];
            assert(logo_mesh_count == 1);
            const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[revision_logo_node_index]];

           const Vec3 logo_local_center =(Vec3(bounding_boxes[mesh_index * 6 + 0], bounding_boxes[mesh_index * 6 + 1], bounding_boxes[mesh_index * 6 + 2]) +
                                          Vec3(bounding_boxes[mesh_index * 6 + 3], bounding_boxes[mesh_index * 6 + 4], bounding_boxes[mesh_index * 6 + 5])) / 2.0f;

           const float distance_to_logo = ((view_cc.world_to_view * logo_ent->global_xfrm * logo_local_center) - (view_cc.world_to_view * camera_ent->global_xfrm.column(3))).length();
           const float ofw = fr / znear * distance_to_logo, ofh = ft / znear * distance_to_logo;
           const Mat4 cam_orthographic = Mat4::translation(Vec3(0.0,0.0,-0.5)) * Mat4::scale(Vec3(1.0f / ofw,1.0f / ofh,-1.0f/1024.0f));


           for(uint i=0;i<16;++i)
               //view_cc.view_to_clip[i] = mix(cam_perspective[i], cam_orthographic[i], std::pow(std::max(0.0,smoothstep(154.0+13.0,160.0+13.0,g_time)),1.0/32.0));
               view_cc.view_to_clip[i] = cam_perspective[i];
        }

        view_cc.world_to_clip = view_cc.view_to_clip * view_cc.world_to_view;
        view_cc.previous_world_to_clip = view_cc.view_to_clip * previous_world_to_view;

        //const Mat4 sun_orientation = Mat4::rotation(-0.2,Vec3(0,1,0)) * Mat4::rotation(0.5,Vec3(1,0,0));
        //const Mat4 sun_orientation = Mat4::rotation(M_PI*0.3,Vec3(1,0,0)) * Mat4::rotation(51.489342,Vec3(0,1,0));
        //const Mat4 sun_orientation = Mat4::rotation(M_PI*0.32*g_time/50.0,Vec3(1,0,0)) * Mat4::rotation(51.489342,Vec3(0,1,0));
        //const Mat4 sun_orientation = Mat4::rotation(-0.2,Vec3(0,1,0)) * Mat4::rotation(0.5,Vec3(1,0,0));

        //const Mat4 sun_orientation = Mat4::rotation(0.5*std::pow(1.0-smoothstep(0.0f,130.0f,fabsf(g_time-g_particles_activate_time)),0.8f),Vec3(1,0,0)) * Mat4::rotation(-0.2,Vec3(0,1,0));
        assert(g_orion);

        const double suntime = g_orion->getMostRecentCameraMarkerTime(g_time);

        //if(vp.camera_entity_index == camera_003_node_index || vp.camera_entity_index == camera_006_node_index || vp.camera_entity_index == camera_007_node_index)
          //  suntime = g_particles_activate_time;

        const Mat4 sun_orientation = Mat4::rotation(0.5*std::pow(1.0-smoothstep(0.0f,130.0f,fabsf(suntime-g_particles_activate_time)),0.8f),Vec3(1,0,0)) * Mat4::rotation(0.1,Vec3(0,1,0));


        const Mat4 sun_orientation_transpose = sun_orientation.transpose();
        Vec3 vsd = sun_orientation.inverse().column(2);
        Vec3 vsu = Vec3(0,1,0);
        view_cc.world_to_view.transform3x3(vsd);
        view_cc.world_to_view.transform3x3(vsu);
        const float view_ray_xfrm[9] = { tan_half_fov, 0.0f, 0.0f,
                                   0.0f, aspect_ratio * tan_half_fov, 0.0f,
                                   0.0f, 0.0f, 1.0f };

        const uint32 shadowbuffer_selection = (g_time > g_big_shadowmap_activate_time) ? 1 : 0;


         if(g_time > g_explosion_time)
         {
            static bool chunks_inited=false;
            if(!chunks_inited)
            {
               chunks_start_time = ct;
               next_chunk_logic_update = chunks_start_time;
               chunks_inited=true;
            }
            for(uint32 i = 0; i < 128 && ct >= next_chunk_logic_update; ++i)
            {
               glBindFramebuffer(GL_DRAW_FRAMEBUFFER,g_chunk_fbos[chunk_framebuffer_flipper ^ 1]);
               glViewport(0,0,g_chunk_texture_size,g_chunk_texture_size);
               const GLenum bs[1] = { GL_COLOR_ATTACHMENT0 };
               glDrawBuffers(1, bs);
               glDisable(GL_DEPTH_TEST);
               glDepthMask(GL_FALSE);
               glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
               chunk_update_program.use();
               SET_UNIFORM(chunk_update_program, world_to_sdf, glUniformMatrix4fv, 1, GL_FALSE, g_world_to_revision_logo_sdf.e);
               //chunk_update_program.uniformMatrix4fv("world_to_sdf",1,GL_FALSE,g_world_to_revision_logo_sdf.e);
               {
                  double tt = (double(next_chunk_logic_update-chunks_start_time))/1000.0;
                  SET_UNIFORM(chunk_update_program, time, glUniform1f, tt);
                  chunk_update_program.uniform1f("time",tt);
               }
               setTextureUnit(0, GL_TEXTURE_2D, g_chunk_textures[chunk_framebuffer_flipper]);
               setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex2);
               //glActiveTexture(GL_TEXTURE1);
               //glBindTexture(GL_TEXTURE_2D,0);
               renderQuad();
               next_chunk_logic_update += chunk_update_interval;
               chunk_framebuffer_flipper ^= 1;
               CHECK_FOR_GL_ERRORS("updating chunks");
            }

        }


         Mat4 base_shadow_world_to_clip_biased[5];
         {
           shadow_program.use();
           glDepthMask(GL_TRUE);
           glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);
           glDrawBuffer(GL_NONE);
           glEnable(GL_DEPTH_TEST);
            glCullFace(GL_FRONT);
            CHECK_FOR_GL_ERRORS("setting state before shadowbuffer render");
            for(uint32 shadow = 0; shadow < num_shadowbuffers; ++shadow)
            {
              CameraConfig shadow_cc;

              Vec3 sp = -camera_ent->global_xfrm.column(3);

               const double sscale = .05 / pow(2.0f, (double)shadow);

              const double su = floor(sp.dot(sun_orientation_transpose.column(0)) * sscale * shadowbuffer_sizes[shadowbuffer_selection] / 2) / double(shadowbuffer_sizes[shadowbuffer_selection] / 2);
              const double sv = floor(sp.dot(sun_orientation_transpose.column(1)) * sscale * shadowbuffer_sizes[shadowbuffer_selection] / 2) / double(shadowbuffer_sizes[shadowbuffer_selection] / 2);

              sp = sun_orientation_transpose.column(0) * su / sscale +
                   sun_orientation_transpose.column(1) * sv / sscale;

              shadow_cc.world_to_view = sun_orientation * Mat4::translation(Vec3(sp));
              shadow_cc.view_to_clip = Mat4::scale(Vec3(sscale,sscale,-1.0/1024.0));
              shadow_cc.world_to_clip = shadow_cc.view_to_clip * shadow_cc.world_to_view;

              base_shadow_world_to_clip_biased[shadow] = bias_matrix * shadow_cc.world_to_clip;

               glBindFramebuffer(GL_DRAW_FRAMEBUFFER, shadowbuffer_fbos[shadowbuffer_selection]);
               CHECK_FOR_GL_ERRORS("binding shadowbuffer FBO");
               glFramebufferTextureLayer(GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, shadowbuffers_texs[shadowbuffer_selection], 0, shadow);
               CHECK_FOR_GL_ERRORS("setting up shadowbuffer");
               CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "shadowbuffers");
               assert(glCheckFramebufferStatus(GL_FRAMEBUFFER) == GL_FRAMEBUFFER_COMPLETE);
               glViewport(0, 0, shadowbuffer_sizes[shadowbuffer_selection], shadowbuffer_sizes[shadowbuffer_selection]);
               glClear(GL_DEPTH_BUFFER_BIT);

               if(g_time > g_explosion_time)
               {
                 chunk_shadow_program.use();
                 glUniformMatrix4fv(chunk_shadow_program.transform_properties.object_to_clip_matrix_location, 1, GL_FALSE, shadow_cc.world_to_clip.e);
                 setTextureUnit(0, GL_TEXTURE_2D, g_chunk_textures[chunk_framebuffer_flipper ^ 1]);
                 setTextureUnit(1, GL_TEXTURE_2D, g_chunk_textures[chunk_framebuffer_flipper]);
                 SET_UNIFORM(chunk_shadow_program, tween, glUniform1f, 1.0f - float(next_chunk_logic_update - ct) / float(chunk_update_interval));
                 //chunk_shadow_program.uniform1f("tween", 1.0f - float(next_chunk_logic_update - ct) / float(chunk_update_interval));
                 renderChunks();
               }
               else
               {
                  g_orion->renderEntities(shadow_cc, shadow_program.transform_properties, false);
                  renderSpheres(shadow_cc,shadow_program.transform_properties,g_time);
                  renderDebris(shadow_cc,shadow_program.transform_properties,g_time);
               }


               if(g_time >= g_subdiv_transformation_time)
               {
                  subdiv_program.use();
                  SET_UNIFORM(subdiv_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, (shadow_cc.world_to_view * rocket_jitter_xfrm * rocket_object_matrix).e);
                  SET_UNIFORM(subdiv_program, projection, glUniformMatrix4fv, 1, GL_FALSE, shadow_cc.view_to_clip.e);
                  //subdiv_program.uniformMatrix4fv("modelview", 1, GL_FALSE, shadow_cc.world_to_view.e);
                  //subdiv_program.uniformMatrix4fv("projection", 1, GL_FALSE, shadow_cc.view_to_clip.e);
                  renderSubdiv();
                  shadow_program.use();
               }

               CHECK_FOR_GL_ERRORS("rendering into shadowbuffer");
            }
            glCullFace(GL_BACK);
         }

        primary_framebuffer.bindForDraw();
        glDepthMask(GL_TRUE);
        glClear(GL_DEPTH_BUFFER_BIT);

      // the sky
      {
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
         sky_program.use();
         static const Mat4 sky_object_matrix = Mat4::scale(Vec3(1024));
         const Mat4 zcrunch=Mat4::scale(Vec3(1,1,1.0/256.0));
         glUniformMatrix4fv(sky_program.transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (view_cc.world_to_view * sky_object_matrix).e);
         glUniformMatrix4fv(sky_program.transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, (zcrunch * view_cc.view_to_clip).e);
         glUniformMatrix4fv(sky_program.transform_properties.object_to_clip_matrix_location, 1, GL_FALSE, (zcrunch * view_cc.world_to_clip * sky_object_matrix).e);
         glUniformMatrix4fv(sky_program.transform_properties.previous_object_to_clip_matrix_location, 1, GL_FALSE, (zcrunch * view_cc.previous_world_to_clip * sky_object_matrix).e);
         glDisable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         glCullFace(GL_FRONT);
         assert(g_cube_vbo != 0);
         glBindBuffer(GL_ARRAY_BUFFER, g_cube_vbo);
         assert(g_cube_ebo != 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_cube_ebo);
         glEnableVertexAttribArray(0);
         glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (const GLvoid*)0);
         glDrawElements(GL_TRIANGLES, 6 * 2 * 3, GL_UNSIGNED_BYTE, (const GLvoid*)0);
         glDisableVertexAttribArray(0);
         glBindBuffer(GL_ARRAY_BUFFER, 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
         glCullFace(GL_BACK);
         glDepthMask(GL_TRUE);
         CHECK_FOR_GL_ERRORS("rendering the sky cube");
      }


        glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);
        glEnable(GL_DEPTH_TEST);

        if(g_time < g_explosion_time)
        {
           shadow_program.use();
           glEnable(GL_POLYGON_OFFSET_FILL);
           glPolygonOffset(2.0,4.0);
#if 1
           g_orion->renderEntities(view_cc, shadow_program.transform_properties, false);
           renderSpheres(view_cc,shadow_program.transform_properties,g_time);
#endif
           renderDebris(view_cc,shadow_program.transform_properties,g_time);
        }


        glDisable(GL_POLYGON_OFFSET_FILL);
        primary_program.use();

        //SET_UNIFORM(primary_program, normal_offset_scale, glUniform1f, (vp.camera_entity_index == camera_003_node_index || vp.camera_entity_index == camera_006_node_index || vp.camera_entity_index == camera_007_node_index) ? 0.01 : 0.06);
        SET_UNIFORM(primary_program, normal_offset_scale, glUniform1f, .08);

        SET_UNIFORM(primary_program, world_to_shadow[0], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[0].e);
        SET_UNIFORM(primary_program, world_to_shadow[1], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[1].e);
        SET_UNIFORM(primary_program, world_to_shadow[2], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[2].e);
        SET_UNIFORM(primary_program, world_to_shadow[3], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[3].e);
        SET_UNIFORM(primary_program, world_to_shadow[4], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[4].e);
        //primary_program.uniformMatrix4fv("world_to_shadow[0]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[0]).e);
        //primary_program.uniformMatrix4fv("world_to_shadow[1]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[1]).e);
        //primary_program.uniformMatrix4fv("world_to_shadow[2]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[2]).e);
        //primary_program.uniformMatrix4fv("world_to_shadow[3]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[3]).e);
        glDepthMask(GL_TRUE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        glEnable(GL_DEPTH_TEST);
        setTextureUnit(0, GL_TEXTURE_2D, g_noise_tex2);
        setTextureUnit(1, GL_TEXTURE_2D_ARRAY, shadowbuffers_texs[shadowbuffer_selection]);
        setTextureUnit(3, GL_TEXTURE_3D, svt0);
        setTextureUnit(4, GL_TEXTURE_2D, paper_texture);
        setTextureUnit(5, GL_TEXTURE_3D, svt1);
        SET_UNIFORM(primary_program, shadow_voxel_tween, glUniform1f, shadow_voxel_tween);
        SET_UNIFORM(primary_program, shadow_voxel_scale, glUniform1f, mix(0.1, 1.0, smoothstep(0.0, 8.0, g_time - g_particles_activate_time)));
        //primary_program.uniform1f("shadow_voxel_tween", shadow_voxel_tween);
        //primary_program.uniform1f("shadow_voxel_scale", mix(0.1, 1.0, smoothstep(0.0, 8.0, g_time - g_particles_activate_time)));
#if 1
        if(g_time > g_explosion_time)
            g_orion->renderSingleEntity(view_cc, revision_logo_node_index, primary_program.transform_properties, false);
        else
        {
            g_orion->renderEntities(view_cc, primary_program.transform_properties, false);
            renderSpheres(view_cc,primary_program.transform_properties,g_time);
            renderDebris(view_cc,primary_program.transform_properties,g_time);
        }
#endif
      // blossoms
      if(g_time > g_blossom_time && g_time < g_explosion_time)
      {
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
         Ran ran(4323);
         Ran ran2(4323);
         for(uint32 i=0;i<g_blossom_locations.size();i+=1)
         {
            const float t = (g_time - g_blossom_time) - float(i) * 0.01;
            const float s = std::max(0.0f, smoothstep(0.0f,0.5f,t) * 0.11f - smoothstep(0.5f,0.65f,t) * 0.01f)*mix(0.8,2.0,ran2.doub());
            const float rx = ran.doub() * M_PI * 2;
            const float ry = ran.doub() * M_PI * 2;
            const Mat4 object_matrix = Mat4::translation(g_blossom_locations[i]) * Mat4::scale(Vec3(s)) * Mat4::rotation(ry,Vec3(0,1,0)) * Mat4::rotation(rx,Vec3(1,0,0));
            glUniformMatrix4fv(primary_program.transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (view_cc.world_to_view * object_matrix).e);
            glUniformMatrix4fv(primary_program.transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, view_cc.view_to_clip.e);
            glUniformMatrix4fv(primary_program.transform_properties.object_to_clip_matrix_location, 1, GL_FALSE, (view_cc.world_to_clip * object_matrix).e);
            glUniformMatrix4fv(primary_program.transform_properties.previous_object_to_clip_matrix_location, 1, GL_FALSE, (view_cc.previous_world_to_clip * object_matrix).e);
            glUniformMatrix4fv(primary_program.transform_properties.object_to_world_matrix_location, 1, GL_FALSE, object_matrix.e);
            glUniform1f(primary_program.transform_properties.material_index_location, 1.0f);
            assert(g_cube_vbo != 0);
            glBindBuffer(GL_ARRAY_BUFFER, g_cube_vbo);
            assert(g_cube_ebo != 0);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_cube_ebo);
            glEnableVertexAttribArray(0);
            glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (const GLvoid*)0);
            glDrawElements(GL_TRIANGLES, 6 * 2 * 3, GL_UNSIGNED_BYTE, (const GLvoid*)0);
            glDisableVertexAttribArray(0);
            glBindBuffer(GL_ARRAY_BUFFER, 0);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
            CHECK_FOR_GL_ERRORS("rendering blossom");
         }
      }


         // chunks
         if(g_time > g_explosion_time)
         {
           primary_framebuffer.bindForDraw();
           chunk_program.use();
           SET_UNIFORM(chunk_program, world_to_shadow[0], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[0].e);
           SET_UNIFORM(chunk_program, world_to_shadow[1], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[1].e);
           SET_UNIFORM(chunk_program, world_to_shadow[2], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[2].e);
           SET_UNIFORM(chunk_program, world_to_shadow[3], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[3].e);
           //chunk_program.uniformMatrix4fv("world_to_shadow[0]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[0]).e);
           //chunk_program.uniformMatrix4fv("world_to_shadow[1]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[1]).e);
           //chunk_program.uniformMatrix4fv("world_to_shadow[2]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[2]).e);
           //chunk_program.uniformMatrix4fv("world_to_shadow[3]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[3]).e);
           SET_UNIFORM(chunk_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, view_cc.world_to_view.e);
           SET_UNIFORM(chunk_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);
           //chunk_program.uniformMatrix4fv("modelview", 1, GL_FALSE, view_cc.world_to_view.e);
           //chunk_program.uniformMatrix4fv("projection", 1, GL_FALSE, view_cc.view_to_clip.e);
           SET_UNIFORM(chunk_program, tween, glUniform1f, 1.0f - float(next_chunk_logic_update - ct) / float(chunk_update_interval));
           //chunk_program.uniform1f("tween", 1.0f - float(next_chunk_logic_update - ct) / float(chunk_update_interval));
           glDepthMask(GL_TRUE);
           glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
           glEnable(GL_DEPTH_TEST);
           setTextureUnit(0, GL_TEXTURE_2D, g_noise_tex2);
           setTextureUnit(1, GL_TEXTURE_2D_ARRAY, shadowbuffers_texs[shadowbuffer_selection]);
           setTextureUnit(2, GL_TEXTURE_2D, paper_texture);
           setTextureUnit(3, GL_TEXTURE_2D, g_chunk_textures[chunk_framebuffer_flipper ^ 1]);
           setTextureUnit(4, GL_TEXTURE_2D, g_chunk_textures[chunk_framebuffer_flipper]);

            glBindBuffer(GL_ARRAY_BUFFER,g_chunk_vbo);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,g_chunk_ebo);
            glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,0,(const GLvoid*)0);
            glVertexAttribPointer(1,2,GL_FLOAT,GL_FALSE,0,(const GLvoid*)g_chunk_vertex_data_length);
            glEnableVertexAttribArray(0);
            glEnableVertexAttribArray(1);
            uint32 idx=0;
            for(uint32 i=0;i<num_materials;++i)
            {
               const uint32 c=g_material_chunk_counts[i];
               if(c>0)
               {
                  assert(chunk_program.transform_properties.material_index_location >= 0);
                  if(i==water_surface_material)
                     glUniform1f(chunk_program.transform_properties.material_index_location, 1.0f);
                  else
                     glUniform1f(chunk_program.transform_properties.material_index_location, (float(i + 1) + 0.5f) / (float)((num_materials + 1) * 2));
                  glDrawElements(GL_TRIANGLES,c*24,GL_UNSIGNED_INT,(const GLvoid*)(sizeof(GLuint)*idx*24));
                  idx+=c;
               }
            }
            glDisableVertexAttribArray(0);
            glDisableVertexAttribArray(1);
            glBindBuffer(GL_ARRAY_BUFFER,0);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,0);

            CHECK_FOR_GL_ERRORS("rendering chunks");
        }



         // subdiv
         if(g_time >= g_subdiv_transformation_time && g_time < g_explosion_time)
         {
           primary_framebuffer.bindForDraw();
            {
               const GLenum bs[2] = { GL_COLOR_ATTACHMENT0, GL_COLOR_ATTACHMENT1 };
               glDrawBuffers(2, bs);
               CHECK_FOR_GL_ERRORS("setting drawbuffers for subdiv");
            }
           subdiv_program.use();
           SET_UNIFORM(subdiv_program, world_to_shadow[0], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[0].e);
           SET_UNIFORM(subdiv_program, world_to_shadow[1], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[1].e);
           SET_UNIFORM(subdiv_program, world_to_shadow[2], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[2].e);
           SET_UNIFORM(subdiv_program, world_to_shadow[3], glUniformMatrix4fv, 1, GL_FALSE, base_shadow_world_to_clip_biased[3].e);
           //subdiv_program.uniformMatrix4fv("world_to_shadow[0]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[0]).e);
           //subdiv_program.uniformMatrix4fv("world_to_shadow[1]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[1]).e);
           //subdiv_program.uniformMatrix4fv("world_to_shadow[2]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[2]).e);
           //subdiv_program.uniformMatrix4fv("world_to_shadow[3]", 1, GL_FALSE, (bias_matrix * base_shadow_world_to_clip[3]).e);
           SET_UNIFORM(subdiv_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, (view_cc.world_to_view * rocket_jitter_xfrm * rocket_object_matrix).e);
           SET_UNIFORM(subdiv_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);
           //subdiv_program.uniformMatrix4fv("modelview", 1, GL_FALSE, view_cc.world_to_view.e);
           //subdiv_program.uniformMatrix4fv("projection", 1, GL_FALSE, view_cc.view_to_clip.e);
           SET_UNIFORM(subdiv_program, time, glUniform1f, std::max(0.0, g_time-g_subdiv_transformation_time)*2.0f);
           //subdiv_program.uniform1f("time", std::max(0.0, g_time-(2*60+13))*2.0f);
           {
             assert(rocket_node_index < g_orion->entities.size());
             assert(rocket_ent);
             const Vec3 p = rocket_ent->global_xfrm.column(3);
             SET_UNIFORM(subdiv_program, spin_pivot, glUniform3f, p.x, p.y, p.z);
             //subdiv_program.uniform3f("spin_pivot", p.x, p.y, p.z);
           }
            glDisable(GL_CULL_FACE);
            glDepthMask(GL_TRUE);
            glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
            glEnable(GL_DEPTH_TEST);
            glBindBuffer(GL_ARRAY_BUFFER, g_subdiv_vbo);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_subdiv_ebo);
            glEnableVertexAttribArray(0);
            glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,0,(const GLvoid*)0);
            setTextureUnit(1, GL_TEXTURE_2D_ARRAY, shadowbuffers_texs[shadowbuffer_selection]);
            uint index = 0;
            for(uint i = 0; i < num_materials * num_materials; ++i)
            {
               const uint32 c = g_material_subdiv_counts[i];
               if(c > 0)
               {
                  const uint32 m0 = i % num_materials;
                  const uint32 m1 = i / num_materials;
                  assert(subdiv_program.transform_properties.material_index_location >= 0);
                  glUniform1f(subdiv_program.transform_properties.material_index_location, float((m0 + 1) + 0.5) / (float)((num_materials + 1) * 2));
                  SET_UNIFORM(subdiv_program, material2, glUniform1f, float((m1 + 1) + 0.5) / (float)((num_materials + 1) * 2));
                  glDrawElements(GL_TRIANGLES_ADJACENCY,c * 6,GL_UNSIGNED_SHORT,(const GLvoid*)(index * sizeof(GLushort) * 6));
                  index += c;
                  CHECK_FOR_GL_ERRORS("rendering subdived meshes");
               }
            }
            assert(index == g_final_subdiv_triangle_count);
            glDisableVertexAttribArray(0);
            glBindBuffer(GL_ARRAY_BUFFER, 0);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
            primary_framebuffer.bindForDraw();
            glEnable(GL_CULL_FACE);
            CHECK_FOR_GL_ERRORS("rendering subdived meshes (end)");
         }



#if RAYTRACE
         if(g_time < g_explosion_time)
         {
            raytrace_program.use();
            SET_UNIFORM(raytrace_program, spread_level, glUniform1f, (g_time - (g_particles_activate_time+24.0)) * 100.0);
            //raytrace_program.uniform1f("spread_level",(g_time - (g_particles_activate_time+24.0)) * 100.0);
            for(uint32 i = 0; i < raytraced_surface_node_count; ++i)
            {
               const uint32 sidx = raytraced_surface_nodes[i * 2 + 0];
               const uint32 gidx = raytraced_surface_nodes[i * 2 + 1];
               Entity* surface_ent = g_orion->entities[sidx];
               assert(surface_ent);
               Entity* ground_ent = g_orion->entities[gidx];
               assert(ground_ent);
               assert(surface_ent->raytracer);
               const uint32 s_mesh_count = node_mesh_offsets[sidx + 1] - node_mesh_offsets[sidx];
               const uint32 g_mesh_count = node_mesh_offsets[gidx + 1] - node_mesh_offsets[gidx];
               assert(s_mesh_count == 1);
               assert(g_mesh_count == 1);
               const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[gidx]];

               const BVHData& bvh = raytrace_bvh_data[i];

               setTextureUnit(0, GL_TEXTURE_2D, bvh.node0_tex);
               setTextureUnit(1, GL_TEXTURE_2D, bvh.node1_tex);
               setTextureUnit(2, GL_TEXTURE_2D, bvh.edge0_tex);
               setTextureUnit(3, GL_TEXTURE_2D, bvh.edge1_tex);
               setTextureUnit(4, GL_TEXTURE_2D, bvh.triangle_tex);
               setTextureUnit(5, GL_TEXTURE_2D, g_noise_tex2);

               SET_UNIFORM(raytrace_program, surface_to_ground, glUniformMatrix4fv, 1, GL_FALSE, (ground_ent->global_xfrm.inverse() * surface_ent->global_xfrm).e);
               //raytrace_program.uniformMatrix4fv("surface_to_ground", 1, GL_FALSE, (ground_ent->global_xfrm.inverse() * surface_ent->global_xfrm).e);
               SET_UNIFORM(raytrace_program, ground_to_view, glUniformMatrix4fv, 1, GL_FALSE, (view_cc.world_to_view * ground_ent->global_xfrm).e);
               //raytrace_program.uniformMatrix4fv("ground_to_view", 1, GL_FALSE, (view_cc.world_to_view * ground_ent->global_xfrm).e);
               SET_UNIFORM(raytrace_program, time, glUniform1f, g_time);
               //raytrace_program.uniform1f("time", g_time);
               SET_UNIFORM(raytrace_program, material, glUniform1f, (float(mesh_materials[mesh_index] + 1 + num_materials * 1) + 0.5f) / (float)((num_materials + 1) * 2));
               //raytrace_program.uniform1f("material", (float(mesh_materials[mesh_index] + 1 + num_materials * 1) + 0.5f) / (float)((num_materials + 1) * 2));

               g_orion->renderSingleEntity(view_cc, sidx, raytrace_program.transform_properties, false);
            }
         }
#endif

#if 0
         // smoketrail
         if(g_time < g_explosion_time && g_time >= g_rocket_liftoff_time)
         {
            primary_framebuffer.bindForDraw();
/*
            {
               const GLenum bs[2] = { GL_COLOR_ATTACHMENT0, GL_COLOR_ATTACHMENT1 };
               glDrawBuffers(2, bs);
               CHECK_FOR_GL_ERRORS("setting drawbuffers for smoketrail");
            }*/
            smoketrail_program.use();
            SET_UNIFORM(smoketrail_program, time, glUniform1f, std::max(0.0, g_time-g_rocket_liftoff_time));
            //smoketrail_program.uniform1f("time",ct/1000.0);
            glUniformMatrix4fv(smoketrail_program.transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (view_cc.world_to_view * smoketrail_object_matrix).e);
            glUniformMatrix4fv(smoketrail_program.transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, view_cc.view_to_clip.e);
            glUniform1f(smoketrail_program.transform_properties.material_index_location, float(0 + 0.5) / (float)((num_materials + 1) * 2));
            glBindBuffer(GL_ARRAY_BUFFER,g_smoketrail_grid_vbo);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,g_smoketrail_grid_ebo);
            glEnableVertexAttribArray(0);
            glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,(const GLvoid*)0);
            glDrawElements(GL_TRIANGLES,g_smoketrail_index_count,GL_UNSIGNED_SHORT,(const GLvoid*)0);
            glDisableVertexAttribArray(0);
            glBindBuffer(GL_ARRAY_BUFFER,0);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,0);
            primary_framebuffer.bindForDraw();
            CHECK_FOR_GL_ERRORS("rendering the smoketrail");
         }
#endif





      // spheres (emissive)
      if(g_time < g_explosion_time)
      {
         sphere_program.use();
         crystal_framebuffer.bindForDraw();
         glFramebufferRenderbuffer(GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, primary_framebuffer.depth_rbo);
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "sphere emission rendering");
         SET_UNIFORM(sphere_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);

          assert(g_sphere_vbo != 0);
          assert(g_sphere_ebo != 0);

          glBindBuffer(GL_ARRAY_BUFFER, g_sphere_vbo);
          glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_sphere_ebo);


         //glEnable(GL_CULL_FACE);
         glEnable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
         glClear(GL_COLOR_BUFFER_BIT);
         glEnableVertexAttribArray(0);
         glEnable(GL_BLEND);
         glBlendFunc(GL_ONE, GL_ONE);
         glBlendEquation(GL_FUNC_ADD);

          glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

         const double t=g_time;
         for(size_t i=0;i<g_spheres.size();++i)
         {
            const Sphere& s=g_spheres[i];

            if(!s.active(t))
               continue;

            const Mat4 global_xfrm = s.evaluate(t);
            const Vec3 c = mix(Vec3(2.7,2.0,1.0),Vec3(0,0,0),std::pow(std::max(0.0,(t-s.appear_time)/1.6),0.25));

            if(c.x > 1e-4 || c.y > 1e-4 || c.z > 1e-4)
            {
               SET_UNIFORM(sphere_program, colour, glUniform3f, c.x, c.y, c.z);
               SET_UNIFORM(sphere_program, object_to_world, glUniformMatrix4fv, 1, GL_FALSE, global_xfrm.e);
               SET_UNIFORM(sphere_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, (view_cc.world_to_view * global_xfrm).e);

               assert(g_sphere_index_count > 0);
               glDrawElements(GL_TRIANGLES, g_sphere_index_count, GL_UNSIGNED_BYTE, (const GLvoid*)0);
               CHECK_FOR_GL_ERRORS("end of single sphere render (emission)");
            }
          }

          glDisableVertexAttribArray(0);

         glBindBuffer(GL_ARRAY_BUFFER, 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
         glDisable(GL_BLEND);
         CHECK_FOR_GL_ERRORS("rendering the sphere emissions");
      }

      static const Vec3 crystal_colours[2] = { Vec3(0.2,0.23,0.4), Vec3(0.4,0.2,0.4) };
      static const Vec3 crystal_light_colours[2] = { Vec3(0.5,0.7,1.0), Vec3(1.0,0.7,1.0) };
      static const int crystal_colour_count=sizeof(crystal_colours)/sizeof(crystal_colours[0]);

      // crystals
      if(g_time < g_explosion_time)
      {
         crystal_program.use();
         crystal_framebuffer.bindForDraw();
         glFramebufferRenderbuffer(GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, primary_framebuffer.depth_rbo);
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "crystal rendering");
         SET_UNIFORM(crystal_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);
         //crystal_program.uniformMatrix4fv("projection", 1, GL_FALSE, view_cc.view_to_clip.e);
         setTextureUnit(0,GL_TEXTURE_2D,g_noise_tex);

         GLuint array_buffer = 0, element_array_buffer = 0;

         glBindBuffer(GL_ARRAY_BUFFER, array_buffer);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer);

         glEnable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
         //glClear(GL_COLOR_BUFFER_BIT);
         glEnableVertexAttribArray(0);
         glEnable(GL_BLEND);
         glBlendFunc(GL_ONE, GL_ONE);
         glBlendEquation(GL_FUNC_ADD);


         const uint32 i=crystals_node_index;
         const uint32 child_count = node_child_offsets[i + 1] - node_child_offsets[i];
         for(uint32 j = 0; j < child_count; ++j)
         {
            const uint32 c = node_child_indices[node_child_offsets[i] + j];
            const uint32 mesh_count = node_mesh_offsets[c + 1] - node_mesh_offsets[c];

            if(mesh_count==0)
               continue;

            const Mat4& global_xfrm = g_orion->entities[c]->global_xfrm;
            SET_UNIFORM(crystal_program, object_to_world, glUniformMatrix4fv, 1, GL_FALSE, global_xfrm.e);
            //crystal_program.uniformMatrix4fv("object_to_world", 1, GL_FALSE, global_xfrm.e);
            SET_UNIFORM(crystal_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, (view_cc.world_to_view * global_xfrm).e);
            //crystal_program.uniformMatrix4fv("modelview", 1, GL_FALSE, (view_cc.world_to_view * global_xfrm).e);
            const Vec3& col=crystal_colours[j%crystal_colour_count];
            SET_UNIFORM(crystal_program, in_colour, glUniform3f, col.x,col.y,col.z);

            for(uint32 j = 0; j < mesh_count; ++j)
            {
               const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[c] + j];
               assert(mesh_index < g_orion->mesh_cache.size());
               const MeshCacheItem* m = g_orion->mesh_cache[mesh_index];
               assert(m);

               if(m->array_buffer != array_buffer)
               {
                 array_buffer = m->array_buffer;
                 glBindBuffer(GL_ARRAY_BUFFER, array_buffer);
                 glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, sizeof(float) * 6, 0);
               }

               if(m->element_array_buffer != element_array_buffer)
               {
                 element_array_buffer = m->element_array_buffer;
                 glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer);
               }

               glDrawElements(GL_TRIANGLES, m->count, m->type, m->indices);
            }
         }


         glDisableVertexAttribArray(0);
         glBindBuffer(GL_ARRAY_BUFFER, 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
         glDisable(GL_BLEND);
         CHECK_FOR_GL_ERRORS("rendering the crystals");

         bloom_program.use();
         glDisable(GL_DEPTH_TEST);
         glDepthMask(GL_FALSE);
         GLuint prev_tex=crystal_framebuffer.texs[0];
         const uint passes = 4;
         static const Vec2 s = Vec2(aspect_ratio, 0), t = Vec2(0, 1);
         static const Vec2 directions[passes] = { s * 0.01, t * 0.01, s * 0.01, t * 0.01 };
         for(uint i=0;i<passes;++i)
         {
            if(i==(passes-1))
            {
               crystal_framebuffer.bindForDraw();
               glEnable(GL_BLEND);
               glBlendFunc(GL_CONSTANT_COLOR,GL_ONE);
               glBlendEquation(GL_FUNC_ADD);
               glBlendColor(0.5,0.5,0.5,0.5);
            }
            else
               bloom_framebuffers[i & 1].bindForDraw();
            SET_UNIFORM(bloom_program, direction, glUniform2f, directions[i].x,directions[i].y);
            //bloom_program.uniform2f("direction",directions[i].x,directions[i].y);
            setTextureUnit(0,GL_TEXTURE_2D,prev_tex);
            renderQuad();
            prev_tex=bloom_framebuffers[i & 1].texs[0];
         }

         // lights
         assert(g_cube_vbo != 0);
         glBindBuffer(GL_ARRAY_BUFFER, g_cube_vbo);
         assert(g_cube_ebo != 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, g_cube_ebo);
         glEnableVertexAttribArray(0);
         glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (const GLvoid*)0);
         light_program.use();
         glUniformMatrix4fv(light_program.transform_properties.view_to_clip_matrix_location, 1, GL_FALSE, view_cc.view_to_clip.e);
         setTextureUnit(0, GL_TEXTURE_2D, primary_framebuffer.texs[0]);
         setTextureUnit(1, GL_TEXTURE_2D, primary_framebuffer.texs[1]);
         setTextureUnit(2, GL_TEXTURE_2D, g_noise_tex2);
         glEnable(GL_DEPTH_TEST);
         glDepthFunc(GL_GEQUAL);
         glDepthMask(GL_FALSE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
         glEnable(GL_BLEND);
         glBlendFunc(GL_ONE, GL_ONE);
         glBlendEquation(GL_FUNC_ADD);

         for(uint32 j = 0; j < child_count; ++j)
         {
            const uint32 c = node_child_indices[node_child_offsets[i] + j];
            const uint32 mesh_count = node_mesh_offsets[c + 1] - node_mesh_offsets[c];

            if(mesh_count==0)
               continue;

            const Vec3& col=crystal_light_colours[j%crystal_colour_count];
            SET_UNIFORM(light_program, in_colour, glUniform3f, col.x,col.y,col.z);

            const Mat4& global_xfrm = g_orion->entities[c]->global_xfrm;

            for(uint32 j = 0; j < mesh_count; ++j)
            {
               const uint32 mesh_index = node_mesh_indices[node_mesh_offsets[c] + j];

               const Vec3 box_min(bounding_boxes[mesh_index*6+0],bounding_boxes[mesh_index*6+1],bounding_boxes[mesh_index*6+2]);
               const Vec3 box_max(bounding_boxes[mesh_index*6+3],bounding_boxes[mesh_index*6+4],bounding_boxes[mesh_index*6+5]);
               const Vec3 world_center = global_xfrm * Vec3((box_min + box_max) * 0.5f);
               const Mat4 object_to_world = Mat4::translation(world_center) * Mat4::scale(Vec3(-1));

               glUniformMatrix4fv(light_program.transform_properties.object_to_view_matrix_location, 1, GL_FALSE, (view_cc.world_to_view * object_to_world).e);
               glDrawElements(GL_TRIANGLES, 6 * 2 * 3, GL_UNSIGNED_BYTE, (const GLvoid*)0);
            }
         }

         glDepthFunc(GL_LEQUAL);
         glDisableVertexAttribArray(0);
         glBindBuffer(GL_ARRAY_BUFFER, 0);
         glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
         CHECK_FOR_GL_ERRORS("rendering the crystal lights");
      }


        glEnable(GL_DEPTH_TEST);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        glDepthMask(GL_FALSE);

        glBlendFunc(GL_ONE, GL_ONE);
        glBlendEquation(GL_FUNC_ADD);
        glDisable(GL_BLEND);

        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);

/*
        final_framebuffer.bindForDraw();
        secondary_program.use();
        SET_UNIFORM(secondary_program, viewspace_sun_direction, glUniform3f, vsd.x, vsd.y, vsd.z);
        //secondary_program.uniform3f("viewspace_sun_direction", vsd.x, vsd.y, vsd.z);
        SET_UNIFORM(secondary_program, viewspace_up, glUniform3f, vsu.x, vsu.y, vsu.z);
        //secondary_program.uniform3f("viewspace_up", vsu.x, vsu.y, vsu.z);
        SET_UNIFORM(secondary_program, ray_xfrm, glUniformMatrix3fv, 1, GL_FALSE, view_ray_xfrm);
        //secondary_program.uniformMatrix3fv("ray_xfrm", 1, GL_FALSE, view_ray_xfrm);
        SET_UNIFORM(secondary_program, view_to_world, glUniformMatrix4fv, 1, GL_FALSE, view_cc.world_to_view.inverse().e);
        //secondary_program.uniformMatrix4fv("view_to_world", 1, GL_FALSE, view_cc.world_to_view.inverse().e);

        assert(g_orion);
        //SET_UNIFORM(secondary_program, stars, glUniform1f, (vp.camera_entity_index == camera_016_node_index) ? ((g_time - g_orion->getMostRecentCameraMarkerTime(g_time)) * 0.063f) : 0.0f);
        //SET_UNIFORM(secondary_program, stars, glUniform1f, smoothstep(50.0,60.0,g_time));
        SET_UNIFORM(secondary_program, stars, glUniform1f, 0);


         if(g_time >= g_explosion_time)
         {
            SET_UNIFORM(secondary_program, white_background, glUniform1f, 1.0f);
         }
         else
         {
            SET_UNIFORM(secondary_program, white_background, glUniform1f, 0.0f);
         }
         SET_UNIFORM(secondary_program, darkness, glUniform1f, smoothstep(g_explosion_time + 5, g_explosion_time + 8, g_time));


         SET_UNIFORM(secondary_program, fade, glUniform1f, 1);


        {
            const Vec3 cp = g_orion->entities[cave_node_index]->global_xfrm.column(3);
            SET_UNIFORM(secondary_program, cave_point, glUniform3f, cp.x, cp.y, cp.z);
            //secondary_program.uniform3f("cave_point", cp.x, cp.y, cp.z);
        }
*/


        final_framebuffer.bindForDraw();
        secondary_program.use();
        SET_UNIFORM(secondary_program, viewspace_sun_direction, glUniform3f, vsd.x, vsd.y, vsd.z);
        //secondary_program.uniform3f("viewspace_sun_direction", vsd.x, vsd.y, vsd.z);
        SET_UNIFORM(secondary_program, viewspace_up, glUniform3f, vsu.x, vsu.y, vsu.z);
        //secondary_program.uniform3f("viewspace_up", vsu.x, vsu.y, vsu.z);
        SET_UNIFORM(secondary_program, ray_xfrm, glUniformMatrix3fv, 1, GL_FALSE, view_ray_xfrm);
        //secondary_program.uniformMatrix3fv("ray_xfrm", 1, GL_FALSE, view_ray_xfrm);
        SET_UNIFORM(secondary_program, view_to_world, glUniformMatrix4fv, 1, GL_FALSE, view_cc.world_to_view.inverse().e);
        //secondary_program.uniformMatrix4fv("view_to_world", 1, GL_FALSE, view_cc.world_to_view.inverse().e);

        //SET_UNIFORM(secondary_program, stars, glUniform1f, (vp.camera_entity_index == camera_016_node_index) ? ((g_time - g_orion->getMostRecentCameraMarkerTime(g_time)) * 0.063f) : 0.0f);
        //SET_UNIFORM(secondary_program, stars, glUniform1f, smoothstep(50.0,60.0,g_time));
        SET_UNIFORM(secondary_program, stars, glUniform1f, smoothstep(145.0,160.0,g_time));

         if(g_time >= g_explosion_time)
         {
            SET_UNIFORM(secondary_program, white_background, glUniform1f, 1.0f);
         }
         else
         {
            SET_UNIFORM(secondary_program, white_background, glUniform1f, 0.0f);
         }
         SET_UNIFORM(secondary_program, darkness, glUniform1f, smoothstep(g_explosion_time + 5, g_explosion_time + 8, g_time));


           // find the nearest marker
          {
              const uint32 frame = (uint32)std::floor(g_time * 24.0);
              uint32 marker0_frame = 0;
              uint32 marker1_frame = ~0U;
              uint32 last_marker = 0;
              for(uint32 i = 0; i < marker_count; ++i)
              {
                  const Marker& m = markers[i];
                  if(m.frame <= frame && m.node_index != 0 && m.frame >= marker0_frame)
                  {
                      marker0_frame = m.frame;
                      last_marker = i;
                  }
              }
              for(uint32 i = 0; i < marker_count; ++i)
              {
                  const Marker& m = markers[i];
                  if(m.frame > frame && m.node_index != 0 && m.frame <= marker1_frame)
                  {
                      marker1_frame = m.frame;
                  }
              }

             assert(marker0_frame <= frame);
             assert(marker1_frame >= frame);
             assert(marker0_frame <= marker1_frame);

              float transition_fade=1;
              if(g_time < g_particles_activate_time)
              {
                 if(last_marker >= 3)
                 {
                    transition_fade=smoothstep(0.0, 1.0, g_time - marker0_frame * 1.0 / 24.0) - smoothstep(-1.0, 0.0, g_time - marker1_frame * 1.0 / 24.0);
                 }
                 else
                 {
                    transition_fade=smoothstep(0.0, 10.0, g_time - marker0_frame * 1.0 / 24.0) - smoothstep(-2.0, 0.0, g_time - marker1_frame * 1.0 / 24.0);
                 }
              }
              transition_fade=mix(transition_fade, 5.0f, smoothstep(60.0+39.0, 60.0+41.0, g_time) - smoothstep(60.0+41.0, 60.0+43.0, g_time)); // second cave exit
              transition_fade=mix(transition_fade, 0.0f, smoothstep(2*60.0+59.0, 3*60.0+8.0, g_time)); // ending
              if(g_time<g_explosion_time)
                 for(int th=0;th<num_thumps;++th)
                 {
                     if(g_time>=(thumps[th]-.1))
                        transition_fade=mix(1.5,1.0,std::min(1.,(g_time-(thumps[th]-.1))/.1));
                 }
              SET_UNIFORM(secondary_program, fade, glUniform1f, mix(transition_fade, 8.0f, smoothstep(48.0, 51.0, g_time) - smoothstep(51.0, 60.0, g_time)));
          }

        {
            const Vec3 cp = g_orion->entities[cave_node_index]->global_xfrm.column(3);
            SET_UNIFORM(secondary_program, cave_point, glUniform3f, cp.x, cp.y, cp.z);
            //secondary_program.uniform3f("cave_point", cp.x, cp.y, cp.z);
        }

        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        setTextureUnit(1, GL_TEXTURE_2D, primary_framebuffer.texs[0]);
        setTextureUnit(2, GL_TEXTURE_2D, primary_framebuffer.texs[1]);
        setTextureUnit(3, GL_TEXTURE_2D, material_palette_tex);
        setTextureUnit(4, GL_TEXTURE_2D, primary_framebuffer.texs[2]);
        setTextureUnit(5, GL_TEXTURE_2D, crystal_framebuffer.texs[0]);
        setTextureUnit(6, GL_TEXTURE_2D, g_noise_tex2);
        setTextureUnit(7, GL_TEXTURE_CUBE_MAP, g_sky_cubemap);
        setTextureUnit(8, GL_TEXTURE_2D, g_stars_texture);
        renderQuad();





      // skyscrapers
      if(g_time > g_particles_activate_time && g_time < (g_explosion_time - 9.0))
      {
         skyscraper_program.use();
         glFramebufferRenderbuffer(GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, primary_framebuffer.depth_rbo);
         CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "skyscraper rendering");
         setTextureUnit(0,GL_TEXTURE_2D,g_noise_tex);
         setTextureUnit(2,GL_TEXTURE_2D,g_noise_tex2);

         glEnable(GL_DEPTH_TEST);
         glDepthMask(GL_TRUE);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
         glDisable(GL_BLEND);

         for(uint32 j=0;j<skyscraper_node_count;++j)
         {
            const uint32 n=skyscraper_nodes[j];
            setTextureUnit(1,GL_TEXTURE_CUBE_MAP,skyscraper_cubemap_texs[j]);
            g_orion->renderSingleEntity(view_cc, n, skyscraper_program.transform_properties, false);
         }

         CHECK_FOR_GL_ERRORS("rendering the skyscrapers");
      }

      glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);

         // particles
         if(g_time > g_particles_activate_time && g_time < g_particles_deactivate_time)
         {
            static bool particles_inited=false;
            static Uint32 partcles_start_time = 0;
            static Uint32 next_logic_update = 0;
            if(!particles_inited)
            {
               partcles_start_time = ct;
               next_logic_update = partcles_start_time;
               particles_inited=true;
            }
            for(uint32 i = 0; i < 128 && ct >= next_logic_update; ++i)
            {
               particle_framebuffers[particle_framebuffer_flipper].bindForDraw();
              glDisable(GL_DEPTH_TEST);
              glDepthMask(GL_FALSE);
              glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
               setTextureUnit(0, GL_TEXTURE_2D, particle_framebuffers[particle_framebuffer_flipper^1].texs[0]);
               assert(collision_voxel_texture != 0);
               setTextureUnit(1, GL_TEXTURE_3D, collision_voxel_texture);
               setTextureUnit(2, GL_TEXTURE_2D, g_noise_tex2);
               setTextureUnit(3, GL_TEXTURE_1D, particle_velocity_map);
               particle_init_program.use();
               {
                  double tt = (double(next_logic_update-partcles_start_time))/1000.0;
                  SET_UNIFORM(particle_init_program, time, glUniform1f, tt);
                  //particle_init_program.uniform1f("time",tt);
               }
               renderQuad();
               particle_update_program.use();
               setTextureUnit(0, GL_TEXTURE_2D, particle_framebuffers[particle_framebuffer_flipper].texs[0]);
               renderQuad();
               next_logic_update += particle_update_interval;
               particle_framebuffer_flipper ^= 1;
            }


           glBindBuffer(GL_ARRAY_BUFFER, particle_coord_vbo);
           glEnableVertexAttribArray(0);


            if((particle_shadow_frame_counter & 7)==0)
            {
                 glBindFramebuffer(GL_DRAW_FRAMEBUFFER,shadow_voxel_fbos[(particle_shadow_frame_counter / 8) & 1]);
                  {
                   const GLenum draw_buffers[] = { GL_COLOR_ATTACHMENT0 };
                   glDrawBuffers(sizeof(draw_buffers)/sizeof(draw_buffers[0]), draw_buffers);
                  }
                  CHECK_FRAMEBUFFER(GL_DRAW_FRAMEBUFFER, "particle voxel plot");
                 glDisable(GL_DEPTH_TEST);
                 glDepthMask(GL_FALSE);
                 glColorMask(GL_TRUE,GL_FALSE,GL_FALSE,GL_FALSE);
                 glViewport(0, 0, g_shadow_voxel_texture_width, g_shadow_voxel_texture_height);
                 glClear(GL_COLOR_BUFFER_BIT);
                 particle_plot_program.use();
                 SET_UNIFORM(particle_plot_program, tween, glUniform1f, 1.0f - float(next_logic_update - ct) / float(particle_update_interval));
                 //particle_plot_program.uniform1f("tween", 1.0f - float(next_logic_update - ct) / float(particle_update_interval));
                 glPointSize(4);
                 CHECK_FOR_GL_ERRORS("plotting particles: glPointSize");
                 setTextureUnit(0, GL_TEXTURE_2D, particle_framebuffers[particle_framebuffer_flipper ^ 1].texs[0]);
                 CHECK_FOR_GL_ERRORS("plotting particles: texture unit 0");
                 setTextureUnit(1, GL_TEXTURE_2D, particle_framebuffers[particle_framebuffer_flipper].texs[0]);
                 CHECK_FOR_GL_ERRORS("plotting particles: texture unit 1");
                 glEnable(GL_BLEND);
                 glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0, NULL);
                 CHECK_FOR_GL_ERRORS("plotting particles: glVertexAttribPointer");
                 glDrawArrays(GL_POINTS,0,particle_framebuffers[0].width*particle_framebuffers[0].height/1000);
                 CHECK_FOR_GL_ERRORS("plotting particles: glDrawArrays");
                 glDisable(GL_BLEND);
            }


            ++particle_shadow_frame_counter;

           glPointSize(2);

           final_framebuffer.bindForDraw();
           glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, g_generic_depth_rbo);
           particle_draw_program.use();
           SET_UNIFORM(particle_draw_program, fade, glUniform1f, transition_fade);
           SET_UNIFORM(particle_draw_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, view_cc.world_to_view.e);
           //particle_draw_program.uniformMatrix4fv("modelview", 1, GL_FALSE, view_cc.world_to_view.e);
           SET_UNIFORM(particle_draw_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);
           //particle_draw_program.uniformMatrix4fv("projection", 1, GL_FALSE, view_cc.view_to_clip.e);
           SET_UNIFORM(particle_draw_program, tween, glUniform1f, 1.0f - float(next_logic_update - ct) / float(particle_update_interval));
           //particle_draw_program.uniform1f("tween", 1.0f - float(next_logic_update - ct) / float(particle_update_interval));

           SET_UNIFORM(particle_draw_program, vertical_spread, glUniform1f, 1.0f);

           glEnable(GL_DEPTH_TEST);

            // dynamic particles
           const Framebuffer* fb = particle_framebuffers;

           setTextureUnit(0, GL_TEXTURE_2D, fb[particle_framebuffer_flipper ^ 1].texs[0]);
           setTextureUnit(1, GL_TEXTURE_2D, fb[particle_framebuffer_flipper].texs[0]);
           assert(svt0 != 0);
           setTextureUnit(2, GL_TEXTURE_3D, svt0);
           assert(collision_voxel_texture != 0);
           setTextureUnit(3, GL_TEXTURE_3D, collision_voxel_texture);
           assert(svt1 != 0);
           setTextureUnit(4, GL_TEXTURE_3D, svt1);
           SET_UNIFORM(particle_draw_program, shadow_voxel_tween, glUniform1f, shadow_voxel_tween);
           //particle_draw_program.uniform1f("shadow_voxel_tween",shadow_voxel_tween);

           glDepthMask(GL_TRUE);
           glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
           glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0, NULL);
           glDrawArrays(GL_POINTS,0,fb[0].width*fb[0].height);
           glDisableVertexAttribArray(0);
           glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, final_framebuffer.depth_rbo);
         }

         // static particles
         if(g_time > (g_particles_activate_time - 15) && g_time < (g_particles_activate_time + 6))
         {
           glEnable(GL_DEPTH_TEST);

           glBindBuffer(GL_ARRAY_BUFFER, particle_coord_vbo);
           glEnableVertexAttribArray(0);

           glPointSize(2);

           final_framebuffer.bindForDraw();
           glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, g_generic_depth_rbo);
           particle_draw_program.use();
           SET_UNIFORM(particle_draw_program, fade, glUniform1f, transition_fade);
           SET_UNIFORM(particle_draw_program, modelview, glUniformMatrix4fv, 1, GL_FALSE, (view_cc.world_to_view * Mat4::translation(Vec3(0.0f, -0.06f, 0.0f))).e);
           //particle_draw_program.uniformMatrix4fv("modelview", 1, GL_FALSE, view_cc.world_to_view.e);
           SET_UNIFORM(particle_draw_program, projection, glUniformMatrix4fv, 1, GL_FALSE, view_cc.view_to_clip.e);
           //particle_draw_program.uniformMatrix4fv("projection", 1, GL_FALSE, view_cc.view_to_clip.e);
           SET_UNIFORM(particle_draw_program, tween, glUniform1f, 0);
           //particle_draw_program.uniform1f("tween", 1.0f - float(next_logic_update - ct) / float(particle_update_interval));

           SET_UNIFORM(particle_draw_program, vertical_spread, glUniform1f, 0.0f);

           setTextureUnit(0, GL_TEXTURE_2D, staticparticle_framebuffer.texs[0]);
           setTextureUnit(1, GL_TEXTURE_2D, staticparticle_framebuffer.texs[0]);
           assert(svt0 != 0);
           setTextureUnit(2, GL_TEXTURE_3D, svt0);
           assert(collision_voxel_texture != 0);
           setTextureUnit(3, GL_TEXTURE_3D, collision_voxel_texture);
           assert(svt1 != 0);
           setTextureUnit(4, GL_TEXTURE_3D, svt1);
           SET_UNIFORM(particle_draw_program, shadow_voxel_tween, glUniform1f, shadow_voxel_tween);
           //particle_draw_program.uniform1f("shadow_voxel_tween",shadow_voxel_tween);

           glDepthMask(GL_TRUE);
           glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
           glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0, NULL);
           glDrawArrays(GL_POINTS,0,staticparticle_framebuffer.width*staticparticle_framebuffer.height);

           glDisableVertexAttribArray(0);
           glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, final_framebuffer.depth_rbo);

        }

      glDisable(GL_BLEND);

         // sparkle effect
        {
         glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_TRUE);

            bloom_program.use();
            glDisable(GL_DEPTH_TEST);
            glDepthMask(GL_FALSE);
            GLuint prev_tex=final_framebuffer.texs[0];
            const uint passes = 3;
            static const Vec2 aspect_scale = Vec2(aspect_ratio, 1);
            static const Vec2 s = Vec2(aspect_ratio, 0), t = Vec2(0, 1);
            static const Vec2 directions[passes] = { aspect_scale * 0.01, aspect_scale * 0.02, aspect_scale * 0.04 };
            for(uint i=0;i<passes;++i)
            {
               if(i==(passes-1))
               {
                  final_framebuffer.bindForDraw();
                  glEnable(GL_BLEND);
                  glBlendFunc(GL_CONSTANT_COLOR,GL_ONE);
                  glBlendEquation(GL_FUNC_ADD);
                  glBlendColor(0.5,0.5,0.5,1);
               }
               else
                  bloom_framebuffers[i & 1].bindForDraw();
               SET_UNIFORM(bloom_program, direction, glUniform2f, directions[i].x,directions[i].y);
               //bloom_program.uniform2f("direction",directions[i].x,directions[i].y);
               setTextureUnit(0,GL_TEXTURE_2D,prev_tex);
               renderQuad();
               prev_tex=bloom_framebuffers[i & 1].texs[0];
            }
         glDisable(GL_BLEND);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
       }
        {
         glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_TRUE);

            bloom_program.use();
            glDisable(GL_DEPTH_TEST);
            glDepthMask(GL_FALSE);
            GLuint prev_tex=final_framebuffer.texs[0];
            const uint passes = 3;
            static const Vec2 aspect_scale = Vec2(aspect_ratio, -1);
            static const Vec2 s = Vec2(aspect_ratio, 0), t = Vec2(0, 1);
            static const Vec2 directions[passes] = { aspect_scale * 0.01, aspect_scale * 0.02, aspect_scale * 0.04 };
            for(uint i=0;i<passes;++i)
            {
               if(i==(passes-1))
               {
                  final_framebuffer.bindForDraw();
                  glEnable(GL_BLEND);
                  glBlendFunc(GL_CONSTANT_COLOR,GL_ONE);
                  glBlendEquation(GL_FUNC_ADD);
                  glBlendColor(0.5,0.5,0.5,1);
               }
               else
                  bloom_framebuffers[i & 1].bindForDraw();
               SET_UNIFORM(bloom_program, direction, glUniform2f, directions[i].x,directions[i].y);
               //bloom_program.uniform2f("direction",directions[i].x,directions[i].y);
               setTextureUnit(0,GL_TEXTURE_2D,prev_tex);
               renderQuad();
               prev_tex=bloom_framebuffers[i & 1].texs[0];
            }
         glDisable(GL_BLEND);
         glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
       }

        motionblur_framebuffer.bindForDraw();
        motionblur_program.use();
        SET_UNIFORM(motionblur_program, scale, glUniform1f, (arg_motion_blur==0)?1.0:0.0);
        //motionblur_program.uniform1f("scale",1.0-smoothstep(144.0,145.0,g_time));
        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        setTextureUnit(0, GL_TEXTURE_2D, final_framebuffer.texs[0]);
        setTextureUnit(1, GL_TEXTURE_2D, primary_framebuffer.texs[0]);
        setTextureUnit(2, GL_TEXTURE_2D, primary_framebuffer.texs[2]);
        renderQuad();


        blur_program.use();
        const float blur_radius = 0.002 * (1.0f-smoothstep(g_info_appear_time-4.0,g_info_appear_time,g_time));
        float peripheralvalue = (1.0f - (smoothstep(49.0, 50.0, g_time) - smoothstep(50.0, 51.0, g_time)));
        for(int th=0;th<num_thumps;++th)
        {
            if(g_time>=(thumps[th]-.1))
               peripheralvalue=std::min(1.,.01+(g_time-(thumps[th]-.1))/.12);
        }
        SET_UNIFORM(blur_program, peripheral, glUniform1f, peripheralvalue);
        SET_UNIFORM(blur_program, direction, glUniform2f, blur_radius,0);
        //blur_program.uniform2f("direction",blur_radius,0);
        blur_framebuffer.bindForDraw();
        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        setTextureUnit(0, GL_TEXTURE_2D, motionblur_framebuffer.texs[0]);
        setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
        renderQuad();

        SET_UNIFORM(blur_program, direction, glUniform2f, 0.0, blur_radius / aspect_ratio);
        //blur_program.uniform2f("direction", 0.0, blur_radius / aspect_ratio);
        final_framebuffer.bindForDraw();
        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
        setTextureUnit(0, GL_TEXTURE_2D, blur_framebuffer.texs[0]);
        setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
        renderQuad();


        bindBackbuffer(GL_DRAW_FRAMEBUFFER);

        post_program.use();
        glDisable(GL_DEPTH_TEST);
        glDepthMask(GL_FALSE);
        glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);
        setTextureUnit(0, GL_TEXTURE_2D, final_framebuffer.texs[0]);
        setTextureUnit(1, GL_TEXTURE_2D, g_noise_tex);
        renderQuad();

        if(g_time>=g_info_appear_time)
        {
           info_program.use();
           SET_UNIFORM(info_program, alpha, glUniform1f, smoothstep(0.0,0.5,g_time-g_info_appear_time));
           renderInfo();
        }
#if 0
         primary_framebuffer.bindForRead(1);
         bindBackbuffer(GL_DRAW_FRAMEBUFFER);
         glBlitFramebuffer(0, 0, g_window_width, g_window_height, 0, 0, g_window_width, g_window_height, GL_COLOR_BUFFER_BIT, GL_LINEAR);
         glBindFramebuffer(GL_READ_FRAMEBUFFER, 0);
#endif
        CHECK_FOR_GL_ERRORS("end of frame");

#if CAPTURE
        {
             GLubyte* in_buffer = new GLubyte[g_window_width * g_window_height * 3];

             const int d=2;
             const int tw=g_window_width/d,th=g_window_height/d;

             glBindFramebuffer(GL_READ_FRAMEBUFFER, 0);
             glReadBuffer(GL_BACK);
             glPixelStorei(GL_PACK_ALIGNMENT,1);
             glPixelStorei(GL_UNPACK_ALIGNMENT,1);
             glReadPixels(0,0,g_window_width,g_window_height,GL_RGB,GL_UNSIGNED_BYTE,in_buffer);

             for(int y=0;y<g_window_height;++y)
               for(int x=0;x<g_window_width;++x)
               {
                  for(int ci=0;ci<3;++ci)
                     accumulated_mb_frames[(x+y*g_window_width)*3+ci]+=in_buffer[(x+y*g_window_width)*3+ci];
               }

            if((local_capture_frame % mb_frame_count) == (mb_frame_count-1))
            {
                GLubyte* out_buffer = new GLubyte[tw * th * 3];

                for(int y=0;y<th;++y)
                  for(int x=0;x<tw;++x)
                  {
                     unsigned long int c[3]={0,0,0};
                     for(int v=0;v<d;++v)
                        for(int u=0;u<d;++u)
                        {
                           for(int ci=0;ci<3;++ci)
                              c[ci]+=accumulated_mb_frames[((x*d+u)+(y*d+v)*g_window_width)*3+ci]/mb_frame_count;
                        }
                     for(int ci=0;ci<3;++ci)
                        out_buffer[(x+y*tw)*3+ci]=c[ci]/(d*d);
                  }

               char fname[256];

               snprintf(fname, sizeof(fname), "frames/%0.6d.bmp", capture_frame/mb_frame_count);

               ILuint image = 0;
               ilGenImages(1, &image);
               ilBindImage(image);
               ilTexImage(tw, th, 1, 3, IL_RGB, IL_UNSIGNED_BYTE, out_buffer);
               ilEnable(IL_FILE_OVERWRITE);
               ilSaveImage(fname);
               ilDeleteImages(1, &image);

               delete[] out_buffer;
               memset(accumulated_mb_frames,0,g_window_width*g_window_height*3*sizeof(unsigned long int));
            }

            ++capture_frame;
            local_capture_frame+=1;
            delete[] in_buffer;
         }
#endif

        SDL_GL_SwapBuffers();


        {
            char str[1024];
            sprintf(str,"%d",SDL_GetTicks()-start_time);
            SDL_WM_SetCaption(str, NULL);
        }
    }

   fclose(g_logfile);

    BASS_Free();

   SDL_Quit();

#if CAPTURE
   ilShutDown();
#endif

   return 0;
}

