; This module contains soundcard IRQ handlers, DMA routines, mixing routines
; as well as some time-critical GUS routines.
;
; Mixing routines are unrolled and optimized as much as my feeble soul can!
; The GUS routines use a weird DMA-caching scheme because the transfer has
; to be split into left and right channels, and into even more in the case of
; a buffer-wrap!

                .386
                .model flat

                include judascfg.inc
                include judasgus.inc

MONO            =       0
EIGHTBIT        =       0
STEREO          =       1
SIXTEENBIT      =       2

VM_OFF          =       0
VM_ON           =       1
VM_LOOP         =       2
VM_16BIT        =       4

DEV_NOSOUND     =       0
DEV_SB          =       1
DEV_SBPRO       =       2
DEV_SB16        =       3
DEV_GUS         =       4

CACHESLOTS      =       16

CACHESLOT       struc
GDC_Pos         dd      0
GDC_Length      dd      0
CACHESLOT       ends

DMACHANNEL      struc
DMA_PagePort    dw      0
DMA_AdrPort     dw      0
DMA_LenPort     dw      0
DMA_MaskPort    dw      0
DMA_ModePort    dw      0
DMA_ClearPort   dw      0
DMA_Mask        db      0
DMA_UnMask      db      0
DMA_Unused      dw      0
DMACHANNEL      ends

CHANNEL         struc
Chn_Pos         dd      0
Chn_Repeat      dd      0
Chn_End         dd      0
Chn_Sample      dd      0
Chn_Freq        dd      0
Chn_FractPos    dw      0
Chn_Vol         db      0
Chn_MasterVol   db      0
Chn_Panning     db      0
Chn_VoiceMode   db      0
CHANNEL         ends


getvolume       macro
                local   @limit
                mov     AL, [ESI.Chn_Vol]
                cmp     AL, 64
                jbe     @limit
                mov     AL, 64
@limit:         mul     [ESI.Chn_MasterVol]
                shr     AX, 6
                or      AL, AL
                jz      zerovolume
                movzx   EBX, AL
                endm

stereoadjust    macro
                local   @limit
                shl     EBX, 1                          ;Stereo can have 2x vol
                cmp     EBX, 255                        ;Check that volume is
                jbe     @limit                          ;within volumetable
                mov     EBX, 255                        ;limits though
@limit:         endm

smoothpanadjust macro
                local   @limit1, @limit2
                movzx   EAX, [ESI.Chn_Panning]
                mov     ECX, EBX
                imul    ECX, EAX
                shr     ECX, 7
                neg     EAX
                add     EAX, 256
                imul    EBX, EAX
                shr     EBX, 7
                cmp     EBX, 255
                jbe     @limit1
                mov     EBX, 255
@limit1:        cmp     ECX, 255
                jbe     @limit2
                mov     ECX, 255
@limit2:        endm

mix8_mono_n     macro   n
                mov     BL, [ESI]
                add     EDX, ECX
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 4 * n], EAX
                endm

mix8_left_n     macro   n
                mov     BL, [ESI]
                add     EDX, ECX
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                endm

mix8_middle_n   macro   n
                mov     BL, [ESI]
                add     EDX, ECX
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                add     [EDI + 8 * n + 4], EAX
                endm

mix8_right_n    macro   n
                mov     BL, [ESI]
                add     EDX, ECX
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mix8_smooth_n   macro   n
                mov     BL, [ESI]
                add     EDX, fractadd
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                mov     CL, BL
                add     [EDI + 8 * n], EAX
                mov     EAX, [ECX * 4]
                add     [EDI + 8 * n + 4], EAX
                endm

mix8_mono_i     macro   n
                movsx   EAX, byte ptr [ESI+1]
                movsx   ECX, byte ptr [ESI]
                sub     EAX, ECX
                movzx   ECX, DH
                imul    AX, CX
                mov     BL, AH
                add     BL, [ESI]
                add     DX, word ptr fractadd + 2
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 4 * n], EAX
                endm

mix8_left_i     macro   n
                movsx   EAX, byte ptr [ESI+1]
                movsx   ECX, byte ptr [ESI]
                sub     EAX, ECX
                movzx   ECX, DH
                imul    AX, CX
                mov     BL, AH
                add     BL, [ESI]
                add     DX, word ptr fractadd + 2
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                endm

mix8_middle_i   macro   n
                movsx   EAX, byte ptr [ESI+1]
                movsx   ECX, byte ptr [ESI]
                sub     EAX, ECX
                movzx   ECX, DH
                imul    AX, CX
                mov     BL, AH
                add     BL, [ESI]
                add     DX, word ptr fractadd + 2
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                add     [EDI + 8 * n + 4], EAX
                endm

mix8_right_i    macro   n
                movsx   EAX, byte ptr [ESI+1]
                movsx   ECX, byte ptr [ESI]
                sub     EAX, ECX
                movzx   ECX, DH
                imul    AX, CX
                mov     BL, AH
                add     BL, [ESI]
                add     DX, word ptr fractadd + 2
                mov     EAX, [EBX * 4]
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mix8_smooth_i   macro   n
                movsx   EAX, byte ptr [ESI+1]
                movsx   EBP, byte ptr [ESI]
                sub     EAX, EBP
                movzx   EBP, DH
                imul    AX, BP
                mov     BL, AH
                add     BL, [ESI]
                add     DX, word ptr fractadd + 2
                mov     EAX, [EBX * 4]
                adc     ESI, integeradd
                mov     CL, BL
                add     [EDI + 8 * n], EAX
                mov     EAX, [ECX * 4]
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_mono_n    macro   n
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, EBX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     EDX, ECX
                adc     ESI, EBP
                add     [EDI + 4 * n], EAX
                endm

mix16_left_n    macro   n
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, EBX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     EDX, ECX
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                endm

mix16_middle_n  macro   n
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, EBX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     EDX, ECX
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_right_n   macro   n
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, EBX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     EDX, ECX
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_smooth_n  macro   n
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, EBX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     [EDI + 8 * n], EAX
                movsx   EAX, word ptr [ESI * 2]
                imul    EAX, ECX
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     EDX, fractadd
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_mono_i    macro   n
                movsx   EAX, word ptr [ESI * 2 + 2]
                movsx   ECX, word ptr [ESI * 2]
                sub     EAX, ECX
                movzx   EBX, DH
                imul    EAX, EBX
                sar     EAX, 8
                add     EAX, ECX
                imul    EAX, leftvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     DX, word ptr fractadd + 2
                adc     ESI, EBP
                add     [EDI + 4 * n], EAX
                endm

mix16_left_i    macro   n
                movsx   EAX, word ptr [ESI * 2 + 2]
                movsx   ECX, word ptr [ESI * 2]
                sub     EAX, ECX
                movzx   EBX, DH
                imul    EAX, EBX
                sar     EAX, 8
                add     EAX, ECX
                imul    EAX, leftvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     DX, word ptr fractadd + 2
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                endm

mix16_middle_i  macro   n
                movsx   EAX, word ptr [ESI * 2 + 2]
                movsx   ECX, word ptr [ESI * 2]
                sub     EAX, ECX
                movzx   EBX, DH
                imul    EAX, EBX
                sar     EAX, 8
                add     EAX, ECX
                imul    EAX, leftvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     DX, word ptr fractadd + 2
                adc     ESI, EBP
                add     [EDI + 8 * n], EAX
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_right_i   macro   n
                movsx   EAX, word ptr [ESI * 2 + 2]
                movsx   ECX, word ptr [ESI * 2]
                sub     EAX, ECX
                movzx   EBX, DH
                imul    EAX, EBX
                sar     EAX, 8
                add     EAX, ECX
                imul    EAX, leftvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     DX, word ptr fractadd + 2
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mix16_smooth_i  macro   n
                movsx   EAX, word ptr [ESI * 2 + 2]
                movsx   ECX, word ptr [ESI * 2]
                sub     EAX, ECX
                movzx   EBX, DH
                imul    EAX, EBX
                sar     EAX, 8
                add     EAX, ECX
                mov     EBX, EAX
                imul    EAX, leftvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     [EDI + 8 * n], EAX
                mov     EAX, EBX
                imul    EAX, rightvol
                sar     EAX, 8 + (16 - SIGNIFICANT_BITS)
                add     DX, word ptr fractadd + 2
                adc     ESI, EBP
                add     [EDI + 8 * n + 4], EAX
                endm

mixloop8        macro   routine, samplesize, ip
                local   @hitend, @subloop, @oneshot, @offsettable,
                local   @offset0, @offset1, @offset2, @offset3
                local   @offset4, @offset5, @offset6, @offset7
                local   @offset8, @offset9, @offseta, @offsetb
                local   @offsetc, @offsetd, @offsete, @offsetf
                mov     DX, [ESI.Chn_FractPos]          ;Get fractional pos
                if ip eq 0
                shl     EDX, 16
                endif
                mov     EAX, [ESI.Chn_End]              ;Get end & endsubtract-
                mov     smpend, EAX                     ;value
                sub     EAX, [ESI.Chn_Repeat]
                mov     smpsubtract, EAX
                mov     ESI, [ESI.Chn_Pos]              ;Get sample position
                mov     EDI, dptr                       ;Get bufferptr
                mov     EAX, samples                    ;Fix loopcount &
                dec     EAX
                shr     EAX, 4                          ;jumpoffset & subtract
                inc     EAX                             ;EDI accordingly
                mov     loopcount, EAX
                mov     EAX, samples
                and     EAX, 15
                mov     EAX, shittable[EAX * 4]
                if samplesize eq 4
                sub     EDI, EAX
                else
                sub     EDI, EAX
                sub     EDI, EAX
                endif
                add     EAX, offset @offsettable
                jmp     [EAX]
@offset0:       routine 0
@offset1:       routine 1
@offset2:       routine 2
@offset3:       routine 3
@offset4:       routine 4
@offset5:       routine 5
@offset6:       routine 6
@offset7:       routine 7
@offset8:       routine 8
@offset9:       routine 9
@offseta:       routine 10
@offsetb:       routine 11
@offsetc:       routine 12
@offsetd:       routine 13
@offsete:       routine 14
@offsetf:       routine 15
                add     EDI, 16 * samplesize
                cmp     ESI, smpend
                jae     @hitend
                dec     loopcount
                jnz     @offset0
                mov     EAX, cptr
                mov     [EAX.Chn_Pos], ESI
                if ip eq 0
                shr     EDX, 16
                endif
                mov     [EAX.Chn_FractPos], DX
                ret
@hitend:        mov     EAX, cptr
                test    [EAX.Chn_VoiceMode], VM_LOOP
                jz      @oneshot
@subloop:       sub     ESI, smpsubtract
                cmp     ESI, smpend
                jae     @subloop
                dec     loopcount
                jnz     @offset0
                mov     [EAX.Chn_Pos], ESI
                if ip eq 0
                shr     EDX, 16
                endif
                mov     [EAX.Chn_FractPos], DX
                ret
@oneshot:       mov     [EAX.Chn_VoiceMode], VM_OFF
                ret
                align   4
@offsettable:   dd      offset @offset0
                dd      offset @offset1
                dd      offset @offset2
                dd      offset @offset3
                dd      offset @offset4
                dd      offset @offset5
                dd      offset @offset6
                dd      offset @offset7
                dd      offset @offset8
                dd      offset @offset9
                dd      offset @offseta
                dd      offset @offsetb
                dd      offset @offsetc
                dd      offset @offsetd
                dd      offset @offsete
                dd      offset @offsetf
                endm

mixloop16       macro   routine, samplesize, ip
                local   @hitend, @subloop, @oneshot, @offsettable,
                local   @offset0, @offset1, @offset2, @offset3
                local   @offset4, @offset5, @offset6, @offset7
                local   @offset8, @offset9, @offseta, @offsetb
                local   @offsetc, @offsetd, @offsete, @offsetf
                mov     DX, [ESI.Chn_FractPos]          ;Get fractional pos
                if ip eq 0
                shl     EDX, 16
                endif
                mov     EAX, [ESI.Chn_End]              ;Get end & endsubtract-
                shr     EAX, 1
                mov     smpend, EAX                     ;value
                mov     EAX, [ESI.Chn_End]
                sub     EAX, [ESI.Chn_Repeat]
                shr     EAX, 1
                mov     smpsubtract, EAX
                mov     ESI, [ESI.Chn_Pos]              ;Get sample position
                shr     ESI, 1
                mov     EDI, dptr                       ;Get bufferptr
                mov     EAX, samples                    ;Fix loopcount &
                dec     EAX
                shr     EAX, 4                          ;jumpoffset & subtract
                inc     EAX                             ;EDI accordingly
                mov     loopcount, EAX
                mov     EAX, samples
                and     EAX, 15
                mov     EAX, shittable[EAX * 4]
                if samplesize eq 4
                sub     EDI, EAX
                else
                sub     EDI, EAX
                sub     EDI, EAX
                endif
                add     EAX, offset @offsettable
                jmp     [EAX]
@offset0:       routine 0
@offset1:       routine 1
@offset2:       routine 2
@offset3:       routine 3
@offset4:       routine 4
@offset5:       routine 5
@offset6:       routine 6
@offset7:       routine 7
@offset8:       routine 8
@offset9:       routine 9
@offseta:       routine 10
@offsetb:       routine 11
@offsetc:       routine 12
@offsetd:       routine 13
@offsete:       routine 14
@offsetf:       routine 15
                add     EDI, 16 * samplesize
                cmp     ESI, smpend
                jae     @hitend
                dec     loopcount
                jnz     @offset0
                mov     EAX, cptr
                shl     ESI, 1
                mov     [EAX.Chn_Pos], ESI
                if ip eq 0
                shr     EDX, 16
                endif
                mov     [EAX.Chn_FractPos], DX
                ret
@hitend:        mov     EAX, cptr
                test    [EAX.Chn_VoiceMode], VM_LOOP
                jz      @oneshot
@subloop:       sub     ESI, smpsubtract
                cmp     ESI, smpend
                jae     @subloop
                dec     loopcount
                jnz     @offset0
                shl     ESI, 1
                mov     [EAX.Chn_Pos], ESI
                if ip eq 0
                shr     EDX, 16
                endif
                mov     [EAX.Chn_FractPos], DX
                ret
@oneshot:       mov     [EAX.Chn_VoiceMode], VM_OFF
                ret
                align   4
@offsettable:   dd      offset @offset0
                dd      offset @offset1
                dd      offset @offset2
                dd      offset @offset3
                dd      offset @offset4
                dd      offset @offset5
                dd      offset @offset6
                dd      offset @offset7
                dd      offset @offset8
                dd      offset @offset9
                dd      offset @offseta
                dd      offset @offsetb
                dd      offset @offsetc
                dd      offset @offsetd
                dd      offset @offsete
                dd      offset @offsetf
                endm

                .code

                public  judas_code_lock_start_
                public  judas_code_lock_end_
                public  judas_update_
                public  judas_get_ds_
                public  sb_handler_
                public  sb_aihandler_
                public  sb16_handler_
                public  gus_handler_
                public  gus_peek_
                public  gus_poke_
                public  gus_handler_
                public  gus_dmawait_
                public  gus_dmainit_
                public  gus_dmaprogram_
                public  gus_startchannels_
                public  mix_
                public  normalmix_
                public  ipmix_
                public  dma_program_

                extrn   _judas_ds:word
                extrn   _judas_initialized:byte
                extrn   _judas_mixmode:byte
                extrn   _judas_samplesize:byte
                extrn   _judas_clipbuffer:dword
                extrn   _judas_cliptable:dword
                extrn   _judas_volumetable:dword
                extrn   _judas_mixrate:dword
                extrn   _judas_channel:dword
                extrn   _judas_mixroutine:dword
                extrn   _judas_device:dword
                extrn   _judas_port:dword
                extrn   _judas_irq:dword
                extrn   _judas_dma:dword
                extrn   _judas_irqcount:dword
                extrn   _judas_bufferlength:dword
                extrn   _judas_buffermask:dword
                extrn   _judas_bpmcount:dword
                extrn   _judas_bpmtempo:byte
                extrn   _judas_player:dword
                extrn   _judas_mixpos:dword
                extrn   _dma_address:dword

judas_get_ds_:  mov     AX, DS
                mov     _judas_ds, AX
                ret

judas_code_lock_start_:

                align   4

DMAChannels     DMACHANNEL <87h, 0h, 1h, 0ah, 0bh, 0ch, 4h, 0h>
                DMACHANNEL <83h, 2h, 3h, 0ah, 0bh, 0ch, 5h, 1h>
                DMACHANNEL <81h, 4h, 5h, 0ah, 0bh, 0ch, 6h, 2h>
                DMACHANNEL <82h, 6h, 7h, 0ah, 0bh, 0ch, 7h, 3h>
                DMACHANNEL <8fh, 0c0h, 0c2h, 0d4h, 0d6h, 0d8h, 4h, 0h>
                DMACHANNEL <8bh, 0c4h, 0c6h, 0d4h, 0d6h, 0d8h, 5h, 1h>
                DMACHANNEL <89h, 0c8h, 0cah, 0d4h, 0d6h, 0d8h, 6h, 2h>
                DMACHANNEL <8ah, 0cch, 0ceh, 0d4h, 0d6h, 0d8h, 7h, 3h>

                align   4

gdc             CACHESLOT CACHESLOTS dup (<>)

                align   4

shittable       dd      0, 60, 56, 52, 48, 44, 40, 36
                dd      32, 28, 24, 20, 16, 12, 8, 4

loopcount       dd      0
fractadd        dd      0
integeradd      dd      0
smpend          dd      0
smpsubtract     dd      0
samples         dd      0
totalwork       dd      0
postproc        dd      0
cptr            dd      0
dptr            dd      0
fptr            dd      0
leftvol         dd      0
rightvol        dd      0
gus_dmainprogress db    0
mix_exec        db      0

                align   4

        ;DMA functions. DMA polling is really fucked up: if reading the
        ;position too often (> 100 Hz) one may get bogus values. This is
        ;compensated by reading two values, and if their offset is too big or
        ;they're outside the buffer, the position is read again.
        ;
        ;Actually GUS fucks up just in the same way when reading the channel
        ;position. Shit, what is wrong with the hardware?!
        ;
        ;Previously I though that EMM386 causes these fuckups, but no, it
        ;wasn't so. However, under DPMI there's no fuckups!
        ;
        ;It would be really nice & simple to just update one bufferhalf at a
        ;time in the soundcard interrupt, but I think it's important to give
        ;the user full control of the sound updating, even at the expense of
        ;PAIN!!!

dma_program_:   push    ESI
                push    EDI
                push    ECX
                mov     ECX, EAX                        ;ECX = mode
                mov     EDI, EDX                        ;EDI = offset
                mov     ESI, _judas_dma                 ;Get channel num
                cmp     ESI, 4
                jae     dma16_program
                shl     ESI, 4                          ;16 = dma struc len
                add     ESI, offset DMAChannels         ;Ptr now ready
                mov     DX, [ESI.DMA_MaskPort]
                mov     AL, [ESI.DMA_Mask]
                out     DX, AL                          ;Mask the DMA channel
                xor     AL, AL
                mov     DX, [ESI.DMA_ClearPort]
                out     DX, AL                          ;Clear byte ptr.
                mov     DX, [ESI.DMA_ModePort]
                mov     AL, CL                          ;Get mode
                or      AL, [ESI.DMA_UnMask]            ;Or with channel num
                out     DX, AL                          ;Set DMA mode
                mov     DX, [ESI.DMA_LenPort]
                dec     EBX                             ;EBX = length
                mov     AL, BL
                out     DX, AL                          ;Set length low and
                mov     AL, BH                          ;high bytes
                out     DX, AL
                mov     DX, [ESI.DMA_AdrPort]
                mov     EBX, _dma_address               ;Get DMA buffer address
                add     EBX, EDI                        ;Add offset
                mov     AL, BL
                out     DX, AL                          ;Set offset
                mov     AL, BH
                out     DX, AL
                mov     DX, [ESI.DMA_PagePort]
                shr     EBX, 16
                mov     AL, BL
                out     DX, AL                          ;Set page
                mov     DX, [ESI.DMA_MaskPort]
                mov     AL, [ESI.DMA_UnMask]
                out     DX, AL                          ;Unmask the DMA channel
                pop     ECX
                pop     EDI
                pop     ESI
                ret
dma16_program:  shl     ESI, 4                          ;16 = dma struc len
                add     ESI, offset DMAChannels         ;Ptr now ready
                mov     DX, [ESI.DMA_MaskPort]
                mov     AL, [ESI.DMA_Mask]
                out     DX, AL                          ;Mask the DMA channel
                xor     AL, AL
                mov     DX, [ESI.DMA_ClearPort]
                out     DX, AL                          ;Clear byte ptr.
                mov     DX, [ESI.DMA_ModePort]
                mov     AL, CL                          ;Get mode
                or      AL, [ESI.DMA_UnMask]            ;Or with channel num
                out     DX, AL                          ;Set DMA mode
                mov     DX, [ESI.DMA_LenPort]
                shr     EBX, 1
                dec     EBX
                mov     AL, BL
                out     DX, AL                          ;Set length low and
                mov     AL, BH                          ;high bytes
                out     DX, AL
                mov     DX, [ESI.DMA_AdrPort]
                mov     EBX, _dma_address               ;Get DMA buffer address
                add     EBX, EDI                        ;Add offset
                shr     EBX, 1                          ;Because of 16-bitness
                mov     AL, BL
                out     DX, AL                          ;Set offset
                mov     AL, BH
                out     DX, AL
                mov     DX, [ESI.DMA_PagePort]
                shr     EBX, 15
                mov     AL, BL
                out     DX, AL                          ;Set page
                mov     DX, [ESI.DMA_MaskPort]
                mov     AL, [ESI.DMA_UnMask]
                out     DX, AL                          ;Unmask the DMA channel
                pop     ECX
                pop     EDI
                pop     ESI
                ret

dma_query_:     cli
                push    EBX
                push    ECX
                push    EDX
                push    ESI
                mov     ESI, _judas_dma
                cmp     ESI, 4
                jae     dma16_query
                shl     ESI, 4                          ;16 = dma struc len
                add     ESI, offset DMAChannels         ;Ptr now ready
                xor     EAX, EAX
                mov     DX, [ESI.DMA_ClearPort]         ;Clear flip-flop
                out     DX, AL
                mov     DX, [ESI.DMA_AdrPort]
dqloop1:        xor     EAX, EAX
                in      AL, DX
                xchg    AL, AH
                in      AL, DX
                xchg    AL, AH
                sub     AX, word ptr _dma_address       ;Subtract page offset
                mov     EBX, EAX                        ;EBX = position 1
                in      AL, DX
                xchg    AL, AH
                in      AL, DX
                xchg    AL, AH
                sub     AX, word ptr _dma_address       ;Subtract page offset
                mov     ECX, EAX                        ;ECX = position 2
                cmp     EBX, _judas_bufferlength        ;Outside buffer?
                jae     dqloop1
                mov     EAX, EBX
                sub     EAX, ECX
                cmp     EAX, 64
                jg      dqloop1
                cmp     EAX, -64
                jl      dqloop1
                mov     EAX, EBX
                pop     ESI
                pop     EDX
                pop     ECX
                pop     EBX
                sti
                ret
dma16_query:    shl     ESI, 4                          ;16 = dma struc len
                add     ESI, offset DMAChannels         ;Ptr now ready
                mov     DX, [ESI.DMA_ClearPort]         ;Clear flip-flop
                xor     EAX, EAX
                out     DX, AL
                mov     DX, [ESI.DMA_AdrPort]
                mov     ESI, _dma_address
                and     ESI, 1ffffh
dqloop2:        xor     EAX, EAX
                in      AL, DX
                xchg    AL, AH
                in      AL, DX
                xchg    AL, AH
                shl     EAX, 1
                sub     EAX, ESI                        ;Subtract page offset
                mov     EBX, EAX                        ;EBX = position 1
                xor     EAX, EAX
                in      AL, DX
                xchg    AL, AH
                in      AL, DX
                xchg    AL, AH
                shl     EAX, 1
                sub     EAX, ESI                        ;Subtract page offset
                mov     ECX, EAX                        ;ECX = position 2
                cmp     EBX, _judas_bufferlength        ;Outside buffer?
                jae     dqloop2
                mov     EAX, EBX
                sub     EAX, ECX
                cmp     EAX, 64
                jg      dqloop2
                cmp     EAX, -64
                jl      dqloop2
                mov     EAX, EBX
                pop     ESI
                pop     EDX
                pop     ECX
                pop     EBX
                sti
                ret

        ;Generic send-EOI routine.

send_eoi:       inc     _judas_irqcount
                cmp     _judas_irq, 8
                jae     highirq
                mov     AL, 20h
                out     20h, AL
                ret
highirq:        mov     AL, 20h
                out     0a0h, AL
                mov     AL, 00001011b
                out     0a0h, AL
                in      AL, 0a0h
                or      AL, AL
                jnz     sb_noeoi
                mov     AL, 20h
                out     20h, AL
sb_noeoi:       ret

        ;Soundblaster IRQ handlers, one for singlecycle, one for 8bit autoinit
        ;and one for 16bit autoinit.

sb_handler_:    pushad
                push    DS
                mov     AX, CS:_judas_ds
                mov     DS, AX
                mov     EDX, _judas_port
                add     EDX, 0eh
                in      AL, DX
                sub     EDX, 2h
sb_wait1:       in      AL, DX
                or      AL, AL
                js      sb_wait1
                mov     AL, 14h
                out     DX, AL
sb_wait2:       in      AL, DX
                or      AL, AL
                js      sb_wait2
                mov     AX, 0fff0h
                out     DX, AL
sb_wait3:       in      AL, DX
                or      AL, AL
                js      sb_wait3
                mov     AL, AH
                out     DX, AL
                sti
                call    send_eoi
                pop     DS
                popad
                iretd

sb_aihandler_:  pushad
                push    DS
                mov     AX, CS:_judas_ds
                mov     DS, AX
                mov     EDX, _judas_port
                add     EDX, 0eh
                in      AL, DX
                sti
                call    send_eoi
                pop     DS
                popad
                iretd

sb16_handler_:  pushad
                push    DS
                mov     AX, CS:_judas_ds
                mov     DS, AX
                mov     EDX, _judas_port
                add     EDX, 0fh
                in      AL, DX
                sti
                call    send_eoi
                pop     DS
                popad
                iretd

        ;GUS IRQ handler

gus_handler_:   pushad
                push    DS
                mov     AX, CS:_judas_ds
                mov     DS, AX
gus_irqloop:    mov     EDX, _judas_port
                add     EDX, GF1_IRQ_STAT
                in      AL, DX
                test    AL, DMA_TC_IRQ
                jz      gus_irqdone
                mov     EDX, _judas_port                ;Acknowledge the DMA
                add     EDX, GF1_REG_SELECT             ;interrupt
                mov     AL, DMA_CONTROL
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_HI
                in      AL, DX
                dec     gus_dmainprogress
                mov     ESI, offset gdc
                mov     ECX, CACHESLOTS
gusirq_seekslot:cmp     [ESI.GDC_Length], 0
                jnz     gusirq_slotfound
                add     ESI, type CACHESLOT
                dec     ECX
                jnz     gusirq_seekslot
                jmp     gus_irqloop
gusirq_slotfound:
                mov     EBX, [ESI.GDC_Pos]              ;DMA offset
                shr     EBX, 4
                mov     CL, DMA_ENABLE or DMA_R0 or DMA_TWOS_COMP or DMA_IRQ_ENABLE
                test    _judas_mixmode, SIXTEENBIT
                jz      gus_dma_eight2
                mov     CL, DMA_ENABLE or DMA_R0 or DMA_DATA_16 or DMA_IRQ_ENABLE
gus_dma_eight2: cmp     _judas_dma, 4
                jb      gus_nohighdma2
                or      CL, DMA_WIDTH_16
                shr     EBX, 1
gus_nohighdma2: mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, SET_DMA_ADDRESS
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                mov     AX, BX
                out     DX, AX
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, DMA_CONTROL
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_HI
                mov     AL, CL
                out     DX, AL
                mov     EBX, [ESI.GDC_Length]
                mov     [ESI.GDC_Length], 0
                mov     EDX, [ESI.GDC_Pos]              ;DMA offset
                mov     EAX, 48h                        ;DMA mode
                call    dma_program_                    ;Program it!
                jmp     gus_irqloop
gus_irqdone:    sti
                call    send_eoi
                pop     DS
                popad
                iretd

        ;Various GUS functions

gus_peek_:      push    EBX
                mov     EBX, EAX
                mov     AL, SET_DRAM_LOW
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                out     DX, AL
                mov     AX, BX
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                out     DX, AX
                mov     AL, SET_DRAM_HIGH
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                out     DX, AL
                shr     EBX, 16
                mov     AL, BL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_HI
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DRAM
                in      AL, DX
                pop     EBX
                ret

gus_poke_:      push    EBX
                push    EDX
                mov     EBX, EAX
                mov     AL, SET_DRAM_LOW
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                out     DX, AL
                mov     AX, BX
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                out     DX, AX
                mov     AL, SET_DRAM_HIGH
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                out     DX, AL
                shr     EBX, 16
                mov     AL, BL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_HI
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DRAM
                pop     EAX
                out     DX, AL
                pop     EBX
                ret

gus_startchannels_:
                push    EBX                             ;This routine starts
                push    ECX                             ;the two channels
                push    EDX                             ;as quickly as possible.
                mov     EBX, _judas_port
                add     EBX, GF1_PAGE
                mov     ECX, _judas_port
                add     ECX, GF1_DATA_HI
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, SET_CONTROL
                out     DX, AL
                test    _judas_mixmode, SIXTEENBIT
                jz      gus_start8
                mov     EDX, EBX
                mov     AL, 0
                out     DX, AL
                mov     EDX, ECX
                mov     AL, VC_LOOP_ENABLE or VC_DATA_TYPE
                out     DX, AL
                mov     EDX, EBX
                mov     AL, 1
                out     DX, AL
                mov     EDX, ECX
                mov     AL, VC_LOOP_ENABLE or VC_DATA_TYPE
                out     DX, AL
                pop     EDX
                pop     ECX
                pop     EBX
                ret
gus_start8:     mov     EDX, EBX
                xor     AL, AL
                out     DX, AL
                mov     EDX, ECX
                mov     AL, VC_LOOP_ENABLE
                out     DX, AL
                mov     EDX, EBX
                mov     AL, 1
                out     DX, AL
                mov     EDX, ECX
                mov     AL, VC_LOOP_ENABLE
                out     DX, AL
                pop     EDX
                pop     ECX
                pop     EBX
                ret

gus_dmaprogram_:
                or      EDX, EDX                        ;Zero length fucks up!
                jz      gus_skipdma
                pushad
                cli
                cmp     gus_dmainprogress, 0            ;Do we have to cache the
                je      gus_dontcache                   ;block?
                mov     EBX, offset gdc
                mov     ECX, CACHESLOTS
gus_seekslot:   cmp     [EBX.GDC_Length], 0
                je      gus_slotfound
                add     EBX, type CACHESLOT
                dec     ECX
                jnz     gus_seekslot
                sti
                popad
gus_skipdma:    ret
gus_slotfound:  mov     [EBX.GDC_Pos], EAX
                mov     [EBX.GDC_Length], EDX
                inc     gus_dmainprogress
                sti
                popad
                ret
gus_dontcache:  sti
                inc     gus_dmainprogress
                mov     ESI, EAX
                mov     EDI, EDX
                mov     EBX, ESI                        ;DMA offset
                shr     EBX, 4
                mov     CL, DMA_ENABLE or DMA_R0 or DMA_TWOS_COMP or DMA_IRQ_ENABLE
                test    _judas_mixmode, SIXTEENBIT
                jz      gus_dma_eight
                mov     CL, DMA_ENABLE or DMA_R0 or DMA_DATA_16 or DMA_IRQ_ENABLE
gus_dma_eight:  cmp     _judas_dma, 4
                jb      gus_nohighdma
                or      CL, DMA_WIDTH_16
                shr     EBX, 1
gus_nohighdma:  mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, SET_DMA_ADDRESS
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                mov     EAX, EBX
                out     DX, AX
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, DMA_CONTROL
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_HI
                mov     AL, CL
                out     DX, AL
                mov     EBX, EDI                        ;DMA length
                mov     EDX, ESI                        ;DMA offset
                mov     EAX, 48h                        ;DMA mode
                call    dma_program_                    ;Program it!
                popad
                ret

gus_dmainit_:   cli
                mov     gus_dmainprogress, 0
                push    EAX
                mov     EAX, offset gdc
diloop:         mov     [EAX.GDC_Pos], 0
                mov     [EAX.GDC_Length], 0
                add     EAX, type CACHESLOT
                cmp     EAX, offset gdc + CACHESLOTS * type CACHESLOT
                jne     diloop
                pop     EAX
                sti
                ret

gus_dmawait_:   mov     EAX, 200000h                    ;Timeout counter
gus_dmawaitloop:cmp     gus_dmainprogress, 0            ;(might time out if
                je      gus_dmadone                     ;there is a DMA
                dec     EAX                             ;conflict.) This routine
                jnz     gus_dmawaitloop                 ;is used just for click
gus_dmadone:    ret                                     ;removal!

gus_getpos:     push    EBX
                push    EDX
                mov     EDX, _judas_port                ;Get the channel
                add     EDX, GF1_PAGE                   ;playing position to
                xor     AL, AL                          ;know where we'll mix
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, GET_ACC_HIGH
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                in      AX, DX
                and     EAX, 8191
                shl     EAX, 7
                mov     EBX, EAX
                mov     EDX, _judas_port
                add     EDX, GF1_REG_SELECT
                mov     AL, GET_ACC_LOW
                out     DX, AL
                mov     EDX, _judas_port
                add     EDX, GF1_DATA_LOW
                in      AX, DX
                shr     AX, 9
                or      EAX, EBX
                test    _judas_mixmode, SIXTEENBIT
                jz      ggp_not16
                shl     EAX, 1
ggp_not16:      pop     EDX
                pop     EBX
                ret

        ;General DMAbuffer update routine (call either from main program or
        ;within your timer interrupt)

judas_update_:  cmp     _judas_device, DEV_NOSOUND
                je      judas_gotohell
                cmp     mix_exec, 0
                jne     judas_gotohell
                cmp     _judas_initialized, 0
                je      judas_gotohell
                inc     mix_exec
                pushad
                cmp     _judas_device, DEV_GUS
                je      updategus                       ;This is a different story
                call    dma_query_
                                                        ;Must be aligned on 8
                and     EAX, _judas_buffermask          ;samples (unrolling!)
                mov     EBX, _judas_mixpos              ;This is the old pos
                cmp     EAX, EBX
                je      judas_donothing
                jb      judas_wrap
judas_normal:   mov     _judas_mixpos, EAX
                mov     EDX, EAX
                sub     EDX, EBX                        ;EDX = length to mix
                mov     EAX, EBX                        ;EAX = pos. to mix
                add     EAX, _dma_address
                call    mix_
judas_donothing:popad
                dec     mix_exec
judas_gotohell: ret
judas_wrap:     mov     _judas_mixpos, EAX
                mov     EAX, EBX                        ;Mix to buffer end
                mov     EDX, _judas_bufferlength
                sub     EDX, EBX
                add     EAX, _dma_address
                call    mix_
                mov     EAX, _dma_address               ;Then to start
                mov     EDX, _judas_mixpos
                or      EDX, EDX
                jz      judas_donothing
                call    mix_
                jmp     judas_donothing

updategus:      cmp     gus_dmainprogress, CACHESLOTS - 4 ;Is there too many
                ja      judas_donothing                 ;DMA's going on?
                cli                                     ;(our DMA might not
                test    _judas_mixmode, STEREO          ;fit in anymore)
                jz      updategus_mono
ipc_s:          xor     EAX, EAX
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                shr     EAX, 1
                call    gus_poke_
                mov     EAX, 1
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                shr     EAX, 1
                inc     EAX
                call    gus_poke_
                mov     EAX, _judas_bufferlength
                shr     EAX, 1
                add     EAX, 32
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                add     EAX, 32
                call    gus_poke_
                mov     EAX, _judas_bufferlength
                shr     EAX, 1
                add     EAX, 33
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                add     EAX, 33
                call    gus_poke_
                mov     EDX, _judas_bufferlength
                shr     EDX, 1
ugs_shitloop:   call    gus_getpos
                mov     EBX, EAX
                call    gus_getpos
                mov     ECX, EAX
                cmp     EBX, EDX
                jae     ugs_shitloop
                mov     EAX, EBX
                sub     EAX, ECX
                cmp     EAX, 64
                jg      ugs_shitloop
                cmp     EAX, -64
                jl      ugs_shitloop
                mov     EAX, EBX
                sti
                and     EAX, _judas_buffermask
                mov     EBX, _judas_mixpos              ;EBX = old mixpos
                cmp     EAX, EBX
                je      judas_donothing
                jb      updategus_wrap
                mov     _judas_mixpos, EAX              ;New "oldpos"
                mov     EDX, EAX
                sub     EDX, EBX                        ;How much to mix
                mov     EAX, EBX                        ;Where to mix
                push    EAX
                push    EDX
                shl     EDX, 1
                add     EAX, _dma_address
                call    mix_
                pop     EDX
                pop     EAX
                call    gus_dmaprogram_
                add     EAX, 32
                mov     EBX, _judas_bufferlength
                shr     EBX, 1
                add     EAX, EBX
                call    gus_dmaprogram_
                jmp     judas_donothing
updategus_wrap: mov     _judas_mixpos, EAX              ;Mix first to buffer
                mov     EAX, EBX                        ;end, then to start
                mov     EDX, _judas_bufferlength
                shr     EDX, 1
                sub     EDX, EBX
                push    EAX
                push    EDX
                shl     EDX, 1
                add     EAX, _dma_address
                call    mix_
                mov     EAX, _dma_address
                mov     EDX, _judas_mixpos
                shl     EDX, 1
                call    mix_
                pop     EDX
                pop     EAX
                call    gus_dmaprogram_
                add     EAX, 32
                mov     EBX, _judas_bufferlength
                shr     EBX, 1
                add     EAX, EBX
                call    gus_dmaprogram_
                xor     EAX, EAX
                mov     EDX, _judas_mixpos
                call    gus_dmaprogram_
                add     EAX, 32
                mov     EBX, _judas_bufferlength
                shr     EBX, 1
                add     EAX, EBX
                call    gus_dmaprogram_
                jmp     judas_donothing

updategus_mono: xor     EAX, EAX
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                call    gus_poke_
                mov     EAX, 1
                call    gus_peek_
                mov     DL, AL
                mov     EAX, _judas_bufferlength
                inc     EAX
                call    gus_poke_
                mov     EDX, _judas_bufferlength
ugm_shitloop:   call    gus_getpos
                mov     EBX, EAX
                call    gus_getpos
                mov     ECX, EAX
                cmp     EBX, EDX
                jae     ugm_shitloop
                mov     EAX, EBX
                sub     EAX, ECX
                cmp     EAX, 64
                jg      ugm_shitloop
                cmp     EAX, -64
                jl      ugm_shitloop
                mov     EAX, EBX
                sti
                and     EAX, _judas_buffermask
                mov     EBX, _judas_mixpos              ;EBX = old mixpos
                cmp     EAX, EBX
                je      judas_donothing
                jb      updategusm_wrap
                mov     _judas_mixpos, EAX              ;New "oldpos"
                mov     EDX, EAX
                sub     EDX, EBX                        ;How much to mix
                mov     EAX, EBX                        ;Where to mix
                push    EAX
                push    EDX
                add     EAX, _dma_address
                call    mix_
                pop     EDX
                pop     EAX
                call    gus_dmaprogram_
                jmp     judas_donothing
updategusm_wrap:mov     _judas_mixpos, EAX              ;Mix first to buffer
                mov     EAX, EBX                        ;end
                mov     EDX, _judas_bufferlength
                sub     EDX, EBX
                push    EAX
                push    EDX
                add     EAX, _dma_address
                call    mix_
                mov     EAX, _dma_address
                mov     EDX, _judas_mixpos
                call    mix_
                pop     EDX
                pop     EAX
                call    gus_dmaprogram_
                xor     EAX, EAX
                mov     EDX, _judas_mixpos
                call    gus_dmaprogram_
                jmp     judas_donothing

        ;Mixing routines start here!
        ;This is the main mixing routine, which mixes EDX bytes of sound into
        ;address EAX, calling the music player at correct intervals. EDX must
        ;be a multiply of (samplesize * 8) because there is an unrolled
        ;postprocessing loop.
        ;WARNING: This routine destroys every register!

mix_:           or      EDX, EDX                        ;Check zero length
                jz      mix_quit
                mov     ECX, EDX
                test    _judas_mixmode, STEREO          ;Stereo or mono?
                jz      mix_noshift1
                shr     EDX, 1
mix_noshift1:   test    _judas_mixmode, SIXTEENBIT      ;8- or 16bit?
                jz      mix_noshift2
                shr     EDX, 1
                shr     ECX, 1
mix_noshift2:   mov     samples, EDX                    ;Save number of samples
                mov     totalwork, EDX                  ;"Total work" counter
                mov     fptr, EAX                       ;Save final destination
                shr     ECX, 3
                mov     postproc, ECX                   ;Save clipbuffer size
                mov     EDI, _judas_clipbuffer          ;Clear the clipbuffer
                mov     dptr, EDI
                xor     EAX, EAX
mix_clearloop:  mov     [EDI], EAX
                mov     [EDI + 4], EAX
                mov     [EDI + 8], EAX
                mov     [EDI + 12], EAX
                mov     [EDI + 16], EAX
                mov     [EDI + 20], EAX
                mov     [EDI + 24], EAX
                mov     [EDI + 28], EAX
                add     EDI, 32
                dec     ECX
                jnz     mix_clearloop
                cmp     _judas_player, 0
                jne     mix_hardwayloop
                call    _judas_mixroutine
                jmp     mix_firstphasedone
mix_hardwayloop:cmp     _judas_bpmcount, 0              ;Time to play?
                jne     mix_skipplaying
                cmp     _judas_player, 0                ;Might change in the
                je      mix_fuckshitup                  ;middle of a loop
                call    [_judas_player]
mix_fuckshitup: mov     EAX, _judas_mixrate
                mov     EBX, 5
                mul     EBX
                shr     EAX, 1
                xor     EDX, EDX
                movzx   EBX, _judas_bpmtempo
                div     EBX
                mov     _judas_bpmcount, EAX
mix_skipplaying:mov     EAX, totalwork
                cmp     EAX, _judas_bpmcount
                jbe     mix_nolimit
                mov     EAX, _judas_bpmcount
mix_nolimit:    mov     samples, EAX
                call    _judas_mixroutine
                mov     EAX, samples
                sub     _judas_bpmcount, EAX
                mov     EBX, EAX
                shl     EBX, 2
                test    _judas_mixmode, STEREO
                jz      mix_noshift3
                shl     EBX, 1
mix_noshift3:   add     dptr, EBX
                sub     totalwork, EAX
                jnz     mix_hardwayloop
mix_firstphasedone:
                test    _judas_mixmode, SIXTEENBIT
                jz      mix_8bit_endphase
mix_16bit_endphase:
                test    _judas_mixmode, STEREO
                jz      mix_nogusshit1
                cmp     _judas_device, DEV_GUS
                je      mix_gus16_endphase
mix_nogusshit1: mov     EDI, fptr
                mov     EBX, postproc
                mov     ESI, _judas_clipbuffer
                mov     ECX, _judas_cliptable
                xor     EAX, EAX
mix_16bit_endphase_loop:
                mov     AX, [ESI]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI], AX
                mov     AX, [ESI + 4]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 2], AX
                mov     AX, [ESI + 8]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 4], AX
                mov     AX, [ESI + 12]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 6], AX
                mov     AX, [ESI + 16]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 8], AX
                mov     AX, [ESI + 20]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 10], AX
                mov     AX, [ESI + 24]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 12], AX
                mov     AX, [ESI + 28]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 14], AX
                add     ESI, 32
                add     EDI, 16
                dec     EBX
                jnz     mix_16bit_endphase_loop
mix_quit:       ret
mix_8bit_endphase:
                test    _judas_mixmode, STEREO
                jz      mix_nogusshit2
                cmp     _judas_device, DEV_GUS
                je      mix_gus8_endphase
mix_nogusshit2: mov     EDI, fptr
                mov     EBX, postproc
                mov     ESI, _judas_clipbuffer
                mov     ECX, _judas_cliptable
                xor     EAX, EAX
mix_8bit_endphase_loop:
                mov     AX, [ESI]
                mov     AL, [ECX + EAX]
                mov     [EDI], AL
                mov     AX, [ESI + 4]
                mov     AL, [ECX + EAX]
                mov     [EDI + 1], AL
                mov     AX, [ESI + 8]
                mov     AL, [ECX + EAX]
                mov     [EDI + 2], AL
                mov     AX, [ESI + 12]
                mov     AL, [ECX + EAX]
                mov     [EDI + 3], AL
                mov     AX, [ESI + 16]
                mov     AL, [ECX + EAX]
                mov     [EDI + 4], AL
                mov     AX, [ESI + 20]
                mov     AL, [ECX + EAX]
                mov     [EDI + 5], AL
                mov     AX, [ESI + 24]
                mov     AL, [ECX + EAX]
                mov     [EDI + 6], AL
                mov     AX, [ESI + 28]
                mov     AL, [ECX + EAX]
                mov     [EDI + 7], AL
                add     ESI, 32
                add     EDI, 8
                dec     EBX
                jnz     mix_8bit_endphase_loop
                jmp     mix_quit
mix_gus16_endphase:
                mov     EDI, fptr
                mov     EBX, postproc
                mov     EDX, _judas_bufferlength
                shr     EDX, 1
                add     EDX, EDI
                add     EDX, 32
                mov     ESI, _judas_clipbuffer
                mov     ECX, _judas_cliptable
                xor     EAX, EAX
mix_gus16_endphase_loop:
                mov     AX, [ESI]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI], AX
                mov     AX, [ESI + 4]
                mov     AX, [ECX + EAX * 2]
                mov     [EDX], AX
                mov     AX, [ESI + 8]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 2], AX
                mov     AX, [ESI + 12]
                mov     AX, [ECX + EAX * 2]
                mov     [EDX + 2], AX
                mov     AX, [ESI + 16]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 4], AX
                mov     AX, [ESI + 20]
                mov     AX, [ECX + EAX * 2]
                mov     [EDX + 4], AX
                mov     AX, [ESI + 24]
                mov     AX, [ECX + EAX * 2]
                mov     [EDI + 6], AX
                mov     AX, [ESI + 28]
                mov     AX, [ECX + EAX * 2]
                mov     [EDX + 6], AX
                add     ESI, 32
                add     EDI, 8
                add     EDX, 8
                dec     EBX
                jnz     mix_gus16_endphase_loop
                jmp     mix_quit
mix_gus8_endphase:
                mov     EDI, fptr
                mov     EBX, postproc
                mov     EDX, _judas_bufferlength
                shr     EDX, 1
                add     EDX, EDI
                add     EDX, 32
                mov     ESI, _judas_clipbuffer
                mov     ECX, _judas_cliptable
                xor     EAX, EAX
mix_gus8_endphase_loop:
                mov     AX, [ESI]
                mov     AL, [ECX + EAX]
                mov     [EDI], AX
                mov     AX, [ESI + 4]
                mov     AL, [ECX + EAX]
                mov     [EDX], AX
                mov     AX, [ESI + 8]
                mov     AL, [ECX + EAX]
                mov     [EDI + 1], AX
                mov     AX, [ESI + 12]
                mov     AL, [ECX + EAX]
                mov     [EDX + 1], AX
                mov     AX, [ESI + 16]
                mov     AX, [ECX + EAX]
                mov     [EDI + 2], AX
                mov     AX, [ESI + 20]
                mov     AL, [ECX + EAX]
                mov     [EDX + 2], AX
                mov     AX, [ESI + 24]
                mov     AL, [ECX + EAX]
                mov     [EDI + 3], AX
                mov     AX, [ESI + 28]
                mov     AL, [ECX + EAX]
                mov     [EDX + 3], AX
                add     ESI, 32
                add     EDI, 4
                add     EDX, 4
                dec     EBX
                jnz     mix_gus8_endphase_loop
                jmp     mix_quit

        ;Mixes [samples] of all channels to 32-bit intermediate buffer
        ;at [dptr]. Destroys every register

normalmix_:     mov     cptr, offset _judas_channel
normalmixloop:  call    mixchannel
                add     cptr, type CHANNEL
                cmp     cptr, offset _judas_channel + CHANNELS * type CHANNEL
                jne     normalmixloop
                ret

ipmix_:         mov     cptr, offset _judas_channel
ipmixloop:      call    ipmixchannel
                add     cptr, type CHANNEL
                cmp     cptr, offset _judas_channel + CHANNELS * type CHANNEL
                jne     ipmixloop
                ret

        ;Mixes [samples] of channel [cptr] to buffer at [dptr]. Destroys
        ;every register.

mixchannel:     mov     ESI, cptr
                test    [ESI.Chn_VoiceMode], VM_ON
                jz      mixchannel_quit
                mov     EAX, [ESI.Chn_Freq]             ;Get playing speed here
                cmp     EAX, 535232                     ;Highest linear freq
                jbe     mixchannel_freqok
                mov     EAX, 535232
mixchannel_freqok:
                mov     EDX, EAX                        ;Don't worry: overflow
                shr     EDX, 16                         ;prevented by check
                shl     EAX, 16                         ;above
                div     _judas_mixrate                  ;DIV is always
                mov     word ptr fractadd + 2, AX       ;frightening!!!
                shr     EAX, 16
                mov     integeradd, EAX
                test    [ESI.Chn_VoiceMode], VM_16BIT   ;16bit takes the branch
                jnz     mixchannel16                    ;because it's unusual
                test    _judas_mixmode, STEREO          ;Mono takes the branch
                jz      mixchannel_mono                 ;because it's faster
mixchannel_stereo:
                getvolume
                cmp     [ESI.Chn_Panning], 0            ;Left panning?
                jne     mc8_notleft
                stereoadjust
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mov     ECX, fractadd                   ;ECX = fraction add
                mixloop8 mix8_left_n, 8, 0              ;DO IT!
mc8_notleft:    cmp     [ESI.Chn_Panning], 128          ;Middle panning?
                jne     mc8_notmiddle
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mov     ECX, fractadd                   ;ECX = fraction add
                mixloop8 mix8_middle_n, 8, 0            ;DO IT!
mc8_notmiddle:  cmp     [ESI.Chn_Panning], 255          ;Right panning?
                jne     mc8_notright
                stereoadjust
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mov     ECX, fractadd                   ;ECX = fraction add
                mixloop8 mix8_right_n, 8, 0             ;DO IT!
mc8_notright:   smoothpanadjust                         ;Oh no, smooth panning!
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                shl     ECX, 8
                add     ECX, _judas_volumetable
                mov     EBP, integeradd                 ;ECX not available!
                mixloop8 mix8_smooth_n, 8, 0            ;But yet we must do it..
mixchannel_mono:getvolume
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mov     ECX, fractadd                   ;ECX = fraction add
                mixloop8 mix8_mono_n, 4, 0              ;DO IT!
mixchannel_quit:ret
mixchannel16:   test    _judas_mixmode, STEREO          ;Mono takes the branch
                jz      mixchannel16_mono               ;because it's faster
mixchannel16_stereo:
                getvolume
                cmp     [ESI.Chn_Panning], 0            ;Left panning?
                jne     mc16_notleft
                stereoadjust
                mov     ECX, fractadd                   ;ECX = fraction add
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_left_n, 8, 0            ;DO IT!
mc16_notleft:   cmp     [ESI.Chn_Panning], 128          ;Middle panning?
                jne     mc16_notmiddle
                mov     ECX, fractadd                   ;ECX = fraction add
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_middle_n, 8, 0          ;DO IT!
mc16_notmiddle: cmp     [ESI.Chn_Panning], 255          ;Right panning?
                jne     mc16_notright
                stereoadjust
                mov     ECX, fractadd                   ;ECX = fraction add
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_right_n, 8, 0           ;DO IT!
mc16_notright:  smoothpanadjust                         ;Oh no, smooth panning!
                mov     EBP, integeradd
                mixloop16 mix16_smooth_n, 8, 0          ;But yet we must do it..
mixchannel16_mono:
                getvolume
                mov     ECX, fractadd                   ;ECX = fraction add
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_mono_n, 4, 0            ;DO IT!
mixchannel16_quit:
                ret

        ;Mixes [samples] of channel [cptr] to buffer at [dptr] with
        ;interpolation. Destroys every register.

ipmixchannel:   mov     ESI, cptr
                test    [ESI.Chn_VoiceMode], VM_ON
                jz      ipmixchannel_quit
                mov     EAX, [ESI.Chn_Freq]             ;Get playing speed here
                cmp     EAX, 535232                     ;Highest linear freq
                jbe     ipmixchannel_freqok
                mov     EAX, 535232
ipmixchannel_freqok:
                mov     EDX, EAX
                shr     EDX, 16
                shl     EAX, 16
                div     _judas_mixrate
                mov     word ptr fractadd + 2, AX
                shr     EAX, 16
                mov     integeradd, EAX
                test    [ESI.Chn_VoiceMode], VM_16BIT   ;16bit takes the branch
                jnz     ipmixchannel16                  ;because it's unusual
                test    _judas_mixmode, STEREO          ;Mono takes the branch
                jz      ipmixchannel_mono               ;because it's faster
ipmixchannel_stereo:
                getvolume
                cmp     [ESI.Chn_Panning], 0            ;Left panning?
                jne     imc8_notleft
                stereoadjust
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop8 mix8_left_i, 8, 1              ;DO IT!
imc8_notleft:   cmp     [ESI.Chn_Panning], 128          ;Middle panning?
                jne     imc8_notmiddle
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop8 mix8_middle_i, 8, 1            ;DO IT!
imc8_notmiddle: cmp     [ESI.Chn_Panning], 255          ;Right panning?
                jne     imc8_notright
                stereoadjust
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop8 mix8_right_i, 8, 1             ;DO IT!
imc8_notright:  smoothpanadjust                         ;Oh no, smooth panning!
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                shl     ECX, 8
                add     ECX, _judas_volumetable
                mixloop8 mix8_smooth_i, 8, 1
ipmixchannel_mono:getvolume
                shl     EBX, 8                          ;Convert to volumetable
                add     EBX, _judas_volumetable         ;ofs.
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop8 mix8_mono_i, 4, 1              ;DO IT!
ipmixchannel_quit:ret
ipmixchannel16: test    _judas_mixmode, STEREO          ;Mono takes the branch
                jz      ipmixchannel16_mono             ;because it's faster
ipmixchannel16_stereo:
                getvolume
                cmp     [ESI.Chn_Panning], 0            ;Left panning?
                jne     imc16_notleft
                stereoadjust
                mov     leftvol, EBX
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_left_i, 8, 1            ;DO IT!
imc16_notleft:  cmp     [ESI.Chn_Panning], 128          ;Middle panning?
                jne     imc16_notmiddle
                mov     leftvol, EBX
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_middle_i, 8, 1          ;DO IT!
imc16_notmiddle:cmp     [ESI.Chn_Panning], 255          ;Right panning?
                jne     imc16_notright
                stereoadjust
                mov     leftvol, EBX
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_right_i, 8, 1           ;DO IT!
imc16_notright: smoothpanadjust                         ;Oh no, smooth panning!
                mov     leftvol, EBX
                mov     rightvol, ECX
                mov     EBP, integeradd
                mixloop16 mix16_smooth_i, 8, 1          ;But yet we must do it..
ipmixchannel16_mono:
                getvolume
                mov     leftvol, EBX
                mov     EBP, integeradd                 ;EBP = integeradd
                mixloop16 mix16_mono_i, 4, 1            ;DO IT!
ipmixchannel16_quit:
                ret

zerovolume:     mov     EBP, samples
                mov     EBX, [ESI.Chn_Pos]
                mov     ECX, EBX
                shr     EBX, 16
                shl     ECX, 16
                mov     CX, [ESI.Chn_FractPos]
                mov     EAX, [ESI.Chn_Freq]
                mov     EDX, EAX
                shr     EDX, 16
                shl     EAX, 16
                div     _judas_mixrate                  ;EAX = mixrate
                test    [ESI.Chn_VoiceMode], VM_16BIT   ;If 16bit then double
                jz      zerovolume_not16bit             ;the count
                shl     EBP, 1
zerovolume_not16bit:
                mul     EBP                             ;EDX:EAX = pos. add
                add     ECX, EAX                        ;Add low part
                adc     EBX, EDX                        ;Add high part
                mov     [ESI.Chn_FractPos], CX          ;Save fractpos
                shl     EBX, 16                         ;(won't change now)
                shr     ECX, 16                         ;Now shift back: ECX
                or      ECX, EBX                        ;is integer pos
                test    [ESI.Chn_VoiceMode], VM_16BIT   ;Final adjust for 16bit
                jz      zerovolume_no16bitadjust
                and     ECX, 0fffffffeh
zerovolume_no16bitadjust:
                test    [ESI.Chn_VoiceMode], VM_LOOP    ;Is it looped?
                jnz     zerovolume_looped
                cmp     ECX, [ESI.Chn_End]
                jae     zerovolume_oneshot_end
zerovolume_ready:
                mov     [ESI.Chn_Pos], ECX              ;Store pos
                ret
zerovolume_oneshot_end:
                mov     [ESI.Chn_VoiceMode], VM_OFF
                ret
zerovolume_looped:
                mov     EAX, [ESI.Chn_End]
                sub     EAX, [ESI.Chn_Repeat]           ;EAX = subtract value
zerovolume_looped_loop:
                cmp     ECX, [ESI.Chn_End]
                jb      zerovolume_ready
                sub     ECX, EAX
                jmp     zerovolume_looped_loop

judas_code_lock_end_:
                end
