//===============================================================================================
// 3D.EXE
// Copyright (c), Firelight Multimedia, 1999,2000.
//
// This test shows EAX, DS3D, A3D and Software all being used together.  For vortex cards
// geometry will obstruct and relfect sounds.
// This example is a has prebuilt or dynamic geometry lists for A3D.  One generates the scene every
// frame (dynamic), which is good for dynamic scenes that change, but slower than a statically generated
// scene which cannot change (but things like doors opening and closing can change).
// This application also displays the use of FSOUND_GetDriverCaps to get information on the 
// 3D capabilities of the selected driver
//===============================================================================================

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#ifndef PLATFORM_LINUX
  #include <conio.h>
  #include <windows.h>
#else
  #include "wincompat.h"
#endif

#include "../../api/inc/fmod.h"
#include "../../api/inc/fmod_errors.h"	// optional


#define INTERFACE_UPDATETIME 50		// 50ms update for interface

#define DYNAMIC_GEOMETRY			// comment this to use precalculated static geometry!


/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
void Close(FSOUND_MATERIAL *material, FSOUND_SAMPLE *samp1, FSOUND_SAMPLE *samp2, FSOUND_SAMPLE *samp3)
{
	FSOUND_Geometry_Material_Free(material);
	
	// you dont need to free samples if you let fsound's sample manager look after samples, as 
	// it will free them all for you.
	FSOUND_Sample_Free(samp1);	
	FSOUND_Sample_Free(samp2);
	FSOUND_Sample_Free(samp3);

	FSOUND_Close();
}


/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
int main()
{
	FSOUND_SAMPLE	*samp1 = NULL, *samp2 = NULL, *samp3 = NULL;
#ifndef DYNAMIC_GEOMETRY
	FSOUND_GEOMLIST *glist;
#endif
	FSOUND_MATERIAL *material = NULL;
	char			key, openflag = 0, listenerflag = 1;
	int				driver, i = 0, channel1 = -1, channel2 = -1;
	float			listenerpos[3] =  { 0,0,0 }; 
	float			openingfactor=0;

	if (FSOUND_GetVersion() < FMOD_VERSION)
	{
		printf("Error : You are using the wrong DLL version!  You should be using FMOD %.02f\n", FMOD_VERSION);
		return 1;
	}

	// ==========================================================================================
	// SELECT OUTPUT METHOD
	// ==========================================================================================
	
	printf("---------------------------------------------------------\n");	
	printf("Output Type\n");	
	printf("---------------------------------------------------------\n");	
#ifndef PLATFORM_LINUX
	printf("1 - Direct Sound\n");
	printf("2 - Windows Multimedia Waveout\n");
	printf("3 - A3D\n");
#else
    printf("1 - Open Sound System (OSS) (Linux, Solaris, freebsd)\n");
    printf("2 - Elightment Sound Daemon (ESD, Linux, more ...)\n");
    printf("3 - Alsa Sound System (Linux)\n");   
#endif
	printf("4 - NoSound\n");
	printf("---------------------------------------------------------\n");	// print driver names
	printf("Press a corresponding number or ESC to quit\n");

	do
	{
		key = getch();
	} while (key != 27 && key < '1' && key > '4');
	
	switch (key)
	{

#ifndef PLATFORM_LINUX     
		case '1' :	FSOUND_SetOutput(FSOUND_OUTPUT_DSOUND);
					break;
		case '2' :	FSOUND_SetOutput(FSOUND_OUTPUT_WINMM);
					break;
		case '3' :	FSOUND_SetOutput(FSOUND_OUTPUT_A3D);
					break;
#else      
  	    case '1' :  FSOUND_SetOutput(FSOUND_OUTPUT_OSS);
	                       break;
	    case '2' :  FSOUND_SetOutput(FSOUND_OUTPUT_ESD);
	                       break;
	    case '3' :  FSOUND_SetOutput(FSOUND_OUTPUT_ALSA);
	                       break;
#endif      
		 
		case '4' :	FSOUND_SetOutput(FSOUND_OUTPUT_NOSOUND);
					break;
		default :	return 1; 
	}
	

	// ==========================================================================================
	// SELECT DRIVER
	// ==========================================================================================

	// The following list are the drivers for the output method selected above.
	printf("---------------------------------------------------------\n");	
	switch (FSOUND_GetOutput())
	{
		case FSOUND_OUTPUT_NOSOUND:	printf("NoSound"); break;
#ifndef PLATFORM_LINUX     	   
		case FSOUND_OUTPUT_WINMM:	printf("Windows Multimedia Waveout"); break;
		case FSOUND_OUTPUT_DSOUND:	printf("Direct Sound"); break;
		case FSOUND_OUTPUT_A3D:		printf("A3D"); break;
#else
   	    case FSOUND_OUTPUT_OSS:     printf("Open Sound System"); break;
	    case FSOUND_OUTPUT_ESD:     printf("Enlightment Sound Daemon"); break;
	    case FSOUND_OUTPUT_ALSA:    printf("Alsa"); break;
#endif
		 
	};
	printf(" Driver list\n");	
	printf("---------------------------------------------------------\n");	

	for (i=0; i < FSOUND_GetNumDrivers(); i++) 
	{
		printf("%d - %s\n", i+1, FSOUND_GetDriverName(i));	// print driver names
		{
			unsigned int caps = 0;

			FSOUND_GetDriverCaps(i, &caps);
			
			if (caps & FSOUND_CAPS_HARDWARE)
				printf("  * Driver supports hardware 3D sound!\n");
			if (caps & FSOUND_CAPS_EAX)
				printf("  * Driver supports EAX reverb!\n");
			if (caps & FSOUND_CAPS_GEOMETRY_OCCLUSIONS)
				printf("  * Driver supports hardware 3d geometry processing with occlusions!\n");
			if (caps & FSOUND_CAPS_GEOMETRY_REFLECTIONS)
				printf("  * Driver supports hardware 3d geometry processing with reflections!\n");
			if (caps & FSOUND_CAPS_EAX2)
				printf("  * Driver supports EAX 2.0 reverb!\n");
		}
	}
	printf("---------------------------------------------------------\n");	// print driver names
	printf("Press a corresponding number or ESC to quit\n");

	do
	{
		key = getch();
		if (key == 27) 
			return 0;
		driver = key - '1';
	} while (driver < 0 || driver >= FSOUND_GetNumDrivers());

	FSOUND_SetDriver(driver);					// Select sound card (0 = default)


	{
		unsigned int caps = 0;

		FSOUND_GetDriverCaps(FSOUND_GetDriver(), &caps);
		
		printf("---------------------------------------------------------\n");	
		printf("Driver capabilities\n");
		printf("---------------------------------------------------------\n");	
		if (!caps)
			printf("- This driver will support software mode only.\n  It does not properly support 3D sound hardware.\n");
		if (caps & FSOUND_CAPS_HARDWARE)
			printf("- Driver supports hardware 3D sound!\n");
		if (caps & FSOUND_CAPS_EAX)
			printf("- Driver supports EAX reverb!\n");
		if (caps & FSOUND_CAPS_GEOMETRY_OCCLUSIONS)
			printf("- Driver supports hardware 3d geometry processing with occlusions!\n");
		if (caps & FSOUND_CAPS_GEOMETRY_REFLECTIONS)
			printf("- Driver supports hardware 3d geometry processing with reflections!\n");
		if (caps & FSOUND_CAPS_EAX2)
			printf("- Driver supports EAX 2.0 reverb!\n");
		printf("---------------------------------------------------------\n");	
	}

	// ==========================================================================================
	// INITIALIZE
	// ==========================================================================================
	if (!FSOUND_Init(44100, 32, 0))
	{
		printf("Init: %s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	// ==========================================================================================
	// LOAD SAMPLES
	// ==========================================================================================

	// ==========================================================================================
	// 3D MONO 
	// ==========================================================================================
	samp1 = FSOUND_Sample_Load(FSOUND_FREE, "../../media/drumloop.wav", FSOUND_HW3D, 0);
	if (!samp1)
	{
		printf("samp1: %s\n", FMOD_ErrorString(FSOUND_GetError()));
		Close(material, samp1, samp2, samp3);
		return 1;
	}
	// increasing mindistnace makes it louder in 3d space
	FSOUND_Sample_SetMinMaxDistance(samp1, 4.0f, 1000.0f);	
	FSOUND_Sample_SetLoopMode(samp1, FSOUND_LOOP_NORMAL);

	// ==========================================================================================
	// 3D MONO 
	// ==========================================================================================
	samp2 = FSOUND_Sample_Load(FSOUND_UNMANAGED, "../../media/jungle.wav", FSOUND_HW3D, 0);
	if (!samp2)
	{
		printf("samp2: %s\n", FMOD_ErrorString(FSOUND_GetError()));
		Close(material, samp1, samp2, samp3);
		return 1;
	}
	// increasing mindistance makes it louder in 3d space
	FSOUND_Sample_SetMinMaxDistance(samp2, 4.0f, 1000.0f);
	FSOUND_Sample_SetLoopMode(samp2, FSOUND_LOOP_NORMAL);

	// ==========================================================================================
	// 2D STEREO
	// ==========================================================================================
	samp3 = FSOUND_Sample_Load(FSOUND_UNMANAGED, "../../media/chimes.wav", FSOUND_2D, 0);
	if (!samp3)
	{
		printf("samp3: %s\n", FMOD_ErrorString(FSOUND_GetError()));
		Close(material, samp1, samp2, samp3);
		return 1;
	}

	// ==========================================================================================
	// DISPLAY HELP
	// ==========================================================================================

	printf("FSOUND Output Method : ");
	switch (FSOUND_GetOutput())
	{
		case FSOUND_OUTPUT_NOSOUND:	printf("FSOUND_OUTPUT_NOSOUND\n"); break;
#ifndef PLATFORM_LINUX    
	    case FSOUND_OUTPUT_WINMM:	printf("FSOUND_OUTPUT_WINMM\n"); break;
		case FSOUND_OUTPUT_DSOUND:	printf("FSOUND_OUTPUT_DSOUND\n"); break;
		case FSOUND_OUTPUT_A3D:		printf("FSOUND_OUTPUT_A3D\n"); break;
#else
	    case FSOUND_OUTPUT_OSS:     printf("Open Sound System"); break;
	    case FSOUND_OUTPUT_ESD:     printf("Enlightment Sound Daemon"); break;
	    case FSOUND_OUTPUT_ALSA:    printf("Alsa"); break;	   
#endif	   
	};

	printf("FSOUND Mixer         : ");
	switch (FSOUND_GetMixer())
	{
		case FSOUND_MIXER_BLENDMODE:	printf("FSOUND_MIXER_BLENDMODE\n"); break;
		case FSOUND_MIXER_MMXP5:		printf("FSOUND_MIXER_MMXP5\n"); break;
		case FSOUND_MIXER_MMXP6:		printf("FSOUND_MIXER_MMXP6\n"); break;
		case FSOUND_MIXER_QUALITY_FPU:	printf("FSOUND_MIXER_QUALITY_FPU\n"); break;
		case FSOUND_MIXER_QUALITY_MMXP5:printf("FSOUND_MIXER_QUALITY_MMXP5\n"); break;
		case FSOUND_MIXER_QUALITY_MMXP6:printf("FSOUND_MIXER_QUALITY_MMXP6\n"); break;
	};
	printf("FSOUND Driver        : ");
	printf("%s\n", FSOUND_GetDriverName(FSOUND_GetDriver()));
	printf("Hardware 3D channels : %d\n", FSOUND_GetNumHardwareChannels());

	printf("=========================================================================\n");
	printf("Press 1		Pause/Unpause 16bit 3D sound at any time\n");
	printf("      2		Pause/Unpause 8bit 3D sound at any time\n");
	printf("      3		Play 16bit STEREO 2D sound at any time\n");
	printf("      4		Change to EAX Reverb mode CONCERTHALL (DirectSound/SBLive only)\n");
	printf("      5		Change to EAX Reverb mode SEWERPIPE (DirectSound/SBLive only)\n");
	printf("      6		Change to EAX Reverb mode PSYCHOTIC (DirectSound/SBLive only)\n");
	printf("      7		Open/Close door on right wall (A3D only)\n");
	printf("      8		Change to dull material type for walls (dynamic A3D only)\n");
	printf("      9		Change to bright material type for walls (dynamic A3D only)\n");
	printf("      <		Move listener left (in still mode)\n");
	printf("      >		Move listener right (in still mode)\n");
	printf("      SPACE Stop/Start listener automatic movement\n");
	printf("      ESC	Quit\n");
	printf("=========================================================================\n");


#ifndef DYNAMIC_GEOMETRY

	// ==========================================================================================
	// PRE-BUILD GEOMETRY
	// ==========================================================================================

	glist = FSOUND_Geometry_List_Create(TRUE);
	if (!glist)
	{
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	FSOUND_Geometry_List_Begin(glist);

	{
		// COORDINATE SYSTEM : X = right, Y = up, Z = forwards. (Left handed)
		int count;
		#define NUMPOLYS 4
		float poly[NUMPOLYS][4][3] = 
		{
			{	// left wall
				{ -35.0, -20.0, -20.0, },
				{ -35.0,  20.0, -20.0, },
				{ -35.0,  20.0,  20.0, },
				{ -35.0, -20.0,  20.0, },
			},
			{	// front wall
				{ -35.0,  20.0,  20.0, },
				{  10.0,  20.0,  20.0, },
				{  10.0, -20.0,  20.0, },
				{ -35.0, -20.0,  20.0, },
			},
			{	// back wall
				{ -35.0, -20.0,  -20.0, },
				{  10.0, -20.0,  -20.0, },
				{  10.0,  20.0,  -20.0, },
				{ -35.0,  20.0,  -20.0, },
			},
			{	// right wall
				{  10.0, -20.0, -20.0, },
				{  10.0, -20.0,  20.0, },
				{  10.0,  20.0,  20.0, },
				{  10.0,  20.0, -20.0, },
			},
		};
		float doorpoly[4][3] = 
		{	// hole in right wall 
			{  10.0, -20.0, -5.0, },
			{  10.0, -20.0,  5.0, },
			{  10.0,  20.0,  5.0, },
			{  10.0,  20.0, -5.0, },
		};
		float normal[NUMPOLYS][3] = 
		{
			{   1.0,  0.0,  0.0, },	// left wall
			{  -1.0,  0.0,  0.0, },	// right wall
			{   0.0,  0.0, -1.0, },	// front wall
			{   0.0,  0.0,  1.0, },	// back wall
		};

		material = FSOUND_Geometry_Material_Create();
		FSOUND_Geometry_Material_SetAttributes(material, 1, 1, 0, 0);

		FSOUND_Geometry_Material_Set(material);

		for (count=0; count<NUMPOLYS; count++)
			FSOUND_Geometry_AddPolygon(poly[count][0],poly[count][1],poly[count][2],poly[count][3], NULL, FSOUND_GEOMETRY_NORMAL | FSOUND_GEOMETRY_REFLECTIVE, 0);

		FSOUND_Geometry_AddPolygon(doorpoly[0],doorpoly[1],doorpoly[2],doorpoly[3], NULL, FSOUND_GEOMETRY_NORMAL | FSOUND_GEOMETRY_REFLECTIVE | FSOUND_GEOMETRY_OPENING_REFERENCE, &openingfactor);
	}

	FSOUND_Geometry_List_End(glist);

#endif	// DYNAMIC_GEOMETRY


	// ==========================================================================================
	// PLAY 2 LOOPING SOUNDS
	// ==========================================================================================

	{
		float pos[3] = { -10.0f, -0.0f, 0.0f };
		float vel[3] = { 0,0,0 };

		channel1 = FSOUND_PlaySound3DAttrib(FSOUND_FREE, samp1, -1, -1, -1, pos, vel);
		FSOUND_Reverb_SetMix(channel1, FSOUND_REVERBMIX_USEDISTANCE);
	}
	{
		float pos[3] = { 15.0f, -0.0f, -0.0f };
		float vel[3] = { 0,0,0 };

		channel2 = FSOUND_PlaySound3DAttrib(FSOUND_FREE, samp2, -1, -1, -1, pos, vel);
		FSOUND_SetPaused(channel2, FALSE);

		FSOUND_Reverb_SetMix(channel2, FSOUND_REVERBMIX_USEDISTANCE);
	}

	// ==========================================================================================
	// MAIN LOOP
	// ==========================================================================================

	material = FSOUND_Geometry_Material_Create();

//	FSOUND_Reverb_SetEnvironment(FSOUND_PRESET_CONCERTHALL);

	do
	{

		if (kbhit())
		{
			key = getch();

			if (key == '1') 
				FSOUND_SetPaused(channel1, !FSOUND_GetPaused(channel1));
			if (key == '2') 
				FSOUND_SetPaused(channel2, !FSOUND_GetPaused(channel2));
			if (key == '3') 
				FSOUND_PlaySound(FSOUND_FREE, samp3);
			if (key == '4') 
				FSOUND_Reverb_SetEnvironment(FSOUND_PRESET_CONCERTHALL);
			if (key == '5') 
				FSOUND_Reverb_SetEnvironment(FSOUND_PRESET_SEWERPIPE);
			if (key == '6')
				FSOUND_Reverb_SetEnvironment(FSOUND_PRESET_PSYCHOTIC);

			if (key == '7') 
				openflag = !openflag;
			
			if (key == '8') FSOUND_Geometry_Material_SetAttributes(material, 0, 0,    0, 0);
			if (key == '9') FSOUND_Geometry_Material_SetAttributes(material, 1, 1, 0.1f, 0.1f);

			if (key == ' ')
				listenerflag = !listenerflag;

			if (!listenerflag)
			{
				if (key == '<') 
				{
					listenerpos[0] -= 1.0f;
					if (listenerpos[0] < -35)
						listenerpos[0] = -35;
				}
				if (key == '>') 
				{
					listenerpos[0] += 1.0f;
					if (listenerpos[0] > 30)
						listenerpos[0] = 30;
				}
			}
		}


#ifdef DYNAMIC_GEOMETRY
		// ==========================================================================================
		// BUILD GEOMETRY
		// ==========================================================================================
		{
			int count;
			
			// COORDINATE SYSTEM : X = right, Y = up, Z = forwards. (Left handed)

			#define NUMPOLYS 4
			float poly[NUMPOLYS][4][3] = 
			{
				{	// left wall
					{ -35.0, -20.0, -20.0, },
					{ -35.0,  20.0, -20.0, },
					{ -35.0,  20.0,  20.0, },
					{ -35.0, -20.0,  20.0, },
				},
				{	// front wall
					{ -35.0,  20.0,  20.0, },
					{  10.0,  20.0,  20.0, },
					{  10.0, -20.0,  20.0, },
					{ -35.0, -20.0,  20.0, },
				},
				{	// back wall
					{ -35.0, -20.0,  -20.0, },
					{  10.0, -20.0,  -20.0, },
					{  10.0,  20.0,  -20.0, },
					{ -35.0,  20.0,  -20.0, },
				},
				{	// right wall
					{  10.0, -20.0, -20.0, },
					{  10.0, -20.0,  20.0, },
					{  10.0,  20.0,  20.0, },
					{  10.0,  20.0, -20.0, },
				},
			};
			float doorpoly[4][3] = 
			{	// hole in right wall 
				{  10.0, -20.0, -5.0, },
				{  10.0, -20.0,  5.0, },
				{  10.0,  20.0,  5.0, },
				{  10.0,  20.0, -5.0, },
			};
			float normal[NUMPOLYS][3] = 
			{
				{   1.0,  0.0,  0.0, },	// left wall
				{  -1.0,  0.0,  0.0, },	// right wall
				{   0.0,  0.0, -1.0, },	// front wall
				{   0.0,  0.0,  1.0, },	// back wall
			};

			FSOUND_Geometry_Material_Set(material);	// remember to put this BEFORE the addpolygons

			for (count=0; count<NUMPOLYS; count++)
				FSOUND_Geometry_AddPolygon(poly[count][0],poly[count][1],poly[count][2],poly[count][3], NULL, FSOUND_GEOMETRY_NORMAL | FSOUND_GEOMETRY_REFLECTIVE, 0);

			openingfactor = openflag ? 1.0f : 0.0f;

			FSOUND_Geometry_AddPolygon(doorpoly[0],doorpoly[1],doorpoly[2],doorpoly[3], NULL, FSOUND_GEOMETRY_NORMAL | FSOUND_GEOMETRY_REFLECTIVE | FSOUND_GEOMETRY_OPENING_REFERENCE, &openingfactor);
		}

#else	// DYNAMIC_GEOMETRY

		// ==========================================================================================
		// UPDATE GEOMETRY
		// ==========================================================================================
		openingfactor = openflag ? 1.0f : 0.0f;		// this is pointed to by the geometry list
		FSOUND_Geometry_AddList(glist);

#endif	// DYNAMIC_GEOMETRY

		// ==========================================================================================
		// UPDATE THE LISTENER
		// ==========================================================================================
		{
			static float t = 0;
			static float lastpos[3] = { 0,0,0 };
			float vel[3];

			if (listenerflag)
				listenerpos[0] = ((float)sin(t*0.05f) * 33.0f); // left right pingpong

			// ********* NOTE ******* READ NEXT COMMENT!!!!!
			// vel = how far we moved last FRAME (m/f), then time compensate it to SECONDS (m/s).
			vel[0] = (listenerpos[0]-lastpos[0]) * (1000 / INTERFACE_UPDATETIME);
			vel[1] = (listenerpos[1]-lastpos[1]) * (1000 / INTERFACE_UPDATETIME);
			vel[2] = (listenerpos[2]-lastpos[2]) * (1000 / INTERFACE_UPDATETIME);

			// store pos for next time
			lastpos[0] = listenerpos[0];
			lastpos[1] = listenerpos[1];
			lastpos[2] = listenerpos[2];

			FSOUND_3D_Listener_SetAttributes(&listenerpos[0], &vel[0], 0, 0, 1.0f, 0, 1.0f, 0);

			t += (30 * (1.0f / (float)INTERFACE_UPDATETIME));	// t is just a time value .. it increments in 30m/s steps in this example

			// print out a small visual display
			{
				char s[80];

				sprintf(s, "|.......................<1>..................|   <2>                Door:%s", openflag ? "Open" : "Closed");
				s[(int)listenerpos[0]+35] = 'L';
				if (openflag)
					s[10+35] = ':';
				
				printf("%s\r", s);
			}			
		}

		FSOUND_3D_Update();

		Sleep(INTERFACE_UPDATETIME-1);	// -2 for printf etc

	} while (key != 27);

	printf("\n");

	// ==========================================================================================
	// CLEANUP AND SHUTDOWN
	// ==========================================================================================

#ifndef DYNAMIC_GEOMETRY
	FSOUND_Geometry_List_Free(glist);
#endif	// DYNAMIC_GEOMETRY

	Close(material, samp1, samp2, samp3);
   
    return 0;
}
