#![warn(clippy::perf, clippy::complexity, clippy::pedantic, clippy::suspicious)]
#![allow(
    clippy::missing_errors_doc,
    clippy::missing_panics_doc,
    reason = "We're not going to write comprehensive docs"
)]
#![allow(
    clippy::cast_precision_loss,
    clippy::cast_sign_loss,
    clippy::cast_possible_truncation,
    reason = "There are no sufficient floating point types"
)]

use anyhow::{Context, Result};
#[cfg(feature = "rocket")]
use rust_rocket::{Event as RocketEvent, Rocket, Tracks};
use sdl3::{
    Sdl,
    event::Event as SdlEvent,
    gpu::{Device, ShaderFormat},
    hint::names::{VIDEO_DRIVER, VIDEO_WAYLAND_MODE_EMULATION, VIDEO_WAYLAND_MODE_SCALING},
    keyboard::Keycode,
    video::{FullscreenType, Window},
};
use sdl3engine::{
    config::{self, Config},
    render::{Renderer, StretchFactor},
};
use std::{
    ffi::OsString,
    path::Path,
    time::{Duration, Instant},
};

struct TimeSource {
    start: Instant,
    offset: Duration,
    paused: bool,
}

impl TimeSource {
    pub fn new() -> Self {
        Self {
            start: Instant::now(),
            offset: Duration::from_secs(0),
            paused: false,
        }
    }

    pub fn get_time(&self) -> Duration {
        if self.paused {
            self.offset
        } else {
            self.start.elapsed() + self.offset
        }
    }

    pub fn pause(&mut self, state: bool) {
        self.offset = self.get_time();
        self.start = Instant::now();
        self.paused = state;
    }

    pub fn seek(&mut self, to: Duration) {
        self.offset = to;
        self.start = Instant::now();
    }
}

#[must_use = "Renderer must be updated with the display aspect ratio"]
fn configure_fullscreen(sdl: &Sdl, window: &mut Window, config: &Config) -> Result<StretchFactor> {
    sdl.mouse().show_cursor(false);

    let mut stretch = None;

    // If on wayland, try to enable an emulated fullscreen mode
    if sdl.video()?.current_video_driver() == "wayland" {
        let display = window.get_display()?;

        // Find required dimensions for aspect ratio preserving hacks
        let dmode = display.get_mode()?;
        let display_aspect_ratio = dmode.w as f32 / dmode.h as f32;
        let width = config.render.resolution.width;
        let height = config.render.resolution.height;
        let render_aspect_ratio = width as f32 / height as f32;

        eprintln!("video: Searching for wp_viewporter-emulated mode {width}x{height}");

        // Find mode with suitable dimensions
        let modes = display.get_fullscreen_modes()?;
        let mode = modes
            .iter()
            .find(|&mode| {
                let mode_w = mode.w as u32;
                let mode_h = mode.h as u32;
                let stretch = display_aspect_ratio / (mode_w as f32 / mode_h as f32);
                if display_aspect_ratio > render_aspect_ratio {
                    mode_h == height && mode_w as f32 >= width as f32 * stretch
                } else {
                    mode_w == width && mode_h as f32 >= height as f32 / stretch
                }
            })
            .copied();

        let mode = if let Some(mode) = mode {
            eprintln!(
                "video: Found compatible mode: {}x{}@{}Hz",
                mode.w, mode.h, mode.refresh_rate
            );
            Some(mode)
        } else {
            // Fall back to searching for closest match
            eprintln!("video: No exact match found, using closest match");
            modes
                .iter()
                .filter(|&mode| {
                    let mode_w = mode.w as u32;
                    let mode_h = mode.h as u32;
                    mode_w >= width && mode_h >= height
                })
                .min_by_key(|&mode| mode.h)
                .copied()
        };

        if let Some(mode) = mode {
            stretch = Some(display_aspect_ratio / (mode.w as f32 / mode.h as f32));
            eprintln!(
                "video: Setting mode {}x{}@{}Hz",
                mode.w, mode.h, mode.refresh_rate
            );
        }
        window
            .set_display_mode(mode)
            .context("Can't set fullscreen mode")?;
    }

    window
        .set_fullscreen(true)
        .context("SDL is unable set fullscreen")?;
    Ok(stretch)
}

#[cfg(feature = "reload")]
fn create_error_renderer_config(config: Option<&Config>, error: &anyhow::Error) -> Config {
    use config::{Draw, Output, Text, TextLayer, TextSource};
    let mut config = config.cloned().unwrap_or_default();
    config.render.passes = vec![];
    config.render.output = Output {
        draw: vec![Draw::Text(Text {
            layers: vec![TextLayer::default()],
            sources: vec![TextSource::Inline {
                text: format!("{error:?}"),
            }],
            refresh: None,
            cursor: false,
        })],
    };
    config
}

#[cfg(feature = "rocket")]
fn save_tracks(tracks: &Tracks) -> Result<()> {
    if cfg!(debug_assertions) {
        let path = Path::new(sdl3engine::RESOURCE_DIR).join("tracks.bin");
        let mut file = std::fs::OpenOptions::new()
            .write(true)
            .truncate(true)
            .create(true)
            .open(&path)
            .with_context(|| format!("Can't write to {}", path.display()))?;
        bincode::encode_into_std_write(tracks, &mut file, bincode::config::standard())
            .with_context(|| format!("Failed to encode {}", path.display()))?;
        eprintln!("rocket: Tracks saved to {}", path.display());
    }

    Ok(())
}

struct MainResources {
    config_path: OsString,
    config: Config,
    sdl: Sdl,
    window: Window,
    gpu: Device,
    renderer: Renderer,
    time_source: TimeSource,
    #[cfg(feature = "rocket")]
    rocket: Rocket,
    #[cfg(feature = "audio")]
    player: Option<sdl3engine::audio::Player<std::io::BufReader<std::fs::File>>>,
}

fn main_loop(
    MainResources {
        config_path,
        config,
        sdl,
        mut window,
        gpu,
        mut renderer,
        mut time_source,
        #[cfg(feature = "rocket")]
        mut rocket,
        #[cfg(feature = "audio")]
        mut player,
    }: MainResources,
) -> Result<()> {
    let mut event_pump = sdl.event_pump().context("Can't initialize SDL events")?;
    'main: loop {
        // Handle events from SDL
        for event in event_pump.poll_iter() {
            match event {
                SdlEvent::Quit { .. }
                | SdlEvent::KeyDown {
                    keycode: Some(Keycode::Escape | Keycode::Q),
                    ..
                } => {
                    break 'main;
                }
                SdlEvent::KeyDown {
                    keycode: Some(Keycode::F),
                    ..
                } => {
                    if window.fullscreen_state() == FullscreenType::Off {
                        let stretch = configure_fullscreen(&sdl, &mut window, &config)?;
                        renderer.set_stretch_factor(stretch);
                    } else {
                        window.set_fullscreen(false)?;
                        renderer.set_stretch_factor(None);
                    }
                }
                #[cfg(feature = "reload")]
                SdlEvent::KeyDown {
                    keycode: Some(Keycode::R),
                    ..
                } => {
                    let config = match Config::load(&config_path) {
                        Ok(config) => config,
                        Err(ref e) => create_error_renderer_config(None, e),
                    };
                    // TODO: Reload rocket and audio too
                    renderer = match Renderer::new(&gpu, &window, &config.render) {
                        Ok(renderer) => renderer,
                        Err(ref e) => {
                            let config = create_error_renderer_config(Some(&config), e);
                            Renderer::new(&gpu, &window, &config.render)?
                        }
                    };
                }
                _ => {}
            }
        }

        // Handle events from the rocket tracker
        #[cfg(feature = "rocket")]
        while let Some(event) = rocket.poll_events() {
            match event {
                RocketEvent::Seek(to) => {
                    time_source.seek(to);
                    #[cfg(feature = "audio")]
                    if let Some(player) = &mut player {
                        player.seek(to).context("Can't seek audio stream")?;
                    }
                }
                RocketEvent::Pause(state) => {
                    time_source.pause(state);
                    #[cfg(feature = "audio")]
                    if let Some(player) = &mut player {
                        player.pause(state);
                    }
                }
                RocketEvent::SaveTracks => {
                    save_tracks(rocket.get_tracks()).unwrap_or_else(|e| eprintln!("{e:?}"));
                }
                RocketEvent::NotConnected => break,
            }
        }

        // Get current frame's time and keep the tracker updated
        let time = time_source.get_time();
        #[cfg(feature = "rocket")]
        rocket.set_time(&time);

        // Exit if release
        #[cfg(all(not(debug_assertions), feature = "demo"))]
        if let Some(player) = &player {
            if &time >= player.length() {
                break 'main;
            }
        }

        // Convert to float
        let time = if cfg!(feature = "rocket") {
            rocket.get_row()
        } else {
            time.as_secs_f32()
        };

        renderer.render(time)?;
    }

    #[cfg(feature = "rocket")]
    save_tracks(rocket.get_tracks()).unwrap_or_else(|e| eprintln!("{e:?}"));

    Ok(())
}

fn main() -> Result<()> {
    let config_path = std::env::args_os()
        .nth(1)
        .unwrap_or_else(|| OsString::from("config.ron"));

    let config = Config::load(&config_path)?;
    let config::Resolution { width, height } = config.render.resolution;

    if cfg!(target_os = "linux") {
        sdl3::hint::set(VIDEO_DRIVER, "wayland,x11");
        sdl3::hint::set(VIDEO_WAYLAND_MODE_EMULATION, "1");
        sdl3::hint::set(VIDEO_WAYLAND_MODE_SCALING, "stretch");
    }

    let sdl = sdl3::init().context("Can't initialize SDL3")?;
    let video = sdl.video().context("Can't initialize SDL3 video")?;

    let mut window = video
        .window("Mehu", width, height)
        .position_centered()
        .resizable()
        .build()
        .context("Can't create a window")?;

    let gpu = sdl3::gpu::Device::new(ShaderFormat::SpirV, cfg!(debug_assertions))
        .context("Can't initialize a GPU device")?
        .with_window(&window)
        .context("Window cannot be used with GPU device")?;

    let mut renderer = Renderer::new(&gpu, &window, &config.render)?;

    if cfg!(not(debug_assertions)) {
        match configure_fullscreen(&sdl, &mut window, &config) {
            Ok(stretch) => renderer.set_stretch_factor(stretch),
            Err(e) => {
                sdl3::messagebox::show_simple_message_box(
                    sdl3::messagebox::MessageBoxFlag::ERROR,
                    "Can't enter fullscreen",
                    &format!("{e}"),
                    &window,
                )?;
            }
        }
    }

    // Initialize rocket
    #[cfg(feature = "rocket")]
    let rocket = {
        let tracks = if cfg!(debug_assertions) {
            Tracks::default()
        } else {
            let path = Path::new(sdl3engine::RESOURCE_DIR).join("tracks.bin");
            let mut file = std::fs::File::open(&path)
                .with_context(|| format!("Can't open, {}", path.display()))?;
            bincode::decode_from_std_read(&mut file, bincode::config::standard())
                .with_context(|| format!("Can't decode {}", path.display()))?
        };
        Rocket::new(tracks, config.audio.bpm)
    };

    // Initialize audio
    #[cfg(feature = "audio")]
    let player = 'blk: {
        // TODO: add configuration and move this to module
        use sdl3engine::audio::Player;
        use std::{fs::File, io::BufReader};
        if let Some(name) = &config.audio.name {
            let audio = sdl.audio().context("Can't initialize SDL3 audio")?;
            let path = Path::new(sdl3engine::RESOURCE_DIR).join(name);
            let rdr = match File::open(&path) {
                Ok(file) => BufReader::new(file),
                Err(e) => {
                    eprintln!("audio: Can't open {}: {}", path.display(), e);
                    break 'blk None;
                }
            };
            let mut player = Player::new(rdr, &audio).context("Failed to start audio")?;
            player.pause(false);
            Some(player)
        } else {
            None
        }
    };

    let time_source = TimeSource::new();

    main_loop(MainResources {
        config_path,
        config,
        sdl,
        window,
        gpu,
        renderer,
        time_source,
        #[cfg(feature = "rocket")]
        rocket,
        #[cfg(feature = "audio")]
        player,
    })
}
