#![cfg(feature = "audio")]

mod opus;

use anyhow::{Context, Result, bail};
use opus::OpusStream;
use sdl3::audio::{AudioCallback, AudioSpec, AudioStreamWithCallback};
use std::{
    io::{Read, Seek},
    num::NonZero,
    time::Duration,
};

const MAX_CHANNELS: usize = 2;
pub const MAX_OPUS_PACKET_MS: usize = 120;
pub const SAMPLE_RATE: i32 = 48000;
const NANOS_PER_SEC: u64 = 1_000_000_000;

struct CallbackContext<T: Read + Seek> {
    opus: OpusStream<T>,
    buf: Vec<f32>,
    skip: usize,
}

impl<T: Read + Seek> CallbackContext<T> {
    pub fn new(opus: OpusStream<T>) -> Self {
        let skip = opus.pre_skip();
        Self {
            opus,
            buf: vec![0f32; MAX_CHANNELS * MAX_OPUS_PACKET_MS * SAMPLE_RATE as usize],
            skip,
        }
    }
}

impl<T: Read + Seek + 'static> AudioCallback<f32> for CallbackContext<T> {
    fn callback(&mut self, stream: &mut sdl3::audio::AudioStream, requested: i32) {
        let mut put = 0;
        while put < requested as usize {
            let Some(samples) = self.opus.decode_packet(&mut self.buf).map_or_else(
                |e| {
                    eprintln!("audio: libopus decode failed: {e}");
                    None
                },
                |o| o.map(NonZero::get),
            ) else {
                return;
            };

            let not_skipped = samples.saturating_sub(self.skip);
            self.skip = self.skip.saturating_sub(samples);

            stream.put_data_f32(&self.buf[0..not_skipped]).unwrap();
            put += not_skipped;
        }
    }
}

pub struct Player<T: Read + Seek + 'static> {
    length: Duration,
    stream: AudioStreamWithCallback<CallbackContext<T>>,
}

impl<T: Read + Seek> Player<T> {
    pub fn new(rdr: T, audio: &sdl3::AudioSubsystem) -> Result<Self> {
        let opus = OpusStream::new(rdr)?;
        let channels: i32 = opus.channel_count().into();

        let nanos = opus.length() * NANOS_PER_SEC / SAMPLE_RATE as u64;
        let length = Duration::from_nanos(nanos);

        let callback = CallbackContext::new(opus);

        let spec = AudioSpec {
            freq: Some(SAMPLE_RATE),
            channels: Some(channels),
            format: Some(sdl3::audio::AudioFormat::F32LE),
        };

        let stream = audio
            .open_playback_stream(&spec, callback)
            .context("Can't open SDL3 audio playback stream")?;
        Ok(Self { length, stream })
    }

    #[must_use]
    pub fn length(&self) -> &Duration {
        &self.length
    }

    pub fn pause(&mut self, state: bool) {
        if state {
            self.stream.pause().expect("Can't pause audio stream");
        } else {
            self.stream.resume().expect("Can't resume audio stream");
        }
    }

    pub fn seek(&mut self, to: Duration) -> Result<()> {
        let to_nanos = to.as_nanos() as u64;
        let pos_goal = (to_nanos * SAMPLE_RATE as u64) / NANOS_PER_SEC;

        {
            let Some(mut stream) = self.stream.lock() else {
                bail!("Can't lock SDL audio stream");
            };
            stream.opus.seek(pos_goal).ok();
        }

        Ok(())
    }
}
