#![warn(clippy::perf, clippy::complexity, clippy::pedantic, clippy::suspicious)]
#![allow(
    clippy::missing_errors_doc,
    clippy::missing_panics_doc,
    reason = "We're not going to write comprehensive docs"
)]
#![allow(
    clippy::cast_precision_loss,
    clippy::cast_sign_loss,
    clippy::cast_possible_truncation,
    reason = "There are no sufficient floating point types"
)]

use anyhow::{Context, Result, bail};
#[cfg(feature = "rocket")]
use rust_rocket::{Event as RocketEvent, Rocket, Tracks};
use sdl3::{
    Sdl,
    event::Event as SdlEvent,
    gpu::{Device, ShaderFormat},
    hint::names::{VIDEO_DRIVER, VIDEO_WAYLAND_MODE_EMULATION, VIDEO_WAYLAND_MODE_SCALING},
    keyboard::Keycode,
    sys::video::{SDL_DisplayID, SDL_DisplayMode},
    video::{DisplayMode, FullscreenType, Window},
};
use sdl3engine::{
    config::{self, Config},
    render::Renderer,
};
use std::{
    ffi::OsString,
    os::raw::c_void,
    path::Path,
    time::{Duration, Instant},
};

unsafe extern "C" {
    fn SDL_GetVideoDisplay(display_id: SDL_DisplayID) -> *mut c_void;
    fn SDL_AddFullscreenDisplayMode(display: *mut c_void, mode: *const SDL_DisplayMode) -> bool;
}

struct TimeSource {
    start: Instant,
    offset: Duration,
    paused: bool,
}

impl TimeSource {
    pub fn new() -> Self {
        Self {
            start: Instant::now(),
            offset: Duration::from_secs(0),
            paused: false,
        }
    }

    pub fn get_time(&self) -> Duration {
        if self.paused {
            self.offset
        } else {
            self.start.elapsed() + self.offset
        }
    }

    pub fn pause(&mut self, state: bool) {
        self.offset = self.get_time();
        self.start = Instant::now();
        self.paused = state;
    }

    pub fn seek(&mut self, to: Duration) {
        self.offset = to;
        self.start = Instant::now();
    }
}

fn configure_fullscreen(sdl: &Sdl, window: &mut Window, config: &Config) -> Result<()> {
    sdl.mouse().show_cursor(false);

    // If on wayland, try to enable an emulated fullscreen mode
    if sdl.video()?.current_video_driver() == "wayland" {
        let display = window.get_display()?;

        // Find required dimensions for aspect ratio preserving hacks
        let dmode = display.get_mode()?;
        let display_aspect_ratio = dmode.w as f32 / dmode.h as f32;
        let mut width = config.render.resolution.width;
        let mut height = config.render.resolution.height;
        let render_aspect_ratio = width as f32 / height as f32;

        if display_aspect_ratio > render_aspect_ratio {
            width = ((height as f32 * display_aspect_ratio) + 0.5) as u32;
        } else {
            height = ((width as f32 / display_aspect_ratio) + 0.5) as u32;
        }

        let mode = SDL_DisplayMode {
            displayID: display.to_ll(),
            format: dmode.format.into(),
            w: i32::try_from(width).unwrap(),
            h: i32::try_from(height).unwrap(),
            pixel_density: 0.,
            refresh_rate: dmode.refresh_rate,
            refresh_rate_numerator: dmode.refresh_rate_numerator,
            refresh_rate_denominator: dmode.refresh_rate_denominator,
            internal: std::ptr::null_mut(),
        };

        unsafe {
            let videodisplay = SDL_GetVideoDisplay(display.to_ll());
            if videodisplay.is_null() {
                bail!("SDL_GetVideoDisplay returned NULL");
            }
            SDL_AddFullscreenDisplayMode(videodisplay, &mode);
        }

        window
            .set_display_mode(unsafe { DisplayMode::from_ll(&mode) })
            .context("Can't set fullscreen mode")?;
    }

    window
        .set_fullscreen(true)
        .context("SDL is unable set fullscreen")?;
    Ok(())
}

#[cfg(feature = "reload")]
fn create_error_renderer_config(config: Option<&Config>, error: &anyhow::Error) -> Config {
    use config::{Draw, Output, Text, TextLayer, TextSource};
    let mut config = config.cloned().unwrap_or_default();
    config.render.passes = vec![];
    config.render.output = Output {
        draw: vec![Draw::Text(Text {
            layers: vec![TextLayer::default()],
            sources: vec![TextSource::Inline {
                text: format!("{error:?}"),
            }],
            refresh: None,
            cursor: false,
            timing: None,
        })],
    };
    config
}

#[cfg(feature = "rocket")]
fn save_tracks(tracks: &Tracks) -> Result<()> {
    if cfg!(debug_assertions) {
        let path = Path::new(sdl3engine::RESOURCE_DIR).join("tracks.bin");
        let mut file = std::fs::OpenOptions::new()
            .write(true)
            .truncate(true)
            .create(true)
            .open(&path)
            .with_context(|| format!("Can't write to {}", path.display()))?;
        bincode::encode_into_std_write(tracks, &mut file, bincode::config::standard())
            .with_context(|| format!("Failed to encode {}", path.display()))?;
        eprintln!("rocket: Tracks saved to {}", path.display());
    }

    Ok(())
}

struct MainResources {
    config_path: OsString,
    config: Config,
    sdl: Sdl,
    window: Window,
    gpu: Device,
    renderer: Renderer,
    time_source: TimeSource,
    #[cfg(feature = "rocket")]
    rocket: Rocket,
    #[cfg(feature = "audio")]
    player: Option<sdl3engine::audio::Player<std::io::BufReader<std::fs::File>>>,
}

fn main_loop(
    MainResources {
        config_path,
        config,
        sdl,
        mut window,
        gpu,
        mut renderer,
        mut time_source,
        #[cfg(feature = "rocket")]
        mut rocket,
        #[cfg(feature = "audio")]
        mut player,
    }: MainResources,
) -> Result<()> {
    let mut event_pump = sdl.event_pump().context("Can't initialize SDL events")?;
    'main: loop {
        // Handle events from SDL
        for event in event_pump.poll_iter() {
            match event {
                SdlEvent::Quit { .. }
                | SdlEvent::KeyDown {
                    keycode: Some(Keycode::Escape | Keycode::Q),
                    ..
                } => {
                    break 'main;
                }
                SdlEvent::KeyDown {
                    keycode: Some(Keycode::F),
                    ..
                } => {
                    if window.fullscreen_state() == FullscreenType::Off {
                        configure_fullscreen(&sdl, &mut window, &config)?;
                    } else {
                        window.set_fullscreen(false)?;
                    }
                }
                #[cfg(feature = "reload")]
                SdlEvent::KeyDown {
                    keycode: Some(Keycode::R),
                    ..
                } => {
                    let config = match Config::load(&config_path) {
                        Ok(config) => config,
                        Err(ref e) => create_error_renderer_config(None, e),
                    };
                    // TODO: Reload rocket and audio too
                    renderer = match Renderer::new(&gpu, &window, &config.render) {
                        Ok(renderer) => renderer,
                        Err(ref e) => {
                            let config = create_error_renderer_config(Some(&config), e);
                            Renderer::new(&gpu, &window, &config.render)?
                        }
                    };
                }
                _ => {}
            }
        }

        // Handle events from the rocket tracker
        #[cfg(feature = "rocket")]
        while let Some(event) = rocket.poll_events() {
            match event {
                RocketEvent::Seek(to) => {
                    time_source.seek(to);
                    #[cfg(feature = "audio")]
                    if let Some(player) = &mut player {
                        player.seek(to).context("Can't seek audio stream")?;
                    }
                }
                RocketEvent::Pause(state) => {
                    time_source.pause(state);
                    #[cfg(feature = "audio")]
                    if let Some(player) = &mut player {
                        player.pause(state);
                    }
                }
                RocketEvent::SaveTracks => {
                    save_tracks(rocket.get_tracks()).unwrap_or_else(|e| eprintln!("{e:?}"));
                }
            }
        }

        // Get current frame's time and keep the tracker updated
        let time = time_source.get_time();
        #[cfg(feature = "rocket")]
        rocket.set_time(&time);

        // Exit if release
        #[cfg(all(not(debug_assertions), feature = "demo"))]
        if let Some(player) = &player {
            if &time >= player.length() {
                break 'main;
            }
        }

        // Convert to float
        let time = if cfg!(feature = "rocket") {
            rocket.get_row()
        } else {
            time.as_secs_f32()
        };

        renderer.render(time)?; // TODO: handle time and rocket separately
    }

    #[cfg(feature = "rocket")]
    save_tracks(rocket.get_tracks()).unwrap_or_else(|e| eprintln!("{e:?}"));

    Ok(())
}

fn main() -> Result<()> {
    let config_path = std::env::args_os()
        .nth(1)
        .unwrap_or_else(|| OsString::from("config.ron"));

    let config = Config::load(&config_path)?;
    let config::Resolution { width, height } = config.render.resolution;

    if cfg!(target_os = "linux") {
        sdl3::hint::set(VIDEO_DRIVER, "wayland,x11");
        sdl3::hint::set(VIDEO_WAYLAND_MODE_EMULATION, "1");
        sdl3::hint::set(VIDEO_WAYLAND_MODE_SCALING, "stretch");
    }

    let sdl = sdl3::init().context("Can't initialize SDL3")?;
    let video = sdl.video().context("Can't initialize SDL3 video")?;

    let mut window = video
        .window("Mehu", width, height)
        .position_centered()
        .resizable()
        .build()
        .context("Can't create a window")?;

    let gpu = sdl3::gpu::Device::new(ShaderFormat::SpirV, cfg!(debug_assertions))
        .context("Can't initialize a GPU device")?
        .with_window(&window)
        .context("Window cannot be used with GPU device")?;

    let renderer = Renderer::new(&gpu, &window, &config.render)?;

    if cfg!(not(debug_assertions)) {
        if let Err(e) = configure_fullscreen(&sdl, &mut window, &config) {
            sdl3::messagebox::show_simple_message_box(
                sdl3::messagebox::MessageBoxFlag::ERROR,
                "Can't enter fullscreen",
                &format!("{e}"),
                &window,
            )?;
        }
    }

    // Initialize rocket
    #[cfg(feature = "rocket")]
    let rocket = {
        let path = Path::new(sdl3engine::RESOURCE_DIR).join("tracks.bin");
        let tracks = if cfg!(not(debug_assertions)) {
            std::fs::File::open(&path)
                .with_context(|| format!("Can't open {}", path.display()))
                .and_then(|mut file| {
                    bincode::decode_from_std_read(&mut file, bincode::config::standard())
                        .with_context(|| format!("Can't decode {}", path.display()))
                })?
        } else {
            Tracks::default()
        };

        Rocket::new(tracks, config.audio.bpm)
    };

    // Initialize audio
    #[cfg(feature = "audio")]
    let player = 'blk: {
        // TODO: add configuration and move this to module
        use sdl3engine::audio::Player;
        use std::{fs::File, io::BufReader};
        if let Some(name) = &config.audio.name {
            let audio = sdl.audio().context("Can't initialize SDL3 audio")?;
            let path = Path::new(sdl3engine::RESOURCE_DIR).join(name);
            let rdr = match File::open(&path) {
                Ok(file) => BufReader::new(file),
                Err(e) => {
                    eprintln!("audio: Can't open {}: {}", path.display(), e);
                    break 'blk None;
                }
            };
            let mut player = Player::new(rdr, &audio).context("Failed to start audio")?;
            player.pause(false);
            Some(player)
        } else {
            None
        }
    };

    let time_source = TimeSource::new();

    main_loop(MainResources {
        config_path,
        config,
        sdl,
        window,
        gpu,
        renderer,
        time_source,
        #[cfg(feature = "rocket")]
        rocket,
        #[cfg(feature = "audio")]
        player,
    })
}
