#include <stdio.h>
#include <math.h>
#include <SDL2/SDL.h>
#include <SDL2/SDL_ttf.h>
#include <SDL2/SDL_mixer.h>

#define WINDOW_WIDTH 1920
#define WINDOW_HEIGHT 1080
#define TEXT_SCROLL_SPEED 2
#define SCROLL_AMPLITUDE 50
#define SCROLL_FREQUENCY 0.5

void drawCube(SDL_Renderer* renderer, int centerX, int centerY, int size, Uint8 r, Uint8 g, Uint8 b, double rotationX, double rotationY, double rotationZ) {
    // Define the vertices of the cube
    double vertices[8][3] = {
        {-size / 2, -size / 2, -size / 2},
        {size / 2, -size / 2, -size / 2},
        {size / 2, size / 2, -size / 2},
        {-size / 2, size / 2, -size / 2},
        {-size / 2, -size / 2, size / 2},
        {size / 2, -size / 2, size / 2},
        {size / 2, size / 2, size / 2},
        {-size / 2, size / 2, size / 2}
    };

    // Calculate the scaling factors for stretching and shrinking
    double scaleX = 1.0 + sin(rotationX) * 0.5; // Stretching/shrinking factor on X-axis
    double scaleY = 1.0 + sin(rotationY) * 0.5; // Stretching/shrinking factor on Y-axis

    // Define the edges of the cube
    int edges[12][2] = {
        {0, 1},
        {1, 2},
        {2, 3},
        {3, 0},
        {4, 5},
        {5, 6},
        {6, 7},
        {7, 4},
        {0, 4},
        {1, 5},
        {2, 6},
        {3, 7}
    };

    // Rotate the cube around X, Y, and Z axes
    for (int i = 0; i < 8; i++) {
        // Rotate around X-axis
        double newY1 = vertices[i][1] * cos(rotationX) - vertices[i][2] * sin(rotationX);
        double newZ1 = vertices[i][1] * sin(rotationX) + vertices[i][2] * cos(rotationX);
        vertices[i][1] = newY1;
        vertices[i][2] = newZ1;

        // Rotate around Y-axis
        double newX2 = vertices[i][0] * cos(rotationY) + vertices[i][2] * sin(rotationY);
        double newZ2 = -vertices[i][0] * sin(rotationY) + vertices[i][2] * cos(rotationY);
        vertices[i][0] = newX2;
        vertices[i][2] = newZ2;

        // Rotate around Z-axis
        double newX3 = vertices[i][0] * cos(rotationZ) - vertices[i][1] * sin(rotationZ);
        double newY3 = vertices[i][0] * sin(rotationZ) + vertices[i][1] * cos(rotationZ);
        vertices[i][0] = newX3;
        vertices[i][1] = newY3;
    }

    // Apply stretching and shrinking to the vertices
    for (int i = 0; i < 8; i++) {
        vertices[i][0] *= scaleX;
        vertices[i][1] *= scaleY;
    }


    // Project and draw the cube edges
    int projectedPoints[8][2];
    for (int i = 0; i < 8; i++) {
        double distance = 3 * size;
        double scaleFactor = distance / (distance - vertices[i][2]);

        projectedPoints[i][0] = centerX + scaleFactor * vertices[i][0];
        projectedPoints[i][1] = centerY + scaleFactor * vertices[i][1];
    }

    for (int i = 0; i < 12; i++) {
        int p1x = projectedPoints[edges[i][0]][0];
        int p1y = projectedPoints[edges[i][0]][1];
        int p2x = projectedPoints[edges[i][1]][0];
        int p2y = projectedPoints[edges[i][1]][1];
        SDL_RenderDrawLine(renderer, p1x, p1y, p2x, p2y);
    }
}

int main() {
    // Initialize SDL
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) != 0) {
        printf("SDL initialization failed: %s\n", SDL_GetError());
        return 1;
    }

    // Initialize SDL_ttf
    if (TTF_Init() != 0) {
        printf("SDL_ttf initialization failed: %s\n", TTF_GetError());
        SDL_Quit();
        return 1;
    }

    // Create a window
    SDL_Window* window = SDL_CreateWindow("Demoscene Demo", SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED, WINDOW_WIDTH, WINDOW_HEIGHT, SDL_WINDOW_FULLSCREEN_DESKTOP);
    if (!window) {
        printf("Failed to create window: %s\n", SDL_GetError());
        return 1;
    }

    // Create a renderer
    SDL_Renderer* renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
    if (!renderer) {
        printf("Failed to create renderer: %s\n", SDL_GetError());
        SDL_DestroyWindow(window);
        return 1;
    }


    // Initialize SDL_mixer
    if (Mix_OpenAudio(44100, MIX_DEFAULT_FORMAT, 2, 2048) < 0) {
        printf("SDL_mixer initialization failed: %s\n", Mix_GetError());
        SDL_DestroyRenderer(renderer);
        SDL_DestroyWindow(window);
        SDL_Quit();
        return 1;
    }

    // Load the music
    Mix_Music* music = Mix_LoadMUS("audio.wav");
    if (!music) {
        printf("Failed to load music: %s\n", Mix_GetError());
        Mix_CloseAudio();
        SDL_DestroyRenderer(renderer);
        SDL_DestroyWindow(window);
        SDL_Quit();
        return 1;
    }

    // Play the music
    Mix_PlayMusic(music, -1);

    // Set up font and text
    TTF_Font* font = TTF_OpenFont("font.ttf", 52);
    if (!font) {
        printf("Failed to load font: %s\n", TTF_GetError());
        SDL_DestroyRenderer(renderer);
        SDL_DestroyWindow(window);
        SDL_Quit();
        return 1;
    }

    SDL_Color textColor = { 255, 255, 255 };  // White color for the text
    char text[] = "HELLO  THERE  BIRDIE!                THIS  MIGHT  SEEM  LIKE  A  BASIC  SCROLLER  AND  A  ROTATING  3D  CUBE                  NOTHING  TOO  FANCY,  HUH?               THE  FUNNY  THING  IS,  IT'S  ENTIRELY  WRITTE  BY  AI           ALL  HAIL  OUR  FUTURE  TECH  OVERLORDS!                    CODE:  CHATGPT              MUSIC:  HYPERUNKNOWN            ORGANIZER:  ZELEST                         SEE YOU ALL NEXT YEAR! <3                                                                      ";
    int textLength = strlen(text);

    SDL_Rect letterRects[textLength];
    int totalTextWidth = 0;
    SDL_Texture* letterTextures[textLength];  // Declare an array to hold letter textures
    for (int i = 0; i < textLength; i++) {
        SDL_Surface* letterSurface = TTF_RenderGlyph_Solid(font, text[i], textColor);
        letterTextures[i] = SDL_CreateTextureFromSurface(renderer, letterSurface);  // Assign texture to the array
        SDL_FreeSurface(letterSurface);

        int letterWidth, letterHeight;
        SDL_QueryTexture(letterTextures[i], NULL, NULL, &letterWidth, &letterHeight);
        totalTextWidth += letterWidth;

        letterRects[i].w = letterWidth;
        letterRects[i].h = letterHeight;
        letterRects[i].x = WINDOW_WIDTH + totalTextWidth - letterWidth;
        letterRects[i].y = (WINDOW_HEIGHT - letterHeight) / 2;

        SDL_RenderCopy(renderer, letterTextures[i], NULL, &letterRects[i]);
    }

    int cubeSize = 300;
    int cubeCenterX = WINDOW_WIDTH / 2;
    int cubeCenterY = WINDOW_HEIGHT / 2;
    double cubeRotationX = 0;
    double cubeRotationY = 0;
    double cubeRotationZ = 0;

    // Main loop
    int running = 1;
    while (running) {
        SDL_Event event;
        while (SDL_PollEvent(&event)) {
            if (event.type == SDL_QUIT) {
                running = 0;
            } else if (event.type == SDL_KEYDOWN) {
                if (event.key.keysym.sym == SDLK_ESCAPE || event.key.keysym.sym == SDLK_q) {
                    running = 0;
                }
            }
        }

        // Clear the screen
        SDL_SetRenderDrawColor(renderer, 0, 0, 0, 255);
        SDL_RenderClear(renderer);

        // Update cube rotation
        cubeRotationX += 0.01;
        cubeRotationY += 0.02;
        cubeRotationZ += 0.03;

        // Draw the rotating cube
        SDL_SetRenderDrawColor(renderer, 255, 165, 0, 255);  // Orange color for the cube
        drawCube(renderer, cubeCenterX, cubeCenterY, cubeSize, 255, 165, 0, cubeRotationX, cubeRotationY, cubeRotationZ);

        // Update letter positions
        double time = SDL_GetTicks() / 1000.0;
        int offsetX = WINDOW_WIDTH - totalTextWidth;
        for (int i = 0; i < textLength; i++) {
            int waveOffsetY = SCROLL_AMPLITUDE * sin(time + i * SCROLL_FREQUENCY);
            letterRects[i].x -= TEXT_SCROLL_SPEED;
            letterRects[i].y = (WINDOW_HEIGHT - letterRects[i].h) / 2 + waveOffsetY;

            SDL_RenderCopy(renderer, letterTextures[i], NULL, &letterRects[i]);

            // If the letter has scrolled off the screen, reset its position to the right side
            if (letterRects[i].x + letterRects[i].w < 0) {
                letterRects[i].x = WINDOW_WIDTH + totalTextWidth;
            }
        }

        // Update the screen
        SDL_RenderPresent(renderer);
        SDL_Delay(10);
    }

    // Clean up resources
    for (int i = 0; i < textLength; i++) {
        SDL_DestroyTexture(letterTextures[i]);  // Destroy letter textures
    }
    TTF_CloseFont(font);
    SDL_DestroyRenderer(renderer);
    SDL_DestroyWindow(window);
    TTF_Quit();
    SDL_Quit();

    return 0;
}

