; A submission to the 'Vintage Computing Christmas Challenge 2022'
; for the 16/48k ZX Spectrum, assembled using sjasmplus assembler.
; https://logiker.com/Vintage-Computing-Christmas-Challenge-2022
;
; Written by spaceWumpus, December 2022.
;
; 

    DEVICE ZXSPECTRUM48

    CSPECTMAP "build/main.map"

ROM_CL_ALL      = $0DAF             ; Address of ROM Routine to clear screen and set cursor to top-left

ATTRIB_ADDR = $5800                 ; Base address of screen attributes
ATTRIB_ACTIVE_BLOCK_OFFSET = $60    ; Attrib offset of 3 character rows

STAR_ATTRIB = 7                     ; Attributes for visible '*' to reveal star - white on black

ATTRIB_ACTIVE_BLOCK_SIZE = 768 - 1 - 32*7 ; Size of attribute region that we will be modifying.

; Character cell coordinates of of top-centre of star
START_COL = 15
START_ROW = 7

START_ADDR = ATTRIB_ADDR + $20 * (START_ROW-1) + START_COL  ; Address of first character cell to be revealed.


;   From https://worldofspectrum.net/pub/sinclair/games-info/z/ZXSpectrumAssembler.txt
;       "In the Spectrum, the address of the first character after the REM in a REM
;       statement in line 1 of a program is always 23760, unless microdrives are in use.""
;   Actually 23760 is the REM statement itself...

    ;--------------------------------------------------
    ; org $8000 ; Start of non-contended RAM
    org 23761 ; First char after REM in line 1 REM in a BASIC program.

    ; Note that SAVE### directives have 'main' listed as entry point, so this is where we start.
main:
    di                  ; 1 Byte. Disable interrupts, as we are going to trash the stack by abusing the SP   


    call ROM_CL_ALL     ; 3 Bytes. Use ROM call to clear screen and reset cursor to top.


    ; 11 Bytes. Fill (most of) screen with '*'
    ; Simple partially unrolled loop of calls to print character restart routine.
    ; Display has 24 rows.
    ; B=224 prints 21 full rows (32 chars) of '*' (21*32 == 672 == 224*3)   
    ld bc, (224 << 8) + '*' ; Equiv to "ld b,224 : ld c,'*'" but saves a byte
fillScreenWithStars:
    ld a,c
    rst $10              ; Print char from A
    ld a,c
    rst $10              ; Print char from A
    ld a,c
    rst $10              ; Print char from A
    djnz fillScreenWithStars



    ; 10 Bytes. Set all attributes to black on black
    ; Set first byte explicitly, then use an overlapping block copy to copy this to the rest.
setAllAttribsToBlackOnBlack:
    ; ASSERTION ATTRIB_ADDR & $ff == 0
    ; ASSERTION L==0 ; HL is $4000 after call ROM_CL_ALL, and not touched by RST $10
    ld h, ATTRIB_ADDR >> 8 ; Equiv to "ld hl, ATTRIB_ADDR" but saves a byte.
    ; ASSERTION E==0 ; DE is $0000 after call ROM_CL_ALL, and not touched by RST $10
    ld (hl), e; Black Ink on Black Paper ; Equiv to "ld (hl), 0" but saves a byte.   
    ld d,h : inc e ; Equiv to "ld de, ATTRIB_ADDR+1" but saves a byte:
    ld bc,767
    ldir


    ; 29 Bytes. Render top-right quarter of star.
    ; This is done by a series of 5 iterations through code at 'startDrawing',
    ; each of which draws an L shape. These L's are nested to form the top-right of the star shape.
    ; We abuse the SP to store the address offset to apply to get from the last character of one L to the first character of the next.
    ld sp, 6 *-32 -3  ; Attrib coords offset 5 rows up, 3 cols left.  Let's hope nobody needs the stack. :-)
startDrawing:
    ld hl, START_ADDR
    ld de,$20   ; Attrib coords offset for one row down
    ld c,5      ; We want 5 nested 'L' segments

renderSegmentLStart: ; Render a vertical line of 5 '*'s:
    ld b,5
renderSegmentVertL: 
    add hl, de
    ld (hl), STAR_ATTRIB
    djnz renderSegmentVertL

    ld b,4
renderSegmentHorizL: ; Randed horizontal line of 4 '*'s to complete the L
horizStep:
    inc l   ; // INC L  is $2c, DEC L is $2d
    ld (hl), STAR_ATTRIB
    djnz renderSegmentHorizL

    add hl,sp    ; Offset ready for next L

    dec c
    jr nz, renderSegmentLStart  ; Draw 5 nested L's


    ; 17 Bytes. Modify the above code to draw top-left instead of top right, and jump back to it if this is the first time we arrived here.
    ; On the second arrival, here we have already replaced 'ld b,0' below with 'ld b,1', which will prevent the jump being taken again.
fakeLoopCount+1:
    ld b,0 ; On first pass, this will be 0, so djnz will jump. In the second pass this has been overwritten with ld b,1, and the djnz will not jump.
    ; We modify the main draw code above to make the horizontal part of the L extend to the left instead of the right.
    ld a, $2d
    ld (horizStep), a ; Replace 'INC L' at horizStep with 'DEC L', so next time around it draws a reversed L
    ld sp, 6 *-32 +3 ; Also correct the offset for reversed L. Attrib coords offset 5 rows up, 3 cols left.
    ; Then we change 'ld b,0' above to 'ld b,1' so that on the next pass the jump will not be taken.
    xor a
    inc a
    ld (fakeLoopCount),a ; Change 'ld b,0' above to 'ld b,1'
    ; Finally, jump back if this was the first time we have arrived here.
    djnz startDrawing


copyBottom:
    ; 16 Bytes.  Create bottom half as reverse copy of top half.
    ld l, (ATTRIB_ADDR + ATTRIB_ACTIVE_BLOCK_OFFSET) & $ff ; 1 byte less than 'ld hl, ATTRIB_ADDR + ATTRIB_ACTIVE_BLOCK_OFFSET'
    ; ASSERTION HL == ATTRIB_ADDR + ATTRIB_ACTIVE_BLOCK_OFFSET
    ld de, ATTRIB_ADDR + ATTRIB_ACTIVE_BLOCK_OFFSET + ATTRIB_ACTIVE_BLOCK_SIZE -1 ; Address of last attrib byte in copy region
    ld bc, ATTRIB_ACTIVE_BLOCK_SIZE/2
fillLoop:
    ldi
    dec de
    dec de 
    ld a, b
    or c
    jr nz, fillLoop


    halt    ; 1 Byte. Infinite loop. Since we have disabled RUPTs we can save a byte by using halt instead of 'JR $'

codeEnd

; In Debug Console:
;   -eval codeEnd - main
; reports:
;   88, 58h, 1011000b
; So, 88 bytes of code.


; Notional top of stack - though stack isn't used in this code.
stack_top:

    ; Deployment, using 'main' as entrypoint, and generic BASIC loader.
    SAVESNA "main.sna", main
    SAVETAP "main.tap", main

    ; Save machine code out as a code file:
    ; This file contains 2 bytes loading address, then 88 bytes of code:
    SAVETAP "code.tap", CODE, "code", main, $-main

    ; This can be loaded into a Spectrum using:
    ;   Spectrum BASIC:         In Fuse emulator, press:
    ;   LOAD "" CODE            J, Ctrl-P, Ctrl-P, Shift-Ctrl, I     
    ;   RANDOMIZE USR 32768     T, Shift-Ctrl, L, 32768
    ;
    ; To create a BASIC loader, create a BASIC program containing the above two lines as lines 1 and 2,
    ; then use:
    ; SAVE "x" LINE 1           S, Ctrl-P, X, Ctrl-P, Shift-Ctrl, Ctrl-3, 1
    ; Result is a standard 19 byte header, then a 29 byte BASIC block (including flag byte and checksum byte),
    ;  so 27 bytes of actual BASIC.
    ; 
    ; Then use e.g. Tapir to assemble the loader followed by code.tap.
    ; 'code.tap' contains standard 19 byte header (which includes the loading address), then a 90 byte code block (including flag byte and checksum byte),
    ;  so 88 bytes of actual code.
    ;
    ; To create REM embedded version (compo submission):
    ; 1. Assemble using 'org 23761' rather than 'org $8000'
    ;
    ; 2. In Spectrum emulator, create the following BASIC program (88 digits in first line):
    ;   1 REM 1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678
    ;   2 RANDOMIZE USR 23761
    ;
    ; 3. Load the assembled code into the REM:
    ;   LOAD "" CODE            J, Ctrl-P, Ctrl-P, Shift-Ctrl, I
    ;
    ; 4. Save the BASIC program with the embedded assembly:
    ;   SAVE "x" LINE 1         S, Ctrl-P, X, Ctrl-P, Shift-Ctrl, Ctrl-3, 1
    ;
    ; Result is standard 19 byte header, followed by 114 bytes of BASIC containing embedded assembly (including flag byte and checksum byte),
    ;  so 112 bytes of combined BASIC loader code + assembly.
