#include "Beacon.as"
#include "Portal.as"

enum TileType 
{
    TL_None = 0,
    TL_Empty = 1,
    TL_Tree1 = 2,
	TL_Tree2 = 3,
    TL_Wall = 4,
    TL_Spawn = 5,
    TL_Enemy = 6,
    TL_Beacon = 7,
    TL_Portal = 8,
    TL_Crystal = 9,
    TL_Well = 10,
}

enum TileKind 
{
    TK_Single = 0,
    TK_Straight = 1,
    TK_Corner = 2,
    TK_T = 3,
    TK_Cross = 4,
    TK_End = 5
}

enum EnemyType
{
    ET_Blight = 0
}

//used by enemy and player to determine walking direction, and when to turn.
//They both use the playfield, so there...
enum Direction 
{
    Dir_Undefined = -1,
    Dir_N = 0,
    Dir_E = 1,
    Dir_S = 2,
    Dir_W = 3
}


class Tile 
{
    Tile() {
        type = TL_None;
    }

    Tile(TileType t) {
        type = t;
    }

    Tile(EnemyType et) {
        type = TL_Enemy;
        etype = et;
        occupied = true;
    }

    Transform xForm;
    TileType type;
    TileKind kind = TK_Single;
    
    bool occupied = false;
    EnemyType etype;

    //Tried to solve this with inherritance,
    //But for some reason, angelscript doesn't like casting Tile@ from arrays to specialized tiles?! No time for further research..
    Beacon@ beacon;
}

class Location 
{
    Location() { }
    Location(pInt _x, pInt _y) {
        x = _x;
        y = _y;
    }

    bool opEquals(Location@ loc) {
        return x == loc.x && y == loc.y;
    }
    
    pInt x = 0;
    pInt y = 0;
}


class EnemyPosition
{
    EnemyPosition() {}
    EnemyPosition(EnemyType _type, Location  _loc) { type = _type; loc = _loc; }
    Location loc;
    
    bool opEquals(EnemyPosition pos) {
        return loc.x == pos.loc.x && loc.y == pos.loc.y && type == pos.type;
    }
    EnemyType type;
}



class PlayField
{
    private float mTileSize = 2;
    private float mFieldWidth = 0;
    private float mFieldHeight = 0; 

    private Scene@ mScene;
    private ObjectFactory@ mFactory;

    private String@ mAssetRoot;
    private String@ mTheme;

    private Object@ mEmptyTile;
    private Object@ mTrees;
    private Object@ mWalls;
    private Object@ mShrooms;
    private Object@ mWells;
    private Object@ mChristals;
    private Object@ mPebbles;

    private Location mPlayerLocation;

    private Transform mSpawn;
    private Array<Array<Tile>> mTiles;

    private Array<EnemyPosition> mEnemies;
    private Array<Beacon@> mBeacons;
    private Portal@ mPortal;


    PlayField(Scene@ scene, ObjectFactory@ of, String@ assetRoot)
    {
        @mScene = scene;
        @mFactory = of;
        @mAssetRoot = assetRoot;
    }

    pFloat get_width() { return mFieldWidth; }
    pFloat get_height() { return mFieldHeight; }
    Transform get_center() { return Transform(Quaternion(), Vector3(mFieldWidth * 0.5, mFieldHeight * 0.5, 0));}
    pFloat get_tileSize() { return mTileSize; }
    Array<EnemyPosition>@ get_enemies() { return mEnemies; }
    Array<Beacon@>@ get_beacons() { return mBeacons; }

    Portal@ get_portal() { return mPortal; }

    Location get_playerLocation() { return mPlayerLocation; }

    Transform getSpawnLocation(Location &out loc) {
        loc = mPlayerLocation;
        return mSpawn;
    }
   

    Transform getWorldLocation(Location loc) {
        if (!isValidLocation(loc.x, loc.y)) {
            echo ("INVALID TILE LOCATION!!!");
            return mTiles[0][0].xForm;
        }

        return mTiles[loc.y][loc.x].xForm;
    }

    Beacon@ getBeacon(pUInt x, pUInt y) {
        if (!isValidLocation(x, y)) return null;
        auto tile = mTiles[y][x];
        if (tile.type == TL_Beacon) {
            return tile.beacon;
        }
        return null;
    }

    bool updatePlayerLocation(pUInt x, pUInt y,  Transform &out xform) {
        Location loc(x, y);
        if (!isWalkableTile(loc)) return false;
        mPlayerLocation = loc;
        xform = mTiles[loc.y][loc.x].xForm;
        return true;
    }

    void updateEnemyPos(Location prev, Location new) {
        if (!isValidLocation(prev.x, prev.y) || !isValidLocation(new.x, new.y)) {
            echo ("Algorithm is broken somehow.. Enemy just left the playfield ?!");
            return;
        } 

        mTiles[prev.y][prev.x].occupied = false;
        mTiles[new.y][new.x].occupied = true;
    }

    bool isWalkableTile(Location loc, bool ignoreOccupied = false) {
        if (!isValidLocation(loc.x, loc.y)) return false;
        auto tile = mTiles[loc.y][loc.x];
     
        bool occupied = ignoreOccupied ? true : tile.occupied == false;
        return  tile.type != TL_Tree1 &&
				tile.type != TL_Tree2 &&
                tile.type != TL_Wall &&
                tile.type != TL_Beacon &&
                tile.type != TL_Crystal &&
                tile.type != TL_Well &&
                tile.type != TL_None && occupied;
    }

    private bool isValidLocation(pInt x, pInt y) {
        if (x < 0 || y < 0) return false;
        if (y >= pInt(mTiles.length())) return false;
        if (x >= pInt(mTiles[y].length())) return false;

        return true;
    }

    bool load(String filename) {
        File f;
        if (f.open(filename, "r") == -1) return false;
        String@ lvl = f.readString(f.getSize());
        
        if (!parseLevel(lvl)) return false;

        //Half tile offset, so the field actually starts at 0.0 (as tile point of reference is the center of the tile)
        pFloat halfTile = 0.5 * mTileSize;
        pFloat y = halfTile;
        pFloat x = halfTile; 

        @mEmptyTile = Object();
        @mTrees = Object();
        @mWalls = Object();
        @mShrooms = Object();
        @mWells = Object();
        @mChristals = Object();
        @mPebbles = Object();

        //We render the lines in reverse, so it matches the layout of the text file ;-)
        for (pUInt i = mTiles.length() - 1; i != pUInt(-1); --i)  {
            auto line = mTiles[i];
            for (pUInt j = 0; j < line.length(); ++j) {
                auto tile = line[j];
                tile.xForm.translate(x, y, 0);
                tile.xForm.setScale(mTileSize);
                switch (tile.type) {
                    case TL_Spawn:
                        mPlayerLocation = Location(j, i);
                        x = initSpawn(x, y);
                        break;
                    case TL_Empty: //Empty tile
                        x = placeEmptyTile(x, y);
                        break;
                    case TL_Tree1: //Tree
                        x = placeTreeTile(tile.type, x, y);
                        break;
                    case TL_Tree2: //Tree
                        x = placeTreeTile(tile.type, x, y);
                        break;
                    case TL_Wall:
                        x = placeWallTile(tile, x, y);
                        break;
                    case TL_Enemy:
                        x = setEnemyLocation(Location(j, i), tile.etype, x, y);
                        break;
                    case TL_Beacon:
                        x = placeBeacon(tile, x, y);
                        break;
                    case TL_Portal:
                        x = placePortal( x, y, Location(j, i));
                        break;
                    case TL_Well:
                        x = placeWell(x, y);
                        break;
                    case TL_Crystal:
                        x = placeCrystal(x, y);
                        break;
                    case TL_None: 
                        x += mTileSize;
                    
                }
            }

            mFieldHeight = max(mFieldHeight, y  + halfTile);         
            mFieldWidth = max(mFieldWidth, x - halfTile);

            x = halfTile;
            y += mTileSize;
        }

        if (mBeacons.length() == 0 ) {
            echo("Error loading level: Level doesn't have beacons and cannot be completed");
            return false;
        }

        if (mPortal is null) {
            echo("Error loading level: Level doesn't a portal and cannot be completed");
            return false;
        }

        mScene.add(mEmptyTile);
        mScene.add(mTrees);
        mScene.add(mWalls);
        mScene.add(mShrooms);

        mScene.add(mWells);
        mScene.add(mChristals);
        mScene.add(mPebbles);

        return true;
    }


    private bool parseLevel(String@ lvl) {
        mTiles.add(Array<Tile>());
        pUInt line = 0;

        bool foundSpawn = false; //To check if we have more than one spawn, which is currently not allowed
        bool foundPortal = false;

        //read theme
        pUInt l = 0;
        l = lvl.findFirst('\r');

        if (l == lvl.length() )
            l == lvl.findFirst('\n');
        if (l == lvl.length() )
            echo("INVALID LEVEL FORMAT: First line must be name of theme folder you want to use");
        
        @mTheme = lvl.subString(0, l);
        echo ("Loading theme: " + mTheme);
        if (lvl[l] == '\n') ++l;
        
        for (pUInt i = l; i < lvl.length(); ++i)
        {
    	    switch (lvl[i]) {
                case ' ': //NO tile
                    mTiles[line].add(Tile(TL_None));
                    break;
                case '0': //Empty tile
                    mTiles[line].add(Tile(TL_Empty));
                    break;
                case 'T': //Tree ?
                    mTiles[line].add(Tile(TL_Tree1));
                    break;
                case 't': //Tree 2 ?
                    mTiles[line].add(Tile(TL_Tree2));
                    break;					
                case 'W': //Wall!
                    mTiles[line].add(Tile(TL_Wall));
                    break;
                case 'E': //For now: Enemy (We only have one for now: Blight!)
                    mTiles[line].add(Tile(ET_Blight));
                    break;
                case 'B':
                    mTiles[line].add(Tile(TL_Beacon));
                    break;
                case 'C':
                    mTiles[line].add(Tile(TL_Crystal));
                    break;
                case 'w': 
                    mTiles[line].add(Tile(TL_Well));
                    break;
                case 'P': 
                    if (foundPortal) {
                        echo ("INVALID LEVEL FORMAT: Found more than one Portal!! Only one portal is allowed!");
                        return false;
                    }
                    foundPortal = true;
                    mTiles[line].add(Tile(TL_Portal));
                    break;
                case 'S': //Spawn point
                    if (foundSpawn) {
                        echo ("INVALID LEVEL FORMAT: Found more than one spawn point! Only one spawnpoint is currently supported!");
                        return false;
                    }
                    foundSpawn = true;
                    mTiles[line].add(Tile(TL_Spawn));
                    break;
                case '\r': 
                    if (lvl[i+1] == '\n') ++i;
                case '\n': //new line
                    mTiles.add(Array<Tile>());
                    ++line;
                    break;
                default:
                  echo("INVALID LEVEL FORMAT: Unexpected character found: " + String(lvl[i]));
                  return false;
            }
        }

        if (!foundSpawn) {
            echo ("INVALID LEVEL FORMAT: Couldn't find a spawn point! Use 'S' to place spawnpoint" );
            return false;
        }


        specializeTiles();

        return true;
    }

    void specializeTiles() {
        for (pUInt i = mTiles.length() - 1; i != pUInt(-1); --i)  {
            auto line = mTiles[i];
            for (pUInt j = 0; j < line.length(); ++j) {
                auto tile = line[j];
                setTileKind(tile, j, i);
            }
        } 
    }

    void setTileKind(Tile@ tile, pUInt x, pUInt y)
    {
        auto type = tile.type;
        pInt count = 0;

        Tile@ t1;    
        Tile@ t2;
        Tile@ t3;
        Tile@ t4; 

        Quaternion left(Vector3(0,0,1), 0.5 * PI);
        Quaternion down(Vector3(0,0,1), 1 * PI);
        Quaternion right(Vector3(0,0,1), 1.5 * PI);


        if (isValidLocation(x, y-1))  { if (mTiles[y-1][x].type == type)  { @t1 = mTiles[y-1][x]; ++count; } }
        if (isValidLocation(x+1, y))  { if (mTiles[y][x+1].type == type)  { @t2 = mTiles[y][x+1]; ++count; } }
        if (isValidLocation(x, y+1))  { if (mTiles[y+1][x].type == type)  { @t3 = mTiles[y+1][x]; ++count; } }
        if (isValidLocation(x-1, y))  { if (mTiles[y][x-1].type == type)  { @t4 = mTiles[y][x-1]; ++count; } }

        switch (count) {
            case 4:
                tile.kind = TK_Cross;
                return;
            case 3:
                if (t1 is null) { tile.xForm.quaternion = down; }
                else if (t2 is null) { tile.xForm.quaternion = left; }
                else if (t3 is null) { /*do nothing */}
                else tile.xForm.quaternion = right; //only one remains!
				tile.kind = TK_T; 
                return;
            case 2:
                if (t1 !is null) {
                    if (t2 !is null) { tile.kind = TK_Corner; return; }
                    if (t3 !is null) { tile.kind = TK_Straight; return; }
                    if (t4 !is null) { tile.xForm.quaternion = left; tile.kind = TK_Corner; return; }
                }

                if (t2 !is null) {
                    if (t3 !is null)  { tile.xForm.quaternion = right; tile.kind = TK_Corner; return; }
                    if (t4 !is null) { tile.xForm.quaternion = right; tile.kind = TK_Straight; return; }
                }
                //only one case left!
                tile.xForm.quaternion = down;
                tile.kind = TK_Corner;
                return;
            case 1:
                if (t1 !is null) {/*do nothing */ }
                else if (t2 !is null) {tile.xForm.quaternion = right;}
                else if (t3 !is null) {tile.xForm.quaternion = down;}
                else  tile.xForm.quaternion = left; //Only one remains!...

                tile.kind = TK_End;
                return;
            case 0:
                tile.kind = TK_Single;
                return;
        }
    }
   

    private pFloat initSpawn(pFloat x, pFloat y) {
        mSpawn.setPosition(x, y, 0);
        return placeEmptyTile(x, y);
    }

   
    private pFloat placeEmptyTile(pFloat x, pFloat y, bool noFlair = false)
    {
        Object@ obj;
        @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/empty.fbx");

        obj.setScale(mTileSize);
        obj.setPosition(x, y, 0);
        mEmptyTile.combine(obj);
        
        if (!noFlair)
            addFlair(x, y);

        return x + mTileSize;
    }

    void addFlair(pFloat x, pFloat y) {
        //for now, only mushroom..
        if (mTheme == "Forest") {
            if (rand() % 15 == 0)
              placeMushroom(x, y);
			if (rand() % 15 == 0)
              placeFireflies(x, y);  
        } else if (mTheme == "Dungeon") {
            if (rand() % 3 == 0)
              placePebbles(x, y);
        }
    }
	
    void placeFireflies(pFloat x, pFloat y) {
		pFloat defheight = 0.5;
		pFloat defspeed = 0.5;
		pFloat defradius = 0.5;
		pFloat c = defradius*0.552;
		pUInt n = rand() % 5 + 1; // at least one, at most 5
		// random height
		defheight = (rand() % 1.0) + defheight;	

		echo(String("Spawning fireflies: ", n));

		for (pUInt i = 0; i < n; ++i) {		
			pFloat radius = (rand() % 0.1) + defradius;
			pFloat speed = (rand() % 2.0) + defspeed;
			// random control points
			c = rand() % radius;
			
			Object@ obj = mFactory.createPoster(mAssetRoot + "/3D/Tiles/" + mTheme + "/firefly_glow.png", MT_OPAQUE, DM_ADDITIVE);
			Surface@ surf = obj.model.getSurface(0);
			
			auto pointLight = PointLight();
			pointLight.color = Color(30,30,30,255);
		
			obj.link.add(pointLight, LM_OVERRIDE, pLinkTransformMask(LT_TRANSLATION | LT_ROTATION));			
			
			mScene.add(pointLight);
			
			obj.model.setColor(WHITE);

			surf.material.mode = DM_ADDITIVE;
			surf.material.color = WHITE;
			surf.useSpriteHack = true;
			
			obj.setScale(0.002);
			@obj.animator = Animator();
			
			ObjectAnimation@ oa = ObjectAnimation("Fly", speed);

			auto kind = rand() % 3;
			switch (kind) {
			case 0:
				//circle
				oa.addFrames(Vector3(x, y + radius, defheight), Vector3(x + c, y + radius, defheight), Vector3(x + radius, y + c, defheight), Vector3(x + radius, y, defheight));
				oa.addFrames(Vector3(x + radius, y-c, defheight), Vector3(x+c, y-radius, defheight), Vector3(x, y-radius, defheight));
				oa.addFrames(Vector3(x-c, y-radius, defheight), Vector3(x-radius, y-c, defheight), Vector3(x-radius, y, defheight));
				oa.addFrames(Vector3(x-radius, y+c, defheight), Vector3(x-c, y+radius, defheight), Vector3(x, y+radius, defheight));
				break;
			case 1:
				// line
				oa.addFrames(Vector3(x, y + radius, defheight), Vector3(x + c, y + radius, defheight), Vector3(x + radius, y + c, defheight), Vector3(x, y + radius, defheight));
				break;
			case 2:
				//some thing else :p
				oa.addFrames(Vector3(x, y + radius, defheight), Vector3(x + c, y + radius, defheight), Vector3(x + radius, y + c, defheight), Vector3(x + radius, y, defheight));
				oa.addFrames(Vector3(x + radius, y-c, defheight), Vector3(x+c, y-radius, defheight), Vector3(x, y + radius, defheight));
			}
			
			oa.build(IM_CURVED);
			obj.animator.play(oa, true, (rand() % 2 == 0) ? -1 : 1);
			
			mScene.add(obj);
		}
    }

    void placeMushroom(pFloat x, pFloat y) {
        int r = rand() % 2 + 1;
        Object@ obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/mushroom_" + String(r) + ".fbx");
        obj.model.setDrawMode(DM_EMISSIVE);

        r = rand() % 100 + 1;
        obj.rotate(0,0,1, PI * pFloat(100)/pFloat(r));

        r = rand() % 10 + 1;
        pFloat offset = 0.3 + pFloat(r)/pFloat(11) * 0.7; //Let's keep it off to the side
        obj.setScale(mTileSize);
        obj.translate(x, y, 0);
        obj.relTranslate(offset, 0, 0);
        
        auto pointLight = PointLight(0.8 * mTileSize);
        pointLight.color = Color(30, 100, 255, 255);
        pointLight.position = obj.position;
        pointLight.translate(0,0,0.3);

        mScene.add(pointLight);

        mShrooms.combine(obj);
    }

    
    void placePebbles(pFloat x, pFloat y) {
        Object@ obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/pebbles.fbx");
        obj.model.setDrawMode(DM_EMISSIVE);

        int r = rand() % 100 + 1;
        obj.rotate(0,0,1, PI * pFloat(100)/pFloat(r));

        r = rand() % 10 + 1;
        pFloat offset = 0.3 + pFloat(r)/pFloat(11) * 0.7; //Let's keep it off to the side
        obj.setScale(mTileSize);
        obj.translate(x, y, 0);
        obj.relTranslate(offset, 0, 0);

        mPebbles.combine(obj);
    }


    private pFloat placeTreeTile(TileType type, pFloat x, pFloat y)
    {
		String tree = mAssetRoot + "/3D/Tiles/"+ mTheme + "/dead_tree.fbx";
		if (rand() % 2 == 0) // if even random number, choose a different tree
			tree = mAssetRoot + "/3D/Tiles/"+ mTheme + "/evil_tree.fbx";
		
		switch (type)
		{
		case TL_Tree2:
			tree = mAssetRoot + "/3D/Tiles/"+ mTheme + "/tree1.fbx";
			break;
		}
	
        Object@ obj = mFactory.load(tree);
        int r = rand() % 100 + 1;
        obj.rotate(0,0,1, PI * pFloat(100)/pFloat(r));
        obj.setScale(mTileSize);
        obj.setPosition(x, y, 0);

        obj.model.getSurface(0).material.type = MT_TRANSPARENT;
        mTrees.combine(obj);

        return placeEmptyTile(x, y, true);
    }

    private pFloat placeWallTile(Tile@ tile, pFloat x, pFloat y)
    {
        Object@ obj;
        switch (tile.kind) {
            case TK_Single: 
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_single.fbx");
                break;
            case TK_Straight:
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_straight.fbx");
                break;
            case TK_Corner:
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_corner.fbx");
                break;
            case TK_T:
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_T.fbx");
                break;
            case TK_Cross:
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_cross.fbx");
                break;
            case TK_End:
                @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/wall_end.fbx");
                break;
        }


        // Quaternion left(Vector3(0,0,1), 0.5 * PI);
        // tile.xForm.quaternion = left;

        obj.xForm = tile.xForm;
        
        
        //mScene.add(obj);
        mWalls.combine(obj);
        return placeEmptyTile(x, y, true);
    }

    private pFloat placeWell(pFloat x, pFloat y) {
        Object@ obj;
        @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/well.fbx");

        obj.setScale(mTileSize);
        obj.setPosition(x, y, 0);

        auto@ light = PointLight(mTileSize * 0.55);
        light.translate(x,y,mTileSize * 0.2);
        light.color = ORANGE;

        mScene.add(light);
        mWells.combine(obj);

        return placeEmptyTile(x, y, true);
    }

    private pFloat placeCrystal(pFloat x, pFloat y) {
        Object@ obj;
        @obj = mFactory.load(mAssetRoot + "/3D/Tiles/"+ mTheme + "/crystal_stone.fbx");

        obj.setScale(mTileSize);
        obj.setPosition(x, y, 0);

        auto@ light = PointLight(mTileSize * 2);
        light.translate(x,y,mTileSize * 0.5);
        light.color = BLUE;//Color(0, 30, 150, 255);

        mScene.add(light);
        mChristals.combine(obj);

        return placeEmptyTile(x, y, true);
    }

    private pFloat placeBeacon(Tile@ tile, pFloat x, pFloat y) 
    {
        @tile.beacon = Beacon(mTheme, x, y, mScene, mTileSize);
        mBeacons.add(tile.beacon);

        return x + mTileSize;
    }

    private pFloat placePortal(pFloat x, pFloat y, Location loc) 
    {

        @mPortal = Portal(loc, x, y, mScene, mTileSize);
        return placeEmptyTile(x, y, true);
    }

    private  pFloat setEnemyLocation( Location loc, EnemyType type, pFloat x, pFloat y) 
    {
        mEnemies.add(EnemyPosition(type, loc)); 
        return placeEmptyTile(x,y);
    }

    bool checkLineOfSight(Location loc, Location target)
    {
        if (loc.x != target.x && loc.y != target.y) return false;
        if (loc == target) return true;
        
        if (loc.x == target.x) {
            pInt step = (loc.y > target.y) ? -1 : 1;
            for (pInt y = loc.y; y != target.y; y += step) {
                if (!isWalkableTile(Location(loc.x, y), true)) return false;
            }
        } else  {
            pInt step = (loc.x > target.x) ? -1 : 1;
            for (pInt x = loc.x; x != target.x; x += step) {
                if (!isWalkableTile(Location(x, loc.y), true)) return false;
            }
        }
        return true;
    }

    void reset() {
        //faster than reload/rebuild
        for (pUInt i = 0; i < mTiles.length(); ++i)  {
            auto line = mTiles[i];
            for (pUInt j = 0; j < line.length(); ++j) {
                auto@ tile = line[j];
                switch (tile.type) {
                    case TL_Enemy: 
                        tile.occupied = true;
                        break;
                    case TL_Beacon:
                        tile.beacon.reset();
                        break;
                    case TL_Portal: 
                        mPortal.reset();
                        tile.occupied = false;
                        break;
                    case TL_Spawn:
                        mPlayerLocation = Location(j, i);
                    default:
                        tile.occupied = false; //just to make sure...
                }
            }
        }
        //reset the portal to be sure...
        if (mPortal !is null) mPortal.reset();
    }
};

