#include "StartupScreen.as";
#include "Menu/MainMenu.as";
#include "Menu/GameMenu.as";
#include "Menu/GameOverScreen.as";
#include "Menu/GameEndScreen.as";
#include "Menu/LevelSelection.as";
#include "LevelInfo.as"
#include "Audio.as"
#include "Fade.as"
#include "../Input/InputMapper.as"

#include "../Misc/OffsetCam.as"
#include "Level/PlayField.as"
#include "Actors/Player.as"
#include "Actors/Enemy.as"



#include "PathFinder.as"

const int RandomSeed = 6;

class Game
{
	Control@ control;
	DebugInfo@ debugInfo;
	
	Audio@ audio;
	ResourcePool@ resourcePool;
	ObjectFactory@ objectFactory;
	StartupScreen@ startupScreen;
	MainMenu@ mainMenu;

	String assetRoot = "../../Assets/";
	Window@ mainWindow;
	
	Display@ activeDisplay;
	private Renderer@ mRenderer;
	private GameMenu@ mGameMenu;
	private GameEndScreen@ mGameEndScreen;
	private GameOverScreen@ mGameOverScreen;
	private Fade@ mTransitionFade;
	LevelInfo@ levelInfo;


	pFloat animationSpeed = 2;
	
	private bool mPaused;
	private bool mPlaying = false;

	private PlayField@ mPlayField;
	private Scene@ mLevelScene;
	private OffsetCam@ mGameCamera;
	private Viewport@ mViewport;
	private Player@ mPlayer;
	private InputMapper@ mInputMapper;
	private pUInt mCurrentLevel = 1;

	Array<Enemy@> mEnemies;
	pUInt beaconsActivated = 0;


	pUInt mViewportIdx = pUInt(-1);
	
	//Camera layers
	private pUInt HUD_LAYER = 0;
	private pUInt GAME_LAYER = 1;


	//get this from the display instead
	int gameWidth = 1280;
	int gameHeight = 720;
	bool fullscreen = false;
	
	Game(Control@ _control)
	{
		@control = _control;
		@resourcePool = ResourcePool();
		@objectFactory = ObjectFactory(resourcePool);
		@audio = Audio(resourcePool, _control, assetRoot);
			
		@mRenderer = OpenGL4Renderer();
		control.setRenderer(mRenderer);
	}

	~Game() {
		echo ("Unloading GAME");
		releaseMenus();
	}

	void releaseMenus() {
    mGameMenu.release();
		mGameOverScreen.release();
		mGameEndScreen.release();
		mainMenu.release();
    }
	
	void Start()
	{
		@mainWindow = mRenderer.createWindow(gameWidth, gameHeight, fullscreen);
		mainWindow.open();
		mainWindow.VSync = true;
		
		echo ("Width: " + String(mainWindow.width) + " . " + String(mainWindow.height));
		@mViewport = Viewport(0, 0, mainWindow.width, mainWindow.height); 
		mViewport.enableClear();
		
		@activeDisplay = mRenderer.getDisplay(mainWindow.getDisplayIndex());


		@startupScreen = StartupScreen(control, mainWindow, resourcePool, assetRoot);
		@levelInfo = LevelInfo();


		
		@startupScreen.StartCB = Action(this.ShowMenu);
		startupScreen.Show();
		
		@mainMenu = MainMenu(control, mainWindow, resourcePool, assetRoot);
		@mGameMenu = GameMenu(control, mainWindow, resourcePool, assetRoot);
		@mGameOverScreen = GameOverScreen(control, mainWindow, resourcePool, assetRoot);
		@mGameEndScreen = GameEndScreen(control, mainWindow, resourcePool, assetRoot);

		@mTransitionFade = Fade(control, mainWindow, resourcePool, assetRoot);
		control.Input.key(PK_r) += KeyCallback(this.onRestart);

		mainWindow.Input.windowResize() += WindowResizeCallback(this.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(mainMenu.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(mGameMenu.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(startupScreen.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(mGameOverScreen.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(mTransitionFade.onWindowResize);
		mainWindow.Input.windowResize() += WindowResizeCallback(mGameEndScreen.onWindowResize);
	}

	private void onWindowResize(Window@ w, pUInt width, pUInt height) {
		mViewport.width = width;
		mViewport.height = height;

		//new projection
		Projection p(60, mainWindow.width / float(mainWindow.height), 0.5, 100);
		if (mGameCamera !is null) mGameCamera.projection = p;
	}


	void pause() {
		mPaused = true;
			if (mPlaying) {
				if (mInputMapper !is null)
					mInputMapper.disable();
			} 
		mTransitionFade.pause();

		echo("PAUSED!");
	}

	void resume() 
	{
		mPaused = false;
		if (mPlaying) {
			if (mInputMapper !is null)
				mInputMapper.enable();
		}
		mTransitionFade.resume();
		echo("UNPAUSED!");
	}

	void quit() {
		unloadLevel();
		releaseMenus();
		control.quit();
	}

	private void onRestart(pKeyCode pk, pByte b) {
		restartLevel();
	}

	void ShowMenu() { 
		unloadLevel();
		startupScreen.Hide();
		mainMenu.Show();
	}

	void restartLevel() {
		srand(RandomSeed);
		mPlayField.reset();
		mTransitionFade.cancel();
		
		for (pUInt i = 0; i < mEnemies.length(); ++i) {
			mEnemies[i].removeFromScene(); //Garbage collector is too slow... 
		}
		mEnemies.clear();
		control.script.CollectGarbage();

		beaconsActivated = 0;

		mPlayer.reset();
		//reset the game camera
		mGameCamera.center = mPlayField.center;
		setupGame();

		if (!mPlaying)
			startPlaying();
		mGameMenu.enable();

	}

	void loadLevel(String@ filename) 
	{
		//Let's set our random seed so that every run for the level will always be deterministic
		srand(RandomSeed);

		unloadLevel();

		@mLevelScene = Scene(OctreePartition());
		control.addScene(mLevelScene);

		//Default light for now..
		auto dl = DirectionalLight();
		dl.rotate(0,0,1, 0.55 * PI);
		dl.relRotate(1,0,0, -0.40 * PI);

		dl.color = Color(64,64,64,255);
		mLevelScene.add(dl);
		dl.setShadowMap(TX_2048,TX_2048);
		
		@mPlayField = PlayField(mLevelScene, objectFactory, assetRoot);
		if (!mPlayField.load(assetRoot + filename)) return;

		//Register beacons
		auto beacons = mPlayField.beacons;
		for (pUInt i = 0; i < beacons.length(); ++i) {
			auto beacon = beacons[i];
			@beacon.activated = Action(this.beaconActivated);
		}

		setupGame();
			
		@mPlayer = Player(mLevelScene,  mPlayField);
		@mPlayer.tick = Action(this.tick);
		@mPlayer.stepped = Action(this.playerStepped);
		@mInputMapper = InputMapper(control, mPlayer);

		Camera@ cam = mGameCamera;
		cam.setLUTTexture(resourcePool.getTexture(GAME.assetRoot + "3D/party_lut_eyes.png"));
		
		mGameCamera.setDistance(6 * mPlayField.tileSize);
		mGameCamera.center = mPlayField.center;
		mGameCamera.offset = mPlayField.tileSize;
		@mGameCamera.observed = Object(mPlayer);
		mLevelScene.add(mGameCamera);
	}
	
	void setupGame() {
		//Spawn enemies;
		auto enemies = mPlayField.enemies;
		for (pUInt i = 0; i < enemies.length(); ++i) {
			auto@ enemy = Enemy(mPlayField, mLevelScene, enemies[i]);
			@enemy.stepped = EnemyStepped(this.enemyStepped);
			mEnemies.add(enemy);
		}
	}

	//End of player animation
	void playerStepped() {

		if (mPlayField.portal.activated) {
			if (mPlayer.location == mPlayField.portal.location)
			{
				//THAT'S THE WIN CONDITION!
				stopPlaying();

				@mTransitionFade.onFadeEnd = Action(this.winLevel);
				mTransitionFade.Show();
				mTransitionFade.fadeOut(WHITE, FM_ADDITIVE, 3000);
				return;
			}
		}
		//Check if we need to activate beacons
		Beacon@ b;
		switch (mPlayer.direction) {
			case Dir_N:
				@b = mPlayField.getBeacon(mPlayer.location.x, mPlayer.location.y - 1);
				break;
			case Dir_E:
				@b = mPlayField.getBeacon(mPlayer.location.x + 1, mPlayer.location.y);
				break;
			case Dir_S:
				@b = mPlayField.getBeacon(mPlayer.location.x, mPlayer.location.y + 1);
				break;
			case Dir_W:
				@b = mPlayField.getBeacon(mPlayer.location.x-1, mPlayer.location.y);
				break;
		}
		if (b is null) return;
		b.activate();
	}

	void beaconActivated() {
		++beaconsActivated;
		echo ("Beacon number: " + String(beaconsActivated) + " activated");
		if (beaconsActivated == mPlayField.beacons.length())
			mPlayField.portal.activate();
	}

	//End of enemy animation
	void enemyStepped(Enemy@ enemy) {
		//check if enemie has caught the player!
		if (enemy.location == mPlayer.location) {
			stopPlaying();
			mPlayer.dead();

			@mTransitionFade.onFadeEnd = Action(this.gameOver);
			mTransitionFade.Show();
			mTransitionFade.fadeOut(BLACK, FM_ALPHA, 2000);
			return;
		}


		pInt distx = mPlayer.location.x - enemy.location.x;
		pInt disty = mPlayer.location.y - enemy.location.y;

		bool flee = false;
		if (enemy.location.x == mPlayer.location.x) {
			switch (mPlayer.direction)	{
				case Dir_N:
					if (disty >= 0 && disty <= mPlayer.lightDistance) flee = true;
					break;
				case Dir_S:
					if (disty <= 0 && abs(disty) <= mPlayer.lightDistance) flee = true;
					break;
				default:
					return; //In line, but the player is looking away
			}
		} else if (enemy.location.y == mPlayer.location.y) {
			switch (mPlayer.direction)	{
				case Dir_W:
					if (distx >= 0 && distx <= mPlayer.lightDistance) flee = true;
					 break;
				case Dir_E:
					if (distx <= 0 && abs(distx) <= mPlayer.lightDistance) flee = true;
					break;
				default:
					return; //In line, but the player is looking away
			}	
		}

		if (flee) {
			//is anything blocking our view ?
			if (!mPlayField.checkLineOfSight(mPlayer.location, enemy.location)) {

				return;
			}

			enemy.flee(mPlayer.location, mPlayer.direction);
		}

		return; //not in line of light;
	}
	
	void tick() {
		for (pUInt i = 0; i < mEnemies.length(); ++i)
			mEnemies[i].tick();
		
	}

	void unloadLevel() {
		for (pUInt i = 0; i < mEnemies.length(); ++i) {
			mEnemies[i].removeFromScene(); //Garbage collector is too slow... 
		}
		mEnemies.clear();
		beaconsActivated = 0;

		@mPlayer = null;
		@mPlayField = null;
		control.removeScene(mLevelScene);
		@mLevelScene = null; 
		@mInputMapper = null;

		control.script.CollectGarbage();
		mGameOverScreen.Hide();
		mGameEndScreen.Hide();
		mTransitionFade.Hide();
		mTransitionFade.cancel();
	}
	
	void NewGame(pUInt selectedLevel = 1) {
		//Maybe create a setup function or something?
		//Let's see how this goes...
		mGameMenu.enable();
		mGameEndScreen.Hide();
		mCurrentLevel = selectedLevel;

		@mGameCamera= OffsetCam(60, mainWindow.width / float(mainWindow.height), 0.5, 100, control);
		Camera@ cam = mGameCamera;
		cam.rotate( 1,0,0 , -0.30 * PI);
		

		mViewportIdx = mainWindow.addViewport(mViewport);
		mViewport.addCamera(GAME_LAYER, mGameCamera);

		audio.playNextLevelMusic();
		loadLevel("Levels/Level" + mCurrentLevel + ".lvl");

		if (!mPlaying)
			startPlaying();
	}

	void loadNextLevel() {
		levelInfo.progress = ++mCurrentLevel;
		loadLevel("Levels/Level" + mCurrentLevel + ".lvl");
	}

	void winLevel() 
	{
		//TODO: End Game!! YOU WIN
		if (mCurrentLevel >= levelInfo.levelcount) {
			unloadLevel();

			mGameMenu.disable();
			mGameEndScreen.Show();
			levelInfo.progress = mCurrentLevel;
			mCurrentLevel = 1;
			return;
		}
		
		
			
		loadNextLevel();
		@mTransitionFade.onFadeEnd = Action(this.startPlaying);
		mTransitionFade.Show();
		mTransitionFade.fadeIn(WHITE, FM_ADDITIVE, 3000);
	}

	void gameOver() {
		mGameMenu.disable();
		mGameOverScreen.Show();
	}

	private void startPlaying() {
		mGameOverScreen.Hide();
		mTransitionFade.Hide();
		mPlaying = true;
		if (mInputMapper !is null  && !mPaused)
			mInputMapper.enable();
	}
	
	private void stopPlaying() {
		mPlaying = false;
		if (mInputMapper !is null && !mPaused)
			mInputMapper.disable();
	}

	void showCredits() {
			mGameEndScreen.Show();
	}



	

}