#include "IPlayer.as"

//All actions are relative to the camera position
enum ActionType
{
	AT_Up, 
	AT_Down,
	AT_Left,
	AT_Right,
	AT_MoveH,
	AT_MoveV,
	AT_StrafeV,
	AT_StrafeH,
	AT_StrafeUp,
	AT_StrafeDown,
	AT_StrafeLeft,
	AT_StrafeRight,
	AT_Jump, 
	AT_Attack1,
	AT_Attack2,
}


class Mapping
{
	weakref<IPlayer> mPlayer;
	pInt mPlayerNumber;
}

class KBMouseMapping : Mapping
{
	Map<pMouseButton, ActionType> mMouseBtnActionMap;
	Map<pKeyCode, ActionType> mKeyActionMap;
}

class ControllerMapping : Mapping
{
	Map<pControllerAxis, ActionType> mAxisActionMap;
	Map<pControllerButton, ActionType> mButtonActionMap;
}

class InputMapper 
{
	
	funcdef IPlayer@ PlayerJoinedCallback(pInt playerNr);
	funcdef void PlayerLeftCallback(pInt playerNr);

	private KBMouseMapping mKBMouseMap;
	//OK so it seems like our piece of shit map can do this...
	private	Map<GameController@, ControllerMapping@> mControllerMap;
	
	private Control@ mControl;
	private PlayerJoinedCallback@ mPlayerJoinedCB;
	private PlayerLeftCallback@ mPlayerLeftCB;
	
	private Array<IPlayer@> mSlots;

	bool mEnabled = false;
	
	
	pKeyCode mKBJoinKey = PK_UNKNOWN;
	pControllerButton mGCJoinBtn = PCB_BUTTON_INVALID;

	//Single player mode maps ALL inputs of ALL devices to the same player.
	private bool mSinglePlayerMode = false;

	//Single player mode.	
	InputMapper(Control@ control, IPlayer@ player)
	{
		@mControl = control;
		enable();

		mSinglePlayerMode = true;

		mKBMouseMap.mPlayer = player;
		setDefaultKBMouseBindings();

		@mControllerMap[null] = ControllerMapping();
		mControllerMap[null].mPlayer = player;
		setDefaultControllerBindings(mControllerMap[null]);
	}

	//Local multiplayer mode.
	InputMapper(Control@ control, pInt playerCount, PlayerJoinedCallback@ pjcb, PlayerLeftCallback@ plcb)
	{
		@mControl = control;
		@mPlayerJoinedCB = pjcb;
		@mPlayerLeftCB = plcb;

		for (pInt i = 0; i < playerCount; ++i)
			mSlots.add(null);
		
		enable();
	}

	void disable() {
		echo ("InputMapper disabled");
		if (!mEnabled) return;
		mEnabled = false;
		mControl.Input.keyDown(PK_ANY) 	-= KeyStateCallback(this.keyDown);
		mControl.Input.keyUp(PK_ANY) 	-= KeyStateCallback(this.keyUp);
		
		mControl.Input.mouseMove() -= MouseMoveCallback(this.mouseMoveCallback);
		
		mControl.Input.controllerButtonDown(PCB_ANY) -= ControllerButtonCallback(this.controllerBtnDown);
		mControl.Input.controllerButtonUp(PCB_ANY)   -= ControllerButtonCallback(this.controllerBtnUp);
		
		mControl.Input.controllerAxisMotion(PCA_LEFTX) -= ControllerAxisCallback(this.controllerAxisMotion);
		mControl.Input.controllerAxisMotion(PCA_LEFTY) -= ControllerAxisCallback(this.controllerAxisMotion);
	}

	void enable() {

		if (mEnabled) return;
		mEnabled = true;

		echo ("InputMapper enabled");
		mControl.Input.keyDown(PK_ANY) 	+= KeyStateCallback(this.keyDown);
		mControl.Input.keyUp(PK_ANY) 	+= KeyStateCallback(this.keyUp);
		
		mControl.Input.mouseMove() += MouseMoveCallback(this.mouseMoveCallback);
		
		mControl.Input.controllerButtonDown(PCB_ANY) += ControllerButtonCallback(this.controllerBtnDown);
		mControl.Input.controllerButtonUp(PCB_ANY)   += ControllerButtonCallback(this.controllerBtnUp);
		
		mControl.Input.controllerAxisMotion(PCA_LEFTX) += ControllerAxisCallback(this.controllerAxisMotion);
		mControl.Input.controllerAxisMotion(PCA_LEFTY) += ControllerAxisCallback(this.controllerAxisMotion);
	}


	void setKBJoinKey(pKeyCode key)
	{
		mControl.Input.keyDown(mKBJoinKey) -= KeyStateCallback(this.joinKeyDown);
		mKBJoinKey = key;
		mControl.Input.keyDown(mKBJoinKey) += KeyStateCallback(this.joinKeyDown);
	}
	
	void setGCJoinBtn(pControllerButton btn)
	{
		mControl.Input.controllerButtonDown(mGCJoinBtn) -= ControllerButtonCallback(this.joinBtnDown);
		mGCJoinBtn = btn;
		mControl.Input.controllerButtonDown(mGCJoinBtn) += ControllerButtonCallback(this.joinBtnDown);
	}
	
	void joinKeyDown(pKeyCode k)
	{
		//Is this the first player to register with keyboard ?
		auto player = mKBMouseMap.mPlayer.get();
		if (player is null)	{
			addPlayer(mKBMouseMap);
			//Here for now. Perhaps not the best place if we need to save stuff.
			setDefaultKBMouseBindings();
		} else {
		//guess not...Joining again means leaving.
			removePlayer(mKBMouseMap);
		}
	}
	
	void joinBtnDown(GameController@ controller, pControllerButton btn)
	{
		ControllerMapping@ mapping = mControllerMap[controller];
		if (mapping is null || mapping.mPlayer is null) { //we don't know this player yet!
			if (mapping is null) 
				@mapping = ControllerMapping(); //if this is a new controller, create a new mapping.
				
			if (!addPlayer(mapping)) 
				return;
			@mControllerMap[controller] = mapping;
			setDefaultControllerBindings(mapping);
			
		}
		else 
		{
			removePlayer(mapping);
		}
	}
	
	bool addPlayer(Mapping@ mapping)
	{
		//Let's see if we have an empty player slot...
		for (pUInt i = 0; i < mSlots.length(); ++i) {
			if (mSlots[i] is null)	{
				IPlayer@ player = mPlayerJoinedCB(i);
				if (player is null) return false;
				
				@mSlots[i] = player;
				mapping.mPlayerNumber = i;
				mapping.mPlayer = player;
				return true;
			}
		}
		return false;
	}
	
	void removePlayer(Mapping@ mapping)
	{
		@mSlots[mapping.mPlayerNumber] = null;
		mPlayerLeftCB(mapping.mPlayerNumber);
		mapping.mPlayer = null;
	}
	
	void doAction(IPlayer@ player, ActionType action, bool down, pFloat v = 1.0f)
	{
		switch (action)
		{
			case AT_Up:
				player.up(down ? v: 0.0f);
				break;
			case AT_Down:
				player.down(down ? v : 0.0f);
				break;
			case AT_Left:
				player.left(down ? v : 0.0f);
				break;
			case AT_Right:
				player.right(down ? v : 0.0f);
				break;
			case AT_MoveH:
				{
					if (v > 0.0f) {
						player.right(v);
						player.left(0.0f);
					} else if (v < 0.0f) {
						player.left(-v);
						player.right(0.0f);
					} else {
						player.left(0.0f);
						player.right(0.0f);
					}
					break;
				}
			case AT_MoveV:
					{
					if (v > 0.0f) {
						player.up(v);
						player.down(0.0f);
					} else if (v < 0.0f) {
						player.down(-v);
						player.up(0.0f);
					} else {
						player.up(0.0f);
						player.down(0.0f);
					}
					break;
				}

			case AT_StrafeH: //Axis controls.... -_-
				{
					if (v > 0.0f) {
						player.strafeRight(v);
						player.strafeLeft(0.0f);
					} else if (v < 0.0f) {
						player.strafeLeft(-v);
						player.strafeRight(0.0f);
					} else {
						player.strafeLeft(0.0f);
						player.strafeRight(0.0f);
					}
					break;
				}
			case AT_StrafeV:
				{
					if (v > 0.0f) {
						player.strafeUp(-v);
						player.strafeDown(0.0f);
					} else if (v < 0.0f) {
						player.strafeDown(v);
						player.strafeUp(0.0f);
					} else {
						player.strafeUp(0.0f);
						player.strafeDown(0.0f);
					}
					break;
				}
			case AT_StrafeUp:
				player.strafeUp(down ? v : 0.0f);
				break;			
			case AT_StrafeDown:
				player.strafeDown(down ? v : 0.0f);
				break;
			case AT_StrafeLeft:
				player.strafeLeft(down ? v : 0.0f);
				break;
			case AT_StrafeRight:
				player.strafeRight(down ? v : 0.0f);
				break;
			case AT_Jump:
				if (down) player.jump();
				break;
			case AT_Attack1:
				if (down) player.attack1();
				break;
			case AT_Attack2:
				if (down) player.attack2();
				break;
		}
	}
	
	
	/**							 	**
	 * Keyboard and mouse functions  *
	 **								**/   								
	void setDefaultKBMouseBindings()
	{
		//movement keys
		mKBMouseMap.mKeyActionMap[PK_UP] = AT_Up;
		mKBMouseMap.mKeyActionMap[PK_DOWN] = AT_Down;
		mKBMouseMap.mKeyActionMap[PK_LEFT] = AT_Left;
		mKBMouseMap.mKeyActionMap[PK_RIGHT] = AT_Right;

		//doubled up...
		mKBMouseMap.mKeyActionMap[PK_w] = AT_Up;
		mKBMouseMap.mKeyActionMap[PK_a] = AT_Left;
		mKBMouseMap.mKeyActionMap[PK_s] = AT_Down;
		mKBMouseMap.mKeyActionMap[PK_d] = AT_Right;
	}
	
	//Mouse and Keyboard event handlers
	void keyDown(pKeyCode k) {
		keyToAction(k, true);
	}
	
	void keyUp(pKeyCode k) {
		keyToAction(k, false);
	}
	
	void keyToAction(pKeyCode k, bool down)
	{
		IPlayer@ player = mKBMouseMap.mPlayer;
		if (player is null) return;
		
		auto it = mKBMouseMap.mKeyActionMap.find_iterator(k);
		if (it == mKBMouseMap.mKeyActionMap.end()) return;

		if (mKBMouseMap.mPlayer !is null)
			doAction(player, it.value, down);
	}
	
	void mouseBtnDown(pMouseButton c, pInt x, pInt y){
		mouseBtnToAction(c, true);
	}
	
	void mouseBtnUp(pMouseButton c, pInt x, pInt y){
		mouseBtnToAction(c, false);
	}
	
	void mouseBtnToAction(pMouseButton c, bool down)
	{
		IPlayer@ player = mKBMouseMap.mPlayer;
		if (player is null) return;
	
		auto it = mKBMouseMap.mMouseBtnActionMap.find_iterator(c);
		if (it == mKBMouseMap.mMouseBtnActionMap.end()) return;
		
		doAction(player, it.value, down);
	}

	void mouseMoveCallback(pInt x, pInt y, pInt xRel, pInt yRel){
		//TODO: Decide what to do with mouse because mouse is shit!
	}
	
	/**							 	**
	 * Controller functions  		 *
	 **								**/
	void setDefaultControllerBindings(ControllerMapping@ mapping)
	{
		//Movement

		//Analog...
		mapping.mAxisActionMap[PCA_LEFTX] = AT_MoveH;
		mapping.mAxisActionMap[PCA_LEFTY] = AT_MoveV;

		//or D-PAD
		mapping.mButtonActionMap[PCB_DPAD_UP] = AT_Up;
		mapping.mButtonActionMap[PCB_DPAD_DOWN] = AT_Down;
		mapping.mButtonActionMap[PCB_DPAD_LEFT] = AT_Left;
		mapping.mButtonActionMap[PCB_DPAD_RIGHT] = AT_Right;
	}
	
	void controllerBtnDown(GameController@ controller, pControllerButton btn)
	{
		if (mSinglePlayerMode) @controller = null;

		auto it = mControllerMap.find_iterator(controller);
		if (it == mControllerMap.end()) return;
				
		doControllerBtnAction(it.value, btn, true);
	}
	
	void controllerBtnUp(GameController@ controller, pControllerButton btn)
	{
		if (mSinglePlayerMode) @controller = null;

		auto it = mControllerMap.find_iterator(controller);
		if (it == mControllerMap.end()) return;
		
		doControllerBtnAction(it.value, btn, false);
	}
	
	void doControllerBtnAction(ControllerMapping@ mapping, pControllerButton btn, bool down)
	{
		IPlayer@ player = mapping.mPlayer.get();
		if (player is null) return;
		
		auto it = mapping.mButtonActionMap.find_iterator(btn);
		if (it == mapping.mButtonActionMap.end()) return;

		doAction(player, it.value, down); 
	}
	
	void controllerAxisMotion(GameController@ controller, pControllerAxis axis, pFloat value)
	{
		if (mSinglePlayerMode) @controller = null;

		auto it = mControllerMap.find_iterator(controller);
		if (it == mControllerMap.end()) return;
		
		ControllerMapping@ mapping = it.value;
		IPlayer@ player = mapping.mPlayer.get();
		if (player is null) return;
		
		auto ait = mapping.mAxisActionMap.find_iterator(axis);
		if (ait == mapping.mAxisActionMap.end()) return;

		doAction(player, ait.value, true, value);
	}


}





