ig.module(
    'game.entities.button'
)

.requires(
    'impact.entity'
)

.defines(function(){
    
// Base class for buttons.
EntityButton = ig.Entity.extend({
    
    zIndex: 15, // How far behind this should be drawn compared to other objects.
    is_activated: true, // If the buttons is activated, the user can interact with it.
    can_be_activate: false, // Need a delay so that when a button is clicked, and a new button is added at the same space, the new button won't be set as clicked right away.
    can_be_active_timer: new ig.Timer(), // The delay set before a new button can be clicked.
    gravityFactor: 0, // Stay where you are.
    notice: { }, // A notice or comment about what is going to happen if user clicks this button."
    scale: 1, // Scale.
    
    init: function( x, y, settings ) {
        this.parent( x, y, settings ); 

        // Increase size of hitbox for all buttons, so you can slighly "miss"; but still hit the button.
        this.size.x += 16;
        this.size.y += 16;
        this.offset.x -= 8;
        this.offset.y -= 8;       

        if(settings.notice === undefined) this.notice = "";
        else this.notice = settings.notice;
        
        if(!ig.editor){

            // Re-sort Entities
            ig.game.sortEntitiesDeferred();
            
            // Set click button delay.
            this.can_be_active_timer.set( 0.01 );
        }
    },

    // Use this for setting position when dynamic offset position is considered.
    setPosition: function(x, y) {
        this.pos.x = x + this.offset.x;
        this.pos.y = y + this.offset.y;
    },
    
    // Check whether the mouse cursor is intersecting this button or not.
    mouseCollision: function(){
        this.parent();

        // If button is recently added and has not been set to 'can be active'.
        if(!this.can_be_activate) {
            // Thus button can not be focused on yet.
            this.unfocus();
            
            // Check if button should be able to be activated.
            if(this.can_be_active_timer.delta() > 0 ) {
                this.can_be_activate = true;
                this.can_be_active_timer.pause(); // Deactivate timer.
            } else {
                return false;
            }
        }
        
        // Mouse cursor is hoovering button and user clicked on it.
        if(this.is_activated && this.has_mouse_focus) {
            this.focus();
            
            if(ig.input.released('mouse_l')) {
                this.action();
            }

            this.animate();

            return true;

        } else {
            this.unfocus();
        }
        
        return false;
    },

    animate: function() {
        if(this.scale_dir === 1) {
            this.scale +=  0.95 * ig.system.tick;

            if(this.scale > 1.08) {
                this.scale_dir = 0;
            }
        } else {
            this.scale -=  0.95 * ig.system.tick;

            if(this.scale < 0.92) {
                this.scale_dir = 1;
            }
        }
    },
    
    // Called when mouse cursor intersects button. Can be used to e.g.change frame.
    focus: function() {
        this.currentAnim = this.anims.focus;
    },
    
    // Called when mouse cursor intersects button. Can be used to e.g.change frame.
    unfocus: function() {
        this.currentAnim = this.anims.unfocus;

        this.scale = 1.0;
    },
    
    // Activate user interaction with button.
    activate: function() { this.is_activated = true; this.hide = false; },
    
    // Deactivate user interaction with button.
    deactivate: function() { this.is_activated = false; this.hide = true; },

    // Disables the button without hiding it.
    disable: function() { this.is_activated = false; },
    
    draw: function() {
        if(!this.hide) {
            var ctx = ig.system.context;
            ctx.save();
            ctx.translate( ig.system.getDrawPos( this.pos.x + ( this.size.x / 2 ) - this.offset.x ),
            ig.system.getDrawPos( this.pos.y + ( this.size.y / 2 ) - this.offset.y  ) );
            ctx.scale( this.scale, this.scale );
            if(this.currentAnim!== null) 
            this.currentAnim.draw( - (this.size.x / 2), - (this.size.y / 2) );
            ctx.restore();
        }
    }
})
});