package skyport.ais;

import java.io.IOException;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import skyport.Coordinate;
import skyport.Game;
import skyport.Resources;
import skyport.SkyportException;
import skyport.algorithms.astar.Node;
import skyport.connection.TcpClient;
import skyport.constants.GameConstants;
import skyport.enums.Direction;
import skyport.enums.DirectionFactory;
import skyport.enums.Tile;
import skyport.enums.WeaponType;
import skyport.enums.WeaponTypeFactory;
import skyport.json.MapBean;
import skyport.json.PlayerBean;
import skyport.json.JsonMessage;
import skyport.json.WeaponBean;

import com.fasterxml.jackson.core.JsonProcessingException;

public abstract class Ai {

	private String ip;
	private int port;
	protected Game game;
	protected JsonMessage message;
	protected String aiName;
	protected TcpClient client;

	public Ai(final String aiName) {
		ip = "127.0.0.1";
		port = 54321;
		game = new Game();
		message = new JsonMessage();
		this.aiName = aiName;
		client = null;
	}

	protected abstract void processTurn() throws JsonProcessingException;

	public void setIp(String ip) {
		this.ip = ip;
	}

	public void setPort(int port) {
		this.port = port;
	}

	private void handshake() throws IOException, SkyportException {
		client.send(message.handshake(aiName));

		// {"message":"connect","status":true}
		String fromServer = client.receive();
		Map<String, Object> map = message.getMap(fromServer);

		if ("connect".equals(map.get("message"))) {
			Boolean status = (Boolean) map.get("status");

			if (status == null || !status) {
				throw new SkyportException("Connection refused by server");
			}
		}
	}

	public void start() throws IOException, SkyportException {
		Socket socket = new Socket(ip, port);
		client = new TcpClient(socket);
		System.out.println("connected to " + ip + ":" + port);

		handshake();

		// {"message":"gamestate","turn":0,"players":[{"name":"stupidAI"},{"name":"idleAI0"},...
		String fromServer = client.receive();

		Map<String, Object> initialGamestate = message.getMap(fromServer);

		if ("gamestate".equals((String) initialGamestate.get("message"))) {

			@SuppressWarnings("unchecked")
			List<PlayerBean> players = (List<PlayerBean>) initialGamestate
					.get("players");

			MapBean mapBean = (MapBean) initialGamestate.get("map");
			game.initialize(mapBean, players);
		}

		client.send(message.loadout());

		fromServer = client.receive();

		System.out.println(aiName);

		while (fromServer != null) {
			long start = System.currentTimeMillis();

			Map<String, Object> gamestate = message.getMap(fromServer);
			String messageType = (String) gamestate.get("message");

			if (messageType != null) {
				switch (messageType) {
				case "gamestate":
					game.setTurn((int) gamestate.get("turn"));

					@SuppressWarnings("unchecked")
					List<PlayerBean> players = (List<PlayerBean>) gamestate
							.get("players");
					game.setPlayers(players);

					MapBean mapBean = (MapBean) gamestate.get("map");
					game.updateMap(mapBean);
					// game.getTilemap().setMapData(mapBean);

					PlayerBean currentPlayer = game.getCurrentPlayer();

					if (aiName.equals(currentPlayer.getName())) {
						processTurn();

						long end = System.currentTimeMillis();
						System.out.println("Calculation time: " + (end - start)
								+ " ms");
					} else {
						System.out.println("waiting");
					}
					break;
				case "endturn":
					//System.out.println("got endturn");
					break;
				case "action":
					String actionAiName = (String) gamestate.get("from");
					String actionType = (String) gamestate.get("type");
					PlayerBean player = game.getPlayerByName(actionAiName);
					Coordinate pos = player.getPosition();

					if (!actionAiName.equals(aiName)) {
						if ("mine".equals(actionType)) {
							Tile tileAtPos = game.getTilemap().getTileAt(pos);

							if (tileAtPos != null && tileAtPos.isMineable()) {
								Resources resources = game.getResources();
								resources.getPlayerResources(actionAiName)
										.incrementResourceCount(tileAtPos);
								resources.decrementResourceCountAt(pos);

								System.out.println(actionAiName + " mined "
										+ tileAtPos.toString());
							} else {
								System.out
										.println("A player mined a non-minable tile!");
							}
						} else if ("move".equals(actionType)) {
							String actionDir = (String) gamestate
									.get("direction");
							Direction dir = DirectionFactory.getInstance()
									.getByString(actionDir);

							pos.changeToCoordinateInDirection(dir);
						} else if ("upgrade".equals(actionType)) {
							String actionWeapon = (String) gamestate
									.get("weapon");
							WeaponType weaponType = WeaponTypeFactory
									.getInstance().getByString(actionWeapon);
							WeaponBean weapon;

							if (player.getPrimaryWeapon().getType() == weaponType) {
								weapon = player.getPrimaryWeapon();
							} else {
								weapon = player.getSecondaryWeapon();
							}

							game.getResources()
									.getPlayerResources(actionAiName)
									.upgradeWeapon(weapon);
						} else {
							System.out.println(fromServer);
						}
					}
					break;
				default:
					System.out.println(fromServer);
					break;
				}
			} else {
				System.out.println(fromServer);
			}

			fromServer = client.receive();
		}
	}

	protected Direction getSafeDirectionFrom(final Coordinate pos) {
		for (Direction dir : Direction.values()) {
			Coordinate adjacentPos = pos.getCoordinateInDirection(dir);
			Tile tile = game.getTilemap().getTileAt(adjacentPos);

			if (tile != null && tile.isTraversable()
					&& !game.opponentAt(adjacentPos)) {

				return dir;
			}
		}

		return null;
	}

	protected List<Direction> getDirectionsFrom(final Coordinate from,
			final List<Node> nodes) {
		List<Direction> sequence = new ArrayList<Direction>();
		Coordinate current = from;

		for (Node node : nodes) {
			Coordinate next = new Coordinate(node.getX(), node.getY());
			Direction dir = current.getDirectionTowards(next);
			sequence.add(dir);
			current = next;
		}

		return sequence;
	}

	protected List<Direction> getDirectionSequenceTowards(int moves,
			final Coordinate from, final Coordinate to) {

		List<Node> nodes = game.getNodemap().getRoute(from, to);

		if (nodes == null) {
			//System.out.println("Could not calculate a path");
			return null;
		} else {
			return getDirectionsFrom(from, nodes);
		}
	}

	protected List<Direction> getDirectionSequence(int maxMoves,
			final Coordinate from, final Coordinate to) {

		List<Node> nodes = game.getNodemap().getRoute(from, to);

		if (nodes == null || nodes.size() > maxMoves) {
			return null;
		} else {
			return getDirectionsFrom(from, nodes);
		}
	}

	protected WeaponBean[] weaponsSortedByDamage(final PlayerBean player) {
		WeaponBean[] weapons = { player.getPrimaryWeapon(),
				player.getSecondaryWeapon() };
		int[] damages = new int[weapons.length];

		for (int i = 0; i < weapons.length; ++i) {
			WeaponType weaponType = weapons[i].getType();
			int level = weapons[i].getLevel();

			switch (weaponType) {
			case DROID:
				damages[i] = GameConstants.DROID_DAMAGE[level - 1];
				break;
			case LASER:
				damages[i] = GameConstants.LASER_DAMAGE[level - 1];
				break;
			case MORTAR:
				damages[i] = GameConstants.MORTAR_DAMAGE[level - 1];
				break;
			default:
				break;
			}
		}

		if (damages[1] > damages[0]) {
			WeaponBean temp = weapons[0];
			weapons[0] = weapons[1];
			weapons[1] = temp;
		}

		return weapons;
	}
}
