/*  DS
 *  Copyright (C) Anders Asplund and Joakim Kolsjö 2005
 *	This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 
 */

#include "mtexture.h"
#include "mdemo.h"
#include "dirlist.h"

using namespace std;

namespace DS
{	
	MTexture::MTexture(const char* pBaseDirectory)
	{
		m_BaseDirectory = pBaseDirectory;
		m_BaseDirectory += "/textures";
	}
	
	MTexture::~MTexture()
	{
		std::map <std::string, Texture*>::iterator iter = m_textureData.begin();
		while(iter != m_textureData.end())
		{
			delete iter->second;
			++iter;
		}
	}
	
	void MTexture::Instance(const char* pBaseDirectory)
	{
		if(!m_pInstance)
			m_pInstance = new MTexture(pBaseDirectory);
	}
	
	MTexture* MTexture::GetPointer()
	{
		return m_pInstance;
	}
	
	int MTexture::LoadSingle(const char* pIdentity)
	{
		string identity = pIdentity;
		
		string file = m_BaseDirectory + "/" + pIdentity + ".png";
	
		// Check if this should be loaded as a mipmapped texture
		bool mipmap;
		if(identity.substr(identity.size() - 2, identity.size()) == "mm")
			mipmap = true;	
		else
			mipmap = false;
		mipmap = true;
							
		Texture* pTex = new Texture();
		if(pTex->Load(file.c_str(), false, mipmap))
			return 1;
		
		m_textureData[identity] = pTex;
		m_textures[identity] = 0; // We get the number of the texture in MakeGLTextures()
		m_textureNames.push_back(identity);			
		return 0;
	}
	
	int MTexture::MakeGLTextures()
	{
		for(int i = 0; i < (int)m_textureNames.size(); i++)
		{
			string identity = m_textureNames[i];
			if(!m_textures[identity])
				m_textures[identity] = m_textureData[identity]->MakeGLTexture();
		}
		
		return 0;
	}
	
	int MTexture::Load(float &state, bool &active)
	{			
		vector<string> textures = DS::DirectoryList::GetPointer()->GetFileList(m_BaseDirectory.c_str());
		
		// Check how many textures we're going to load
		int total = 0, current = 1;
		for(int i = 0; i < (int)textures.size(); i++)
			if(textures[i].substr(textures[i].size()-3, 3) == "png")
				if(!m_textures[textures[i].substr(m_BaseDirectory.size()+1, textures[i].size() - m_BaseDirectory.size()-5)])
					total++;
		
		// Load textures
		for(int i = 0; i < (int)textures.size(); i++)
		{
			if(textures[i].substr(textures[i].size()-3, 3) == "png")
			{
				string identity = textures[i].substr(m_BaseDirectory.size()+1, textures[i].size() - m_BaseDirectory.size()-5);

				if(!active)
					return 0;
				if(!m_textures[identity])
				{
					state = ((float)current++/total);
					if(LoadSingle(identity.c_str())) {
						cout << "[MTexture::Load] Failed to load texture: " << identity << endl;
						return 1;
					}
				}
			}
		}
		return 0;
	}
	
	GLuint MTexture::GetTexture(const char* pIdentity)
	{
		if(!m_textures[pIdentity] && LoadSingle(pIdentity)) {
			cout << "[MTexture::GetTexture] Texture \"" << pIdentity << "\" does not exist." << endl;
			return 0;
		}
		
		return m_textures[pIdentity];
	}
	
	std::vector<std::string> MTexture::GetList()
	{
		return m_textureNames;
	}
	
	std::vector<std::string> MTexture::GetList(const char* pHead)
	{
		// Get list of texture Identity's that start with the contents of pHead.
		vector<string> outlist;
		string Head = pHead;
		int Size = Head.size();
		if(Size > 0)
		{
			for(int i = 0; i < (int)m_textureNames.size(); i++)
			{
				if(m_textureNames[i].substr(0, Size) == Head)
					outlist.push_back(m_textureNames[i]);
			}
		}
		
		sort(outlist.begin(), outlist.end());
		return outlist;
	}
}
