/*	Texture
 *
 *			written by Alexander Zaprjagaev
 *			frustum@frustum.org
 *			http://frustum.org
 */
#include <iostream>
#include "texture.h"

#include <png.h>
#include <setjmp.h>
extern "C" {
#include <jpeglib.h>
}

/*
 */
Texture::Texture(int width,int height,int bpp, GLuint target,int flag) : width(width), height(height), bpp(bpp), target(target) 
{
    GLuint mode;
    switch (bpp) {
        case 1: mode = GL_ALPHA; break;
        case 3: mode = GL_RGB; break;
        case 4: mode = GL_RGBA; break;
        default: std::cerr << "bpp of png-image not supported (" << bpp << ")" << std::endl;
        return;
    }
	glGenTextures(1,&id);
	glBindTexture(target,id);
	if(flag & NEAREST) {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
	} else if(flag & LINEAR) {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	} else {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
	}
	if(flag & CLAMP) {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_CLAMP);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_CLAMP);
	} else if(flag & CLAMP_TO_EDGE) {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
	} else {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_REPEAT);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_REPEAT);
	}
	if(flag & MIPMAP_SGIS) glTexParameteri(target,GL_GENERATE_MIPMAP_SGIS,GL_TRUE);
	//GLuint internalformat = GL_RGBA;
	GLuint type = GL_UNSIGNED_BYTE;
	if(target == TEXTURE_1D) glTexImage1D(target,0,mode,width,0,mode,type,NULL);
	else if(target == TEXTURE_2D) glTexImage2D(target,0,mode,width,height,0,mode,type,NULL);
	else if(target == TEXTURE_RECT) glTexImage2D(target,0,mode,width,height,0,mode,type,NULL);
	else if(target == TEXTURE_CUBE) {
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X,0,mode,width,height,0,mode,type,NULL);
		glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_X,0,mode,width,height,0,mode,type,NULL);
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Y,0,mode,width,height,0,mode,type,NULL);
		glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,0,mode,width,height,0,mode,type,NULL);
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Z,0,mode,width,height,0,mode,type,NULL);
		glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Z,0,mode,width,height,0,mode,type,NULL);
	}
}

/*
 */
Texture::Texture(const char *name,GLuint target,int flag) : target(target) 
{
	glGenTextures(1,&id);
	glBindTexture(target,id);
	if(flag & NEAREST) {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_NEAREST_MIPMAP_NEAREST);
	} else if(flag & LINEAR) {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
	} else {
		glTexParameteri(target,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(target,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
	}
	if(flag & CLAMP) {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_CLAMP);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_CLAMP);
	} else if(flag & CLAMP_TO_EDGE) {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
	} else {
		glTexParameteri(target,GL_TEXTURE_WRAP_S,GL_REPEAT);
		glTexParameteri(target,GL_TEXTURE_WRAP_T,GL_REPEAT);
	}
	glTexParameteri(target,GL_GENERATE_MIPMAP_SGIS,GL_TRUE);
	if(target == TEXTURE_1D || target == TEXTURE_2D || target == TEXTURE_RECT) {
		unsigned char *data = load(name,width,height,bpp);
        GLuint mode;
        switch (bpp) {
            case 1: mode = GL_ALPHA; break;
            case 3: mode = GL_RGB; break;
            case 4: mode = GL_RGBA; break;
            default: std::cerr << "bpp of png-image not supported (" << bpp << ")" << std::endl;
                return;break;
        }
		if(flag & MIPMAP_SGIS) {
			if(target == TEXTURE_1D) glTexImage1D(target,0,mode,width,0,mode,GL_UNSIGNED_BYTE,data);
			else glTexImage2D(target,0,mode,width,height,0,mode,GL_UNSIGNED_BYTE,data);
		} else {
			if(target == TEXTURE_1D) gluBuild1DMipmaps(target,mode,width,mode,GL_UNSIGNED_BYTE,data);
			else gluBuild2DMipmaps(target,mode,width,height,mode,GL_UNSIGNED_BYTE,data);
		}
		if(data) delete data;
	} else if(target == TEXTURE_CUBE) {
		GLuint targets[6] = {
			GL_TEXTURE_CUBE_MAP_POSITIVE_X, GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
			GL_TEXTURE_CUBE_MAP_POSITIVE_Y, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
			GL_TEXTURE_CUBE_MAP_POSITIVE_Z, GL_TEXTURE_CUBE_MAP_NEGATIVE_Z
		};
		char *suffix[6] = { "px", "nx", "py", "ny", "pz", "nz" };
		for(int i = 0; i < 6; i++) {
			char buf[1024];
			sprintf(buf,name,suffix[i]);
			unsigned char *data = load(buf,width,height,bpp);
            GLuint mode;
            switch (bpp) {
                case 1: mode = GL_ALPHA; break;
                case 3: mode = GL_RGB; break;
                case 4: mode = GL_RGBA; break;
                default: std::cerr << "bpp of png-image not supported (" << bpp << ")" << std::endl;
                    return;break;
            }
			if(flag & MIPMAP_SGIS) glTexImage2D(targets[i],0,mode,width,height,0,mode,GL_UNSIGNED_BYTE,data);
			else gluBuild2DMipmaps(targets[i],mode,width,height,mode,GL_UNSIGNED_BYTE,data);
			if(data) delete data;
		}
	}
}

/*
 */
Texture::~Texture() {
	glDeleteTextures(1,&id);
}

/*
 */
void Texture::enable() {
	glEnable(target);
}

/*
 */
void Texture::disable() {
	glDisable(target);
}

/*
 */
void Texture::bind() {
	glBindTexture(target,id);
}

/*
 */
void Texture::copy(GLuint t) {
	if(t == 0) glCopyTexSubImage2D(target,0,0,0,0,0,width,height);
	else glCopyTexSubImage2D(t,0,0,0,0,0,width,height);
}

/*
 */
void Texture::render(float x0,float y0,float x1,float y1) {
	glBegin(GL_QUADS);
	if(target == TEXTURE_RECT) {
		glTexCoord2f(0,0);
		glVertex2f(x0,y0);
		glTexCoord2f(width,0);
		glVertex2f(x1,y0);
		glTexCoord2f(width,height);
		glVertex2f(x1,y1);
		glTexCoord2f(0,height);
		glVertex2f(x0,y1);
	} else {
		glTexCoord2f(0,0);
		glVertex2f(x0,y0);
		glTexCoord2f(1,0);
		glVertex2f(x1,y0);
		glTexCoord2f(1,1);
		glVertex2f(x1,y1);
		glTexCoord2f(0,1);
		glVertex2f(x0,y1);
	}
	glEnd();
}

/* images
 */
unsigned char *Texture::load(const char *name,int &width,int &height, int &bpp) {
	char *ext = strrchr(name,'.');
	if(!ext) {
		fprintf(stderr,"Texture::load(): unknown format of \"%s\" file",name);
		return NULL;
	}
	unsigned char *data = NULL;
    if(!strcmp(ext,".png")) data = load_png(name,width,height,bpp);
	else if(!strcmp(ext,".jpg")) data = load_jpeg(name,width,height,bpp);
    
	else std::cerr << "Texture::load(): unknown format of " << name << std::endl;
	return data;
}

unsigned char *Texture::load_png(const char *name,int &width,int &height, int &bpp) {
	FILE *file = fopen(name,"rb");
	if(!file) {
		fprintf(stderr,"error open PNG file \"%s\"\n",name);
		return NULL;
	}
	png_byte sig[8];
	fread(sig,8,1,file);
	if(!png_check_sig(sig,8)) {
		fprintf(stderr,"error load png file \"%s\": wrong signature\n",name);
		fclose(file);
		return NULL;
	}
	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,0,0,0);
	if(!png_ptr) {
		fclose(file);
		return NULL;
	}
	png_infop info_ptr = png_create_info_struct(png_ptr);
	if(!info_ptr) {
		png_destroy_read_struct(&png_ptr,0,0);
		fclose(file);
		return NULL;
	}
	png_init_io(png_ptr,file);
	png_set_sig_bytes(png_ptr,8);
	png_read_info(png_ptr,info_ptr);
	unsigned long w,h;
	int bit_depth,color_type;
	png_get_IHDR(png_ptr,info_ptr,&w,&h,&bit_depth,&color_type,0,0,0);
	if(bit_depth == 16) png_set_strip_16(png_ptr);
	//if(color_type == PNG_COLOR_TYPE_PALETTE) png_set_expand(png_ptr);
	if(bit_depth < 8) png_set_expand(png_ptr);
	if(png_get_valid(png_ptr,info_ptr,PNG_INFO_tRNS)) png_set_expand(png_ptr);
	//if(color_type == PNG_COLOR_TYPE_GRAY || color_type == PNG_COLOR_TYPE_GRAY_ALPHA) png_set_gray_to_rgb(png_ptr);
	double gamma;
	if(png_get_gAMA(png_ptr,info_ptr,&gamma)) png_set_gamma(png_ptr,(double)2.2,gamma);
	png_read_update_info(png_ptr,info_ptr);
	png_get_IHDR(png_ptr,info_ptr,&w,&h,&bit_depth,&color_type,0,0,0);
	png_uint_32 row_bytes = png_get_rowbytes(png_ptr,info_ptr);
	png_uint_32 channels = png_get_channels(png_ptr,info_ptr);
	png_byte *img = new png_byte[row_bytes * h];
	png_byte **row = new png_byte*[h];
	for(int i = 0; i < (int)h; i++) row[i] = img + row_bytes * i;
	png_read_image(png_ptr,row);
	png_read_end(png_ptr,NULL);
	png_destroy_read_struct(&png_ptr,0,0);
	fclose(file);
	delete row;
	width = w;
	height = h;
    bpp = channels;
	unsigned char *data = new unsigned char[width * height * bpp];
	unsigned char *ptr = data;
	for(int i = 0; i < height; i++) {
		for(int j = 0; j < width; j++) {
			int k = row_bytes * i + j * bpp;
            for (int l = 0; l < bpp; l++)
                *ptr++ = img[k+l];
		}
	}
	delete img;
	return data;
}

struct my_error_mgr {
	struct jpeg_error_mgr pub;
	jmp_buf setjmp_buffer;
};

typedef struct my_error_mgr *my_error_ptr;

static void my_error_exit(j_common_ptr cinfo) {
	my_error_ptr myerr = (my_error_ptr)cinfo->err;
	(*cinfo->err->output_message)(cinfo);
	longjmp(myerr->setjmp_buffer,1);
}

unsigned char *Texture::load_jpeg(const char *name,int &width,int &height, int& bpp) {
	struct jpeg_decompress_struct cinfo;
	struct my_error_mgr jerr;
	FILE *file;
	JSAMPARRAY buffer;
	int row_stride;
	long cont;
	JSAMPLE *data_buffer;
	int i,j;
	unsigned char *data;
	file = fopen(name,"rb");
	if(!file) {
		fprintf(stderr,"error open JPEG file \"%s\"\n",name);
		return NULL;
	}
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	if(setjmp(jerr.setjmp_buffer)) {
		jpeg_destroy_decompress(&cinfo);
		fclose(file);
		return NULL;
	}
	jpeg_create_decompress(&cinfo);
	jpeg_stdio_src(&cinfo,file);
	jpeg_read_header(&cinfo,TRUE);
	jpeg_start_decompress(&cinfo);
	row_stride = cinfo.output_width * cinfo.output_components;
	buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr)&cinfo,JPOOL_IMAGE,row_stride,1);
	data_buffer = new JSAMPLE[cinfo.image_width * cinfo.image_height * cinfo.output_components];
	cont = 0;
	while(cinfo.output_scanline < cinfo.output_height) {
		jpeg_read_scanlines(&cinfo,buffer,1);
		memcpy(data_buffer + cinfo.image_width * cinfo.output_components * cont,buffer[0],row_stride);
		cont++;
	}
	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	width = cinfo.image_width;
	height = cinfo.image_height;
    bpp = 4;
	data = new unsigned char[width * height * 4];
	switch(cinfo.output_components) {
		case 1:
			for(i = 0, j = 0; i < width * height; i++, j += 4) {
				data[j] = data[j + 1] = data[j + 2] = data_buffer[i];
				data[j + 3] = 255;
			}
			break;
		case 3:
			for(i = 0, j = 0; i < width * height * 3; i += 3, j += 4) {
				data[j] = data_buffer[i];
				data[j + 1] = data_buffer[i + 1];
				data[j + 2] = data_buffer[i + 2];
				data[j + 3] = 255;
			}
			break;
		default:
			delete data;
			delete data_buffer;
			return NULL;
	}
	delete data_buffer;
	fclose(file);
	return data;
}
/*
int Texture::save_jpeg(const char *name,const unsigned char *data,int width,int height,int quality) {
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	JSAMPROW row_pointer[1];
	int i,j,row_stride;
	unsigned char *data_buffer;
	FILE *file = fopen(name,"wb");
	if(!file) {
		fprintf(stderr,"error create JPEG file \"%s\"\n",name);
		return -1;
	}
	data_buffer = new unsigned char[width * height * 3];
	for(i = 0, j = 0; i < width * height * 4; i += 4, j += 3) {
		data_buffer[j + 0] = data[i + 0];
		data_buffer[j + 1] = data[i + 1];
		data_buffer[j + 2] = data[i + 2];
	}
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	jpeg_stdio_dest(&cinfo,file);
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 3;
	cinfo.in_color_space = JCS_RGB;
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo,quality,TRUE);
	jpeg_start_compress(&cinfo,TRUE);
	row_stride = width * 3;
	while(cinfo.next_scanline < cinfo.image_height) {
		row_pointer[0] = &data_buffer[cinfo.next_scanline * row_stride];
		jpeg_write_scanlines(&cinfo,row_pointer,1);
	}
	jpeg_finish_compress(&cinfo);
	jpeg_destroy_compress(&cinfo);
	delete data_buffer;
	fclose(file);
	return 0;
}
*/
