import java.awt.*;
import java.awt.event.*;
import javax.media.j3d.*;

/**
 * Example06 illustrates the basics of Java 3D.  
 * <p>
 * IMPORTANT:  This source code example is offered under the license at:
 * http://wireless.java.sun.com/berkeley_license.html
 *
 * @author Bill Day <bill.day@sun.com>
 * @version 1.2
 * @see java.awt.GraphicsConfiguration
 * @see java.awt.GraphicsDevice
 * @see java.awt.GraphicsEnvironment
 * @see javax.media.j3d.BranchGroup
 * @see javax.media.j3d.Canvas3D
 * @see javax.media.j3d.GraphicsConfigTemplate3D
 * @see javax.media.j3d.Locale
 * @see javax.media.j3d.PhysicalBody
 * @see javax.media.j3d.PhysicalEnvironment
 * @see javax.media.j3d.TransformGroup
 * @see javax.media.j3d.View
 * @see javax.media.j3d.ViewPlatform
 * @see javax.media.j3d.VirtualUniverse
**/

public class Example06 extends Frame {
  /**
   * Instantiates an Example06 object.
  **/
  public static void main(String args[]) {
    new Example06();
  }

  /**
   * The Example06 constructor sets the frame's size, adds the
   * visual components, and then makes them visible to the user.
   * <P>
   * We place a Canvas3D object into the Frame so that Java 3D
   * has the heavyweight component it needs to render 3D
   * graphics into.  We then call methods to construct the
   * View and Content branches of our scene graph.
  **/
  public Example06() {
    //Title our frame and set its size.
    super("Java 3D Example06, Basics");
    setSize(400,300);

    //Here is our first Java 3D-specific code.  We add a
    //Canvas3D to our Frame so that we can render our 3D
    //graphics.  Java 3D requires a heavyweight component
    //Canvas3D into which to render, and in order to instantiate
    //this component, we first have to access a 3D GraphicsConfiguration
    //using GraphicsConfigTemplate3D and java.awt GraphicsEnvironment
    //and related classes.  Note how we use GraphicsEnvironment's static
    //method getLocalGraphicsEnvironment() to return reference to the
    //systems GraphicsEnvironment.
    GraphicsConfigTemplate3D myGraphicsConfigTemplate3D = new GraphicsConfigTemplate3D();
    GraphicsEnvironment myGraphicsEnvironment = GraphicsEnvironment.getLocalGraphicsEnvironment();
    GraphicsDevice myGraphicsDevice = myGraphicsEnvironment.getDefaultScreenDevice();
    GraphicsConfiguration myGraphicsConfiguration = myGraphicsDevice.getBestConfiguration(myGraphicsConfigTemplate3D);
    Canvas3D myCanvas3D = new Canvas3D(myGraphicsConfiguration);
    add(myCanvas3D,BorderLayout.CENTER);

    //Turn on the visibility of our frame.
    setVisible(true);

    //We want to be sure we properly dispose of resources 
    //this frame is using when the window is closed.  We use
    //an anonymous inner class adapter for this.
    addWindowListener(new WindowAdapter()
      {public void windowClosing(WindowEvent e)
         {dispose(); System.exit(0);}
      }
    );

    //Now that we have our Frame and Canvas3D, we are ready
    //to start building our scene graph.  We need to construct
    //both a view branch and a content branch.  In order to
    //actually use our view branch, we also need to construct
    //a View and connect it to our view branch's ViewPlatform.
    View myView = constructView(myCanvas3D);
    Locale myLocale = constructViewBranch(myView);
    constructContentBranch(myLocale);
  }

  /**
   * constructView() takes a Canvas3D reference and constructs
   * a View to display in that Canvas3D.  It uses the default
   * PhysicalBody and PhysicalEnvironment (both required to be
   * set or else the 3D runtime will throw exceptions).  The
   * returned View is used by constructViewBranch() to attach
   * the scene graph's ViewPlatform to a Canvas3D for rendering.
   *
   * @see constructViewBranch(View)
  **/
  private View constructView(Canvas3D myCanvas3D) {
    View myView = new View();
    myView.addCanvas3D(myCanvas3D);
    myView.setPhysicalBody(new PhysicalBody());
    myView.setPhysicalEnvironment(new PhysicalEnvironment());
    return(myView);
  }


  /**
   * constructViewBranch() takes as input a View which we
   * attached to our Canvas3D in constructView().  It constructs
   * a default view branch for the scene graph, attaches
   * the View to the ViewPlatform, and returns a reference to
   * our Locale for use by constructContentBranch()
   * in creating content for our scene graph.
   *
   * @see constructView(Canvas3D)
   * @see constructContentBranch(Locale)
  **/
  private Locale constructViewBranch(View myView) {

    //First, we create the necessary coordinate systems
    //(VirtualUniverse, Locale), container nodes
    //(BranchGroup, TransformGroup), and platform which
    //determines our viewing position and direction (ViewPlatform).
    VirtualUniverse myUniverse = new VirtualUniverse();
    Locale myLocale = new Locale(myUniverse);
    BranchGroup myBranchGroup = new BranchGroup();
    TransformGroup myTransformGroup = new TransformGroup();
    ViewPlatform myViewPlatform = new ViewPlatform();

    //Next, we insert the platform into the transform group,
    //the transform group into the branch group, and the branch
    //group into the locale's branch graph portion of the
    //scene graph.
    myTransformGroup.addChild(myViewPlatform);
    myBranchGroup.addChild(myTransformGroup);
    myLocale.addBranchGraph(myBranchGroup);

    //Finally, we attach our view to the view platform and we
    //return a reference to our new universe.  We are ready to
    //render 3D content!
    myView.attachViewPlatform(myViewPlatform);
    return(myLocale);
  }

  /**
   * constructContentBranch() is where we specify the 3D graphics
   * content to be rendered into the Locale referenced
   * in the passed parameter.  Nothing is currently specified, so we
   * render an empty universe.
   *
   * @see constructViewBranch(View)
  **/
  private void constructContentBranch(Locale myLocale) {
    //Insert content to be rendered here.
  }
}
