//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cHurtable.hpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================
#ifndef cHurtable_hpp
#define cHurtable_hpp
//==============================================================================
// Includes
#include "cCollidable.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace std;
namespace ShootingStar {
//------------------------------------------------------------------------------
// Forward declarations
//==============================================================================
	
//==============================================================================
//! Object that can be hurt by harmful objects
//------------------------------------------------------------------------------
class cHurtable : public cCollidable
{
	// Constructor & Destructor
	public:
		//! Constructor
		cHurtable (void);
		//! Destructor
		virtual ~cHurtable (void);
	
	// Public methods
	public:
		// --- Armor ---
		//! Sets the maximal armor for this object
		void SetMaxArmor (int maxArmor) 
		{ 
			mMaxArmor = maxArmor; 
			LimitArmor ();
		};
		//! Returns the maximal armor of this object
		int GetMaxArmor (void) const { return mMaxArmor; };
		//! Sets the armor of this object
		void SetArmor (int armor)
		{
			mArmor = armor;
			LimitArmor ();
		};
		//! Returns the armor of this object
		int GetArmor (void) const { return mArmor; };
		//! Increase the armor of this object
		void GainArmor (int amount)
		{
			mArmor += amount;
			LimitArmor ();
		};
		
		// --- Health ---
		//! Sets the maximal health for this object
		void SetMaxHealth (int maxHealth)
		{
			mMaxHealth = maxHealth;
			LimitHealth ();
		};
		//! Returns the maximal health of this object
		int GetMaxHealth (void) const { return mMaxHealth; };
		//! Sets the health of this object
		void SetHealth (int health)
		{
			mHealth = health;
			LimitHealth ();
		};
		//! Returns the health of this object
		int GetHealth (void) const { return mHealth; };
		//! Increase the health of this object
		void GainHealth (int health)
		{
			mHealth += health;
			LimitHealth ();
		};
		
		// --- Damage ---
		void DoDamage (int amount, const cVector2f &direction, ObjectID attacker = 0)
		{
			OnHurt (amount, direction, attacker);
		
			mArmor -= amount;
			if ( mArmor < 0 )
			{
				mHealth += mArmor;
				mArmor = 0;
			
				// Check if the object has died
				if ( mHealth < 0 )
					OnDeath ();
			}
		};
		//! Called when object is hurt
		virtual void OnHurt (int amount, const cVector2f &direction, ObjectID attacker) { }; 
		
		//! Called on death
		virtual void OnDeath (void) { Kill (); };
	
	// Private methods
	private:
		void LimitArmor (void)
		{
			if ( mArmor > mMaxArmor )
				mArmor = mMaxArmor;
		};
		void LimitHealth (void) 
		{
			if ( mHealth > mMaxHealth )
				mHealth = mMaxHealth;
		};
	
	// Member variables
	private:
		int mMaxArmor;		//!< Maximal armor
		int mArmor;			//!< Current armor
		int mMaxHealth;		//!< Maximal health
		int mHealth;		//!< Current health
};
//==============================================================================

//==============================================================================
}		// End of the ShootingStar namespace
#endif // cHurtable_hpp
//------------------------------------------------------------------------------
// EOF
//==============================================================================
