;* Small 3D Engine // Simply (?) key frame tracking system
;* (C)oded by Sergey Chaban
;* Thanks to Digisnap


TIME_SCALE=1


;_TRACKDEBUG=1 ; Uncomment for debugging

SplineCase EQU 1  ; Bit mask for the incoming/outgoing flag



COMMENT ~
This tracking system implements the Kochanek-Bartels method for "Interpolating
splines with local tension, continuity and bias control".
It uses splines with a cubic interpolation between each pair of predefined 
key points.
The track points are defined using:
a) 3D position vector (X,Y,Z); 
b) 3D rotation (3 Euler angles - roll, pitch and yaw) 
   and 
c) properties (for both pos and rot) for each key point.
The track preparation routine converts Euler angles into quaternions, and
then quaternions are used in calculations. It also precalculates tangent
vectors for the key points.
Rotations are applied in a following order:
pitch, then roll, then yaw - YXZ.
The properties for the key are varies between [-1,1] and defined as follows:
Continuity - controls the smoothness of the curve at the control point.
             When C<>0 curve has a 'corner' at the control point (its 
             direction depends on the sign of C).
Tension    - controls the length of the tangent vector at the control point.
             A smaller tangent (T near 1) leading to a tightening of the
             curve at the control point, a larger one (T near -1) leading to a 
             slackening.
Bias       - defines which tangent vector (in or out) will dominate the 
             direction of the path of the curve at the point. When B=0 the
             tangents are equally weighted. When B near -1 the outgoing 
             dominates. When B near 1 the incoming dominates.
~


;*** Calculates one element of tangent vector for the Hermite spline
; BX=element offset for the integer vector
; AX=element offset for the float vector
CalcTanVec proc
 fld dword ptr [bp].Temp_Bias
 fld st(0)
 fadd dword ptr ds:[fp_One]                 ; (1+B)
 fild word ptr [si].[bx]                    ; pn
 fisub word ptr [si-SIZE KeyFrame_Raw].[bx] ; pn-p(n-1)
 fmulp st(1),st                             ; g1=(pn-p(n-1))*(1+B)
 fxch
 fchs                                       ; -B
 fadd dword ptr ds:[fp_One]                 ; (1-B)
 fild word ptr [si+SIZE KeyFrame_Raw].[bx]  ; p(n+1)
 fisub word ptr [si].[bx]                   ; p(n+1)-pn
 fmulp st(1),st                             ; g2=(p(n+1)-pn)*(1-B)
 fsub st,st(1)                              ; g3=g2-g1
 fld dword ptr [bp].Temp_Continuity         ; C
 test byte ptr [bp].Temp_flags,SplineCase
 jz @@Incoming
 fchs                                       ; -C for the outgoing spline
@@Incoming:
 fadd dword ptr ds:[fp_One]                 ; (1+C) or (1-C)
 fmul dword ptr ds:[fp_Half]                ; 0.5*(1+-C)
 fmulp st(1),st                             ; g3*0.5*(1+-C)
 faddp st(1),st                             ; g1+g3*0.5*(1+-C)
 fld1
 fsub dword ptr [bp].Temp_Tension           ; (1-T)
 fmulp st(1),st                             ; (g1+g3*0.5*(1+-C))*(1-T)
 push di
 add di,ax
 fstp dword ptr es:[di]
 pop di
 ret
CalcTanVec endp

; Calc control point for Bezier spline
CalcControlPoint proc
 push si
 push di
 fld dword ptr ds:[fp_Three]                ; 3.0
 fdivr dword ptr ds:[fp_One]                ; 1/3
 sub di,SIZE KeyFrame-offset Key_Rotation   ; point DI to q(n-1)
 ; Copy three quatenions q(n-1), qn & q(n+1) from track's structures
 ; into temporal storage (quaternions qn_1, qn & qn1).
 mov si,offset qn_1
 mov cl,3
 sub bx,bx
@@copyquats:
 mov ch,4
@@copyonequat:
 mov eax,es:[di+bx]
 mov [si+bx],eax
 add bl,4
 dec ch
 jnz @@copyonequat
 add di,SIZE KeyFrame-SIZE Quaternion
 loop @@copyquats

 fld dword ptr [bp].Temp_Bias
 fld st(0)
 fadd dword ptr ds:[fp_One]                 ; 1+B, B, 1/3
 fchs                                       ; -(1+B), B, 1/3
 fmul st,st(2)                              ; -(1+B)/3, B, 1/3
 fxch                                       ; B, -(1+B)/3, 1/3
 fsubr dword ptr ds:[fp_One]                ; (1-B), -(1+B)/3, 1/3
 fmulp st(2),st                             ; -(1+B)/3, (1-B)/3
 ;** g1=slerp (qn,q(n-1),-(1+B)/3)
 mov si,offset qn                           ; SI=ptr on qn
 mov di,offset qn_1                         ; DI=ptr on q(n-1)
 mov bx,offset TempQuat1
 call Slerp
 ;** g2=slerp (qn,q(n+1),(1-B)/3)
 push bx                                    ; save g1 ptr
 mov di,offset qn1                          ; DI=ptr on q(n+1)
 mov bx,offset TempQuat2
 call Slerp
 fld dword ptr [bp].Temp_Tension
 fsub dword ptr ds:[fp_One]                 ; T-1
 fld dword ptr [bp].Temp_Continuity         ; C, T-1
 test byte ptr [bp].Temp_Flags,SplineCase
 mov dx,offset KeyFrame.Key_C_in            ; Assume incoming control point
 jz @@Incoming
 mov dl,offset KeyFrame.Key_C_out           ; Nope -> outgoing
 ; Change signs for the outgoing spline
 fchs                                       ; -C, (T-1)
 fldz                                       ; 0, -C, (T-1)
 fsubrp st(2),st                            ; -C, -(T-1)
@@Incoming:
 fadd dword ptr ds:[fp_One]                 ; (1+C) or (1-C)
 fmul dword ptr ds:[fp_Half]                ; 0.5*(1+-C)
 mov di,bx                                  ; g2 ptr
 pop cx
 push si                                    ; save qn ptr
 mov si,cx                                  ; g1 ptr
 ;** g3=(g1,g2,0.5+-0.5*C)
 ;Note that g2 will be overwritten (BX=DI)
 call Slerp
 mov di,bx                                  ; DI=g3 ptr
 mov bx,si                                  ; BX=TempQuat1 / erase g1
 pop si                                     ; restore qn ptr in SI
 ;** Result=slerp(qn,g3,+-(T-1))
 call Slerp
 mov si,bx
 pop bx                                     ; pop saved DI (track data ptr)
 ;; mov di,bx
 ;; add di,dx
 lea di,[ebx+edx]
 mov cx,(SIZE Quaternion)/2
 rep movsw
 mov di,bx
 pop si
 ret
CalcControlPoint endp

CopyKeyParam MACRO ParamName, PopFlag
LOCAL L
 PUSHSTATE
 MASM51
  SrcStr   CATSTR <word ptr [si].[bx].>,<ParamName>
  DestStr  CATSTR <dword ptr [bp].Temp_>,<ParamName>

   fild SrcStr

  IFB <PopFlag>
   fmul st,st(1)  ; Convert from fixed to float
  ELSE
   fmulp st(1),st ; Convert from fixed to float and remove scalemult from stack
  ENDIF

    IFDEF __DEBUG
     push bx
     cmp bx,offset RawKey_PosParameters
     mov bx,offset Key_PosParameters
     jz L
     mov bx,offset Key_RotParameters
   L:
     DbgDest CATSTR <dword ptr es:[di].[bx].f>,<ParamName>
     fst DbgDest
     pop bx
    ENDIF

   fstp DestStr

 POPSTATE
ENDM

;* Convert pos/rot parameters from fixed point to float
; SI=KeyFrame_Raw ptr
; BX=offset of KeyFrame_Raw structure member (Pos/Rot parameters)
ConvertKeyParameters proc
 fld dword ptr ds:[fp_32767]
 fdivr dword ptr ds:[fp_One]  ; 1/32767
 CopyKeyParam Continuity
 CopyKeyParam Tension
 CopyKeyParam Bias,POP
 ret
ConvertKeyParameters endp


;/////////////////////////////////////////////////////////////////////////
;***************** Prepare track for the interpolation *******************
;/////////////////////////////////////////////////////////////////////////
; DS:SI=Ptr to source (raw) track
; ES:DI=Ptr to destination track
;    CX=Number of points in the track
; NOTE: There are two 'auxiliary' points, one before the beginning and one
;       after the end of the track, which points are needed to calc the
;       tangent vectors and control points for the first and the last points
;       of the track. This way allowes me to avoid the special case handling
;       for these points and thus makes the code smaller. The alternative for
;       that is to having the looped tracks, but it's not always a convenient.
;       I refer to these points as follows:
;        'head' point for the beginning
;        'tail' point for the end
PrepareTrack proc
 ;* Make room for the local vars
 enter 3*4+2,0
 ;* Point ES to the track data segment
 mov es,word ptr ds:[TrackData]

 ;* Define offsets for the local vars
 Temp_Continuity = -14
 Temp_Tension    = -10
 Temp_Bias       = -6
 Temp_Flags      = -2

 ;** First convert all rotations into quaternions
 ;NOTE: Camera must be properly initialized, i. e. plased into quaternion mode
 pusha
 ; Take 'head' and 'tail' points into account
 inc cx                    ; Two extra points
 inc cx
 sub si,SIZE KeyFrame_Raw  ; Ptr onto the source's 'head' point
 sub di,SIZE KeyFrame      ; Ptr onto the dest's 'head' point
@@ConvertRotations:
 add si,offset RawKey_Angles
 push cx
 push si di
 call Angles2Quaternion
 pop di si
 ; Copy resulting quaternion from [BX] to ES:[DI]
 ;!!NOTE: Key_Rotation field supposed to be the last member of structure!
 xchg si,bx
 add di,offset KeyFrame.Key_Rotation
 mov cx,(SIZE Quaternion)/2
 rep movsw
 lea si,[bx+(SIZE KeyFrame_Raw-offset RawKey_Angles)] ; go to p(n+1)
 pop cx
 loop @@ConvertRotations
 popa


IFDEF _TRACKDEBUG ;================================================
 ;*Debugging mode - check if all rotations were converted correctly
 pushad
@@doCheck:
 push cx di si
 ; Create the first debugging camera and put it into Euler mode
 sub al,al
 mov si,offset EulerDebugCam
 call Init_Camera
 mov bx,[esp] ; BX=RawKey ptr
 ; Calc rotation matrix using Euler angles obtained from the raw frame data
 mov ax,[bx].KeyFrame_Raw.RawKey_Angles.Angle_X
 mov [si].Camera_AngleX,ax
 mov ax,[bx].KeyFrame_Raw.RawKey_Angles.Angle_Y
 mov [si].Camera_AngleY,ax
 mov ax,[bx].KeyFrame_Raw.RawKey_Angles.Angle_Z
 mov [si].Camera_AngleZ,ax
 call Rotate_Camera
 push si
 ; Create second debugging cam, this time the quaternion one
 mov al,CamFlag_Quaternions
 mov si,offset QuatDebugCam
 call Init_Camera
 mov eax,es:[di].KeyFrame.Key_Rotation.Quat_W
 mov [si].Camera_QuatRot.Quat_W,eax
 mov eax,es:[di].KeyFrame.Key_Rotation.Quat_X
 mov [si].Camera_QuatRot.Quat_X,eax
 mov eax,es:[di].KeyFrame.Key_Rotation.Quat_Y
 mov [si].Camera_QuatRot.Quat_Y,eax
 mov eax,es:[di].KeyFrame.Key_Rotation.Quat_Z
 mov [si].Camera_QuatRot.Quat_Z,eax
 call Quaternion2CamMatrix
 pop bx
 mov ax,offset Camera.Camera_RotationMatrix
 add bx,ax
 mov di,offset EulerInt
 call Matrix2Int
 add si,ax
 mov bx,si
 mov si,di
 mov di,offset QuatInt
 call Matrix2Int
 push es
 push ds
 pop es
 mov cx,SIZE Matrix SHR 1
 rep cmpsw
 pop es
 pop si di cx
 jz @@Okay
 int 3  ; Should not be that!
 db 'Error converting raw track''s rotations!',0
@@Okay:
 dec cx
 jnz @@doCheck
 popad
ENDIF ;============================================================


@@ConvertKeys:
 push cx
 movzx eax,word ptr [si].RawKey_FrameNumber    ; frame time (number) in secs
 imul eax,1000                                 ; convert to milliseconds
 IF NOT(TIME_SCALE EQ 1)
 sub edx,edx
 mov ebx,TIME_SCALE
 div ebx
 ENDIF
 mov es:[di].Key_FrameNumber,eax

 ;* Convert position vector from integer to floating point
 fild word ptr [si].KeyFrame_Raw.RawKey_X
 fstp es:[di].KeyFrame.Key_Position.Vector3.VecX
 fild word ptr [si].KeyFrame_Raw.RawKey_Y
 fstp es:[di].KeyFrame.Key_Position.Vector3.VecY
 fild word ptr [si].KeyFrame_Raw.RawKey_Z
 fstp es:[di].KeyFrame.Key_Position.Vector3.VecZ

 mov ax,offset KeyFrame.Key_Tout.Vector3.VecX
 mov cl,2 ; two tangent vectors for every point
 mov byte ptr [bp].Temp_Flags,SplineCase    ; Start with outgoing spline
 mov bx,offset KeyFrame_Raw.RawKey_PosParameters
 call ConvertKeyParameters
@@CalcTn_Outter:
 mov bx,offset KeyFrame_Raw.RawKey_X
 mov ch,3 ; repeat for the whole position vector
@@CalcTn:
 call CalcTanVec
 inc bx
 inc bx
 add ax,4
 dec ch
 jnz @@CalcTn
 xor byte ptr [bp].Temp_Flags,SplineCase    ; Switch to incoming
 loop @@CalcTn_Outter
 ;; mov byte ptr [bp].Temp_Flags,SplineCase ; Flag has already switched to out
 mov bx,offset KeyFrame_Raw.RawKey_RotParameters
 call ConvertKeyParameters  ; Convert parameters for rotation track
 mov cl,2
@@Calc_CPoints:
 push cx
 call CalcControlPoint
 pop cx
 xor byte ptr [bp].Temp_Flags,SplineCase
 loop @@Calc_CPoints

 add si,SIZE KeyFrame_Raw                   ; next source frame
 add di,SIZE KeyFrame                       ; next destination frame
 pop cx
 loop @@ConvertKeys
 leave
 ret
PrepareTrack endp

InterpolateTrack proc
 @@t=-10
 enter 10,0
 mov es,word ptr ds:[TrackData]
;////////////////////////////////////////////////////////////////////////
; Calculate the Hermite spline for the position track.
; H(t)=p1*(2t^3-3t^2+1)+T1*(t^3-2t^2+t)+p2*(-2t^3+3t^2)+T2*(t^3-t^2)
; t is a local time parameter t=(F-f1)/(f2-f1), where F is the global
; time (frame number), f1 is the frame number for the point p1 (starting
; point) and f2 is the frame number for the p2 (ending point).
; t ranges from 0.0 to 1.0, F runs from f1 to f2 and when it reaches f2
; our ending point p2 becomes the new starting point and we've making the
; next point in the track being our new ending point.
; T1 and T2 are tangent vectors at points p1 and p2, these vectors are
; precalculated using four consecutive points from the track (see above).
;
;----------------------------------------------------------------------------
; [TimeCounter] is a global time counter which is
; periodically incremented by the 1kHz timer routine,
; i. e. it counts milliseconds.
 mov eax,dword ptr ds:[TimeCounter] ; global time F (frame number) in millisecs
 mov di,0
CurrentTrackPointPtr EQU $-2
; Search for the current segment of the track. Segment means two points for
; which the global time lies inbetween.
FindCurrTrackSegment:
 cmp eax,es:[di].Key_FrameNumber
 je CopyThisKeyframe         ; this frame exactly -> just use its data
 add di,SIZE KeyFrame
 cmp eax,es:[di].Key_FrameNumber
 jae FindCurrTrackSegment
 ; We need unsigned dword, so we've performing qword reading, high dword
 ; of counter must be set to zero!
 fild qword ptr [TimeCounter]                         ; F
 fild dword ptr es:[di-SIZE KeyFrame].Key_FrameNumber ; f1, F
 fsub st(1),st                                        ; f1, F-f1
 fisubr dword ptr es:[di].Key_FrameNumber             ; f2-f1, F-f1
 fdivp st(1),st                                       ; t=(F-f1)/(f2-f1)
 sub di,SIZE KeyFrame
 mov word ptr [CurrentTrackPointPtr],di
 fld st(0)
 fstp tbyte ptr [bp].@@t                              ; save t
;----------------------------------------------------------------------------
; Now ST contains t
; Calculate the Hermite spline's coefficients:
;   hp1=2t^3-3t^2+1
;   ht1=t^3-2t^2+t
;   hp2=-2t^3+3t^2
;   ht2=t^3-t^2
 fld st(0)
 fld st(0)                  ; t, t, t
 fmul st(1),st              ; t, t^2, t
 fxch                       ; t^2, t, t
 fmul st(2),st              ; t^2, t, t^3
 fld st(0)                  ; t^2, t^2, t, t^3
 fxch st(3)                 ; t^3, t^2, t, t^2
 fsubr st(1),st             ; t^3, t^3-t^2, t, t^2
 fadd st(2),st              ; t^3, t^3-t^3, t^3+t, t^2
 fadd st,st(0)              ; 2t^3, t^3-t^2, t^3+t, t^2
 fld st(3)                  ; t^2, 2t^3, t^3-t^2, t^3+t, t^2
 fsubr st(1),st             ; t^2, -2t^3+t^2, t^3-t^2, t^3+t, t^2
 faddp st(4),st             ; -2t^3+t^2, t^3-t^2, t^3+t, 2t^2
 fxch st(3)                 ; 2t^2, t^3-t^2, t^3+t, -2t^3+t^2
 fsub st(2),st              ; 2t^2, t^3-t^2, t^3-2t^2+t, -2t^3+t^2
 faddp st(3),st             ; t^3-t^2, t^3-2t^2+t, -2t^3+3t^2
 fld dword ptr ds:[fp_One]  ; 1, t^3-t^2, t^3-2t^2+t, -2t^3+3t^2
 fsub st,st(3)              ; 2t^3-3t^2+1, t^3-t^2, t^3-2t^2+t, -2t^3+3t^2
 ; Now we has all elements in the FPU stack: hp1, ht2, ht1, hp2
 ; Calculate the spline.
 sub bx,bx                  ; BX is used as a Vector3's element offset
                            ; for the Camera and KeyFrame structs members,
                            ; it's also used as a loop counter.
CalcHermite:
 fld dword ptr es:[di].Key_Position[bx] ; p1, hp1, ht2, ht1, hp2
 fld dword ptr es:[di+SIZE KeyFrame].Key_Position[bx] ; p2,p1,hp1,ht2,ht1,hp2
 fld dword ptr es:[di].Key_Tout[bx]     ; T1, p2, p1, hp1, ht2, ht1, hp2
 fld dword ptr es:[di+SIZE KeyFrame].Key_Tin[bx] ; T2, T1, p2, p1, hp1, ht2, ht1, hp2
 fmul st,st(5)            ; T2*ht2, T1, p2, p1, hp1, ht2, ht1, hp2
 fxch                     ; T1, T2*ht2, p2, p1, hp1, ht2, ht1, hp2
 fmul st,st(6)            ; T1*ht1, T2*ht2, p2, p1, hp1, ht2, ht1, hp2
 fxch st(2)               ; p2, T2*ht2, T1*ht1, p1, hp1, ht2, ht1, hp2
 fmul st,st(7)            ; p2*hp2, T2*ht2, T1*ht1, p1, hp1, ht2, ht1, hp2
 fxch st(3)               ; p1, T2*ht2, T1*ht1, p2*hp2, hp1, ht2, ht1, hp2
 fmul st,st(4)            ; p1*hp1, T2*ht2, T1*ht1, p2*hp2, hp1, ht2, ht1, hp2
 fxch                     ; T2*ht2, p1*hp1, T1*ht1, p2*hp2, hp1, ht2, ht1, hp2
 faddp st(3),st
 faddp st(1),st
 faddp st(1),st
 fstp dword ptr [si].Camera_Position[bx]
 add bx,4
 cmp bx,SIZE Vector3
 jb CalcHermite
 ; Move four numbers out of the FPU using two FCOMPPs (P5 did it faster,
 ; P6 don't cares, saves 4 bytes, I like it :).
 fcompp
 fcompp

 _qn=offset qroll
 _qn1=offset qpitch
 _bn=offset qyaw           ; outgoing curve for pn
 _an1=offset TempQuat1     ; incoming curve for p(n+1)
 _q0=offset TempQuat2
 _q1=_qn
 _q2=_bn
 _q3=_qn1
 _q4=_an1
 Num=0
 REPT 4
 mov eax,dword ptr es:[di].Key_Rotation[Num]               ; qn
 mov ebx,dword ptr es:[di+SIZE KeyFrame].Key_Rotation[Num] ; q(n+1)
 mov ecx,dword ptr es:[di].Key_C_Out[Num]                  ; bn
 mov edx,dword ptr es:[di+SIZE KeyFrame].Key_C_In[Num]     ; a(n+1)
 mov dword ptr _qn[Num],eax
 mov dword ptr _qn1[Num],ebx
 mov dword ptr _bn[Num],ecx
 mov dword ptr _an1[Num],edx
 Num=Num+4
 ENDM

 push di si
 fld tbyte ptr [bp].@@t
 mov si,offset _qn
 mov di,offset _bn
 mov bx,offset _q0
 call Slerp
 fld tbyte ptr [bp].@@t
 mov si,offset _bn
 mov di,offset _an1
 mov bx,offset _q1
 call Slerp
 fld tbyte ptr [bp].@@t
 mov si,offset _an1
 mov di,offset _qn1
 mov bx,offset _q2
 call Slerp
 fld tbyte ptr [bp].@@t
 mov si,offset _q0
 mov di,offset _q1
 mov bx,offset _q3
 call Slerp
 fld tbyte ptr [bp].@@t
 mov si,offset _q1
 mov di,offset _q2
 mov bx,offset _q4
 call Slerp
 fld tbyte ptr [bp].@@t
 mov si,offset _q3
 mov di,offset _q4
 pop bx
 push bx
 add bx,offset Camera.Camera_QuatRot
 call Slerp
 pop si di

@@CalcMatrix:

; Derive camera rotation matrix from the quaternion q=[w,(x,y,z)]
; [si]=camera struc
; It's CAMERA matrix
;     [ 1-2yy-2zz    2wz-2xy    2xz+2wy]
;     [  -2xy-2wz  1-2xx-2zz    2wx-2yz]
;     [   2xz-2wy   -2yz-2wx  1-2xx-2yy]
;
; SIDENOTES:
; All quaternion's members are loaded once and all of FPU regs are used.
; This way is probably faster. This also results in a damn cryptic code -
; the true pleasure for the one who decide to disassemble it ;-)
; Camera structure and its members must be properly aligned to
; achieve max speed. Taken care of a cache lines filling.
Quaternion2CamMatrix:
 fld dword ptr [si].Camera_QuatRot.Quat_W
 fld dword ptr [si].Camera_QuatRot.Quat_X
 fld st(0)
 fmul st,st(0)               ; xx, x, w
 fld dword ptr [si].Camera_QuatRot.Quat_Y
                             ; y, xx, x, w
 fld st(0)                   ; y, y, xx, x, w
 fmul st,st(0)               ; yy, y, xx, x, w
 fxch st(3)                  ; x, y, xx, yy, w
 fld dword ptr [si].Camera_QuatRot.Quat_Z
                             ; z, x, y, xx, yy, w
 fld st(0)                   ; z, z, x, y, xx, yy, w
 fmul st,st(0)               ; zz, z, x, y, xx, yy, w
 fld st(0)                   ; zz, zz, z, x, y, xx, yy, w
 fadd st,st(6)               ; zz+yy, zz, z, x, y, xx, yy, w
 fadd st,st(0)               ; 2*(zz+yy), zz, z, x, y, xx, yy, w
 fsubr dword ptr ds:[fp_One] ; 1-2yy-2zz, zz, z, x, y, xx, yy, w
 fstp dword ptr [si].Camera_RotationMatrix.matr_Xx
 ; zz, z, x, y, xx, yy, w
 fadd st,st(4)               ; zz+xx, z, x, y, xx, yy
 fadd st,st(0)               ; 2*(zz+xx), z, x, y, xx, yy
 fsubr dword ptr ds:[fp_One] ; 1-2xx-2zz, z, x, y, xx, yy
 fstp dword ptr [si].Camera_RotationMatrix.matr_Yy
 ; z, x, y, xx, yy, w
 fld st(5)                   ; w, z, x, y, xx, yy, w
 fmul st,st(1)               ; wz, z, x, y, xx, yy, w
 fxch st(5)                  ; yy, z, x, y, xx, wz, w
 fld st(6)                   ; w, yy, z, x, y, xx, wz, w
 fmul st,st(4)               ; wy, yy, z, x, y, xx, wz, w
 fxch st(5)                  ; xx, yy, z, x, y, wy, wz, w
 faddp st(1),st              ; (xx+yy), z, x, y, wy, wz, w
 fadd st,st(0)               ; 2*(xx+yy), z, x, y, wy, wz, w
 fsubr dword ptr ds:[fp_One] ; 1-2xx-2yy, z, x, y, wy, wz, w
 fstp dword ptr [si].Camera_RotationMatrix.matr_Zz
 ; z, x, y, wy, wz, w
 fld st(1)                   ; x, z, x, y, wy, wz, w
 fmul st(6),st               ; x, z, x, y, wy, wz, wx
 fmul st,st(1)               ; xz, z, x, y, wy, wz, wx
 fxch st(3)                  ; y, z, x, xz, wy, wz, wx
 fmul st(2),st               ; y, z, xy, xz, wy, wz, wx
 fmulp st(1),st              ; yz, xy, xz, wy, wz, wx
 fld st(4)                   ; wz, yz, xy, xz, wy, wz, wx
 fsub st,st(2)               ; wz-xy, yz, xy, xz, wy, wz, wx
 fadd st,st(0)               ; 2*(wz-xy)
 fstp dword ptr [si].Camera_RotationMatrix.matr_Xy  ; [2wz-2xy]
 ; yz, xy, xz, wy, wz, wx
 fld st(3)                   ; wy, yz, xy, xz, wy, wz, wx
 fadd st,st(3)               ; xz+wy, yz, xy, xz, wy, wz, wx
 fadd st,st(0)               ; 2xz+2wy, yz, xy, xz, wy, wz, wx
 fstp dword ptr [si].Camera_RotationMatrix.matr_Xz  ; [2xz+2wy]
 ; yz, xy, xz, wy, wz, wx
 fld st(5)                   ; wx, yz, xy, xz, wy, wz, wx
 fsub st,st(1)               ; wx-yz, yz, xy, xz, wy, wz, wx
 fxch                        ; yz, wx-yz, xy, xz, wy, wz, wx
 faddp st(6),st              ; wx-yz, xy, xz, wy, wz, wx+yz
 fadd st,st(0)               ; 2*(wx-yz), xy, xz, wy, wz, wx+yz
 fstp dword ptr [si].Camera_RotationMatrix.matr_Yz  ; [2wx-2yz]
 ; xy, xz, wy, wz, wx+yz
 faddp st(3),st              ; xz, wy, xy+wz, wx+yz
 fsubrp st(1),st             ; xz-wy, xy+wz, wx+yz
 fadd st,st(0)               ; 2xz-2wy
 fstp dword ptr [si].Camera_RotationMatrix.matr_Zx  ; [2xz-2wy]
 fadd st,st(0)               ; 2xy+2wz, wx+yz
 fchs                        ; -2xy-2wz, wx+yz
 fstp dword ptr [si].Camera_RotationMatrix.matr_Yx  ; [-2xy-2wz]
 fadd st,st(0)               ; 2wx+2yz
 fchs                        ; -2yz-2wx
 fstp dword ptr [si].Camera_RotationMatrix.matr_Zy  ; [-2yz-2wx]
 leave
 ret
CopyThisKeyframe:
 Num=0
 REPT 3
 mov eax,dword ptr es:[di].Key_Position[Num]
 mov edx,dword ptr es:[di].Key_Rotation[Num]
 mov dword ptr [si].Camera_Position[Num],eax
 mov dword ptr [si].Camera_QuatRot[Num],edx
 Num=Num+4
 ENDM
 mov eax,dword ptr es:[di].Key_Rotation[Num]
 mov dword ptr [si].Camera_QuatRot[Num],eax
 mov word ptr [CurrentTrackPointPtr],di
 jmp @@CalcMatrix
InterpolateTrack endp


IFDEF _TRACKDEBUG
 EVEN
 ; Temporal cameras needed for the track's debugging
 EulerDebugCam Camera <>
 QuatDebugCam  Camera <>
 ; Integer matrices
 EulerInt         Matrix <>
 QuatInt          Matrix <>
ENDIF
