/*--------------------------------------------------------------------------
 * File: vesa16.c
 * Written by: Fredrik Kling, 1998-01-27
 * Description: A VESA driver for any 16 bit resolution.
 *
 * Updates:
 * -- Date -- | ----- Name ----- |-- Did what....
 * 1998-04-13 | Fredrik Kling    | 15/16/24 bits support!!
 * 1998-04-xx | Gnilk & Krikkit  | Diffrent features for Xcopy/Xmove buff...
 * 1998-01-27 | Fredrik Kling    | 16 bit implementation...
 * 1997-08-16 | Fredrik Kling    | Added xcopybuff (eXtended buffer copy...)
 * 1997-05-27 | Alexander Boczar | Added a pragma pack(1) to be sure to have matching vesa structures
 *
 * TODO: fixa till alla special rutiner frn dx16.c
 *
 *
 -------------------------------------------------------------------------------*/

#include <conio.h>
#include <i86.h>
#include <dos.h>
#include <float.h>
#include "system/dpmi.h"
#include "system/xstdio.h"
#include "system/xstring.h"
#include "drivers/drv16.h"
#include "draw16/draw16.h"

#pragma pack( push, 1)

/* ######################################################################### */

typedef struct s_VESAINFO VESAINFO;
typedef struct s_MODEINFO MODEINFO;
typedef struct s_GFXMODE GFXMODE;
typedef enum
{
	ERR_NOERR,
	ERR_NODOSMEM,
	ERR_NOMEM,
	ERR_VESAERR,
	ERR_NOVESA,
	ERR_INVMODE,
	ERR_MAPFAIL,
	ERR_NOMOUSE
} ERRORS;

struct s_VESAINFO
{
  BYTE vbesignature[4];		// VBE Signature
	union
	{
		WORD vbeversion;				// VBE Version
		struct
		{
			BYTE vbeversion_low;
			BYTE vbeversion_high;
		};
	};
  BYTE *oemstringptr;			// Pointer to OEM String
	union
	{
		DWORD capabilities;			// Capabilities of graphics cont.
		struct
		{
			DWORD capabilities_notfixwidth:1;
			DWORD capabilities_notvgacomp:1;
			DWORD capabilities_specialramdac:1;
			DWORD capabilities_1:29;
		};
	};
  WORD *videomodeptr;			// Pointer to Video Mode List
  WORD totalmemory;				// Number of 64kb memory blocks

	// Added for VBE 2.0

	union
	{
		WORD oemsoftwarerev;				// VBE implementation Software revision
		struct
		{
			BYTE oemsoftwarerev_low;
			BYTE oemsoftwarerev_high;
		};
	};
  BYTE *oemvendornameptr;	// Pointer to Vendor Name String
  BYTE *oemproductnameptr;// Pointer to Product Name String
  BYTE *oemproductrevptr;	// Pointer to Product Revision String
  BYTE reserved[222];			// Reserved for VBE implementation, scratch area
  BYTE oemdata[256];			// Data Area for OEM Strings
};

struct s_MODEINFO
{
	// Mandatory information for all VBE revisions
	union
	{
		WORD modeattributes;		// mode attributes
		struct
		{
			WORD modeattributes_hdwsup:1;
			WORD modeattributes_1:1;
			WORD modeattributes_biossup:1;
			WORD modeattributes_color:1;
			WORD modeattributes_gfx:1;
			WORD modeattributes_novgacomp:1;
			WORD modeattributes_novgacompmem:1;
			WORD modeattributes_linear:1;
			WORD modeattributes_2:8;
		};
	};
	union
	{
		BYTE winaattributes;		// window A attributes
		struct
		{
			BYTE winaattributes_relocatable:1;
			BYTE winaattributes_readable:1;
			BYTE winaattributes_writeable:1;
			BYTE winaattributes_1:5;
		};
	};
	union
	{
		BYTE winbattributes;		// window B attributes
		struct
		{
			BYTE winbattributes_relocatable:1;
			BYTE winbattributes_readable:1;
			BYTE winbattributes_writeable:1;
			BYTE winbattributes_1:5;
		};
	};
  WORD wingranularity;		//	window granularity
  WORD winsize;						//	window size
  WORD winasegment;				//	window A start segment
  WORD winbsegment;				//	window B start segment
  void *winfuncptr;				//	pointer to window function
  WORD bytesperscanline;	//	bytes per scan line

  // Mandatory information for VBE 1.2 and above
  WORD xresolution;				//	horizontal resolution in pixels or chars
  WORD yresolution;				//	vertical resolution in pixels or chars
  BYTE xcharsize;					//	character cell width in pixels
  BYTE ycharsize;					//	character cell height in pixels
  BYTE numberofplanes;		//	number of memory planes
  BYTE bitsperpixel;			//	bits per pixel
  BYTE numberofbanks;			//	number of banks
  BYTE memorymodel;				//	memory model type
  BYTE banksize;					//	bank size in KB
  BYTE numberofimagepages;//	number of images
  BYTE reserved;					//	reserved for page function

  // Direct Color fields (required for direct/6 and YUV/7 memory models)
  BYTE redmasksize;				//	size of direct color red mask in bits
  BYTE redfieldposition;	//	bit position of lsb of red mask
  BYTE greenmasksize;			//	size of direct color green mask in bits
  BYTE greenfieldposition;//	bit position of lsb of green mask
  BYTE bluemasksize;			//	size of direct color blue mask in bits
  BYTE bluefieldposition;	//	bit position of lsb of blue mask
  BYTE rsvdmasksize;			//	size of direct color reserved mask in bits
  BYTE rsvdfieldposition;	//	bit position of lsb of reserved mask
  BYTE directcolormodeinfo;//	direct color mode attributes

  // Mandatory information for VBE 2.0 and above
  BYTE *physbaseptr;			//	physical address for flat frame buffer
  BYTE *offscreenmemoffset;//	pointer to start of off screen memory
  WORD offscreenmemsize;	//	amount of off screen memory in 1k units
  BYTE _reserved[206]; 		//	remainder of modeinfoblock
};

struct s_GFXMODE
{
	int width,height,bpp;
  int red_shift,green_shift,blue_shift;
  int linear;
	DWORD mode;
};

#pragma pack( pop)

extern DRV vesa16DRV;

/* ######################################################################### */

static ERRORS error = ERR_NOERR;

static int mouseactive = FALSE;

static DWORD modes = 0;
static GFXMODE modelist[256];

static WORD *videoptr;
static DWORD bytesperline;
static WORD RGB_r[256],RGB_g[256],RGB_b[256];



WORD *getcopy_of_screen (WORD *buffer)
{
	memcpy (buffer,videoptr,640*480*2);
	return buffer;
}
static void *segofs2ptr( void *segofs)
{
	return( (void *)((((DWORD)segofs >> 12) & 0xffff0) + ((DWORD)segofs & 0xffff)));
}

/* ######################################################################### */
static void fixcoltab (int rs, int gs, int bs,int bpp)
{
	float ra,ga,ba;
	float max_r,max_b,max_g;
	int i;

	if (bpp==15)
	{
		max_r = 31.0;
		max_g = 31.0;
		max_b = 31.0;
	} else
		if (bpp==16)
		{
			max_r = 31.0;
			max_g = 63.0;
			max_b = 31.0;
		} else if ((bpp==24) || (bpp==32))
			{
				max_r = max_g = max_b = 256;
			}



	ra = max_r / 256.0;
	ga = max_g / 256.0;
	ba = max_b / 256.0;

  // replace with stuff found in vesa_info...

	for (i=0;i<256;i++)
	{
    RGB_r[i]=(int)(i*ra) << (rs);  // red_shift
    RGB_g[i]=(int)(i*ga) << (gs);  // green_shift
    RGB_b[i]=(int)(i*ba) << (bs);  // blue_shift
	}

}

static int setup()
{
	RMREGS regs;
	VESAINFO far *vesainfo;
	MODEINFO far *modeinfo;
	WORD vesainfo_sel,vesainfo_seg;
	WORD modeinfo_sel,modeinfo_seg;
	WORD *modeptr;
	int i;


	if( !(_dpmi_dosalloc( sizeof( VESAINFO), &vesainfo_sel, &vesainfo_seg)) )
	{
		error = ERR_NODOSMEM;
		return( FALSE);
	}

	vesainfo = MK_FP( vesainfo_sel, 0);
	vesainfo->vbesignature[0] = 'V';
	vesainfo->vbesignature[1] = 'B';
	vesainfo->vbesignature[2] = 'E';
	vesainfo->vbesignature[3] = '2';


	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f00;
	regs.es = vesainfo_seg;
	regs.di = 0;
	_dpmi_realmodeint( 0x10, &regs);

	if( regs.ax != 0x004f)
	{
		_dpmi_dosfree( vesainfo_sel);
		error = ERR_NOVESA;
		return( FALSE);
	}

	if( !(_dpmi_dosalloc( sizeof( MODEINFO), &modeinfo_sel, &modeinfo_seg)) )
	{
		_dpmi_dosfree( vesainfo_sel);
		error = ERR_NODOSMEM;
		return( FALSE);
	}
	modeinfo = MK_FP( modeinfo_sel, 0);

	modes = 0;
	modeptr = (WORD *)segofs2ptr( vesainfo->videomodeptr);
	while( *modeptr != 0xffff)
	{
		memset( &regs, 0, sizeof( RMREGS));
		regs.ax = 0x4f01;
		regs.cx = *modeptr;
		regs.es = modeinfo_seg;
		regs.di = 0;
		_dpmi_realmodeint( 0x10, &regs);

/*
		if( regs.ax != 0x004f)
		{
			_dpmi_dosfree( modeinfo_sel);
			_dpmi_dosfree( vesainfo_sel);
			error = ERR_VESAERR;
			return( FALSE);
		}
*/
		/*
		 *
		 * 0x04 - packed pixel mode!!
		 *
		 * check if 8 bps...
		 *
		 */
    if (regs.ax == 0x004f)
    {


//      if( modeinfo->bitsperpixel == 16)
      {
/*
        printf ("%4ix%4i-%2ibit - ",modeinfo->xresolution,modeinfo->yresolution,modeinfo->bitsperpixel);
        if (!modeinfo->modeattributes_linear) printf ("banked");
          else printf ("linear");
*/
 /*
        printf ("    RGB info: %i,%i,%i\n",
                modeinfo->redfieldposition,  //  bit position of lsb of red mask
                modeinfo->greenfieldposition,//  bit position of lsb of green mask
                modeinfo->bluefieldposition); //  bit position of lsb of blue mask
 */


        modelist[modes].width = modeinfo->xresolution;
        modelist[modes].height = modeinfo->yresolution;
        modelist[modes].mode = *modeptr;
				modelist[modes].bpp = modeinfo->bitsperpixel;
        modelist[modes].red_shift   = modeinfo->redfieldposition;  //  bit position of lsb of red mask
        modelist[modes].green_shift = modeinfo->greenfieldposition;//  bit position of lsb of green mask
        modelist[modes].blue_shift  = modeinfo->bluefieldposition; //  bit position of lsb of blue mask

        if (!modeinfo->modeattributes_linear) modelist[modes].linear = FALSE;
          else modelist[modes].linear = TRUE;

        modes++;
      }
    } // if (regs.ax == 0x004f)
    modeptr++;
  } // while...

	_dpmi_dosfree( modeinfo_sel);
	_dpmi_dosfree( vesainfo_sel);


	vesa16DRV.modes = modes;
	vesa16DRV.modelist = (DRVMODE *)xmalloc( modes * sizeof( DRVMODE));
	for( i=0; i<modes; i++)
  {
    vesa16DRV.modelist[i].width = modelist[i].width;
		vesa16DRV.modelist[i].height = modelist[i].height;
		vesa16DRV.modelist[i].bpp = modelist[i].bpp;

  }

  return (TRUE);
}


static int init( int width, int height, int config)
{
	RMREGS regs;
	MODEINFO far *modeinfo;
	WORD modeinfo_sel,modeinfo_seg;
	DWORD mode = 0;
	int i,bpp;


	if( modes == 0)
    if(!setup())
			return( FALSE);

	//vesa16DRV.palette = (RGB *)xmalloc( 256 * sizeof( RGB));

	if (config & DRVCFG_15BITS) bpp = 15;
	else if (config & DRVCFG_16BITS) bpp = 16;
	else if (config & DRVCFG_24BITS) bpp = 24;
	else if (config & DRVCFG_32BITS) bpp = 32;

	for( i=0; i<modes; i++)
	{
		if( (modelist[i].width == width) && (modelist[i].height == height) && (modelist[i].bpp == bpp))
		{
			mode = modelist[i].mode;
			break;
		}
	}

	if( mode == 0)
	{
		error = ERR_INVMODE;
		return(FALSE);
	}

	if( !(_dpmi_dosalloc( sizeof( MODEINFO), &modeinfo_sel, &modeinfo_seg)) )
	{
		error = ERR_NODOSMEM;
		return( FALSE);
	}

	modeinfo = MK_FP( modeinfo_sel, 0);
	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f01;
	regs.cx = mode | 0x4000;
	regs.es = modeinfo_seg;
	regs.di = 0;
	_dpmi_realmodeint( 0x10, &regs);

	if( regs.ax != 0x004f)
	{
		_dpmi_dosfree( modeinfo_sel);
		error = ERR_VESAERR;
		return( FALSE);
	}

	videoptr = (WORD *)modeinfo->physbaseptr;
	bytesperline = modeinfo->bytesperscanline;
	if ( (videoptr = _dpmi_mapmem( videoptr, bytesperline * height)) == NULL)
	{
		_dpmi_dosfree( modeinfo_sel);
		error = ERR_MAPFAIL;
		return( FALSE);
	}

	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f02;
	regs.bx = mode | 0x4000;
	_dpmi_realmodeint( 0x10, &regs);

	vesa16DRV.width  = modeinfo->xresolution;
	vesa16DRV.height = modeinfo->yresolution;
	vesa16DRV.minx = vesa16DRV.miny = 0;
	vesa16DRV.maxx = vesa16DRV.width;
	vesa16DRV.maxy = vesa16DRV.height - 1;
	vesa16DRV.config = config;

  fixcoltab (modeinfo->redfieldposition,
             modeinfo->greenfieldposition,
             modeinfo->bluefieldposition,
						 modeinfo->bitsperpixel);

  _dpmi_dosfree( modeinfo_sel);

  return TRUE;
}

static int reinit(int width, int height)
{
	RMREGS regs;
	MODEINFO far *modeinfo;
	WORD modeinfo_sel,modeinfo_seg;
	DWORD mode = 0;
	int i;

	for( i=0; i<modes; i++)
	{
		if( (modelist[i].width == width) && (modelist[i].height == height))
		{
			mode = modelist[i].mode;
			break;
		}
	}

	if( mode == 0)
	{
		error = ERR_INVMODE;
		return(FALSE);
	}

	if( !(_dpmi_dosalloc( sizeof( MODEINFO), &modeinfo_sel, &modeinfo_seg)) )
	{
		error = ERR_NODOSMEM;
		return( FALSE);
	}
	modeinfo = MK_FP( modeinfo_sel, 0);

	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f01;
	regs.cx = mode | 0x4000;
	regs.es = modeinfo_seg;
	regs.di = 0;
	_dpmi_realmodeint( 0x10, &regs);

	if( regs.ax != 0x004f)
	{
		_dpmi_dosfree( modeinfo_sel);
		error = ERR_VESAERR;
		return( FALSE);
	}

	videoptr = (WORD *)modeinfo->physbaseptr;
	bytesperline = modeinfo->bytesperscanline;
	if ( (videoptr = _dpmi_mapmem( videoptr, bytesperline * height)) == NULL)
	{
		_dpmi_dosfree( modeinfo_sel);
		error = ERR_MAPFAIL;
		return( FALSE);
	}

	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f02;
	regs.bx = mode | 0x4000;
	_dpmi_realmodeint( 0x10, &regs);

	vesa16DRV.width = modeinfo->xresolution;
	vesa16DRV.height = modeinfo->yresolution;
	vesa16DRV.minx = vesa16DRV.miny = 0;
	vesa16DRV.maxx = vesa16DRV.width;
	vesa16DRV.maxy = vesa16DRV.height - 1;

	_dpmi_dosfree( modeinfo_sel);

	mouseinit( 0, 0, vesa16DRV.width - 1, vesa16DRV.height - 1);
	setpalette( NULL);

	return( TRUE);
}

static void exit()
{
	union REGS r;

	r.h.ah = 0x0;
	r.h.al = 0x03;
	int386( 0x10, &r, &r);
}

static void setpalette( RGB *newpal)
{
	RMREGS regs;
	BGRA far *tmppal;
	WORD tmppal_sel,tmppal_seg;
  int i;

	if( newpal != NULL)
		memcpy( vesa16DRV.palette, newpal, sizeof(RGB) * 256);

	_dpmi_dosalloc( 256 * sizeof( BGRA), &tmppal_sel, &tmppal_seg);
	tmppal = MK_FP( tmppal_sel, 0);

	for( i=0; i<256; i++)
	{
		tmppal[i].r = vesa16DRV.palette[i].r >> 2;
		tmppal[i].g = vesa16DRV.palette[i].g >> 2;
		tmppal[i].b = vesa16DRV.palette[i].b >> 2;
		tmppal[i].a = 0;
	}

	memset( &regs, 0, sizeof( RMREGS));
	regs.ax = 0x4f09;
	regs.bx = 0x0000;
	regs.cx = 256;
	regs.dx = 0;
	regs.es = tmppal_seg;
	regs.di = 0;
	_dpmi_realmodeint( 0x10, &regs);

	_dpmi_dosfree( tmppal_sel);

}

static void vsync()
{
/*
	while ( !(inp( 0x3da) & 0x08));
	while ( (inp( 0x3da) & 0x08));
*/
}

static void clearscreen( RGBA col)
{
	int y;
	DWORD dcol;
	WORD *screen;

	dcol = (RGB_r[col.r] | RGB_g[col.g] | RGB_b[col.b]);

	if( vesa16DRV.width == bytesperline>>1)
    memset4( videoptr, dcol | (dcol<<16), (vesa16DRV.width * vesa16DRV.height) >> 1);
	else
	{
		screen = videoptr;
		for( y=0; y<vesa16DRV.height; y++)
		{
      memset4( screen, dcol | (dcol<<16), vesa16DRV.width);
			screen += (bytesperline>>1);
		}
	}
}

/* ######################################################################### */

static BUFF *createbuff( int width, int height)
{
	BUFF *buff, *oldactive;
	int i;

	if( (buff = (BUFF *)xmalloc( sizeof(BUFF))) == NULL)
	{
		error = ERR_NOMEM;
		return( NULL);
	}

	buff->drv = &vesa16DRV;

	if( (buff->image = (RGBA *)xmalloc( sizeof( RGBA) * width * height)) == NULL)
	{
		xfree( buff);
		error = ERR_NOMEM;
		return( NULL);
	}

	if( (buff->ytab = (int *)xmalloc( sizeof( int) * height)) == NULL)
	{
		xfree( buff->image);
		xfree( buff);
		error = ERR_NOMEM;
		return( NULL);
	}

	if( vesa16DRV.config & DRVCFG_ZBUFFER)
	{
    if( (buff->zbuffer = (int *)xmalloc( sizeof( int) * width * height)) == NULL)
		{
			xfree( buff->ytab);
			xfree( buff->image);
			xfree( buff);
			error = ERR_NOMEM;
			return( NULL);
		}
	}

	for (i=0; i<height; i++)
		buff->ytab[i] = i * width;

	buff->width = width;
	buff->height = height;

	buff->xorigo = width/2;
	buff->yorigo = height/2;

	buff->minx = 0;
	buff->maxx = width - 1;
	buff->miny = 0;
	buff->maxy = height - 1;

	buff->lighttab = NULL;
	buff->transtab = NULL;

	oldactive = vesa16DRV.activebuff;
	vesa16DRV.activebuff = buff;
	clearbuff();
	vesa16DRV.activebuff = oldactive;

	return( buff);
}

static void destroybuff( BUFF *buff)
{
	if( buff != NULL)
	{
		if( buff->zbuffer != NULL) xfree( buff->zbuffer);
		if( buff->ytab != NULL) xfree( buff->ytab);
		if( buff->image != NULL) xfree( buff->image);
		xfree( buff);
	}
}

static void setbuff( BUFF *buff)
{
	vesa16DRV.activebuff = buff;
}

static void xcopybuff (int x, int y, int dis, int tmp, BYTE *msk)
{
	int sxs,ixs,xlen;
  int sys,iys,ylen,dis_ofs;
	BYTE r,g,b;
	register int i,j;
	WORD w,bw,bh;


	// Check boundaries...

	bw = vesa16DRV.activebuff->width;
	bh = vesa16DRV.activebuff->height;


	if ( x >= vesa16DRV.minx )
	{
		sxs = x;
		ixs = 0;
		if ( (x + bw /*dx16DRV.activebuff->width*/) <= vesa16DRV.maxx)
			xlen = bw; //dx16DRV.activebuff->width;
		else
			xlen = vesa16DRV.maxx - x;
	}
	else
	{
		sxs = vesa16DRV.minx;
		ixs = vesa16DRV.minx - x;
		if ( (x + bw/*dx16DRV.activebuff->width*/) <= vesa16DRV.maxx)
			xlen = (x + bw /*dx16DRV.activebuff->width*/) - vesa16DRV.minx;
		else
			xlen = vesa16DRV.maxx - vesa16DRV.minx;
	}
	// Check Y

	if ( y >= vesa16DRV.miny )
	{
		sys = y;
		iys = 0;
		if ( (y + bh /*dx16DRV.activebuff->height*/) <= vesa16DRV.maxy)
			ylen = bh /*dx16DRV.activebuff->height*/;
		else
			ylen = vesa16DRV.maxy - y;
	}
	else
	{
		sys = vesa16DRV.miny;
		iys = vesa16DRV.miny - y;
		if ( (y + bh /*dx16DRV.activebuff->height*/) <= vesa16DRV.maxy)
			ylen = (y + bh/*dx16DRV.activebuff->height*/) - vesa16DRV.miny;
		else
			ylen = vesa16DRV.maxy - vesa16DRV.miny;
	}

	if ((vesa16DRV.config & DRVCFG_15BITS) || (vesa16DRV.config & DRVCFG_16BITS))
	{
		WORD *screen,*pscreen;
  	BYTE *image, *mask;

	  switch (tmp)
	  {
	    case 0:
	        // double pixels...  plot 4x4 pixels to screen.
	        for ( j=0; j<ylen; j++)
	        {
	          image = (BYTE *)&vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
	          screen = &videoptr[ sxs + vesa16DRV.width * (2*j + sys)];
	          pscreen  =&videoptr[ sxs + vesa16DRV.width * (2*j + sys+1)];
	          for ( i=0; i<xlen; i++)
	          {
	            r = image[0];
	            g = image[1];
	            b = image[2];
	            w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
	            *(screen) = w;
	            *(screen+1) = w;

	            *(pscreen) = w;
	            *(pscreen+1) = w;

	            screen += 2;
	            pscreen += 2;
	            image+=sizeof (RGBA);

	          }
	        }
	        break;
	    case 1:
	        // With displacement...  plot 1 of 4x4 pixels to screen
	        dis = dis & 3;
	        switch(dis)
	        {
	          case 0 : dis_ofs=0; break;
	          case 1 : dis_ofs=1; break;
	          case 2 : dis_ofs=1+vesa16DRV.width; break;
	          case 3 : dis_ofs=vesa16DRV.width; break;
	          default : dis_ofs = 0; break;
	        }
	        if(msk == NULL)
	          for(j=0 ; j<ylen ; j++)
	          {
	            image = (BYTE *)&vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
	            screen = &videoptr[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
	            for ( i=0; i<xlen; i++)
	            {
	              r = image[0];
	              g = image[1];
	              b = image[2];
	              w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
	              *(screen) = w;

	              screen += 2;
	              image  += sizeof(RGBA);
	            }
	          }
	        else
	          for(j=0; j<ylen; j++)
	          {
	            image  = (BYTE*) &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
	            screen = &videoptr[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
	            mask   = &msk[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
	            for(i=0 ; i<xlen ; i++)
	            {
	              if(*mask)
	              {
	                r = image[0];
	                g = image[1];
	                b = image[2];
	                w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
	                *(screen) = w;
	              }
	              mask   += 2;
	              screen += 2;
	              image  += sizeof(RGBA);
	            }
	          }
	        break;
	    case 2:
	        if(msk == NULL)
	          for(j=0 ; j<ylen ; j++)
	          {
	            image  = (BYTE*) &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
	            screen = &videoptr[sxs + vesa16DRV.width * (j + sys)];
	            for(i=0 ; i<xlen ; i++)
	            {
	              r = image[0];
	              g = image[1];
	              b = image[2];
	              w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
	              //if (w>0) *(screen) = w;
	              *(screen) = w;

	              ++screen;
	              image+=sizeof (RGBA);
	            }
	          }
	        else
	          for(j=0 ; j<ylen ; j++)
	          {
	            image  = (BYTE*) &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
	            screen = &videoptr[sxs + vesa16DRV.width * (j + sys)];
	            for(i=0 ; i<xlen ; i++)
	            {
	              if(*msk)
	              {
	                r = image[0];
	                g = image[1];
	                b = image[2];
	                w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
	                //if (w>0) *(screen) = w;
	                *(screen) = w;
	              }

	              ++msk;
	              ++screen;
	              image+=sizeof (RGBA);
	            }
	          }
	        break;
	  } // switch

	} else
		{
			RGBA *screen,*pscreen,*image;
			RGBA *vptr;
			RGBA col;
  		BYTE *mask;

			vptr = (RGBA *)videoptr;

		  switch (tmp)
		  {
		    case 0:
		        // double pixels...  plot 4x4 pixels to screen.
		        for ( j=0; j<ylen; j++)
		        {

		          image = &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
		          screen = &vptr[ (sxs + vesa16DRV.width * (2*j + sys))];
		          pscreen  = &vptr[sxs + vesa16DRV.width * (2*j + sys+1)];
		          for ( i=0; i<xlen; i++)
		          {
								col = image[0];

								*(screen)=col;
								*(screen+1)=col;
								*(pscreen)=col;
								*(pscreen+1)=col;

		            screen++;
		            pscreen++;

		            image++;

		          }
		        }
		        break;
		    case 1:
		        // With displacement...  plot 1 of 4x4 pixels to screen
		        dis = dis & 3;
		        switch(dis)
		        {
		          case 0 : dis_ofs=0; break;
		          case 1 : dis_ofs=1; break;
		          case 2 : dis_ofs=1+vesa16DRV.width; break;
		          case 3 : dis_ofs=vesa16DRV.width; break;
		          default : dis_ofs = 0; break;
		        }
		        if(msk == NULL)
		          for(j=0 ; j<ylen ; j++)
		          {
		            image = &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
		            screen = &vptr[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
		            for ( i=0; i<xlen; i++)
		            {
									*screen=*image;
									screen+=2;
									image++;
		            }
		          }
		        else
		          for(j=0; j<ylen; j++)
		          {
		            image  = &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
		            screen = &vptr[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
		            mask   = (BYTE *)&msk[dis_ofs + sxs + vesa16DRV.width * (2*j + sys)];
		            for(i=0 ; i<xlen ; i++)
		            {
		              if(*mask) *screen = *image;
		              mask   += 2;
		              screen += 2;
		              image++;
		            }
		          }
		        break;
		    case 2:
		        if(msk == NULL)
		          for(j=0 ; j<ylen ; j++)
		          {
		            image  =  &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
		            screen =  &vptr[sxs + vesa16DRV.width * (j + sys)];
								memcpy (screen,image,xlen);
		          }
		        else
		          for(j=0 ; j<ylen ; j++)
		          {
		            image  = &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
		            screen = &vptr[sxs + vesa16DRV.width * (j + sys)];
		            for(i=0 ; i<xlen ; i++)
		            {
		              if(*msk) *screen = *image;

		              msk++;
			            screen++;
		              image++;
		            }
		          }
		        break;

		  } // switch

		} // else 24 bits
}

static void xmovebuff (int x, int y, int displace, int tmp, RGBA col)
{
	int sxs,ixs,xlen;
	int sys,iys,ylen,ofs;
	BYTE r,g,b;
	register int i,j;
	WORD w,bw,bh;
	BYTE *image;

	//clw = (RGB_r[col.r] | RGB_g[col.g] | RGB_b[col.b]);


	// Check boundaries...

	bw = vesa16DRV.activebuff->width;
	bh = vesa16DRV.activebuff->height;

	switch (displace)
	{
		case 0 : ofs = 0; break;
		case 1 : ofs = 1; break;
		case 2 : ofs = 1+vesa16DRV.width; break;
		case 3 : ofs = vesa16DRV.width; break;
	}

	if ( x >= vesa16DRV.minx )
	{
		sxs = x;
		ixs = 0;
		if ( (x + bw /*dx16DRV.activebuff->width*/) <= vesa16DRV.maxx)
			xlen = bw; //dx16DRV.activebuff->width;
		else
			xlen = vesa16DRV.maxx - x;
	}
	else
	{
		sxs = vesa16DRV.minx;
		ixs = vesa16DRV.minx - x;
		if ( (x + bw/*dx16DRV.activebuff->width*/) <= vesa16DRV.maxx)
			xlen = (x + bw /*dx16DRV.activebuff->width*/) - vesa16DRV.minx;
		else
			xlen = vesa16DRV.maxx - vesa16DRV.minx;
	}
	// Check Y

	if ( y >= vesa16DRV.miny )
	{
		sys = y;
		iys = 0;
		if ( (y + bh /*dx16DRV.activebuff->height*/) <= vesa16DRV.maxy)
			ylen = bh /*dx16DRV.activebuff->height*/;
		else
			ylen = vesa16DRV.maxy - y;
	}
	else
	{
		sys = vesa16DRV.miny;
		iys = vesa16DRV.miny - y;
		if ( (y + bh /*dx16DRV.activebuff->height*/) <= vesa16DRV.maxy)
			ylen = (y + bh/*dx16DRV.activebuff->height*/) - vesa16DRV.miny;
		else
			ylen = vesa16DRV.maxy - vesa16DRV.miny;
	}


	if ((vesa16DRV.config & DRVCFG_15BITS) || (vesa16DRV.config & DRVCFG_16BITS))
	{
		WORD *screen;
		int yofs;
		yofs = sys * vesa16DRV.width;

		for ( j=0; j<ylen; j++,yofs+=2*vesa16DRV.width)
		{
			image = (BYTE *)&vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
			screen = &videoptr[ofs+sxs + yofs];
			for ( i=0; i<xlen; i++)
			{
				r = image[0];
				g = image[1];
				b = image[2];
				w = (RGB_r[r] | RGB_g[g] | RGB_b[b]);
				*(screen) = w;
				*(int *)image=(int)0;
				screen += 2;
				image+=sizeof (RGBA);
			}
		}
	} else
		if (vesa16DRV.config & DRVCFG_32BITS)
		{
			RGBA *screen,*vptr;
			RGBA *r_image;
			int yofs;

			vptr = (RGBA *) videoptr;
			yofs = sys * vesa16DRV.width;

			for ( j=0; j<ylen; j++,yofs += 2*vesa16DRV.width)
			{
				r_image = (RGBA *)&vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
				screen = &vptr[ofs+sxs + yofs];
				for ( i=0; i<xlen; i++)
				{
					*(screen) = *(r_image);
					screen += 2;
					*r_image=col;
					r_image++;
				}
			}

		}
	if ( vesa16DRV.activebuff->zbuffer != NULL) //  & DRVCFG_ZBUFFER)
    memset4( vesa16DRV.activebuff->zbuffer, 0x7fffffff, (vesa16DRV.activebuff->width * vesa16DRV.activebuff->height));

}

static void copybuff( int x, int y)
{

/*

	if( (x == y == 0) && (vesa16DRV.activebuff->width == vesa16DRV.width) && (vesa16DRV.activebuff->height == vesa16DRV.height))
	{
		BYTE *image = vesa16DRV.activebuff->image;
		WORD *screen = videoptr;

		for ( j=0; j<vesa16DRV.height; j++)
		{
			if( (vesa16DRV.width & 0x0f) == 0)
			{
				//memcpy16( screen, image, vesa16DRV.width >> 4);
				image += vesa16DRV.width;
				screen += bytesperline;
			}
			else if( (vesa16DRV.width & 0x03) == 0)
			{
				//memcpy4( screen, image, vesa16DRV.width >> 2);
				image += vesa16DRV.width;
				screen += bytesperline;
			}
			else
			{
				//memcpy4( screen, image, vesa16DRV.width >> 2);
				image += vesa16DRV.width ^ 0x03;
				screen += vesa16DRV.width ^ 0x03;

				for ( i=0; i<(vesa16DRV.width & 0x03); i++)
					*(screen++) = *(image++);
				image += vesa16DRV.width & 0x03;
				screen += vesa16DRV.width & 0x03 + (bytesperline - vesa16DRV.width);
			}
		}
	}
	else
{ */
		int sxs,ixs,xlen;
		int sys,iys,ylen;
		register int i,j;
		WORD r,g,b;

		// Check X

		if ( x >= vesa16DRV.minx )
		{
			sxs = x;
			ixs = 0;
			if ( (x + vesa16DRV.activebuff->width) <= vesa16DRV.maxx)
				xlen = vesa16DRV.activebuff->width;
			else
				xlen = vesa16DRV.maxx - x;
		}
		else
		{
			sxs = vesa16DRV.minx;
			ixs = vesa16DRV.minx - x;
			if ( (x + vesa16DRV.activebuff->width) <= vesa16DRV.maxx)
				xlen = (x + vesa16DRV.activebuff->width) - vesa16DRV.minx;
			else
				xlen = vesa16DRV.maxx - vesa16DRV.minx;
		}

		// Check Y

		if ( y >= vesa16DRV.miny )
		{
			sys = y;
			iys = 0;
			if ( (y + vesa16DRV.activebuff->height) <= vesa16DRV.maxy)
				ylen = vesa16DRV.activebuff->height;
			else
				ylen = vesa16DRV.maxy - y;
		}
		else
		{
			sys = vesa16DRV.miny;
			iys = vesa16DRV.miny - y;
			if ( (y + vesa16DRV.activebuff->height) <= vesa16DRV.maxy)
				ylen = (y + vesa16DRV.activebuff->height) - vesa16DRV.miny;
			else
				ylen = vesa16DRV.maxy - vesa16DRV.miny;
		}

	if ((vesa16DRV.config & DRVCFG_15BITS) || (vesa16DRV.config & DRVCFG_16BITS))
	{

		BYTE *image;
		WORD *screen;
		//if (xlen>200) xlen++;
		for ( j=0; j<ylen; j++)
		{
			image = (BYTE *)&vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
			screen = &videoptr[ sxs + vesa16DRV.width * (j + sys)];
			for ( i=0; i<xlen; i++)
			{
				r = image[0];
				g = image[1];
				b = image[2];


				*(screen) =  (RGB_r[r] | RGB_g[g] | RGB_b[b]);

				screen++;
				image += sizeof (RGBA);

			}
		}
	} else
		if (vesa16DRV.config & DRVCFG_32BITS)
		{
			RGBA *screen,*vptr,*image;
			vptr = (RGBA *)videoptr;
			//if (xlen>200) xlen++;

			for (j=0; j<ylen; j++)
			{
				image = &vesa16DRV.activebuff->image[ ixs + vesa16DRV.activebuff->ytab[j + iys]];
				screen = &vptr[ sxs + vesa16DRV.width * (j + sys)];
				memcpy4 (screen,image,xlen);
			}



		}
}

static void clearbuff( RGBA col)
{
	DWORD dcol;
	memcpy4 ((void *)&dcol,(void *)&col,1);
  memset4( vesa16DRV.activebuff->image, dcol, (vesa16DRV.activebuff->width * vesa16DRV.activebuff->height));
	//if ( vesa16DRV.config & DRVCFG_ZBUFFER)
	if (vesa16DRV.activebuff->zbuffer != NULL)
    memset4( vesa16DRV.activebuff->zbuffer, 0x7fffffff, (vesa16DRV.activebuff->width * vesa16DRV.activebuff->height));
}

static void movebuff( int x, int y, RGBA col)
{
  copybuff( x, y);
  clearbuff( col);
}

/* ######################################################################### */

static void _plot( XYZ p, RGBA col)
{
  plot16( vesa16DRV.activebuff, p, col);
}

static void _line( XYZ p1, XYZ p2, RGBA col)
{
  line16( vesa16DRV.activebuff, p1, p2, col);
}

static void _poly( XYZ p1, XYZ p2, XYZ p3, RGBA col )
{
  poly16( vesa16DRV.activebuff, p1, p2, p3, col);
}

static void _tpoly( XYZ p1, XYZ p2, XYZ p3, BYTE *data )
{
  tpoly16( vesa16DRV.activebuff, p1, p2, p3, data );
}

static void _blit( XYZ p, RGBA *image, int *zimage, int width, int height)
{
  blit16( vesa16DRV.activebuff, p, image, zimage, width, height);
}

/* ######################################################################### */

static int mouseinit( int minx, int miny, int maxx, int maxy)
{
	union REGS r;

	if( mouseactive)
	{
		r.w.ax = 0x0007;
		r.w.cx = minx;
		r.w.dx = maxx;
		int386( 0x33, &r, &r);

		r.w.ax = 0x0008;
		r.w.cx = miny;
		r.w.dx = maxy;
		int386( 0x33, &r, &r);
		return( TRUE);
	}

	r.w.ax = 0x0;
	int386( 0x33, &r, &r);
	if ( r.w.ax == 0x0ffff)
	{
		r.w.ax = 0x0007;
		r.w.cx = minx;
		r.w.dx = maxx;
		int386( 0x33, &r, &r);

		r.w.ax = 0x0008;
		r.w.cx = miny;
		r.w.dx = maxy;
		int386( 0x33, &r, &r);

		r.w.ax = 0x0004;
		r.w.cx = minx + ((maxx - minx) >> 1);
		r.w.dx = miny + ((maxy - miny) >> 1);
		int386( 0x33, &r, &r);

		mouseactive = TRUE;
		return( TRUE);
	}
	else
	{
		mouseactive = FALSE;
		error = ERR_NOMOUSE;
		return( FALSE);
	}
}

static int mousecallback( void ( *callback)( DRVMOUSE change))
{
	return( TRUE);
}

static void getmousepos( int *x, int *y)
{
	union REGS r;

	if( mouseactive)
	{
		r.w.ax = 0x0003;
		int386( 0x33, &r, &r);
    *x = (signed short)r.w.cx;
    *y = (signed short)r.w.dx;
	}
}

static void setmousepos( int x, int y)
{
	union REGS r;

	if( mouseactive)
	{
		r.w.ax = 0x0004;
		r.w.cx = x;
		r.w.dx = y;
		int386( 0x33, &r, &r);
	}
}

static void getmousediff( int *x, int *y)
{
	union REGS r;

	if( mouseactive)
	{
		r.w.ax = 0x000b;
		int386( 0x33, &r, &r);
		*x = (signed short)r.w.cx;
		*y = (signed short)r.w.dx;
	}
}


/* ######################################################################### */

static char getkey()
{
	return( getch());
}

/* ######################################################################### */

static char *geterror()
{
	ERRORS olderr = error;
	error = ERR_NOERR;
	switch( olderr)
	{
		case ERR_NOMEM: return( "Not enough memory");
		case ERR_NOMOUSE: return( "Mouse driver not found");
    case ERR_VESAERR: return ("Error while detecting!");
		default : return( "Unknown error");
	}
}

/* ######################################################################### */


DRV vesa16DRV =
{
	"Vesa 1.2/2.0 16bitDriver",

	0,
	NULL,

	0,0,
	0,0,0,0,
	1,1,

	0,/*config*/

	NULL,

	NULL,

	setup,
	init,
	reinit,
	exit,

	setpalette,
	vsync,
	clearscreen,

	createbuff,
	destroybuff,
	setbuff,

	xcopybuff,
	xmovebuff,
	copybuff,
	clearbuff,
	movebuff,

  _plot,
  _line,
  _poly,
  _tpoly,
  _blit,

	mouseinit,
	mousecallback,
	getmousepos,
	setmousepos,
	getmousediff,

	kbhit,
	getkey,

	geterror,
};














