/*
  UNIT VGALIB
  GargonScene 3: Curso de programacin grfica.
  
   BIBLIOTECA PARA USO EN MODO 13H. GargonScene n3.                 
  
*/
#ifndef __STDIO_H
  #include <STDIO.H>
#endif

/*------ Constantes --------------------------------------------------------*/
#define DIR_VIDEO       0xA0000000L
#define VideoX          320
#define VideoY          200
#define Colors          256

/*------ Declaracin de las funciones de la librera. ----------------------*/
void PutPixel (int X, int Y, char Color, unsigned char *Where);
char GetPixel(int X, int Y, unsigned char *Where);

void Box      (int X, int Y, int X1, int Y1, char Color, unsigned char *Where);
void FillBox  (int X, int Y, int Ancho, int Alto, char Color, unsigned char *Where);
void Circle   (int X, int Y, int Radio, char Color, unsigned char *Where);
void LineH    (int X, int Y, int Ancho, char Color, unsigned char *Where);
void LineV    (int X, int Y, int Alto, char Color, unsigned char *Where);
void Line(int x1, int y1, int x2, int y2, char Color, unsigned char *Where);

void Flip     (unsigned char *Source);
void FlipTo   (unsigned char *Source, unsigned char *Where);
void Cls      (unsigned char *Where);

void GetSprite(int X, int Y, int Ancho, int Alto, unsigned char *Source, unsigned char *Sprite);
void PutSprite(int X, int Y, unsigned char *Sprite, unsigned char *Where);
void PutSpriteMul4(int X, int Y, unsigned char *Sprite, unsigned char *Where);
void PutSpriteMask(int X, int Y, unsigned char *Sprite, unsigned char *Where);
char PutSpriteClipMask(int X, int Y, unsigned char *Sprite, unsigned char *Where);

char GetPcx   (char *Fichero, unsigned char *Where, unsigned char *Palette, unsigned long int Offset);
char GetRaw   (char *Fichero, unsigned char *Where, unsigned long int Offset);

void IniMode  (char Mode);
char GetMode  ();
char ThisVga  ();
void WRetrace ();
void SetBorder(char Color);

int Sgn(int x);
int Abs(int x);
int ABS(int x);

FILE *Fp;
unsigned char *Video=(unsigned char *) DIR_VIDEO;

/*------ Cabezera del PCX -------------------------------------------------*/
struct PcxHeader
	{
	 char Manufact;
	 char Version;
	 char Encode;
	 char BPP;
	 int  xmin,ymin;
	 int  xmax,ymax;
	 int  Hres;
	 int  Vres;
	 char Palette16[48];
	 char Reserved;
	 char Color_planes;
	 int  Bytes_per_line;
	 int  Palette_type;
	 char Filler[58];
	}PcxHead;

/*------ PutPixel() --------------------------------------------------------
   Esta funcin es bsica. Pone un pixel en el lugar indicado (Where) y en
 el sitio indicado (X,Y) del color dado (Color). Si queremos ponerlo direc-
 tamente en pantalla si pantallas virutuales, le pasamos en *Where el pun-
 tero Video, p.e. PutPixel(2,3,15,Video);
 ---------------------------------------------------------------------------*/
void PutPixel(int X, int Y, char Color, unsigned char *Where)
{
 asm les di, [Where]            /* Cargamos direccin efectiva del lugar a
				   poner el pixel. */
 asm mov ax, Y                  /* Movemos en 'ax' la posicin Y */
 asm mov di, ax                 /* Ahora 'ax'=='di' */
 asm shl ax, 8                  /* Multiplicamos por 320 mediante desplaza- */
 asm shl di, 6                  /* mientos (ms rpido). 256+64=320. */
 asm add di, ax                 /* Sumamos a 'di' el contenido de 'ax' */
 asm add di, X                  /* Sumamos  X a 'di' */

 asm mov al, Color              /* 'al'=Color a poner. */
 asm mov es:[di], al            /* Ya tenemos offset, ponemos pixel. */
}

/*------ GetPixel() --------------------------------------------------------
	Lee un pixel de una posicin determinada y lo devuelve en formato
 unsigned char.
 ---------------------------------------------------------------------------*/
char GetPixel(int X, int Y, unsigned char *Where)
{
 asm les di, [Where]            /* Cargamos direccin efectiva del lugar a
				   poner el pixel. */
 asm mov ax, Y                  /* Movemos en 'ax' la posicin Y */
 asm mov di, ax                 /* Ahora 'ax'=='di' */
 asm shl ax, 8                  /* Multiplicamos por 320 mediante desplaza- */
 asm shl di, 6                  /* mientos (ms rpido). 256+64=320. */
 asm add di, ax                 /* Sumamos a 'di' el contenido de 'ax' */
 asm add di, X                  /* Sumamos  X a 'di' */

 asm mov al, es:[di]

 return _AL;
}

/*------ Box() ------------------------------------------------------------
  Dibuja un cuadrado desde la posicin X,Y hasta la X1,Y1 del color especi-
  ficado en un lugar (*Where).
 --------------------------------------------------------------------------*/
void Box(int X, int Y, int X1, int Y1, char Color, unsigned char *Where)
{
 LineV(X1,Y,Y1-Y+1,Color,Where);        /* Dibujamos la primera linea vertical. */
 LineV(X,Y,Y1-Y,Color,Where);           /* Dibujamos la segunda linea vertical. */
 LineH(X,Y,X1-X,Color,Where);           /* Dibujamos la primera linea horizontal. */
 LineH(X,Y1,X1-X,Color,Where);          /* Dibujamos la segunda linea horizontal. */
}

/*------ FillBox() --------------------------------------------------------
  Dibuja un cuadrado relleno del color especificado desde la posicin X,Y
  hasta la X1,Y1.
 --------------------------------------------------------------------------*/
void FillBox(int X, int Y, int Ancho, int Alto, char Color, unsigned char *Where)
{
 asm push es
 asm push di
 asm les di, [Where]                    /* Cargamos direccin destino */

 asm mov ax, Y                          /* 'ax' ahora tiene valor de Y */
 asm mov di, ax                         /* 'di' =  'ax' Ms rpido trabajar con registros... ;) */
 asm shl di, 8                          /* Multiplicamos por 320 mediante des- */
 asm shl ax, 6                          /* plazamientos. */
 asm add di, ax                         /* Aadimos resultado de ax shl 6 a di */
 asm add di, X                          /* Aadimos pos. X */

 asm mov al, Color                      /* Cargamos color */
 asm mov dx, Alto                       /* En dx metemos altura del cuadrado */
 asm mov bx, VideoX                     /* En bx 320 (VideoX) */
 asm sub bx, Ancho

Bucle1:
 asm mov cx, Ancho
 asm rep stosb
 asm add di, bx
 asm dec dx
 asm jnz Bucle1
 asm pop di
 asm pop es
}

/*------ Circle() ---------------------------------------------------------
 Dibuja un circulo en el lugar deseado del color especificado.
      __
    /  R \              R= Radio del crculo.
   |  ---|             = Coordenada (x,y) del crculo
    \    /
      --

 Entrada:
   X e Y -> Se corresponden al centro del crculo (ver dibujo).
   Radio -> Longitud de la que ser el circulo (r+r) (ver dibujo).
   Color -> Color del circulo.
   *Where -> Lugar de la video memoria en donde se dibujar el crculo.
 --------------------------------------------------------------------------*/
void Circle(int X, int Y, int Radio, char Color, unsigned char *Where)
{
 register int xl, yl;

 if(!Radio)
     PutPixel(X,Y,Color,Where);

 else {
	xl=0;
	yl=Radio;
	Radio=Radio*Radio+1;

	while(yl!=0)
		 {
		  PutPixel(X+xl,Y+yl,Color,Where);
		  PutPixel(X-xl,Y+yl,Color,Where);
		  PutPixel(X+xl,Y-yl,Color,Where);
		  PutPixel(X-xl,Y-yl,Color,Where);

		  if(((xl*xl)+(yl*yl)) >= Radio)
					       yl--;
		  else xl++;
		 }
	PutPixel(X+xl,Y+yl,Color,Where);
	PutPixel(X-xl,Y+yl,Color,Where);
	PutPixel(X+xl,Y-yl,Color,Where);
	PutPixel(X-xl,Y-yl,Color,Where);
     }
}

/*------ LineH() ----------------------------------------------------------
   Dibuja una linea horizontal en el lugar que deseemos.

 Entrada:
    X, Y -> Coordenadas en donde comienza la linea.
    Ancho -> Anchura en pixels de la linea.
    Color -> Color del que ser la linea.
    *Where -> Lugar en donde se dibuja la linea horizontal.
 --------------------------------------------------------------------------*/
void LineH(int X, int Y, int Ancho, char Color, unsigned char *Where)
{
 asm les di, [Where]

 asm mov ax, Y
 asm mov di, ax
 asm shl ax, 8
 asm shl di, 6
 asm add di, ax
 asm add di, X
 asm mov al, Color
 asm mov cx, Ancho
 asm rep stosb
}

/*------ LineV() ----------------------------------------------------------
   Funcin que dibuja una linea vertical en pantalla. (No clipping!!)

 Entrada:
    X, Y -> Coordenadas en donde comienza la linea.
    Alto -> Altura en pixels de la linea.
    Color -> Color del que ser la linea.
    *Where -> Lugar en donde se dibuja la linea vertical.
 --------------------------------------------------------------------------*/
void LineV(int X, int Y, int Alto, char Color, unsigned char *Where)
{
 asm les di, [Where]

 asm mov ax, Y
 asm mov di, ax
 asm shl di, 8
 asm shl ax, 6
 asm add di, ax
 asm add di, X
 asm mov cx, Alto
 asm mov al, Color
 asm mov dx, VideoX

Bucle:
 asm mov es:[di], al
 asm add di, dx
 asm dec cx
 asm jnz Bucle
}

/*------ Line() -----------------------------------------------------------
   Esta funcin dibuja una linea en inclinada en cualquier ngulo. Utiliza
 el algoritmo de Breshenham.

 Entrada:
   x1 e y1 -> Coordenadas del origen (desde) de la linea.
   x2 e y2 -> Coordenadas del destino (hasta) de la linea.
   Color -> Color del que ser la linea.
   *Where -> Lugar en donde se dibuja la linea.
 --------------------------------------------------------------------------*/
void Line(int x1, int y1, int x2, int y2, char Color, unsigned char *Where)
{
 int i, s, d1x, d1y, d2x, d2y, u=x2-x1, v=y2 - y1, m, n;

 d2x = Sgn(u); d1y= Sgn(v);
 d2y = 0;      d1x= Sgn(u);
 m   = Abs(u); n  = Abs(v);
 
 if(m>n==0)
      {
       d2x = 0 ;
       d2y = Sgn(v);
       m = Abs(v);
       n = Abs(u);
      }
 
 s=m>>1;
 for(i=0;i<=m;i++)
       {
	PutPixel(x1,y1,Color,Where);
	s+=n;
	
	if(s<m==0) 
	      {
	       s -= m;
	       x1 += d1x;
	       y1 += d1y;
	      }
	else 
	  {
	   x1 += d2x;
	   y1 += d2y;
	  }
      }
}


/*------ Flip() -----------------------------------------------------------
    Esta funcin vuelca una pantalla VIRTUAL a la videomemoria. Lo que le
 demos a esta se vera inmediatamente en pantalla.

 Entrada:
   *Source -> Pantalla a volcar.
 --------------------------------------------------------------------------*/
void Flip(unsigned char *Source)
{
 asm push ds
 asm push si
 asm lds si, [Source]
 asm mov ax, 0xA000
 asm mov es, ax
 asm xor di, di
 asm mov cx, 16000
 asm cld
 asm db 66h
 asm rep movsw
 asm pop si
 asm pop ds
}

/*------ FlipTo() ---------------------------------------------------------
 Esta funcin vuelca dos pantallas virtuales.

 Entrada:
   *Source -> Pantalla (de 64000 bytes) que se volcar. Origen.
   *Where  -> Pantalla en donde se meter el contenido de *Source. Destino.
 --------------------------------------------------------------------------*/
void FlipTo(unsigned char *Source, unsigned char *Where)
{
 asm push ds
 asm push si
 asm lds si, [Source]
 asm les di, [Where]
 asm mov cx, 16000
 asm cld
 asm db 66h
 asm rep movsw
 asm pop si
 asm pop ds
}

/*------ Cls() ------------------------------------------------------------
   Esta funcin es la encargada de borrar la pantalla, es parecida a lo que
 hariamos con un memset(), lo nico es que no se por qu es como 5 veces ms
 rpida...  ;)

 Entrada:
   *Where -> Nombre del puntero que referencia la pantalla a borrar.
 --------------------------------------------------------------------------*/
void Cls(unsigned char *Where)
{
 asm les di, [Where]                    /* Cargamos direccin del puntero que queremos borrar. */
 asm xor ax, ax                               /* Borramos registro ax con un xor */
 asm mov cx, 16000                      /* Tamao en CX.(64000/4=16000Bytes). */
 asm db 66h
 asm rep stosw
}

/*------ GetSprite() ------------------------------------------------------
   Esta funcin lee un sprite del lugar especifiado, y lo guarda en el lugar
 pasado como puntero *Sprite.

 Formato del sprite:

  Byte   Tamao    Descripcin
  
   1       1byte   Anchura en bytes del sprite.
   2       1byte   Altura total en byte del sprite.
   3       x*x     Datos. Bytes que forman el sprite.

 Entrada:
   X -> Posicin x de donde se empieza a leer.
   Y -> Posicin y de donde se empieza a leer.
   Ancho -> Ancho del sprite en bytes. (1-255)
   Alto -> Altura del sprite en bytes. (1-255)
   *Source -> Lugar donde se encuetra el sprite a leer.
   *Sprite -> Lugar final en donde se almacenar el sprite.
 --------------------------------------------------------------------------*/
void GetSprite(int X, int Y, int Ancho, int Alto, unsigned char *Source, unsigned char *Sprite)
{
 register int y, x;
 unsigned int Posicion=1;
 Sprite[0]=Ancho;
 Sprite[1]=Alto;

 for(y=Y;y<Y+Alto;y++)
	for(x=X;x<X+Ancho;x++)
		  Sprite[++Posicion]=Source[(y<<8)+(y<<6)+x];
}

/*------ PutSprite() ------------------------------------------------------
   Funcin que pone un sprite en la posicin especificada. No tiene ningn
 tipo de comprobacin, ni clipping, ni respeta fondo.

 Entrada:
   X e Y --> Coordenadas en donde se situar al sprite.
   *Sprite --> Sprite a poner.
   *Where  --> Lugar en donde se dibuja el sprite.
 --------------------------------------------------------------------------*/
void PutSprite(int X, int Y, unsigned char *Sprite, unsigned char *Where)
{
 int Ancho=Sprite[0],
     Alto =Sprite[1];

 asm push ds
 asm les di, [Where]
 asm lds si, [Sprite]
 asm inc si
 asm inc si

 asm mov ax, Y
 asm mov di, ax
 asm shl di, 8
 asm shl ax, 6
 asm add di, ax
 asm add di, X
 asm mov dx, Alto
 asm mov bx, VideoX
 asm sub bx, Ancho

Bucle:;
 asm mov cx, Ancho
 asm cld
 asm rep movsb
 asm add di, bx
 asm dec dx
 asm jnz Bucle
 asm pop ds
}

/*------ PutSpriteMul4() --------------------------------------------------
   Esta funcin pone un sprite cuya ANCHURA sea MULTIPLO DE CUATRO. As se
 acelera el proceso y la rutina es el triple de rpida ya que se utilizan
 'movsd' en vez de 'movsb' para poner los sprites (66h + movsw = movsd).

 Entrada:
   X e Y --> Posicin a partir de la cual se pone el sprite.
   *Sprite --> Sprite a volcar.
   *Where  --> Puntero al lugar en donde se dibuja el sprite.

 NOTA: La anchura del sprite (que se encuentra en Sprite[0]) debe de ser for-
       zosamente multiple de 4 (4,8,16,20,24...), si no es as NO FUNCIONARA.
 --------------------------------------------------------------------------*/
void PutSpriteMul4(int X, int Y, unsigned char *Sprite, unsigned char *Where)
{
 int Ancho=Sprite[0]>>2,
     Alto =Sprite[1];

 asm push ds
 asm les di, [Where]
 asm lds si, [Sprite]
 asm inc si
 asm inc si

 asm mov ax, Y
 asm mov di, ax
 asm shl di, 8
 asm shl ax, 6
 asm add di, ax
 asm add di, X
 asm mov dx, Alto
 asm mov bx, VideoX
 asm sub bx, Ancho

Bucle:;
 asm mov cx, Ancho
 asm cld
 asm db 66h
 asm rep movsw
 asm add di, bx
 asm dec dx
 asm jnz Bucle
 asm pop ds
}

/*------ PutSpriteMask() --------------------------------------------------
   Pone un sprite en la posicin dada (X,Y), y en el lugar deseado de la me-
 moria. Tiene el color 0 como mscara para respetar el fondo al dibujar.

 Entrada:
   X e Y --> Coordenadas en donde se quiere poner el sprite.
   *Sprite --> Sprite a volcar.
   *Where --> Lugar de la memoria donde se dibuja el sprite.
 --------------------------------------------------------------------------*/
void PutSpriteMask(int X, int Y, unsigned char *Sprite, unsigned char *Where)
{
 int Ancho=Sprite[0],
     Alto =Sprite[1];

 asm push ds
 asm lds si, [Sprite]
 asm les di, [Where]
 asm mov di, Y
 asm mov ax, di
 asm shl di, 8
 asm shl ax, 6
 asm add di, ax
 asm add di, X
 asm mov dx, Alto
 asm mov bx, Ancho
 asm xor ah, ah
 asm inc si
 asm inc si

DrawLoopSprite:;
 asm mov cx, bx

LineLoopSprite:;
 asm mov al, byte ptr [si]
 asm cmp al, ah
 asm je NoDraw
 asm mov es:[di], al

NoDraw:;
 asm inc si
 asm inc di
 asm dec cx
 asm jnz LineLoopSprite

 asm add di, VideoX
 asm sub di, bx
 asm dec dx
 asm jnz DrawLoopSprite
 asm pop ds
}

/*------ PutSpriteClipMask() ----------------------------------------------
   Pone un sprite en la posicin indicada y el lugar dado como puntero. A
 este sprite se le controla el 'clipping', vamos, que se le recortan los la-
 dos si hace falta por que se sale de la pantalla. Adems respeta el fondo,
 con el color 0.

 Entrada:
   X --> Posicin X en pantalla del sprite.
   Y --> Posicin Y en pantalla del sprite.
   *Sprite --> Sprite que se desea poner.
   *Where --> Lugar de la memoria donde se quiere poner el sprite.

 No devuelve nada.
 --------------------------------------------------------------------------*/
char PutSpriteClipMask(int X, int Y, unsigned char *Sprite, unsigned char *Where)
{
 int Ancho=Sprite[0],
     Alto =Sprite[1];
 int ClipX=0, ClipY=0, ClipX1=0, ClipY1=0;

 if(X<0) { ClipX= X*(-1); if(ClipX>=Ancho) return; }
 if(Y<0) { ClipY= Y*(-1); if(ClipY>=Alto) return;  }

 if(X+Ancho > 320) { ClipX1= X+Ancho-320; if(ClipX1>=Ancho) return; }
 if(Y+Alto > 200)  { ClipY1= Y+Alto-200;  if(ClipY1>=Alto) return; }

 asm push ds
 asm lds si, [Sprite]
 asm les di, [Where]
 asm mov di, Y
 asm add di, ClipY
 asm mov ax, di
 asm shl di, 8
 asm shl ax, 6
 asm add di, ax
 asm add di, X

 asm mov dx, Alto
 asm sub dx, ClipY1
 asm sub dx, ClipY

 asm mov bx, Ancho
 asm sub bx, ClipX1
 asm sub bx, ClipX

 asm xor ah, ah
 asm inc si
 asm inc si

 asm mov cx, ClipY
 asm cmp cx, 0
 asm je  DrawLoopSprite

ADD:
 asm add si, Ancho
 asm add si, ClipX
 asm add si, ClipX1
 asm dec cx
 asm jnz ADD:;

DrawLoopSprite:;
 asm mov cx, bx
 asm add di, ClipX
 asm add si, ClipX

LineLoopSprite:;
 asm mov al, byte ptr [si]
 asm cmp al, ah
 asm je NoDraw
 asm mov es:[di], al

NoDraw:;
 asm inc si
 asm inc di
 asm dec cx
 asm jnz LineLoopSprite

 asm add si, ClipX1

 asm add di, VideoX
 asm sub di, bx
 asm sub di, ClipX
 asm dec dx
 asm jnz DrawLoopSprite
 asm pop ds

 return 0;
}

/*------ GetPcx() ---------------------------------------------------------
 Lee un fichero grfico pcx de 320X200X256 y lo pone en el lugar especificado.

 Entrada:
   *Fichero --> Puntero al nombre del directorio y archivo en donde se
		 encuentra el fichero a leer.
   *Where   --> Lugar en donde se descomprime la imagen.
   *Palette --> Lugar en donde se almacena la Paleta correspondiente a la
		 imagen del fichero pcx (est en los 768 ultimos bytes del fichero).
   Offset   --> A partir de donde se empieza a leer el fichero.

 Salida:
   1 --> Fichero descomprimido correctamente.
   0 --> No se encontr el fichero.
 --------------------------------------------------------------------------*/
char GetPcx(char *Fichero, unsigned char *Where, unsigned char *Palette, unsigned long int Offset)
{
 unsigned int x, i=0, ysize, xsize;
 unsigned int Points;
 char c;

 if((Fp=fopen(Fichero,"rb"))==NULL)
				  return 0;

 fseek(Fp,Offset,SEEK_SET);
 fread(&PcxHead,sizeof(PcxHead),1,Fp);

 xsize=(PcxHead.xmax-PcxHead.xmin)+1;
 ysize=(PcxHead.ymax-PcxHead.ymin)+1;

 Points=xsize*ysize;

 for(i=0;i<Points;++i)
	  {
	   c=fgetc(Fp) & 0xff;

	   if((c&0xc0)==0xc0)
		       {
			x=c & 0x3f;
			c=fgetc(Fp);
			while(x--)
			   {
			    *(Where++)=c;
			    i++;
			   }
			i--;
		       }
	   else
	      *(Where++)=c;
	  }
 fgetc(Fp);
 fread(Palette,768,1,Fp);

 for(i=0;i<768;i++)
		Palette[i]=Palette[i]/4;
 fclose(Fp);
 return 1;
}

/*------ GetRaw() ---------------------------------------------------------
   Lee un fichero grfico en formato crudo (64000 bytes), y lo pone en la
 direccin de memoria dada.

 Entrada:
   *Fichero --> Puntero al nombre del directorio y archivo en donde se
		 encuentra el fichero a leer.
   *Where   --> Lugar en donde se descomprime la imagen.
   Offset   --> A partir de donde se empieza a leer el fichero.

 Salida:
   1 --> Fichero descomprimido correctamente.
   0 --> No se encontr el fichero.
 --------------------------------------------------------------------------*/
char GetRaw(char *Fichero, unsigned char *Where, unsigned long int Offset)
{
 if((Fp=fopen(Fichero,"rb"))==NULL)
				 return 0;
 if(Offset)
    fseek(Fp, Offset, SEEK_SET);

 fread(Where, 64000U, 1, Fp);

 fclose(Fp);
 return 1;
}

/*------ IniMode() --------------------------------------------------------
   Inicia un modo grfico mediante la int 10h. de BIOS.

 Entrada:
   Mode --> Modo de video a iniciar. (0x3 ->Modo texto | 0x13 ->Modo 320x200)
 --------------------------------------------------------------------------*/
void IniMode(char Mode)
{
 asm xor ah, ah
 asm mov al, Mode
 asm int 10h
}

/*------ GetMode() --------------------------------------------------------
   Nos devuelve un byte con el modo de video actual.

 Salida:
   Modo de video (1byte)
 --------------------------------------------------------------------------*/
char GetMode()
{
 asm mov ah, 0Fh
 asm int 10h
 return _AL;
}

/*------ ThisVga() --------------------------------------------------------
   Comprueba mediante la funcin 1a de la int 10h. de BIOS si hay una tarje-
 ta vga instalada en el ordenador.

 Salida:
   0  --> No encontrada
   1  --> Encontrada tarjeta vga
 --------------------------------------------------------------------------*/
char ThisVga()
{
 asm mov ax, 1a00h
 asm int 10h
 asm cmp al, 1ah
 asm je  Ok
 return 0;

Ok:
 return 1;
}

/*------ SetBorder() ------------------------------------------------------
   Esta funcin cambia el color del borde de la pantalla mediante la funcin
 10h del BIOS. (int. 10h Servicios de video)

 Entrada:
   Color  --> Color (0-255 en modo 13h) del que se quiere el borde.
 --------------------------------------------------------------------------*/
void SetBorder(char Color)
{
 asm mov ax, 1001h
 asm mov bh, Color
 asm int 10h
}

/*------ WRetrace() -------------------------------------------------------
   Esta rutina espera a que acabe el retrazado vertical de la pantalla, as
 evitamos que las pantallas se corten, y sincronizamos mejor.
 --------------------------------------------------------------------------*/
void WRetrace()
{
	 asm mov dx, 3DAh

	et1:
	 asm in  al,dx
	 asm test al, 8
	 asm jnz et1

	et2:
	 asm in  al,dx
	 asm test al, 8
	 asm jz  et2
}


/*
 --------------------------------------------------------------------------
  R U T I N A S     C O M P L E M E N T A R I A S
 --------------------------------------------------------------------------

 Estas rutinas no tinen una funcin en concreto, slo son para apoyar a otras
 funciones que las usan.
*/

int Sgn(int x)
{
 if(x>0)
     return(+1);
 if(x<0)
     return(-1);
 return (0);
}

int ABS(int x)
{
 return (+x);
}

int Abs(int x)
{
 if(x<0)
    return (-x);

 return(x);
}

/*----------- [Fin de la VgaLib] -------------------------------------------*/

