/****************************************************************************
 math3d.c, ver 1.0 (08/07/1997)
 ----------------------------------------------------------------------------
 Written by Timur Davidenko (aka Adept/Esteem), 1997.
 Send any comments/suggestions to: adept@aquanet.co.il
 I`ll be glad to see any responses.
 ----------------------------------------------------------------------------
	Usefull 3D geometry routings.
 ----------------------------------------------------------------------------
																 CREDITS
 Huge thanks to follow people that helped me to get it done:
	 Torgeir Hagland (aka Xanthome),VOR,Wog,some others...

 References:
	"Quaternion Calculus and Fast Animation" -  Ken Shoemake,
	Notes for SIGGRAPH'87 course #10,
	"Computer Animation: 3D Motion Specification and Control".
	Graphics Gems sources collection.
 ----------------------------------------------------------------------------
																DISCALIMER
 I can not be held responsible for any damages this source may produce,
 Use it on your own risk.
 You are free to distribute,copy,use and modify this source code with one
 restriction do not distribute modified copy of this source without firstly
 asking me for permission.
****************************************************************************/

#include <math.h>
#include "math3d.h"

//**************************** Quaternions **********************************
void qset( Quat *q,float w,float x,float y,float z )	{
	q->w = w;
	q->x = x;
	q->y = y;
	q->z = z;
}

void	qscale( Quat q,float s,Quat *dest )	// Scale quaternion by value.
{
	dest->w = q.w*s;
	dest->x = q.x*s;
	dest->y = q.y*s;
	dest->z = q.z*s;
}

float	qmod( Quat q )	// Returns modul of quaternion
{
	float d;
	d = sqrt(q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w);
	if (d == 0) d = 1;	// for some case.
	return d;
}

void qunit( Quat q,Quat *dest )	// Normilize quaternion.
{
	float s;
	s = 1.0/qmod(q);
	qscale( q,s,dest );
}

float	qdot( Quat q1, Quat q2 )	// Returns dot product of q1*q2
{
	float d;
	d = (q1.x*q2.x + q1.y*q2.y + q1.z*q2.z + q1.w*q2.w)/(qmod(q1)*qmod(q2));
	return d;
}

// Returns dot product of normilized quternions q1*q2
float	qdotunit( Quat q1, Quat q2 )	{
	return q1.x*q2.x + q1.y*q2.y + q1.z*q2.z + q1.w*q2.w;
}

//	Calculates quaternion product dest = q1 * q2.
void qmul( Quat q1, Quat q2, Quat *dest ) {
	dest->w = q1.w*q2.w - q1.x*q2.x - q1.y*q2.y - q1.z*q2.z;
	dest->x = q1.w*q2.x + q1.x*q2.w + q1.y*q2.z - q1.z*q2.y;
	dest->y = q1.w*q2.y + q1.y*q2.w + q1.z*q2.x - q1.x*q2.z;
	dest->z = q1.w*q2.z + q1.z*q2.w + q1.x*q2.y - q1.y*q2.x;
}

void qinv( Quat q, Quat *dest )	//  Multiplicative inverse of q.
{
	float d;
	d = q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w;
	if (d != 0) d = 1.0/d;	else	d = 1;
	dest->w =  q.w * d;
	dest->x = -q.x * d;
	dest->y = -q.y * d;
	dest->z = -q.z * d;
}

void qnegate( Quat *q )	// Negates q;
{
	float d;
	d = 1.0/qmod(*q);
	q->w *=  d;
	q->x *= -d;
	q->y *= -d;
	q->z *= -d;
}

void qexp( Quat q, Quat *dest ) // Calculate quaternion`s exponent.
{
	float d,d1;
	d = sqrt( q.x*q.x + q.y*q.y + q.z*q.z );
	if (d > 0) d1 = sin(d)/d; else d1 = 1;
	dest->w = cos(d);
	dest->x = q.x*d1;
	dest->y = q.y*d1;
	dest->z = q.z*d1;
}

void qlog( Quat q, Quat *dest )	// Calculate quaternion`s logarithm.
{
	float d;
	d = sqrt( q.x*q.x + q.y*q.y + q.z*q.z );
	if (q.w != 0.0)	d = atan(d/q.w);	else	d = M_PI_2;
	dest->w = 0.0;
	dest->x = q.x*d;
	dest->y = q.y*d;
	dest->z = q.z*d;
}

// Calculate logarithm of the relative rotation from p to q
void qlndif( Quat p, Quat q, Quat *dest )
{
	Quat inv,dif;
	float d,d1;
	float s;

	qinv( p,&inv );			// inv = -p;
	qmul( inv,q,&dif );	// dif = -p*q

	d = sqrt( dif.x*dif.x + dif.y*dif.y + dif.z*dif.z );
	s = p.x*q.x + p.y*q.y + p.z*q.z + p.w*q.w;
	if (s != 0)	d1 = atan(d/s);	else	d1 = M_PI_2;
	if (d != 0) d1 /= d;

	dest->w = 0;

	dest->x = dif.x*d1;
	dest->y = dif.y*d1;
	dest->z = dif.z*d1;
}

// Converts quaternion to matrix.
void quat2matrix( Quat q,Matrix *M )	{
	float d,s,xs,ys,zs,wx,wy,wz,xx,xy,xz,yy,yz,zz;

	d = q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w;
	if (d == 0.0) s = 1.0; else s = 2.0/d;

	xs = q.x * s;   ys = q.y * s;  zs = q.z * s;
	wx = q.w * xs;  wy = q.w * ys; wz = q.w * zs;
	xx = q.x * xs;  xy = q.x * ys; xz = q.x * zs;
	yy = q.y * ys;  yz = q.y * zs; zz = q.z * zs;

	M->xx = 1.0 - (yy + zz);
	M->xy = xy - wz;
	M->xz = xz + wy;

	M->yx = xy + wz;
	M->yy = 1.0 - (xx + zz);
	M->yz = yz - wx;

	M->zx = xz - wy;
	M->zy = yz + wx;
	M->zz = 1.0 - (xx + yy);
}

// Converts quaternion to inverse matrix.
void quat2inverseMatrix( Quat q,Matrix *M )	{
	float d,s,xs,ys,zs,wx,wy,wz,xx,xy,xz,yy,yz,zz;

	d = q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w;
	if (d == 0.0) s = 1.0; else s = 2.0/d;

	xs = q.x * s;   ys = q.y * s;  zs = q.z * s;
	wx = q.w * xs;  wy = q.w * ys; wz = q.w * zs;
	xx = q.x * xs;  xy = q.x * ys; xz = q.x * zs;
	yy = q.y * ys;  yz = q.y * zs; zz = q.z * zs;

	M->xx = 1.0 - (yy + zz);
	M->xy = xy + wz;
	M->xz = xz - wy;

	M->yx = xy - wz;
	M->yy = 1.0 - (xx + zz);
	M->yz = yz + wx;

	M->zx = xz + wy;
	M->zy = yz - wx;
	M->zz = 1.0 - (xx + yy);
}

// Converts Axis-Angle quaternion representation to classic form.
// Note: Angle should be in radians.
void	qAxisAngle2quat( float x,float y,float z,float angle,Quat *q )	{
	float s;
	s = sin(angle/2);
	q->x = x*s;
	q->y = y*s;
	q->z = z*s;
	q->w = cos(angle/2);
}

// Converts Axis-Angle quaternion representation to classic form.
// Note: Angle should be in degres.
void	qAxisAngleDeg2quat( float x,float y,float z,float angle,Quat *q )	{
	float s;
	angle *= M_PI/(2*180.0);	// convert angle to radians / 2.
	s = sin(angle);
	q->x = x*s;
	q->y = y*s;
	q->z = z*s;
	q->w = cos(angle);
}

// Converts classic quaternion representation to Angle-Axis form.
void quat2AngleAxis( Quat q, float *x,float *y,float *z,float *angle ) {
	int i;
	float halfang,s;
	Quat qn;

	qunit( q,&qn );
	halfang = acos( qn.w );
	*angle = 2.0*halfang;
	s = sin(halfang);
	*x = qn.x/s;
	*y = qn.y/s;
	*z = qn.z/s;
}

// Converts Matrix to quaternion.
void matrix2quat( Matrix *m, Quat *q )	{
	float s,v;
	int i,j,k;

	v = m->xx + m->yy + m->zz;
	if (v > 0.0) {
		s = sqrt(v + 1.0);
		q->w = 0.5 * s;
		s = 0.5 / s;
		q->x = (m->zy - m->yz) * s;
		q->y = (m->xz - m->zx) * s;
		q->z = (m->yx - m->xy) * s;
	}	else {
		if (m->yy > m->xx)	{
			if (m->zz > m->yy)	{
				s = sqrt( (m->zz - (m->xx + m->yy)) + 1.0 );
				q->z = s * 0.5;
				if (s != 0.0)	s = 0.5/s;
				q->w = (m->yx - m->xy) * s;
				q->x = (m->xz + m->zx) * s;
				q->y = (m->yz + m->zy) * s;
			}	else	{
				s = sqrt( (m->yy - (m->zz + m->xx)) + 1.0 );
				q->y = s * 0.5;
				if (s != 0.0)	s = 0.5/s;
				q->w = (m->xz - m->zx) * s;
				q->z = (m->zy + m->yz) * s;
				q->x = (m->xy + m->yx) * s;
			}
		}	else
		if (m->zz > m->xx)	{
			s = sqrt( (m->zz - (m->xx + m->yy)) + 1.0 );
			q->z = s * 0.5;
			if (s != 0.0)	s = 0.5/s;
			q->w = (m->yx - m->xy) * s;
			q->x = (m->xz + m->zx) * s;
			q->y = (m->yz + m->zy) * s;
		}	else	{
			s = sqrt( (m->xx - (m->yy + m->zz)) + 1.0 );
			q->x = s * 0.5;
			if (s != 0.0)	s = 0.5/s;
			q->w = (m->zy - m->yz) * s;
			q->y = (m->yx + m->xy) * s;
			q->z = (m->zx + m->xz) * s;
		}
	}
}

#define	EPSILON 1.0E-06

void	qslerp( Quat a,Quat b, Quat *dest, float time,float spin )	{
	double k1,k2;					// interpolation coefficions.
	double angle;					// angle between A and B
	double angleSpin;			// angle between A and B plus spin.
	double sin_a, cos_a;	// sine, cosine of angle
	int flipk2;						// use negation of k2.

	cos_a = qdotunit( a,b );
	if (cos_a < 0.0) 	cos_a = -cos_a, flipk2 = -1;  else flipk2 = 1;

	if ((1.0 - cos_a) < EPSILON) {
		k1 = 1.0 - time;
		k2 = time;
	} else {				/* normal case */
		angle = acos(cos_a);
		sin_a = sin(angle);
		angleSpin = angle + spin*M_PI;
		k1 = sin( angle - time*angleSpin ) / sin_a;
		k2 = sin( time*angleSpin ) / sin_a;
	}
	k2 *= flipk2;

	dest->x = k1*a.x + k2*b.x;
	dest->y = k1*a.y + k2*b.y;
	dest->z = k1*a.z + k2*b.z;
	dest->w = k1*a.w + k2*b.w;
}

void	qslerplong( Quat a,Quat b, Quat *dest, float time,float spin )	{
	double k1,k2;					// interpolation coefficions.
	double angle;					// angle between A and B
	double angleSpin;			// angle between A and B plus spin.
	double sin_a, cos_a;	// sine, cosine of angle
	int flipk2;						// use negation of k2.

	cos_a = qdotunit( a,b );

	if (1.0 - fabs(cos_a) < EPSILON) {
		k1 = 1.0 - time;
		k2 = time;
	} else {				/* normal case */
		angle = acos(cos_a);
		sin_a = sin(angle);
		angleSpin = angle + spin*M_PI;
		k1 = sin( angle - time*angleSpin ) / sin_a;
		k2 = sin( time*angleSpin ) / sin_a;
	}

	dest->x = k1*a.x + k2*b.x;
	dest->y = k1*a.y + k2*b.y;
	dest->z = k1*a.z + k2*b.z;
	dest->w = k1*a.w + k2*b.w;
}