
;ͻ;
; Heap management.                                                          ;
;ͼ;

COMMENT #

RAW/XMS :

  Initialization - A large 'heap pool' is used.  The linear address is
  Heap_Base.  pFirst_Handle is the offset of the first handle.  The
  first handle contains a link to the last handle (last 4 bytes of
  'pool').  Last 4 bytes of 'pool' contain a terminator (00000000h) as a
  handle link.

  Functionality - Each 'heap block' consists of a 'handle' and the
  memory requested.  A 'handle' consists of a link to the next 'handle'
  and the size of the current 'block' (including 'handle' size).

  Allocation - Each handle and its link is tested to determine whether
  the area between is large enough.  If there is sufficient space, a
  link is inserted into the list.  If the terminator is ever reached, a
  memory block of sufficient size was not available.

  Release - Each handle is tested against the address of the memory
  'block' to be released.  When a match is determined, the link is
  removed from the list.


DPMI:

  Initialization - A 200h*8 byte block of memory is allocated to contain
  the heap 'handles'.  All handles are initialized to free (offset
  00000000h).

  Functionality - A 'heap handle pool' is used to hold all the handles.
  There may only be N handles available at a time.  If more handles are
  requested the 'pool' is resized at run-time.  The 'pool' is never
  shrunk.

  Allocation - The 'heap handle pool' is scaned for a free handle
  (offset 00000000h).  DPMI memory allocation is used.  The DPMI memory
  handle and address of block are stored in the 'heap handle'.

  Release - The 'heap handle pool' is scaned for a matching block
  offset.  When a match is determined, the memory is freed through DPMI
  memory release.

#



DPMI_HANDLESPERBLOCK  EQU (200h)  ; number of handles per DPMI handle block
                                  ; (DPMI allocations are usually aligned
                                  ;  on a 4Kb boundary.  Each handle requires
                                  ;  8 bytes.  8*200h = 4Kb.)

  ; structure for heap handles
HeapHandleTYPE STRUC
  hh_pLink        LABEL DWORD     ; link to next block handle (RAW/XMS)
  hh_BlockHandle  LABEL DWORD     ; handle for block (DPMI)
                  dd ?

  hh_BlockSize    LABEL DWORD     ; size of block in memory (RAW/XMS)
  hh_pBlock       LABEL DWORD     ; pointer to block (DPMI)
                  dd ?
ENDS


;*****************************************************************************;
Code32  SEGMENT
  ASSUME CS:Code32, DS:Code32, SS:Code32

; imformation relating to handles
Heap_Size         LABEL   DWORD   ; size of heap (RAW/XMS)
Heap_Handle       LABEL   DWORD   ; DPMI handle for memory block (DPMI)
                  dd      0

Heap_Base         LABEL   DWORD   ; address of memory block (RAW/XMS)
Handle_Total      LABEL   DWORD   ; total number of handles available (DPMI)
                  dd      100000h

pFirst_Handle     LABEL   DWORD   ; address of first handle (RAW/XMS)
pHeap_Handles     LABEL   DWORD   ; address of handle block (DPMI)
                  dd      0


; error messages
HeapAllocErrorMSG db 'Memory allocation error.$'
HeapReleaseErrorMSG db 'Memory release error.$'


;Ŀ;
; Allocate memory block                                                     ;
; Entry: EDX - Size of memory block                                         ;
; Exit : EAX - Pointer to memory block                                      ;
;;
Alloc             LABEL   DWORD   ; pointer to procedure
                  dd      o RAWXMS_Alloc


;Ŀ;
; Release memory block                                                      ;
; Entry: EAX - Pointer to memory block                                      ;
;;
Release           LABEL   DWORD   ; pointer to procedure
                  dd      o RAWXMS_Release


;Ŀ;
; Amount of free memory available                                           ;
; Exit : EAX - The amount of free memory available                          ;
;;
MemAvail          LABEL   DWORD   ; pointer to procedure
                  dd      o RAWXMS_MemAvail


;Ŀ;
; Initialize heap manager                                                   ;
;;
Init_Heap:

  cmp pm_EntryType,pm_ENTRY_RAWXMS
  jz @@RAWXMS

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  @@DPMI:

  ; update function pointers
  mov m Alloc,o DPMI_Alloc
  mov m Release,o DPMI_Release
  mov m MemAvail,o DPMI_MemAvail

  ; allocate memory block from DPMI
  xor ebx,ebx
  mov cx,DPMI_HANDLESPERBLOCK*8
  mov ax,0501h
  int 31h
  jnc @@AllocatedMemory
    mov edx,o HeapAllocErrorMsg
    jmp ErrorExit32
  @@AllocatedMemory:

  ; save handle
  mov m w Heap_Handle,di
  mov m w Heap_Handle+2,si

  ; save offset of memory block
  shl ebx,16
  mov bx,cx
  sub ebx,m Code32addr
  mov m pHeap_Handles,ebx

  ; initialize all handles
  mov edi,ebx
  mov ecx,DPMI_HANDLESPERBLOCK*8/4
  xor eax,eax
  rep stosd

  ; set number of handles in 'pool'
  mov Handle_Total,DPMI_HANDLESPERBLOCK

  ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  @@RAWXMS:

  ; calculate offset of first and last handle
  mov eax,m Heap_Base
  sub eax,m Code32addr          ; relative to Code32 Segment
  mov edi,eax                   ; copy for last handle
  add eax,1Fh                   ; 32 byte alignment (for cache coherency)
  and eax,NOT 1Fh
  mov m pFirst_Handle,eax       ; save first handle offset
  add edi,m Heap_Size
  sub edi,4                     ; offset of last handle

  ; setup first handle
  mov [eax.hh_pLink],edi        ; link to last handle
  mov [eax.hh_BlockSize],20h    ; size of 32 (must include size of handle)

  ; setup last handle
  mov [edi.hh_pLink],0          ; link to 00000000h (handle terminator)

  ret


;Ŀ;
; RAW/XMS version of Alloc                                                  ;
;;
RAWXMS_Alloc:

  pushad

  ; add size of handle (32 bytes) so that size of block includes handle
  ; align size on 32 byte boundary
  ; so that block sizes are in 32 byte chunks (for cache coherency)
  add edx,3Fh
  and edx,NOT 1Fh

  ; search for memory block of sufficient size
  mov esi,m pFirst_Handle       ; current handle
  mov edi,[esi.hh_pLink]        ; forward handle
  @@Search:
    ; determine size between current block and forward block
    mov eax,edi                 ;   start of next block
    sub eax,esi                 ; - current block
    sub eax,[esi.hh_BlockSize]  ; - block size (includes handle size)
    cmp eax,edx                 ; = area between handles
    jae @@Found

    ; next block
    mov esi,edi
    mov edi,[edi.hh_pLink]
    test edi,edi
    jnz @@Search

    ; insufficient memory available
    mov edx,o HeapAllocErrorMSG
    jmp ErrorExit32

  ; large enough block found
  @@Found:

  ; point EBX to offset of handle
  mov ebx,esi                 ; previous handle
  add ebx,[esi.hh_BlockSize]  ; + size of block

  ; insert link
  mov [esi.hh_pLink],ebx
  mov [ebx.hh_pLink],edi
  mov [ebx.hh_BlockSize],edx

  ; point EBX to offset of memory block
  add ebx,20h

  mov [esp+28],ebx              ; return EBX as EAX

  popad

  ret


;Ŀ;
; RAW/XMS version of Release                                                ;
;;
RAWXMS_Release:

  push edi esi

  ; offset points to handle
  sub eax,20h

  ; search for handle that matches address
  mov esi,m pFirst_Handle
  mov edi,[esi.hh_pLink]
  @@Search:
    cmp eax,edi
    jz @@Found
    mov esi,edi
    mov edi,[edi.hh_pLink]
    test edi,edi
    jnz @@Search

    mov edx,o HeapReleaseErrorMSG
    jmp ErrorExit32

  @@Found:

  ; remove link
  mov eax,[edi.hh_pLink]
  mov [esi.hh_pLink],eax

  pop esi edi

  ret


;Ŀ;
; RAW/XMS version of MemAvail                                               ;
;;
RAWXMS_MemAvail:
  push edi esi

  ; zero memory available counter
  xor eax,eax

  mov esi,m pFirst_Handle   ; current handle
  mov edi,[esi.hh_pLink]    ; forward handle
  @@Search:
    ; determine size between current block and forward block
    add eax,edi
    sub eax,esi
    sub eax,[esi.hh_BlockSize]

    ; next block
    mov esi,edi
    mov edi,[edi.hh_pLink]
    test edi,edi
    jnz @@Search

  pop esi edi

  ret



;Ŀ;
; Resizes the DPMI heap handle pool                                         ;
;;
DPMI_Resize_HandleBlock:

  ; increase size by minimum amount
  add m Handle_Total,DPMI_HANDLESPERBLOCK

  ; resize block
  mov edx,m Handle_Total
  shl edx,3                     ; Handle is 8 bytes
  shld ebx,edx,16               ; BX:CX is new size (Total Handles * 8 bytes)
  mov ecx,edx

  mov di,m w Heap_Handle
  mov si,m w Heap_Handle+2      ; SI:DI is old handle
  mov ax,0503h
  int 31h
  jnc @@AllocatedMemory
    mov edx,o HeapAllocErrorMsg
    jmp ErrorExit32
  @@AllocatedMemory:

  ; save handle
  mov m w Heap_Handle,di
  mov m w Heap_Handle+2,si

  ; save location
  shl ebx,16
  mov bx,cx
  sub ebx,m Code32addr
  mov m pHeap_Handles,ebx

  ; initialize all new handles
  mov edi,m Handle_Total
  mov ecx,DPMI_HANDLESPERBLOCK*8/4
  lea edi,[edi*8+ebx-DPMI_HANDLESPERBLOCK*8]
  xor eax,eax
  rep stosd

  popad

;  jmp DPMI_Alloc


;Ŀ;
; DPMI version of Alloc                                                     ;
;;
DPMI_Alloc:

  pushad

  ; search for free handle
  mov ecx,m Handle_Total
  mov esi,m pHeap_Handles
  @@DetermineFreeHandle:
    lodsd                       ; DPMI Handle
    lodsd                       ; address of block
    test eax,eax                ; if address is 00000000h then handle is free
    jz @@FoundHandle
    loop @@DetermineFreeHandle

    ; insufficient handles
    ; resize DPMI Handle block and try again
    jmp DPMI_Resize_HandleBlock


  ; found empty handle
  @@FoundHandle:
  sub esi,8
  mov ebp,esi

  ; if there isn't 16 byte over read present allocate another page
  add edx,16

  ; allocate memory block from DPMI
  shld ebx,edx,16
  mov ecx,edx
  mov ax,0501h
  int 31h
  jnc @@AllocatedMemory
    mov edx,o HeapAllocErrorMsg
    jmp ErrorExit32
  @@AllocatedMemory:

  ; store block information

  ; store handle
  mov w [ebp.hh_BlockHandle  ],di
  mov w [ebp.hh_BlockHandle+2],si

  ; store pointer
  mov eax,ebx
  shl eax,16
  mov ax,cx
  sub eax,m Code32addr
  mov [ebp.hh_pBlock],eax

  mov [esp+28],eax              ; return EAX

  popad

  ret


;Ŀ;
; DPMI version of Release                                                   ;
;;
DPMI_Release:

  pushad

  ; search for handle that matches address
  mov ebx,m pHeap_Handles
  mov ecx,m Handle_Total
  @@Search:
    cmp eax,[ebx.hh_pBlock]
    jz @@Found
    add ebx,8
    loop @@Search

    mov edx,o HeapReleaseErrorMSG
    jmp ErrorExit32

  @@Found:

  ; release DPMI memory
  mov di,w [ebx.hh_BlockHandle  ]
  mov si,w [ebx.hh_BlockHandle+2]
  mov ax,0502h
  int 31h
  jnc @@NoError
    mov edx,o HeapReleaseErrorMSG
    jmp ErrorExit32
  @@NoError:

  ; free handle
  mov d [ebx.hh_pBlock],0

  popad

  ret


;Ŀ;
; DPMI version of MemAvail                                                  ;
;;
DPMI_MemAvail:

  push edi

  ; allocate memory for free memory information table
  sub esp,30h

  ; get free memory info
  mov edi,esp
  mov ax,0500h
  int 31h

  mov eax,[edi+14h]
  shl eax,12

  add esp,30h

  pop edi

  ret




;*****************************************************************************;
Code32 ENDS
