#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "sample.h"

// constructors and destructors
Sample::Sample(void)
  : error_message(0),
    name(0),
    data(0),
    length(0),
    loop_start(0),
    loop_end(0),
    frequency(0),
    finetune(0),
    relative_note(0),
    flags(0),
    volume(64), // dont set the volume to zero
    panning(0)
{
}

Sample::~Sample(void)
{
  dump();
}

// This only tests the filename extension
bool Sample::is_sample(const char *filename) 
{
  if (strstr(filename,".iff") ||
      strstr(filename,".voc") ||
      strstr(filename,".wav") ||   
      strstr(filename,".raw") ||
      strstr(filename,".aif") ||
      strstr(filename,".pcm") ||
      strstr(filename,".snd") ||
      strstr(filename, ".au"))
    return true;
  
  return false;
}

int Sample::dump(void)
{
  if (name)
    {
      free(name);
      name = 0;
    }
  if (data)
    {
      free(data);
      data = 0;
    }
  return 1;
}
//////////////////////////////////////////////////////////////////////////////
// DECODERS AND ENCODERS FOR VARIOUS FORMATS
//////////////////////////////////////////////////////////////////////////////
// decodeMuLawSample, decodeMuLaw, decodeALawSample, decodeALaw, 
// decodeExponentialDPCM, and decodeFibonacciDPCM
// were taken from "A Programer's Guide to Sound" by Tim Kientzle
// see copyright for details on his copyright

// mu-law
static uword *muLawDecodeTable = 0;

uword decodeMuLawSample(ubyte ulaw)
{
  ulaw = ~ulaw;
  const ubyte exponent = (ulaw >> 4) & 0x7;
  const ubyte mantissa = (ulaw & 0xf) + 16;
  udword adjusted = (mantissa << (exponent + 3)) - 128 - 4;
  return (ulaw & 0x80) ? adjusted : -adjusted;
}

int decodeMuLaw(ubyte *src, uword *&dest, udword length)
{
  // create decode table if it doesn't exist
  if (!muLawDecodeTable)
    {
      muLawDecodeTable = (uword *)malloc(256*sizeof(uword));
      if (!muLawDecodeTable)
	return 0;
      for (int i=0;i<256;i++)
	muLawDecodeTable[i] = decodeMuLawSample(i);      
    }  
  for (long i = length-1; i>=0; i--)
    dest[i] = muLawDecodeTable[src[i]];
  return 1;
}

// A-law
// this was taken from "A Programer's Guide to Sound" by Tim Kientzle
static uword *aLawDecodeTable = 0;

uword decodeALawSample(ubyte alaw)
{
  alaw ^= 0x55;
  const ubyte exponent = (alaw >> 4) & 0x7;
  const ubyte mantissa = (alaw & 0xf) + (exponent ? 16 : 0);  
  udword adjusted = (mantissa << (exponent + 4));
  return (alaw & 0x80) ? -adjusted : adjusted;
}

int decodeALaw(const ubyte *src, uword *&dest, const udword length)
{
  // create decode table if it doesn't exist
  if (!aLawDecodeTable)
    {
      aLawDecodeTable = (uword *)malloc(256*sizeof(uword));
      if (!aLawDecodeTable)
	return 0;
      for (int i=0;i<256;i++)
	aLawDecodeTable[i] = decodeALawSample(i)<<8;
    }  
  for (long i = length-1; i>=0; i--)
    dest[i] = aLawDecodeTable[src[i]];
  return 1;
}
// fibonacci DPCM
static const sbyte fibonacci[] = {
  -34, -21, -13, -8, -5, -3, -2, -1, 0, 1, 2, 3, 5, 8, 13, 21
};

// dest must be twice as long as src, length is the length of src
int decodeFibonacciDPCM(const ubyte *src, ubyte *&dest, const udword length)
{
  sbyte previous = src[0];
  dest[0] = previous^0x80;
  for (udword i = 0; i < length-1; i++)
    {
      previous += fibonacci[(src[i+1]>>4) & 0xf];
      (sbyte)dest[i*2] = previous^0x80; 
      previous += fibonacci[src[i+1] & 0xf];
      (sbyte)dest[i*2+1] = previous^0x80;
    }
  return 1;
}
// exponential DPCM
static const sbyte exponential[] = {
  -128, -64, -32, -16, -8, -4 ,-2, -1, 0, 1, 2, 4, 8, 16, 32, 64
};
int decodeExponentialDPCM(ubyte *src, ubyte *&dest, udword length)
{
  sbyte previous =  src[0];
  dest[0] = previous^0x80;
  for (udword i = 0; i < length-1; i++)
    {
      previous += exponential[(src[i+1]>>4)&0xf];
      dest[i*2] = previous^0x80; 
      previous += exponential[src[i+1]&0xf];
      dest[i*2+1] = previous^0x80;
    }
  return 1;
}

///////////////////////////////////////////////////////////////////////////////
//                  Sample loaders - iff, raw,  voc, wav                     //
///////////////////////////////////////////////////////////////////////////////
// generic sample loader
int Sample::load(const char *filename)
{
  if       (strstr(filename,".iff")) 
    return loadIFF(filename);
  else if  (strstr(filename,".voc")) 
    return loadVOC(filename);
  else if  (strstr(filename,".wav")) 
    return loadWAV(filename);
  else if  (strstr(filename,".au"))
    return  loadAU(filename);
  else 
    return loadRAW(filename);
}

int Sample::loadAU(const char *filename)
{
  udword magic;
  AU_HEADER header;
  struct stat filestat; /* used to get the length of the file */
  ubyte *temp_data = 0;
  FILE *fp = fopen(filename, "rb");
  
  if (fp == 0) 
    {
      error_message = "error opening file";
      return 0;
    }

  stat(filename, &filestat);
  length = filestat.st_size;
  
  if (fread(&magic,4,1,fp)!=1)
    {
      error_message = "read error";
      return 0;
    }
  if (!memcmp(&magic, ".snd", 4)) // header is present 
    {
      if (fread(&header,sizeof(AU_HEADER),1,fp)!=1)
	{
	  error_message = "read error";
	  return 0;
	}
      frequency = B_ENDIAN_32(header.sample_rate);
      length -= B_ENDIAN_32(header.header_size);
#ifdef DEBUG
      printf("header size %ld\n",B_ENDIAN_32(header.header_size));
      printf("data length %ld\n", length);
      printf("format      %ld\n",B_ENDIAN_32(header.format));
      printf("sample rate %d\n",frequency);
      printf("channels    %ld\n",B_ENDIAN_32(header.channels));
#endif      
      temp_data = (ubyte *)malloc(length*sizeof(ubyte));
      if (!temp_data)
	{
	  error_message = "malloc error";
	  return 0;
	}
      if (fread(temp_data,length,1,fp)!=1)
	{
	  error_message = "read error";
	  return 0;
	}
      data = (ubyte *)malloc(length*sizeof(uword));
      if (!data)
	{
	  error_message = "malloc error";
	  return 0;
	}      
      switch (B_ENDIAN_32(header.format))
	{
	case 1: //8 bit mu-law G.711
	  decodeMuLaw(temp_data, (uword *&)data, length);
	  // multiply length by two (as samples are 16 bit)
	  length <<=1;
	  flags |= SMP_16BIT;
	  return 1;
	case 27: // 8-bit A-Law G.711
	  decodeALaw(temp_data, (uword *&)data, length);
	  // multiply length by two (as samples are 16 bit)
	  length <<=1;
	  flags |= SMP_16BIT;
	  return 1;
	case 2: // 8-bit linear
	  data = temp_data;
	  // convert to unsigned
	  for (unsigned int i = 0; i < length; i++) 
	    *(data + i) ^= 0x80;
	  flags &= ~SMP_16BIT;
	  return 1;
	case 3: // 16-bit linear
	  data = temp_data;
	  flags |= SMP_16BIT;
	  return 1;
	case 4: // 24-bit linear
	case 5: // 32-bit linear
	case 6: // Floating-point
	case 7: // Double precision floating point
	case 8: // Fragmented sample data
	case 10: // DSP program 
	case 11: // 8-bit fixed point
	case 12: // 16-bit fixed point
	case 13: // 24-bit fixed point
	case 14: // 32-bit fixed point 
	case 18: // 16-bit linear with emphasis
	case 19: // 16-bit linear compressed
	case 20: // 16-bit linear with emphasis and compression
	case 21: // Music kit DSP commands
	case 23: // ADPCM G.721
	case 24: // ADPCM G.722
	case 25: // ADPCM G.723.3
	case 26: // ADPCM G.723.5
	default:
	  error_message = "Unknown AU format";
	  return 0;
	}
    }
  // NO header present
  // assume this is a 8khz mu-Law dump
  frequency = 8000;
  // read and decode data
  temp_data = (ubyte *)malloc(length*sizeof(ubyte));
  if (!temp_data)
    {
      error_message = "malloc error";
      return 0;
    }
  if (fread(temp_data,length,1,fp)!=1)
    {
      error_message = "read error";
      return 0;
    }
  data = (ubyte *)malloc(length*sizeof(uword));
  if (!data)
    {
      error_message = "malloc error";
      return 0;
    }      
  decodeMuLaw(temp_data, (uword *&)data, length);
  // multiply length by two (as samples are 16 bit)
  length <<=1;
  flags |= SMP_16BIT;
  return 1;
}

int Sample::loadIFF(const char *filename) 
{
  int nextseek; /* used to get the next chunk */
  Voice8Header Header;
  long Len;
  udword chunkid;
  udword chunksize;
  struct stat filestat; /* used to get the length of the file */
  FILE *fp = fopen(filename, "rb");
  
  if (fp == 0) 
    {
      error_message = "error opening file";
      return 0;
    }
  
  if (fread(&chunkid, 4, 1, fp) != 1)
    {
      error_message = "read error 1";
      return 0;
    }
  if (memcmp(&chunkid, "FORM",4)) 
    {
      error_message = "FORM not found.";
      return 0;
    }
  if (fread(&Len, 4, 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error 2";
      return 0;
    }

  if (fread(&chunkid, 1, 4, fp) != 4) 
    {
      error_message = "read error";
      return 0;
    }
  if (!memcmp(&chunkid, "8SVX", 4)) 
    flags &= ~SMP_16BIT; // clear 16-bit bit in flags
  else if (!memcmp(&chunkid, "16SV",4))     
    flags |= SMP_16BIT; 
  else 
    { /* UNKNOWN IFF SAMPLE FORMAT */
      fclose(fp);
      error_message = "Unknown IFF sample format";
      return 0;
    }
  
  /* READ IN CHUNKS IN ANY ORDER */
  nextseek = ftell(fp);
  /*ltmp = B_ENDIAN_32(Len);*/
  /*  Len = ltmp + nextseek - 4; */
  /* DONT TRUST THE FILESIZE GIVEN IN THE FILE !!!!!!! */
  stat(filename, &filestat);
  Len = filestat.st_size;
#ifdef DEBUG
  printf("  Len 0x%lX\n", Len);
#endif
  while (nextseek < Len) 
    {
      /* SEEK TO NEXT CHUNK */
      if (fseek(fp, nextseek, SEEK_SET)) 
	perror("seek");
      
    /* READ CHUNK ID */
      if (fread(&chunkid, 1, 4, fp) != 4) 
	{
	  perror("fread");
	  fclose(fp);
	  error_message = "read error 3";
	  return 0;
	}
      /* READ CHUNK SIZE */
      if (fread(&chunksize, 4, 1, fp) != 1) 
	perror("fread");	
      long ltmp = B_ENDIAN_32(chunksize);
      chunksize = ltmp + (ltmp % 2); /* pad if not word aligned */
#ifdef DEBUG
      /* SHOW CHUNK ID */
     printf("\n----- %.4s Chunk, Size %lu -----\n",(char *)&chunkid,chunksize);
#endif
    nextseek = chunksize + ftell(fp);

    switch (chunkid)
      {	
      case CKID_IFF_NAME: /* "NAME" */
        this->name = (char *)malloc(chunksize + 1);
	if (!this->name)
	  {
	    error_message = "malloc error";
	    return 0;
	  }
        if (fread(this->name, 1,chunksize, fp) != chunksize) 
	  {
	    fclose(fp);
	    error_message = "read error 6";
	    return 0; 
	  }
	name[chunksize] = 0; /* do this to stop printf printing junk*/
#ifdef DEBUG
        printf("  Name : %s\n", name);
#endif
        break;
      case CKID_IFF_VHDR: /* "VHDR" */
        if (fread(&Header, sizeof(Voice8Header), 1, fp) != 1) 
	  {
	    fclose(fp);
	    error_message = "read error 8";
	    return 0;
	  }
        /* BODY SHOULD COME AFTER VHDR  - Not always :)*/
#ifdef DEBUG
        printf("  OneShotHi     %ld\n", B_ENDIAN_32(Header.OneShotHi));
        printf("  RepeatHi      %ld\n", B_ENDIAN_32(Header.RepeatHi));
        printf("  SmpPerHiCycle %ld\n", B_ENDIAN_32(Header.SmpPerHiCycle));
        printf("  SmpPerSec     %d\n",  B_ENDIAN_16(Header.SmpPerSec));
        printf("  NumOctaves    %hd\n", Header.NumOctaves);
        printf("  Cmp           %hd\n", Header.Cmp); 
        printf("  Volume        %ld\n",  B_ENDIAN_32(Header.Volume));
        printf("POSITION 0x%lX\n",      ftell(fp));
#endif
	break;
      case CKID_IFF_BODY: /* "BODY" */
	{
#ifdef DEBUG
	  printf("POSITION 0x%lX\n",      ftell(fp));
#endif
	  length = chunksize;
	  nextseek = ftell(fp) + length + (length % 2);

	  if (Header.Cmp)
	    length >>=1;

	  ubyte *temp_data = (ubyte *)malloc(length);
	  if (!temp_data) 
	    {
	      error_message = "Malloc error"; 
	      return 0;
	    }
	  /* READ IN SAMPLE DATA */
	  if (fread(temp_data, length, 1, fp) != 1) 
	    {
	      if (ferror(fp))
		{
		  perror("fread");
		  fclose(fp);		  
		  error_message = "read error";
		  return 0;
		}
	    }	
	  switch (Header.Cmp)
	    {
	    case 0: // PCM
	      data = temp_data;
	      break;
	    case 1: // Fibonacci coding
	      data = (ubyte *)malloc(length*2);
	      if (!data) 
		{
		  error_message = "Malloc error"; 
		  return 0;
		}
	      decodeFibonacciDPCM(temp_data,data,length);
	      length <<= 1;
	      free(temp_data);
	      break;
	    case 2: // exponential coding
	      data = (ubyte *)malloc(length*2);
	      if (!data) 
		{
		  error_message = "Malloc error"; 
		  return 0;
		}
	      decodeExponentialDPCM(temp_data,data,length);
	      length <<= 1;
	      free(temp_data);
	      break;
	    }
	  break;
	}
      
#ifdef DEBUG
      default:
	printf("  Unknown Chunk : %4s 0x%lX\n", (char *)&chunkid, chunkid);
#endif
      } /* END SWITCH */
    } /* END WHILE */

  if (Header.Volume)
    volume    = B_ENDIAN_32(Header.Volume) / 1024;
  if (Header.SmpPerSec != 0) 
    frequency = B_ENDIAN_16(Header.SmpPerSec);
  else 
    {
      if (flags & SMP_16BIT) 	
	frequency = 22048;
      else 	
	frequency = 11024;
    }

  /* convert 8bit data to unsigned */
  if (!flags & SMP_16BIT) 
    for (unsigned int i = 0; i < length; i++) 
      *(data + i) ^= 0x80;
  fclose(fp);
  return 1;
}

/* assume the sample is 8bit unsigned */
int Sample::loadRAW(const char *filename)
{
  FILE *fp = fopen(filename, "rb");

  if (fp) 
    {
      if (fseek(fp, 0, SEEK_END) == 0) 
	{
	  length = ftell(fp);
	  if (length != (udword)-1) 
	    {
	      if (fseek(fp, 0, SEEK_SET) == 0) 
		{
		  data = (ubyte *)malloc(length);
		  if (data) 
		    {
		      if (fread(data, length, 1, fp) == 1) 
			{
			  flags &= ~SMP_16BIT;
			  frequency = 11024; /* just guess at frequency */
			} 
		      else 
			error_message = "Error reading file!";
		    } 
		  else
		    error_message = "Out of memory!";
		} 
	      else
		error_message = "Seek error!";
	    } 
	  else
	    error_message = "Tell error!";
	} 
      else
	error_message = "Seek error!";
    } 
  else 
    {
      perror("fopen");
      error_message = "Error opening file";
    }

  fclose(fp);

  loop_start = 0;
  loop_end   = 0;
  name = strdup(filename);

  if (error_message)
    return 0;
  return 1;
}

int Sample::loadVOC(const char *filename)
{
  char magic[20];
  ubyte blockid, temp, *tBuf;
  uword firstBlock, version, id;
  udword blocklen = 0;

  length = 0;

  FILE *fp = fopen(filename, "rb");

  if (!fp) 
    {
      error_message = "No Such File";
      return 0;
    }
  
  fread(magic, 1, 20, fp);
  // \x1a is the end-of-file marker in MS-DOS
  if (strncmp(magic,"Creative Voice File\x1a", 20)) 
    {
      error_message = "Not a valid .VOC file!";
      return 0;
    }

  // read in size of header (or position of the first block)
  fread(&firstBlock, 1, 2, fp); 
#ifdef DEBUG
  printf("First block %d\n", firstBlock);
#endif
  fread(&version, 1, 2, fp);
#ifdef DEBUG
  printf("Version: %d.%d\n", version >> 8, version & 0xff);
#endif
  if (version > 0x010A) /* I don't have the specs for new .VOCs */
    {
      error_message = "Unsupported .VOC file version!";   
      return 0;
    }
  fread(&id,1,2,fp);
  if (id != (~version) + 0x1234)
    {
      error_message = "Not a valid .VOC file!";
      return 0;
    }

  fseek(fp, firstBlock, SEEK_SET);
  do {
    fread(&blockid, 1, 1, fp);
#ifdef DEBUG
    printf("BLOCKID : %d\n", blockid);
#endif
    // a blockid of zero signifies the end of the voc file
    if (blockid) 
      {
	// is this right ? 
	fread(&blocklen, 1, 3, fp);
#ifdef DEBUG
	printf("Block length : %ld\n",blocklen);
#endif	
	switch(blockid) 
	  {
	  case 1: // Sound Data Block
	    // some voc files have more than one sound data block
	    // so we need to keep reallocating spce for the data
#ifdef DEBUG
	    printf("Sound Data Block\n");
#endif	    
	    fread(&temp, 1, 1, fp);   /* Time-constant */	      
	    if (temp < 256)
	      frequency = 1000000 / (256 - temp);	      
	    
	    fread(&temp, 1, 1,fp);
#ifdef DEBUG
	    printf("Format : %d\n",temp);
#endif	    
	    switch (temp)
	      {
	      case 0: // unsigned 8 bit PCM
		{
		  length = length + blocklen - 2;
		  if (!data)
		    data = (byte *)malloc(length);
		  else
		    data = (byte *)realloc(data,length);
		  if (!data) 
		    {
		      fclose(fp);
		      error_message = "malloc error";
		      return 0;
		    }
		  flags &= ~SMP_16BIT;
		  ubyte *position = data + length + 2 - blocklen;
		  fread(position, (blocklen - 2), 1, fp);
		  break;
		}
	      case 1:   // compressed 4 bits per sample
	      case 2:   // compressed 2.6 bits per sample
	      case 3:   // compressed 2 bits per sample
	      default:
		fclose(fp);
		error_message = "Unknown sample encoding";
		return 0;
	      }
	    break;
	  case 2: // sound continuation block
#ifdef DEBUG
	    printf("Sound Continuation Block");
#endif	    
	    tBuf = (ubyte *)malloc(length + blocklen);
	    if (!tBuf)
	      {
		free(data);
		error_message = "malloc error";
		return 0;
	      }
	    memcpy(tBuf, data, length);
	    fread(tBuf + length, 1, blocklen, fp);
	    length += blocklen;
	    free(data);
	    data = tBuf;
	    break;
	  case 3: // a block of silence
	  case 4: // Marker Block
	  case 5: // Text Block
	  case 6: // loop start
	  case 7: // loop end
	  case 8: // stereo extension block
	  case 9: // compression extension block	      
	  default:
	    // fix this up !
	    // use fseek instead !
	    while (blocklen--)
	      fgetc(fp);	      
	    break;
	  }
      }
  } while (blockid != 0);
  name = strdup(filename);
  return 1;
}

int Sample::loadWAV(const char *filename) 
{
  char        *xchar = 0;
  RiffHeader  Riff;
  CommonChunk Common;
  uword       extrasize;
  udword      SampleLength;
  int         i;
  udword      nextseek;
  udword      listseek; /* Used in lists */
  udword      chunkid; /* four letter id*/
  udword      chunksize;
  FILE        *fp = fopen(filename, "rb");
  uword       SamplesPerBlock; 
  udword      ltmp;

  /* MSADPCM VAIABLES */
  ADPCMCOEFSET *coefset = 0;
  word         NumCoef; /* adpcmcoefset points to NumCoef ADPCMCOEFSET's */
  /**/

  if (fp == 0) 
    {
      error_message = "No Such File";
      return 0;
    }
  

  if (fread(&Riff, sizeof(RiffHeader), 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "Error Reading File 1";
      return 0;
    }
  if (strncmp((char *)Riff.RIFF, "RIFF",4)) 
    {
      fclose(fp);
      error_message = "Not a riff file";
      return 0;
    }
  Riff.Size += ftell(fp);
  if (fread(&chunkid, 4, 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error";
      return 0;
    }
  if (memcmp(&chunkid, "WAVE", 4)) 
    {
      fclose(fp);      
      error_message = "Not a WAVE file";
      return 0;
    }
  
  /* READ IN CHUNKS */
  nextseek = ftell(fp);
  while(nextseek < Riff.Size) 
    {
      if (fseek(fp, nextseek, SEEK_SET))
	{
	  // FIX THIS UP!
	  printf("seek error\n");
	}
      if (fread(&chunkid, sizeof(char), 4, fp) != 4) 
	{
	  fclose(fp);
	  error_message = "read error chunkid";
	  return 0;
	}
      if (fread(&chunksize, 4, 1, fp) != 1) 
	{
	  fclose(fp);
	  error_message = "read error chunksize";
	  return 0;
	}
#ifdef DEBUG
     printf("\n----- %.4s Chunk, Size %lu -----\n",(char *)&chunkid,chunksize);
#endif
    nextseek = chunksize + ftell(fp); /* get position of next chunk */
    
    switch (chunkid)
      {
      case CKID_RIFF_FMT: /* "fmt "*/
        if (fread(&Common, sizeof(CommonChunk), 1, fp) != 1) 
	  {
	    fclose(fp);
	    error_message = "read error 2";
	    return 0;
	  }
        frequency = Common.SamplesPerSec;
#ifdef DEBUG
        printf("  FormatTag      : %d\n",  Common.FormatTag);
        printf("  SamplesPerSec  : %ld\n", Common.SamplesPerSec);
        printf("  AvgBytesPerSec : %ld\n", Common.AvgBytesPerSec);
        printf("  BlockAlign     : %d\n",  Common.BlockAlign);
        printf("  BitsPerSample  : %d\n",  Common.BitsPerSample);
#endif

        if (chunksize > 16) /* skip junk*/
	  { 
	    if (fread(&extrasize, sizeof(uword), 1, fp) != 1) 
	      {
		if (ferror(fp))
		  {
		    fclose(fp);
		    error_message = "read error 4";
		    return 0;
		  }
	      }
	    printf("  Extra size: %d\n", extrasize);

	    if (Common.FormatTag == WAVE_FORMAT_ADPCM)
	      {
#ifdef DEBUG
		printf("  WAVE_FORMAT_ADPCM\n");
#endif
	    
		/* read in Samples Per Block */
		if (fread(&SamplesPerBlock, 2, 1, fp) != 1) 
		  {
		    if (ferror(fp))
			{
			  perror("fread");
			  fclose(fp);
			  error_message = "read error SPB";
			  return 0;
			}
		  }
		if (fread(&NumCoef, 2, 1, fp) != 1) 
		  {
		    if (ferror(fp))
		      {
			perror("fread");
			fclose(fp);
			error_message = "read error NumCoef";
			return 0;
		      }
		  }
		/* allocate memory for coefset */
		coefset = (ADPCMCOEFSET *)malloc(NumCoef*sizeof(ADPCMCOEFSET));
		if (!coefset) 
		  {		    
		    fclose(fp);
		    error_message = "malloc error";
		    return 0;
		  }
		if (fread(coefset, NumCoef*sizeof(ADPCMCOEFSET), 1, fp) != 1) 
		  {
		    if (ferror(fp))
		      {
			perror("fread");
			error_message = "read error coefset";
			return 0;
		      }
		  }
#ifdef DEBUG
		printf("  SamplesPerBlock %d\n", SamplesPerBlock);
		printf("  NumCoef         %d\n", NumCoef);
		for (i = 0; i < NumCoef; i++) 
		  printf("  [%d] %d, %d\n", i,coefset[i].iCoef1,
			 coefset[i].iCoef2);
#endif
	    
	      } 
	    else if (Common.FormatTag == WAVE_FORMAT_DVI_ADPCM) 
	      {  
#ifdef DEBUG
		printf("  WAVE_FORMAT_DVI_ADPCM\n");
#endif
	    /* read in Samples Per Block */
		if (fread(&SamplesPerBlock, 2, 1, fp) != 1) 
		  {
		    if (ferror(fp))
		      {
			perror("fread");
			fclose(fp);
			error_message = "read error SPB2";
			return 0;
		      }
		  }
#ifdef DEBUG
		printf("  SamplesPerBlock %d\n", SamplesPerBlock);
#endif
	    
	      } 
	    else if (chunksize > 18) 
	      {
		fseek(fp,(chunksize - 18), SEEK_CUR);
		printf("  Junk Skipped %ld\n", (chunksize - 18));
	      }
	    
	  }
        break;
      case CKID_RIFF_DATA: /* "data" */        
        length = chunksize;
	switch (Common.FormatTag)
	  { // Format switch
	  case WAVE_FORMAT_PCM:
	    /* READ IN SAMPLE DATA */
	    // should be 8 or 16 bit
	    if (Common.BitsPerSample == 16)
	      flags |= SMP_16BIT; /* set 16 bit flag */
	    else if (Common.BitsPerSample == 8)
	      flags &= ~SMP_16BIT;
	    
	    data = (ubyte *)malloc(length);
	    if (data)
	      {
		if (fread(data, length, 1, fp)!=1)
		  {
		    if (ferror(fp))
		      {
			fclose(fp);
			error_message = "read error data 1";
			return 0;
		      }
		  }
	      } 
	    else 
	      {
		fclose(fp);
		error_message = "Memory Allocation Error";
		return 0;
	      }
	    break;
	  case WAVE_FORMAT_ALAW:
	    {
#ifdef DEBUG
	      printf("WAVE_FORMAT_ALAW\n");
#endif
	      ubyte *temp_data = (ubyte *)malloc(length*sizeof(ubyte));
	      if (!temp_data)
		{
		  error_message = "malloc error";
		  return 0;
		}
	      if (fread(temp_data,length,1,fp)!=1)
		{
		  error_message = "read error temp_data";
		  return 0;
		}
	      data = (ubyte *)malloc(length*sizeof(uword));
	      if (!data)
		{
		  error_message = "malloc error";
		  return 0;
		}      
	      decodeALaw(temp_data, (uword *&)data, length);
	      flags |= SMP_16BIT;
	      // multiply length by two (as samples are 16 bit)
	      length <<=1;
	      break;
	    }
	  case WAVE_FORMAT_MULAW:
	    {
#ifdef DEBUG
	      printf("WAVE_FORMAT_MULAW\n");
#endif
	      ubyte *temp_data = (ubyte *)malloc(length*sizeof(ubyte));
	      if (!temp_data)
		{
		  error_message = "malloc error";
		  return 0;
		}
	      if (fread(temp_data,length,1,fp)!=1)
		{
		  error_message = "read error temp_data";
		  return 0;
		}
	      data = (ubyte *)malloc(length*sizeof(uword));
	      if (!data)
		{
		  error_message = "malloc error";
		  return 0;
		}      
	      decodeMuLaw(temp_data, (uword *&)data, length);
	      flags |= SMP_16BIT;
	      // multiply length by two (as samples are 16 bit)
	      length <<=1;
	      break;
	    }
	  case WAVE_FORMAT_ADPCM:
	    {
#ifdef DEBUG
	    printf("WAVE_FORMAT_ADPCM\n");
#endif
	    // I haven't been able to find the tables microsoft uses for
	    // their adpcm format so this is still unfinished
	    /* VARIABLES USED FOR MS ADPCM */
	    ADPCMBLOCKHEADER blockheader;
	    ubyte *adpcmdata = 0;
	    
	    /* read the ADPCM BLOCK HEADER */
	    if (fread(&blockheader,sizeof(ADPCMBLOCKHEADER),1,fp)!=1) 
	      {
		if (ferror(fp))
		  {
		    perror("fread");
		    fclose(fp);
		    error_message = "read error blockheader";
		    return 0;
		  }
	      }
#ifdef DEBUG
	    printf("  Predictor %hd\n", blockheader.bPredictor[0]);
	    printf("  iDelta    %d\n",  blockheader.iDelta[0]);
	    printf("  iSamp1    %d\n",  blockheader.iSamp1[0]);
	    printf("  iSamp2    %d\n",  blockheader.iSamp2[0]);
#endif
	    adpcmdata = (ubyte *)malloc(chunksize);
	    if (!adpcmdata) 
	      {
		error_message = "malloc error";
		return 0;    
	      }
	    /* READ IN ENCODED DATA */
	    if (fread(adpcmdata, chunksize, 1, fp) != 1) 
	      {
		if (ferror(fp))
		  {
		    perror("fread");
		    error_message = "read error adpcmdata";
		    return 0;
		  }
	      }	      
	    data = (ubyte *)malloc(SampleLength);
	    if (!data) 
	      {
		error_message = "malloc error";
		return 0;
	      }
	    /*
	      lPredSamp = ((iSamp1 * iCoef1) +  
	      (iSamp2 *iCoef2)) / FIXED_POINT_COEF_BASE
	      */
	      
	    /* DECODE IT! real sample length is given in a Fact chunk */
	    free(adpcmdata);
	    adpcmdata = 0;
	    error_message = "NOT IMPLEMENTED YET!!";
	    return 0;
	    } 
	  case WAVE_FORMAT_DVI_ADPCM:
	    { 
#ifdef DEBUG
	    printf("WAVE_FORMAT_DVI_ADPCM\n");
#endif
	      /*VARIABLES USED FOR DVI_ADPCM */
	      DVI_ADPCMBLOCKHEADER dviblockheader;
	      byte Delta; /* only use 4bits */
	      long valprev;
	      long valpred;
	      int vpdiff;
	      int StepTableIndex; /* Step Table Index */
	      long blocklength;
	      ubyte *BlockData = 0;
	      ubyte *DataPointer = 0;
	      unsigned int DataRead;
	      
	      /* 
		 Read in each block and decode it until all 
		 blocks are read 
		 allocate memory for Sample Data 
		 */
	      length = SampleLength*2; /* Taken from fact chunk */
	      data = (byte *)malloc(length);
	      DataPointer = data; 
	      /* allocate memory for BlockData*/            
	      /* subtract 4 from length to get the length of the data */
	      blocklength=Common.BlockAlign-sizeof(DVI_ADPCMBLOCKHEADER);
	      BlockData = (byte *)malloc(blocklength);
	      if (!BlockData)
		{
		  error_message = "malloc error";
		  return 0;
		}
	      DataRead = 0;
	      while (DataRead < chunksize) 
		{		      
		  if(fread(&dviblockheader,sizeof(DVI_ADPCMBLOCKHEADER),1,fp)!=1)
		    {
		      if (ferror(fp))
			{
			  perror("fread");
			  error_message = "read error dviblockheader";
			  return 0;
			}
		    }
#ifdef DEBUG
		  if (dviblockheader.Reserved)
		    printf("Error in DVI block header\n");
#endif
		  /* Output Samp0 */
		  *DataPointer++ = dviblockheader.Samp0 & 0xff;
		  *DataPointer++ = ((dviblockheader.Samp0 >> 8) & 0xff);
		      
		  valprev = dviblockheader.Samp0;
		  StepTableIndex = dviblockheader.StepTableIndex;
		  
		  DataRead += Common.BlockAlign;
		  /* Read in Block Data */
		  if (fread(BlockData, blocklength, 1, fp) != 1)
		    {
		      if (ferror(fp))
			{
			  perror("fread");
			  error_message = "read error blockdata";
			  return 0;
			}
		    }
		  for (i = 0; i < blocklength<<1; i++) 
		    {
		      /* while still Samples to decode */
		      /* Get Next Sample Code SampXCode is only 4 bits !*/
		      if (i % 2)
			Delta = (*(BlockData + (i >> 1)) >> 4) & 0xf;
		      else
			Delta =  *(BlockData + (i >> 1)) & 0xf;
		      
		      /* Calculate the new sample */
		      
		      /* Calculate the difference */
		      // this is difference = (vpdiff +.5)*step/4
		      vpdiff = StepTab[StepTableIndex] >> 3;
		      if (Delta & 1)
			vpdiff += StepTab[StepTableIndex] >> 2;
		      if (Delta & 2)
			vpdiff += StepTab[StepTableIndex] >> 1;
		      if (Delta & 4) 
			vpdiff += StepTab[StepTableIndex];
		      if (Delta & 8) /* Check the sign bit */
			valpred = valprev - vpdiff;
		      else
			valpred = valprev + vpdiff;
		      /* Check for overflow and underflow errors */
		      if (valpred > 32767) 
			valpred = 32767;
		      else if (valpred < -32768)
			valpred = -32768;
		      /* Output new Sample - SampX */
		      *DataPointer++ = valpred & 0xff;
		      *DataPointer++ = ((valpred >> 8) & 0xff);
		      /* Save the previous sample */
		      valprev = valpred;

		      /* Adjust the Step Table Index */
		      StepTableIndex += IndexTab[Delta];
		      /* Check for step table index 
			 overflow and underflow */
		      if (StepTableIndex > 88)
			StepTableIndex = 88;
		      else if (StepTableIndex < 0)
			StepTableIndex = 0;

		    }
		}
	  /* sample data has been converted to 16 bit signed */
#ifdef DEBUG
	      printf("Sample Length: %ld\n", length);
#endif
	      flags |= SMP_16BIT; /* 16 bit */
	      break;
	    }	   
	  case WAVE_FORMAT_UNKNOWN:
#ifdef DEBUG
	    printf("WAVE_FORMAT_UNKNOWN\n");
#endif
	  default: 	    
	    fclose(fp);
	    error_message = "Unknown Format";
	    return 0;
	  }       
	break;
      case CKID_RIFF_DISP:
        /* read in 0-terminated string and pad */
        xchar = (char *)malloc(chunksize + (chunksize % 2));
        if (!xchar) 
	  {
	    fclose(fp);
	    error_message = "malloc error"; 
	    return 0;
	  }
        if (fread(xchar, chunksize + (chunksize % 2),1, fp) != 1) 
	  {
	    perror("fread");
	    error_message = "read error xchar";
	    return 0;
	  }
#ifdef DEBUG
        printf("  Display : %s\n", xchar);
#endif
        break;
      case CKID_RIFF_FACT: /* "fact" */
        if (fread(&SampleLength, sizeof(udword), 1, fp) != 1) 
	  {
	    fclose(fp);
	    error_message = "read error FACT";
	    return 0;
	  }
        printf("FACT FileSize %ld\n", SampleLength);
        break;
      case CKID_RIFF_LIST: /* "LIST" */
        if (fread(&chunkid, 1, 4, fp) != 4) 
	  {
	    fclose(fp);
	    error_message = "read error LIST 1"; 
	    return 0;
	  }
        /* LISTs have an list type after their size */
#ifdef DEBUG
        printf("  --- %.4s Chunk ---\n",(char *)&chunkid);
#endif
        switch(chunkid)
          {
          case CKID_RIFF_INFO: /* "INFO" */
            /* NO ChunkSize after "INFO" */
            ltmp = chunksize  - 4 + ftell(fp); /* ltmp = size of INFO list */
            listseek = ftell(fp);
            while (listseek < ltmp) 
	      { 
		fseek(fp, listseek, SEEK_SET);
		if (fread(&chunkid, sizeof(char), 4,fp) != 4) 
		  {
		    fclose(fp);
		    error_message = "read error INFO chunkid";
		    return 0;
		  }
		fread(&chunksize, 4, 1, fp);
		listseek = chunksize + ftell(fp) + chunksize % 2;
		/* read in 0-terminated string and pad */
		xchar = (char *)malloc(chunksize + (chunksize % 2));
		if (!xchar) 
		  {
		    fclose(fp);
		    error_message = "malloc error";
		    return 0;
		  }
		fread(xchar, 1, chunksize + (chunksize % 2), fp);
		switch(chunkid) 
		  {
		  case CKID_RIFF_ICMT:
#ifdef DEBUG
		    printf("  Comment       : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ICOP: /* Copyright */
#ifdef DEBUG
		    printf("  Copyright     : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ICRD: /* "ICRD" */
#ifdef DEBUG
		    printf("  Creation Date : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_IENG: /* Engineer */
#ifdef DEBUG
		    printf("  Engineer      : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_IGNR:
#ifdef DEBUG
		    printf("  Genre         : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_IKEY:
#ifdef DEBUG
		    printf("  Keywords      : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_IMED:
#ifdef DEBUG
		    printf("  Medium        : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_INAM:
		    name = (char *)malloc(chunksize);
		    if (!name) 
		      {
			fclose(fp);
			error_message = "malloc error";
			return 0;
		      }
		    strncpy(name, xchar, chunksize);
#ifdef DEBUG
		    printf("  Name          : %s\n", name);
#endif
		    break;
		  case CKID_RIFF_IPRD: /* "IPRD" */
#ifdef DEBUG
		    printf("  Product       : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ISBJ:
#ifdef DEBUG
		    printf("  Subject       : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ISFT: /* "ISFT" */
#ifdef DEBUG
		    printf("  Software      : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ISRC:
#ifdef DEBUG
		    printf("  Source        : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ISRF:
#ifdef DEBUG
		    printf("  Src Reference : %s\n", xchar);
#endif
		    break;
		  case CKID_RIFF_ITCH:
#ifdef DEBUG
		    printf("  Technician    : %s\n", xchar);
#endif
		    break;
#ifdef DEBUG
		  default:
		    printf("  Unknown INFO chunk 0x%lX\n", chunkid);
#endif
		  }
		free(xchar);
		xchar = 0;
	      }
            break;
          default:
            printf("  Unknown LIST type");
          }
        break;
#ifdef DEBUG
      default:	
        printf("  Unknown Chunk : %4s 0x%lX \n", (char *)&chunkid , chunkid);
#endif
      } /*END SWITCH*/
    
    } /*END WHILE*/
  fclose(fp);
  if (!name) 
    name = strdup(filename);
  return 1;
}

/*****************************************************************************\
 .                  Sample savers - iff, raw,  voc, wav                     .
\*****************************************************************************/

int Sample::saveRAW(const char *filename)
{
  FILE *fp = fopen(filename, "wb");
  
  if (fp != NULL) 
    {
      if (length != 0) 
	{
	  if (fwrite(data, length, 1, fp) == 1) 
	    {
	      fclose(fp);
	      return 1;
	    }
	  else 
	    {
	      perror("fwrite");
	      fclose(fp);
	      error_message = "write error";
	      return 0;
	    }
	}
      else /* if no sample data don't need to do anything */ 
	{
	  fclose(fp);
	  error_message = "No Sample data to save";
	  return 0;
	}
    }
  perror("fopen");
  error_message = "Error opening file.";
  return 0;
}

int Sample::saveIFF(const char *filename)
{
  Voice8Header Header;
  udword chunksize;
  FILE *fp = fopen(filename, "wb");

  if (fp == NULL) 
    {
      perror("fopen");
      error_message = "error opening file";
      return 0;
    }
  /* FORM */
  /* leave space for filesize and come back later to write it */ 
  if (fwrite("FORM    ", 8,1,fp) != 1) 
    {
      error_message = "write error";
      return 0;
    }
  /* LENGTH */
  /* CHUNK TYPE - 16SV or 8SVX */
  if (flags & SMP_16BIT) 
    {
      if (fwrite("16SV", 4, 1, fp) != 1) 
	{
	  perror("fwrite");
	  error_message = "write error";
	  return 0;
	}
    }   
  else 
    {
      if (fwrite("8SVX", 4, 1, fp) != 1) 
	{
	  error_message = "write error";
	  return 0;
	}
    }

  /* NAME */
  if (fwrite("NAME",4,1,fp)!= 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  chunksize = 22;
  if (fwrite(&chunksize, 4,1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  if (fwrite(name, 22,1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }

  /* VHDR */
  if (fwrite("VHDR", 4,1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }

  chunksize = sizeof(Voice8Header);

  if (fwrite(&chunksize, sizeof(chunksize),1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  /* no of samples in the section that doesn;t repeat */
  Header.OneShotHi = B_ENDIAN_32(length); 
  Header.RepeatHi = 0;/* Sample->LoopEnd - Sample->LoopStart */
  Header.SmpPerHiCycle = 0;
  Header.SmpPerSec = B_ENDIAN_32(frequency);
  Header.NumOctaves = 0;
  Header.Cmp = 0; /* 0 for no compression */
  Header.Volume = 65535; /* full volume */
  if (fwrite(&Header, sizeof(Voice8Header), 1, fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  /* BODY */
  /* write id */
  if (fwrite("BODY", 4,1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  /* write chunk size */
  chunksize = length;
  if (fwrite(&chunksize, sizeof(chunksize),1,fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  /* convert 8bit data to signed */
  if (!(flags & SMP_16BIT)) 
    for (unsigned int i = 0; i < length; i++) 
      *(data + i) ^= 0x80;
  if (fwrite(data, chunksize, 1, fp) != 1) 
    {
      perror("fwrite");
      error_message = "write error";
      return 0;
    }
  /* convert 8bit data to unsigned -- this is really stupid!!!*/
  if (!(flags & SMP_16BIT)) 
    for (unsigned int i = 0; i < length; i++) 
      *(data + i) ^= 0x80;
  
  return 1;
}
