#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "instrument.h"

Instrument::Instrument(void)
{
  name = 0;
  samples = 0;
  NumSamples = 0;  
}

Instrument::~Instrument(void)
{
  dump();
}

bool Instrument::is_instrument(const char *filename)
{
  if (strstr(filename, ".xi") || strstr(filename, ".pat"))
    return true;
  
  return false;  
}

int Instrument::dump(void)
{
  if (name)
    {
      free(name);
      name = 0;
    }

  if (samples)
    {
      for (int i=NumSamples; i; i--)
	samples[i-1]->dump();
      free(samples);
      samples = 0;
    }

  return 1;
}

/*****************************************************************************\
 .                    Instrument loaders - PAT, XI                           .
\*****************************************************************************/
int Instrument::load(const char *filename)
{
  if (strstr(filename, ".xi")) 
    return loadXI(filename);
  else if (strstr(filename, ".pat")) 
    return loadPAT(filename);
  else
    return 0;  
}

// I haven't really finished these loaders
// I've fixed up the instrument class but still have add parts that assign 
// the proper envelope values and other such things

/* need to exchange sample for inst - and create a better Instument struct */
int Instrument::loadPAT(const char *filename)
{
  PATCHHEADER ph;
  INSTRUMENTDATA in; 
  LAYERDATA layer;
  PATCHDATA *pd; /* ph.instruments*PATCHDATA's */
  ubyte swap;

  FILE *fp = fopen(filename, "rb");

  if (!fp) 
    {
      error_message = "No such file";
      return 0;
    }

  if (fread(&ph, sizeof(PATCHHEADER), 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error";
      return 0;
    }

#ifdef DEBUG
  printf("----- PATCH HEADER -----\n");
  printf("  Header        [%.12s]\n", ph.header);
  printf("  gravis_id     [%.10s]\n", ph.gravis_id);
  printf("  description   [%.60s]\n", ph.description);
  printf("  instruments   %u\n",      ph.instruments);
  printf("  voices        %hd\n",     ph.voices);
  printf("  channels      %hd\n",     ph.channels);
  printf("  wave_forms    %u\n",      ph.wave_forms);
  printf("  master_volume %u\n",      ph.master_volume);
  printf("  data_size     %lX\n",     ph.data_size);
#endif
   
  NumSamples = ph.wave_forms;
  //  NumSamples = ph.instruments;

  //  samples = new Sample [NumSamples];
  if ((samples = (Sample **)calloc(NumSamples,sizeof(Sample *))) == 0) 
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumSamples; i++)
    samples[i] = (Sample *)calloc(1,sizeof(Sample)); 

  if (fread(&in, sizeof(INSTRUMENTDATA), 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error";
      return 0;
    }

#ifdef DEBUG
  printf("\n----- INSTRUMENTDATA -----\n");
  printf("  instrument %hu\n",   in.instrument);
  printf("  name       %.16s\n", in.instrument_name);
  printf("  size       %lX\n",   in.instrument_size);
  printf("  layers     %hd\n",   in.layers);
#endif

  name = strdup(in.instrument_name);

  if (fread(&layer, sizeof(LAYERDATA), 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error";
      return 0;
    }

#ifdef DEBUG
  printf("\n----- LAYERDATA -----\n");
  printf("  layer_duplicate %hd\n", layer.layer_duplicate);
  printf("  layer           %hd\n", layer.layer);
  printf("  layer_size      %ld\n", layer.layer_size);
  printf("  samples         %hd\n", layer.samples);
#endif

  pd = (PATCHDATA *)calloc(ph.wave_forms,sizeof(PATCHDATA));

  if (!pd)
    {
      error_message = "calloc error";
      return 0;
    }

  for (int j = 0; j < NumSamples; j++) 
    {
      if (fread(&pd[j], sizeof(PATCHDATA), 1, fp) != 1) 
	{
	  fclose(fp);
	  error_message = "read error";
	  return 0;
	}

      /* DO ALL THE SAMPLES HAVE THE SAME VIBRATO? */
#ifdef DEBUG
      printf("\n----- PATCHDATA -----\n");
      printf("  wave_name       %.7s\n", pd[j].wave_name);
      printf("  fractions       %hu\n",  pd[j].fractions);
      printf("  wave_size       %ld\n",  pd[j].wave_size);
      printf("  start_loop      %ld\n",  pd[j].start_loop);
      printf("  end_loop        %ld\n",  pd[j].end_loop);
      printf("  sample_rate     %hu\n",  pd[j].sample_rate);
      printf("  low_frequency   %ld\n",  pd[j].low_frequency);
      printf("  high_frequency  %ld\n",  pd[j].high_frequency);
      printf("  root_frequency  %ld\n",  pd[j].root_frequency);
      printf("  tune            %hd\n",  pd[j].tune);
      printf("  balance         %hu\n",  pd[j].balance);

      for (int i = 0; i < ENVELOPES; i++) 
	{
	  printf("  envelope_rate [%.1d] %.3hu",   i, pd[j].envelope_rate[i]);
	  printf("  envelopee_offset [%.1d] %.3hu\n",
		 i,pd[j].envelope_offset[i]);
	}

      printf("  tremolo_sweep   %hu\n",  pd[j].tremolo_sweep);
      printf("  tremolo_rate    %hu\n",  pd[j].tremolo_rate);
      printf("  tremolo_depth   %hu\n",  pd[j].tremolo_depth);
      printf("  vibrato_sweep   %hu\n",  pd[j].vibrato_sweep);
      printf("  vibrato_rate    %hu\n",  pd[j].vibrato_depth);
      printf("  modes           %hd\n",  pd[j].modes);

      if (pd[j].modes & 1) 
	printf("    16bit ");
       else 
	 printf("    8bit ");

      if (pd[j].modes & 2) 
	printf("unsigned\n");
      else
	printf("signed\n");      

      printf("  scale_frequency %hd\n",  pd[j].scale_frequency);
      printf("  scale_factor    %hd\n",  pd[j].scale_factor);
#endif

      /* NOW READ IN SAMPLE DATA ! */
      /* SAMPLE DATA IS BIG ENDIAN BUT HEADERS ARE LITTLE ENDIAN !!!!! */
      samples[j]->frequency = pd[j].sample_rate;
      samples[j]->length    = pd[j].wave_size;
      
      if (pd[j].modes & 1) /* 16 bit wave data */
	{ 
	  samples[j]->flags |= Sample::SMP_16BIT;
	  samples[j]->data = (ubyte *)malloc(samples[j]->length);

  	  if (!samples[j]->data)
	    {
	      error_message = "malloc error";
	      return 0;
	    }

	  if (fread(samples[j]->data,samples[j]->length,1, fp)!=1)
	    {
	      perror("fread ");
	      fclose(fp);
	      error_message = "read error";
	      return 0;
	    }

	  /* SWAP bytes */
	  for (unsigned int i = 0; i < (samples[j]->length >> 1); i++) 
	    {
	      swap = *(samples[j]->data + 2*i + 1);   
	      *(samples[j]->data + 2*i + 1) = *(samples[j]->data + 2*i); 
	      *(samples[j]->data + 2*i + 1) = swap ^ 0x80;
	    }
	} 
      else 
	{ /* 8 bit */
	  samples[j]->flags &= ~Sample::SMP_16BIT;
	  samples[j]->data = (ubyte *)malloc(samples[j]->length);

	  if (!samples[j]->data)
	    {
	      error_message = "malloc error";
	      return 0;
	    }
	  
	  if(fread(samples[j]->data, samples[j]->length,1, fp)!=1) 
	    {
	      fclose(fp);
	      error_message = "8bit data read error";
	      return 0;
	    }
	}
    }
  return 1;
}

int Instrument::loadXI(const char *filename)
{
  XIFileHeader FHeader;
  XIinst in;
  SMPHeader *SHeader = 0;
  FILE *fp = fopen(filename, "rb");
  ubyte junk;
  int newp;
  int old = 0;

  if (!fp) 
    {
      error_message = "No Such File"; 
      return 0;
    } 

  if (fread(&FHeader, sizeof(XIFileHeader), 1, fp) != 1) 
    {
      fclose(fp);
      error_message = "read error 1";
      return 0;
    }

  if (strncmp(FHeader.Magic1, "Extended Instrument: ", 21)) 
    {
      fclose(fp);
      error_message = "Not an Extended Instrument";
      return 0;
    }

  // copy name of instrument
  name = strdup(FHeader.Name);

#ifdef DEBUG
  printf("Name : %.20s\nTracker : %.20s", FHeader.Name,   FHeader.Tracker);
  printf(" version %d.%d\n",              FHeader.VMajor, FHeader.VMinor ); 
#endif

  if (fread(&in, sizeof(XIinst), 1, fp) != 1) 
    {
      fclose(fp);
      perror("fread");
      error_message ="read error 2";
      return 0;
    }

  for (int i = 0; i < 48; i++)
    volume.envelope[i] = in.Vol_Envelope[i];
  volume.points      = in.Num_Vol_Points;
  volume.sustain     = in.Vol_Sus;
  volume.loop_start  = in.Vol_Loop_Start;
  volume.loop_end    = in.Vol_Loop_End;
  volume.fadeout     = in.Vol_Fadeout;

  for (int i = 0; i < 48; i++)
    panning.envelope[i] = in.Pan_Envelope[i];
  panning.points     = in.Num_Pan_Points;
  panning.sustain    = in.Pan_Sus;
  panning.loop_start = in.Pan_Loop_Start;
  panning.loop_end   = in.Pan_Loop_End;
  panning.fadeout    = 0;

  vibrato.type       = in.Vibrato_Type;
  vibrato.sweep      = in.Vibrato_Sweep;
  vibrato.depth      = in.Vibrato_Depth;
  vibrato.rate       = in.Vibrato_Rate;

#ifdef DEBUG
  printf("\n----- XInst -----\n");
  /* Sample_Number[96]; 
   *Vol_Envelope[48];
   *Pan_Envelope[48];*/
  printf("  Num_Vol_Points %d\n", in.Num_Vol_Points);
  printf("  Num_Pan_Points %d\n", in.Num_Pan_Points);
  printf("  Vol_Sus        %d\n", in.Vol_Sus);
  printf("  Vol_Loop_Start %d\n", in.Vol_Loop_Start);
  printf("  Vol_Loop_End   %d\n", in.Vol_Loop_End);
  printf("  Pan_Sus        %d\n", in.Pan_Sus);
  printf("  Pan_Loop_Start %d\n", in.Pan_Loop_Start);
  printf("  Pan_Loop_End   %d\n", in.Pan_Loop_End);
  printf("  Vol_Type       %d\n", in.Vol_Type);
  printf("  Pan_Type       %d\n", in.Pan_Type);
  printf("  Vibrato_Type   %d\n", in.Vibrato_Type);
  printf("  Vibrato_Sweep  %d\n", in.Vibrato_Sweep);
  printf("  Vibrato_Depth  %d\n", in.Vibrato_Depth);
  printf("  Vibrato_Rate   %d\n", in.Vibrato_Rate);
  printf("  Vol_Fadeout    %hd\n",in.Vol_Fadeout);
#endif

  if (fread(&junk, 1,1,fp) != 1) 
    {
      fclose(fp);
      perror("fread");
      error_message = "read error 3";
      return 0;
    }
  NumSamples = junk;
#ifdef DEBUG
  printf("\nNumSamples : %d\n", NumSamples);
#endif

  // create space for samples
  //  samples = new Sample [NumSamples];
  if ((samples = (Sample **)calloc(NumSamples,sizeof(Sample *))) == 0) 
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumSamples; i++)
    samples[i] = (Sample *)calloc(1,sizeof(Sample)); 

  if (fread(&junk, 1, 1, fp) != 1) 
    {
      fclose(fp);
      perror("fread");
      error_message = "read error 4";
      return 0;
    }
  if (NumSamples > 0 ) 
    {
      SHeader = (SMPHeader *)calloc(NumSamples,sizeof(SMPHeader));
      /*
	Read in the SMPHeaders for all the samples
	*/
      if (fread(SHeader, sizeof(SMPHeader)*NumSamples, 1, fp) != 1) 
	{
	  fclose(fp);
	  perror("fread");
	  error_message = "read error 5";
	  return 0;
	}
  
      for (int j = 0; j < NumSamples; j++) 
	{
#ifdef DEBUG
	  printf("\n----- SMPHeader chunk %d ------\n", j);
	  if (SHeader[j].Length != 0) 
	    {
	      printf("  Length       : 0x%lX\n", SHeader[j].Length);
	      printf("  Loop_Start   : 0x%lX\n", SHeader[j].Loop_Start);
	      printf("  Loop_Length  : 0x%lX\n", SHeader[j].Loop_Length);
	      printf("  Volume       : 0x%X\n",  SHeader[j].Volume);
	      printf("  Type         : 0x%X\n",  SHeader[j].Type);
	      printf("  Panning      : 0x%X\n",  SHeader[j].Panning);
	      printf("  RelativeNote : 0x%X\n",  SHeader[j].RelativeNote);
	    }
	  printf("  Name       : %.22s\n", SHeader[j].Name);
	  
#endif
	  samples[j]->length = SHeader[j].Length;
	  if (SHeader[j].Length != 0) 
	    {
	      if ((SHeader[j].Type & 16) == 0) 
		{ /* 8 bit */
		  samples[j]->frequency = 22024;
		  /* clear 16 bit flag */
		  samples[j]->flags &= ~Sample::SMP_16BIT;
		  samples[j]->data = (ubyte *)malloc(samples[j]->length); 
		  if (!samples[j]->data)
		    {
		      error_message = "malloc error";
		      return 0;
		    }
		  if (fread(samples[j]->data, samples[j]->length,1,fp)!=1) 
		    {
		      if (ferror(fp))
			{
			  fclose(fp);
			  perror("fread");
			  error_message = "read error 6";
			  return 0;
			}
		    }
		  /* convert data to real data from deltas */
          
		  for (unsigned int i = 0; i < samples[j]->length; i++) 
		    {
		      newp = *(samples[j]->data + i) + old;
		      *(samples[j]->data + i) = newp ^ 0x80;
		      old = newp;
		    }
		} 
	      else 
		{ /* 16 bit */
		  samples[j]->frequency = 44096;
		  samples[j]->flags |= Sample::SMP_16BIT;
		  samples[j]->data = (ubyte *)malloc(samples[j]->length);
		  if (fread(samples[j]->data, samples[j]->length, 1, fp) != 1) 
		    {
		      if (ferror(fp))
			{
			  fclose(fp);
			  perror("fread");
			  error_message = "read error 7";
			  return 0;
			}
		    }
		  /* convert data to real unsigned data from deltas */
		  for (unsigned int i=0; i < (samples[j]->length>>1); i++) 
		    {
		      /* first need to construct a word from 2 bytes */
		      newp = *(samples[j]->data + 2*i) + 
			(*(samples[j]->data + 2*i + 1) << 8);
		      newp += old;
		      /* LOWER byte */
		      *(samples[j]->data + 2*i) = newp & 0xff; 
		      /* UPPER byte */ 
		      *(samples[j]->data + 2*i + 1) = (newp >> 8) & 0xff;
		      old = newp;
		    }
		}
	    } 
	  else
	    {
	      //	      free(samples[j]);
	      //	      samples[j] = 0;
	    }
	}
    }
  fclose(fp);
  return 1;
}
