/*
 TODO:
	WaweIn device selector
	Support for other bitdepths and samplerates
	Better error handling (don't show messageboxes in Work())

*/



#define VC_EXTRALEAN

#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <float.h>
#include "../../MachineInterface.h"

#pragma optimize ("a", on)

CMachineParameter const paraGain = 
{ 
	pt_byte,										// type
	"Gain",
	"Gain (0=-inf.dB, 1=-64dB, 40=0dB, 80=64dB)",	// description
	0,												// MinValue	
	0x80,  											// MaxValue
	255,											// NoValue
	MPF_STATE,										// Flags
	0x40
};

CMachineParameter const *pParameters[] = { &paraGain };

CMachineAttribute const attrDevice =
{
	"Device",
	0,
	32,
	0
};

CMachineAttribute const attrNumBufs =
{
	"Number of buffers",
	1,
	8,
	3
};

CMachineAttribute const attrBufSize =
{
	"Buffer size",
	128,
	16384,
	4096
};

CMachineAttribute const *pAttributes[] = 
{
	&attrDevice,
	&attrNumBufs,
	&attrBufSize
};


#pragma pack(1)
 
class gvals
{
public:
	byte gain;
};

class avals
{
public:
	int device;
	int numbufs;
	int bufsize;
};

#pragma pack()

class CBlock
{
public:
	HANDLE Handle;
	byte *pData;
	WAVEHDR *phdr;
	HANDLE HeaderHandle;
	bool Prepared;
};

#define IBUF_SIZE	(128*1024)

CMachineInfo const MacInfo = 
{
	MT_GENERATOR,							// type
	MI_VERSION,
	0,										// flags
	0,										// min tracks
	0,          							// max tracks
	1,										// numGlobalParameters
	0,										// numTrackParameters
	pParameters,        
	3, 
	pAttributes,
#ifdef _DEBUG
	"Jeskola WaveIn interface (Debug build)",			// name
#else
	"Jeskola WaveIn interface",
#endif
	"WaveIn",								// short name
	"Oskari Tammelin", 						// author
	NULL
};

class mi : public CMachineInterface
{
public:
	mi();
	virtual ~mi();

	virtual void Init(CMachineDataInput * const pi);
	virtual void Tick();
	virtual bool Work(float *psamples, int numsamples, int const mode);
	virtual void AttributesChanged();
	virtual char const *DescribeValue(int const param, int const value);

	void AddBlock(CBlock &blk);

	void WriteBuffer(short *pin, int c);
	void ReadBuffer(float *pout, int c);

	void Start();
	void Stoppa();		

public:
	double DCBOutput;
	double DCBInput;

	HANDLE hWaveIn;

	int Device;
	int numBufs;
	int BufSize;
	CBlock *pBlock;
	CBlock *Blocks;

	float Gain;
	short *pBufRead;
	short *pBufWrite;
	short Buffer[IBUF_SIZE];
	
	avals aval;
	gvals gval;

	bool Running;

};

DLL_EXPORTS

mi::mi()
{
	GlobalVals = &gval;
	TrackVals = NULL;
	AttrVals = (int *)&aval;
}

mi::~mi()
{
	Stoppa();
}

void mi::Start()
{
	if (Running)
		return;
	
	Blocks = new CBlock [numBufs];
	
	WAVEFORMATEX f;
	f.wFormatTag = WAVE_FORMAT_PCM;
	f.nChannels = 1;
	f.nSamplesPerSec = pMasterInfo->SamplesPerSec;
	f.wBitsPerSample = 16;
	f.nBlockAlign = f.nChannels * f.wBitsPerSample / 8;
	f.nAvgBytesPerSec = f.nBlockAlign * f.nSamplesPerSec;
	f.cbSize = 0;
	
	hWaveIn = NULL;

	if (waveInOpen(&hWaveIn, Device, &f, NULL, 0, CALLBACK_NULL) || hWaveIn == 0)
	{
		pCB->MessageBox("waveInOpen() failed");
		return;
	}
	 
	// init blocks
	CBlock *pb = Blocks;
	for (int c = 0; c < numBufs; c++, pb++)
	{
		pb->HeaderHandle = GlobalAlloc(GHND | GMEM_SHARE, sizeof(WAVEHDR));
		pb->phdr = (LPWAVEHDR)GlobalLock(pb->HeaderHandle);
		pb->Handle = GlobalAlloc(GHND | GMEM_SHARE, BufSize);
		pb->pData = (byte *)GlobalLock(pb->Handle);
		
		pb->phdr->lpData = (char *)pb->pData;
		pb->phdr->dwBufferLength = BufSize;
		pb->phdr->dwBytesRecorded = 0;
		pb->phdr->dwUser = 0;
		pb->phdr->dwFlags = 0;
		pb->phdr->dwLoops = 0;
		pb->phdr->lpNext = NULL;
		pb->phdr->reserved = 0;

	}

	DCBInput = DCBOutput = 0;
	pBufRead = pBufWrite = Buffer;
	
	for (c = 0; c < numBufs; c++)
		AddBlock(Blocks[c]);

	pBlock = Blocks;

	waveInStart(hWaveIn);

	Running = true;
}

void mi::Stoppa()
{
	if (!Running)
		return;

	if (waveInReset(hWaveIn))
	{
		pCB->MessageBox("waveInReset() failed");
		return;
	}
	
	if (waveInClose(hWaveIn))
	{
		pCB->MessageBox("waveinClose() failed");
		return;
	}

	for (int c = 0; c < numBufs; c++)
	{
		GlobalUnlock(Blocks[c].phdr);
		GlobalUnlock(Blocks[c].pData);
		GlobalFree(Blocks[c].HeaderHandle);
		GlobalFree(Blocks[c].Handle);
	}

	delete[] Blocks;
	Blocks = NULL;

	Running = false;
}

void mi::Init(CMachineDataInput * const pi)
{
	Gain = 1.0f;
	Blocks = NULL;
	Running = false;
	Device = attrDevice.DefValue;
	numBufs = attrNumBufs.DefValue;
	BufSize = attrBufSize.DefValue;

	Start();
	
}

void mi::AttributesChanged()
{
	Stoppa();
	
	Device = aval.device;
	numBufs = aval.numbufs;
	BufSize = aval.bufsize;
	
	Start();
}


void mi::AddBlock(CBlock &blk)
{
	if (waveInPrepareHeader(hWaveIn, blk.phdr, sizeof(WAVEHDR)))
	{
		pCB->MessageBox("waveInPrepareHeader() failed");
		return;
	}

	if (waveInAddBuffer(hWaveIn, blk.phdr, sizeof(WAVEHDR)))
	{
		pCB->MessageBox("waveInAddBuffer() failed");

	}
}

char const *mi::DescribeValue(int const param, int const value)
{
	static char txt[16];

	switch(param)
	{
	case 0:
		if (value == 0)
			return "-inf.dB";
		else
			sprintf(txt, "%.1fdB", (float)value-0x40);
		break;
	default:
		return NULL;
	}

	return txt;
}


inline double DBToAmplitude(double const a)
{
	return pow(10, (a * 0.05));
}

void mi::Tick()
{
	if (gval.gain != paraGain.NoValue)
	{
		if (gval.gain > 0)
			Gain = (float)DBToAmplitude(gval.gain - 0x40);
		else
			Gain = 0;
	}
}

void mi::WriteBuffer(short *pin, int numsamples)
{
	short *pout = pBufWrite;
	
	do
	{
		int c = min(numsamples, (Buffer + IBUF_SIZE) - pout);
		
		memcpy(pout, pin, c*sizeof(short));

		numsamples -= c;
		pout += c;
		pin += c;

		if (pout == Buffer + IBUF_SIZE)
			pout = Buffer;

	}
	while(numsamples > 0);

	pBufWrite = pout;	
}

void mi::ReadBuffer(float *pout, int numsamples)
{
	double dcbout = DCBOutput;
	double dcbin = DCBInput;
	double const g = Gain;
	short *pin = pBufRead;

	do
	{
		int c = min(numsamples, (Buffer + IBUF_SIZE) - pin);
		numsamples -= c;
		
		do
		{
			// block dc with a highpass filter
			double i = *pin++;
			dcbout = i - dcbin + (0.99 * dcbout);
			dcbin = i;
			
			*pout++ = (float)(dcbout * g);

		} while(--c);
	
		if (pin == Buffer + IBUF_SIZE)
			pin = Buffer;
	
	} while(numsamples > 0);

	pBufRead = pin;
	DCBInput = dcbin;
	DCBOutput = dcbout;
}

bool mi::Work(float *psamples, int numsamples, int const)
{
	while (pBlock->phdr->dwFlags & WHDR_DONE)
	{
		WriteBuffer((short *)pBlock->pData, BufSize/2); // assume 16bit mono

		if (waveInUnprepareHeader(hWaveIn, pBlock->phdr, sizeof(WAVEHDR)))
		{
			pCB->MessageBox("waveInUnprepareHeader() failed");
			return false;
		}
	
		AddBlock(*pBlock);
		
		pBlock++;
		if (pBlock == Blocks + numBufs)
			pBlock = Blocks;
	}

	ReadBuffer(psamples, numsamples);
	
	return true;
}

